/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.io;

import java.sql.Connection;
// import java.sql.Date;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.sql.Statement;
// import java.sql.Types;
// import java.util.Locale;

import java.util.Date;
import java.util.Calendar;

import org.opengion.fukurou.util.Closer;
import org.opengion.fukurou.util.LogWriter;
import org.opengion.fukurou.util.HybsDateUtil;

import org.jfree.data.gantt.TaskSeriesCollection;
import org.jfree.data.gantt.TaskSeries;
import org.jfree.data.gantt.Task;
// import org.jfree.data.time.SimpleTimePeriod;

/**
 * HybsTaskSeriesCollection は、org.jfree.data.gantt.TaskSeriesCollection を継承したサブクラスで、
 * オブジェクト作成とともに JDBC接続して、TaskSeries データを作成し、セットします。
 * TaskSeriesCollection は、IntervalCategoryDataset, GanttCategoryDataset インターフェースを継承しています。
 *
 * データ形式は、シリーズ名、タスク名、開始日時、終了日時 の順で、シリーズ名でソートしておく必要があります。
 * シリーズ名 は、キーブレイクで、設定する為です。
 * 
 *  select series,task,st(時間),ed(時間) from XX order by series,task,st(時間)
 *
 * @og.rev 5.6.1.0 (2013/02/01) 新規作成
 *
 * @version  0.9.0  2001/05/05
 * @author   Kazuhiko Hasegawa
 * @since    JDK1.1,
 */
public class HybsTaskSeriesCollection extends TaskSeriesCollection {
	private static final long serialVersionUID = 5604L ;

	/**
	 * HybsTaskSeriesCollection オブジェクトの内部に、DB検索結果のデータを設定します(縦持)。
	 *
	 *  select series,task,st(時間),ed(時間) from XX order by series,task,st(時間)
	 * シリーズ名 は、キーブレイクで、設定します。
	 *
	 * @param con  the connection.
	 * @param query  the query.
	 *
	 */
	public void executeQuery( final Connection con, final String query ) throws SQLException {

		Statement statement = null;
		ResultSet resultSet = null;
		try {
			statement = con.createStatement();
			resultSet = statement.executeQuery(query);
			ResultSetMetaData metaData = resultSet.getMetaData();

			int columnCount = metaData.getColumnCount();

			if(columnCount < 4) {
				String errMsg = "HybsTaskSeriesCollection.executeQuery() : 実行できません。\n"
							+ "select series,task,st(時間),ed(時間) は、最低必要です。それ以降は無視します。"
							+ " SQL=" + query ;
				throw new SQLException( errMsg );
			}

			String bkSeries = null;		// キーブレイクのための過去のSeries

			TaskSeries taskseries = null;
			while (resultSet.next()) {
				// first column contains the row key...
				String seriVal	= resultSet.getString(1);					// シリーズ名
				if( seriVal != null && !seriVal.equals( bkSeries ) ) {
					if( taskseries != null ) { add( taskseries ); }			// キーブレイクでTaskSeriesCollectionにセット
					taskseries = new TaskSeries( seriVal );
					bkSeries = seriVal ;
				}

				String taskVal		= resultSet.getString(2);			// タスク名
				String stDataVal	= resultSet.getString(3);			// st(時間)
				String edDateVal	= resultSet.getString(4);			// ed(時間)

				Date stDate = HybsDateUtil.getCalendar( stDataVal ).getTime() ;
				Date edDate = HybsDateUtil.getCalendar( edDateVal ).getTime() ;

				Task task = new Task( taskVal, stDate, edDate );

				taskseries.add( task );
			}
			if( taskseries != null ) { add( taskseries ); }		// キーブレイクでTaskSeriesCollectionにセット
		}
		finally {
			Closer.resultClose( resultSet ) ;
			Closer.stmtClose( statement ) ;
		}
	}
}
