/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.report2;

import java.util.ArrayList;
import java.util.List;

import org.opengion.hayabusa.common.HybsSystemException;

/**
 * シート単位のcontent.xmlを管理するためのクラスです。
 * シートのヘッダー、行の配列、フッター及びシート名を管理します。
 *
 * @og.group 帳票システム
 *
 * @version  4.0
 * @author   Hiroki.Nakamura
 * @since    JDK1.6
 */
class OdsSheet {

	//======== content.xmlのパースで使用 ========================================

	/* 行の開始終了タグ */
	private static final String ROW_START_TAG = "<table:table-row ";
	private static final String ROW_END_TAG = "</table:table-row>";

	/* シート名を取得するための開始終了文字 */
	private static final String SHEET_NAME_START = "table:name=\"";
	private static final String SHEET_NAME_END = "\"";

	/* 変数定義の開始終了文字及び区切り文字 */
	private static final String VAR_START = "{@";
	private static final String VAR_END = "}";
	private static final String VAR_CON = "_";

	/* ラインコピー文字列 5.0.0.2 (2009/09/15) */
	private static final String LINE_COPY = "LINECOPY";

	private final List<String>	sheetRows	= new ArrayList<String>();
	private String			sheetHeader;
	private String			sheetFooter;
	private String			sheetName;
	private String			origSheetName;
	private String			confSheetName;

	/**
	 * シートを行単位に分解します。
	 *
	 * @og.rev 5.0.0.2 (2009/09/15) ボディ部のカウントを引数に追加し、LINECOPY機能実装。
	 * @og.rev 5.2.1.0 (2010/10/01) シート名定義対応
	 *
	 * @param sheet
	 * @param bodyRowCount
	 */
	public void analyze( final String sheet, final int bodyRowCount ) {
		String[] tags = TagParser.tag2Array( sheet, ROW_START_TAG, ROW_END_TAG );
		sheetHeader = tags[0];
		sheetFooter = tags[1];
		for( int i = 2; i < tags.length; i++ ) {
			sheetRows.add( tags[i] );
			lineCopy( tags[i], bodyRowCount ); // 5.0.0.2 (2009/09/15)
		}

		sheetName = TagParser.getValueFromTag( sheetHeader, SHEET_NAME_START, SHEET_NAME_END );
		origSheetName = sheetName;

		confSheetName = null;
		if( sheetName != null ) {
			int cnfIdx = sheetName.indexOf( "__" );
			if( cnfIdx > 0 && !sheetName.endsWith( "__" ) ) {
				confSheetName = sheetName.substring( cnfIdx + 2 );
				sheetName = sheetName.substring( 0, cnfIdx );
			}
		}
	}

	/**
	 * ラインコピーに関する処理を行います。
	 *
	 * {&#064;LINE_COPY}が存在した場合に、テーブルモデル分だけ
	 * 行をコピーします。
	 * その際、{&#064;xxx_y}のyをカウントアップしてコピーします。
	 *
	 * 整合性等のエラーハンドリングはこのメソッドでは行わず、
	 * 実際のパース処理中で行います。
	 *
	 * @og.rev 5.0.0.2 (2009/09/15) 追加
	 * @og.rev 5.1.8.0 (2010/07/01) パース処理の内部実装を変更
	 *
	 * @param row
	 * @param rowCount
	 */
	private void lineCopy( final String row, final int rowCount ) {
//		int preOffset = -1;
//		int curOffset = -1;
//
//		preOffset = row.indexOf( VAR_START + LINE_COPY );
//		// この段階で存在しなければ即終了
//		if( preOffset < 0 ) { return; }
//
//		curOffset = row.indexOf( VAR_END, preOffset );
//		String lcKey = row.substring( preOffset + VAR_START.length(), curOffset );
//		StringBuilder tmpBuf = new StringBuilder( row );
//		if( LINE_COPY.equals( OdsParseUtil.checkKey( lcKey, tmpBuf ) ) ) {
//			// 存在すればテーブルモデル行数-1回ループ(自身を除くため)
//			for( int i = 1; i < rowCount; i++ ) {
//				tmpBuf = new StringBuilder();
//				preOffset = -1;
//				curOffset = 0;
//				while( ( preOffset = row.indexOf( VAR_START, preOffset + 1 ) ) >= 0 ) {
//					tmpBuf.append( row.substring( curOffset, preOffset ) );
//					curOffset = row.indexOf( VAR_END, preOffset + 1 ) + VAR_END.length();
//
//					// {@XXXX_XX}のXXXX_XXの部分
//					String key = row.substring( preOffset + VAR_START.length(), curOffset - VAR_END.length() );
//					key = OdsParseUtil.checkKey( key, tmpBuf );
//					// 不整合はreturnで何もしないで返しておく
//					int keyCheck = key.indexOf( '<' );
//					if( keyCheck >= 0 ) { return; }
//					tmpBuf.append( VAR_START ).append( incrementKey( key, i ) ).append( VAR_END );
//				}
//				tmpBuf.append( row.substring( curOffset, row.length() ) );
//
//				sheetRows.add( tmpBuf.toString() ); // シートに追加
//			}
//		}

		// この段階で存在しなければ即終了
		int lcStrOffset = row.indexOf( VAR_START + LINE_COPY );
		if( lcStrOffset < 0 ) { return; }
		int lcEndOffset = row.indexOf( VAR_END, lcStrOffset );
		if( lcEndOffset < 0 ) { return; }

		StringBuilder lcStrBuf = new StringBuilder( row );
		String lcKey = TagParser.checkKey( row.substring( lcStrOffset + VAR_START.length(), lcEndOffset ), lcStrBuf );
		if( lcKey == null || !LINE_COPY.equals( lcKey ) ) { return; }

		// 存在すればテーブルモデル行数-1回ループ(自身を除くため)
		for( int i = 1; i < rowCount; i++ ) {
			final int cRow = i;
			String rowStr = new TagParser() {
				@Override
				protected void exec( final String str, final StringBuilder buf, final int offset ) {
					String key = TagParser.checkKey( str, buf );
					if( key.indexOf( '<' ) >= 0 ){
						String errMsg = "[ERROR]PARSE:{@と}の整合性が不正です。変数内の特定の文字列に書式設定がされている可能性があります。キー=" + key;
						throw new HybsSystemException( errMsg );
					}
					buf.append( VAR_START ).append( incrementKey( key, cRow ) ).append( VAR_END );
				}
			}.doParse( lcStrBuf.toString(), VAR_START, VAR_END, false );
			sheetRows.add( rowStr );
		}
	}

	/**
	 * xxx_yのy部分を引数分追加して返します。
	 * yが数字でない場合や、_が無い場合はそのまま返します。
	 *
	 * @og.rev 5.0.0.2 LINE_COPYで利用するために追加
	 *
	 * @param key
	 * @param inc
	 *
	 * @return 変更後キー
	 */
	private String incrementKey( final String key, final int inc ) {
		int conOffset = key.lastIndexOf( VAR_CON );
		if( conOffset < 0 ) { return key; }

		String name = key.substring( 0, conOffset );
		int rownum = -1;
		try {
			rownum = Integer.valueOf( key.substring( conOffset + VAR_CON.length(), key.length() ) );
		}
		// エラーが起きてもなにもしない。
		catch( NumberFormatException ex ) {}

		// アンダースコア後が数字に変換できない場合はヘッダフッタとして認識
		if( rownum < 0 ){ return key; }
		else			{ return name + VAR_CON + ( rownum + inc ); }
	}

	/**
	 * シートのヘッダー部分を返します。
	 *
	 * @return ヘッダー
	 */
	public String getHeader() {
		return sheetHeader;
	}

	/**
	 * シートのフッター部分を返します。
	 *
	 * @return フッター
	 */
	public String getFooter() {
		return sheetFooter;
	}

	/**
	 * シート名称を返します。
	 *
	 * @return シート名称
	 */
	public String getSheetName() {
		return sheetName;
	}

	/**
	 * 定義済シート名称を返します。
	 *
	 * @og.rev 5.2.1.0 (2010/10/01) シート名定義対応
	 *
	 * @return 定義済シート名称
	 */
	public String getConfSheetName() {
		return confSheetName;
	}

	/**
	 * 定義名変換前のシート名称を返します。
	 *
	 * @og.rev 5.2.1.0 (2010/10/01) シート名定義対応
	 *
	 * @return 定義済シート名称
	 */
	public String getOrigSheetName() {
		return origSheetName;
	}

	/**
	 * シートの各行を配列で返します。
	 *
	 * @og.rev 4.3.1.1 (2008/08/23) あらかじめ、必要な配列の長さを確保しておきます。
	 *
	 * @return シートの各行の配列
	 */
	public String[] getRows() {
		return sheetRows.toArray( new String[sheetRows.size()] );
	}
}

