/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.fukurou.util.StringUtil;
import org.opengion.fukurou.util.TagBuffer;

/**
 * 配置図／座席表を表す特殊系のビューです。
 *
 * itd タグは、ガントヘッダー部の TDタグの繰返しに使用されます。
 * この繰返しは、ganttParam タグの minDuration で指定された間隔で行われます。
 * (例えば、0.5 を指定すれば、半日単位で処理されます。)
 * itd タグの colspan 属性を指定した場合は、itd 自身が、td タグに colspan を
 * 追加すると共に、繰返し自身を、その指定数だけに抑制します。
 * 具体的には、colspan="2" とすると、２回に一回しか、itd タグが呼び出されなく
 * なります。
 *
 * @og.formSample
 * ●形式：&lt;og:iMatrix  ... /&gt;
 * ●body：なし
 * ●前提：headタグで、adjustEvent="Matrix" を指定してください。
 *
 * ●Tag定義：
 *   &lt;og:iMatrix
 *       mode               【TAG】Layout:レイアウト編集 Value:設定値の編集([Layout/Value])
 *       countPro           【TAG】データノードの属性数( 行, 列, 名称, 値, 色, 編集可否, ステータス )
 *                                 (SQL文のカラム数(非表示項目を除く))
 *       inputWidth         【TAG】編集時の入力欄幅(px)
 *       cellWidth          【TAG】セル幅(px)
 *       cellHeight         【TAG】セル高さ(px)
 *       showStatus         【TAG】ステータスの出力をする(true)/しない(false)
 *       countStatus        【TAG】ステータスの種類数
 *       writable           【TAG】画面編集可(true)/編集不可(false)
 *       editColor          【TAG】セル背景色の編集可否（true:編集可/false:編集不可）
 *       separator          【TAG】セル名称と設定値の間の区切り文字
 *       paramVal           【TAG】セルの設定値はパラメータより渡す
 *       paramColor         【TAG】セルの背景色はパラメータより渡す
 *       notEditBgColor     【TAG】編集不可のセルの背景色
 *       debug              【TAG】デバッグ情報を出力するかどうか[true/false]を指定します(初期値:false)
 *   /&gt;
 *
 * ●使用例
 *  Select 文は、行, 列, 名称, 値, 色, 編集可否, ステータス の順で検索します。
 *  &lt;og:query &gt;
 *    select 行番号,列番号,セル名称,セル設定値,セル背景色,セルの編集可否,セルのステータス
 *    from   レイアウトテーブル
 *    where  ロケーションID = 'XXXX'
 *    order by 行番号,列番号
 *  &lt;/og:query&gt;
 *
 *  viewタグによるHTML出力 は、この通りにしてください。
 *  &lt;og:view
 *      viewFormType    = "HTMLDynamic"
 *      useScrollBar    = "false"
 *      useHilightRow   = "false"
 *      noMessage       = "true"
 *      pageSize        = "0"
 *  /&gt;
 *
 *  &lt;og:iMatrix
 *     mode             =   "Layout/Value" 
 *     countPro         =   "7"            
 *     inputWidth       =   "80px"         
 *     cellWidth        =   "100px"        
 *     cellHeight       =   "30px"         
 *     showStatus       =   "true/false"   
 *     countStatus      =   "6"            
 *     writable         =   "true/false"   
 *     editColor        =   "true/false"   
 *     separator        =   ":"            
 *     paramVal         =   "{@XXX}"       
 *     paramColor       =   "{@XXX}"       
 *     notEditBgColor   =   "#XXXXXX"      
 *  /&gt;
 *
 * @og.rev 5.6.3.2 (2013/04/12) 新規作成
 * @og.group 画面部品
 *
 * @version  5.0
 * @author	 Kazuhiko Hasegawa
 * @since    JDK6.0,
 */
public class ViewIMatrix extends CommonTagSupport {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "5.6.3.2 (2013/04/12)" ;

	private static final long serialVersionUID = 5632 ;		// 5.6.3.2 (2013/04/12)

	/** mode 引数に渡す事の出来る モードリスト  */
	private static final String[] MODE_LIST = new String[] { "Layout" , "Value" };

	private TagBuffer tag = new TagBuffer( "iMatrix" ) ;

	/**
	 * Taglibの終了タグが見つかったときに処理する doEndTag() を オーバーライドします。
	 *
	 * @return	後続処理の指示
	 */
	@Override
	public int doEndTag() {
		debugPrint();		// 4.0.0 (2005/02/28)

		jspPrint( tag.makeTag() );

		return(EVAL_PAGE);		// ページの残りを評価する。
	}

	/**
	 * タグリブオブジェクトをリリースします。
	 * キャッシュされて再利用されるので、フィールドの初期設定を行います。
	 *
	 */
	@Override
	protected void release2() {
		super.release2();
		tag = new TagBuffer( "iMatrix" );
	}

	/**
	 * 【TAG】配置図／座席表の編集モード(Layout:レイアウト編集/Value:設定値の編集)を指定します(必須)。
	 *
	 * @og.tag
	 * mode は、Layout:レイアウト編集を行うのか、Value:設定値の編集を行うかを指定します。
	 * Layout:レイアウト編集
	 * Value:設定値の編集
	 *
	 * @param   mode 編集モード(Layout:レイアウト編集/Value:設定値の編集)
	 */
	public void setMode( final String mode ) {
		String tmpMode = StringUtil.nval( getRequestParameter( mode ),null );

		if( tmpMode == null || !check( tmpMode, MODE_LIST ) ) {
			String errMsg = "指定のモード(mode)は指定できません。モード指定エラー"
							+ HybsSystem.CR
							+ "mode=[" + tmpMode + "] "
							+ HybsSystem.CR
							+ StringUtil.array2csv( MODE_LIST ) ;
			throw new HybsSystemException( errMsg );
		}

		tag.add( "mode",tmpMode );
	}

	/**
	 * 【TAG】データノードの属性数( 行, 列, 名称, 値, 色, 編集可否, ステータス )を指定します。
	 *
	 * @og.tag
	 * SQL文のカラム数(非表示項目を除く)を指定します。
	 * 行, 列, 名称, 値, 色, 編集可否, ステータス の場合は、"7" を指定します。
	 *
	 * @param   countPro データノードの属性数
	 */
	public void setCountPro( final String countPro ) {
		tag.add( "countPro",StringUtil.nval( getRequestParameter( countPro ),null ) );
	}

	/**
	 * 【TAG】編集時の入力欄幅(px)を指定します。
	 *
	 * @og.tag
	 * 編集時の入力欄幅(px)を指定します。
	 *
	 * @param   inputWidth 編集時の入力欄幅(px)
	 */
	public void setInputWidth( final String inputWidth ) {
		tag.add( "inputWidth",StringUtil.nval( getRequestParameter( inputWidth ),null ) );
	}

	/**
	 * 【TAG】セル幅(px)を指定します。
	 *
	 * @og.tag
	 * セル幅(px)を指定します。
	 *
	 * @param   cellWidth セル幅(px)
	 */
	public void setCellWidth( final String cellWidth ) {
		tag.add( "cellWidth",StringUtil.nval( getRequestParameter( cellWidth ),null ) );
	}

	/**
	 * 【TAG】セル高さ(px)を指定します。
	 *
	 * @og.tag
	 * セル高さ(px)を指定します。
	 *
	 * @param   cellHeight セル高さ(px)
	 */
	public void setCellHeight( final String cellHeight ) {
		tag.add( "cellHeight",StringUtil.nval( getRequestParameter( cellHeight ),null ) );
	}

	/**
	 * 【TAG】ステータスの出力をする(true)/しない(false)を指定します(初期値:false)。
	 *
	 * @og.tag
	 * ステータスの出力をする(true)/しない(false)を指定します。
	 *
	 * @param   showStatus ステータスの出力可否[true/false]
	 */
	public void setShowStatus( final String showStatus ) {
		tag.add( "showStatus",StringUtil.nval( getRequestParameter( showStatus ),null ) );
	}

	/**
	 * 【TAG】ステータスの種類数を指定します。
	 *
	 * @og.tag
	 * ステータスの種類数を指定します。
	 *
	 * @param   countStatus ステータスの種類数
	 */
	public void setCountStatus( final String countStatus ) {
		tag.add( "countStatus",StringUtil.nval( getRequestParameter( countStatus ),null ) );
	}

	/**
	 * 【TAG】画面編集可(true)/編集不可(false)を指定します(初期値:false)。
	 *
	 * @og.tag
	 * 画面編集可(true)/編集不可(false)を指定します。
	 *
	 * @param   writable 画面編集可否[true/false]
	 */
	public void setWritable( final String writable ) {
		tag.add( "writable",StringUtil.nval( getRequestParameter( writable ),null ) );
	}

	/**
	 * 【TAG】セル背景色の編集可否(true:編集可/false:編集不可)を指定します(初期値:false)。
	 *
	 * @og.tag
	 * セル背景色の編集可否(true:編集可/false:編集不可)を指定します。
	 *
	 * @param   editColor セル背景色の編集可否[true/false]
	 */
	public void setEditColor( final String editColor ) {
		tag.add( "editColor",StringUtil.nval( getRequestParameter( editColor ),null ) );
	}

	/**
	 * 【TAG】セル名称と設定値の間の区切り文字を指定します(初期値:":")。
	 *
	 * @og.tag
	 * セル名称と設定値の間の区切り文字を指定します。
	 *
	 * @param   separator 区切り文字
	 */
	public void setSeparator( final String separator ) {
		tag.add( "separator",StringUtil.nval( getRequestParameter( separator ),null ) );
	}

	/**
	 * 【TAG】セルの設定値を指定します。
	 *
	 * @og.tag
	 * セルの設定値を指定します。
	 *
	 * @param   paramVal セルの設定値
	 */
	public void setParamVal( final String paramVal ) {
		tag.add( "paramVal",StringUtil.nval( getRequestParameter( paramVal ),null ) );
	}

	/**
	 * 【TAG】セルの背景色を指定します。
	 *
	 * @og.tag
	 * セルの背景色を指定します。
	 *
	 * @param   paramColor セルの背景色
	 */
	public void setParamColor( final String paramColor ) {
		tag.add( "paramColor",StringUtil.nval( getRequestParameter( paramColor ),null ) );
	}

	/**
	 * 【TAG】編集不可のセルの背景色を指定します。
	 *
	 * @og.tag
	 * 編集不可のセルの背景色を指定します。
	 *
	 * @param   notEditBgColor 編集不可のセルの背景色
	 */
	public void setNotEditBgColor( final String notEditBgColor ) {
		tag.add( "notEditBgColor",StringUtil.nval( getRequestParameter( notEditBgColor ),null ) );
	}

	/**
	 * タグの名称を、返します。
	 * 自分自身のクラス名より、自動的に取り出せないため、このメソッドをオーバーライドします。
	 *
	 * @return  タグの名称
	 */
	@Override
	protected String getTagName() {
		return "iMatrix" ;
	}

	/**
	 * このオブジェクトの文字列表現を返します。
	 * 基本的にデバッグ目的に使用します。
	 *
	 * @return このクラスの文字列表現
	 */
	@Override
	public String toString() {
		return org.opengion.fukurou.util.ToString.title( this.getClass().getName() )
				.println( "VERSION"		,VERSION	)
				.println( "tag"			,tag.makeTag()	)
				.println( "Other..."	,getAttributes().getAttribute() )
				.fixForm().toString() ;
	}
}
