/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.fukurou.util.StringUtil;
import org.opengion.fukurou.util.TagBuffer;

/**
 * JavaScriptを利用してクライアント側でテーブルの左右分割します。
 *
 * iTable タグは、ガント全体の左右２分割表示したいカラム数を、fixedCols 属性に指定します。
 * 通常の view タグの後に記述します。
 *
 * @og.formSample
 * ●形式：&lt;og:iTable  ... /&gt;
 * ●body：なし
 * ●前提：headタグで、adjustEvent="Table" を指定してください。
 *
 * ●Tag定義：
 *   &lt;og:iTable
 *       fixedCols          【TAG】左右２分割で、固定したいカラム数(必須)
 *       debug              【TAG】デバッグ情報を出力するかどうか[true/false]を指定します(初期値:false)
 *   /&gt;
 *
 * ●使用例
 *  &lt;og:view
 *      viewFormType = "HTMLTable"
 *      command      = "{&#064;command}"
 *  /&gt;

 *   &lt;og:iTable
 *       fixedCols  = "5"
 *   /&gt;
 *
 * @og.rev 5.6.3.2 (2013/04/12) 新規作成
 * @og.group 画面部品
 *
 * @version  5.0
 * @author	 Kazuhiko Hasegawa
 * @since    JDK6.0,
 */
public class ViewITableTag extends CommonTagSupport {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "5.6.3.2 (2013/04/12)" ;

	private static final long serialVersionUID = 5632 ;		// 5.6.3.2 (2013/04/12)

	private TagBuffer tag = new TagBuffer( "iTable" ) ;

	/**
	 * Taglibの終了タグが見つかったときに処理する doEndTag() を オーバーライドします。
	 *
	 * @return	後続処理の指示
	 */
	@Override
	public int doEndTag() {
		debugPrint();		// 4.0.0 (2005/02/28)

		jspPrint( tag.makeTag() );

		return(EVAL_PAGE);		// ページの残りを評価する。
	}

	/**
	 * タグリブオブジェクトをリリースします。
	 * キャッシュされて再利用されるので、フィールドの初期設定を行います。
	 *
	 */
	@Override
	protected void release2() {
		super.release2();
		tag = new TagBuffer( "iTable" );
	}

	/**
	 * 【TAG】左右２分割で、固定したいカラム数を指定します(必須)。
	 *
	 * @og.tag
	 * １段組でも２段組でも、固定したいカラム数を指定します。
	 *
	 * @param   fixedCols 固定したいカラム数
	 */
	public void setFixedCols( final String fixedCols ) {
		tag.add( "fixedCols",StringUtil.nval( getRequestParameter( fixedCols ),null ) );
	}

	/**
	 * タグの名称を、返します。
	 * 自分自身のクラス名より、自動的に取り出せないため、このメソッドをオーバーライドします。
	 *
	 * @return  タグの名称
	 */
	@Override
	protected String getTagName() {
		return "iTable" ;
	}

	/**
	 * このオブジェクトの文字列表現を返します。
	 * 基本的にデバッグ目的に使用します。
	 *
	 * @return このクラスの文字列表現
	 */
	@Override
	public String toString() {
		return org.opengion.fukurou.util.ToString.title( this.getClass().getName() )
				.println( "VERSION"		,VERSION	)
				.println( "tag"			,tag.makeTag()	)
				.println( "Other..."	,getAttributes().getAttribute() )
				.fixForm().toString() ;
	}
}
