/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.column;

import org.opengion.hayabusa.db.AbstractRenderer;
import org.opengion.hayabusa.db.CellRenderer;
import org.opengion.hayabusa.db.DBColumn;
import org.opengion.fukurou.util.StringUtil;

/**
 * 特定のHTMLタグのエスケープ文字を元のタグに戻して表示するクラスです。
 *
 * HTMLレンデラーで、HTMLタグ('&lt;' や '&gt;')が('&amp;lt;' や '&amp;gt;' に)変換
 * されますが、この FILTERでは、特定の変換済みタグを元のHTMLに戻す処理を行います。
 * 現時点では、&lt;u&gt; , &lt;/u&gt; , &lt;br /&gt; の３種類です。
 * これは、帳票システムで、データに含まれるHTMLを変換しないと、HTMLのレイアウトが
 * 崩れる為、3.7.0.1 (2005/01/31) で、帳票データのHTMLエスケープ処理を導入しました。
 * この時点で、セル内の改行を表す &lt;br&gt; も、エスケープされる為、3.7.1.1 (2005/05/31)にて
 * &lt;br&gt;のみ、そのまま元に戻す処理が入っています。
 * 今回は、指定のデータに下線を引く &lt;u&gt; タグと、今後もこのような変換対象が現れる
 * 可能性を考慮して、既存のレンデラーに実装しました。
 * 現状の帳票システムでは、エンジンのレンデラー経由で変換され、HTML可されているため、
 * カラムリソース(システムIDがGE)に逆変換したいカラムをこの FILTER レンデラーで
 * 登録すれば、元に戻すことが可能になります。
 *
 * クロスサイトスクリプティング問題に対応するフィールドに対して
 * 定義することにより、エスケープ処理を行います。
 *
 *  カラムの表示に必要な属性は, DBColumn オブジェクト より取り出します。
 * このクラスは、DBColumn オブジェクト毎に１つ作成されます。
 *
 * @og.rev 3.8.1.1 (2005/11/21) 新規追加
 * @og.group データ表示
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class Renderer_FILTER extends AbstractRenderer {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "4.0.0 (2005/08/31)" ;

	private static final CellRenderer dbCell = new Renderer_FILTER();
	private static final String[] KEYS = new String[] { "&lt;u&gt;"	,"&lt;/u&gt;","&lt;br/&gt;" };
	private static final String[] VALS = new String[] { "<u>"		,"</u>"		 ,"<br/>"       };

	/**
	 * 各オブジェクトから自分のインスタンスを返します。
	 * 自分自身をキャッシュするのか、新たに作成するのかは、各サブクラスの実装に
	 * まかされます。
	 *
	 * @param	clm	DBColumnオブジェクト
	 *
	 * @return	CellRendererオブジェクト
	 */
	public CellRenderer newInstance( final DBColumn clm ) {
		return dbCell;
	}

	/**
	 * データの表示用文字列を返します。
	 *
	 * @param   value 入力値
	 *
	 * @return  データの表示用文字列
	 */
	@Override
	public String getValue( final String value ) {
		String val = ( value == null ) ? "" : value;

		for( int i=0; i<KEYS.length; i++ ) {
			val = StringUtil.replace( val,KEYS[i],VALS[i] );
		}

		return val;
	}
}
