/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.fukurou.process;

import org.opengion.fukurou.util.Argument;
import org.opengion.fukurou.util.FileUtil;
import org.opengion.fukurou.util.FileString;
import org.opengion.fukurou.util.Closer ;
import org.opengion.fukurou.util.StringUtil ;
import org.opengion.fukurou.util.LogWriter;

import java.util.Map ;
import java.util.LinkedHashMap ;
import java.util.List ;
import java.util.ArrayList ;

import java.io.File;
import java.io.PrintWriter;
import java.io.BufferedReader;
import java.io.IOException;

/**
 * Process_GrepChange は、上流から受け取った FileLineModelから、語句を
 * 置換する、ChainProcess インターフェースの実装クラスです。
 *
 * Process_Grep との違いは、チェックするファイルのコピーを(キーワードが存在
 * しなくとも)作成することと、検索キーに正規表現が使えない、複数行置き換えが
 * 出来ないことです。
 *
 * keywordFile より、置換する語句を含むキーと値のペアー(タブ区切り)を読取り、
 * 対象とする語句を置換します。
 * keywordFile に、タブが含まれない行や、先頭にタブが存在している場合は、
 * その行を読み飛ばします。また、区切りタブは何個存在しても構いません。
 * ただし、タブで区切った前(キー)と後ろ(値)は、trim() されますので、スペース
 * が前後に存在している場合は、ご注意ください。
 * 置換文字(値)は、\t と \n の特殊文字が使用できます。
 * この GrepChange では、語句に、正規表現は使用できません。正規表現のキーワード
 * や文字列を複数行の文字列と置き換える場合は、Process_Grep を使用してください。
 * このプログラムでは、上流から受け取った FileLineModel のファイルに対して、
 * 置き換えた結果も、同じファイルにセーブします。
 * 元のファイルを保存したい場合は、予めバックアップを取得しておいてください。
 * -inEncode は、入力ファイルのエンコード指定になります。
 * -outEncode は、出力ファイルのエンコードや、キーワードファイルの
 * エンコード指定になります。(keywordFile は、必ず 出力ファイルと同じエンコードです。)
 * これらのエンコードが無指定の場合は、System.getProperty("file.encoding") で
 * 求まる値を使用します。
 *
 * 上流プロセスでは、Name 属性として、『File』を持ち、値は、Fileオブジェクト
 * である、Process_FileSearch を使用するのが、便利です。それ以外のクラスを
 * 使用する場合でも、Name属性と、File オブジェクトを持つ LineModel を受け渡し
 * できれば、使用可能です。
 *
 * 引数文字列中にスペースを含む場合は、ダブルコーテーション("") で括って下さい。
 * 引数文字列の 『=』の前後には、スペースは挟めません。必ず、-key=value の様に
 * 繋げてください。
 *
 *  Process_GrepChange -keyword=検索文字列 -ignoreCase=true -outfile=OUTFILE -encode=UTF-8
 *
 *    -keywordFile=キーワード    ：置換する語句を含むキーと値のペアー(タブ区切り)
 *   [-ignoreCase=大文字小文字 ] ：検索時に大文字小文字を区別しない(true)かどうか(初期値:false[区別する])
 *   [-isChange=置換可否       ] ：置換処理を実施する(true)かどうか(初期値:置換する[true])
 *   [-inEncode=入力エンコード ] ：入力ファイルのエンコードタイプ
 *   [-outEncode=出力エンコード] ：出力ファイルやキーワードファイルのエンコードタイプ
 *   [-display=false|true      ] ：結果を標準出力に表示する(true)かしない(false)か(初期値:false[表示しない])
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class Process_GrepChange extends AbstractProcess implements ChainProcess {
	private String[]	keyword	= null;
	private String[]	change	= null;
	private boolean		ignoreCase	= false;
	private boolean		isChange	= true;		// 5.1.2.0 (2010/01/01) 置換するかどうかを指定可能にする
	private String		inEncode	= null;
	private String		outEncode	= null;
	private boolean		display		= false;	// 表示しない

	private int		inCount		= 0;
	private int		findCount	= 0;
	private int		cngCount	= 0;

	private static final Map<String,String> mustProparty   ;		// ［プロパティ］必須チェック用 Map
	private static final Map<String,String> usableProparty ;		// ［プロパティ］整合性チェック Map

	static {
		mustProparty = new LinkedHashMap<String,String>();
		mustProparty.put( "keywordFile",	"置換する語句を含むキーと値のペアー(タブ区切り)(必須)" );

		usableProparty = new LinkedHashMap<String,String>();
		usableProparty.put( "ignoreCase",	"検索時に大文字小文字を区別しない(true)かどうか。" +
										CR + "(初期値:区別する[false])" );
		usableProparty.put( "isChange",		"置換処理を実施する(true)かどうか" +
										CR + "(初期値:置換する[true])" );
		usableProparty.put( "inEncode",		"入力ファイルのエンコードタイプ" );
		usableProparty.put( "outEncode",	"出力ファイルやキーワードファイルのエンコードタイプ" );
		usableProparty.put( "display",		"結果を標準出力に表示する(true)かしない(false)か" +
										CR + "(初期値:false:表示しない)" );
	}

	/**
	 * デフォルトコンストラクター。
	 * このクラスは、動的作成されます。デフォルトコンストラクターで、
	 * super クラスに対して、必要な初期化を行っておきます。
	 *
	 */
	public Process_GrepChange() {
		super( "org.opengion.fukurou.process.Process_GrepChange",mustProparty,usableProparty );
	}

	/**
	 * プロセスの初期化を行います。初めに一度だけ、呼び出されます。
	 * 初期処理(ファイルオープン、ＤＢオープン等)に使用します。
	 *
	 * @og.rev 5.1.2.0 (2010/01/01) 置換するかどうかを指定可能にする(isChange)属性追加
	 *
	 * @param   paramProcess データベースの接続先情報などを持っているオブジェクト
	 */
	public void init( final ParamProcess paramProcess ) {
		Argument arg = getArgument();

		String keywordFile = arg.getProparty("keywordFile" );
		ignoreCase		= arg.getProparty("ignoreCase",ignoreCase);
		isChange		= arg.getProparty("isChange",isChange);			// 5.1.2.0 (2010/01/01)
		inEncode		= arg.getProparty("inEncode",System.getProperty("file.encoding"));
		outEncode		= arg.getProparty("outEncode",System.getProperty("file.encoding"));
		display			= arg.getProparty("display",display);

		FileString fs = new FileString();
		fs.setFilename( keywordFile );
		fs.setEncode( outEncode );
		String[] lines = fs.getValue( "\n" );
		int len = lines.length;
		if( len == 0 ) {
			String errMsg = "keywordFile の内容が 読み取れませんでした。[" + keywordFile + "]" ;
			throw new RuntimeException( errMsg );
		}

		println( "keywordFile を、" + len + "件読み取りました。" );
		List<String> keyList = new ArrayList<String>( len );
		List<String> cngList = new ArrayList<String>( len );

		for( int i=0; i<len; i++ ) {
	//		String line = lines[i].trim();
			String line = lines[i];
			int indx = line.indexOf( '\t' );
			if( indx <= 0 ) { continue ; }	// TAB が先頭や、存在しない行は読み飛ばす。
			keyList.add( line.substring( 0,indx ).trim() );
			String cng = line.substring( indx+1 ).trim();
			cng = StringUtil.replace( cng,"\\n",CR );
			cng = StringUtil.replace( cng,"\\t","\t" );
			cngList.add( cng );
		}
		keyword	= keyList.toArray( new String[keyList.size()] );
		change	= cngList.toArray( new String[cngList.size()] );
	}

	/**
	 * プロセスの終了を行います。最後に一度だけ、呼び出されます。
	 * 終了処理(ファイルクローズ、ＤＢクローズ等)に使用します。
	 *
	 * @param   isOK トータルで、OKだったかどうか[true:成功/false:失敗]
	 */
	public void end( final boolean isOK ) {
		// ここでは処理を行いません。
	}

	/**
	 * 引数の LineModel を処理するメソッドです。
	 * 変換処理後の LineModel を返します。
	 * 後続処理を行わない場合(データのフィルタリングを行う場合)は、
	 * null データを返します。つまり、null データは、後続処理を行わない
	 * フラグの代わりにも使用しています。
	 * なお、変換処理後の LineModel と、オリジナルの LineModel が、
	 * 同一か、コピー(クローン)かは、各処理メソッド内で決めています。
	 * ドキュメントに明記されていない場合は、副作用が問題になる場合は、
	 * 各処理ごとに自分でコピー(クローン)して下さい。
	 *
	 * @og.rev 4.0.0.0 (2007/11/28) メソッドの戻り値をチェックします。
	 * @og.rev 5.1.2.0 (2010/01/01) 置換するかどうかを指定可能にする(isChange)属性追加
	 * @og.rev 5.7.2.2 (2014/01/24) エラー時にデータも出力します。
	 *
	 * @param	data	オリジナルのLineModel
	 *
	 * @return	処理変換後のLineModel
	 */
	public LineModel action( final LineModel data ) {
		inCount++ ;
		final FileLineModel fileData ;
		if( data instanceof FileLineModel ) {
			fileData = (FileLineModel)data ;
		}
		else {
			String errMsg = "データが FileLineModel オブジェクトではありません。" + CR ;
			throw new RuntimeException( errMsg );
		}

		File org = fileData.getFile() ;
		String orgName = org.getPath();
		if( ! org.isFile() ) { return data; }

		BufferedReader reader = FileUtil.getBufferedReader( org,inEncode );
//		File		tempFile  = new File( org.getPath() + "_temp" );
//		PrintWriter	tempWrt   = FileUtil.getPrintWriter( tempFile,outEncode );
		File		tempFile  = null;
		PrintWriter	tempWrt   = null;

		// 5.1.2.0 (2010/01/01) 置換する場合の前処理
		if( isChange ) {
			tempFile  = new File( orgName + "_temp" );
			tempWrt   = FileUtil.getPrintWriter( tempFile,outEncode );
		}

		boolean nextFlag  = false;

		try {
			String line ;
			int    lineNo = 0;
			while((line = reader.readLine()) != null) {
				lineNo++ ;
				StringBuilder buf = new StringBuilder( line );
//				boolean foundFlag = false;		// 行単位に初期化する。
				for( int i=0; i<keyword.length; i++ ) {
					int indx = buf.indexOf( keyword[i] );
					// 置換対象発見。行出力用に見つかれば、true にする。
					if( indx >= 0 ) {
//						foundFlag = true;
						nextFlag  = true;		// １度でも見つかれば、true にセット
						if( display ) { println( orgName + ":" + lineNo + ":" + keyword[i] + ":" + line ); }
					}
					// 置換対象が見つかっても、isChange=true でなければ、置換処理は行わない。
					if( isChange ) {
						while( indx >= 0 ) {
							buf.replace( indx,indx+keyword[i].length(),change[i] );
							indx = buf.indexOf( keyword[i],indx+change[i].length() );
	//						nextFlag = true;			// キーワードが存在したファイル。
							cngCount++ ;
							findCount++ ;
						}
					}
				}
				// 5.1.2.0 (2010/01/01) 置換する場合の処理
				if( isChange ) {
					tempWrt.println( buf.toString() );
				}
			}
		}
		catch ( IOException ex ) {
			String errMsg = "処理中にエラーが発生しました。[" + data.getRowNo() + "]件目" + CR
//						+ data.toString() ;
						+	"data=[" + data.dataLine() + "]" + CR ;		// 5.7.2.2 (2014/01/24) エラー時にデータも出力します。
			throw new RuntimeException( errMsg,ex );
		}
		finally {
			Closer.ioClose( reader );
			Closer.ioClose( tempWrt );
		}

		// 5.1.2.0 (2010/01/01) 置換する場合の処理
		if( isChange ) {
			if( nextFlag ) {
				if( !org.delete() ) {
					String errMsg = "所定のファイルを削除できませんでした。[" + org + "]" + CR
							+	"data=[" + data.dataLine() + "]" + CR ;		// 5.7.2.2 (2014/01/24) エラー時にデータも出力します。
					throw new RuntimeException( errMsg );
				}
				if( !tempFile.renameTo( org ) ) {
					String errMsg = "所定のファイルをリネームできませんでした。[" + tempFile + "]" + CR
							+	"data=[" + data.dataLine() + "]" + CR ;		// 5.7.2.2 (2014/01/24) エラー時にデータも出力します。
					throw new RuntimeException( errMsg );
				}
			}
			else {
				if( !tempFile.delete() ) {
					String errMsg = "所定のファイルを削除できませんでした。[" + tempFile + "]" + CR
							+	"data=[" + data.dataLine() + "]" + CR ;		// 5.7.2.2 (2014/01/24) エラー時にデータも出力します。
					throw new RuntimeException( errMsg );
				}
			}
		}

		return (nextFlag) ? data : null ;
	}

	/**
	 * プロセスの処理結果のレポート表現を返します。
	 * 処理プログラム名、入力件数、出力件数などの情報です。
	 * この文字列をそのまま、標準出力に出すことで、結果レポートと出来るような
	 * 形式で出してください。
	 *
	 * @return   処理結果のレポート
	 */
	public String report() {
		String report = "[" + getClass().getName() + "]" + CR
				+ TAB + "Search File Count : " + inCount    + CR
				+ TAB + "Key Find    Count : " + findCount  + CR
				+ TAB + "Key Change  Count : " + cngCount ;

		return report ;
	}

	/**
	 * このクラスの使用方法を返します。
	 *
	 * @return	このクラスの使用方法
	 */
	public String usage() {
		StringBuilder buf = new StringBuilder();

		buf.append( "Process_GrepChange は、上流から受け取った FileLineModelから、語句を"			).append( CR );
		buf.append( "置換する、ChainProcess インターフェースの実装クラスです。"						).append( CR );
		buf.append( "Process_Grep との違いは、チェックするファイルのコピーを(キーワードが存在"		).append( CR );
		buf.append( "しなくとも)作成することと、検索キーに正規表現が使えない、複数行置き換えが"		).append( CR );
		buf.append( "出来ないことです。"															).append( CR );
		buf.append( CR );
		buf.append( "keywordFile より、置換する語句を含むキーと値のペアー(タブ区切り)を読取り、"	).append( CR );
		buf.append( "対象とする語句を置換します。"													).append( CR );
		buf.append( "keywordFile に、タブが含まれない行や、先頭にタブが存在している場合は、"		).append( CR );
		buf.append( "その行を読み飛ばします。また、区切りタブは何個存在しても構いません。"			).append( CR );
		buf.append( "ただし、タブで区切った前(キー)と後ろ(値)は、trim() されますので、スペース"		).append( CR );
		buf.append( "が前後に存在している場合は、ご注意ください。"									).append( CR );
		buf.append( "置換文字(値)は、\t と \n の特殊文字が使用できます。" 							).append( CR );
		buf.append( "この GrepChange では、語句に、正規表現は使用できません。正規表現のキーワード"	).append( CR );
		buf.append( "や文字列を複数行の文字列と置き換える場合は、Process_Grep を使用して下さい。"	).append( CR );
		buf.append( "このプログラムでは、上流から受け取った FileLineModel のファイルに対して、"		).append( CR );
		buf.append( "置き換えた結果も、同じファイルにセーブします。"								).append( CR );
		buf.append( "元のファイルを保存したい場合は、予めバックアップを取得しておいてください。"	).append( CR );
		buf.append( "-inEncode は、入力ファイルのエンコード指定になります。" 						).append( CR );
		buf.append( "-outEncode は、出力ファイルのエンコードや、キーワードファイルのエンコード" 	).append( CR );
		buf.append( "指定になります。(keywordFile は、必ず 出力ファイルと同じエンコードです。)" 	).append( CR );
		buf.append( "これらのエンコードが無指定の場合は、System.getProperty(\"file.encoding\") " 	).append( CR );
		buf.append( "で求まる値を使用します。" 														).append( CR );
		buf.append( CR );
		buf.append( "上流プロセスでは、Name 属性として、『File』を持ち、値は、Fileオブジェクト"		).append( CR );
		buf.append( "である、Process_FileSearch を使用するのが、便利です。それ以外のクラスを"		).append( CR );
		buf.append( "使用する場合でも、Name属性と、File オブジェクトを持つ LineModel を受け渡し"	).append( CR );
		buf.append( "できれば、使用可能です。"														).append( CR );
		buf.append( CR );
		buf.append( "引数文字列中に空白を含む場合は、ダブルコーテーション(\"\") で括って下さい。"	).append( CR );
		buf.append( "引数文字列の 『=』の前後には、空白は挟めません。必ず、-key=value の様に"		).append( CR );
		buf.append( "繋げてください。"																).append( CR );
		buf.append( CR ).append( CR );

		buf.append( getArgument().usage() ).append( CR );

		return buf.toString();
	}

	/**
	 * このクラスは、main メソッドから実行できません。
	 *
	 * @param	args	コマンド引数配列
	 */
	public static void main( final String[] args ) {
		LogWriter.log( new Process_GrepChange().usage() );
	}
}
