/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import org.opengion.fukurou.db.ConnectionFactory;
import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;
import static org.opengion.fukurou.util.StringUtil.nval ;

import java.util.ArrayList;
import java.util.Locale ;

import java.sql.ResultSetMetaData ;
import java.sql.DatabaseMetaData ;
import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Types;

/**
 * DatabaseMetaData の情報を検索するタグです。
 *
 * データベースに関する包括的な情報を提供する、DatabaseMetaData の内容を
 * 表示する、タグです。テスト的に使用します。
 *
 * @og.formSample
 * ●形式：
 *  &lt;og:databaseMetaData
 *      dbid            = &quot;{&#064;DBID}&quot;
 *      catalog         = &quot;{&#064;catalog}&quot;
 *      schema          = &quot;{&#064;schema}&quot;
 *      tableName       = &quot;{&#064;tableName}&quot;
 *      columnName      = &quot;{&#064;columnName}&quot;
 *      typeName        = &quot;{&#064;typeName}&quot;
 *      attributeName   = &quot;{&#064;attriName}&quot;
 *      procedureName   = &quot;{&#064;procName}&quot;
 *      maxRowCount     = &quot;{&#064;maxRowCount}&quot;
 *      likeKey         = &quot;{&#064;likeKey}&quot;
 *  /&gt;
 *
 * ●body：なし
 *
 * ●Tag定義：
 *   &lt;og:databaseMetaData
 *       dbid               【TAG】(通常は使いません)Queryオブジェクトを作成する時のDB接続IDを指定します
 *       catalog            【TAG】(通常は使いません)Queryオブジェクトを作成する時のカタログを指定します
 *       schema             【TAG】(通常は使いません)Queryオブジェクトを作成する時のスキーマを指定します
 *       attributeName      【TAG】(通常は使いません)Queryオブジェクトを作成する時の属性名を指定します
 *       columnName         【TAG】(通常は使いません)Queryオブジェクトを作成する時のカラム名を指定します
 *       procedureName      【TAG】(通常は使いません)Queryオブジェクトを作成する時のプロダクト名を指定します
 *       tableName          【TAG】(通常は使いません)Queryオブジェクトを作成する時のテーブル名を指定します
 *       typeName           【TAG】(通常は使いません)Queryオブジェクトを作成する時のタイプ名を指定します
 *       maxRowCount        【TAG】maxRowCount ResultSetで検索するパラメータの最大検索件数を指定します(初期値:20)
 *       likeKey            【TAG】実行するメソッドを選別するための部分一致するキーを指定します
 *       debug              【TAG】デバッグ情報を出力するかどうか[true/false]を指定します(初期値:false)
 *   /&gt;
 *
 * ●使用例
 *       &lt;og:databaseMetaData /&gt;
 *
 * @og.group ＤＢ検索
 *
 * @version  0.9.0	2000/10/17
 * @author	 Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class DatabaseMetaDataTag extends CommonTagSupport {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "5.5.8.5 (2012/11/27)" ;

	private static final long serialVersionUID = 558520121127L ;

	// 4.0.0.0 (2007/10/10) dbid の初期値を、"DEFAULT" から null に変更
//	private String	dbid			= "DEFAULT";
	private String	dbid			= null ;
	private String	catalog			= null ;
	private String	schema			= null ;
	private String	tableName		= null ;
	private String	columnName		= null ;
	private String	typeName		= null ;
	private String	attributeName	= null ;
	private String	procedureName	= null ;
	private int		rowNo			= 0;
	private int		maxRowCount		= 20;		// 5.5.3.2 (2012/06/08) 最大検索件数を、指定できるようにする。
	private String	likeKey			= null ;	// 5.5.3.2 (2012/06/08) 特定の処理だけを実行できるように、あいまい検索

	/**
	 * Taglibの終了タグが見つかったときに処理する doEndTag() を オーバーライドします。
	 *
	 * @og.rev 3.8.7.0 (2006/12/15) アクセスログ取得の為,ApplicationInfoオブジェクトを設定
	 * @og.rev 4.0.0.1 (2007/12/03) try ～ catch ～ finally をきちんと行う。
	 *
	 * @return	後続処理の指示(EVAL_PAGE)
	 */
	@Override
	public int doEndTag() {
		debugPrint();		// 4.0.0 (2005/02/28)
		Connection conn = null;
		boolean errFlag = true;
		try {
			conn = ConnectionFactory.connection( dbid,getApplicationInfo() );	// 3.8.7.0 (2006/12/15)
			DatabaseMetaData metaData = conn.getMetaData() ;

			callPrintMessage( metaData );
			errFlag = false;		// エラーではない
		}
		catch (Throwable ex) {
			String errMsg = "データベース処理を実行できませんでした。"
						+ ex.getMessage()  + HybsSystem.CR ;
//			errMsg = errMsg + HybsSystem.CR ;
//			errMsg = errMsg + ex.getMessage() ;

//			ConnectionFactory.remove( conn,dbid );
//			conn = null;
			throw new HybsSystemException( errMsg,ex );		// 3.5.5.4 (2004/04/15) 引数の並び順変更
		}
		finally {
			// 以下のコーディングの見直しを考える必要有り。
			// Connection はプールしている為、close() しても再利用されるだけ。
//			ConnectionFactory.close( conn,dbid );
			if( errFlag ) { ConnectionFactory.remove( conn,dbid ); }	// 削除
			else {			ConnectionFactory.close( conn,dbid );  }	// 返却
//			conn = null;
		}

		jspPrint( "処理が完了しました。" + HybsSystem.CR );		// 5.5.3.2 (2012/06/08) 終了メッセージ追加
		return(EVAL_PAGE);
	}

	/**
	 * タグリブオブジェクトをリリースします。
	 * キャッシュされて再利用されるので、フィールドの初期設定を行います。
	 *
	 * @og.rev 4.0.0.0 (2007/10/10) dbid の初期値を、"DEFAULT" から null に変更
	 * @og.rev 5.5.3.2 (2012/06/08) maxRowCount 最大検索件数を、指定できるようにする。
	 * @og.rev 5.5.3.2 (2012/06/08) likeKey 特定の処理だけを実行できるように、あいまい検索
	 */
	@Override
	protected void release2() {
		super.release2();
//		dbid			= "DEFAULT";
		dbid			= null ;
		catalog			= null ;
		schema			= null ;
		tableName		= null ;
		columnName		= null ;
		typeName		= null ;
		procedureName	= null ;
		attributeName	= null ;
		rowNo			= 0;
		maxRowCount		= 20;		// 5.5.3.2 (2012/06/08) 最大検索件数を、指定できるようにする。
		likeKey			= null;		// 5.5.3.2 (2012/06/08) 特定の処理だけを実行できるように、あいまい検索
	}

	/**
	 * 画面にテーブルタグの(TR)として、情報を出力します。
	 *
	 * 第一引数が、boolean の場合のオーバーロードメソッドです。
	 *
	 * @param	in	第一引数(boolean)
	 * @param	lbl	ラベル
	 * @param	msg	メッセージ
	 */
	private void printMessage( final boolean in, final String lbl, final String msg ) {
		String str = "<tr class=\"row_" + (rowNo++ % 2) + "\"><td>"
				 + rowNo + "</td><td>"			// 5.5.3.2 (2012/06/08) 行番号を表示
				 + lbl + "</td><td>" + in + "</td><td>" + msg + "</td></tr>" + HybsSystem.CR ;
		jspPrint( str );
	}

	/**
	 * 画面にテーブルタグの(TR)として、情報を出力します。
	 *
	 * 第一引数が、int の場合のオーバーロードメソッドです。
	 *
	 * @param	in	第一引数(int)
	 * @param	lbl	ラベル
	 * @param	msg	メッセージ
	 */
	private void printMessage( final int in, final String lbl, final String msg ) {
		String str = "<tr class=\"row_" + (rowNo++ % 2) + "\"><td>"
				 + rowNo + "</td><td>"			// 5.5.3.2 (2012/06/08) 行番号を表示
				 + lbl + "</td><td>" + in + "</td><td>" + msg + "</td></tr>" + HybsSystem.CR ;
		jspPrint( str );
	}

	/**
	 * 画面にテーブルタグの(TR)として、情報を出力します。
	 *
	 * 第一引数が、String の場合のオーバーロードメソッドです。
	 *
	 * @param	in	第一引数(String)
	 * @param	lbl	ラベル
	 * @param	msg	メッセージ
	 */
	private void printMessage( final String in, final String lbl, final String msg ) {
		String str = "<tr class=\"row_" + (rowNo++ % 2) + "\"><td>"
				 + rowNo + "</td><td>"			// 5.5.3.2 (2012/06/08) 行番号を表示
				 + lbl + "</td><td>" + in + "</td><td>" + msg + "</td></tr>" + HybsSystem.CR ;
		jspPrint( str );
	}

	/**
	 * 画面にテーブルとして、情報を出力します。
	 *
	 * 第一引数が、ResultSet の場合のオーバーロードメソッドです。
	 *
	 * @param	resultSet	ResultSetオブジェクト
	 * @param	msg			メッセージ
	 */
	private void printMessage( final ResultSet resultSet, final String msg ) {
//		jspPrint( msg + HybsSystem.BR );
		jspPrint( makeTableTag( resultToString( resultSet ) ) );
	}

	/**
	 * 画面にテーブルとして、情報を出力します。
	 *
	 * 第一引数が、存在しない場合のオーバーロードメソッドです。
	 *
	 * @param	lbl	ラベル
	 * @param	msg	メッセージ
	 */
	private void printMessage( final String lbl, final String msg ) {
//		jspPrint( msg + HybsSystem.BR );
		String str = "<table><tr><td>" + lbl + "</td></tr></table>" + HybsSystem.CR ;
		jspPrint( str );
	}

	/**
	 * 指定の文字列配列をテーブル上に表示する為の、テーブルを作成します。
	 *
	 * @og.rev 5.5.3.2 (2012/06/08) 行番号を表示するように修正
	 *
	 * @param	data	文字列配列
	 *
	 * @return	文字列配列を表示する為のHTMLテーブル書式
	 */
	private String makeTableTag( final String[][] data ) {
		if( data == null || data.length == 0 ) { return "" ; }

		StringBuilder buf = new StringBuilder( HybsSystem.BUFFER_LARGE );
		buf.append( "<table>" ).append( HybsSystem.CR );

		int maxRow = data.length ;
		int maxClm = data[0].length ;

		buf.append( "<tr class=\"row_h\">" );
		buf.append( "<th>No</th>" );					// 5.5.3.2 (2012/06/08) 行番号を表示
		for( int clm=0; clm<maxClm; clm++ ) {
			buf.append( "<th>" ).append( data[0][clm] ).append( "</th>" );
		}
		buf.append( "</tr>" ).append( HybsSystem.CR );
		for( int row=1; row<maxRow; row++ ) {
			buf.append( "<tr class=\"row_" ).append( (row-1) % 2 ).append( "\">" );
			buf.append( "<td>" ).append( row ).append( "</td>" );			// 5.5.3.2 (2012/06/08) 行番号を表示
			for( int clm=0; clm<maxClm; clm++ ) {
				buf.append( "<td>" ).append( data[row][clm] ).append( "</td>" );
			}
			buf.append( "</tr>" ).append( HybsSystem.CR );
		}
		buf.append( "</table>" ).append( HybsSystem.CR );

		return buf.toString();
	}

	/**
	 * ステートメントと引数により、Queryを実行します。
	 * これは、Prepared クエリーの検索のみ実行します。
	 * 結果は,すべて文字列に変換されて格納されます。
	 *
	 * @og.rev 5.5.3.2 (2012/06/08) 最大検索件数を、指定できるようにする。
	 *
	 * @param   resultSet ResultSetオブジェクト
	 *
	 * @return  ResultSetの検索結果配列
	 */
	private String[][] resultToString( final ResultSet resultSet ) {
		ArrayList<String[]> data = new ArrayList<String[]>();
		try {
			ResultSetMetaData metaData  = resultSet.getMetaData();
			int numberOfColumns =  metaData.getColumnCount();

			String[] columnNames = new String[numberOfColumns];
			for(int column = 0; column < numberOfColumns; column++) {
				columnNames[column]	 = (metaData.getColumnLabel(column+1)).toUpperCase(Locale.JAPAN) ;
			}
			data.add( columnNames );

			// とりあえず MAX １０件とする。
			int maxcnt = 0 ;
	 		// 5.5.3.2 (2012/06/08) 最大検索件数を、指定できるようにする。
			while( resultSet.next() && ( maxcnt++ < maxRowCount ) ) {
				String[] columnValues = new String[numberOfColumns];
				for(int i = 1; i <= numberOfColumns; i++) {
					Object obj = resultSet.getObject(i);
					if( obj == null ) {
						columnValues[i-1] = "";
					}
					else {
						columnValues[i-1] = String.valueOf( obj );
					}
				}
				data.add( columnValues );
			}
		}
		catch (SQLException ex) {
			String errMsg = "処理結果を実行できませんでした。"
						+ ex.getMessage();		// 5.1.8.0 (2010/07/01) errMsg 修正
			throw new HybsSystemException( errMsg,ex );		// 3.5.5.4 (2004/04/15) 引数の並び順変更
		}

		int size = data.size();
		String[][] rtn = new String[size][];
		for( int i=0; i<size; i++ ) {
			rtn[i] = data.get(i);
		}

		return rtn;
	}

	/**
	 * 【TAG】(通常は使いません)Queryオブジェクトを作成する時のDB接続IDを指定します。
	 *
	 * @og.tag Queryオブジェクトを作成する時のDB接続IDを指定します。
	 *
	 * @param	id データベース接続ID
	 */
	public void setDbid( final String id ) {
		dbid = nval( getRequestParameter( id ),dbid );
	}

	/**
	 * 【TAG】(通常は使いません)Queryオブジェクトを作成する時のカタログを指定します。
	 *
	 * @og.tag Queryオブジェクトを作成する時のカタログを指定します。
	 *
	 * @param	val	カタログ情報
	 */
	public void setCatalog( final String val ) {
		catalog = nval( getRequestParameter( val ),catalog );
	}

	/**
	 * 【TAG】(通常は使いません)Queryオブジェクトを作成する時のスキーマを指定します。
	 *
	 * @og.tag Queryオブジェクトを作成する時のスキーマを指定します。
	 *
	 * @param	val	スキーマ情報
	 */
	public void setSchema( final String val ) {
		schema = nval( getRequestParameter( val ),schema );
	}

	/**
	 * 【TAG】(通常は使いません)Queryオブジェクトを作成する時のテーブル名を指定します。
	 *
	 * @og.tag Queryオブジェクトを作成する時のテーブル名を指定します。
	 *
	 * @param	val	テーブル名情報
	 */
	public void setTableName( final String val ) {
		tableName = nval( getRequestParameter( val ),tableName );
	}

	/**
	 * 【TAG】(通常は使いません)Queryオブジェクトを作成する時のカラム名を指定します。
	 *
	 * @og.tag Queryオブジェクトを作成する時のカラム名を指定します。
	 *
	 * @param	val	カラム名情報
	 */
	public void setColumnName( final String val ) {
		columnName = nval( getRequestParameter( val ),columnName );
	}

	/**
	 * 【TAG】(通常は使いません)Queryオブジェクトを作成する時のタイプ名を指定します。
	 *
	 * @og.tag Queryオブジェクトを作成する時のタイプ名を指定します。
	 *
	 * @param	val	タイプ名情報
	 */
	public void setTypeName( final String val ) {
		typeName = nval( getRequestParameter( val ),typeName );
	}

	/**
	 * 【TAG】(通常は使いません)Queryオブジェクトを作成する時のプロダクト名を指定します。
	 *
	 * @og.tag Queryオブジェクトを作成する時のプロダクト名を指定します。
	 *
	 * @param	val	プロダクト名情報
	 */
	public void setProcedureName( final String val ) {
		procedureName = nval( getRequestParameter( val ),procedureName );
	}

	/**
	 * 【TAG】(通常は使いません)Queryオブジェクトを作成する時の属性名を指定します。
	 *
	 * @og.tag Queryオブジェクトを作成する時の属性名を指定します。
	 *
	 * @param	val	属性名情報
	 */
	public void setAttributeName( final String val ) {
		attributeName = nval( getRequestParameter( val ),attributeName );
	}

	/**
	 * 【TAG】maxRowCount ResultSetで検索するパラメータの最大検索件数を指定します(初期値:20)。
	 *
	 * @og.tag
	 * 大量に検索されて、処理速度が低下するのを避けるため、最大件数を指定します。
	 * 他のタグの maxRowCount の初期値設定(初期値:DB_MAX_ROW_COUNT)は利用せず、独自に設定値を持っています。
	 * 0は、無制限とします。
	 *
	 * @og.rev 5.5.3.2 (2012/06/08) likeKey 特定の処理だけを実行できるように、あいまい検索
	 * @og.rev 5.5.8.5 (2012/11/27) 0を無制限として処理します。
	 *
	 * @param	maxSize	ResultSetで検索する最大検索件数
	 */
	public void setMaxRowCount( final String maxSize ) {
		maxRowCount = nval( getRequestParameter( maxSize ),maxRowCount );
		if( maxRowCount == 0 ) { maxRowCount = Integer.MAX_VALUE ; }			// 5.5.8.5 (2012/11/27)
	}

	/**
	 * 【TAG】実行するメソッドを選別するための部分一致するキーを指定します。
	 *
	 * @og.tag
	 * DatabaseMetaData で取得できる各種情報をアクセスするにあたり、指定のキーで部分一致する
	 * メソッドだけ処理するように指定します。
	 * メソッドの部分一致で判断します。
	 * 処理は、大文字、小文字を認識したうえで、部分一致で判定します。
	 *
	 * @og.rev 5.5.3.2 (2012/06/08) likeKey 特定の処理だけを実行できるように、あいまい検索
	 *
	 * @param	key	実行するメソッドを選別するためのキー
	 */
	public void setLikeKey( final String key ) {
		likeKey = nval( getRequestParameter( key ),likeKey );
	}

	/**
	 * DatabaseMetaData のフィールドで定義されている値に対応する メッセージを返します。
	 *
	 * @param	code short のコード番号
	 */
//	private String dbMessage( final short code ) {
//		String msg = null ;
//		switch( code ) {
//			case DatabaseMetaData.attributeNoNulls :
//					msg = "NULL 値が許可されない可能性があります。" ;
//					break ;
//			case DatabaseMetaData.attributeNullable :
//					msg = "NULL 値が必ず許可されます。" ;
//					break ;
//			case DatabaseMetaData.attributeNullableUnknown :
//					msg = "NULL 値が許可されるかどうかが不明です。" ;
//					break ;
//			case DatabaseMetaData.tableIndexClustered :
//					msg = "このテーブルインデックスがクラスタ化されたインデックスです。" ;
//					break ;
//			case DatabaseMetaData.tableIndexHashed :
//					msg = "このテーブルインデックスがハッシュ化されたインデックスです。" ;
//					break ;
//			case DatabaseMetaData.tableIndexOther :
//					msg = "このテーブルインデックスがクラスタ化されたインデックスでも、ハッシュ化されたインデックスでも、テーブルの統計情報でもありません。" ;
//					break ;
//			case DatabaseMetaData.tableIndexStatistic :
//					msg = "この列がテーブルのインデックスの記述と連動して返されるテーブルの統計情報を格納します。" ;
//					break ;
//			default :
//					msg = "対象のメッセージがありません。" ;
//					break ;
//		}
//
//		return msg ;
//	}

	/**
	 * DatabaseMetaData のフィールドで定義されている値に対応する メッセージを返します。
	 *
	 * @param	code short のコード番号
	 */
//	private String dbMessage( final int code ) {
//		String msg = null ;
//		switch( code ) {
//			case DatabaseMetaData.bestRowNotPseudo :
//					msg = "最適な行識別子が擬似列ではありません。" ;
//					break ;
//			case DatabaseMetaData.bestRowPseudo :
//					msg = "最適な行識別子が擬似列です。" ;
//					break ;
//			case DatabaseMetaData.bestRowSession :
//					msg = "最適な行識別子のスコープが現在のセッションの残りの部分です。" ;
//					break ;
//			case DatabaseMetaData.bestRowTemporary :
//					msg = "最適な行識別子のスコープが一時的 (行が使用されている間だけ) です。" ;
//					break ;
//			case DatabaseMetaData.bestRowTransaction :
//					msg = "最適な行識別子のスコープが現在のトランザクションの残りの部分です。" ;
//					break ;
//			case DatabaseMetaData.bestRowUnknown :
//					msg = "最適な行識別子が擬似列である場合とそうでない場合があります。" ;
//					break ;
//			case DatabaseMetaData.columnNoNulls :
//					msg = "列が NULL 値を許可しない場合があります。" ;
//					break ;
//			case DatabaseMetaData.columnNullable :
//					msg = "列が必ず NULL 値を許可されます。" ;
//					break ;
//			case DatabaseMetaData.columnNullableUnknown :
//					msg = "列が NULL 値を許可するかどうか不明です。" ;
//					break ;
//			case DatabaseMetaData.importedKeyCascade :
//					msg = "列 UPDATE_RULE の場合は、主キーが更新されるときに、外部キー (インポートされたキー) がそれに合致するように変更されます。" ;
//					break ;
//			case DatabaseMetaData.importedKeyInitiallyDeferred :
//					msg = "遅延可能性を示します。" ;
//					break ;
//			case DatabaseMetaData.importedKeyInitiallyImmediate :
//					msg = "遅延可能性を示します。" ;
//					break ;
//			case DatabaseMetaData.importedKeyNoAction :
//					msg = "列 UPDATE_RULE および DELETE_RULE の場合は、主キーがインポートされている場合、それが更新および削除ができません。" ;
//					break ;
//			case DatabaseMetaData.importedKeyNotDeferrable :
//					msg = "遅延可能性を示します。" ;
//					break ;
//			case DatabaseMetaData.importedKeyRestrict :
//					msg = "列 UPDATE_RULE の場合は、別のテーブルによって外部キーとしてインポートされている場合に、主キーは更新できません。" ;
//					break ;
//			case DatabaseMetaData.importedKeySetDefault :
//					msg = "列 UPDATE_RULE および DELETE_RULE の場合は、主キーが更新または削除された場合に、外部キー (インポートされたキー) がデフォルト値に変更されます。" ;
//					break ;
//			case DatabaseMetaData.importedKeySetNull :
//					msg = "列 UPDATE_RULE および DELETE_RULE の場合、主キーが更新または削除されたときに、外部キー (インポートされたキー) が NULL に変更されます。" ;
//					break ;
//			case DatabaseMetaData.procedureColumnIn :
//					msg = "列が IN パラメータを格納されます。" ;
//					break ;
//			case DatabaseMetaData.procedureColumnInOut :
//					msg = "列が INOUT パラメータを格納されます。" ;
//					break ;
//			case DatabaseMetaData.procedureColumnOut :
//					msg = "列が OUT パラメータを格納されます。" ;
//					break ;
//			case DatabaseMetaData.procedureColumnResult :
//					msg = "列が結果を格納されます。" ;
//					break ;
//			case DatabaseMetaData.procedureColumnReturn :
//					msg = "列が戻り値を格納されます。" ;
//					break ;
//			case DatabaseMetaData.procedureColumnUnknown :
//					msg = "列の型が不明です。" ;
//					break ;
//			case DatabaseMetaData.procedureNoNulls :
//					msg = "NULL 値が許可されません。" ;
//					break ;
//			case DatabaseMetaData.procedureNoResult :
//					msg = "プロシージャが結果を返しません。" ;
//					break ;
//			case DatabaseMetaData.procedureNullable :
//					msg = "NULL 値が許可されます。" ;
//					break ;
//			case DatabaseMetaData.procedureNullableUnknown :
//					msg = "NULL 値が許可されるかどうかが不明です。" ;
//					break ;
//			case DatabaseMetaData.procedureResultUnknown :
//					msg = "プロシージャが結果を返すかどうかが不明です。" ;
//					break ;
//			case DatabaseMetaData.procedureReturnsResult :
//					msg = "プロシージャが結果を返すことを示します。" ;
//					break ;
//			case DatabaseMetaData.sqlStateSQL99 :
//					msg = "値が SQL99 SQLSTATE 値です。" ;
//					break ;
//			case DatabaseMetaData.sqlStateXOpen :
//					msg = "値が X/Open (現在は Open Group) の SQL CLI SQLSTATE 値です。" ;
//					break ;
//			case DatabaseMetaData.typeNoNulls :
//					msg = "このデータの型で NULL 値が許可されません。" ;
//					break ;
//			case DatabaseMetaData.typeNullable :
//					msg = "このデータの型で NULL 値が許可されます。" ;
//					break ;
//			case DatabaseMetaData.typeNullableUnknown :
//					msg = "このデータの型で NULL 値が許可されるかどうかが不明です。" ;
//					break ;
//			case DatabaseMetaData.typePredBasic :
//					msg = "WHERE . . .LIKE を除くすべての WHERE 検索の節がこのデータの型に基づくことができます。" ;
//					break ;
//			case DatabaseMetaData.typePredChar :
//					msg = "この型に基づくことができる唯一の WHERE 検索の節が WHERE . . .LIKE です。" ;
//					break ;
//			case DatabaseMetaData.typePredNone :
//					msg = "この型で WHERE 検索の節がサポートされていません。" ;
//					break ;
//			case DatabaseMetaData.typeSearchable :
//					msg = "すべての WHERE 検索の節がこの型に基づくことができます。" ;
//					break ;
//			case DatabaseMetaData.versionColumnNotPseudo :
//					msg = "このバージョンの列が擬似列でありません。" ;
//					break ;
//			case DatabaseMetaData.versionColumnPseudo :
//					msg = "このバージョンの列が擬似列です。" ;
//					break ;
//			case DatabaseMetaData.versionColumnUnknown :
//					msg = "このバージョンの列が擬似列である場合とそうでない場合があります。" ;
//					break ;
//			default :
//					msg = "対象のメッセージがありません。" ;
//					break ;
//		}
//
//		return msg ;
//	}

	/**
	 * 各メソッド情報を画面に出力していきます。
	 *
	 * @og.rev 5.5.3.2 (2012/06/08) JDk1.6 用のメソッドを追加します。
	 *
	 * @param metaData DatabaseMetaDataオブジェクト
	 * @throws Throwable
	 */
	private void callPrintMessage( final DatabaseMetaData metaData ) throws Throwable {
		// 単体のリターン値を持つ メタデータ

//		int type = ResultSet.TYPE_FORWARD_ONLY ;
//		int concurrency = ResultSet.CONCUR_READ_ONLY;
//		int holdability = ResultSet.CLOSE_CURSORS_AT_COMMIT;

		Integer type = ResultSet.TYPE_FORWARD_ONLY ;
		Integer concurrency = ResultSet.CONCUR_READ_ONLY;
		Integer holdability = ResultSet.CLOSE_CURSORS_AT_COMMIT;

		jspPrint( "<table>" );
		jspPrint( "<tr class=\"row_h\"><th>No</th><th>キー</th><th>値</th><th>説明</th></tr>" );

		// 5.5.3.2 (2012/06/08) 個別に呼び出す箇所を、リフレクションに変更。
		for( int i=0; i<METHOD_LIST.length; i++ ) {
			MetaDataInvoke inv = METHOD_LIST[i];
			if( inv.isUseMethod( likeKey ) && inv.getType() < 10 ) {
				Object obj = null;
				switch( inv.getType() ) {
					case 0: obj = inv.invokeA( metaData ); break;
					case 1: obj = inv.invokeA( metaData,type ); break;
					case 2: obj = inv.invokeA( metaData,type,concurrency ); break;
					case 3: obj = inv.invokeA( metaData,holdability ); break;
					case 4: obj = inv.invokeA( metaData ); break;
					default : jspPrint( "*** タイプ不明 ***" + HybsSystem.BR ); continue;
				}
				String msg = (obj == null) ? inv.getErrMsg() : String.valueOf( obj );

				String str = "<tr class=\"row_" + (rowNo++ % 2) + "\"><td>"
						+ rowNo + "</td><td>"			// 5.5.3.2 (2012/06/08) 行番号を表示
						+ inv.getMethod() + "</td><td>" + msg + "</td><td>" + inv.getComment() + "</td></tr>" + HybsSystem.CR ;
				jspPrint( str );
			}
		}

		// 5.5.3.2 (2012/06/08) 個別に呼び出す箇所を、リフレクションに変更。
//		for( int i=0; i<METHOD_LIST.length; i++ ) {
//			String methodKey = METHOD_LIST[i][0];
//			int    flag  = Integer.parseInt( METHOD_LIST[i][2] );			// 0:通常 1:type 2:type, concurrency 3:holdability 4:ResultSet.TYPE_FORWARD_ONLY 5:Connection.TRANSACTION_READ_COMMITTED
//			if( likeKey == null || methodKey.indexOf( likeKey ) >= 0 ) {
//				try {
//					Object obj= null;
//					if( flag == 0 ) {
//						obj= metaData.getClass().getMethod( methodKey ).invoke(metaData);
//					}
//					else if( flag == 1 ) {
//						obj= metaData.getClass().getMethod( methodKey,int.class ).invoke(metaData,type);
//					}
//					else if( flag == 2 ) {
//						obj= metaData.getClass().getMethod( methodKey,int.class,int.class ).invoke(metaData,type,concurrency);
//					}
//					else if( flag == 3 ) {
//						obj= metaData.getClass().getMethod( methodKey,int.class ).invoke(metaData,holdability);
//					}
//					else if( flag == 4 ) {
//						obj= metaData.getClass().getMethod( methodKey,int.class ).invoke(metaData,Integer.valueOf(ResultSet.TYPE_FORWARD_ONLY));
//						// TYPE_SCROLL_INSENSITIVE , TYPE_SCROLL_SENSITIVE
//					}
//					else if( flag == 5 ) {
//						obj= metaData.getClass().getMethod( methodKey,int.class ).invoke(metaData,Integer.valueOf(Connection.TRANSACTION_READ_COMMITTED));
//						// TRANSACTION_READ_UNCOMMITTED , TRANSACTION_REPEATABLE_READ , TRANSACTION_SERIALIZABLE
//					}
//
//					String str = "<tr class=\"row_" + (rowNo++ % 2) + "\"><td>"
//							+ rowNo + "</td><td>"			// 5.5.3.2 (2012/06/08) 行番号を表示
//							+ methodKey + "</td><td>" + obj + "</td><td>" + METHOD_LIST[i][1] + "</td></tr>" + HybsSystem.CR ;
//					jspPrint( str );
//				}
//				catch( Throwable ex ) {
//					String str = "<tr class=\"row_" + (rowNo++ % 2) + "\"><td>"
//							+ rowNo + "</td><td>"			// 5.5.3.2 (2012/06/08) 行番号を表示
//							+ methodKey + "</td><td>" + "*** 実行時エラー ***" + "</td><td>" + METHOD_LIST[i][1] + "</td></tr>" + HybsSystem.CR ;
//					jspPrint( str );
//				}
//			}
//		}

//		try { printMessage( metaData.getCatalogSeparator() ,"getCatalogSeparator",getCatalogSeparator); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"getCatalogSeparator",getCatalogSeparator); }
//		try { printMessage( metaData.getCatalogTerm() ,"getCatalogTerm",getCatalogTerm); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"getCatalogTerm",getCatalogTerm); }
//		try { printMessage( metaData.getDatabaseProductName() ,"getDatabaseProductName",getDatabaseProductName); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"getDatabaseProductName",getDatabaseProductName); }
//		try { printMessage( metaData.getDatabaseProductVersion() ,"getDatabaseProductVersion",getDatabaseProductVersion); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"getDatabaseProductVersion",getDatabaseProductVersion); }
//		try { printMessage( metaData.getDriverMajorVersion() ,"getDriverMajorVersion",getDriverMajorVersion); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"getDriverMajorVersion",getDriverMajorVersion); }
//		try { printMessage( metaData.getDriverMinorVersion() ,"getDriverMinorVersion",getDriverMinorVersion); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"getDriverMinorVersion",getDriverMinorVersion); }
//		try { printMessage( metaData.getDriverName() ,"getDriverName",getDriverName); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"getDriverName",getDriverName); }
//		try { printMessage( metaData.getDriverVersion() ,"getDriverVersion",getDriverVersion); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"getDriverVersion",getDriverVersion); }
//		try { printMessage( metaData.getExtraNameCharacters() ,"getExtraNameCharacters",getExtraNameCharacters); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"getExtraNameCharacters",getExtraNameCharacters); }
//		try { printMessage( metaData.getIdentifierQuoteString() ,"getIdentifierQuoteString",getIdentifierQuoteString); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"getIdentifierQuoteString",getIdentifierQuoteString); }
//		try { printMessage( metaData.getNumericFunctions() ,"getNumericFunctions",getNumericFunctions); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"getNumericFunctions",getNumericFunctions); }
//		try { printMessage( metaData.getProcedureTerm() ,"getProcedureTerm",getProcedureTerm); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"getProcedureTerm",getProcedureTerm); }
//		try { printMessage( metaData.getSchemaTerm() ,"getSchemaTerm",getSchemaTerm); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"getSchemaTerm",getSchemaTerm); }
//		try { printMessage( metaData.getSearchStringEscape() ,"getSearchStringEscape",getSearchStringEscape); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"getSearchStringEscape",getSearchStringEscape); }
//		try { printMessage( metaData.getSQLKeywords() ,"getSQLKeywords",getSQLKeywords); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"getSQLKeywords",getSQLKeywords); }
//		try { printMessage( metaData.getStringFunctions() ,"getStringFunctions",getStringFunctions); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"getStringFunctions",getStringFunctions); }
//		try { printMessage( metaData.getSystemFunctions() ,"getSystemFunctions",getSystemFunctions); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"getSystemFunctions",getSystemFunctions); }
		try { printMessage( metaData.getTimeDateFunctions() ,"getTimeDateFunctions",getTimeDateFunctions); }
			catch( Throwable e ) { printMessage( e.getMessage(),"getTimeDateFunctions",getTimeDateFunctions); }
//		try { printMessage( metaData.getURL() ,"getURL",getURL); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"getURL",getURL); }
//		try { printMessage( metaData.getUserName() ,"getUserName",getUserName); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"getUserName",getUserName); }
//		try { printMessage( metaData.getDefaultTransactionIsolation() ,"getDefaultTransactionIsolation",getDefaultTransactionIsolation); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"getDefaultTransactionIsolation",getDefaultTransactionIsolation); }
//		try { printMessage( metaData.getSQLStateType() ,"getSQLStateType",getSQLStateType); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"getSQLStateType",getSQLStateType); }
//		try { printMessage( metaData.getResultSetHoldability() ,"getResultSetHoldability",getResultSetHoldability); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"getResultSetHoldability",getResultSetHoldability); }
//		try { printMessage( metaData.allProceduresAreCallable() ,"allProceduresAreCallable",allProceduresAreCallable); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"allProceduresAreCallable",allProceduresAreCallable); }
//		try { printMessage( metaData.allTablesAreSelectable() ,"allTablesAreSelectable",allTablesAreSelectable); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"allTablesAreSelectable",allTablesAreSelectable); }
//		try { printMessage( metaData.dataDefinitionCausesTransactionCommit() ,"dataDefinitionCausesTransactionCommit",dataDefinitionCausesTransactionCommit); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"dataDefinitionCausesTransactionCommit",dataDefinitionCausesTransactionCommit); }
//		try { printMessage( metaData.dataDefinitionIgnoredInTransactions() ,"dataDefinitionIgnoredInTransactions",dataDefinitionIgnoredInTransactions); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"dataDefinitionIgnoredInTransactions",dataDefinitionIgnoredInTransactions); }
//		try { printMessage( metaData.deletesAreDetected(type) ,"deletesAreDetected",deletesAreDetected); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"deletesAreDetected",deletesAreDetected); }
//		try { printMessage( metaData.doesMaxRowSizeIncludeBlobs() ,"doesMaxRowSizeIncludeBlobs",doesMaxRowSizeIncludeBlobs); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"doesMaxRowSizeIncludeBlobs",doesMaxRowSizeIncludeBlobs); }
//		try { printMessage( metaData.insertsAreDetected(type) ,"insertsAreDetected",insertsAreDetected); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"insertsAreDetected",insertsAreDetected); }
//		try { printMessage( metaData.isCatalogAtStart() ,"isCatalogAtStart",isCatalogAtStart); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"isCatalogAtStart",isCatalogAtStart); }
//		try { printMessage( metaData.isReadOnly() ,"isReadOnly",isReadOnly); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"isReadOnly",isReadOnly); }
//		try { printMessage( metaData.locatorsUpdateCopy() ,"locatorsUpdateCopy",locatorsUpdateCopy); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"locatorsUpdateCopy",locatorsUpdateCopy); }
//		try { printMessage( metaData.nullPlusNonNullIsNull() ,"nullPlusNonNullIsNull",nullPlusNonNullIsNull); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"nullPlusNonNullIsNull",nullPlusNonNullIsNull); }
//		try { printMessage( metaData.nullsAreSortedAtEnd() ,"nullsAreSortedAtEnd",nullsAreSortedAtEnd); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"nullsAreSortedAtEnd",nullsAreSortedAtEnd); }
//		try { printMessage( metaData.nullsAreSortedAtStart() ,"nullsAreSortedAtStart",nullsAreSortedAtStart); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"nullsAreSortedAtStart",nullsAreSortedAtStart); }
//		try { printMessage( metaData.nullsAreSortedHigh() ,"nullsAreSortedHigh",nullsAreSortedHigh); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"nullsAreSortedHigh",nullsAreSortedHigh); }
//		try { printMessage( metaData.nullsAreSortedLow() ,"nullsAreSortedLow",nullsAreSortedLow); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"nullsAreSortedLow",nullsAreSortedLow); }
//		try { printMessage( metaData.othersDeletesAreVisible(type) ,"othersDeletesAreVisible",othersDeletesAreVisible); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"othersDeletesAreVisible",othersDeletesAreVisible); }
//		try { printMessage( metaData.othersInsertsAreVisible(type) ,"othersInsertsAreVisible",othersInsertsAreVisible); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"othersInsertsAreVisible",othersInsertsAreVisible); }
//		try { printMessage( metaData.othersUpdatesAreVisible(type) ,"othersUpdatesAreVisible",othersUpdatesAreVisible); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"othersUpdatesAreVisible",othersUpdatesAreVisible); }
//		try { printMessage( metaData.ownDeletesAreVisible(type) ,"ownDeletesAreVisible",ownDeletesAreVisible); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"ownDeletesAreVisible",ownDeletesAreVisible); }
//		try { printMessage( metaData.ownInsertsAreVisible(type) ,"ownInsertsAreVisible",ownInsertsAreVisible); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"ownInsertsAreVisible",ownInsertsAreVisible); }
//		try { printMessage( metaData.ownUpdatesAreVisible(type) ,"ownUpdatesAreVisible",ownUpdatesAreVisible); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"ownUpdatesAreVisible",ownUpdatesAreVisible); }
//		try { printMessage( metaData.storesLowerCaseIdentifiers() ,"storesLowerCaseIdentifiers",storesLowerCaseIdentifiers); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"storesLowerCaseIdentifiers",storesLowerCaseIdentifiers); }
//		try { printMessage( metaData.storesLowerCaseQuotedIdentifiers() ,"storesLowerCaseQuotedIdentifiers",storesLowerCaseQuotedIdentifiers); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"storesLowerCaseQuotedIdentifiers",storesLowerCaseQuotedIdentifiers); }
//		try { printMessage( metaData.storesMixedCaseIdentifiers() ,"storesMixedCaseIdentifiers",storesMixedCaseIdentifiers); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"storesMixedCaseIdentifiers",storesMixedCaseIdentifiers); }
//		try { printMessage( metaData.storesMixedCaseQuotedIdentifiers() ,"storesMixedCaseQuotedIdentifiers",storesMixedCaseQuotedIdentifiers); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"storesMixedCaseQuotedIdentifiers",storesMixedCaseQuotedIdentifiers); }
//		try { printMessage( metaData.storesUpperCaseIdentifiers() ,"storesUpperCaseIdentifiers",storesUpperCaseIdentifiers); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"storesUpperCaseIdentifiers",storesUpperCaseIdentifiers); }
//		try { printMessage( metaData.storesUpperCaseQuotedIdentifiers() ,"storesUpperCaseQuotedIdentifiers",storesUpperCaseQuotedIdentifiers); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"storesUpperCaseQuotedIdentifiers",storesUpperCaseQuotedIdentifiers); }
//		try { printMessage( metaData.supportsAlterTableWithAddColumn() ,"supportsAlterTableWithAddColumn",supportsAlterTableWithAddColumn); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"supportsAlterTableWithAddColumn",supportsAlterTableWithAddColumn); }
//		try { printMessage( metaData.supportsAlterTableWithDropColumn() ,"supportsAlterTableWithDropColumn",supportsAlterTableWithDropColumn); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"supportsAlterTableWithDropColumn",supportsAlterTableWithDropColumn); }
//		try { printMessage( metaData.supportsANSI92EntryLevelSQL() ,"supportsANSI92EntryLevelSQL",supportsANSI92EntryLevelSQL); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"supportsANSI92EntryLevelSQL",supportsANSI92EntryLevelSQL); }
//		try { printMessage( metaData.supportsANSI92FullSQL() ,"supportsANSI92FullSQL",supportsANSI92FullSQL); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"supportsANSI92FullSQL",supportsANSI92FullSQL); }
//		try { printMessage( metaData.supportsANSI92IntermediateSQL() ,"supportsANSI92IntermediateSQL",supportsANSI92IntermediateSQL); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"supportsANSI92IntermediateSQL",supportsANSI92IntermediateSQL); }
//		try { printMessage( metaData.supportsBatchUpdates() ,"supportsBatchUpdates",supportsBatchUpdates); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"supportsBatchUpdates",supportsBatchUpdates); }
//		try { printMessage( metaData.supportsCatalogsInDataManipulation() ,"supportsCatalogsInDataManipulation",supportsCatalogsInDataManipulation); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"supportsCatalogsInDataManipulation",supportsCatalogsInDataManipulation); }
//		try { printMessage( metaData.supportsCatalogsInIndexDefinitions() ,"supportsCatalogsInIndexDefinitions",supportsCatalogsInIndexDefinitions); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"supportsCatalogsInIndexDefinitions",supportsCatalogsInIndexDefinitions); }
//		try { printMessage( metaData.supportsCatalogsInPrivilegeDefinitions() ,"supportsCatalogsInPrivilegeDefinitions",supportsCatalogsInPrivilegeDefinitions); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"supportsCatalogsInPrivilegeDefinitions",supportsCatalogsInPrivilegeDefinitions); }
//		try { printMessage( metaData.supportsCatalogsInProcedureCalls() ,"supportsCatalogsInProcedureCalls",supportsCatalogsInProcedureCalls); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"supportsCatalogsInProcedureCalls",supportsCatalogsInProcedureCalls); }
//		try { printMessage( metaData.supportsCatalogsInTableDefinitions() ,"supportsCatalogsInTableDefinitions",supportsCatalogsInTableDefinitions); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"supportsCatalogsInTableDefinitions",supportsCatalogsInTableDefinitions); }
//		try { printMessage( metaData.supportsColumnAliasing() ,"supportsColumnAliasing",supportsColumnAliasing); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"supportsColumnAliasing",supportsColumnAliasing); }
//		try { printMessage( metaData.supportsConvert() ,"supportsConvert",supportsConvertFunc); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"supportsConvert",supportsConvertFunc); }
//		try { printMessage( metaData.supportsCoreSQLGrammar() ,"supportsCoreSQLGrammar",supportsCoreSQLGrammar); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"supportsCoreSQLGrammar",supportsCoreSQLGrammar); }
//		try { printMessage( metaData.supportsCorrelatedSubqueries() ,"supportsCorrelatedSubqueries",supportsCorrelatedSubqueries); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"supportsCorrelatedSubqueries",supportsCorrelatedSubqueries); }
//		try { printMessage( metaData.supportsDataDefinitionAndDataManipulationTransactions() ,"supportsDataDefinitionAndDataManipulationTransactions",supportsDataDefinitionAndDataManipulationTransactions); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"supportsDataDefinitionAndDataManipulationTransactions",supportsDataDefinitionAndDataManipulationTransactions); }
//		try { printMessage( metaData.supportsDataManipulationTransactionsOnly() ,"supportsDataManipulationTransactionsOnly",supportsDataManipulationTransactionsOnly); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"supportsDataManipulationTransactionsOnly",supportsDataManipulationTransactionsOnly); }
//		try { printMessage( metaData.supportsDifferentTableCorrelationNames() ,"supportsDifferentTableCorrelationNames",supportsDifferentTableCorrelationNames); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"supportsDifferentTableCorrelationNames",supportsDifferentTableCorrelationNames); }
//		try { printMessage( metaData.supportsExpressionsInOrderBy() ,"supportsExpressionsInOrderBy",supportsExpressionsInOrderBy); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"supportsExpressionsInOrderBy",supportsExpressionsInOrderBy); }
//		try { printMessage( metaData.supportsExtendedSQLGrammar() ,"supportsExtendedSQLGrammar",supportsExtendedSQLGrammar); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"supportsExtendedSQLGrammar",supportsExtendedSQLGrammar); }
//		try { printMessage( metaData.supportsFullOuterJoins() ,"supportsFullOuterJoins",supportsFullOuterJoins); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"supportsFullOuterJoins",supportsFullOuterJoins); }
//		try { printMessage( metaData.supportsGetGeneratedKeys() ,"supportsGetGeneratedKeys",supportsGetGeneratedKeys); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"supportsGetGeneratedKeys",supportsGetGeneratedKeys); }
//		try { printMessage( metaData.supportsGroupBy() ,"supportsGroupBy",supportsGroupBy); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"supportsGroupBy",supportsGroupBy); }
//		try { printMessage( metaData.supportsGroupByBeyondSelect() ,"supportsGroupByBeyondSelect",supportsGroupByBeyondSelect); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"supportsGroupByBeyondSelect",supportsGroupByBeyondSelect); }
//		try { printMessage( metaData.supportsGroupByUnrelated() ,"supportsGroupByUnrelated",supportsGroupByUnrelated); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"supportsGroupByUnrelated",supportsGroupByUnrelated); }
//		try { printMessage( metaData.supportsIntegrityEnhancementFacility() ,"supportsIntegrityEnhancementFacility",supportsIntegrityEnhancementFacility); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"supportsIntegrityEnhancementFacility",supportsIntegrityEnhancementFacility); }
//		try { printMessage( metaData.supportsLikeEscapeClause() ,"supportsLikeEscapeClause",supportsLikeEscapeClause); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"supportsLikeEscapeClause",supportsLikeEscapeClause); }
//		try { printMessage( metaData.supportsLimitedOuterJoins() ,"supportsLimitedOuterJoins",supportsLimitedOuterJoins); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"supportsLimitedOuterJoins",supportsLimitedOuterJoins); }
//		try { printMessage( metaData.supportsMinimumSQLGrammar() ,"supportsMinimumSQLGrammar",supportsMinimumSQLGrammar); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"supportsMinimumSQLGrammar",supportsMinimumSQLGrammar); }
//		try { printMessage( metaData.supportsMixedCaseIdentifiers() ,"supportsMixedCaseIdentifiers",supportsMixedCaseIdentifiers); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"supportsMixedCaseIdentifiers",supportsMixedCaseIdentifiers); }
//		try { printMessage( metaData.supportsMixedCaseQuotedIdentifiers() ,"supportsMixedCaseQuotedIdentifiers",supportsMixedCaseQuotedIdentifiers); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"supportsMixedCaseQuotedIdentifiers",supportsMixedCaseQuotedIdentifiers); }
//		try { printMessage( metaData.supportsMultipleOpenResults() ,"supportsMultipleOpenResults",supportsMultipleOpenResults); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"supportsMultipleOpenResults",supportsMultipleOpenResults); }
//		try { printMessage( metaData.supportsMultipleResultSets() ,"supportsMultipleResultSets",supportsMultipleResultSets); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"supportsMultipleResultSets",supportsMultipleResultSets); }
//		try { printMessage( metaData.supportsMultipleTransactions() ,"supportsMultipleTransactions",supportsMultipleTransactions); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"supportsMultipleTransactions",supportsMultipleTransactions); }
//		try { printMessage( metaData.supportsNamedParameters() ,"supportsNamedParameters",supportsNamedParameters); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"supportsNamedParameters",supportsNamedParameters); }
//		try { printMessage( metaData.supportsNonNullableColumns() ,"supportsNonNullableColumns",supportsNonNullableColumns); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"supportsNonNullableColumns",supportsNonNullableColumns); }
//		try { printMessage( metaData.supportsOpenCursorsAcrossCommit() ,"supportsOpenCursorsAcrossCommit",supportsOpenCursorsAcrossCommit); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"supportsOpenCursorsAcrossCommit",supportsOpenCursorsAcrossCommit); }
//		try { printMessage( metaData.supportsOpenCursorsAcrossRollback() ,"supportsOpenCursorsAcrossRollback",supportsOpenCursorsAcrossRollback); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"supportsOpenCursorsAcrossRollback",supportsOpenCursorsAcrossRollback); }
//		try { printMessage( metaData.supportsOpenStatementsAcrossCommit() ,"supportsOpenStatementsAcrossCommit",supportsOpenStatementsAcrossCommit); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"supportsOpenStatementsAcrossCommit",supportsOpenStatementsAcrossCommit); }
//		try { printMessage( metaData.supportsOpenStatementsAcrossRollback() ,"supportsOpenStatementsAcrossRollback",supportsOpenStatementsAcrossRollback); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"supportsOpenStatementsAcrossRollback",supportsOpenStatementsAcrossRollback); }
//		try { printMessage( metaData.supportsOrderByUnrelated() ,"supportsOrderByUnrelated",supportsOrderByUnrelated); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"supportsOrderByUnrelated",supportsOrderByUnrelated); }
//		try { printMessage( metaData.supportsOuterJoins() ,"supportsOuterJoins",supportsOuterJoins); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"supportsOuterJoins",supportsOuterJoins); }
//		try { printMessage( metaData.supportsPositionedDelete() ,"supportsPositionedDelete",supportsPositionedDelete); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"supportsPositionedDelete",supportsPositionedDelete); }
//		try { printMessage( metaData.supportsPositionedUpdate() ,"supportsPositionedUpdate",supportsPositionedUpdate); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"supportsPositionedUpdate",supportsPositionedUpdate); }
//		try { printMessage( metaData.supportsResultSetConcurrency(type, concurrency) ,"supportsResultSetConcurrency",supportsResultSetConcurrency); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"supportsResultSetConcurrency",supportsResultSetConcurrency); }
//		try { printMessage( metaData.supportsResultSetHoldability(holdability) ,"supportsResultSetHoldability",supportsResultSetHoldability); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"supportsResultSetHoldability",supportsResultSetHoldability); }
//		try { printMessage( metaData.supportsResultSetType(ResultSet.TYPE_FORWARD_ONLY) ,"supportsResultSetType[TYPE_FORWARD_ONLY]",supportsResultSetType); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"supportsResultSetType",supportsResultSetType); }
//		try { printMessage( metaData.supportsResultSetType(ResultSet.TYPE_SCROLL_INSENSITIVE) ,"supportsResultSetType[TYPE_SCROLL_INSENSITIVE]", supportsResultSetType); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"supportsResultSetType",supportsResultSetType); }
//		try { printMessage( metaData.supportsResultSetType(ResultSet.TYPE_SCROLL_SENSITIVE) ,"supportsResultSetType[TYPE_SCROLL_SENSITIVE]", supportsResultSetType); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"supportsResultSetType",supportsResultSetType); }
//		try { printMessage( metaData.supportsSavepoints() ,"supportsSavepoints",supportsSavepoints); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"supportsSavepoints",supportsSavepoints); }
//		try { printMessage( metaData.supportsSchemasInDataManipulation() ,"supportsSchemasInDataManipulation",supportsSchemasInDataManipulation); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"supportsSchemasInDataManipulation",supportsSchemasInDataManipulation); }
//		try { printMessage( metaData.supportsSchemasInIndexDefinitions() ,"supportsSchemasInIndexDefinitions",supportsSchemasInIndexDefinitions); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"supportsSchemasInIndexDefinitions",supportsSchemasInIndexDefinitions); }
//		try { printMessage( metaData.supportsSchemasInPrivilegeDefinitions() ,"supportsSchemasInPrivilegeDefinitions",supportsSchemasInPrivilegeDefinitions); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"supportsSchemasInPrivilegeDefinitions",supportsSchemasInPrivilegeDefinitions); }
//		try { printMessage( metaData.supportsSchemasInProcedureCalls() ,"supportsSchemasInProcedureCalls",supportsSchemasInProcedureCalls); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"supportsSchemasInProcedureCalls",supportsSchemasInProcedureCalls); }
//		try { printMessage( metaData.supportsSchemasInTableDefinitions() ,"supportsSchemasInTableDefinitions",supportsSchemasInTableDefinitions); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"supportsSchemasInTableDefinitions",supportsSchemasInTableDefinitions); }
//		try { printMessage( metaData.supportsSelectForUpdate() ,"supportsSelectForUpdate",supportsSelectForUpdate); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"supportsSelectForUpdate",supportsSelectForUpdate); }
//		try { printMessage( metaData.supportsStatementPooling() ,"supportsStatementPooling",supportsStatementPooling); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"supportsStatementPooling",supportsStatementPooling); }
//		try { printMessage( metaData.supportsStoredProcedures() ,"supportsStoredProcedures",supportsStoredProcedures); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"supportsStoredProcedures",supportsStoredProcedures); }
//		try { printMessage( metaData.supportsSubqueriesInComparisons() ,"supportsSubqueriesInComparisons",supportsSubqueriesInComparisons); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"supportsSubqueriesInComparisons",supportsSubqueriesInComparisons); }
//		try { printMessage( metaData.supportsSubqueriesInExists() ,"supportsSubqueriesInExists",supportsSubqueriesInExists); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"supportsSubqueriesInExists",supportsSubqueriesInExists); }
//		try { printMessage( metaData.supportsSubqueriesInIns() ,"supportsSubqueriesInIns",supportsSubqueriesInIns); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"supportsSubqueriesInIns",supportsSubqueriesInIns); }
//		try { printMessage( metaData.supportsSubqueriesInQuantifieds() ,"supportsSubqueriesInQuantifieds",supportsSubqueriesInQuantifieds); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"supportsSubqueriesInQuantifieds",supportsSubqueriesInQuantifieds); }
//		try { printMessage( metaData.supportsTableCorrelationNames() ,"supportsTableCorrelationNames",supportsTableCorrelationNames); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"supportsTableCorrelationNames",supportsTableCorrelationNames); }
//		try { printMessage( metaData.supportsTransactions() ,"supportsTransactions",supportsTransactions); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"supportsTransactions",supportsTransactions); }
//		try { printMessage( metaData.supportsTransactionIsolationLevel(Connection.TRANSACTION_READ_COMMITTED) ,"supportsTransactionIsolationLevel[TRANSACTION_READ_COMMITTED]",supportsTransactionIsolationLevel); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"supportsTransactionIsolationLevel",supportsTransactionIsolationLevel); }
//		try { printMessage( metaData.supportsTransactionIsolationLevel(Connection.TRANSACTION_READ_UNCOMMITTED) ,"supportsTransactionIsolationLevel[TRANSACTION_READ_UNCOMMITTED]",supportsTransactionIsolationLevel); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"supportsTransactionIsolationLevel",supportsTransactionIsolationLevel); }
//		try { printMessage( metaData.supportsTransactionIsolationLevel(Connection.TRANSACTION_REPEATABLE_READ) ,"supportsTransactionIsolationLevel[TRANSACTION_REPEATABLE_READ]",supportsTransactionIsolationLevel); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"supportsTransactionIsolationLevel",supportsTransactionIsolationLevel); }
//		try { printMessage( metaData.supportsTransactionIsolationLevel(Connection.TRANSACTION_SERIALIZABLE) ,"supportsTransactionIsolationLevel[TRANSACTION_SERIALIZABLE]",supportsTransactionIsolationLevel); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"supportsTransactionIsolationLevel",supportsTransactionIsolationLevel); }
//		try { printMessage( metaData.supportsUnion() ,"supportsUnion",supportsUnion); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"supportsUnion",supportsUnion); }
//		try { printMessage( metaData.supportsUnionAll() ,"supportsUnionAll",supportsUnionAll); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"supportsUnionAll",supportsUnionAll); }
//		try { printMessage( metaData.updatesAreDetected(type) ,"updatesAreDetected",updatesAreDetected); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"updatesAreDetected",updatesAreDetected); }
//		try { printMessage( metaData.usesLocalFilePerTable() ,"usesLocalFilePerTable",usesLocalFilePerTable); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"usesLocalFilePerTable",usesLocalFilePerTable); }
//		try { printMessage( metaData.usesLocalFiles() ,"usesLocalFiles",usesLocalFiles); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"usesLocalFiles",usesLocalFiles); }
//
//		// 5.5.3.2 (2012/06/08) JDk1.6 用のメソッドを追加します。
//		try { printMessage( metaData.autoCommitFailureClosesAllResultSets() ,"autoCommitFailureClosesAllResultSets",autoCommitFailureClosesAllResultSets); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"autoCommitFailureClosesAllResultSets",autoCommitFailureClosesAllResultSets); }
//		try { printMessage( metaData.getDatabaseMajorVersion() ,"getDatabaseMajorVersion",getDatabaseMajorVersion); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"getDatabaseMajorVersion",getDatabaseMajorVersion); }
//		try { printMessage( metaData.getDatabaseMinorVersion() ,"getDatabaseMinorVersion",getDatabaseMinorVersion); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"getDatabaseMinorVersion",getDatabaseMinorVersion); }
//		try { printMessage( metaData.getJDBCMajorVersion() ,"getJDBCMajorVersion",getJDBCMajorVersion); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"getJDBCMajorVersion",getJDBCMajorVersion); }
//		try { printMessage( metaData.getJDBCMinorVersion() ,"getJDBCMinorVersion",getJDBCMinorVersion); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"getJDBCMinorVersion",getJDBCMinorVersion); }
//		try { printMessage( metaData.getMaxBinaryLiteralLength() ,"getMaxBinaryLiteralLength",getMaxBinaryLiteralLength); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"getMaxBinaryLiteralLength",getMaxBinaryLiteralLength); }
//		try { printMessage( metaData.getMaxCatalogNameLength() ,"getMaxCatalogNameLength",getMaxCatalogNameLength); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"getMaxCatalogNameLength",getMaxCatalogNameLength); }
//		try { printMessage( metaData.getMaxCharLiteralLength() ,"getMaxCharLiteralLength",getMaxCharLiteralLength); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"getMaxCharLiteralLength",getMaxCharLiteralLength); }
//		try { printMessage( metaData.getMaxColumnNameLength() ,"getMaxColumnNameLength",getMaxColumnNameLength); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"getMaxColumnNameLength",getMaxColumnNameLength); }
//		try { printMessage( metaData.getMaxColumnsInGroupBy() ,"getMaxColumnsInGroupBy",getMaxColumnsInGroupBy); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"getMaxColumnsInGroupBy",getMaxColumnsInGroupBy); }
//		try { printMessage( metaData.getMaxColumnsInIndex() ,"getMaxColumnsInIndex",getMaxColumnsInIndex); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"getMaxColumnsInIndex",getMaxColumnsInIndex); }
//		try { printMessage( metaData.getMaxColumnsInOrderBy() ,"getMaxColumnsInOrderBy",getMaxColumnsInOrderBy); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"getMaxColumnsInOrderBy",getMaxColumnsInOrderBy); }
//		try { printMessage( metaData.getMaxColumnsInSelect() ,"getMaxColumnsInSelect",getMaxColumnsInSelect); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"getMaxColumnsInSelect",getMaxColumnsInSelect); }
//		try { printMessage( metaData.getMaxColumnsInTable() ,"getMaxColumnsInTable",getMaxColumnsInTable); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"getMaxColumnsInTable",getMaxColumnsInTable); }
//		try { printMessage( metaData.getMaxConnections() ,"getMaxConnections",getMaxConnections); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"getMaxConnections",getMaxConnections); }
//		try { printMessage( metaData.getMaxCursorNameLength() ,"getMaxCursorNameLength",getMaxCursorNameLength); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"getMaxCursorNameLength",getMaxCursorNameLength); }
//		try { printMessage( metaData.getMaxIndexLength() ,"getMaxIndexLength",getMaxIndexLength); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"getMaxIndexLength",getMaxIndexLength); }
//		try { printMessage( metaData.getMaxProcedureNameLength() ,"getMaxProcedureNameLength",getMaxProcedureNameLength); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"getMaxProcedureNameLength",getMaxProcedureNameLength); }
//		try { printMessage( metaData.getMaxRowSize() ,"getMaxRowSize",getMaxRowSize); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"getMaxRowSize",getMaxRowSize); }
//		try { printMessage( metaData.getMaxSchemaNameLength() ,"getMaxSchemaNameLength",getMaxSchemaNameLength); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"getMaxSchemaNameLength",getMaxSchemaNameLength); }
//		try { printMessage( metaData.getMaxStatementLength() ,"getMaxStatementLength",getMaxStatementLength); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"getMaxStatementLength",getMaxStatementLength); }
//		try { printMessage( metaData.getMaxStatements() ,"getMaxStatements",getMaxStatements); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"getMaxStatements",getMaxStatements); }
//		try { printMessage( metaData.getMaxTableNameLength() ,"getMaxTableNameLength",getMaxTableNameLength); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"getMaxTableNameLength",getMaxTableNameLength); }
//		try { printMessage( metaData.getMaxTablesInSelect() ,"getMaxTablesInSelect",getMaxTablesInSelect); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"getMaxTablesInSelect",getMaxTablesInSelect); }
//		try { printMessage( metaData.getMaxUserNameLength() ,"getMaxUserNameLength",getMaxUserNameLength); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"getMaxUserNameLength",getMaxUserNameLength); }
//		try { printMessage( metaData.supportsConvert() ,"supportsConvert",supportsConvert); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"supportsConvert",supportsConvert); }
//		try { printMessage( metaData.supportsStoredFunctionsUsingCallSyntax() ,"supportsStoredFunctionsUsingCallSyntax",supportsStoredFunctionsUsingCallSyntax); }
//			catch( Throwable e ) { printMessage( e.getMessage(),"supportsStoredFunctionsUsingCallSyntax",supportsStoredFunctionsUsingCallSyntax); }

		jspPrint( "</table>" );

		if( metaData.supportsConvert() ) {
			supportsConvert( metaData );
		}

		// ResultSet のリターン値を持つ メタデータ
		boolean nullable = true;
		int scope = 0;

		// 5.5.3.2 (2012/06/08) 個別に呼び出す箇所を、リフレクションに変更。
		for( int i=0; i<METHOD_LIST.length; i++ ) {
			MetaDataInvoke inv = METHOD_LIST[i];
			if( inv.isUseMethod( likeKey ) && inv.getType() >= 10 ) {
				ResultSet obj = null;
				switch( inv.getType() ) {
					case 50: obj = inv.invokeB( metaData ); break;
					case 60:
					case 62:
					case 63:
					case 64:
					case 65: obj = inv.invokeB( metaData, catalog, schema, tableName ); break;
					case 70:
					case 71: obj = inv.invokeB( metaData, catalog, schema, typeName ); break;
					case 61: obj = inv.invokeB( metaData, catalog, schema, tableName, columnName ); break;
					case 72: obj = inv.invokeB( metaData, catalog, schema, typeName, attributeName ); break;
					case 81: obj = inv.invokeB( metaData, catalog, schema, procedureName, columnName ); break;
					case 80: obj = inv.invokeB( metaData, catalog, schema, procedureName ); break;
					default : jspPrint( "*** タイプ不明 ***" + HybsSystem.BR ); continue;
				}

				jspPrint( inv.getMethod() + ":" + inv.getComment() + HybsSystem.BR );
				if( obj != null ) {
					jspPrint( makeTableTag( resultToString( obj ) ) );
				}
				else {
					jspPrint( inv.getErrMsg() + HybsSystem.BR );
				}
			}
		}

		// 以下、リフレクションが、エラーなどで結果を返さないメソッド。
		jspPrint( "****************************************************************************************************" + HybsSystem.BR );

		if( likeKey == null || "getAttributes".indexOf( likeKey ) >= 0 ) {
			jspPrint( "getAttributes:" + getAttributes + HybsSystem.BR );
			try { printMessage( metaData.getAttributes(catalog, schema, typeName, attributeName) ,getAttributes); }
				catch( Throwable e ) { printMessage( e.getMessage() ,getAttributes); }
		}
//		if( likeKey == null || "getBestRowIdentifier".indexOf( likeKey ) >= 0 ) {
//			jspPrint( "getBestRowIdentifier:B:" + getBestRowIdentifier + HybsSystem.BR );
//			try { printMessage( metaData.getBestRowIdentifier(catalog, schema, tableName, scope, nullable) ,getBestRowIdentifier); }
//				catch( Throwable e ) { printMessage( e.getMessage() ,getBestRowIdentifier); }
//		}
//		if( likeKey == null || "getCatalogs".indexOf( likeKey ) >= 0 ) {
//			jspPrint( "getCatalogs:" + getCatalogs + HybsSystem.BR );
//			try { printMessage( metaData.getCatalogs() ,getCatalogs); }
//				catch( Throwable e ) { printMessage( e.getMessage() ,getCatalogs); }
//		}
//		if( likeKey == null || "getColumnPrivileges".indexOf( likeKey ) >= 0 ) {
//			jspPrint( "getColumnPrivileges:" + getColumnPrivileges + HybsSystem.BR );
//			try { printMessage( metaData.getColumnPrivileges(catalog, schema, tableName, columnName) ,getColumnPrivileges); }
//				catch( Throwable e ) { printMessage( e.getMessage() ,getColumnPrivileges); }
//		}
		if( likeKey == null || "getColumns".indexOf( likeKey ) >= 0 ) {
			jspPrint( "getColumns:" + getColumns + HybsSystem.BR );
			try { printMessage( metaData.getColumns(catalog, schema, tableName, columnName) ,getColumns); }
				catch( Throwable e ) { printMessage( e.getMessage() ,getColumns); }
		}
//		if( likeKey == null || "getCrossReference".indexOf( likeKey ) >= 0 ) {
//			jspPrint( "getCrossReference:" + getCrossReference + HybsSystem.BR );
////			try { printMessage( metaData.getCrossReference(catalog, schema, tableName, catalog, schema, tableName) ,getCrossReference); }
//			try { printMessage( metaData.getCrossReference(catalog, schema, tableName, null, null, null) ,getCrossReference); }
//				catch( Throwable e ) { printMessage( e.getMessage() ,getCrossReference); }
//		}
//		if( likeKey == null || "getExportedKeys".indexOf( likeKey ) >= 0 ) {
//			jspPrint( "getExportedKeys:" + getExportedKeys + HybsSystem.BR );
//			try { printMessage( metaData.getExportedKeys(catalog, schema, tableName) ,getExportedKeys); }
//				catch( Throwable e ) { printMessage( e.getMessage() ,getExportedKeys); }
//		}
//		if( likeKey == null || "getImportedKeys".indexOf( likeKey ) >= 0 ) {
//			jspPrint( "getImportedKeys:" + getImportedKeys + HybsSystem.BR );
//			try { printMessage( metaData.getImportedKeys(catalog, schema, tableName) ,getImportedKeys); }
//				catch( Throwable e ) { printMessage( e.getMessage() ,getImportedKeys); }
//		}
		if( likeKey == null || "getIndexInfo".indexOf( likeKey ) >= 0 ) {
			jspPrint( "getIndexInfo:" + getIndexInfo + HybsSystem.BR );
			try { printMessage( metaData.getIndexInfo(catalog, schema, tableName, false, false) ,getIndexInfo); }
				catch( Throwable e ) { printMessage( e.getMessage() ,getIndexInfo); }
		}
//		if( likeKey == null || "getPrimaryKeys".indexOf( likeKey ) >= 0 ) {
//			jspPrint( "getPrimaryKeys:" + getPrimaryKeys + HybsSystem.BR );
//			try { printMessage( metaData.getPrimaryKeys(catalog, schema, tableName) ,getPrimaryKeys); }
//				catch( Throwable e ) { printMessage( e.getMessage() ,getPrimaryKeys); }
//		}
//		if( likeKey == null || "getProcedureColumns".indexOf( likeKey ) >= 0 ) {
//			jspPrint( "getProcedureColumns:" + getProcedureColumns + HybsSystem.BR );
//			try { printMessage( metaData.getProcedureColumns(catalog, schema, procedureName, columnName) ,getProcedureColumns); }
//				catch( Throwable e ) { printMessage( e.getMessage() ,getProcedureColumns); }
//		}
//		if( likeKey == null || "getProcedures".indexOf( likeKey ) >= 0 ) {
//			jspPrint( "getProcedures:" + getProcedures + HybsSystem.BR );
//			try { printMessage( metaData.getProcedures(catalog, schema, procedureName) ,getProcedures); }
//				catch( Throwable e ) { printMessage( e.getMessage() ,getProcedures); }
//		}
//		if( likeKey == null || "getSchemas".indexOf( likeKey ) >= 0 ) {
//			jspPrint( "getSchemas:" + getSchemas + HybsSystem.BR );
//			try { printMessage( metaData.getSchemas() ,getSchemas); }
//				catch( Throwable e ) { printMessage( e.getMessage() ,getSchemas); }
//		}
		if( likeKey == null || "getSuperTables".indexOf( likeKey ) >= 0 ) {
			jspPrint( "getSuperTables:" + getSuperTables + HybsSystem.BR );
			try { printMessage( metaData.getSuperTables(catalog, schema, tableName) ,getSuperTables); }
				catch( Throwable e ) { printMessage( e.getMessage() ,getSuperTables); }
		}
		if( likeKey == null || "getSuperTypes".indexOf( likeKey ) >= 0 ) {
			jspPrint( "getSuperTypes:" + getSuperTypes + HybsSystem.BR );
			try { printMessage( metaData.getSuperTypes(catalog, schema, typeName) ,getSuperTypes); }
				catch( Throwable e ) { printMessage( e.getMessage() ,getSuperTypes); }
		}
//		if( likeKey == null || "getTablePrivileges".indexOf( likeKey ) >= 0 ) {
//			jspPrint( "getTablePrivileges:" + getTablePrivileges + HybsSystem.BR );
//			try { printMessage( metaData.getTablePrivileges(catalog, schema, tableName) ,getTablePrivileges); }
//				catch( Throwable e ) { printMessage( e.getMessage() ,getTablePrivileges); }
//		}
//		if( likeKey == null || "getTables".indexOf( likeKey ) >= 0 ) {
//			jspPrint( "getTables:" + getTables + HybsSystem.BR );
//			try { printMessage( metaData.getTables(catalog, schema, tableName, null) ,getTables); }
//				catch( Throwable e ) { printMessage( e.getMessage() ,getTables); }
//		}
//		if( likeKey == null || "getTableTypes".indexOf( likeKey ) >= 0 ) {
//			jspPrint( "getTableTypes:" + getTableTypes + HybsSystem.BR );
//			try { printMessage( metaData.getTableTypes() ,getTableTypes); }
//				catch( Throwable e ) { printMessage( e.getMessage() ,getTableTypes); }
//		}
		if( likeKey == null || "getTypeInfo".indexOf( likeKey ) >= 0 ) {
			jspPrint( "getTypeInfo:" + getTypeInfo + HybsSystem.BR );
			try { printMessage( metaData.getTypeInfo() ,getTypeInfo); }
				catch( Throwable e ) { printMessage( e.getMessage() ,getTypeInfo); }
		}
//		if( likeKey == null || "getUDTs".indexOf( likeKey ) >= 0 ) {
//			jspPrint( "getUDTs:" + getUDTs + HybsSystem.BR );
//			try { printMessage( metaData.getUDTs(catalog, schema, typeName, null) ,getUDTs); }
//				catch( Throwable e ) { printMessage( e.getMessage() ,getUDTs); }
//		}
//		if( likeKey == null || "getVersionColumns".indexOf( likeKey ) >= 0 ) {
//			jspPrint( "getVersionColumns:" + getVersionColumns + HybsSystem.BR );
//			try { printMessage( metaData.getVersionColumns(catalog, schema, tableName) ,getVersionColumns); }
//				catch( Throwable e ) { printMessage( e.getMessage() ,getVersionColumns); }
//		}
//
//		// 5.5.3.2 (2012/06/08) JDk1.6 用のメソッドを追加します。
//		if( likeKey == null || "getClientInfoProperties".indexOf( likeKey ) >= 0 ) {
//			jspPrint( "getClientInfoProperties:" + getClientInfoProperties + HybsSystem.BR );
//			try { printMessage( metaData.getClientInfoProperties() ,getClientInfoProperties); }
//				catch( Throwable e ) { printMessage( e.getMessage() ,getClientInfoProperties); }
//		}
//		if( likeKey == null || "getFunctionColumns".indexOf( likeKey ) >= 0 ) {
//			jspPrint( "getFunctionColumns:" + getFunctionColumns + HybsSystem.BR );
//			try { printMessage( metaData.getFunctionColumns(catalog, schema, procedureName ,columnName),getFunctionColumns); }
//				catch( Throwable e ) { printMessage( e.getMessage() ,getFunctionColumns); }
//		}
//		if( likeKey == null || "getFunctions".indexOf( likeKey ) >= 0 ) {
//			jspPrint( "getFunctions:" + getFunctions + HybsSystem.BR );
//			try { printMessage( metaData.getFunctions(catalog, schema, procedureName) ,getFunctions); }
//				catch( Throwable e ) { printMessage( e.getMessage() ,getFunctions); }
//		}
	}

	/**
	 * データベースの Types 定数を文字列に変換したテーブルを画面に出力します。
	 *
	 * @param metaData DatabaseMetaDataオブジェクト
	 * @throws Throwable
	 * @see java.sql.Types
	 */
	private void supportsConvert( final DatabaseMetaData metaData ) throws Throwable {

		int cnt = 36;
		int[]    key = new int[cnt] ;
		String[] msg = new String[cnt] ;

		int i = 0;
		key[i] = Types.ARRAY ;			msg[i++] = "ARRAY" ;			// 0
		key[i] = Types.BIGINT ;			msg[i++] = "BIGINT" ;			// 1
		key[i] = Types.BINARY ;			msg[i++] = "BINARY" ;			// 2
		key[i] = Types.BIT ;			msg[i++] = "BIT" ;				// 3
		key[i] = Types.BLOB ;			msg[i++] = "BLOB" ;				// 4
		key[i] = Types.BOOLEAN ;		msg[i++] = "BOOLEAN" ;			// 5
		key[i] = Types.CHAR ;			msg[i++] = "CHAR" ;				// 6
		key[i] = Types.CLOB ;			msg[i++] = "CLOB" ;				// 7
		key[i] = Types.DATALINK ;		msg[i++] = "DATALINK" ;			// 8
		key[i] = Types.DATE ;			msg[i++] = "DATE" ;				// 9
		key[i] = Types.DECIMAL ;		msg[i++] = "DECIMAL" ;			// 10
		key[i] = Types.DISTINCT ;		msg[i++] = "DISTINCT" ;			// 11
		key[i] = Types.DOUBLE ;			msg[i++] = "DOUBLE" ;			// 12
		key[i] = Types.FLOAT ;			msg[i++] = "FLOAT" ;			// 13
		key[i] = Types.INTEGER ;		msg[i++] = "INTEGER" ;			// 14
		key[i] = Types.JAVA_OBJECT ;	msg[i++] = "JAVA_OBJECT" ;		// 15
		key[i] = Types.LONGNVARCHAR ;	msg[i++] = "LONGNVARCHAR" ;		// 16	5.5.3.2 (2012/06/08)
		key[i] = Types.LONGVARBINARY ;	msg[i++] = "LONGVARBINARY" ;	// 17
		key[i] = Types.LONGVARCHAR ;	msg[i++] = "LONGVARCHAR" ;		// 18
		key[i] = Types.NCHAR ;			msg[i++] = "NCHAR" ;			// 19	5.5.3.2 (2012/06/08)
		key[i] = Types.NCLOB ;			msg[i++] = "NCLOB" ;			// 20	5.5.3.2 (2012/06/08)
		key[i] = Types.NULL ;			msg[i++] = "NULL" ;				// 21
		key[i] = Types.NUMERIC ;		msg[i++] = "NUMERIC" ;			// 22
		key[i] = Types.NVARCHAR ;		msg[i++] = "NVARCHAR" ;			// 23	5.5.3.2 (2012/06/08)
		key[i] = Types.OTHER ;			msg[i++] = "OTHER" ;			// 24
		key[i] = Types.REAL ;			msg[i++] = "REAL" ;				// 25
		key[i] = Types.REF ;			msg[i++] = "REF" ;				// 26
		key[i] = Types.ROWID ;			msg[i++] = "ROWID" ;			// 27	5.5.3.2 (2012/06/08)
		key[i] = Types.SMALLINT ;		msg[i++] = "SMALLINT" ;			// 28	5.5.3.2 (2012/06/08)
		key[i] = Types.SQLXML ;			msg[i++] = "SQLXML" ;			// 29
		key[i] = Types.STRUCT ;			msg[i++] = "STRUCT" ;			// 30
		key[i] = Types.TIME ;			msg[i++] = "TIME" ;				// 31
		key[i] = Types.TIMESTAMP ;		msg[i++] = "TIMESTAMP" ;		// 32
		key[i] = Types.TINYINT ;		msg[i++] = "TINYINT" ;			// 33
		key[i] = Types.VARBINARY ;		msg[i++] = "VARBINARY" ;		// 34
		key[i] = Types.VARCHAR ;		msg[i  ] = "VARCHAR" ;			// 35

		// それぞれ、Types 名称用のフィールドを追加
		String[][] table = new String[cnt+1][cnt+1];		// ヘッダー行を追加
		for( int hd=0; hd<cnt; hd++ ) {
			table[0][hd+1] = msg[hd];		// 行のヘッダー
			table[hd+1][0] = msg[hd];		// 列のヘッダー
		}

		for( int row=0; row<cnt; row++ ) {
			for( int clm=0; clm<cnt; clm++ ) {
				if( metaData.supportsConvert( key[row], key[clm]) ) {
					table[row+1][clm+1] = "○" ;
				}
				else {
					table[row+1][clm+1] = "×" ;
				}
			}
		}

		jspPrint( supportsConvertFunc + HybsSystem.BR );
		jspPrint( makeTableTag( table ) );
	}

	// 5.5.3.2 (2012/06/08) メソッドのMapです。順序を登録順に保つため、LinkedHashMap を使います。

//	private static final String[][] METHOD_LIST = new String[][] {
//		  { "getCatalogSeparator" , "このデータベースがカタログ名とテーブル名のセパレータとして使用する String"  , "0" }
//		, { "getCatalogTerm" , "「catalog」に対するデータベースベンダーの推奨用語を取得します。 "  , "0" }
//		, { "getDatabaseProductName" , "このデータベース製品の名前を取得します。 "  , "0" }
//		, { "getDatabaseProductVersion" , "このデータベース製品のバージョンを取得します。 "  , "0" }
//		, { "getDriverMajorVersion" , "この JDBC ドライバのメジャーバージョンを取得します。 "  , "0" }
//		, { "getDriverMinorVersion" , "この JDBC ドライバのマイナーバージョンを取得します。 "  , "0" }
//		, { "getDriverName" , "この JDBC ドライバの名前を取得します。 "  , "0" }
//		, { "getDriverVersion" , "この JDBC ドライバのバージョンを String"  , "0" }
//		, { "getExtraNameCharacters" , "引用符で囲まれていない識別名に使用できるすべての「特殊」文字 (a-z、A-Z、0-9、および _ 以外) を取得します。 "  , "0" }
//		, { "getIdentifierQuoteString" , "SQL 識別子を引用するのに使用する文字列を取得します。 "  , "0" }
//		, { "getNumericFunctions" , "このデータベースで使用可能なコンマで区切った数学関数のリストを取得します。 "  , "0" }
//		, { "getProcedureTerm" , "「procedure」に対するデータベースベンダーの推奨用語を取得します。 "  , "0" }
//		, { "getSchemaTerm" , "「schema」に対するデータベースベンダーの推奨用語を取得します。 "  , "0" }
//		, { "getSearchStringEscape" , "ワイルドカード文字をエスケープするのに使用できる文字列を取得します。 "  , "0" }
//		, { "getSQLKeywords" , "このデータベースの SQL キーワードであって、SQL92 のキーワードではない、すべてのキーワードをコンマで区切ったリストを取得します。 "  , "0" }
//		, { "getStringFunctions" , "このデータベースで使用可能なコンマで区切った文字列関数のリストを取得します。 "  , "0" }
//		, { "getSystemFunctions" , "このデータベースで使用可能なコンマで区切ったシステム関数のリストを取得します。 "  , "0" }
//		, { "getTimeDateFunctions" , "このデータベースで使用可能な時間関数と日付関数をコンマで区切ったリストを取得します。 "  , "0" }
//		, { "getURL" , "この DBMS の URL を取得します。 "  , "0" }
//		, { "getUserName" , "このデータベースに記録されているユーザ名を取得します。 "  , "0" }
//
//		, { "getDefaultTransactionIsolation" , "このデータベースのデフォルトのトランザクション遮断レベルを取得します。 "  , "0" }
//		, { "getSQLStateType" , "Throwable.getSQLState によって返される SQLSTATE が X/Open (現在は Open Group) の SQL CLI であるか SQL99 であるかを示します。 "  , "0" }
//		, { "getResultSetHoldability" , "この ResultSet オブジェクトのデフォルトの保持機能を取得します。 "  , "0" }
//
//		, { "allProceduresAreCallable" , "getProcedures メソッドによって返されるすべてのプロシージャが、現在のユーザから呼び出せるかどうかを取得します。 "  , "0" }
//		, { "allTablesAreSelectable" , "getTables メソッドによって返されるすべてのテーブルが、現在のユーザによって使用できるかどうかを取得します。 "  , "0" }
//		, { "dataDefinitionCausesTransactionCommit" , "トランザクションのデータ定義文が、トランザクションを強制的にコミットさせるかどうかを取得します。 "  , "0" }
//		, { "dataDefinitionIgnoredInTransactions" , "このデータベースによって、トランザクションでデータ定義文が無視されるかどうかを取得します。 "  , "0" }
//		, { "deletesAreDetected" , "ResultSet.rowDeleted メソッドを呼び出すことによって可視の行が削除されたことを検出できるかどうかを取得します。 "  , "1" }
//		, { "doesMaxRowSizeIncludeBlobs" , "getMaxRowSize メソッドの戻り値が SQL データの型の LONGVARCHAR および LONGVARBINARY を含むかどうかを取得します。 "  , "0" }
//		, { "insertsAreDetected" , "ResultSet.rowInserted メソッドを呼び出すことによって可視の行が挿入されたことを検出できるかどうかを取得します。 "  , "1" }
//		, { "isCatalogAtStart" , "完全修飾されたテーブル名の開始部分 (または終了部分) にカタログが現れるかどうかを取得します。 "  , "0" }
//		, { "isReadOnly" , "このデータベースが読み込み専用モードかどうかを取得します。 "  , "0" }
//		, { "locatorsUpdateCopy" , "LOB への変更が、コピーに対して行われたのか、LOB に直接行われたのかを示します。 "  , "0" }
//		, { "nullPlusNonNullIsNull" , "このデータベースが、NULL 値と非 NULL 値の連結を NULL とするかどうかを取得します。 "  , "0" }
//		, { "nullsAreSortedAtEnd" , "NULL 値が、終了時にソート順にかかわらずソートされるかどうかを取得します。 "  , "0" }
//		, { "nullsAreSortedAtStart" , "NULL 値が、開始時にソート順にかかわらずソートされるかどうかを取得します。 "  , "0" }
//		, { "nullsAreSortedHigh" , "NULL 値が高位にソートされるかどうかを取得します。 "  , "0" }
//		, { "nullsAreSortedLow" , "NULL 値が下位にソートされるかどうかを取得します。 "  , "0" }
//		, { "othersDeletesAreVisible" , "他で行われた削除が可視かどうかを取得します。 "  , "1" }
//		, { "othersInsertsAreVisible" , "他で行われた挿入が可視かどうかを取得します。 "  , "1" }
//		, { "othersUpdatesAreVisible" , "他で行われた更新が可視かどうかを取得します。 "  , "1" }
//		, { "ownDeletesAreVisible" , "結果セット自身の削除が可視かどうかを取得します。 "  , "1" }
//		, { "ownInsertsAreVisible" , "結果セット自身の挿入が可視かどうかを取得します。 "  , "1" }
//		, { "ownUpdatesAreVisible" , "指定された ResultSet オブジェクトについて、結果セット自身の更新が可視かどうかを取得します。 "  , "1" }
//		, { "storesLowerCaseIdentifiers" , "このデータベースが、大文字小文字が混在する引用符なしの SQL 識別子を、大文字小文字を区別しないで処理し、小文字で格納するかどうかを取得します。 "  , "0" }
//		, { "storesLowerCaseQuotedIdentifiers" , "このデータベースが、大文字小文字が混在する引用符付きの SQL 識別子を、大文字小文字を区別しないで処理し、小文字で格納するかどうかを取得します。 "  , "0" }
//		, { "storesMixedCaseIdentifiers" , "このデータベースが、大文字小文字が混在する引用符なしの SQL 識別子を、大文字小文字を区別しないで処理し、大文字小文字混在で格納するかどうかを取得します。 "  , "0" }
//		, { "storesMixedCaseQuotedIdentifiers" , "このデータベースが、大文字小文字が混在する引用符付きの SQL 識別子を、大文字小文字を区別しないで処理し、大文字小文字混在で格納するかどうかを取得します。 "  , "0" }
//		, { "storesUpperCaseIdentifiers" , "このデータベースが、大文字小文字が混在する引用符なしの SQL 識別子を、大文字小文字を区別しないで処理し、大文字で格納するかどうかを取得します。 "  , "0" }
//		, { "storesUpperCaseQuotedIdentifiers" , "このデータベースが、大文字小文字が混在する引用符付きの SQL 識別子を、大文字小文字を区別しないで処理し、大文字で格納するかどうかを取得します。 "  , "0" }
//		, { "supportsAlterTableWithAddColumn" , "このデータベースによって、追加列のある ALTER TABLE がサポートされるかどうかを取得します。 "  , "0" }
//		, { "supportsAlterTableWithDropColumn" , "このデータベースによって、ドロップ列のある ALTER TABLE がサポートされるかどうかを取得します。 "  , "0" }
//		, { "supportsANSI92EntryLevelSQL" , "このデータベースによって、ANSI92 エントリレベルの SQL 文法がサポートされるかどうかを取得します。 "  , "0" }
//		, { "supportsANSI92FullSQL" , "このデータベースによって、ANSI92 完全レベルの SQL 文法がサポートされるかどうかを取得します。 "  , "0" }
//		, { "supportsANSI92IntermediateSQL" , "このデータベースによって、ANSI92 中間レベルの SQL 文法がサポートされるかどうかを取得します。 "  , "0" }
//		, { "supportsBatchUpdates" , "このデータベースによってバッチ更新がサポートされるかどうかを取得します。 "  , "0" }
//		, { "supportsCatalogsInDataManipulation" , "データ操作文でカタログ名を使用できるかどうかを取得します。 "  , "0" }
//		, { "supportsCatalogsInIndexDefinitions" , "インデックス定義文でカタログ名を使用できるかどうかを取得します。 "  , "0" }
//		, { "supportsCatalogsInPrivilegeDefinitions" , "特権定義文でカタログ名を使用できるかどうかを取得します。 "  , "0" }
//		, { "supportsCatalogsInProcedureCalls" , "プロシージャ呼び出し文でカタログ名を使用できるかどうかを取得します。 "  , "0" }
//		, { "supportsCatalogsInTableDefinitions" , "テーブル定義文でカタログ名を使用できるかどうかを取得します。 "  , "0" }
//		, { "supportsColumnAliasing" , "このデータベースによって列の別名がサポートされるかどうかを取得します。  "  , "0" }
//	//	, { "supportsConvertFunc" , "このデータベースによって、SQL の型間の CONVERT 関数がサポートされるかどうかを取得します。 "  , "0" }
//		, { "supportsCoreSQLGrammar" , "このデータベースによって、ODBC Core SQL 文法がサポートされるかどうかを取得します。 "  , "0" }
//		, { "supportsCorrelatedSubqueries" , "このデータベースによって照合関係サブクエリーがサポートされるかどうかを取得します。 "  , "0" }
//		, { "supportsDataDefinitionAndDataManipulationTransactions" , "このデータベースによって、トランザクションで、データ定義文とデータ操作文の両方がサポートされるかどうかを取得します。 "  , "0" }
//		, { "supportsDataManipulationTransactionsOnly" , "このデータベースによって、トランザクションでデータ操作文だけがサポートされるかどうかを取得します。 "  , "0" }
//		, { "supportsDifferentTableCorrelationNames" , "テーブル相互関係名がサポートされる場合、テーブルの名前と異なる名前であるという制限を付けるかどうかを取得します。 "  , "0" }
//		, { "supportsExpressionsInOrderBy" , "このデータベースによって、ORDER BY リスト中で式がサポートされるかどうかを取得します。 "  , "0" }
//		, { "supportsExtendedSQLGrammar" , "このデータベースによって、ODBC Extended SQL 文法がサポートされるかどうかを取得します。 "  , "0" }
//		, { "supportsFullOuterJoins" , "このデータベースによって、完全入れ子の外部結合がサポートされるかどうかを取得します。 "  , "0" }
//		, { "supportsGetGeneratedKeys" , "文が実行されたあとに自動生成キーを取得できるかどうかを取得します。 "  , "0" }
//		, { "supportsGroupBy" , "このデータベースによって、GROUP BY 節のフォームがサポートされるかどうかを取得します。 "  , "0" }
//		, { "supportsGroupByBeyondSelect" , "SELECT 文中のすべての列が GROUP BY 節に含まれるという条件で、このデータベースによって、GROUP BY 節で SELECT 文中にない列の使用がサポートされるかどうかを取得します。 "  , "0" }
//		, { "supportsGroupByUnrelated" , "このデータベースによって、GROUP BY 節で SELECT 文中にない列の使用がサポートされるかどうかを取得します。 "  , "0" }
//		, { "supportsIntegrityEnhancementFacility" , "このデータベースによって、SQL Integrity Enhancement Facility がサポートされるかどうかを取得します。 "  , "0" }
//		, { "supportsLikeEscapeClause" , "このデータベースによって、LIKE エスケープ節の指定がサポートされるかどうかを取得します。 "  , "0" }
//		, { "supportsLimitedOuterJoins" , "このデータベースによって、外部結合に関し、制限されたサポートが提供されるかどうかを取得します (supportsFullOuterJoins メソッドが true を返す場合は true)。 "  , "0" }
//		, { "supportsMinimumSQLGrammar" , "このデータベースによって、ODBC Minimum SQL 文法がサポートされるかどうかを取得します。 "  , "0" }
//		, { "supportsMixedCaseIdentifiers" , "このデータベースが、大文字小文字が混在する引用符なしの SQL 識別子を、大文字小文字を区別して処理し、大文字小文字混在で格納するかどうかを取得します。 "  , "0" }
//		, { "supportsMixedCaseQuotedIdentifiers" , "このデータベースが、大文字小文字が混在する引用符付きの SQL 識別子を、大文字小文字を区別して処理し、結果として大文字小文字混在で格納するかどうかを取得します。 "  , "0" }
//		, { "supportsMultipleOpenResults" , "CallableStatement オブジェクトから同時に返された複数の ResultSet オブジェクトを持つことが可能かどうかを取得します。 "  , "0" }
//		, { "supportsMultipleResultSets" , "このデータベースによって、execute メソッドの単一の呼び出しからの複数の ResultSet オブジェクトの取得がサポートされるかどうかを取得します。 "  , "0" }
//		, { "supportsMultipleTransactions" , "このデータベースが一度に複数のトランザクションを (異なった接続で) オープンできるかどうかを取得します。 "  , "0" }
//		, { "supportsNamedParameters" , "このデータベースによって、callable 文への名前付きパラメータがサポートされるかどうかを取得します。 "  , "0" }
//		, { "supportsNonNullableColumns" , "このデータベースの列を非 null として定義できるかどうかを取得します。 "  , "0" }
//		, { "supportsOpenCursorsAcrossCommit" , "このデータベースによって、コミット間でカーソルがオープンされたままの状態がサポートされるかどうかを取得します。 "  , "0" }
//		, { "supportsOpenCursorsAcrossRollback" , "このデータベースによって、ロールバック間でカーソルがオープンされたままの状態がサポートされるかどうかを取得します。 "  , "0" }
//		, { "supportsOpenStatementsAcrossCommit" , "このデータベースによって、コミット間で文がオープンされたままの状態がサポートされるかどうかを取得します。 "  , "0" }
//		, { "supportsOpenStatementsAcrossRollback" , "このデータベースによって、ロールバック間で文がオープンされたままの状態がサポートされるかどうかを取得します。 "  , "0" }
//		, { "supportsOrderByUnrelated" , "このデータベースによって、ORDER BY 節で SELECT 文中にない列の使用がサポートされるかどうかを取得します。 "  , "0" }
//		, { "supportsOuterJoins" , "このデータベースによって、外部結合のなんらかの形式がサポートされるかどうかを取得します。 "  , "0" }
//		, { "supportsPositionedDelete" , "このデータベースによって、位置指定された DELETE 文がサポートされるかどうかを取得します。 "  , "0" }
//		, { "supportsPositionedUpdate" , "このデータベースによって、位置指定された UPDATE 文がサポートされるかどうかを取得します。 "  , "0" }
//		, { "supportsResultSetConcurrency" , "このデータベースが、指定された結果セットの型と与えられた並行処理の種類の組み合わせをサポートするかどうかを取得します。 "  , "2" }
//		, { "supportsResultSetHoldability" , "このデータベースが、指定された結果セットの保持機能をサポートするかどうかを取得します。 "  , "3" }
//		, { "supportsResultSetType" , "このデータベースが、[TYPE_FORWARD_ONLY]指定された結果セットの型をサポートするかどうかを取得します。 "  , "41" }
//		, { "supportsResultSetType" , "このデータベースが、[TYPE_SCROLL_INSENSITIVE]指定された結果セットの型をサポートするかどうかを取得します。 "  , "42" }
//		, { "supportsResultSetType" , "このデータベースが、[TYPE_SCROLL_SENSITIVE]指定された結果セットの型をサポートするかどうかを取得します。 "  , "43" }
//		, { "supportsSavepoints" , "このデータベースによってセーブポイントがサポートされるかどうかを取得します。 "  , "0" }
//		, { "supportsSchemasInDataManipulation" , "データ操作文でスキーマ名を使用できるかどうかを取得します。 "  , "0" }
//		, { "supportsSchemasInIndexDefinitions" , "インデックス定義文でスキーマ名を使用できるかどうかを取得します。 "  , "0" }
//		, { "supportsSchemasInPrivilegeDefinitions" , "特権定義文でスキーマ名を使用できるかどうかを取得します。 "  , "0" }
//		, { "supportsSchemasInProcedureCalls" , "プロシージャ呼び出し文でスキーマ名を使用できるかどうかを取得します。 "  , "0" }
//		, { "supportsSchemasInTableDefinitions" , "テーブル定義文でスキーマ名を使用できるかどうかを取得します。 "  , "0" }
//		, { "supportsSelectForUpdate" , "このデータベースによって SELECT FOR UPDATE 文がサポートされるかどうかを取得します。 "  , "0" }
//		, { "supportsStatementPooling" , "このデータベースによって文のプールがサポートされるかどうかを取得します。 "  , "0" }
//		, { "supportsStoredProcedures" , "このデータベースによって、ストアドプロシージャエスケープ構文を使用するストアドプロシージャコールがサポートされるかどうかを判定します。 "  , "0" }
//		, { "supportsSubqueriesInComparisons" , "このデータベースによって、比較式中でサブクエリーがサポートされるかどうかを取得します。 "  , "0" }
//		, { "supportsSubqueriesInExists" , "このデータベースによって、EXISTS 式中でサブクエリーがサポートされるかどうかを取得します。 "  , "0" }
//		, { "supportsSubqueriesInIns" , "このデータベースによって、IN 文中でサブクエリーがサポートされるかどうかを取得します。 "  , "0" }
//		, { "supportsSubqueriesInQuantifieds" , "このデータベースによって、定量化された式中でサブクエリーがサポートされるかどうかを取得します。 "  , "0" }
//		, { "supportsTableCorrelationNames" , "このデータベースによってテーブル相互関係名がサポートされるかどうかを取得します。 "  , "0" }
//		, { "supportsTransactionIsolationLevel" , "このデータベースが、指定されたトランザクション遮断レベルをサポートするかどうかを取得します。 "  , "5" }
//		, { "supportsTransactions" , "このデータベースによってトランザクションがサポートされるかどうかを取得します。 "  , "0" }
//		, { "supportsUnion" , "このデータベースによって SQL UNION がサポートされるかどうかを取得します。 "  , "0" }
//		, { "supportsUnionAll" , "このデータベースによって SQL UNION ALL がサポートされるかどうかを取得します。 "  , "0" }
//		, { "updatesAreDetected" , "ResultSet.rowUpdated メソッドを呼び出すことによって可視の行が更新されたことを検出できるかどうかを取得します。 "  , "1" }
//		, { "usesLocalFilePerTable" , "このデータベースが、各テーブルにファイルを使用するかどうかを取得します。 "  , "0" }
//		, { "usesLocalFiles" , "このデータベースが、ローカルファイルにテーブルを格納するかどうかを取得します。 "  , "0" }
//
//	// これらは、引数が必要なので、リフレクション処理は、とりあえず保留
//	//	, { "getAttributes" , "指定されたスキーマおよびカタログで使用可能なユーザ定義の型 (UDT) のための指定された型の指定された属性に関する記述を取得します。  "  , "0" }
//	//	, { "getBestRowIdentifier" , "行を一意に識別するテーブルの最適な列セットに関する記述を取得します。 "  , "0" }
//	//	, { "getCatalogs" , "このデータベースで使用可能なカタログ名を取得します。 "  , "0" }
//	//	, { "getColumnPrivileges" , "テーブルの列へのアクセス権に関する記述を取得します。  "  , "0" }
//	//	, { "getColumns" , "指定されたカタログで使用可能なテーブル列の記述を取得します。  "  , "0" }
//	//	, { "getCrossReference" , "指定された主キーテーブルの主キー列を参照する指定された外部のキーテーブル中の、外部のキー列に関する記述 (テーブルが別のキーをインポートする方法を記述) を取得します。 "  , "0" }
//	//	, { "getExportedKeys" , "指定されたテーブルの主キー列 (テーブルによってエクスポートされた外部キー) を参照する外部キー列に関する記述を取得します。 "  , "0" }
//	//	, { "getImportedKeys" , "テーブルの外部キー列 (テーブルによってインポートされる主キー) を参照する主キー列に関する記述を取得します。 "  , "0" }
//	//	, { "getIndexInfo" , "指定されたテーブルのインデックスと統計情報に関する記述を取得します。 "  , "0" }
//	//	, { "getPrimaryKeys" , "指定されたテーブルの主キー列の記述を取得します。 "  , "0" }
//	//	, { "getProcedureColumns" , "指定されたカタログのストアドプロシージャパラメータと結果列に関する記述を取得します。  "  , "0" }
//	//	, { "getProcedures" , "指定されたカタログで使用可能なストアドプロシージャに関する記述を取得します。  "  , "0" }
//	//	, { "getSchemas" , "このデータベースで使用可能なスキーマ名を取得します。 "  , "0" }
//	//	, { "getSuperTables" , "このデータベースの特定のスキーマで定義されているテーブル階層の説明を取得します。  "  , "0" }
//	//	, { "getSuperTypes" , "このデータベースの特定のスキーマで定義されているユーザ定義型 (UDT) 階層の説明を取得します。 "  , "0" }
//	//	, { "getTablePrivileges" , "カタログで使用可能な各テーブルに対するアクセス権に関する記述を取得します。 "  , "0" }
//	//	, { "getTables" , "指定されたカタログで使用可能なテーブルに関する記述を取得します。 "  , "0" }
//	//	, { "getTableTypes" , "このデータベースで使用可能なテーブルの型を取得します。 "  , "0" }
//	//	, { "getTypeInfo" , "このデータベースでサポートされているすべての標準 SQL の型に関する記述を取得します。 "  , "0" }
//	//	, { "getUDTs" , "特定のスキーマで定義されているユーザ定義型 (UDT) の説明を取得します。 "  , "0" }
//	//	, { "getVersionColumns" , "行の任意の値が変更された場合に、自動的に更新されるテーブルの列に関する記述を取得します。 "  , "0" }
//
//	// 5.5.3.2 (2012/06/08) JDk1.6 用のメソッドを追加します。
//		, { "autoCommitFailureClosesAllResultSets" , "autoCommit が true の場合に、ResultSet が保持可能であっても、すべてのオープンされた ResultSet がクローズされたことを SQLException が示すかどうかを取得します。"  , "0" }
//		, { "supportsConvert" , "このデータベースによって、JDBC 型 fromType と toType の間の変換に使用される JDBC スカラー関数 CONVERT がサポートされるかどうかを取得します。"  , "0" }
//		, { "supportsStoredFunctionsUsingCallSyntax" , "このデータベースが、ストアドプロシージャーエスケープ構文を使用した、ユーザー定義関数またはベンダー関数の呼び出しをサポートするかどうかを取得します。"  , "0" }
//		, { "getDatabaseMajorVersion" , "基本となるデータベースのメジャーバージョン番号を取得します。"  , "0" }
//		, { "getDatabaseMinorVersion" , "基本となるデータベースのマイナーバージョン番号を取得します。"  , "0" }
//		, { "getJDBCMajorVersion" , "このドライバの JDBC メジャーバージョン番号を取得します。"  , "0" }
//		, { "getJDBCMinorVersion" , "このドライバの JDBC マイナーバージョン番号を取得します。"  , "0" }
//		, { "getMaxBinaryLiteralLength" , "このデータベースで、インラインバイナリリテラル中に入れられる 16 進数の最大文字数を取得します。"  , "0" }
//		, { "getMaxCatalogNameLength" , "このデータベースでの、カタログ名の最大文字数を取得します。"  , "0" }
//		, { "getMaxCharLiteralLength" , "このデータベースでの、キャラクタリテラルの最大文字数を取得します。"  , "0" }
//		, { "getMaxColumnNameLength" , "このデータベースでの、列名の最大文字数を取得します。"  , "0" }
//		, { "getMaxColumnsInGroupBy" , "このデータベースでの、GROUP BY 節中の列数の最大値を取得します。"  , "0" }
//		, { "getMaxColumnsInIndex" , "このデータベースでの、インデックス中の列数の最大値を取得します。"  , "0" }
//		, { "getMaxColumnsInOrderBy" , "このデータベースでの、ORDER BY 節中の列数の最大値を取得します。"  , "0" }
//		, { "getMaxColumnsInSelect" , "このデータベースでの、SELECT リスト中の列数の最大値を取得します。"  , "0" }
//		, { "getMaxColumnsInTable" , "このデータベースでの、テーブル中の列数の最大値を取得します。"  , "0" }
//		, { "getMaxConnections" , "このデータベースに対して可能な並行接続の最大数を取得します。"  , "0" }
//		, { "getMaxCursorNameLength" , "このデータベースでの、カーソル名の最大文字数を取得します。"  , "0" }
//		, { "getMaxIndexLength" , "このデータベースでの、インデックスの全部分を含む、インデックスの最大バイト数を取得します。"  , "0" }
//		, { "getMaxProcedureNameLength" , "このデータベースでの、プロシージャー名の最大文字数を取得します。"  , "0" }
//		, { "getMaxRowSize" , "このデータベースでの、1 行の最大バイト数を取得します。"  , "0" }
//		, { "getMaxSchemaNameLength" , "このデータベースでの、スキーマ名の最大文字数を取得します。"  , "0" }
//		, { "getMaxStatementLength" , "このデータベースでの、SQL 文の最大文字数を取得します。"  , "0" }
//		, { "getMaxStatements" , "このデータベースの同時にオープンできるアクティブな文の最大数を取得します。"  , "0" }
//		, { "getMaxTableNameLength" , "このデータベースでの、テーブル名の最大文字数を取得します。"  , "0" }
//		, { "getMaxTablesInSelect" , "このデータベースでの、SELECT 文の最大テーブル数を取得します。"  , "0" }
//		, { "getMaxUserNameLength" , "このデータベースでの、ユーザー名の最大文字数を取得します。"  , "0" }
//
//	// これらは、引数が必要なので、リフレクション処理は、とりあえず保留
//	//	, { "getClientInfoProperties" , "ドライバがサポートするクライアント情報プロパティーのリストを取得します。"  , "0" }
//	//	, { "getFunctionColumns" , "指定されたカタログのシステム関数またはユーザー関数のパラメータと返される型に関する記述を取得します。"  , "0" }
//	//	, { "getFunctions" , "指定されたカタログで使用可能なシステム関数およびユーザー関数に関する記述を取得します。"  , "0" }
//	};

	private static final MetaDataInvoke[] METHOD_LIST = new MetaDataInvoke[] {
		  new MetaDataInvoke( "getCatalogSeparator" 	,	0	,	-1	,	 "このデータベースがカタログ名とテーブル名のセパレータとして使用する String" 	)
		, new MetaDataInvoke( "getCatalogTerm" 	, 	0	, 	-1	, 	 "「catalog」に対するデータベースベンダーの推奨用語を取得します。 " 	)
		, new MetaDataInvoke( "getDatabaseProductName" 	, 	0	, 	-1	, 	 "このデータベース製品の名前を取得します。 " 	)
		, new MetaDataInvoke( "getDatabaseProductVersion" 	, 	0	, 	-1	, 	 "このデータベース製品のバージョンを取得します。 " 	)
		, new MetaDataInvoke( "getDriverMajorVersion" 	, 	0	, 	-1	, 	 "この JDBC ドライバのメジャーバージョンを取得します。 " 	)
		, new MetaDataInvoke( "getDriverMinorVersion" 	, 	0	, 	-1	, 	 "この JDBC ドライバのマイナーバージョンを取得します。 " 	)
		, new MetaDataInvoke( "getDriverName" 	, 	0	, 	-1	, 	 "この JDBC ドライバの名前を取得します。 " 	)
		, new MetaDataInvoke( "getDriverVersion" 	, 	0	, 	-1	, 	 "この JDBC ドライバのバージョンを String" 	)
		, new MetaDataInvoke( "getExtraNameCharacters" 	, 	0	, 	-1	, 	 "引用符で囲まれていない識別名に使用できるすべての「特殊」文字 (a-z、A-Z、0-9、および _ 以外) を取得します。 " 	)
		, new MetaDataInvoke( "getIdentifierQuoteString" 	, 	0	, 	-1	, 	 "SQL 識別子を引用するのに使用する文字列を取得します。 " 	)
		, new MetaDataInvoke( "getNumericFunctions" 	, 	0	, 	-1	, 	 "このデータベースで使用可能なコンマで区切った数学関数のリストを取得します。 " 	)
		, new MetaDataInvoke( "getProcedureTerm" 	, 	0	, 	-1	, 	 "「procedure」に対するデータベースベンダーの推奨用語を取得します。 " 	)
		, new MetaDataInvoke( "getSchemaTerm" 	, 	0	, 	-1	, 	 "「schema」に対するデータベースベンダーの推奨用語を取得します。 " 	)
		, new MetaDataInvoke( "getSearchStringEscape" 	, 	0	, 	-1	, 	 "ワイルドカード文字をエスケープするのに使用できる文字列を取得します。 " 	)
		, new MetaDataInvoke( "getSQLKeywords" 	, 	0	, 	-1	, 	 "このデータベースの SQL キーワードであって、SQL92 のキーワードではない、すべてのキーワードをコンマで区切ったリストを取得します。 " 	)
		, new MetaDataInvoke( "getStringFunctions" 	, 	0	, 	-1	, 	 "このデータベースで使用可能なコンマで区切った文字列関数のリストを取得します。 " 	)
		, new MetaDataInvoke( "getSystemFunctions" 	, 	0	, 	-1	, 	 "このデータベースで使用可能なコンマで区切ったシステム関数のリストを取得します。 " 	)
		, new MetaDataInvoke( "getTimeDateFunctions" 	, 	0	, 	-1	, 	 "このデータベースで使用可能な時間関数と日付関数をコンマで区切ったリストを取得します。 " 	)
		, new MetaDataInvoke( "getURL" 	, 	0	, 	-1	, 	 "この DBMS の URL を取得します。 " 	)
		, new MetaDataInvoke( "getUserName" 	, 	0	, 	-1	, 	 "このデータベースに記録されているユーザ名を取得します。 " 	)

		, new MetaDataInvoke( "getDefaultTransactionIsolation" 	, 	0	, 	-1	, 	 "このデータベースのデフォルトのトランザクション遮断レベルを取得します。 " 	)
		, new MetaDataInvoke( "getSQLStateType" 	, 	0	, 	-1	, 	 "Throwable.getSQLState によって返される SQLSTATE が X/Open (現在は Open Group) の SQL CLI であるか SQL99 であるかを示します。 " 	)
		, new MetaDataInvoke( "getResultSetHoldability" 	, 	0	, 	-1	, 	 "この ResultSet オブジェクトのデフォルトの保持機能を取得します。 " 	)

		, new MetaDataInvoke( "allProceduresAreCallable" 	, 	0	, 	-1	, 	 "getProcedures メソッドによって返されるすべてのプロシージャが、現在のユーザから呼び出せるかどうかを取得します。 " 	)
		, new MetaDataInvoke( "allTablesAreSelectable" 	, 	0	, 	-1	, 	 "getTables メソッドによって返されるすべてのテーブルが、現在のユーザによって使用できるかどうかを取得します。 " 	)
		, new MetaDataInvoke( "dataDefinitionCausesTransactionCommit" 	, 	0	, 	-1	, 	 "トランザクションのデータ定義文が、トランザクションを強制的にコミットさせるかどうかを取得します。 " 	)
		, new MetaDataInvoke( "dataDefinitionIgnoredInTransactions" 	, 	0	, 	-1	, 	 "このデータベースによって、トランザクションでデータ定義文が無視されるかどうかを取得します。 " 	)
		, new MetaDataInvoke( "deletesAreDetected" 	, 	1	, 	-1	, 	 "ResultSet.rowDeleted メソッドを呼び出すことによって可視の行が削除されたことを検出できるかどうかを取得します。 " 	)
		, new MetaDataInvoke( "doesMaxRowSizeIncludeBlobs" 	, 	0	, 	-1	, 	 "getMaxRowSize メソッドの戻り値が SQL データの型の LONGVARCHAR および LONGVARBINARY を含むかどうかを取得します。 " 	)
		, new MetaDataInvoke( "insertsAreDetected" 	, 	1	, 	-1	, 	 "ResultSet.rowInserted メソッドを呼び出すことによって可視の行が挿入されたことを検出できるかどうかを取得します。 " 	)
		, new MetaDataInvoke( "isCatalogAtStart" 	, 	0	, 	-1	, 	 "完全修飾されたテーブル名の開始部分 (または終了部分) にカタログが現れるかどうかを取得します。 " 	)
		, new MetaDataInvoke( "isReadOnly" 	, 	0	, 	-1	, 	 "このデータベースが読み込み専用モードかどうかを取得します。 " 	)
		, new MetaDataInvoke( "locatorsUpdateCopy" 	, 	0	, 	-1	, 	 "LOB への変更が、コピーに対して行われたのか、LOB に直接行われたのかを示します。 " 	)
		, new MetaDataInvoke( "nullPlusNonNullIsNull" 	, 	0	, 	-1	, 	 "このデータベースが、NULL 値と非 NULL 値の連結を NULL とするかどうかを取得します。 " 	)
		, new MetaDataInvoke( "nullsAreSortedAtEnd" 	, 	0	, 	-1	, 	 "NULL 値が、終了時にソート順にかかわらずソートされるかどうかを取得します。 " 	)
		, new MetaDataInvoke( "nullsAreSortedAtStart" 	, 	0	, 	-1	, 	 "NULL 値が、開始時にソート順にかかわらずソートされるかどうかを取得します。 " 	)
		, new MetaDataInvoke( "nullsAreSortedHigh" 	, 	0	, 	-1	, 	 "NULL 値が高位にソートされるかどうかを取得します。 " 	)
		, new MetaDataInvoke( "nullsAreSortedLow" 	, 	0	, 	-1	, 	 "NULL 値が下位にソートされるかどうかを取得します。 " 	)
		, new MetaDataInvoke( "othersDeletesAreVisible" 	, 	1	, 	-1	, 	 "他で行われた削除が可視かどうかを取得します。 " 	)
		, new MetaDataInvoke( "othersInsertsAreVisible" 	, 	1	, 	-1	, 	 "他で行われた挿入が可視かどうかを取得します。 " 	)
		, new MetaDataInvoke( "othersUpdatesAreVisible" 	, 	1	, 	-1	, 	 "他で行われた更新が可視かどうかを取得します。 " 	)
		, new MetaDataInvoke( "ownDeletesAreVisible" 	, 	1	, 	-1	, 	 "結果セット自身の削除が可視かどうかを取得します。 " 	)
		, new MetaDataInvoke( "ownInsertsAreVisible" 	, 	1	, 	-1	, 	 "結果セット自身の挿入が可視かどうかを取得します。 " 	)
		, new MetaDataInvoke( "ownUpdatesAreVisible" 	, 	1	, 	-1	, 	 "指定された ResultSet オブジェクトについて、結果セット自身の更新が可視かどうかを取得します。 " 	)
		, new MetaDataInvoke( "storesLowerCaseIdentifiers" 	, 	0	, 	-1	, 	 "このデータベースが、大文字小文字が混在する引用符なしの SQL 識別子を、大文字小文字を区別しないで処理し、小文字で格納するかどうかを取得します。 " 	)
		, new MetaDataInvoke( "storesLowerCaseQuotedIdentifiers" 	, 	0	, 	-1	, 	 "このデータベースが、大文字小文字が混在する引用符付きの SQL 識別子を、大文字小文字を区別しないで処理し、小文字で格納するかどうかを取得します。 " 	)
		, new MetaDataInvoke( "storesMixedCaseIdentifiers" 	, 	0	, 	-1	, 	 "このデータベースが、大文字小文字が混在する引用符なしの SQL 識別子を、大文字小文字を区別しないで処理し、大文字小文字混在で格納するかどうかを取得します。 " 	)
		, new MetaDataInvoke( "storesMixedCaseQuotedIdentifiers" 	, 	0	, 	-1	, 	 "このデータベースが、大文字小文字が混在する引用符付きの SQL 識別子を、大文字小文字を区別しないで処理し、大文字小文字混在で格納するかどうかを取得します。 " 	)
		, new MetaDataInvoke( "storesUpperCaseIdentifiers" 	, 	0	, 	-1	, 	 "このデータベースが、大文字小文字が混在する引用符なしの SQL 識別子を、大文字小文字を区別しないで処理し、大文字で格納するかどうかを取得します。 " 	)
		, new MetaDataInvoke( "storesUpperCaseQuotedIdentifiers" 	, 	0	, 	-1	, 	 "このデータベースが、大文字小文字が混在する引用符付きの SQL 識別子を、大文字小文字を区別しないで処理し、大文字で格納するかどうかを取得します。 " 	)
		, new MetaDataInvoke( "supportsAlterTableWithAddColumn" 	, 	0	, 	-1	, 	 "このデータベースによって、追加列のある ALTER TABLE がサポートされるかどうかを取得します。 " 	)
		, new MetaDataInvoke( "supportsAlterTableWithDropColumn" 	, 	0	, 	-1	, 	 "このデータベースによって、ドロップ列のある ALTER TABLE がサポートされるかどうかを取得します。 " 	)
		, new MetaDataInvoke( "supportsANSI92EntryLevelSQL" 	, 	0	, 	-1	, 	 "このデータベースによって、ANSI92 エントリレベルの SQL 文法がサポートされるかどうかを取得します。 " 	)
		, new MetaDataInvoke( "supportsANSI92FullSQL" 	, 	0	, 	-1	, 	 "このデータベースによって、ANSI92 完全レベルの SQL 文法がサポートされるかどうかを取得します。 " 	)
		, new MetaDataInvoke( "supportsANSI92IntermediateSQL" 	, 	0	, 	-1	, 	 "このデータベースによって、ANSI92 中間レベルの SQL 文法がサポートされるかどうかを取得します。 " 	)
		, new MetaDataInvoke( "supportsBatchUpdates" 	, 	0	, 	-1	, 	 "このデータベースによってバッチ更新がサポートされるかどうかを取得します。 " 	)
		, new MetaDataInvoke( "supportsCatalogsInDataManipulation" 	, 	0	, 	-1	, 	 "データ操作文でカタログ名を使用できるかどうかを取得します。 " 	)
		, new MetaDataInvoke( "supportsCatalogsInIndexDefinitions" 	, 	0	, 	-1	, 	 "インデックス定義文でカタログ名を使用できるかどうかを取得します。 " 	)
		, new MetaDataInvoke( "supportsCatalogsInPrivilegeDefinitions" 	, 	0	, 	-1	, 	 "特権定義文でカタログ名を使用できるかどうかを取得します。 " 	)
		, new MetaDataInvoke( "supportsCatalogsInProcedureCalls" 	, 	0	, 	-1	, 	 "プロシージャ呼び出し文でカタログ名を使用できるかどうかを取得します。 " 	)
		, new MetaDataInvoke( "supportsCatalogsInTableDefinitions" 	, 	0	, 	-1	, 	 "テーブル定義文でカタログ名を使用できるかどうかを取得します。 " 	)
		, new MetaDataInvoke( "supportsColumnAliasing" 	, 	0	, 	-1	, 	 "このデータベースによって列の別名がサポートされるかどうかを取得します。  " 	)
	//	, new MetaDataInvoke( "supportsConvertFunc" 	, 	0	, 	-1	, 	 "このデータベースによって、SQL の型間の CONVERT 関数がサポートされるかどうかを取得します。 " 	)
		, new MetaDataInvoke( "supportsCoreSQLGrammar" 	, 	0	, 	-1	, 	 "このデータベースによって、ODBC Core SQL 文法がサポートされるかどうかを取得します。 " 	)
		, new MetaDataInvoke( "supportsCorrelatedSubqueries" 	, 	0	, 	-1	, 	 "このデータベースによって照合関係サブクエリーがサポートされるかどうかを取得します。 " 	)
		, new MetaDataInvoke( "supportsDataDefinitionAndDataManipulationTransactions" 	, 	0	, 	-1	, 	 "このデータベースによって、トランザクションで、データ定義文とデータ操作文の両方がサポートされるかどうかを取得します。 " 	)
		, new MetaDataInvoke( "supportsDataManipulationTransactionsOnly" 	, 	0	, 	-1	, 	 "このデータベースによって、トランザクションでデータ操作文だけがサポートされるかどうかを取得します。 " 	)
		, new MetaDataInvoke( "supportsDifferentTableCorrelationNames" 	, 	0	, 	-1	, 	 "テーブル相互関係名がサポートされる場合、テーブルの名前と異なる名前であるという制限を付けるかどうかを取得します。 " 	)
		, new MetaDataInvoke( "supportsExpressionsInOrderBy" 	, 	0	, 	-1	, 	 "このデータベースによって、ORDER BY リスト中で式がサポートされるかどうかを取得します。 " 	)
		, new MetaDataInvoke( "supportsExtendedSQLGrammar" 	, 	0	, 	-1	, 	 "このデータベースによって、ODBC Extended SQL 文法がサポートされるかどうかを取得します。 " 	)
		, new MetaDataInvoke( "supportsFullOuterJoins" 	, 	0	, 	-1	, 	 "このデータベースによって、完全入れ子の外部結合がサポートされるかどうかを取得します。 " 	)
		, new MetaDataInvoke( "supportsGetGeneratedKeys" 	, 	0	, 	-1	, 	 "文が実行されたあとに自動生成キーを取得できるかどうかを取得します。 " 	)
		, new MetaDataInvoke( "supportsGroupBy" 	, 	0	, 	-1	, 	 "このデータベースによって、GROUP BY 節のフォームがサポートされるかどうかを取得します。 " 	)
		, new MetaDataInvoke( "supportsGroupByBeyondSelect" 	, 	0	, 	-1	, 	 "SELECT 文中のすべての列が GROUP BY 節に含まれるという条件で、このデータベースによって、GROUP BY 節で SELECT 文中にない列の使用がサポートされるかどうかを取得します。 " 	)
		, new MetaDataInvoke( "supportsGroupByUnrelated" 	, 	0	, 	-1	, 	 "このデータベースによって、GROUP BY 節で SELECT 文中にない列の使用がサポートされるかどうかを取得します。 " 	)
		, new MetaDataInvoke( "supportsIntegrityEnhancementFacility" 	, 	0	, 	-1	, 	 "このデータベースによって、SQL Integrity Enhancement Facility がサポートされるかどうかを取得します。 " 	)
		, new MetaDataInvoke( "supportsLikeEscapeClause" 	, 	0	, 	-1	, 	 "このデータベースによって、LIKE エスケープ節の指定がサポートされるかどうかを取得します。 " 	)
		, new MetaDataInvoke( "supportsLimitedOuterJoins" 	, 	0	, 	-1	, 	 "このデータベースによって、外部結合に関し、制限されたサポートが提供されるかどうかを取得します (supportsFullOuterJoins メソッドが true を返す場合は true)。 " 	)
		, new MetaDataInvoke( "supportsMinimumSQLGrammar" 	, 	0	, 	-1	, 	 "このデータベースによって、ODBC Minimum SQL 文法がサポートされるかどうかを取得します。 " 	)
		, new MetaDataInvoke( "supportsMixedCaseIdentifiers" 	, 	0	, 	-1	, 	 "このデータベースが、大文字小文字が混在する引用符なしの SQL 識別子を、大文字小文字を区別して処理し、大文字小文字混在で格納するかどうかを取得します。 " 	)
		, new MetaDataInvoke( "supportsMixedCaseQuotedIdentifiers" 	, 	0	, 	-1	, 	 "このデータベースが、大文字小文字が混在する引用符付きの SQL 識別子を、大文字小文字を区別して処理し、結果として大文字小文字混在で格納するかどうかを取得します。 " 	)
		, new MetaDataInvoke( "supportsMultipleOpenResults" 	, 	0	, 	-1	, 	 "CallableStatement オブジェクトから同時に返された複数の ResultSet オブジェクトを持つことが可能かどうかを取得します。 " 	)
		, new MetaDataInvoke( "supportsMultipleResultSets" 	, 	0	, 	-1	, 	 "このデータベースによって、execute メソッドの単一の呼び出しからの複数の ResultSet オブジェクトの取得がサポートされるかどうかを取得します。 " 	)
		, new MetaDataInvoke( "supportsMultipleTransactions" 	, 	0	, 	-1	, 	 "このデータベースが一度に複数のトランザクションを (異なった接続で) オープンできるかどうかを取得します。 " 	)
		, new MetaDataInvoke( "supportsNamedParameters" 	, 	0	, 	-1	, 	 "このデータベースによって、callable 文への名前付きパラメータがサポートされるかどうかを取得します。 " 	)
		, new MetaDataInvoke( "supportsNonNullableColumns" 	, 	0	, 	-1	, 	 "このデータベースの列を非 null として定義できるかどうかを取得します。 " 	)
		, new MetaDataInvoke( "supportsOpenCursorsAcrossCommit" 	, 	0	, 	-1	, 	 "このデータベースによって、コミット間でカーソルがオープンされたままの状態がサポートされるかどうかを取得します。 " 	)
		, new MetaDataInvoke( "supportsOpenCursorsAcrossRollback" 	, 	0	, 	-1	, 	 "このデータベースによって、ロールバック間でカーソルがオープンされたままの状態がサポートされるかどうかを取得します。 " 	)
		, new MetaDataInvoke( "supportsOpenStatementsAcrossCommit" 	, 	0	, 	-1	, 	 "このデータベースによって、コミット間で文がオープンされたままの状態がサポートされるかどうかを取得します。 " 	)
		, new MetaDataInvoke( "supportsOpenStatementsAcrossRollback" 	, 	0	, 	-1	, 	 "このデータベースによって、ロールバック間で文がオープンされたままの状態がサポートされるかどうかを取得します。 " 	)
		, new MetaDataInvoke( "supportsOrderByUnrelated" 	, 	0	, 	-1	, 	 "このデータベースによって、ORDER BY 節で SELECT 文中にない列の使用がサポートされるかどうかを取得します。 " 	)
		, new MetaDataInvoke( "supportsOuterJoins" 	, 	0	, 	-1	, 	 "このデータベースによって、外部結合のなんらかの形式がサポートされるかどうかを取得します。 " 	)
		, new MetaDataInvoke( "supportsPositionedDelete" 	, 	0	, 	-1	, 	 "このデータベースによって、位置指定された DELETE 文がサポートされるかどうかを取得します。 " 	)
		, new MetaDataInvoke( "supportsPositionedUpdate" 	, 	0	, 	-1	, 	 "このデータベースによって、位置指定された UPDATE 文がサポートされるかどうかを取得します。 " 	)
		, new MetaDataInvoke( "supportsResultSetConcurrency" 	, 	2	, 	-1	, 	 "このデータベースが、指定された結果セットの型と与えられた並行処理の種類の組み合わせをサポートするかどうかを取得します。 " 	)
		, new MetaDataInvoke( "supportsResultSetHoldability" 	, 	3	, 	-1	, 	 "このデータベースが、指定された結果セットの保持機能をサポートするかどうかを取得します。 " 	)
		, new MetaDataInvoke( "supportsResultSetType" 	, 	4	, 	ResultSet.TYPE_FORWARD_ONLY	, 	 "このデータベースが、[TYPE_FORWARD_ONLY]指定された結果セットの型をサポートするかどうかを取得します。 " 	)
		, new MetaDataInvoke( "supportsResultSetType" 	, 	4	, 	ResultSet.TYPE_SCROLL_INSENSITIVE	, 	 "このデータベースが、[TYPE_SCROLL_INSENSITIVE]指定された結果セットの型をサポートするかどうかを取得します。 " 	)
		, new MetaDataInvoke( "supportsResultSetType" 	, 	4	, 	ResultSet.TYPE_SCROLL_SENSITIVE	, 	 "このデータベースが、[TYPE_SCROLL_SENSITIVE]指定された結果セットの型をサポートするかどうかを取得します。 " 	)
		, new MetaDataInvoke( "supportsSavepoints" 	, 	0	, 	-1	, 	 "このデータベースによってセーブポイントがサポートされるかどうかを取得します。 " 	)
		, new MetaDataInvoke( "supportsSchemasInDataManipulation" 	, 	0	, 	-1	, 	 "データ操作文でスキーマ名を使用できるかどうかを取得します。 " 	)
		, new MetaDataInvoke( "supportsSchemasInIndexDefinitions" 	, 	0	, 	-1	, 	 "インデックス定義文でスキーマ名を使用できるかどうかを取得します。 " 	)
		, new MetaDataInvoke( "supportsSchemasInPrivilegeDefinitions" 	, 	0	, 	-1	, 	 "特権定義文でスキーマ名を使用できるかどうかを取得します。 " 	)
		, new MetaDataInvoke( "supportsSchemasInProcedureCalls" 	, 	0	, 	-1	, 	 "プロシージャ呼び出し文でスキーマ名を使用できるかどうかを取得します。 " 	)
		, new MetaDataInvoke( "supportsSchemasInTableDefinitions" 	, 	0	, 	-1	, 	 "テーブル定義文でスキーマ名を使用できるかどうかを取得します。 " 	)
		, new MetaDataInvoke( "supportsSelectForUpdate" 	, 	0	, 	-1	, 	 "このデータベースによって SELECT FOR UPDATE 文がサポートされるかどうかを取得します。 " 	)
		, new MetaDataInvoke( "supportsStatementPooling" 	, 	0	, 	-1	, 	 "このデータベースによって文のプールがサポートされるかどうかを取得します。 " 	)
		, new MetaDataInvoke( "supportsStoredProcedures" 	, 	0	, 	-1	, 	 "このデータベースによって、ストアドプロシージャエスケープ構文を使用するストアドプロシージャコールがサポートされるかどうかを判定します。 " 	)
		, new MetaDataInvoke( "supportsSubqueriesInComparisons" 	, 	0	, 	-1	, 	 "このデータベースによって、比較式中でサブクエリーがサポートされるかどうかを取得します。 " 	)
		, new MetaDataInvoke( "supportsSubqueriesInExists" 	, 	0	, 	-1	, 	 "このデータベースによって、EXISTS 式中でサブクエリーがサポートされるかどうかを取得します。 " 	)
		, new MetaDataInvoke( "supportsSubqueriesInIns" 	, 	0	, 	-1	, 	 "このデータベースによって、IN 文中でサブクエリーがサポートされるかどうかを取得します。 " 	)
		, new MetaDataInvoke( "supportsSubqueriesInQuantifieds" 	, 	0	, 	-1	, 	 "このデータベースによって、定量化された式中でサブクエリーがサポートされるかどうかを取得します。 " 	)
		, new MetaDataInvoke( "supportsTableCorrelationNames" 	, 	0	, 	-1	, 	 "このデータベースによってテーブル相互関係名がサポートされるかどうかを取得します。 " 	)
		, new MetaDataInvoke( "supportsTransactionIsolationLevel" 	, 	4	, 	Connection.TRANSACTION_READ_COMMITTED	, 	 "このデータベースが、指定されたトランザクション遮断レベルをサポートするかどうかを取得します。 " 	)
		, new MetaDataInvoke( "supportsTransactionIsolationLevel" 	, 	4	, 	Connection.TRANSACTION_READ_UNCOMMITTED	, 	 "このデータベースが、指定されたトランザクション遮断レベルをサポートするかどうかを取得します。 " 	)
		, new MetaDataInvoke( "supportsTransactionIsolationLevel" 	, 	4	, 	Connection.TRANSACTION_REPEATABLE_READ	, 	 "このデータベースが、指定されたトランザクション遮断レベルをサポートするかどうかを取得します。 " 	)
		, new MetaDataInvoke( "supportsTransactionIsolationLevel" 	, 	4	, 	Connection.TRANSACTION_SERIALIZABLE	, 	 "このデータベースが、指定されたトランザクション遮断レベルをサポートするかどうかを取得します。 " 	)
		, new MetaDataInvoke( "supportsTransactions" 	, 	0	, 	-1	, 	 "このデータベースによってトランザクションがサポートされるかどうかを取得します。 " 	)
		, new MetaDataInvoke( "supportsUnion" 	, 	0	, 	-1	, 	 "このデータベースによって SQL UNION がサポートされるかどうかを取得します。 " 	)
		, new MetaDataInvoke( "supportsUnionAll" 	, 	0	, 	-1	, 	 "このデータベースによって SQL UNION ALL がサポートされるかどうかを取得します。 " 	)
		, new MetaDataInvoke( "updatesAreDetected" 	, 	1	, 	-1	, 	 "ResultSet.rowUpdated メソッドを呼び出すことによって可視の行が更新されたことを検出できるかどうかを取得します。 " 	)
		, new MetaDataInvoke( "usesLocalFilePerTable" 	, 	0	, 	-1	, 	 "このデータベースが、各テーブルにファイルを使用するかどうかを取得します。 " 	)
		, new MetaDataInvoke( "usesLocalFiles" 	, 	0	, 	-1	, 	 "このデータベースが、ローカルファイルにテーブルを格納するかどうかを取得します。 " 	)

		, new MetaDataInvoke( "getAttributes" 	, 	72	, 	-1	, 	 "指定されたスキーマおよびカタログで使用可能なユーザ定義の型 (UDT) のための指定された型の指定された属性に関する記述を取得します。  " 	)
		, new MetaDataInvoke( "getBestRowIdentifier" 	, 	64	, 	-1	, 	 "行を一意に識別するテーブルの最適な列セットに関する記述を取得します。 " 	)
		, new MetaDataInvoke( "getCatalogs" 	, 	50	, 	-1	, 	 "このデータベースで使用可能なカタログ名を取得します。 " 	)
		, new MetaDataInvoke( "getColumnPrivileges" 	, 	61	, 	-1	, 	 "テーブルの列へのアクセス権に関する記述を取得します。  " 	)
		, new MetaDataInvoke( "getColumns" 	, 	61	, 	-1	, 	 "指定されたカタログで使用可能なテーブル列の記述を取得します。  " 	)
		, new MetaDataInvoke( "getCrossReference" 	, 	65	, 	-1	, 	 "指定された主キーテーブルの主キー列を参照する指定された外部のキーテーブル中の、外部のキー列に関する記述 (テーブルが別のキーをインポートする方法を記述) を取得します。 " 	)
		, new MetaDataInvoke( "getExportedKeys" 	, 	60	, 	-1	, 	 "指定されたテーブルの主キー列 (テーブルによってエクスポートされた外部キー) を参照する外部キー列に関する記述を取得します。 " 	)
		, new MetaDataInvoke( "getImportedKeys" 	, 	60	, 	-1	, 	 "テーブルの外部キー列 (テーブルによってインポートされる主キー) を参照する主キー列に関する記述を取得します。 " 	)
		, new MetaDataInvoke( "getIndexInfo" 	, 	63	, 	-1	, 	 "指定されたテーブルのインデックスと統計情報に関する記述を取得します。 " 	)
		, new MetaDataInvoke( "getPrimaryKeys" 	, 	60	, 	-1	, 	 "指定されたテーブルの主キー列の記述を取得します。 " 	)
		, new MetaDataInvoke( "getProcedureColumns" 	, 	81	, 	-1	, 	 "指定されたカタログのストアドプロシージャパラメータと結果列に関する記述を取得します。  " 	)
		, new MetaDataInvoke( "getProcedures" 	, 	80	, 	-1	, 	 "指定されたカタログで使用可能なストアドプロシージャに関する記述を取得します。  " 	)
		, new MetaDataInvoke( "getSchemas" 	, 	50	, 	-1	, 	 "このデータベースで使用可能なスキーマ名を取得します。 " 	)
		, new MetaDataInvoke( "getSuperTables" 	, 	60	, 	-1	, 	 "このデータベースの特定のスキーマで定義されているテーブル階層の説明を取得します。  " 	)
		, new MetaDataInvoke( "getSuperTypes" 	, 	70	, 	-1	, 	 "このデータベースの特定のスキーマで定義されているユーザ定義型 (UDT) 階層の説明を取得します。 " 	)
		, new MetaDataInvoke( "getTablePrivileges" 	, 	60	, 	-1	, 	 "カタログで使用可能な各テーブルに対するアクセス権に関する記述を取得します。 " 	)
		, new MetaDataInvoke( "getTables" 	, 	62	, 	-1	, 	 "指定されたカタログで使用可能なテーブルに関する記述を取得します。 " 	)
		, new MetaDataInvoke( "getTableTypes" 	, 	50	, 	-1	, 	 "このデータベースで使用可能なテーブルの型を取得します。 " 	)
		, new MetaDataInvoke( "getTypeInfo" 	, 	50	, 	-1	, 	 "このデータベースでサポートされているすべての標準 SQL の型に関する記述を取得します。 " 	)
		, new MetaDataInvoke( "getUDTs" 	, 	71	, 	-1	, 	 "特定のスキーマで定義されているユーザ定義型 (UDT) の説明を取得します。 " 	)
		, new MetaDataInvoke( "getVersionColumns" 	, 	60	, 	-1	, 	 "行の任意の値が変更された場合に、自動的に更新されるテーブルの列に関する記述を取得します。 " 	)

	// 5.5.3.2 (2012/06/08) JDk1.6 用のメソッドを追加します。
		, new MetaDataInvoke( "autoCommitFailureClosesAllResultSets" 	, 	0	, 	-1	, 	 "autoCommit が true の場合に、ResultSet が保持可能であっても、すべてのオープンされた ResultSet がクローズされたことを SQLException が示すかどうかを取得します。" 	)
		, new MetaDataInvoke( "supportsConvert" 	, 	0	, 	-1	, 	 "このデータベースによって、JDBC 型 fromType と toType の間の変換に使用される JDBC スカラー関数 CONVERT がサポートされるかどうかを取得します。" 	)
		, new MetaDataInvoke( "supportsStoredFunctionsUsingCallSyntax" 	, 	0	, 	-1	, 	 "このデータベースが、ストアドプロシージャーエスケープ構文を使用した、ユーザー定義関数またはベンダー関数の呼び出しをサポートするかどうかを取得します。" 	)
		, new MetaDataInvoke( "getDatabaseMajorVersion" 	, 	0	, 	-1	, 	 "基本となるデータベースのメジャーバージョン番号を取得します。" 	)
		, new MetaDataInvoke( "getDatabaseMinorVersion" 	, 	0	, 	-1	, 	 "基本となるデータベースのマイナーバージョン番号を取得します。" 	)
		, new MetaDataInvoke( "getJDBCMajorVersion" 	, 	0	, 	-1	, 	 "このドライバの JDBC メジャーバージョン番号を取得します。" 	)
		, new MetaDataInvoke( "getJDBCMinorVersion" 	, 	0	, 	-1	, 	 "このドライバの JDBC マイナーバージョン番号を取得します。" 	)
		, new MetaDataInvoke( "getMaxBinaryLiteralLength" 	, 	0	, 	-1	, 	 "このデータベースで、インラインバイナリリテラル中に入れられる 16 進数の最大文字数を取得します。" 	)
		, new MetaDataInvoke( "getMaxCatalogNameLength" 	, 	0	, 	-1	, 	 "このデータベースでの、カタログ名の最大文字数を取得します。" 	)
		, new MetaDataInvoke( "getMaxCharLiteralLength" 	, 	0	, 	-1	, 	 "このデータベースでの、キャラクタリテラルの最大文字数を取得します。" 	)
		, new MetaDataInvoke( "getMaxColumnNameLength" 	, 	0	, 	-1	, 	 "このデータベースでの、列名の最大文字数を取得します。" 	)
		, new MetaDataInvoke( "getMaxColumnsInGroupBy" 	, 	0	, 	-1	, 	 "このデータベースでの、GROUP BY 節中の列数の最大値を取得します。" 	)
		, new MetaDataInvoke( "getMaxColumnsInIndex" 	, 	0	, 	-1	, 	 "このデータベースでの、インデックス中の列数の最大値を取得します。" 	)
		, new MetaDataInvoke( "getMaxColumnsInOrderBy" 	, 	0	, 	-1	, 	 "このデータベースでの、ORDER BY 節中の列数の最大値を取得します。" 	)
		, new MetaDataInvoke( "getMaxColumnsInSelect" 	, 	0	, 	-1	, 	 "このデータベースでの、SELECT リスト中の列数の最大値を取得します。" 	)
		, new MetaDataInvoke( "getMaxColumnsInTable" 	, 	0	, 	-1	, 	 "このデータベースでの、テーブル中の列数の最大値を取得します。" 	)
		, new MetaDataInvoke( "getMaxConnections" 	, 	0	, 	-1	, 	 "このデータベースに対して可能な並行接続の最大数を取得します。" 	)
		, new MetaDataInvoke( "getMaxCursorNameLength" 	, 	0	, 	-1	, 	 "このデータベースでの、カーソル名の最大文字数を取得します。" 	)
		, new MetaDataInvoke( "getMaxIndexLength" 	, 	0	, 	-1	, 	 "このデータベースでの、インデックスの全部分を含む、インデックスの最大バイト数を取得します。" 	)
		, new MetaDataInvoke( "getMaxProcedureNameLength" 	, 	0	, 	-1	, 	 "このデータベースでの、プロシージャー名の最大文字数を取得します。" 	)
		, new MetaDataInvoke( "getMaxRowSize" 	, 	0	, 	-1	, 	 "このデータベースでの、1 行の最大バイト数を取得します。" 	)
		, new MetaDataInvoke( "getMaxSchemaNameLength" 	, 	0	, 	-1	, 	 "このデータベースでの、スキーマ名の最大文字数を取得します。" 	)
		, new MetaDataInvoke( "getMaxStatementLength" 	, 	0	, 	-1	, 	 "このデータベースでの、SQL 文の最大文字数を取得します。" 	)
		, new MetaDataInvoke( "getMaxStatements" 	, 	0	, 	-1	, 	 "このデータベースの同時にオープンできるアクティブな文の最大数を取得します。" 	)
		, new MetaDataInvoke( "getMaxTableNameLength" 	, 	0	, 	-1	, 	 "このデータベースでの、テーブル名の最大文字数を取得します。" 	)
		, new MetaDataInvoke( "getMaxTablesInSelect" 	, 	0	, 	-1	, 	 "このデータベースでの、SELECT 文の最大テーブル数を取得します。" 	)
		, new MetaDataInvoke( "getMaxUserNameLength" 	, 	0	, 	-1	, 	 "このデータベースでの、ユーザー名の最大文字数を取得します。" 	)

		, new MetaDataInvoke( "getClientInfoProperties"	, 	50	, 	-1	, 	"ドライバがサポートするクライアント情報プロパティーのリストを取得します。"	)
		, new MetaDataInvoke( "getFunctionColumns"	, 	81	, 	-1	, 	"指定されたカタログのシステム関数またはユーザー関数のパラメータと返される型に関する記述を取得します。"	)
		, new MetaDataInvoke( "getFunctions"	, 	80	, 	-1	, 	"指定されたカタログで使用可能なシステム関数およびユーザー関数に関する記述を取得します。"	)
	} ;

	private static final class MetaDataInvoke {
		private final String  method  ;
		private final int     type    ;			// 0:引数なし 1: 2: 3: 4:内部value
		private final Integer value   ;
		private final String  comment ;

		public MetaDataInvoke( String method,int type,int value,String comment ) {
			this.method  = method  ;
			this.type	 = type    ;
			this.value   = Integer.valueOf(value) ;
			this.comment = comment ;
		}

		public String getMethod() { return method; }

		public boolean isUseMethod( final String likeKey ) {
			return ( likeKey == null || method.indexOf( likeKey ) >= 0 );
		}

		public int getType() { return type; }

		public String getComment() { return comment; }

		public String getErrMsg() { return errMsg; }

		public Object invokeA( final DatabaseMetaData metaData , final Integer... vals  ) {
			Object obj = null ;
synchronized( metaData ) {
				try {
					if( type == 0 ) {
						obj = metaData.getClass().getMethod( method ).invoke(metaData);
					} else if( type == 1 || type == 3 ) {
						obj = metaData.getClass().getMethod( method,int.class ).invoke(metaData,vals[0]);
					} else if( type == 2 ) {
						obj = metaData.getClass().getMethod( method,int.class,int.class ).invoke(metaData,vals[0],vals[1]);
					} else if( type == 4 ) {
						obj = metaData.getClass().getMethod( method,int.class ).invoke(metaData,value);
					} else {
						errMsg = "*** リフレクションタイプなしエラー ***";
					}
				}
				catch( Throwable ex ) {
					errMsg = "*** 実行時エラー ***" + ex.getMessage();
					obj = null;				// "*** 実行時エラー ***"
				}
}
				if( obj == null && errMsg == null ) { errMsg = "サポートされない機能です。"; }
			return obj ;
		}

		// 50	引数なし
		// 60	(catalog, schema, tableName)
		// 70	(catalog, schema, typeName)
		// 80	(catalog, schema, procedureName)
		// 62	(catalog, schema, tableName, (String[])null)	// String[]
		// 71	(catalog, schema, typeName, (int[])null)		// int[]
		// 63	(catalog, schema, tableName, false, false)
		// 61	(catalog, schema, tableName, columnName)
		// 72	(catalog, schema, typeName, attributeName)
		// 81	(catalog, schema, procedureName, columnName)
		// 64	(catalog, schema, tableName, (int)scope, (boolean)nullable)
		// 65	(catalog, schema, tableName, (String)null, (String)null, (String)null)

		private static final int     scope    = 0;
		private static final boolean nullable = true;

		private String errMsg = null;

		public ResultSet invokeB( final DatabaseMetaData metaData , final String... vals  ) {
			ResultSet obj = null ;
synchronized( metaData ) {
			try {
				if( type == 50 ) {	// 引数なし
					obj = (ResultSet)metaData.getClass().getMethod( method ).invoke(metaData);
				} else if( type == 60 || type == 70 || type == 80 ) {
					obj = (ResultSet)metaData.getClass().getMethod( method,String.class,String.class,String.class ).invoke(metaData,vals[0],vals[1],vals[2]);
				} else if( type == 62 ) {
					obj = (ResultSet)metaData.getClass().getMethod( method,String.class,String.class,String.class,String[].class ).invoke(metaData,vals[0],vals[1],vals[2],null);
				} else if( type == 71 ) {
					obj = (ResultSet)metaData.getClass().getMethod( method,String.class,String.class,String.class,int[].class ).invoke(metaData,vals[0],vals[1],vals[2],null);
				} else if( type == 63 ) {
					obj = (ResultSet)metaData.getClass().getMethod( method,String.class,String.class,String.class,boolean.class,boolean.class ).invoke(metaData,vals[0],vals[1],vals[2],false,false);
				} else if( type == 61 || type == 72 || type == 81 ) {
					obj = (ResultSet)metaData.getClass().getMethod( method,String.class,String.class,String.class,String.class ).invoke(metaData,vals[0],vals[1],vals[2],vals[3]);
				} else if( type == 64 ) {
					obj = (ResultSet)metaData.getClass().getMethod( method,String.class,String.class,String.class,int.class,boolean.class ).invoke(metaData,vals[0],vals[1],vals[2],scope,nullable);
				} else if( type == 65 ) {
					obj = (ResultSet)metaData.getClass().getMethod( method,String.class,String.class,String.class,String.class,String.class,String.class ).invoke(metaData,vals[0],vals[1],vals[2],null,null,null);
				} else {
					errMsg = "*** リフレクションタイプなしエラー ***";
				}
			}
			catch( Throwable ex ) {
				errMsg = "*** 実行時エラー ***" + ex.getMessage();
				obj = null;				// "*** 実行時エラー ***"
			}
}
			if( obj == null && errMsg == null ) { errMsg = "サポートされない機能です。"; }
			return obj ;
		}
	}

//	private static final String getCatalogSeparator = "このデータベースがカタログ名とテーブル名のセパレータとして使用する String" ;
//	private static final String getCatalogTerm = "「catalog」に対するデータベースベンダーの推奨用語を取得します。 " ;
//	private static final String getDatabaseProductName = "このデータベース製品の名前を取得します。 " ;
//	private static final String getDatabaseProductVersion = "このデータベース製品のバージョンを取得します。 " ;
//	private static final String getDriverMajorVersion = "この JDBC ドライバのメジャーバージョンを取得します。 " ;
//	private static final String getDriverMinorVersion = "この JDBC ドライバのマイナーバージョンを取得します。 " ;
//	private static final String getDriverName = "この JDBC ドライバの名前を取得します。 " ;
//	private static final String getDriverVersion = "この JDBC ドライバのバージョンを String" ;
//	private static final String getExtraNameCharacters = "引用符で囲まれていない識別名に使用できるすべての「特殊」文字 (a-z、A-Z、0-9、および _ 以外) を取得します。 " ;
//	private static final String getIdentifierQuoteString = "SQL 識別子を引用するのに使用する文字列を取得します。 " ;
//	private static final String getNumericFunctions = "このデータベースで使用可能なコンマで区切った数学関数のリストを取得します。 " ;
//	private static final String getProcedureTerm = "「procedure」に対するデータベースベンダーの推奨用語を取得します。 " ;
//	private static final String getSchemaTerm = "「schema」に対するデータベースベンダーの推奨用語を取得します。 " ;
//	private static final String getSearchStringEscape = "ワイルドカード文字をエスケープするのに使用できる文字列を取得します。 " ;
//	private static final String getSQLKeywords = "このデータベースの SQL キーワードであって、SQL92 のキーワードではない、すべてのキーワードをコンマで区切ったリストを取得します。 " ;
//	private static final String getStringFunctions = "このデータベースで使用可能なコンマで区切った文字列関数のリストを取得します。 " ;
//	private static final String getSystemFunctions = "このデータベースで使用可能なコンマで区切ったシステム関数のリストを取得します。 " ;
	private static final String getTimeDateFunctions = "このデータベースで使用可能な時間関数と日付関数をコンマで区切ったリストを取得します。 " ;
//	private static final String getURL = "この DBMS の URL を取得します。 " ;
//	private static final String getUserName = "このデータベースに記録されているユーザ名を取得します。 " ;
//
//	private static final String getDefaultTransactionIsolation = "このデータベースのデフォルトのトランザクション遮断レベルを取得します。 " ;
//	private static final String getSQLStateType = "Throwable.getSQLState によって返される SQLSTATE が X/Open (現在は Open Group) の SQL CLI であるか SQL99 であるかを示します。 " ;
//	private static final String getResultSetHoldability = "この ResultSet オブジェクトのデフォルトの保持機能を取得します。 " ;
//
//	private static final String allProceduresAreCallable = "getProcedures メソッドによって返されるすべてのプロシージャが、現在のユーザから呼び出せるかどうかを取得します。 " ;
//	private static final String allTablesAreSelectable = "getTables メソッドによって返されるすべてのテーブルが、現在のユーザによって使用できるかどうかを取得します。 " ;
//	private static final String dataDefinitionCausesTransactionCommit = "トランザクションのデータ定義文が、トランザクションを強制的にコミットさせるかどうかを取得します。 " ;
//	private static final String dataDefinitionIgnoredInTransactions = "このデータベースによって、トランザクションでデータ定義文が無視されるかどうかを取得します。 " ;
//	private static final String deletesAreDetected = "ResultSet.rowDeleted メソッドを呼び出すことによって可視の行が削除されたことを検出できるかどうかを取得します。 " ;
//	private static final String doesMaxRowSizeIncludeBlobs = "getMaxRowSize メソッドの戻り値が SQL データの型の LONGVARCHAR および LONGVARBINARY を含むかどうかを取得します。 " ;
//	private static final String insertsAreDetected = "ResultSet.rowInserted メソッドを呼び出すことによって可視の行が挿入されたことを検出できるかどうかを取得します。 " ;
//	private static final String isCatalogAtStart = "完全修飾されたテーブル名の開始部分 (または終了部分) にカタログが現れるかどうかを取得します。 " ;
//	private static final String isReadOnly = "このデータベースが読み込み専用モードかどうかを取得します。 " ;
//	private static final String locatorsUpdateCopy = "LOB への変更が、コピーに対して行われたのか、LOB に直接行われたのかを示します。 " ;
//	private static final String nullPlusNonNullIsNull = "このデータベースが、NULL 値と非 NULL 値の連結を NULL とするかどうかを取得します。 " ;
//	private static final String nullsAreSortedAtEnd = "NULL 値が、終了時にソート順にかかわらずソートされるかどうかを取得します。 " ;
//	private static final String nullsAreSortedAtStart = "NULL 値が、開始時にソート順にかかわらずソートされるかどうかを取得します。 " ;
//	private static final String nullsAreSortedHigh = "NULL 値が高位にソートされるかどうかを取得します。 " ;
//	private static final String nullsAreSortedLow = "NULL 値が下位にソートされるかどうかを取得します。 " ;
//	private static final String othersDeletesAreVisible = "他で行われた削除が可視かどうかを取得します。 " ;
//	private static final String othersInsertsAreVisible = "他で行われた挿入が可視かどうかを取得します。 " ;
//	private static final String othersUpdatesAreVisible = "他で行われた更新が可視かどうかを取得します。 " ;
//	private static final String ownDeletesAreVisible = "結果セット自身の削除が可視かどうかを取得します。 " ;
//	private static final String ownInsertsAreVisible = "結果セット自身の挿入が可視かどうかを取得します。 " ;
//	private static final String ownUpdatesAreVisible = "指定された ResultSet オブジェクトについて、結果セット自身の更新が可視かどうかを取得します。 " ;
//	private static final String storesLowerCaseIdentifiers = "このデータベースが、大文字小文字が混在する引用符なしの SQL 識別子を、大文字小文字を区別しないで処理し、小文字で格納するかどうかを取得します。 " ;
//	private static final String storesLowerCaseQuotedIdentifiers = "このデータベースが、大文字小文字が混在する引用符付きの SQL 識別子を、大文字小文字を区別しないで処理し、小文字で格納するかどうかを取得します。 " ;
//	private static final String storesMixedCaseIdentifiers = "このデータベースが、大文字小文字が混在する引用符なしの SQL 識別子を、大文字小文字を区別しないで処理し、大文字小文字混在で格納するかどうかを取得します。 " ;
//	private static final String storesMixedCaseQuotedIdentifiers = "このデータベースが、大文字小文字が混在する引用符付きの SQL 識別子を、大文字小文字を区別しないで処理し、大文字小文字混在で格納するかどうかを取得します。 " ;
//	private static final String storesUpperCaseIdentifiers = "このデータベースが、大文字小文字が混在する引用符なしの SQL 識別子を、大文字小文字を区別しないで処理し、大文字で格納するかどうかを取得します。 " ;
//	private static final String storesUpperCaseQuotedIdentifiers = "このデータベースが、大文字小文字が混在する引用符付きの SQL 識別子を、大文字小文字を区別しないで処理し、大文字で格納するかどうかを取得します。 " ;
//	private static final String supportsAlterTableWithAddColumn = "このデータベースによって、追加列のある ALTER TABLE がサポートされるかどうかを取得します。 " ;
//	private static final String supportsAlterTableWithDropColumn = "このデータベースによって、ドロップ列のある ALTER TABLE がサポートされるかどうかを取得します。 " ;
//	private static final String supportsANSI92EntryLevelSQL = "このデータベースによって、ANSI92 エントリレベルの SQL 文法がサポートされるかどうかを取得します。 " ;
//	private static final String supportsANSI92FullSQL = "このデータベースによって、ANSI92 完全レベルの SQL 文法がサポートされるかどうかを取得します。 " ;
//	private static final String supportsANSI92IntermediateSQL = "このデータベースによって、ANSI92 中間レベルの SQL 文法がサポートされるかどうかを取得します。 " ;
//	private static final String supportsBatchUpdates = "このデータベースによってバッチ更新がサポートされるかどうかを取得します。 " ;
//	private static final String supportsCatalogsInDataManipulation = "データ操作文でカタログ名を使用できるかどうかを取得します。 " ;
//	private static final String supportsCatalogsInIndexDefinitions = "インデックス定義文でカタログ名を使用できるかどうかを取得します。 " ;
//	private static final String supportsCatalogsInPrivilegeDefinitions = "特権定義文でカタログ名を使用できるかどうかを取得します。 " ;
//	private static final String supportsCatalogsInProcedureCalls = "プロシージャ呼び出し文でカタログ名を使用できるかどうかを取得します。 " ;
//	private static final String supportsCatalogsInTableDefinitions = "テーブル定義文でカタログ名を使用できるかどうかを取得します。 " ;
//	private static final String supportsColumnAliasing = "このデータベースによって列の別名がサポートされるかどうかを取得します。  " ;
	private static final String supportsConvertFunc = "このデータベースによって、SQL の型間の CONVERT 関数がサポートされるかどうかを取得します。 " ;
//	private static final String supportsCoreSQLGrammar = "このデータベースによって、ODBC Core SQL 文法がサポートされるかどうかを取得します。 " ;
//	private static final String supportsCorrelatedSubqueries = "このデータベースによって照合関係サブクエリーがサポートされるかどうかを取得します。 " ;
//	private static final String supportsDataDefinitionAndDataManipulationTransactions = "このデータベースによって、トランザクションで、データ定義文とデータ操作文の両方がサポートされるかどうかを取得します。 " ;
//	private static final String supportsDataManipulationTransactionsOnly = "このデータベースによって、トランザクションでデータ操作文だけがサポートされるかどうかを取得します。 " ;
//	private static final String supportsDifferentTableCorrelationNames = "テーブル相互関係名がサポートされる場合、テーブルの名前と異なる名前であるという制限を付けるかどうかを取得します。 " ;
//	private static final String supportsExpressionsInOrderBy = "このデータベースによって、ORDER BY リスト中で式がサポートされるかどうかを取得します。 " ;
//	private static final String supportsExtendedSQLGrammar = "このデータベースによって、ODBC Extended SQL 文法がサポートされるかどうかを取得します。 " ;
//	private static final String supportsFullOuterJoins = "このデータベースによって、完全入れ子の外部結合がサポートされるかどうかを取得します。 " ;
//	private static final String supportsGetGeneratedKeys = "文が実行されたあとに自動生成キーを取得できるかどうかを取得します。 " ;
//	private static final String supportsGroupBy = "このデータベースによって、GROUP BY 節のフォームがサポートされるかどうかを取得します。 " ;
//	private static final String supportsGroupByBeyondSelect = "SELECT 文中のすべての列が GROUP BY 節に含まれるという条件で、このデータベースによって、GROUP BY 節で SELECT 文中にない列の使用がサポートされるかどうかを取得します。 " ;
//	private static final String supportsGroupByUnrelated = "このデータベースによって、GROUP BY 節で SELECT 文中にない列の使用がサポートされるかどうかを取得します。 " ;
//	private static final String supportsIntegrityEnhancementFacility = "このデータベースによって、SQL Integrity Enhancement Facility がサポートされるかどうかを取得します。 " ;
//	private static final String supportsLikeEscapeClause = "このデータベースによって、LIKE エスケープ節の指定がサポートされるかどうかを取得します。 " ;
//	private static final String supportsLimitedOuterJoins = "このデータベースによって、外部結合に関し、制限されたサポートが提供されるかどうかを取得します (supportsFullOuterJoins メソッドが true を返す場合は true)。 " ;
//	private static final String supportsMinimumSQLGrammar = "このデータベースによって、ODBC Minimum SQL 文法がサポートされるかどうかを取得します。 " ;
//	private static final String supportsMixedCaseIdentifiers = "このデータベースが、大文字小文字が混在する引用符なしの SQL 識別子を、大文字小文字を区別して処理し、大文字小文字混在で格納するかどうかを取得します。 " ;
//	private static final String supportsMixedCaseQuotedIdentifiers = "このデータベースが、大文字小文字が混在する引用符付きの SQL 識別子を、大文字小文字を区別して処理し、結果として大文字小文字混在で格納するかどうかを取得します。 " ;
//	private static final String supportsMultipleOpenResults = "CallableStatement オブジェクトから同時に返された複数の ResultSet オブジェクトを持つことが可能かどうかを取得します。 " ;
//	private static final String supportsMultipleResultSets = "このデータベースによって、execute メソッドの単一の呼び出しからの複数の ResultSet オブジェクトの取得がサポートされるかどうかを取得します。 " ;
//	private static final String supportsMultipleTransactions = "このデータベースが一度に複数のトランザクションを (異なった接続で) オープンできるかどうかを取得します。 " ;
//	private static final String supportsNamedParameters = "このデータベースによって、callable 文への名前付きパラメータがサポートされるかどうかを取得します。 " ;
//	private static final String supportsNonNullableColumns = "このデータベースの列を非 null として定義できるかどうかを取得します。 " ;
//	private static final String supportsOpenCursorsAcrossCommit = "このデータベースによって、コミット間でカーソルがオープンされたままの状態がサポートされるかどうかを取得します。 " ;
//	private static final String supportsOpenCursorsAcrossRollback = "このデータベースによって、ロールバック間でカーソルがオープンされたままの状態がサポートされるかどうかを取得します。 " ;
//	private static final String supportsOpenStatementsAcrossCommit = "このデータベースによって、コミット間で文がオープンされたままの状態がサポートされるかどうかを取得します。 " ;
//	private static final String supportsOpenStatementsAcrossRollback = "このデータベースによって、ロールバック間で文がオープンされたままの状態がサポートされるかどうかを取得します。 " ;
//	private static final String supportsOrderByUnrelated = "このデータベースによって、ORDER BY 節で SELECT 文中にない列の使用がサポートされるかどうかを取得します。 " ;
//	private static final String supportsOuterJoins = "このデータベースによって、外部結合のなんらかの形式がサポートされるかどうかを取得します。 " ;
//	private static final String supportsPositionedDelete = "このデータベースによって、位置指定された DELETE 文がサポートされるかどうかを取得します。 " ;
//	private static final String supportsPositionedUpdate = "このデータベースによって、位置指定された UPDATE 文がサポートされるかどうかを取得します。 " ;
//	private static final String supportsResultSetConcurrency = "このデータベースが、指定された結果セットの型と与えられた並行処理の種類の組み合わせをサポートするかどうかを取得します。 " ;
//	private static final String supportsResultSetHoldability = "このデータベースが、指定された結果セットの保持機能をサポートするかどうかを取得します。 " ;
//	private static final String supportsResultSetType = "このデータベースが、指定された結果セットの型をサポートするかどうかを取得します。 " ;
//	private static final String supportsSavepoints = "このデータベースによってセーブポイントがサポートされるかどうかを取得します。 " ;
//	private static final String supportsSchemasInDataManipulation = "データ操作文でスキーマ名を使用できるかどうかを取得します。 " ;
//	private static final String supportsSchemasInIndexDefinitions = "インデックス定義文でスキーマ名を使用できるかどうかを取得します。 " ;
//	private static final String supportsSchemasInPrivilegeDefinitions = "特権定義文でスキーマ名を使用できるかどうかを取得します。 " ;
//	private static final String supportsSchemasInProcedureCalls = "プロシージャ呼び出し文でスキーマ名を使用できるかどうかを取得します。 " ;
//	private static final String supportsSchemasInTableDefinitions = "テーブル定義文でスキーマ名を使用できるかどうかを取得します。 " ;
//	private static final String supportsSelectForUpdate = "このデータベースによって SELECT FOR UPDATE 文がサポートされるかどうかを取得します。 " ;
//	private static final String supportsStatementPooling = "このデータベースによって文のプールがサポートされるかどうかを取得します。 " ;
//	private static final String supportsStoredProcedures = "このデータベースによって、ストアドプロシージャエスケープ構文を使用するストアドプロシージャコールがサポートされるかどうかを判定します。 " ;
//	private static final String supportsSubqueriesInComparisons = "このデータベースによって、比較式中でサブクエリーがサポートされるかどうかを取得します。 " ;
//	private static final String supportsSubqueriesInExists = "このデータベースによって、EXISTS 式中でサブクエリーがサポートされるかどうかを取得します。 " ;
//	private static final String supportsSubqueriesInIns = "このデータベースによって、IN 文中でサブクエリーがサポートされるかどうかを取得します。 " ;
//	private static final String supportsSubqueriesInQuantifieds = "このデータベースによって、定量化された式中でサブクエリーがサポートされるかどうかを取得します。 " ;
//	private static final String supportsTableCorrelationNames = "このデータベースによってテーブル相互関係名がサポートされるかどうかを取得します。 " ;
//	private static final String supportsTransactionIsolationLevel = "このデータベースが、指定されたトランザクション遮断レベルをサポートするかどうかを取得します。 " ;
//	private static final String supportsTransactions = "このデータベースによってトランザクションがサポートされるかどうかを取得します。 " ;
//	private static final String supportsUnion = "このデータベースによって SQL UNION がサポートされるかどうかを取得します。 " ;
//	private static final String supportsUnionAll = "このデータベースによって SQL UNION ALL がサポートされるかどうかを取得します。 " ;
//	private static final String updatesAreDetected = "ResultSet.rowUpdated メソッドを呼び出すことによって可視の行が更新されたことを検出できるかどうかを取得します。 " ;
//	private static final String usesLocalFilePerTable = "このデータベースが、各テーブルにファイルを使用するかどうかを取得します。 " ;
//	private static final String usesLocalFiles = "このデータベースが、ローカルファイルにテーブルを格納するかどうかを取得します。 " ;

	private static final String getAttributes = "指定されたスキーマおよびカタログで使用可能なユーザ定義の型 (UDT) のための指定された型の指定された属性に関する記述を取得します。  " ;
//	private static final String getBestRowIdentifier = "行を一意に識別するテーブルの最適な列セットに関する記述を取得します。 " ;
//	private static final String getCatalogs = "このデータベースで使用可能なカタログ名を取得します。 " ;
//	private static final String getColumnPrivileges = "テーブルの列へのアクセス権に関する記述を取得します。  " ;
	private static final String getColumns = "指定されたカタログで使用可能なテーブル列の記述を取得します。  " ;
//	private static final String getCrossReference = "指定された主キーテーブルの主キー列を参照する指定された外部のキーテーブル中の、外部のキー列に関する記述 (テーブルが別のキーをインポートする方法を記述) を取得します。 " ;
//	private static final String getExportedKeys = "指定されたテーブルの主キー列 (テーブルによってエクスポートされた外部キー) を参照する外部キー列に関する記述を取得します。 " ;
//	private static final String getImportedKeys = "テーブルの外部キー列 (テーブルによってインポートされる主キー) を参照する主キー列に関する記述を取得します。 " ;
	private static final String getIndexInfo = "指定されたテーブルのインデックスと統計情報に関する記述を取得します。 " ;
//	private static final String getPrimaryKeys = "指定されたテーブルの主キー列の記述を取得します。 " ;
//	private static final String getProcedureColumns = "指定されたカタログのストアドプロシージャパラメータと結果列に関する記述を取得します。  " ;
//	private static final String getProcedures = "指定されたカタログで使用可能なストアドプロシージャに関する記述を取得します。  " ;
//	private static final String getSchemas = "このデータベースで使用可能なスキーマ名を取得します。 " ;
	private static final String getSuperTables = "このデータベースの特定のスキーマで定義されているテーブル階層の説明を取得します。  " ;
	private static final String getSuperTypes = "このデータベースの特定のスキーマで定義されているユーザ定義型 (UDT) 階層の説明を取得します。 " ;
//	private static final String getTablePrivileges = "カタログで使用可能な各テーブルに対するアクセス権に関する記述を取得します。 " ;
//	private static final String getTables = "指定されたカタログで使用可能なテーブルに関する記述を取得します。 " ;
//	private static final String getTableTypes = "このデータベースで使用可能なテーブルの型を取得します。 " ;
	private static final String getTypeInfo = "このデータベースでサポートされているすべての標準 SQL の型に関する記述を取得します。 " ;
//	private static final String getUDTs = "特定のスキーマで定義されているユーザ定義型 (UDT) の説明を取得します。 " ;
//	private static final String getVersionColumns = "行の任意の値が変更された場合に、自動的に更新されるテーブルの列に関する記述を取得します。 " ;

	// 5.5.3.2 (2012/06/08) JDk1.6 用のメソッドを追加します。
//	private static final String autoCommitFailureClosesAllResultSets = "autoCommit が true の場合に、ResultSet が保持可能であっても、すべてのオープンされた ResultSet がクローズされたことを SQLException が示すかどうかを取得します。" ;
//	private static final String supportsConvert = "このデータベースによって、JDBC 型 fromType と toType の間の変換に使用される JDBC スカラー関数 CONVERT がサポートされるかどうかを取得します。" ;
//	private static final String supportsStoredFunctionsUsingCallSyntax = "このデータベースが、ストアドプロシージャーエスケープ構文を使用した、ユーザー定義関数またはベンダー関数の呼び出しをサポートするかどうかを取得します。" ;
//	private static final String getDatabaseMajorVersion = "基本となるデータベースのメジャーバージョン番号を取得します。" ;
//	private static final String getDatabaseMinorVersion = "基本となるデータベースのマイナーバージョン番号を取得します。" ;
//	private static final String getJDBCMajorVersion = "このドライバの JDBC メジャーバージョン番号を取得します。" ;
//	private static final String getJDBCMinorVersion = "このドライバの JDBC マイナーバージョン番号を取得します。" ;
//	private static final String getMaxBinaryLiteralLength = "このデータベースで、インラインバイナリリテラル中に入れられる 16 進数の最大文字数を取得します。" ;
//	private static final String getMaxCatalogNameLength = "このデータベースでの、カタログ名の最大文字数を取得します。" ;
//	private static final String getMaxCharLiteralLength = "このデータベースでの、キャラクタリテラルの最大文字数を取得します。" ;
//	private static final String getMaxColumnNameLength = "このデータベースでの、列名の最大文字数を取得します。" ;
//	private static final String getMaxColumnsInGroupBy = "このデータベースでの、GROUP BY 節中の列数の最大値を取得します。" ;
//	private static final String getMaxColumnsInIndex = "このデータベースでの、インデックス中の列数の最大値を取得します。" ;
//	private static final String getMaxColumnsInOrderBy = "このデータベースでの、ORDER BY 節中の列数の最大値を取得します。" ;
//	private static final String getMaxColumnsInSelect = "このデータベースでの、SELECT リスト中の列数の最大値を取得します。" ;
//	private static final String getMaxColumnsInTable = "このデータベースでの、テーブル中の列数の最大値を取得します。" ;
//	private static final String getMaxConnections = "このデータベースに対して可能な並行接続の最大数を取得します。" ;
//	private static final String getMaxCursorNameLength = "このデータベースでの、カーソル名の最大文字数を取得します。" ;
//	private static final String getMaxIndexLength = "このデータベースでの、インデックスの全部分を含む、インデックスの最大バイト数を取得します。" ;
//	private static final String getMaxProcedureNameLength = "このデータベースでの、プロシージャー名の最大文字数を取得します。" ;
//	private static final String getMaxRowSize = "このデータベースでの、1 行の最大バイト数を取得します。" ;
//	private static final String getMaxSchemaNameLength = "このデータベースでの、スキーマ名の最大文字数を取得します。" ;
//	private static final String getMaxStatementLength = "このデータベースでの、SQL 文の最大文字数を取得します。" ;
//	private static final String getMaxStatements = "このデータベースの同時にオープンできるアクティブな文の最大数を取得します。" ;
//	private static final String getMaxTableNameLength = "このデータベースでの、テーブル名の最大文字数を取得します。" ;
//	private static final String getMaxTablesInSelect = "このデータベースでの、SELECT 文の最大テーブル数を取得します。" ;
//	private static final String getMaxUserNameLength = "このデータベースでの、ユーザー名の最大文字数を取得します。" ;

//	private static final String getClientInfoProperties = "ドライバがサポートするクライアント情報プロパティーのリストを取得します。" ;
//	private static final String getFunctionColumns = "指定されたカタログのシステム関数またはユーザー関数のパラメータと返される型に関する記述を取得します。" ;
//	private static final String getFunctions = "指定されたカタログで使用可能なシステム関数およびユーザー関数に関する記述を取得します。" ;

	/**
	 * このオブジェクトの文字列表現を返します。
	 * 基本的にデバッグ目的に使用します。
	 *
	 * @return このクラスの文字列表現
	 */
	@Override
	public String toString() {
		return org.opengion.fukurou.util.ToString.title( this.getClass().getName() )
				.println( "VERSION"			,VERSION		)
				.println( "dbid"			,dbid			)
				.println( "catalog"			,catalog		)
				.println( "schema"			,schema			)
				.println( "tableName"		,tableName		)
				.println( "columnName"		,columnName		)
				.println( "typeName"		,typeName		)
				.println( "procedureName"	,procedureName	)
				.println( "attributeName"	,attributeName	)
				.println( "rowNo"			,rowNo			)
				.println( "Other..."		,getAttributes().getAttribute() )
				.fixForm().toString() ;
	}
}
