/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.table;

import java.io.File;
import java.io.PrintWriter;
import java.util.Map;

import org.opengion.fukurou.db.DBUtil;
import org.opengion.fukurou.db.Transaction;			// 5.5.2.6 (2012/05/25)
import org.opengion.fukurou.util.ErrorMessage;
import org.opengion.fukurou.util.FileUtil;
import org.opengion.fukurou.util.FixLengthData;
import org.opengion.fukurou.util.StringUtil;
import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.db.AbstractTableFilter;
import org.opengion.hayabusa.db.DBTableModel;

/**
 * TableFilter_INDEX は、TableUpda インターフェースを継承した、DBTableModel 処理用の
 * 実装クラスです。
 *
 * ここでは、インデックス一覧の検索結果より、GF07 のインデックスカラム定義テーブルから
 * 必要な情報を取得し、インデックス作成スクリプトを作成します。
 * 出力ファイルは、テーブル名＋"C.sql" という命名規則で作成します。
 * 検索では、(SYSTEM_ID,TBLSYU,TABLE_NAME,TABLE_LABEL,INDEX_NAME,NAME_JA,INDTYPE,TABLESPACE_NAME,INITIAL_EXTENT,NEXT_EXTENT)
 * の項目を取得する必要があります。
 *
 * パラメータは、tableFilterタグの keys, vals にそれぞれ記述するか、BODY 部にCSS形式で記述します。
 * 【パラメータ】
 *  {
 *       DIR : {&#064;BASE_DIR}/sql/02_INDEX ;    出力ファイルの基準フォルダ(必須)
 *       XML : false ;                            XML出力を行うかどうか[true/false]を指定します(初期値:false)。
 *       DROP: false ;                            INDEX構文の前に、DROP構文を出力するかどうか[true/false]を指定します(初期値:false)。
 *  }
 *
 * @og.formSample
 * ●形式：
 *      select SYSTEM_ID,TBLSYU,TABLE_NAME,TABLE_LABEL,INDEX_NAME,NAME_JA,INDTYPE,TABLESPACE_NAME,INITIAL_EXTENT,NEXT_EXTENT from GF07
 *      ① &lt;og:tableFilter classId="INDEX" keys="DIR,XML" vals="{&#064;BASE_DIR}/sql/02_INDEX,false" /&gt;
 *
 *      ② &lt;og:tableFilter classId="INDEX" &gt;
 *               {
 *                   DIR : {&#064;BASE_DIR}/sql/02_INDEX ;
 *                   XML : false ;
 *                   DROP: false ;
 *               }
 *         &lt;/og:tableFilter&gt;
 *
 * @og.rev 5.6.6.0 (2013/07/05) keys の整合性チェックを追加
 *
 * @version  0.9.0  2000/10/17
 * @author   Kazuhiko Hasegawa
 * @since    JDK1.1,
 */
public class TableFilter_INDEX extends AbstractTableFilter {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "5.6.9.2 (2013/10/18)" ;

	/**
	 * keys の整合性チェックを行うための初期設定を行います。
	 *
	 * @og.rev 5.6.6.1 (2013/07/12) keys の整合性チェック対応
	 *
	 * @param	keysMap keys の整合性チェックを行うための Map
	 */
	@Override
	protected void init( final Map<String,String> keysMap ) {
		keysMap.put( "DIR"	, "出力ファイルの基準フォルダ(必須)"							);
		keysMap.put( "XML"	, "XML出力を行うかどうか[true/false]を指定(初期値:false)"		);
		keysMap.put( "DROP"	, "INDEX構文の前に、DROP構文を出力するかどうか(初期値:false)"	);
	}

	private static final String[] DBKEY = {"SYSTEM_ID","TBLSYU","TABLE_NAME","TABLE_LABEL","INDEX_NAME","NAME_JA","INDTYPE",
							"TABLESPACE_NAME","INITIAL_EXTENT","NEXT_EXTENT" };

	// 5.1.1.0 (2009/12/01) データのアクセス用の配列番号のIDを private ⇒ protected にします。
	/** データのアクセス用の配列番号 {@value} */
	protected static final int SYSTEM_ID		= 0;
	/** データのアクセス用の配列番号 {@value} */
	protected static final int TBLSYU			= 1;
	/** データのアクセス用の配列番号 {@value} */
	protected static final int TABLE_NAME		= 2;
	/** データのアクセス用の配列番号 {@value} */
	protected static final int TABLE_LABEL		= 3;	// GF02 の NAME_JA より JOIN
	/** データのアクセス用の配列番号 {@value} */
	protected static final int INDEX_NAME		= 4;
	/** データのアクセス用の配列番号 {@value} */
	protected static final int INDTYPE			= 6;
	/** データのアクセス用の配列番号 {@value} */
	protected static final int TABLESPACE_NAME	= 7;
	/** データのアクセス用の配列番号 {@value} */
	protected static final int INITIAL_EXTENT	= 8;
	/** データのアクセス用の配列番号 {@value} */
	protected static final int NEXT_EXTENT		= 9;

	// 5.1.1.2 (2009/12/10)
//	private static final String GF07_SEL = "select CLM"
//											+ " from GF07"
//											+ " where SYSTEM_ID=? and TBLSYU=? and TABLE_NAME=? and INDEX_NAME=?"
//											+ " and   FGJ='1'"
//											+ " order by SEQNO" ;
	private static final String GF07_SEL = "select A.CLM, B.USE_LENGTH"
											+ " from GF07 A left outer join GF05 B"
											+ " on    A.SYSTEM_ID  = B.SYSTEM_ID"
											+ " and   A.TBLSYU     = B.TBLSYU"
											+ " and   A.TABLE_NAME = B.TABLE_NAME"
											+ " and   A.CLM        = B.CLM"
											+ " and   B.FGJ        = '1'"
											+ " where A.SYSTEM_ID=? and A.TBLSYU=? and A.TABLE_NAME=? and A.INDEX_NAME=?"
											+ " and   A.FGJ='1'"
											+ " order by A.SEQNO" ;

//	private static final String ENCODE = "Windows-31J" ;
	private static final String ENCODE = "UTF-8" ; // 4.3.6.6 (2009/05/15)
//	private static final String CR     = HybsSystem.CR ;	// 5.1.1.0 (2009/12/01) CR 定義をAbstractTableFilterで行う。

	private static final String CMNT  = "************************************************************************" ;

	private static final int X = FixLengthData.X ;
	private static final int S = FixLengthData.S ;
	private static final int K = FixLengthData.K ;

	/** 各種定数 */
	protected static final String XML_START_TAG	= "<?xml version='1.0' encoding='UTF-8'?>" + CR + "<ROWSET tableName='xxx'>";
	protected static final String XML_END_TAG	= "</ROWSET>";
	protected static final String EXEC_START_TAG= "<EXEC_SQL>";
	protected static final String EXEC_END_TAG	= "</EXEC_SQL>";

	/** XML形式かどうか  */
	protected boolean		isXml				= false; // 4.3.7.0 (2009/06/01)

	/**
	 * DBTableModel処理を実行します。
	 *
	 * @og.rev 3.8.7.0 (2006/12/15) アクセスログ取得の為,ApplicationInfoオブジェクトを設定
	 * @og.rev 4.0.0.0 (2007/11/28) メソッドの戻り値をチェックします。
	 * @og.rev 4.3.7.0 (2009/06/01) XML出力機能追加
	 * @og.rev 5.1.1.0 (2009/12/01) XML_START_TAG に、tableName をセットします。
	 * @og.rev 5.1.9.0 (2010/08/01) Transaction 対応
	 * @og.rev 5.5.2.6 (2012/05/25) protected変数を、private化したため、getterメソッドで取得するように変更
	 * @og.rev 5.6.9.2 (2013/10/18) INDEXを作成する前に、削除構文を入れるかどうかを指定。
	 *
	 * @return	実行結果のテーブルモデル
	 */
	public DBTableModel execute() {
		DBTableModel table = getDBTableModel();		// 5.5.2.6 (2012/05/25) インターフェースにgetterメソッド追加

		isXml  = StringUtil.nval( getValue( "XML" ), false );

		// 5.6.9.2 (2013/10/18) DROP構文を出力するかどうか 
		boolean isDrop = StringUtil.nval( getValue( "DROP" ), false );

		int[] clmNo = getTableColumnNo( DBKEY );
		int rowCnt = table.getRowCount();

		File dir = new File( getValue( "DIR" ) );
		if( ! dir.exists() && ! dir.mkdirs() ) {
			String errMsg = "所定のフォルダが作成できませんでした。[" + dir + "]" ;
			// 4.3.4.4 (2009/01/01)
			throw new HybsSystemException( errMsg );
		}

		String[] data  = null;
		String bkTableName = null;
		PrintWriter writer = null;
		Transaction tran = getTransaction();	// 5.5.2.6 (2012/05/25)
		for( int row=0; row<rowCnt; row++ ) {
			String tableName = null;
	//		String tableLbl  = null;
			String indexName = null;
			try {
				data  = table.getValues( row );
				String systemId  = data[clmNo[SYSTEM_ID]];
				String tblsyu    = data[clmNo[TBLSYU]];
				tableName = data[clmNo[TABLE_NAME]];
	//			tableLbl  = data[clmNo[TABLE_LABEL]];
				indexName = data[clmNo[INDEX_NAME]];

				// テーブルがキーブレイクすると、セーブファイルを切り替える。
				if( ! tableName.equals( bkTableName ) ) {
					if( writer != null ) {
						if( isXml ) { writer.println( XML_END_TAG ); }
						writer.close();
					}
					bkTableName = tableName;
					writer = FileUtil.getPrintWriter( new File( dir,tableName + ( isXml ? "C.xml" : "C.sql" ) ),ENCODE );
					if( isXml ) { writer.println( XML_START_TAG.replace( "xxx",tableName ) ); }		// 5.1.1.0 (2009/12/01) tableName をセット
					writer.print( makeHeadLine( clmNo,data ) );
				}

				String[] vals = new String[] { systemId,tblsyu,tableName,indexName };
//				String[][] gf07 = DBUtil.dbExecute( GF07_SEL,vals,appInfo );
				String[][] gf07 = DBUtil.dbExecute( GF07_SEL,vals,tran );	// 5.1.9.0 (2010/08/01) Transaction 対応
				if( gf07.length == 0 ) {
					System.out.println( "TABLE=[" + tableName + "],INDEX=[" + indexName + "] is Not Found!" );
					continue;
				}
				// テーブルに対するカラム列
				StringBuilder buf = new StringBuilder() ;
				for( int j=0; j<gf07.length; j++ ) {
					// 5.1.1.2 (2009/12/10)
//					buf.append( gf07[j][0] ).append( "," );
					buf.append( makeIndexClmStr( gf07[j][0], gf07[j][1] ) ).append( "," );
				}
				buf.deleteCharAt( buf.length()-1 );			// 最後の "," を取り除く処理

				// 5.6.9.2 (2013/10/18) DROP構文を出力する。
				if( isDrop ) {
					writer.print( makeDropLine( clmNo,data ) );
				}

				String clms = buf.toString();
				writer.print( makeLineList( clmNo,data,clms ) );
				writer.println( makeEndLine( clmNo,data ) );
			}
			catch( RuntimeException ex ) {
				ErrorMessage errMessage = makeErrorMessage( "TableFilter_INDEX Error",ErrorMessage.NG );
				errMessage.addMessage( row+1,ErrorMessage.NG,"INDEX",ex.getMessage() );
				errMessage.addMessage( row+1,ErrorMessage.NG,"INDEX",StringUtil.array2csv( data ) );
				errMessage.addMessage( row+1,ErrorMessage.NG,"INDEX","TABLE=[" + tableName + "],INDEX=[" + indexName + "]" );
				// BAT から呼び出す場合があるため、標準エラー出力にも情報を出しておきます。
				System.out.println( errMessage );
			}
		}
		if( isXml ) { writer.println( XML_END_TAG ); }
		if( writer != null ) { writer.close(); }

		return table;
	}

	/**
	 * ヘッダー部分の処理を実行します。
	 *
	 * @og.rev 5.6.6.0 (2013/07/05) FixLengthData の簡易コンストラクタを使用
	 *
	 * @param	clmNo	カラム番号配列
	 * @param	data	１行分のデータ配列
	 *
	 * @return	ヘッダー部分の文字列
	 */
	protected String makeHeadLine( final int[] clmNo,final String[] data ) {
//		String LINE1 = data[clmNo[TABLE_NAME]] ;
		String tableName = data[clmNo[TABLE_NAME]];
		String LINE1 = tableName + " ( " + data[clmNo[TABLE_LABEL]] + " )" ;
		String LINE2 = "Created : " + HybsSystem.getDate() ;

		// 5.6.6.0 (2013/07/05) FixLengthData の簡易コンストラクタを使用
//		FixLengthData fixData = new FixLengthData(3);

		int[] addLen = new int[] { 0,0,0 };	// 各データ間のスペース
		int[] type   = new int[] { X,K,X };	// 各データの種別 X:半角 S:空白前埋め K:全角混在
//		fixData.setAddLength( addLen );
//		fixData.setType( type );
		FixLengthData fixData = new FixLengthData( addLen,type );

		String[][] outData = new String[][] {
			{ "/**",	CMNT ,	"**/" },
			{ "/* ",	LINE1,	" */" },
			{ "/* ",	LINE2,	" */" },
			{ "/**",	CMNT ,	"**/" },
		};

		fixData.addAllListData( outData );

		return fixData.getAllFixData();

//		for( int i=0; i<outData.length; i++ ) {
//			fixData.addListData( outData[i] );
//		}

//		StringBuilder buf = new StringBuilder();
//		for( int i=0; i<outData.length; i++ ) {
//			buf.append( fixData.getFixData( i ) ).append( CR );
//		}

//		return buf.toString();
	}

	/**
	 * インデックス作成の処理を実行します。
	 *
	 * @og.rev 5.3.8.0 (2011/08/01) プライマリキー対応
	 * @og.rev 5.6.9.2 (2013/10/18) INDTYPE で、その他ではなく、2:通常 で判断する。
	 *
	 * @param	clmNo	カラム番号配列
	 * @param	data	１行分のデータ配列
	 * @param   clms	カラム名(CSV形式)
	 *
	 * @return	作成された１行分の文字列
	 */
	protected String makeLineList( final int[] clmNo,final String[] data,final String clms ) {
		String tableName = data[clmNo[TABLE_NAME]];
		String indexName = data[clmNo[INDEX_NAME]];
		String idxtype   = data[clmNo[INDTYPE]];

		StringBuilder buf = new StringBuilder();

		buf.append( CR );		// 先頭に、改行を入れておきます。
		if( isXml ) { buf.append( EXEC_START_TAG ).append( CR ); }

		// 5.3.8.0 (2011/08/01) プライマリキー対応
		if( "0".equals( idxtype ) ) {	// 0:プライマリキー
			buf.append( "ALTER TABLE " ).append( tableName ).append( " ADD CONSTRAINT " );
			buf.append( indexName ).append( " PRIMARY KEY ( " ).append( clms );
			buf.append( " )" );
		}
		else if( "1".equals( idxtype ) ) {	// 1:ユニークキー
			buf.append( "ALTER TABLE " ).append( tableName ).append( " ADD CONSTRAINT " );
			buf.append( indexName ).append( " UNIQUE( " ).append( clms );
			buf.append( " )" );
		}
		// 5.6.9.2 (2013/10/18) INDTYPE で、その他ではなく、2:通常 で判断する。
		else if( "2".equals( idxtype ) ) {	// 2:通常
			buf.append( "CREATE INDEX " ).append( indexName ).append( " ON " );
			buf.append( tableName ).append( "( " ).append( clms );
			buf.append( " )" );
		}
		else {
			String errMsg = "INDTYPE が、0,1,2 以外の値が使われています。INDTYPE=[" + idxtype + "]"
							+ " TABLE_NAME=[" + tableName + "] INDEX_NAME=[" + indexName + "]" ;
			System.out.println( errMsg );
		}

		return buf.toString();
	}

	/**
	 * 定義の最後の部分の処理を実行します。
	 *
	 * @og.rev 5.3.9.0 (2011/09/01) プライマリキー対応２
	 *
	 * @param	clmNo	カラム番号配列
	 * @param	data	１行分のデータ配列
	 *
	 * @return	定義の最後の部分
	 */
	protected String makeEndLine( final int[] clmNo,final String[] data ) {
		StringBuilder buf = new StringBuilder();
		buf.append( CR );		// 先頭に、改行を入れておきます。

		String idxtype   = data[clmNo[INDTYPE]];
//		if( "1".equals( idxtype ) ) {	// プライマリキー
		if( "0".equals( idxtype ) || "1".equals( idxtype ) ) {	// 0:プライマリキー , 1:ユニークキー
			buf.append( "USING INDEX " );
		}

		buf.append( "TABLESPACE " ).append( data[clmNo[TABLESPACE_NAME]] ).append( CR );
		buf.append( "STORAGE( INITIAL " ).append(  data[clmNo[INITIAL_EXTENT]] );
		buf.append( "K NEXT " ).append(  data[clmNo[NEXT_EXTENT]] );
		buf.append( "K PCTINCREASE 0 )" );

		if( isXml )	{ buf.append( CR ).append( EXEC_END_TAG ); }
		else		{ buf.append( ";" ); }

		return buf.toString();
	}

	/**
	 * インデックス削除の構文を、作成します。
	 *
	 * @og.rev 5.6.9.2 (2013/10/18) 新規作成
	 *
	 * @param	clmNo	カラム番号配列
	 * @param	data	１行分のデータ配列
	 *
	 * @return	作成された１行分の文字列
	 */
	protected String makeDropLine( final int[] clmNo,final String[] data ) {
		String tableName = data[clmNo[TABLE_NAME]];
		String indexName = data[clmNo[INDEX_NAME]];
		String idxtype   = data[clmNo[INDTYPE]];

		StringBuilder buf = new StringBuilder();

		buf.append( CR );		// 先頭に、改行を入れておきます。
		if( isXml ) { buf.append( EXEC_START_TAG ).append( CR ); }

		// 0:プライマリキー , 1:ユニークキー
		if( "0".equals( idxtype ) || "1".equals( idxtype ) ) {
			buf.append( "ALTER TABLE " ).append( tableName ).append( " DROP CONSTRAINT " );
			buf.append( indexName );
		}
		// 5.6.9.2 (2013/10/18) INDTYPE で、その他ではなく、2:通常 で判断する。
		else if( "2".equals( idxtype ) ) {	// 2:通常
			buf.append( "DROP INDEX " ).append( indexName );
		}
	//  一連の処理で、makeLineList ですでにエラーが出ているハズなので、ここでは出しません。
	//	else {
	//		String errMsg = "INDTYPE が、0,1,2 以外の値が使われています。INDTYPE=[" + idxtype + "]"
	//						+ " TABLE_NAME=[" + tableName + "] INDEX_NAME=[" + indexName + "]" ;
	//		System.out.println( errMsg );
	//	}

		if( isXml )	{ buf.append( CR ).append( EXEC_END_TAG ); }
		else		{ buf.append( ";" ); }

		return buf.toString();
	}

	/**
	 * インデックスを作成するための文字列を返します。
	 * 通常、カラム名をそのまま返します。
	 * 但し、唯一、MySQLの場合、500バイト以上のカラムについては、TEXTで定義しており、
	 * この場合、インデックス化するバイト数(最大255)を指定する必要があります。
	 * このケースに対応するため、カラム名とバイト数を元に判定し、部分インデックスを
	 * 作成するための文字列を作成します。
	 *
	 * @param	clm		カラム名
	 * @param	useLen	カラムのバイト数
	 *
	 * @return	インデックスカラムの文字列
	 * @see TableFilter_INDEX_MYSQL
	 */
	protected String makeIndexClmStr( final String clm, final String useLen ) {
		return clm;
	}
}
