/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.db;

import org.opengion.hayabusa.common.HybsSystemException;

/**
 * DBTableModel の１行分の特性を管理するクラスです。
 * このクラスに１行分の特性をセットして、DBTableModelImpl#setDBRowHeader( int aRow, DBRowHeader rowHeader )
 * を用いて、ヘッダーデータを登録します。
 * なお、DBTableModel インターフェース 自体には、ヘッダー情報を登録する手段はありませんので
 * DBTableModel 実装クラスの独自メソッドで、ヘッダーデータを与える必要があります。
 *
 * @og.rev 3.5.6.0 (2004/06/18) クラスのパッケージプライベート化を行います。
 * @og.group テーブル管理
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
class DBRowHeader {

	private boolean  writable ;
	private boolean  checked  ;
	private String[] bkupdata ;
	private String   type     ;
	private String   marker   ;

	/**
	 * パッケージプライベートな、コンストラクター
	 *
	 */
	DBRowHeader() {
		initialise();
	}

	/**
	 * データを初期化します。
	 *
	 */
	void clear() {
		initialise();
	}

	private void initialise() {
		writable = DBTableModel.DEFAULT_WRITABLE;
		checked  = DBTableModel.DEFAULT_CHECKED ;
		bkupdata = null ;
		type     = "";
		marker   = "";
	}

	/**
	 * 書込み許可を返します。
	 *
	 * @return  書込み可能(true)／不可能(false)
	 */
	boolean isWritable() {
		return writable;
	}

	/**
	 * 行が書き込み可能かどうかをセットします。
	 * デフォルト/およびなにも設定しない場合は, DBTableModel.DEFAULT_WRITABLE が
	 * 与えられています。
	 * これが true の場合は,書込み許可です。(チェックボックスを表示)
	 * false の場合は,書込み不許可(チェックボックスは表示されません。)
	 *
	 * @og.rev 3.1.1.0 (2003/03/28) 同期メソッド(synchronized付き)を非同期に変更する。
	 *
	 * @param   rw 書込み可能(true)／不可能(false)
	 */
	void setWritable( final boolean rw ) {
		writable = rw;
	}

	/**
	 * 書き込み可能な行(rowWritable == true)のチェックボックスに対して
	 * 初期値を 選択済みか、非選択済みかを返します。
	 *
	 * @return 選択済みの場合は、true , そうでないときは、flse を返します。
	 */
	boolean isChecked() {
		return checked;
	}

	/**
	 * 書き込み可能な行(rowWritable == true)のチェックボックスに対して
	 * 初期値を 選択済みにするか、非選択済みにするかを指定します。
	 *
	 * @og.rev 3.1.1.0 (2003/03/28) 同期メソッド(synchronized付き)を非同期に変更する。
	 *
	 * @param   rw チェックON(true)／チェックOFF(false)
	 */
	void setChecked( final boolean rw ) {
		checked = rw;
	}

	/**
	 * row 単位に変更タイプ(追加/変更/削除)をセットします。
	 * タイプは始めに一度登録するとそれ以降に変更はかかりません。
	 * つまり、始めに 追加で作成したデータは、その後変更があっても追加のままです。
	 * なにも変更されていない場合は, ""(ゼロストリング)の状態です。
	 *
	 * @og.rev 3.1.1.0 (2003/03/28) 同期メソッド(synchronized付き)を非同期に変更する。
	 *
	 * @param   modType  変更タイプ(追加/変更/削除)
	 */
	void setType( final String modType ) {
		if( type == null || type.isEmpty() ) {
			type = modType ;
		}
	}

	/**
	 * row 単位に変更タイプ(追加/変更/削除)を返します。
	 * なにも変更されていない場合は, ""(ゼロストリング)の状態です。
	 *
	 * @return   変更タイプ(追加/変更/削除)
	 *
	 */
	String getType() {
		return type;
	}

	/**
	 * row 単位にバックアップデータを保管します。
	 * このデータは,最初に１度しか設定できません。つまり、同一行を何度も
	 * 書き換えたとしても、初めの状態にしか戻すことは出来ません。
	 * このデータを 履歴を管理できるようにすれば, 書き換えた逆順に
	 * データを取り出すことが可能になりますが,現在は対応していません。
	 * データ設定時に、配列コピーを取得します。
	 *
	 * @og.rev 3.1.1.0 (2003/03/28) 同期メソッド(synchronized付き)を非同期に変更する。
	 * @og.rev 3.5.6.0 (2004/06/18) 配列をそのまま受け取らずに、arraycopy してセットする。
	 *
	 * @param	bkup	バックアップデータ配列(可変長引数)
	 */
//	void setBackupData( final String[] bkup ) {
	void setBackupData( final String... bkup ) {
		if( bkupdata == null ) {
			final int size = bkup.length ;
			bkupdata = new String[size];
			System.arraycopy( bkup,0,bkupdata,0,size );
		}
	}

	/**
	 * row 単位にバックアップデータを取り出します。
	 * 最初に登録されたデータを返します。
	 * まだ、なにも修正(追加/変更/削除)が行われていない場合は、null を返します。
	 *
	 * @og.rev 3.5.6.0 (2004/06/18) 配列をそのまま返さずに、clone して返します。
	 * @og.rev 3.6.0.0 (2004/09/22) バックアップデータが存在しない場合はエラーとする。
	 *
	 * @return   バックアップデータ(存在しない場合は、エラー)
	 */
	String[] getBackupData() {
		if( bkupdata != null ) {
			return bkupdata.clone();
		}

		final String errMsg = "バックアップデータが存在しません。" +
					"事前に、getType() で更新状態を取得後、" +
					"この処理の実行可否を判断して下さい。type=[" + type + "]";
		throw new HybsSystemException( errMsg );
	}

	/**
	 * row 単位に指定のマーカーを付けます。
	 * マーカーは、表示等に行に色をつけたりするのに使用できます。
	 * 初期値は、空文字 "" です。
	 *
	 * @param	marker	マーカー文字列
	 *
	 */
	void setMarker( final String marker ) {
		if( marker == null ) { this.marker = ""; }
		else {                 this.marker = marker; }
	}

	/**
	 * row 単位に指定されたマーカーを返します。
	 * マーカーは、表示等に行に色をつけたりするのに使用できます。
	 * 初期値は、空文字 "" です。
	 *
	 * @return   マーカー文字列
	 *
	 */
	String getMarker() {
		return marker;
	}
}
