/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.column;

import org.opengion.hayabusa.db.AbstractRenderer;
import org.opengion.hayabusa.db.CellRenderer;
import org.opengion.hayabusa.db.DBColumn;
import org.opengion.fukurou.util.StringUtil;

/**
 * NBSP レンデラーは、内部のスペースを、&amp;nbsp; という文字列に置き換えます。
 * これは、文字列にスペースが含まれている場合、&amp;nbsp; と言うコードにエスケープすることで、
 * HTML 上で、連続したスペースを表示します。
 * 通常、PRE レンデラーで表示するなどの方法もありますが、NBSP でないとスペースに
 * ならない場合(たとえば、プルダウンメニューのオプション文字列など)に対応できます。
 * また、レンデラーパラメータに、数字を指定すれば、その文字数で強制的に Fill埋めするため、
 * 固定長の表示にも使用できます。
 * これにより、連続するスペースを、そのまま表示することが出来ます。
 *
 * カラムの表示に必要な属性は, DBColumn オブジェクト より取り出します。
 * このクラスは、DBColumn オブジェクト毎に１つ作成されます。
 *
 * @og.rev 3.7.1.0 (2005/04/26) 新規追加
 * @og.group データ表示
 *
 * @version  0.9.0  2000/10/17
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class Renderer_NBSP extends AbstractRenderer {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "6.0.4.0 (2014/11/28)" ;

	private final int dataSize ;

	/**
	 * デフォルトコンストラクター。
	 * このコンストラクターで、基本オブジェクトを作成します。
	 *
	 */
	public Renderer_NBSP() {
		dataSize   = 0;
	}

	/**
	 * デフォルトコンストラクター。
	 *
	 * @param	clm	DBColumnオブジェクト
	 */
	private Renderer_NBSP( final DBColumn clm ) {
		final String tmp = clm.getRendererParam();
//		if( tmp == null || tmp.length() == 0 ) { dataSize = 0; }
		if( tmp == null || tmp.isEmpty() ) { dataSize = 0; }
		else {
			dataSize = Integer.parseInt( tmp ) ;
		}
	}

	/**
	 * 各オブジェクトから自分のインスタンスを返します。
	 * 自分自身をキャッシュするのか、新たに作成するのかは、各サブクラスの実装に
	 * まかされます。
	 *
	 * @param	clm	DBColumnオブジェクト
	 *
	 * @return	CellRendererオブジェクト
	 */
	public CellRenderer newInstance( final DBColumn clm ) {
		return new Renderer_NBSP( clm );
	}

	/**
	 * データの表示用文字列を返します。
	 *
	 * ここでは、内部のスペースを、&amp;nbsp; という文字列に置き換えます。
	 * これにより、HTML 上で、連続したスペースを表示できます。
	 * また、レンデラーパラメータに、数字を指定すれば、その文字数で強制的に Fill埋めするため、
	 * 固定長の表示にも使用できます。
	 *
	 * @param   value 入力値
	 *
	 * @return  データの表示用文字列
	 */
	@Override
	public String getValue( final String value ) {
		String rtn = ( value == null ) ? "" : value ;

		if( dataSize > 0 ) {
			rtn = StringUtil.stringFill( rtn,dataSize,"Windows-31J" );
		}

		return StringUtil.replace( rtn," ","&nbsp;" );
	}

	/**
	 * データ出力用の文字列を作成します。
	 * ファイル等に出力する形式を想定しますので、HTMLタグを含まない
	 * データを返します。
	 * 基本は、#getValue( String ) をそのまま返します。
	 *
	 * @og.rev 6.0.4.0 (2014/11/28) データ出力用のレンデラー
	 *
	 * @param   value 入力値
	 *
	 * @return  データ出力用の文字列
	 * @see		#getValue( String )
	 */
	@Override
	public String getWriteValue( final String value ) {
		return value==null ? "" : value;
	}
}
