/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.fukurou.util;

import java.util.List;
import java.util.ArrayList;
import java.util.Map;							// 6.1.0.0 (2014/12/26) ConstData を Mapで管理
import java.util.HashMap;						// 6.1.0.0 (2014/12/26) ConstData を Mapで管理

/**
 * EXCELファイルを、イベント方式に準拠して、テキストとして読み込み処理を行います。
 * ExcelReaderEventイベントは、openGion形式のファイル読み取りに準拠した方法をサポートします。
 * ※ openGion形式のEXCELファイルとは、#NAME 列に、カラム名があり、#で始まる
 *    レコードは、コメントとして判断し、読み飛ばす処理の事です。
 *
 * このイベントクラスは、サブクラスを作成し、ExcelReader_HSSF、ExcelReader_XSSF、
 * ExcelReader_SS の excelReader メソッドの引数に指定します。
 * HSSF,XSSF は、イベントモデルで処理します。SS は、DOMモデルですが、(.xls/.xlsx)の
 * 両方をサポートします。
 * 実際は、POIUtil#excelReader( String , ExcelReaderEvent ) を使用すれば、拡張子に応じて
 * 使用するクラスを選択します。
 *
 * @og.rev 6.0.3.0 (2014/11/13) 新規作成
 * @og.group ファイル入力
 *
 * @version  6.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK7.0,
 */
public class ExcelReaderEvent {
	private int		nowRowNo	= -1;		// 現在の行番号
	private boolean	isNowSkip	;			// スキップ中かどうか(true:スキップ中)
	private boolean	isNowName	;			// カラム名配列の収集中かどうか(true:収集中)
	private boolean	isShtBreak	;			// シートの読み取り処理を中止するかどうか(true:中止)

	private int		skipRowCnt	;			// 読み飛ばす行数(読み飛ばし中でも ContDataは取得する)

	private int		 clmSize	= -1 ;		// カラムサイズ
	private String[] names		;			// カラム名配列
	private String[] vals		;			// 値の文字列配列(１行分)
	private boolean	useVals		;			// 値配列に設定されたか？

	private List<Integer> colList	;		// カラム番号:name が null かゼロ文字列の場合は、飛ばします。
	private List<String>  nmsList	;		// カラム名:カラム番号に対応したカラム名

	private ConstData	cnstData	;

	/**
	 * シートの読み取り開始時にイベントが発生します。
	 *
	 * 新しいシートの読み取り開始毎に、１回呼ばれます。
	 * 戻り値が、true の場合は、そのシートの読み取りを継続します。
	 * false の場合は、そのシートの読み取りは行わず、次のシートまで
	 * イベントは発行されません。
	 * 初期実装では、固定値設定処理を行っていますので、オーバーライドする場合は、
	 * super.startSheet(String,int) してください。
	 * 初期実装では、true を返します。
	 *
	 * @og.rev 6.0.3.0 (2014/11/13) 新規作成
	 *
	 * @param   shtNm  シート名
	 * @param   shtNo  シート番号(0～)
	 * @return  true:シートの読み取り処理を継続します/false:このシートは読み取りません。
	 */
	public boolean startSheet( final String shtNm,final int shtNo ) {
		// シート名を設定する。
		if( cnstData != null ) { cnstData.putConstSheet( shtNm ); }
		return true;
	}

	/**
	 * シートの読み取り終了時にイベントが発生します。
	 *
	 * #columnNames( String[] ) や、#values( String[] ,int ) などは、行の処理が完了した時点で
	 * イベントが呼ばれるため、一番最後のレコードの終了条件が判りません。
	 * そこで、このイベントを呼ぶことで、シートの終了(=最終行の終了)処理を行うことができます。
	 * 
	 * 引数のシート番号は、参考情報で、#startSheet( String,int ) で呼ばれたシート番号と
	 * 比較できるようにしています。
	 * 初期実装では、固定値設定処理を行っていますので、オーバーライドする場合は、
	 * super.endSheet(int) してください。
	 *
	 * @og.rev 6.0.3.0 (2014/11/13) 新規作成
	 * @og.rev 6.1.0.0 (2014/12/26) シートブレイク処理追加
	 *
	 * @param   shtNo  シート番号(0～)
	 */
	public void endSheet( final int shtNo ) {
		isShtBreak = false;				// 
		endRow( -1 );					// 行終了処理
		if( cnstData != null ) { cnstData.clearValsMap(); }
	}

	/**
	 * 読み取り状態の時に、rowNo,colNo にあるセルの値を引数にイベントが発生します。
	 *
	 * 戻り値が、true の場合は、その行の読み取りを継続します。
	 * false の場合は、その行の読み取りは行わず、次の行まで
	 * イベントは発行されません。
	 *
	 * openGion での標準的な処理は、colNo==0 の時に、val の先頭が、# で
	 * 始まる場合は、その行はスキップします。
	 * ここでの return は、#isSkip( int ) と逆になりますので、ご注意ください。
	 * 初期実装では、#NAME処理、行スキップ、行終了処理等を実行します。
	 * オーバーライドする場合は、super.value(String,int,int) してください。
	 *
	 * @og.rev 6.0.3.0 (2014/11/13) 新規作成
	 *
	 * @param   val     文字列値
	 * @param   rowNo   行番号(0～)
	 * @param   colNo   列番号(0～)
	 * @return  読み取りするかどうか(true:読み取りする/false:読み取りしない)
	 * @see		#isSkip( int )
	 */
	public boolean value( final String val,final int rowNo,final int colNo ) {
		// 値が存在する場合のみ、処理する。
		if( val!=null && val.length()>0 ) {
			// 行番号が異なった場合は、行の終了処理を実行する。
			if( nowRowNo != rowNo ) {
				endRow( rowNo );							// 行終了処理
			}

			// 固定文字の設定なので、コメントもスキップも無関係
			if( cnstData != null ) { cnstData.putConstValue( val,rowNo,colNo ); }

			// 先頭カラムのコメント判断と、#NAME列判断
			if( colNo==0 && val.charAt(0)=='#' ) {
				if( "#NAME".equalsIgnoreCase( val ) && clmSize < 0 ) {	// clmSize < 0 は、#NAME未設定という意味
					isNowName = true ;
					colList = new ArrayList<Integer>() ;	// 初期化
					nmsList = new ArrayList<String>() ;		// 初期化
				}
				isNowSkip = !isNowName;						// #NAME の場合は、false(SKIPしない)
			}
			else if( isNowName ) {							// #NAME処理中
				// #NAME 設定も、null や、ゼロ文字列の場合は、登録しない。(一番上の if で対処)
				colList.add( Integer.valueOf( colNo ) );
				nmsList.add( val );
			}
			// clmSize >= 0 は、#NAME が設定済みという意味
			else if( clmSize >= 0 && skipRowCnt <= rowNo ) {				// skipRowCnt は、値セットだけ影響する。
				final int indx = colList.indexOf( Integer.valueOf( colNo ) );		// カラム番号の位置が 値配列の配列番号
				if( indx >= 0 ) {
					vals[indx] = val;
					useVals = true;
				}
			}
		}

		return !isNowSkip;
	}

	/**
	 * rowNo を元に、この行をスキップするかどうか判定のイベントが発生します。
	 *
	 * ※ この処理は、行が新しくなると、false が返ります。つまり、カラムの値を求める
	 *    イベントやループの直前に入れることで、同一カラム処理のループ処理を
	 *    短縮するのが狙いです。
	 *    引数から、行のループに入れてしまいそうですが、行列のループに入れてください。
	 *
	 * 戻り値が、true の場合は、その行の読み取りをスキップします。
	 * false の場合は、読み取り処理を継続します。
	 * スキップ中は、行に関するイベントは発行されません。
	 *
	 * 値(val)を求める前に、行情報を入手し、スキップ中の現在行と同じ場合に、スキップ(=true)と判定します。
	 * スキップ中かどうかは、#value( String,int,int ) で、判定します。
	 * 初期実装では、スキップ中 かつ 現在行と同じ行の場合、または、シートブレーク中の場合に、true を返します。
	 *
	 * @og.rev 6.0.3.0 (2014/11/13) 新規作成
	 * @og.rev 6.1.0.0 (2014/12/26) シートブレイク処理追加
	 *
	 * @param   rowNo   行番号(0～)
	 * @return  スキップするかどうか(true:スキップする/false:スキップしない)
	 * @see		#value( String,int,int )
	 */
	public boolean isSkip( final int rowNo ) {
//		return isNowSkip && nowRowNo == rowNo ;
		return isNowSkip && nowRowNo == rowNo || isShtBreak ;
	}

	/**
	 * 行の終了時に実行されます。
	 *
	 * ここでは、rowNo がブレークするか、シート終了時に呼ぶことで、
	 * 一番最後の行の処理を行います。
	 *
	 * #NAME 処理中の場合(isNowName == true) は、カラム名配列を作成して、
	 * columnNames イベントを呼び出します。
	 * 値配列が設定されている場合(useVals == true) は、values イベントを
	 * 呼び出します。
	 * #NAME 処理が完了している場合は、値配列の初期化を行います。
	 * そのうえで、スキップの解除(isNowSkip = false)と行データ
	 * 未設定(useVals = false)に、初期化します。
	 *
	 * @og.rev 6.0.3.0 (2014/11/13) 新規作成
	 *
	 * @param   rowNo   行番号(0～)
	 * @see		#columnNames( String[] )
	 * @see		#values( String[],int )
	 */
	private void endRow( final int rowNo ) {
		if( isNowName ) {								// #NAME 処理中の場合
			clmSize = colList.size();
			names = nmsList.toArray( new String[clmSize] );
			if( cnstData != null ) { cnstData.setColumns( names ); }
			columnNames( names.clone() );				// イベント
			isNowName = false;
			nmsList   = null ;		// 不要
		}
		else if( useVals ) {							// 値配列が設定されている場合
			if( cnstData != null ) { vals = cnstData.getConstVals( vals ); }
			values( vals , nowRowNo );					// イベント発行(現在行)
		}

		if( clmSize >= 0 ) {
			vals = new String[clmSize];					// 値配列の初期化
		}

		isNowSkip = false;						// スキップの解除
		useVals	  = false;						// 行データ未設定にする。
		nowRowNo  = rowNo;						// 現在行の設定
	}

	/**
	 * シート数のイベントが発生します。
	 *
	 * 処理の開始前に、シート数のイベントが発生します。
	 * これを元に、処理するシート番号の選別が可能です。
	 * 初期実装は、何もありません。
	 *
	 * @og.rev 6.1.0.0 (2014/12/26) シートの数のイベント
	 *
	 * @param   size  シート数
	 */
	public void sheetSize( final int size ) {
		// refactoring : Document empty method 対策
	}

	/**
	 * カラム名配列がそろった段階で、イベントが発生します。
	 *
	 * openGion での標準的な処理は、colNo==0 の時に、val の先頭が、#NAME
	 * で始まるレコードを、名前配列として認識します。
	 * #value( String,int,int ) で、この #NAME だけは、継続処理されます。
	 * その上で、#NAME レコードが終了した時点で、カラム名配列が完成するので
	 * そこで初めて、このメソッドが呼ばれます。
	 * 初期実装は、何もありません。
	 *
	 * @og.rev 6.0.3.0 (2014/11/13) 新規作成
	 *
	 * @param   names  カラム名配列(可変長引数)
	 * @see		#value( String,int,int )
	 */
//	public void columnNames( final String[] names ) {
	public void columnNames( final String... names ) {
		// refactoring : Document empty method 対策
	}

	/**
	 * row にあるセルのオブジェクト値がそろった段階で、イベントが発生します。
	 *
	 * 初期実装は、何もありません。
	 *
	 * @og.rev 6.0.3.0 (2014/11/13) 新規作成
	 *
	 * @param   vals    文字列値の１行分の配列
	 * @param   rowNo   行番号(0～)
	 */
	public void values( final String[] vals,final int rowNo ) {
		// refactoring : Document empty method 対策
	}

	/**
	 * 外部からCSV形式のカラム名文字列を設定します。
	 *
	 * カラム名配列を、#NAME で始まるレコードではなく、外部から指定します。
	 * ここで設定した場合、#columnNames( String[] )イベントも発生します。
	 * null か、長さゼロのカラム名は、設定されません。
	 *
	 * @og.rev 6.1.0.0 (2014/12/26) カラム名配列設定の対応
	 *
	 * @param   clms  CSV形式のカラム名文字列
	 * @param	useNumber	行番号情報 [true:使用している/false:していない]
	 * @see		#columnNames( String[] )
	 */
	public final void setNames( final String clms , final boolean useNumber ) {
		if( clms != null && clms.length() > 0 ) {
			colList = new ArrayList<Integer>() ;	// 初期化
			nmsList = new ArrayList<String>() ;		// 初期化

			final String[] nms = StringUtil.csv2Array( clms );
			final int adrs = useNumber ? 1:0 ;	// useNumber =true の場合は、１件目(No)は読み飛ばす。
			for( int i=0; i<nms.length; i++ ) {
				// null か、長さゼロのカラム名は、設定されません。
				final String nm = nms[i];
				if( nm != null && nm.length() > 0 ) {
					colList.add( Integer.valueOf( i+adrs ) );
					nmsList.add( nm );
				}
			}
			clmSize = colList.size();
			names   = nmsList.toArray( new String[clmSize] );
			if( cnstData != null ) { cnstData.setColumns( names ); }
			columnNames( names.clone() );			// イベントを発生させます。
			nmsList   = null ;		// 不要
		}
	}

	/**
	 * カラム名配列が、設定されたかどうか、返します。
	 *
	 * カラム名配列は、#NAME で始まるレコードか、#setNames( String,boolean )メソッドを
	 * 使用して、外部から指定します。
	 * カラム名配列が、見つからない場合、データもセットされていない可能性がある為、
	 * ここで、判定します。
	 *
	 * @og.rev 6.1.0.0 (2014/12/26) カラム名配列設定の対応
	 *
	 * @return	カラム名配列が、設定されたかどうか(true:設定済み/false:未設定)
	 * @see		#setNames( String,boolean )
	 */
	public final boolean isNameSet() {
		return names != null && names.length > 0 ;
	}

	/**
	 * このシートの以降のデータを読み飛ばすかどうかを指定します(初期値:false)。
	 *
	 * データを読み込む途中で、それ以降のデータを読み込む必要がなくなった場合に、
	 * true に設定すると、以降のデータを今のシートが終了するまで、スキップします。
	 * 例えば、読み込むべき必須カラムで判定する、nullBreakClm 機能を実現できます。
	 * 初期値は、ここで設定した場合、#columnNames( String[] )イベントは発生しません。
	 * 引数が、false:読み飛ばさない です。
	 *
	 * @og.rev 6.1.0.0 (2014/12/26) シートブレイク処理追加
	 *
	 * @param   flag  このシートの以降のデータを読み飛ばすかどうか[true:読み飛ばす/false:読み飛ばさない]
	 * @see		#isSkip( int )
	 */
	public void setSheetBreak( final boolean flag ) {
		isShtBreak = flag ;
	}

	/**
	 * データの読み飛ばし件数を設定します。
	 *
	 * EXCELのデータの読み始めの行を指定します。
	 * ファイルの先頭行が、０行としてカウントしますので、設定値は、読み飛ばす
	 * 件数になります。(１と指定すると、１件読み飛ばし、２行目から読み込みます。)
	 * 読み飛ばしは、コメント行などは、無視しますので、実際の行数分読み飛ばします。
	 * ＃NAME属性や、columns 属性は、有効です。
	 *
	 * @og.rev 6.1.0.0 (2014/12/26) Excel関係改善
	 *
	 * @param	count 読み始めの初期値(0なら、読み飛ばしなし)
	 */
	public void setSkipRowCount( final int count ) {
		skipRowCnt = count;
	}

	/**
	 * 固定値となるカラム名(CSV形式)と、constAdrs 固定値となるアドレス(行-列,行-列,・・・)を設定します。
	 *
	 * アドレスは、EXCEL上の行-列をCSV形式で指定します。
	 * 行列は、EXCELオブジェクトに準拠するため、０から始まる整数です。
	 * 0-0 ⇒ A1 , 1-0 ⇒ A2 , 0-1 ⇒ B1 になります。
	 * これにより、シートの一か所に書かれている情報を、DBTableModel のカラムに固定値として
	 * 設定することができます。
	 * 例として、DB定義書で、テーブル名をシートの全レコードに設定したい場合などに使います。
	 * このメソッドは、isExcel() == true の場合のみ利用されます。
	 *
	 * 5.7.6.3 (2014/05/23) より、
	 *   ①EXCEL表記に準拠した、A1,A2,B1 の記述も処理できるように対応します。
	 *     なお、A1,A2,B1 の記述は、必ず、英字1文字＋数字 にしてください。(A～Zまで)
	 *   ②処理中のEXCELシート名をカラムに割り当てるために、"SHEET" という記号に対応します。
	 * 例えば、sheetConstKeys="CLM,LANG,NAME" とし、sheetConstAdrs="0-0,A2,SHEET" とすると、
	 * NAMEカラムには、シート名を読み込むことができます。
	 * これは、内部処理の簡素化のためです。
	 *
	 * ちなみに、EXCELのセルに、シート名を表示させる場合の関数は、下記の様になります。
	 * =RIGHT(CELL("filename",$A$1),LEN(CELL("filename",$A$1))-FIND("]",CELL("filename",$A$1)))
	 *
	 * @og.rev 5.5.8.2 (2012/11/09) 新規追加
	 * @og.rev 5.7.6.3 (2014/05/23) EXCEL表記(A2,B1等)の対応と、特殊記号(SHEET)の対応
	 * @og.rev 6.1.0.0 (2014/12/26) カラム名配列設定の対応
	 *
	 * @param	constKeys	固定値となるカラム名(CSV形式)
	 * @param	constAdrs	固定値となるアドレス(行-列,行-列,・・・)
	 */
	public void setSheetConstData( final String constKeys,final String constAdrs ) {
		if( constKeys == null || constKeys.isEmpty() || constAdrs == null || constAdrs.isEmpty() ) {
			return ;
		}

		cnstData = new ConstData( constKeys,constAdrs );
		// setNames( String , boolean ) と、このメソッドの呼び出し順に影響がないようにするため。
		if( names != null ) { cnstData.setColumns( names ); }
	}

	/**
	 * EXCELファイルの所定の位置から、固定値を取り出す処理を管理します。
	 * 
	 * この固定値の取出しは、内部処理に、非常に依存しているため、今は、
	 * ExcelReaderEvent クラスに含めていますが、将来的には、分ける予定です。
	 *
	 * @og.rev 6.1.0.0 (2014/12/26) Excel関係改善
	 *
	 * @version  6.0
	 * @author   Kazuhiko Hasegawa
	 * @since    JDK7.0,
	 */
	private static final class ConstData {
		// ①cnstMap は、cnstKey をキーに、拾ってきた値を持っています。(Sheet毎のトランザクション)
		private final Map<String,String> cnstMap   = new HashMap<String,String>();			// 6.1.0.0 (2014/12/26) Mapで管理
		// ②rowcolMap は、rowcol をキーに、アドレスを持っています。
		private final Map<String,Integer> rowcolMap = new HashMap<String,Integer>();		// 6.1.0.0 (2014/12/26) Mapで管理
		// ③valsMap は、アドレス をキーに、拾ってきた値を持っています。(Sheet毎のトランザクション)
		private final Map<Integer,String> valsMap   = new HashMap<Integer,String>();			// 6.1.0.0 (2014/12/26) Mapで管理

		private int maxRow = -1 ;		// 最大値持っておき、判定処理を早める。

		/**
		 * 固定値となるカラム名(CSV形式)と、constAdrs 固定値となるアドレス(行-列,行-列,・・・)を設定します。
		 *
		 * アドレスは、EXCEL上の行-列をCSV形式で指定します。
		 * 行列は、EXCELオブジェクトに準拠するため、０から始まる整数です。
		 * 0-0 ⇒ A1 , 1-0 ⇒ A2 , 0-1 ⇒ B1 になります。
		 * これにより、シートの一か所に書かれている情報を、DBTableModel のカラムに固定値として
		 * 設定することができます。
		 * 例として、DB定義書で、テーブル名をシートの全レコードに設定したい場合などに使います。
		 * このメソッドは、isExcel() == true の場合のみ利用されます。
		 *
		 * 5.7.6.3 (2014/05/23) より、
		 *   ①EXCEL表記に準拠した、A1,A2,B1 の記述も処理できるように対応します。
		 *     なお、A1,A2,B1 の記述は、必ず、英字1文字＋数字 にしてください。(A～Zまで)
		 *   ②処理中のEXCELシート名をカラムに割り当てるために、"SHEET" という記号に対応します。
		 * 例えば、sheetConstKeys="CLM,LANG,NAME" とし、sheetConstAdrs="0-0,A2,SHEET" とすると、
		 * NAMEカラムには、シート名を読み込むことができます。
		 * これは、内部処理の簡素化のためです。
		 *
		 * ちなみに、EXCELのセルに、シート名を表示させる場合の関数は、下記の様になります。
		 * =RIGHT(CELL("filename",$A$1),LEN(CELL("filename",$A$1))-FIND("]",CELL("filename",$A$1)))
		 *
		 * @og.rev 5.5.8.2 (2012/11/09) 新規追加
		 * @og.rev 5.7.6.3 (2014/05/23) EXCEL表記(A2,B1等)の対応と、特殊記号(SHEET)の対応
		 * @og.rev 6.1.0.0 (2014/12/26) カラム名配列設定の対応
		 *
		 * @param	constKeys	固定値となるカラム名(CSV形式)
		 * @param	constAdrs	固定値となるアドレス(行-列,行-列,・・・)
		 */
		ConstData( final String constKeys,final String constAdrs ) {
			final String[] cnstKeys = constKeys.split( "," );
			final String[] row_col  = constAdrs.split( "," ) ;

			if( cnstKeys.length != row_col.length ) {
				final String errMsg = "キーに対するアドレスの個数が不一致です。Keys=[" + constKeys + "]"
							+ " , Adrs=[" + constAdrs + "]" ;
				throw new RuntimeException( errMsg );
			}

			for( int j=0; j<cnstKeys.length; j++ ) {
				final String cnstKey = cnstKeys[j].trim();	// 前後の不要なスペースを削除
				if( cnstKey != null && cnstKey.length() > 0 ) {
					String rowcol  = row_col[j].trim();		// 前後の不要なスペースを削除
					if( rowcol != null && rowcol.length() > 0 ) {
						// 5.7.6.3 (2014/05/23) EXCEL表記(A2,B1等)の対応と、特殊記号(SHEET)の対応
						final int sep = rowcol.indexOf( '-' );
						if( sep > 0 ) {
							final int row = Integer.parseInt( rowcol.substring( 0,sep ) );
			//				int col = Integer.parseInt( rowcol.substring( sep+1 ) );
							if( maxRow < row ) { maxRow = row; }
			//				rowcol = String.valueOf( (char)('A' + col) ) + String.valueOf( row + 1 ) ;	// row-col 形式なので、不要
						}
						else if( "SHEET".equalsIgnoreCase( rowcol ) ) {
							rowcol = "SHEET" ;			// 大文字化に統一
						}
						// rowcol が SHEET 以外の場合は、A1,A2,B1 の記述とみなす。
						else {
							final int row = Integer.parseInt( rowcol.substring( 1 ) ) -1;		// C6 の場合、rowは、6-1=5
							final int col = rowcol.charAt(0) - 'A' ;							// C6 の場合、colは、'C'-'A'=2
							if( maxRow < row ) { maxRow = row; }
							rowcol = row + "-" + col ;
						}
						cnstMap.put( cnstKey , rowcol );		// 6.1.0.0 (2014/12/26) cnstMap に行列情報を設定する
					}
				}
			}
		}

		/**
		 * カラム名配列を元に、固定値カラムのアドレスを求めます。
		 * カラム名配列は、順番に、指定する必要があります。
		 *
		 * @og.rev 6.1.0.0 (2014/12/26) カラム名配列設定の対応
		 *
		 * @param	names	カラム列配列(可変長引数)
		 */
//		void setColumns( final String[] names ) {
		void setColumns( final String... names ) {
			// 6.1.1.0 (2015/01/17) 可変長引数でもnullは来る。
			if( names != null && names.length > 0 ) {
//				final int columnSize = names.length ;
				// 5.5.8.2 (2012/11/09) 固定値取得用の cnstIndx の設定を行う。
//				for( int i=0; i<columnSize; i++ ) {
				for( int i=0; i<names.length; i++ ) {
					final String rowcol = cnstMap.get( names[i] );		// cnstKey があれば、rowcol が取得できるはず
					if( rowcol != null ) {
						rowcolMap.put( rowcol , Integer.valueOf( i ) );
					}
				}
			}
		}

		/**
		 * 読み取り時に、rowNo,colNo にあるセルの値を、固定値となるカラム名に関連付けます。
		 *
		 * イベントモデルでは、固定値の指定アドレス(rowNo,colNo)をピンポイントで取得することが
		 * できないため、イベント発生毎に、チェックする必要があります。
		 * そのため、固定値を使用すると、処理速度が低下します。
		 *
		 * @og.rev 6.1.0.0 (2014/12/26) Excel関係改善
		 *
		 * @param   val     文字列値(null またはゼロ文字列は、不可)
		 * @param   rowNo   行番号(0～)
		 * @param   colNo   列番号(0～)
		 */
		void putConstValue( final String val,final int rowNo,final int colNo ) {
			if( rowNo <= maxRow ) {
//				String rowcol = String.valueOf( (char)('A' + colNo) ) + String.valueOf( rowNo + 1 ) ;
				final String rowcol = rowNo + "-" + colNo ;
				final Integer adrs = rowcolMap.get( rowcol );
				if( adrs != null ) {
					valsMap.put( adrs,val );
				}
			}
		}

		/**
		 * シート名を外部から指定します。
		 * アドレス指定の特殊系として、"SHEET" 文字列を指定できます。
		 * これは、新しいシートの読み取り開始時に、設定します。
		 *
		 * @og.rev 6.1.0.0 (2014/12/26) Excel関係改善
		 *
		 * @param	shtNm	シート名
		 */
		void putConstSheet( final String shtNm ) {
			final Integer adrs = rowcolMap.get( "SHEET" );
			if( adrs != null ) {
				valsMap.put( adrs,shtNm );
			}
		}

		/**
		 * 内部の valsMap を初期化します。
		 * 固定値の読み取りは、シートごとに行います。
		 * 新しいシートにデータが設定されていない場合、前のシートの値が残ります。
		 * ここでは、シート呼出しごとに、毎回クリアします。
		 *
		 * @og.rev 6.1.0.0 (2014/12/26) Excel関係改善
		 *
		 */
		void clearValsMap() { valsMap.clear(); }

		/**
		 * 値配列のデータに、固定値を設定します。
		 * 引数の文字列配列に、固定値を設定しますので、配列オブジェクト自体を更新します。
		 *
		 * @og.rev 6.1.0.0 (2014/12/26) Excel関係改善
		 *
		 * @param   vals    文字列値の１行分の配列
		 * @return  固定値を設定された１行分の文字列配列
		 */
		String[] getConstVals( final String[] vals ) {
			if( vals != null && vals.length > 0 ) {
				for( final Map.Entry<Integer,String> entry : valsMap.entrySet() ) {
					final int adrs   = entry.getKey().intValue();				// Autoボクシングでよい？
					final String cnst = entry.getValue();
					if( adrs < vals.length ) {
						vals[adrs] = cnst;
					}
				}
			}
			return vals ;
		}
	}
}
