/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.fukurou.util;

import java.io.InputStream;
import java.io.FileInputStream;
import java.io.BufferedInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;

import java.text.DecimalFormat;
import java.text.NumberFormat;
import java.util.Set;								// 6.0.2.3 (2014/10/10)
import java.util.TreeSet;							// 6.0.2.3 (2014/10/10)

import org.apache.xmlbeans.XmlException;
import org.apache.poi.POITextExtractor;
import org.apache.poi.extractor.ExtractorFactory;

import org.apache.poi.hwpf.HWPFDocument;
import org.apache.poi.hwpf.usermodel.Range;
import org.apache.poi.hwpf.usermodel.Section;
import org.apache.poi.hwpf.usermodel.Paragraph;
import org.apache.poi.hwpf.usermodel.CharacterRun;

import org.apache.poi.hssf.usermodel.HSSFCellStyle;

import org.apache.poi.hslf.HSLFSlideShow;
import org.apache.poi.hslf.usermodel.SlideShow;
import org.apache.poi.hslf.model.Slide;
import org.apache.poi.hslf.model.TextRun;

import org.apache.poi.openxml4j.exceptions.InvalidFormatException;
import org.apache.poi.openxml4j.exceptions.OpenXML4JException ;			// 6.1.0.0 (2014/12/26) findBugs

import org.apache.poi.ss.usermodel.WorkbookFactory;
import org.apache.poi.ss.usermodel.Workbook;
import org.apache.poi.ss.usermodel.Sheet;
import org.apache.poi.ss.usermodel.Row;
import org.apache.poi.ss.usermodel.Cell;
import org.apache.poi.ss.usermodel.CellStyle;
import org.apache.poi.ss.usermodel.CreationHelper;
import org.apache.poi.ss.usermodel.RichTextString;
import org.apache.poi.ss.usermodel.DateUtil;
import org.apache.poi.ss.usermodel.FormulaEvaluator;
import org.apache.poi.ss.usermodel.Name;				// 6.0.2.3 (2014/10/10)
import org.apache.poi.ss.util.SheetUtil;

import static org.opengion.fukurou.util.HybsConst.CR;				// 6.1.0.0 (2014/12/26) refactoring
import static org.opengion.fukurou.util.HybsConst.BUFFER_MIDDLE;	// 6.1.0.0 (2014/12/26) refactoring

/**
 * POI による、Excel/Word/PoworPoint等に対する、ユーティリティクラスです。
 *
 * 基本的には、ネイティブファイルを読み取り、テキストを取得する機能が主です。
 * Excel、Word、PowerPoint、Visio、Publisher からのテキスト取得が可能です。
 *
 * @og.rev 6.0.2.0 (2014/09/19) 新規作成
 * @og.group その他
 *
 * @version  6.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK7.0,
 */
public final class POIUtil {
	/** このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "6.0.3.0 (2014/11/13)" ;

//	private static final String CR = System.getProperty("line.separator");

	/**
	 * すべてが staticメソッドなので、コンストラクタを呼び出さなくしておきます。
	 *
	 */
	private POIUtil() {}

	/**
	 * 引数ファイルを、POITextExtractor を使用してテキスト化します。
	 *
	 * Excel、Word、PowerPoint、Visio、Publisher からのテキスト取得が可能です。
	 * 拡張子から、ファイルの種類を自動判別します。
	 *
	 * @og.rev 6.0.2.0 (2014/09/19) 新規作成
	 *
	 * @param	fname 入力ファイル名
	 * @return	変換後のテキスト
	 * @og.rtnNotNull
	 */
	public static String getText( final String fname ) {
		InputStream fis = null;
		POITextExtractor extractor = null;
		try {
			fis = new BufferedInputStream( new FileInputStream( fname ) );
			extractor = ExtractorFactory.createExtractor( fis );
			return extractor.getText();
		}
//		catch( Exception ex ) {					// 6.1.0.0 (2014/12/26) refactoring
		catch( FileNotFoundException ex ) {
			final String errMsg = "ファイルが存在しません[" + fname + "]" + CR + ex.getMessage() ;
			throw new RuntimeException( errMsg,ex );
		}
		catch( IOException ex ) {
			final String errMsg = "ファイル処理エラー[" + fname + "]" + CR + ex.getMessage() ;
			throw new RuntimeException( errMsg,ex );
		}
		catch( InvalidFormatException ex ) {
			final String errMsg = "ファイルフォーマットエラー[" + fname + "]" + CR + ex.getMessage() ;
			throw new RuntimeException( errMsg,ex );
		}
		catch( OpenXML4JException ex ) {
			final String errMsg = "ODF-XML処理エラー[" + fname + "]" + CR + ex.getMessage() ;
			throw new RuntimeException( errMsg,ex );
		}
		catch( XmlException ex ) {
			final String errMsg = "XML処理エラー[" + fname + "]" + CR + ex.getMessage() ;
			throw new RuntimeException( errMsg,ex );
		}
		finally {
			Closer.ioClose( extractor );
			Closer.ioClose( fis );
		}
	}

	/**
	 * 引数ファイル(Word)を、HWPFDocument を使用してテキスト化します。
	 *
	 * POIEventは、唯一の void readText( String,String int... ) メソッドを持ち、
	 * 処理単位に、このメソッドが呼ばれます。つまり、内部にメモリを溜め込みません。
	 * int...可変長引数は、0:Section番号 1:page番号 2:Paragraph番号 がセットされます。
	 *
	 * POIEventは、データとして設定されているレコードのみCallされます。
	 *
	 * @og.rev 6.0.2.0 (2014/09/19) 新規作成
	 *
	 * @param	fname 入力ファイル名
	 * @param	event イベント処理させるI/F
	 */
	public static void wordReader( final String fname , final POIEvent event ) {
		InputStream fis = null;
		final StringBuilder buf = new StringBuilder( BUFFER_MIDDLE );

		try {
			fis = new BufferedInputStream( new FileInputStream( fname ) );
			final HWPFDocument doc = new HWPFDocument( fis );
			final Range rng = doc.getRange();
			// HeaderStories header = new HeaderStories( doc );

			int pCnt = 0;		// ページ番号
			for (int sno = 0; sno < rng.numSections(); sno++) {
				final Section sec = rng.getSection(sno);
				String title = null;									// HeaderStories からうまく取れなかった。
				for (int pno = 0; pno < sec.numParagraphs(); pno++) {
					final Paragraph para = sec.getParagraph(pno);

//					if( para.text().indexOf("\f") >= 0 ) {
					if( para.text().indexOf('\f') >= 0 ) {				// 6.0.2.5 (2014/10/31) refactoring
						pCnt++;
			//			title = header.getHeader( pCnt ).trim();
					}

					buf.setLength(0);		// Clearの事
					for (int cno = 0; cno < para.numCharacterRuns(); cno++) {
						final CharacterRun run = para.getCharacterRun(cno);
						// Removes any fields (eg macros, page markers etc) from the string
						buf.append( Range.stripFields( run.text() ) );
					}
					final String text = buf.toString().trim();
					// データとして設定されているレコードのみイベントを発生させる。
					if( text.length() > 0 ) {
						if( title == null ) { title = text; }		// Section の最初の有効なParagraphをタイトルにする。
						event.readText( text,title,sno,pCnt,pno );	// テキスト タイトル 0:Section番号 1:page番号 2:Paragraph番号
					}
				}
			}
		}
		catch( IOException ex ) {
			final String errMsg = "ファイル読込みエラー[" + fname + "]" + CR + ex.getMessage() ;
			throw new RuntimeException( errMsg,ex );
		}
		finally {
			Closer.ioClose( fis );
		}
	}

	/**
	 * 引数ファイル(PoworPoint)を、HSLFSlideShow を使用してテキスト化します。
	 *
	 * POIEventは、唯一の void readText( String,String int... ) メソッドを持ち、
	 * 処理単位に、このメソッドが呼ばれます。つまり、内部にメモリを溜め込みません。
	 * int...可変長引数は、0:Slide番号 1:TextRun番号 がセットされます。
	 *
	 * POIEventは、データとして設定されているレコードのみCallされます。
	 *
	 * @og.rev 6.0.2.0 (2014/09/19) 新規作成
	 *
	 * @param	fname 入力ファイル名
	 * @param	event イベント処理させるI/F
	 */
	public static void pptReader( final String fname , final POIEvent event ) {
		InputStream fis = null;

		try {
			fis = new BufferedInputStream( new FileInputStream( fname ) );
			final SlideShow ss = new SlideShow( new HSLFSlideShow( fis ) );
			final Slide[] slides = ss.getSlides();
			for (int sno = 0; sno < slides.length; sno++) {
				final String    title   = slides[sno].getTitle();				// title は、あまり当てにならない。
				final TextRun[] textRun = slides[sno].getTextRuns();
				for (int tno = 0; tno < textRun.length; tno++) {
					final String text = textRun[tno].getText().trim();
					// データとして設定されているレコードのみイベントを発生させる。
					if( text.length() > 0 ) {
						event.readText( text,title,sno,tno );			// テキスト タイトル 0:Slide番号 1:TextRun番号
					}
				}
			}
		}
		catch( IOException ex ) {
			final String errMsg = "ファイル読込みエラー[" + fname + "]" + CR + ex.getMessage() ;
			throw new RuntimeException( errMsg,ex );
		}
		finally {
			Closer.ioClose( fis );
		}
	}

	/**
	 * 引数ファイル(Excel)を、テキスト化します。
	 *
	 * ExcelReaderEvent のサブクラスを与えることで、EXCELデータをテキスト化できます。
	 * ここでは、XSSF(.xlsx)形式と、HSSF(.xls)形式で、処理を分けます。
	 * どちらも、イベントモデルでEXCELファイルを処理します。
	 * SS(.xlsx/.xls)では、巨大なXSSF(.xlsx) ファイルでは、OutOfMemoryエラーが発生します。
	 *
	 * @og.rev 6.0.2.0 (2014/09/19) 新規作成
	 * @og.rev 6.0.3.0 (2014/11/13) 引数を POIEvent → ExcelReaderEvent に変更
	 *
	 * @param	fname 入力ファイル名
	 * @param	event イベントオブジェクト(ExcelReaderEvent)
	 * @see		org.opengion.fukurou.model.ExcelModel
	 * @see		org.opengion.fukurou.util.ExcelReader_HSSF
	 * @see		org.opengion.fukurou.util.ExcelReader_XSSF
	 */
	public static void excelReader( final String fname , final ExcelReaderEvent event ) {
		if( fname.endsWith( ".xls" ) ) {
			ExcelReader_HSSF.excelReader( fname,event );
		}
		else if( fname.endsWith( ".xlsx" ) ) {
			ExcelReader_XSSF.excelReader( fname,event );
		}
		else {
			final String errMsg = "拡張子は、「.xls」 か、「.xlsx」にしてください。" ;
			throw new RuntimeException( errMsg );
		}
	}

	/**
	 * Excelの行列記号を、行番号と列番号に分解します。
	 *
	 * Excelの行列記号とは、A1 , B5 , AA23 などの形式を指します。
	 * これを、行番号と列番号に分解します。例えば、A1→0行0列、B5→4行1列、AA23→22行26列 となります。
	 * 分解した結果は、内部変数の、rowNo と colNo にセットされます。
	 * これらは、0 から始まる int型の数字で表します。
	 *
	 * @og.rev 6.0.3.0 (2014/11/13) 新規作成
	 *
	 * @param	kigo	Excelの行列記号( A1 , B5 , AA23 など )
	 * @return	行と列の番号を持った配列([0]=行=ROW , [1]=列=COL)
	 * @og.rtnNotNull
	 */
	public static int[] kigo2rowCol( final String kigo ) {
		int rowNo = 0;
		int colNo = -1;										// +1 して、26 かける処理をしているので、辻褄合わせ
		for( int i=0; i<kigo.length(); i++ ) {
			final char ch = kigo.charAt(i);
			if( 'A' <= ch && ch <= 'Z' ) { colNo = (colNo+1)*26 + ch-'A'; }
			else {
				// アルファベットでなくなったら、残りは 行番号(ただし、-1する)
				rowNo = Integer.parseInt( kigo.substring( i ) ) -1;
				break;
			}
		}
		return new int[] { rowNo,colNo };
	}

//	/**
//	 * Excelの行番号と列番号を、行列記号に合成します。
//	 *
//	 * Excelの行列記号とは、A1 , B5 , AA23 などの形式を指します。
//	 * これを、行番号と列番号にから合成します。例えば、0行0列→A1、4行1列→B5、22行26列→AA23 となります。
//	 * 合成した結果は、列記号+行番号 になりますが、、列番号は、引数の列番号＋１ になっています。
//	 * これは、内部配列処理は、0 から始まる数字ですが、EXCEL上の行番号は、１から始まります。
//	 *
//	 * @og.rev 6.0.3.0 (2014/11/13) 新規作成
//	 *
//	 * @param	rowNo	行番号
//	 * @param	colNo	列番号
//	 * @return	Excelの行列記号( A1 , B5 , AA23 など )
//	 */
//	public static String rowCol2kigo( final int rowNo , final int colNo ) {
//		String kigo = "";
//
//		// 手抜き：最大３ケタの記号までしか処理できません。
//		if(      colNo < 26 )    { kigo = "" + (char)('A'+colNo); }
//		else if( colNo < 26*27 ) { kigo = "" + (char)('A'+colNo/26-1)      + (char)('A'+colNo%26); }
//		else                     { kigo = "" + (char)('A'+colNo/(26*27)-1) + (char)('A'+(colNo/26-1)%26) + (char)('A'+colNo%26); }
//
//		return kigo;
//	}

	/**
	 * セルオブジェクト(Cell)から値を取り出します。
	 *
	 * セルオブジェクトが存在しない場合は、null を返します。
	 * それ以外で、うまく値を取得できなかった場合は、ゼロ文字列を返します。
	 *
	 * @og.rev 3.8.5.3 (2006/08/07) 取り出し方法を少し修正
	 * @og.rev 5.5.1.2 (2012/04/06) フォーマットセルを実行して、その結果を再帰的に処理する。
	 * @og.rev 6.0.3.0 (2014/11/13) セルフォーマットエラー時に、RuntimeException を throw しない。
	 *
	 * @param	oCell EXCELのセルオブジェクト
	 *
	 * @return	セルの値
	 */
	public static String getValue( final Cell oCell ) {
		if( oCell == null ) { return null; }

		String strText = "";
		final int nCellType = oCell.getCellType();
		switch(nCellType) {
			case Cell.CELL_TYPE_NUMERIC:
					strText = getNumericTypeString( oCell );
					break;
			case Cell.CELL_TYPE_STRING:
	// POI3.0		strText = oCell.getStringCellValue();
					final RichTextString richText = oCell.getRichStringCellValue();
					if( richText != null ) {
						strText = richText.getString();
					}
					break;
			case Cell.CELL_TYPE_FORMULA:
	// POI3.0		strText = oCell.getStringCellValue();
					// 5.5.1.2 (2012/04/06) フォーマットセルを実行して、その結果を再帰的に処理する。
					final Workbook wb = oCell.getSheet().getWorkbook();
					final CreationHelper crateHelper = wb.getCreationHelper();
					final FormulaEvaluator evaluator = crateHelper.createFormulaEvaluator();

					try {
						strText = getValue(evaluator.evaluateInCell(oCell));
					}
					catch ( Throwable th ) {
						final String errMsg = "セルフォーマットが解析できません。Formula=[" + oCell.getCellFormula() + "]"
									+ CR + getCellMsg( oCell );
	//					throw new RuntimeException( errMsg,th );
						System.err.println( errMsg );								// 6.0.3.0 (2014/11/13)
						System.err.println( StringUtil.ogStackTrace( th ) );
					}
					break;
			case Cell.CELL_TYPE_BOOLEAN:
					strText = String.valueOf(oCell.getBooleanCellValue());
					break;
			case Cell.CELL_TYPE_BLANK :
					break;
			case Cell.CELL_TYPE_ERROR:
					break;
			default :
				break;
		}
		return strText ;
	}

	/**
	 * セル値が数字の場合に、数字か日付かを判断して、対応する文字列を返します。
	 *
	 * @og.rev 3.8.5.3 (2006/08/07) 新規追加
	 * @og.rev 5.5.7.2 (2012/10/09) HybsDateUtil を利用するように修正します。
	 *
	 * @param	oCell EXCELのセルオブジェクト
	 *
	 * @return	数字の場合は、文字列に変換した結果を、日付の場合は、"yyyyMMddHHmmss" 形式で返します。
	 */
	public static String getNumericTypeString( final Cell oCell ) {
		final String strText ;

		final double dd = oCell.getNumericCellValue() ;
		if( DateUtil.isCellDateFormatted( oCell ) ) {
			strText = HybsDateUtil.getDate( DateUtil.getJavaDate( dd ).getTime() , "yyyyMMddHHmmss" );	// 5.5.7.2 (2012/10/09) HybsDateUtil を利用
		}
		else {
			final NumberFormat numFormat = NumberFormat.getInstance();
			if( numFormat instanceof DecimalFormat ) {
				((DecimalFormat)numFormat).applyPattern( "#.####" );
			}
			strText = numFormat.format( dd );
		}
		return strText ;
	}

	/**
	 * 全てのSheetに対して、autoSizeColumn設定を行います。
	 *
	 * 重たい処理なので、ファイルの書き出し直前に一度だけ実行するのがよいでしょう。
	 * autoSize設定で、カラム幅が大きすぎる場合、現状では、
	 * 初期カラム幅のmaxColCount倍を限度に設定します。
	 * ただし、maxColCount がマイナスの場合は、無制限になります。
	 *
	 * @og.rev 6.0.2.0 (2014/09/19) 新規作成
	 *
	 * @param	wkbook		処理対象のWorkbook
	 * @param	maxColCount	最大幅を標準セル幅の何倍にするかを指定。マイナスの場合は、無制限
	 * @param	dataStRow	データ行の開始位置。未設定時は、-1
	 */
	public static void autoCellSize( final Workbook wkbook , final int maxColCount , final int dataStRow ) {
		final int shCnt = wkbook.getNumberOfSheets();

		for( int shNo=0; shNo<shCnt; shNo++ ) {
			final Sheet sht = wkbook.getSheetAt( shNo );
			final int defW = sht.getDefaultColumnWidth();		// 標準カラムの文字数
			final int maxWidth = defW*256*maxColCount ;		// Widthは、文字数(文字幅)*256*最大セル数

			int stR = sht.getFirstRowNum();
			final int edR = sht.getLastRowNum();

			final Row rowObj = sht.getRow( stR );
			final int stC = rowObj.getFirstCellNum();
			final int edC = rowObj.getLastCellNum();				// 含まない

			// SheetUtil を使用して、計算範囲を指定します。
			if( stR < dataStRow ) { stR = dataStRow; }		// 計算範囲
			for( int colNo=stC; colNo<edC; colNo++ ) {
				final double wpx = SheetUtil.getColumnWidth( sht,colNo,true,stR,edR );
				if( wpx >= 0.0d ) {							// Cellがないと、マイナス値が戻る。
					int wd = (int)Math.ceil(wpx * 256) ;
					if( maxWidth >= 0 && wd > maxWidth ) { wd = maxWidth; }	// 最大値が有効な場合は、置き換える
					sht.setColumnWidth( colNo,wd );
				}
			}

			// Sheet#autoSizeColumn(int) を使用して、自動計算させる場合。
	//		for( int colNo=stC; colNo<edC; colNo++ ) {
	//			sht.autoSizeColumn( colNo );
	//			if( maxWidth >= 0 ) {					// 最大値が有効な場合は、置き換える
	//				int wd = sht.getColumnWidth( colNo );
	//				if( wd > maxWidth ) { sht.setColumnWidth( colNo,maxWidth ); }
	//			}
	//		}
		}
	}

	/**
	 * 指定の Workbook の全Sheetを対象に、空行を取り除き、全体をシュリンクします。
	 *
	 * この処理は、#saveFile( String ) の直前に行うのがよいでしょう。
	 *
	 * ここでは、Row を逆順にスキャンし、Cellが 存在しない間は、行を削除します。
	 * 途中の空行の削除ではなく、最終行かららの連続した空行の削除です。
	 * 
	 * isCellDel=true を指定すると、Cellの末尾削除を行います。
	 * 有効行の最後のCellから空セルを削除していきます。
	 * 表形式などの場合は、Cellのあるなしで、レイアウトが崩れる場合がありますので
	 * 処理が不要な場合は、isCellDel=false を指定してください。
	 *
	 * @og.rev 6.0.2.0 (2014/09/19) 新規作成
	 * @og.rev 6.0.2.3 (2014/10/10) CellStyle の有無も判定基準に含めます。
	 * @og.rev 6.0.2.5 (2014/10/31) Cellの開始、終了番号が、マイナスのケースの対応
	 *
	 * @param	wkbook		処理対象のWorkbook
	 * @param	isCellDel	Cellの末尾削除を行うかどうか(true:行う/false:行わない)
	 */
	public static void activeWorkbook( final Workbook wkbook , final boolean isCellDel ) {
		final int shCnt = wkbook.getNumberOfSheets();
		for( int shNo=0; shNo<shCnt; shNo++ ) {
			final Sheet sht = wkbook.getSheetAt( shNo );

			final int stR = sht.getFirstRowNum();
			final int edR = sht.getLastRowNum();

			boolean isRowDel = true;											// 行の削除は、Cellが見つかるまで。
			for( int rowNo=edR; rowNo>=stR && rowNo>=0; rowNo-- ) {				// 逆順に処理します。
				final Row rowObj = sht.getRow( rowNo );
				if( rowObj != null ) {
					final int stC = rowObj.getFirstCellNum();
					final int edC = rowObj.getLastCellNum();
//					for( int colNo=edC; colNo>=stC; colNo-- ) {
					for( int colNo=edC; colNo>=stC && colNo>=0; colNo-- ) {		// 6.0.2.5 (2014/10/31) stC,edC が、マイナスのケースがある。
						final Cell colObj = rowObj.getCell( colNo );
						if( colObj != null ) {
							final String val = getValue( colObj );
							if( colObj.getCellType() != Cell.CELL_TYPE_BLANK && val != null && val.length() > 0 ) { 
								isRowDel = false;					// 一つでも現れれば、行の削除は中止
								break;
							}
							// 6.0.2.3 (2014/10/10) CellStyle の有無も判定基準に含めます。
							else if( colObj.getCellStyle() != null ) {
								isRowDel = false;					// 一つでも現れれば、行の削除は中止
								break;
							}
							else if( isCellDel ) {
								rowObj.removeCell( colObj );		// CELL_TYPE_BLANK の場合は、削除
							}
						}
					}
					if( isRowDel ) { sht.removeRow( rowObj );	}
					else if( !isCellDel ) { break; }				// Cell の末尾削除を行わない場合は、break すればよい。
				}
			}
		}
	}

	/**
	 * ファイルから、Workbookオブジェクトを新規に作成します。
	 *
	 * @og.rev 6.0.2.3 (2014/10/10) 新規作成
	 *
	 * @param	fname	入力ファイル
	 * @return	Workbookオブジェクト
	 * @og.rtnNotNull
	 */
	public static Workbook createWorkbook( final String fname ) {
		InputStream fis = null;
		try {
			// File オブジェクトでcreate すると、ファイルがオープンされたままになってしまう。
			fis = new BufferedInputStream( new FileInputStream( fname ) );
			return WorkbookFactory.create( fis );
		}
		catch( IOException ex ) {
			final String errMsg = "ファイル読込みエラー[" + fname + "]" + CR + ex.getMessage() ;
			throw new RuntimeException( errMsg,ex );
		}
		catch( InvalidFormatException ex ) {
			final String errMsg = "ファイル形式エラー[" + fname + "]" + CR + ex.getMessage() ;
			throw new RuntimeException( errMsg,ex );
		}
		finally {
			Closer.ioClose( fis );
		}
	}

	/**
	 * シート一覧を、Workbook から取得します。
	 *
	 * 取得元が、Workbook なので、xls , xlsx どちらの形式でも取り出せます。
	 *
	 * EXCEL上のシート名を、配列で返します。
	 *
	 * @og.rev 6.0.2.3 (2014/10/10) 新規作成
	 *
	 * @param	wkbook Workbookオブジェクト
	 * @return	シート名の配列
	 */
	public static String[] getSheetNames( final Workbook wkbook ) {
		final int shCnt = wkbook.getNumberOfSheets();

		String[] shtNms = new String[shCnt];

		for( int i=0; i<shCnt; i++ ) {
			final Sheet sht = wkbook.getSheetAt( i );
			shtNms[i] = sht.getSheetName();
		}

		return shtNms;
	}

	/**
	 * 名前定義一覧を取得します。
	 *
	 * EXCEL上に定義された名前を、配列で返します。
	 * ここでは、名前とFormulaをタブで連結した文字列を配列で返します。
	 * Name オブジェクトを削除すると、EXCELが開かなくなったりするので、
	 * 取りあえず一覧を作成して、手動で削除してください。
	 * なお、名前定義には、非表示というのがありますので、ご注意ください。
	 *
	 * ◆ 非表示になっている名前の定義を表示にする EXCEL VBA マクロ
	 * http://dev.classmethod.jp/tool/excel-delete-name/
	 *    Sub VisibleNames()
	 *        Dim name
	 *        For Each name In ActiveWorkbook.Names
	 *            If name.Visible = False Then
	 *                name.Visible = True
	 *            End If
	 *        Next
	 *        MsgBox "すべての名前の定義を表示しました。", vbOKOnly
	 *    End Sub
	 *
	 * ※ EXCEL2010 数式タブ→名前の管理 で、複数選択で、削除できます。
	 *    ただし、非表示の名前定義は、先に表示しないと、削除できません。
	 * ◆ 名前の一括削除 EXCEL VBA マクロ
	 * http://komitsudo.blog70.fc2.com/blog-entry-104.html
	 *    Sub DeleteNames()
	 *        Dim name
	 *        On Error Resume Next
	 *        For Each name In ActiveWorkbook.Names
	 *            If Not name.BuiltIn Then
	 *                name.Delete
	 *            End If
	 *        Next
	 *        MsgBox "すべての名前の定義を削除しました。", vbOKOnly
	 *    End Sub
	 *
	 * @og.rev 6.0.2.3 (2014/10/10) 新規作成
	 *
	 * @param	wkbook Workbookオブジェクト
	 * @return	名前定義(名前+TAB+Formula)の配列
	 * @og.rtnNotNull
	 */
	public static String[] getNames( final Workbook wkbook ) {
		final int cnt = wkbook.getNumberOfNames();

		final Set<String> nmSet = new TreeSet<String>();

		for( int i=0; i<cnt; i++ ) {
			String name	= null;
			String ref	= null;

			final Name nm = wkbook.getNameAt(i);
			try {
				name = nm.getNameName();
				ref  = nm.getRefersToFormula();
			}
	//		catch( Exception ex ) {					// 6.1.0.0 (2014/12/26) refactoring
			catch( RuntimeException ex ) {
				final String errMsg = "POIUtil:RefersToFormula Error! name=[" + name + "]" + ex.getMessage() ;
				System.out.println( errMsg );
				// Excel97形式の場合、getRefersToFormula() でエラーが発生することがある。
			}

			nmSet.add( name + "\t" + ref );

			// 削除するとEXCELが壊れる？ なお、削除時には逆順で廻さないとアドレスがずれます。
			// if( nm.isDeleted() ) { wkbook.removeName(i); }
		}

		return nmSet.toArray( new String[nmSet.size()] );
	}

	/**
	 * 書式のスタイル一覧を取得します。
	 *
	 * EXCEL上に定義された書式のスタイルを、配列で返します。
	 * 書式のスタイルの名称は、CellStyle にメソッドが定義されていません。
	 * 実クラスである HSSFCellStyle にキャストして使用する
	 * 必要があります。(XSSFCellStyle にも名称を取得するメソッドがありません。)
	 *
	 * ※ EXCEL2010 ホームタブ→セルのスタイル は、一つづつしか削除できません。
	 *    マクロは、開発タブ→Visual Basic で、挿入→標準モジュール を開き
	 *    テキストを張り付けてください。
	 *    実行は、開発タブ→マクロ で、マクロ名を選択して、実行します。
	 *    最後は、削除してください。
	 *
	 * ◆ スタイルの一括削除 EXCEL VBA マクロ
	 * http://komitsudo.blog70.fc2.com/blog-entry-104.html
	 *    Sub DeleteStyle()
	 *        Dim styl
	 *        On Error Resume Next
	 *        For Each styl In ActiveWorkbook.Styles
	 *            If Not styl.BuiltIn Then
	 *                styl.Delete
	 *            End If
	 *        Next
	 *        MsgBox "すべての追加スタイルを削除しました。", vbOKOnly
	 *    End Sub
	 *
	 * ◆ 名前の表示、削除、スタイルの削除の一括実行 EXCEL VBA マクロ
	 *    Sub AllDelete()
	 *        Call VisibleNames
	 *        Call DeleteNames
	 *        Call DeleteStyle
	 *        MsgBox "すべての処理を完了しました。", vbOKOnly
	 *    End Sub
	 *
	 * @og.rev 6.0.2.3 (2014/10/10) 新規作成
	 *
	 * @param	wkbook Workbookオブジェクト
	 * @return	書式のスタイル一覧
	 * @og.rtnNotNull
	 */
	public static String[] getStyleNames( final Workbook wkbook ) {
		final int cnt = wkbook.getNumCellStyles();		// return 値は、short

		final Set<String> nmSet = new TreeSet<String>();

		for( int s=0; s<cnt; s++ ) {
			final CellStyle cs = wkbook.getCellStyleAt( (short)s );
			if( cs instanceof HSSFCellStyle ) {
				final HSSFCellStyle hcs = (HSSFCellStyle)cs;
				final String name = hcs.getUserStyleName();
				if( name != null ) { nmSet.add( name ); }
				else {												// この処理は不要かも。
					final HSSFCellStyle pst = hcs.getParentStyle();
					if( pst != null ) {
						final String pname = pst.getUserStyleName();
						if( pname != null ) { nmSet.add( pname ); }
					}
				}
			}
		}

		return nmSet.toArray( new String[nmSet.size()] );
	}

	/**
	 * セル情報を返します。
	 *
	 * エラー発生時に、どのセルでエラーが発生したかの情報を取得できるようにします。
	 *
	 * @og.rev 6.0.2.0 (2014/09/19) 新規作成
	 * @og.rev 6.0.3.0 (2014/11/13) セル情報を作成する時に、値もセットします。
	 *
	 * @param	oCell EXCELのセルオブジェクト
	 * @return	セル情報の文字列
	 */
	public static String getCellMsg( final Cell oCell ) {
		String lastMsg = null;

		if( oCell != null ) {
			final String shtNm = oCell.getSheet().getSheetName();
			final int  rowNo = oCell.getRowIndex();
			final int  celNo = oCell.getColumnIndex();
			final String  celKigo = ( celNo <= 26 ) ? 
										"" + (char)('A'+ celNo%26)
									:	"" + (char)('A'+ celNo/26-1) + (char)('A'+ celNo%26);

			// 6.0.3.0 (2014/11/13) セル情報を作成する時に、値もセットします。
//			lastMsg = " Sheet=" + shtNm + ", Row=" + rowNo + ", Cel=" + celNo ;
			lastMsg = " Sheet=" + shtNm + ", Row=" + rowNo + ", Cel=" + celNo
						 + "(" + celKigo + ") , Val=" + oCell.toString() ;
		}

		return lastMsg;
	}

	/**
	 * アプリケーションのサンプルです。
	 *
	 * 入力ファイル名 は必須で、第一引数固定です。
	 * 第二引数は、処理方法で、指定しない場合は、-ALL として処理します。
	 *
	 * Usage: java org.opengion.fukurou.util.POIUtil 入力ファイル名 [パラメータ]
	 *   -A(LL)        ･･･ ALL 一括処理(初期値)
	 *   -W(ORD)       ･･･ Word
	 *   -P(PT)        ･･･ PoworPoint
	 *   -E(XCEL)      ･･･ Excel
	 *   -S(heet)      ･･･ Sheet名一覧
	 *   -N(AME)       ･･･ NAME:名前定義
	 *   -C(ellStyle)  ･･･ CellStyle:書式のスタイル
	 *
	 * @og.rev 6.0.2.0 (2014/09/19) 新規作成
	 *
	 * @param	args	コマンド引数配列
	 */
	public static void main( final String[] args ) {
		final String usageMsg = "Usage: java org.opengion.fukurou.util.POIUtil 入力ファイル名 [パラメータ]" + "\n" +
								"\t -A(LL)        ･･･ ALL 一括処理(初期値)      \n" +
								"\t -W(ORD)       ･･･ Word                      \n" +
								"\t -P(PT)        ･･･ PoworPoint                \n" +
								"\t -E(XCEL)      ･･･ Excel                     \n" +
								"\t -S(heet)      ･･･ Sheet名一覧               \n" +
								"\t -N(AME)       ･･･ NAME:名前定義             \n" +
								"\t -C(ellStyle)  ･･･ CellStyle:書式のスタイル  \n" ;
		if( args.length == 0 ) {
			System.err.println( usageMsg );
			return ;
		}

		final String fname = args[0];
		final char   type  = (args.length == 1) ? 'A' : args[1].charAt(1) ;

		switch( type ) {
			case 'A' :  System.out.println( POIUtil.getText(fname) );
						break;
			case 'W' :  POIUtil.wordReader(
							fname,
							new POIEvent() {
								/**
								 * 読み込みイベント処理のメソッドです。
								 *
								 * @param	text テキスト
								 * @param	cmnt 説明
								 * @param	args... 可変長引数(ページ数や行番号など)
								 */
								public void readText( final String text , final String cmnt , final int... args ) {
									System.out.println( "S[" + args[0] + "],P[" + fname + "],G[" + args[2] + "],C[" + cmnt + "]=" + text );
								}
							}
						);
						break;
			case 'P' :  POIUtil.pptReader(
							fname,
							new POIEvent() {
								/**
								 * 読み込みイベント処理のメソッドです。
								 *
								 * @param	text テキスト
								 * @param	cmnt 説明
								 * @param	args... 可変長引数(ページ数や行番号など)
								 */
								public void readText( final String text , final String cmnt , final int... args ) {
						//			System.out.println( "S[" + args[0] + "],T[" + fname + "],C[" + cmnt + "]=" + text );
									System.out.println( "S[" + args[0] + "],T[" + fname + "]=" + text );
								}
							}
						);
						break;
			case 'E' :  POIUtil.excelReader(
							fname,
							new ExcelReaderEvent() {

								/**
								 * シートの読み取り開始時にイベントが発生します。
								 *
								 * @param   shtNm  シート名
								 * @param   shtNo  シート番号(0～)
								 * @return  true:シートの読み取り処理を継続します/false:このシートは読み取りません。
								 */
								public boolean startSheet( final String shtNm,final int shtNo ) {
									System.out.println( "S[" + shtNo + "]=" + shtNm );
									return super.startSheet( shtNm,shtNo );
								}

						//		public void columnNames( final String[] names ) {
						//			System.out.println( "NM=" + java.util.Arrays.toString( names ) );
						//		}

						//		public void values( final String[] vals,final int rowNo ) {
						//			System.out.println( "V[" + rowNo + "]=" + java.util.Arrays.toString( vals ) );
						//		}

						//		public boolean isSkip( final int rowNo ) {
						//			super.isSkip( rowNo );
						//			return false;
						//		}

								/**
								 * 読み取り状態の時に、rowNo,colNo にあるセルの値を引数にイベントが発生します。
								 *
								 * @param   val     文字列値
								 * @param   rowNo   行番号(0～)
								 * @param   colNo   列番号(0～)
								 * @return  読み取りするかどうか(true:読み取りする/false:読み取りしない)
								 */
								public boolean value( final String val,final int rowNo,final int colNo ) {
									System.out.println( "R[" + rowNo + "],C[" + colNo + "]=" + val );
									return super.value( val,rowNo,colNo );
								}
							}
						);
						break;
			case 'S' :  final String[] shts = POIUtil.getSheetNames( POIUtil.createWorkbook(fname) );
						System.out.println( "No:\tSheetName" );
						for( int i=0; i<shts.length; i++ ) {
							System.out.println( i + "\t" + shts[i] );
						}
						break;
			case 'N' :  final String[] nms = POIUtil.getNames( POIUtil.createWorkbook(fname) );
						System.out.println( "No:\tName\tFormula" );
						for( int i=0; i<nms.length; i++ ) {
							System.out.println( i + "\t" + nms[i] );
						}
						break;
			case 'C' :  final String[] sns = POIUtil.getStyleNames( POIUtil.createWorkbook(fname) );
						System.out.println( "No:\tStyleName" );
						for( int i=0; i<sns.length; i++ ) {
							System.out.println( i + "\t" + sns[i] );
						}
						break;
			default :   System.err.println( usageMsg );
						break;
		}
	}
}
