/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.db;

import java.math.BigDecimal;
import java.sql.ResultSet;
// import java.sql.ResultSetMetaData;		// 6.0.4.0 (2014/11/28) 廃止
import java.sql.SQLException;
// import java.sql.Types;					// 6.0.2.5 (2014/10/31) , 6.0.4.0 (2014/11/28) 廃止
import java.text.DecimalFormat;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
// import java.util.Locale;
import java.util.Map;

// import org.opengion.fukurou.db.DBUtil;
import org.opengion.fukurou.db.ResultSetValue;				// 6.0.4.0 (2014/11/28)
import org.opengion.fukurou.util.StringUtil;
import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;
// import org.opengion.hayabusa.resource.LabelData;
import org.opengion.hayabusa.resource.ResourceManager;
import static org.opengion.fukurou.util.HybsConst.BUFFER_MIDDLE;	// 6.1.0.0 (2014/12/26) refactoring

/**
 * DBTableModelを継承した TableModelの編集設定による変換を行うための実装クラスです。
 *
 * このクラスでは、オブジェクト初期化後は、通常のDBTableModelと同じ振る舞いをします。
 * オブジェクト初期化時(createメソッド呼び出し時)に、検索結果オブジェクトから直接、編集設定に
 * 応じて変換されたDBTableModelを生成します。
 *
 * このような実装を行う理由は、メモリ使用量を節約するためです。
 * この編集設定では、集計機能を備えていますが、一旦DBTableModel作成後に集計処理を行うと、
 * メモリを大量に使用する恐れがあるため、検索結果オブジェクトから直接集計処理を行い、DBTableModelを
 * 生成しています。
 *
 * DBTableModel インターフェースは，データベースの検索結果(Resultset)をラップする
 * インターフェースとして使用して下さい。
 *
 * @og.rev 5.3.6.0 (2011/06/01) 新規作成
 * @og.group テーブル管理
 *
 * @version  5.0
 * @author   Hiroki Nakamura
 * @since    JDK6.0,
 */
public class DBTableModelEditor extends DBTableModelImpl {
	private static final String			JS		= HybsSystem.JOINT_STRING;
	private static final DecimalFormat	FORMAT	= new DecimalFormat( "0.#########" );

	private int rowCountColumn = -1;
	private DBEditConfig config;

	/**
	 * DBTableModel を設定し、このオブジェクトを初期化します。
	 *
	 * @og.rev 5.7.1.2 (2013/12/20) msg ⇒ errMsg 変更
	 * @og.rev 6.0.2.1 (2014/09/26) queryタグが複数あり、mainTrans=false で制御されていない場合、エラーが発生する
	 * @og.rev 6.0.2.5 (2014/10/31) FireBardでの日付型取得対応
	 * @og.rev 6.0.4.0 (2014/11/28) ResultSetValue を使用するように変更。
	 * @og.rev 6.0.4.0 (2014/11/28) queryタグが複数ある場合の事前チェックの条件訂正
	 *
	 * @param	result			検索結果オブジェクト
	 * @param	skipRowCount	読み飛ばし件数
	 * @param	maxRowCount		最大検索件数
	 * @param	resource		ResourceManagerオブジェクト
	 * @param	config			編集設定オブジェクト
	 * @throws	SQLException データベースアクセスエラー
	 */
	public void create( final ResultSet result, final int skipRowCount, final int maxRowCount, final ResourceManager resource, final DBEditConfig config ) throws SQLException {
		if( result == null || config == null || resource == null ) {
			final String errMsg = "DBTableModelまたは、DBEditConfigが設定されていません。";
			throw new HybsSystemException( errMsg );	// 5.7.1.2 (2013/12/20) msg ⇒ errMsg 変更
		}

		this.config = config;

		/**********************************************************************
		 * 各パラメーターの初期化処理
		 **********************************************************************/
		// 6.0.4.0 (2014/11/28) ResultSetValue を使用するように変更。
		final ResultSetValue rsv = new ResultSetValue( result );		// 6.0.4.0 (2014/11/28)
//		ResultSetMetaData metaData	= result.getMetaData();
//		int colCnt = metaData.getColumnCount();
		int colCnt = rsv.getColumnCount();						// 6.0.4.0 (2014/11/28)

		if( config.useGroup() || config.useSubTotal() || config.useTotal() || config.useGrandTotal() ) {
			rowCountColumn = colCnt;
			colCnt++;
		}
		init( colCnt );

		DBColumn[] dbColumn = new DBColumn[numberOfColumns];
//		boolean   isOther = false;									// 6.0.2.5 (2014/10/31) FireBardでの日付型取得対応
//		int[] types  = new int[numberOfColumns];
		boolean[] sumFilter			= new boolean[numberOfColumns];
		boolean[] groupFilter		= new boolean[numberOfColumns];
		boolean[] subTotalFilter	= new boolean[numberOfColumns];
		boolean[] totalFilter		= new boolean[numberOfColumns];
		boolean   sumFilterCheck	= false;						// 6.0.2.1 (2014/09/26)
		if( config.useGrandTotal() ) { sumFilterCheck = true; }		// 6.0.4.0 (2014/11/28)
		for( int column=0; column<numberOfColumns; column++ ) {
			String name = null;
			// 6.1.0.0 (2014/12/26) refactoring : Avoid if (x != y) ..; else ..;
			if( column == rowCountColumn ) {
				name = "rowCount";
				dbColumn[column] = resource.makeDBColumn( name );
			}
			else {
				name = rsv.getColumnName(column);					// 6.0.4.0 (2014/11/28)
				dbColumn[column] = resource.getDBColumn( name );
				if( dbColumn[column] == null ) {
					dbColumn[column] = DBTableModelUtil.makeDBColumn( name,column,rsv,resource );	// 6.0.4.0 (2014/11/28)
				}
			}
//			if( column != rowCountColumn ) {
////				name = metaData.getColumnLabel(column+1).toUpperCase(Locale.JAPAN);
//				name = rsv.getColumnName(column);					// 6.0.4.0 (2014/11/28)
//				dbColumn[column] = resource.getDBColumn( name );
//				if( dbColumn[column] == null ) {
////					LabelData labelData  = resource.getLabelData( name );
////					dbColumn[column] = DBTableModelUtil.makeDBColumn( name,labelData,metaData,column,resource.getLang() );
//					dbColumn[column] = DBTableModelUtil.makeDBColumn( name,column,rsv,resource );	// 6.0.4.0 (2014/11/28)
//				}
//				// 6.0.4.0 (2014/11/28) ResultSetValue を使用するように変更。
////				// 6.0.2.5 (2014/10/31) types 配列のセットと、isOther フラグセット
////				int clmType = metaData.getColumnType(column+1);
////				types[column] = clmType;
////				if( clmType == Types.CLOB || clmType == Types.ROWID || clmType == Types.TIMESTAMP ) {	// JDK 6.0以降 ROWID
////					isOther = true;
////				}
//			}
//			else {
//				name = "rowCount";
//				dbColumn[column] = resource.makeDBColumn( name );
////				types[column] = Types.INTEGER ;						// 6.0.2.5 (2014/10/31)
//			}

			setDBColumn( column,dbColumn[column] );
			sumFilter[column]		= config.isSumClm( name );
			groupFilter[column]		= config.isGroupClm( name );
			subTotalFilter[column]	= config.isSubTotalClm( name );
			totalFilter[column]		= config.isTotalClm( name );
			// 6.0.4.0 (2014/11/28) queryタグが複数ある場合の事前チェックの条件訂正
//			if( sumFilter[column] ) { sumFilterCheck = true; }		// 6.0.2.1 (2014/09/26)
			if( sumFilter[column] || groupFilter[column] || subTotalFilter[column] || totalFilter[column] ) {
				sumFilterCheck = true;
			}
		}

		/**********************************************************************
		 * 集計、ソート、合計処理
		 **********************************************************************/
		// 集計キーに基づく集計処理を行いデータを追加します。
		if( config.useGroup() ) {
			// 6.0.4.0 (2014/11/28) ResultSetValue を使用するように変更。
//			addGroupRows( result, types, skipRowCount, maxRowCount, sumFilter, groupFilter  );
			addGroupRows( rsv, skipRowCount, maxRowCount, sumFilter, groupFilter );
		}
		// 通常と同じように結果カーソルからデータを読込みデータを追加します。
		else {
	 		// 5.5.2.4 (2012/05/16) int[] types は使われていないので、削除します。
			// 6.0.2.5 (2014/10/31) int[] types 復活。isOther ﾌﾗｸﾞも使います。
//			addPlainRows( result, skipRowCount, maxRowCount );
			// 6.0.4.0 (2014/11/28) ResultSetValue を使用するように変更。
//			addPlainRows( result, types, skipRowCount, maxRowCount , isOther );
			addPlainRows( rsv, skipRowCount, maxRowCount );
		}

		// ソート処理
		if( getRowCount() > 0 && config.useOrderBy() ) {
			sort();
		}

		// 小計・合計行を追加します。
		if( getRowCount() > 0 && !isOverflow()
			&& ( config.useSubTotal() || config.useTotal() || config.useGrandTotal() ) ) {

		 	// 6.0.2.1 (2014/09/26) queryタグが複数あり、mainTrans=false で制御されていない場合、エラーが発生する
			if( !sumFilterCheck ) {
				final String errMsg = "小計、合計カラムが存在しません。"
								+ " これは、queryタグが複数あり、mainTrans=false で制御されていない可能性があります。" ;
				throw new HybsSystemException( errMsg );
			}

			addTotalRows( maxRowCount, resource, sumFilter, groupFilter, subTotalFilter, totalFilter );
		}
	}

	/**
	 * 集計キーの設定に基づき、DBTableModelの行を追加します。
	 * 内部的には、キーブレイクではなく、内部マップにより集計処理を行っているため、
	 * 集計キーが検索順により散在した場合でも1まとまりで集計されます。
	 *
	 * @og.rev 5.3.9.0 (2011/09/01) 値がNULLの場合にエラーになるバグを修正
	 * @og.rev 5.6.1.0 (2013/02/01) doubleをBigDecimalに変更
	 * @og.rev 6.0.4.0 (2014/11/28) ResultSetValue を使用するように変更。
	 *
	 * @param rsv ResultSetValueオブジェクト
	 * @param skipRowCount 読み飛ばし件数
	 * @param maxRowCount 最大検索件数
	 * @param sumFilter 集計項目フィルター
	 * @param groupFilter グループキーフィルター
	 * @throws SQLException データベースアクセスエラー
	 */
//	private void addGroupRows( final ResultSet result, final int[] types, final int skipRowCount, final int maxRowCount
	private void addGroupRows( final ResultSetValue rsv, final int skipRowCount, final int maxRowCount
								, final boolean[] sumFilter, final boolean[] groupFilter ) throws SQLException {
		int numberOfRows = 0;
//		while( numberOfRows < skipRowCount && result.next() ) {
		while( numberOfRows < skipRowCount && rsv.next() ) {
			// 注意 resultSet.next() を先に判定すると必ず１件読み飛ばしてしまう。
			numberOfRows ++ ;
		}
		numberOfRows = 0;

		final Map<String,String[]> groupLinkedMap	= new LinkedHashMap<String,String[]>();
		final Map<String,Integer> groupCountMap	= new HashMap<String,Integer>();
		final Map<String,BigDecimal[]> sumMap		= new HashMap<String,BigDecimal[]>();	// 5.6.1.0 (2013/02/01)
//		while( numberOfRows < maxRowCount && result.next() ) {
		final StringBuilder groupKey = new StringBuilder( BUFFER_MIDDLE );		// 6.1.0.0 (2014/12/26) refactoring
		while( numberOfRows < maxRowCount && rsv.next() ) {
//			final StringBuilder groupKey = new StringBuilder();
			groupKey.setLength(0);															// 6.1.0.0 (2014/12/26) refactoring
			BigDecimal[] sumVals = new BigDecimal[config.getSumClmCount()]; // 5.6.1.0 (2013/02/01) 
			String[]   groupVals = new String[config.getGroupClmCount()];
			int sc = 0;
			int gc = 0;
			for( int column=0; column<numberOfColumns; column++ ) {
				if( column != rowCountColumn ) {
//					String val = DBUtil.getValue( result, column, types[column] );
					final String val = rsv.getValue( column );
					if( sumFilter[column] ) {
						// 5.3.9.0 (2011/09/01) 値がNULLの場合の対応漏れ
						// sumVals[sc++] = ( val != null && val.length() > 0 ? Double.valueOf( val ) : 0 );
						sumVals[sc++] = ( val != null && val.length() > 0 ? new BigDecimal( val ) : BigDecimal.ZERO ); // 5.6.1.0 (2013/02/01)
					}
					if( groupFilter[column] ) {
						groupVals[gc++] = val;
						groupKey.append( val ).append( JS );
					}
				}
			}

			final String key = groupKey.toString();
			int groupCount = 0;
			if( groupLinkedMap.containsKey( key ) ) {
				final BigDecimal[] eSumVals = sumMap.get( key ); // 5.6.1.0 (2013/02/01)
				for( int i=0; i<config.getSumClmCount(); i++ ) {
					sumVals[i] = sumVals[i] == null ? BigDecimal.ZERO : sumVals[i].add( eSumVals[i] ); // 5.6.1.0 (2013/02/01)
				}
				sumMap.put( key, sumVals );
				groupCount = groupCountMap.get( key ).intValue() + 1;
			}
			else {
				groupLinkedMap.put( key, groupVals );
				groupCount = 1;
				numberOfRows++;
			}
			sumMap.put( key, sumVals );
			groupCountMap.put( key, Integer.valueOf( groupCount ) );
		}

		for( final Map.Entry<String, String[]> entry : groupLinkedMap.entrySet() ) {
			final String key = entry.getKey();
			addRow( groupFilter, entry.getValue(), groupCountMap.get( key ), sumFilter, sumMap.get( key ) );
		}

		// 最大件数が、超えた場合でかつ次のデータがある場合は、オーバーフロー
//		if( numberOfRows >= maxRowCount && result.next() ) {
		if( numberOfRows >= maxRowCount && rsv.next() ) {
			setOverflow( true );
		}
	}

	/**
	 * 検索結果オブジェクトを順に読み取り、そのままDBTableModelの行を追加します。
	 *
	 * @og.rev 5.5.2.4 (2012/05/16) int[] types は使われていないので、削除します。
	 * @og.rev 6.0.2.5 (2014/10/31) int[] types 復活。isOther ﾌﾗｸﾞも使います。
	 * @og.rev 6.0.4.0 (2014/11/28) ResultSetValue を使用するように変更。
	 *
	 * @param rsv ResultSetValueオブジェクト
	 * @param skipRowCount 読み飛ばし件数
	 * @param maxRowCount 最大検索件数
	 * @throws	SQLException データベースアクセスエラー
	 */
//	private void addPlainRows( final ResultSet result, final int skipRowCount, final int maxRowCount ) throws SQLException {
//	private void addPlainRows( final ResultSet result, final int[] types, final int skipRowCount, final int maxRowCount, final boolean isOther ) throws SQLException {
	private void addPlainRows( final ResultSetValue rsv, final int skipRowCount, final int maxRowCount ) throws SQLException {
		int numberOfRows = 0;
//		while( numberOfRows < skipRowCount && result.next() ) {
		while( numberOfRows < skipRowCount && rsv.next() ) {
			// 注意 resultSet.next() を先に判定すると必ず１件読み飛ばしてしまう。
			numberOfRows ++ ;
		}
		numberOfRows = 0;

		// 6.0.2.5 (2014/10/31) 行列のループなので、 CLOB 使用可否でループを分ける。
//		if( isOther ) {
			// 6.0.2.5 (2014/10/31) typesを考慮した処理
//			while( numberOfRows < maxRowCount && result.next() ) {
			while( numberOfRows < maxRowCount && rsv.next() ) {
				numberOfRows++ ;
				String[] columnValues = new String[numberOfColumns];
				for( int column=0; column<numberOfColumns; column++ ) {
					// 6.1.0.0 (2014/12/26) refactoring : Avoid if (x != y) ..; else ..;
					if( column == rowCountColumn ) {
						columnValues[column] = "";
					}
					else {
						columnValues[column] = rsv.getValue( column );
					}
//					if( column != rowCountColumn ) {
////						Object obj = result.getObject(column+1);
////						columnValues[column] = ( obj == null ? "" : String.valueOf( obj ) );
////						columnValues[column] = DBUtil.getValue( result, column, types[column] );
//						columnValues[column] = rsv.getValue( column );
//					}
//					else {
//						columnValues[column] = "";
//					}
				}
				addColumnValues( columnValues );
			}
//		}
//		else {
//			// 6.0.2.5 (2014/10/31) 従来の処理
//			while( numberOfRows < maxRowCount && result.next() ) {
//				numberOfRows++ ;
//				String[] columnValues = new String[numberOfColumns];
//				for( int column=0; column<numberOfColumns; column++ ) {
//					if( column != rowCountColumn ) {
//						Object obj = result.getObject(column+1);
//						columnValues[column] = ( obj == null ? "" : String.valueOf( obj ) );
//					}
//					else {
//						columnValues[column] = "";
//					}
//				}
//				addColumnValues( columnValues );
//			}
//		}

		// 最大件数が、超えた場合でかつ次のデータがある場合は、オーバーフロー
//		if( numberOfRows >= maxRowCount && result.next() ) {
		if( numberOfRows >= maxRowCount && rsv.next() ) {
			setOverflow( true );
		}
	}

	/**
	 * DBTableModelのソート処理を行います。
	 *
	 */
	private void sort() {
		// orderByClmsによる並び替え
		final DBTableModelSorter temp = new DBTableModelSorter();
		temp.setModel( this );
		final String[] oClms = StringUtil.csv2Array( config.getOrderByClms() );
		for( int i=oClms.length-1; i>=0; i-- ) {
			String oc = oClms[i];
			boolean ascending = true;
			if( oc.startsWith( "!" ) ) {
				oc = oc.substring( 1 );
				ascending = false;
			}
			final int clmNo = getColumnNo( oc );
			temp.sortByColumn( clmNo, ascending );
		}
		this.data = temp.data;
		this.rowHeader = temp.rowHeader;
	}

	/**
	 * DBTableModelからデータを読み取り、編集設定情報を元に合計行の追加処理を行います。
	 * 合計行の追加は、キーブレイクにより行われますので、同じキーが複数回出現した場合は、
	 * それぞれの行に対して、合計行が付加されます。
	 *
	 * @og.rev 5.3.7.0 (2011/07/01) 小計、合計行追加処理でオーバーフローフラグがセットされないバグを修正
	 * @og.rev 5.6.1.0 (2013/02/01) 誤差回避のため、doubleではなくdecimalで計算する
	 * @og.rev 5.6.8.1 (2013/09/13) 1行目が合計されていなかったので修正
	 * @og.rev 6.0.2.1 (2014/09/26) queryタグが複数あり、mainTrans=false で制御されていない場合、エラーが発生する対応
	 *
	 * @param	maxRowCount 最大検索件数
	 * @param	resource リソースマネージャー
	 * @param	sumFilter 集計項目フィルター
	 * @param	groupFilter グループキーフィルター
	 * @param	subTotalFilter 小計キーフィルター
	 * @param	totalFilter 合計キーフィルター
	 *
	 * @return	オーバーフローしたかどうか(最大件数が超えた場合でかつ次のデータがある場合は、true)
	 */
	private boolean addTotalRows( final int maxRowCount, final ResourceManager resource
									, final boolean[] sumFilter
									, final boolean[] groupFilter
									, final boolean[] subTotalFilter
									, final boolean[] totalFilter ) {

		final String subTotalLabel	= ( config.useSubTotal()   ? resource.makeDBColumn( "EDIT_SUBTOTAL_VALUE"   ).getLongLabel() : null );
		final String totalLabel		= ( config.useTotal()	   ? resource.makeDBColumn( "EDIT_TOTAL_VALUE"      ).getLongLabel() : null );
		final String grandTotalLabel= ( config.useGrandTotal() ? resource.makeDBColumn( "EDIT_GRANDTOTAL_VALUE" ).getLongLabel() : null );

		int numberOfRows = getRowCount();
		final int sumClmCount  = config.getSumClmCount();
		BigDecimal subTotalSum[]   = new BigDecimal[sumClmCount]; // 5.6.1.0 (2013/02/01)
		BigDecimal totalSum[]      = new BigDecimal[sumClmCount];
		BigDecimal grandTotalSum[] = new BigDecimal[sumClmCount];

		String lastSubTotalKey = null;
		String lastTotalKey    = null;

		int subTotalCount   = 0;
		int totalCount      = 0;
		int grandTotalCount = 0;
		int rowCount =0;

		int tblIdx = 0;
		final StringBuilder groupKey    = new StringBuilder( BUFFER_MIDDLE );	// 6.1.0.0 (2014/12/26) refactoring
		final StringBuilder subTotalKey = new StringBuilder( BUFFER_MIDDLE );	// 6.1.0.0 (2014/12/26) refactoring
		final StringBuilder totalKey    = new StringBuilder( BUFFER_MIDDLE );	// 6.1.0.0 (2014/12/26) refactoring
		while( numberOfRows < maxRowCount && tblIdx < getRowCount() ) {
			BigDecimal[] sumVals      = new BigDecimal[sumClmCount];	// 5.6.1.0 (2013/02/01)
//			final StringBuilder groupKey    = new StringBuilder( BUFFER_MIDDLE );
//			final StringBuilder subTotalKey = new StringBuilder( BUFFER_MIDDLE );
//			final StringBuilder totalKey    = new StringBuilder( BUFFER_MIDDLE );
			groupKey.setLength(0);															// 6.1.0.0 (2014/12/26) refactoring
			subTotalKey.setLength(0);														// 6.1.0.0 (2014/12/26) refactoring
			totalKey.setLength(0);															// 6.1.0.0 (2014/12/26) refactoring

			int sc = 0;
			for( int column=0; column<numberOfColumns; column++ ) {
				final String val = getValue( tblIdx, column );
				if( groupFilter[column] )		{ groupKey.append( val ).append( JS ); }
				if( sumFilter[column] )			{ sumVals[sc++] = ( val != null && val.length() > 0 ? new BigDecimal( val ) : BigDecimal.ZERO ); } // 5.6.1.0 (2013/02/01)
				if( subTotalFilter[column] )	{ subTotalKey.append( val ).append( JS ); }
				if( totalFilter[column] )		{ totalKey.append( val ).append( JS ); }
//				if( column == rowCountColumn )	{ rowCount = ( val != null && val.length() > 0 ? Integer.valueOf( val ) : 0 ); }
				if( column == rowCountColumn )	{ rowCount = ( val != null && val.length() > 0 ? Integer.parseInt( val ) : 0 ); }	// 6.0.2.4 (2014/10/17) メソッド間違い
			}

			// 小計キーブレイク処理
			if( numberOfRows < maxRowCount && config.useSubTotal() && lastSubTotalKey != null && lastSubTotalKey.length() > 0
				&& !lastSubTotalKey.equals( subTotalKey.toString() ) ) {
				addRow( subTotalFilter, subTotalLabel, subTotalCount, sumFilter, subTotalSum, tblIdx );
				subTotalSum = new BigDecimal[sumClmCount]; // 5.6.1.0 (2013/02/01)
				subTotalCount = 0;
				numberOfRows++;
				tblIdx++;
			}

			// 合計キーブレイク処理
			if( numberOfRows < maxRowCount && config.useTotal() && lastTotalKey != null && lastTotalKey.length() > 0
				&& !lastTotalKey.equals( totalKey.toString() ) ) {
				addRow( totalFilter, totalLabel, totalCount, sumFilter, totalSum, tblIdx );
				totalSum = new BigDecimal[sumClmCount]; // 5.6.1.0 (2013/02/01)
				totalCount = 0;
				numberOfRows++;
				tblIdx++;
			}

			// 小計、合計、総合計単位に集計項目の合計値を加算します。
			// 6.0.2.0 (2014/09/19) BigDecimal.ZERO.add で、null エラーが発生するのは、query が複数あり、mainTrans=false で制御されていない場合
//			for( int cnt=0; cnt<sumClmCount; cnt++ ) {
			for( int cnt=0; cnt<sc; cnt++ ) {
				subTotalSum[cnt]   = subTotalSum[cnt]   == null ? BigDecimal.ZERO.add(sumVals[cnt]) : subTotalSum[cnt].add(  sumVals[cnt]); // 5.6.8.1 (2013/09/13)
				totalSum[cnt]      = totalSum[cnt]      == null ? BigDecimal.ZERO.add(sumVals[cnt]) : totalSum[cnt].add(     sumVals[cnt]);
				grandTotalSum[cnt] = grandTotalSum[cnt] == null ? BigDecimal.ZERO.add(sumVals[cnt]) : grandTotalSum[cnt].add(sumVals[cnt]);
			}

			lastSubTotalKey = subTotalKey.toString();
			lastTotalKey = totalKey.toString();

			// グループ集計時はグルーピングした行数を加算する。
			int gcnt = 1;
			if( config.useGroup() && rowCountColumn >= 0 && rowCount > 0 ) {
				gcnt = rowCount;
			}
			subTotalCount += gcnt;
			totalCount    += gcnt;
			grandTotalCount += gcnt;

			tblIdx++;
		}

		// 最大件数が、超えた場合でかつ次のデータがある場合は、オーバーフロー
		boolean isOverFlow = tblIdx < getRowCount() ;

		// 小計キー最終行処理
		if( config.useSubTotal() && lastSubTotalKey != null ) {
			if( numberOfRows < maxRowCount ) {
				addRow( subTotalFilter, subTotalLabel, subTotalCount, sumFilter, subTotalSum, tblIdx );
				numberOfRows++;
				tblIdx++;
			}
			else {
				isOverFlow = true;
			}
		}

		// 合計キー最終行処理
		if( config.useTotal() && lastTotalKey != null ) {
			if( numberOfRows < maxRowCount ) {
				addRow( totalFilter, totalLabel, totalCount, sumFilter, totalSum, tblIdx );
				numberOfRows++;
				tblIdx++;
			}
			else {
				isOverFlow = true;
			}
		}

		// 総合計処理
		if( config.useGrandTotal() && numberOfRows > 0 ) {
			if( numberOfRows < maxRowCount ) {
				final boolean[] grandTotalFilter = new boolean[numberOfColumns];
				// 総合計のラベル表示廃止
				// grandTotalFilter[0] = true;

				if( config.useFirstTotal() ) {		// 6.1.1.0 (2015/01/17)
					addRow( grandTotalFilter, grandTotalLabel, grandTotalCount, sumFilter, grandTotalSum, 0 );
				}
				else {
					addRow( grandTotalFilter, grandTotalLabel, grandTotalCount, sumFilter, grandTotalSum, tblIdx );
					tblIdx++;
				}
				numberOfRows++;
			}
			else {
				isOverFlow = true;
			}
		}

		if( isOverFlow ) {
			setOverflow( true );
		}

		return isOverFlow;
	}

	/**
	 * キーの値配列、集計値の配列を引数として、追加行を生成し、DBTableModelに追加します。
	 * キー、及び集計値がDBTableModel上のどのカラムに位置するかは、キーフィルタ、集計フィルタで指定します。
	 * 
	 * @og.rev 5.6.1.0 (2013/02/01) doubleをdecimalに
	 *
	 * @param keyFilter キーフィルタ
	 * @param keyVals キーの値配列
	 * @param keyCount 集計した行のカウント
	 * @param sumFilter 集計フィルタ
	 * @param sumVals 集計値配列
	 * @param aRow 挿入する行番号
	 */
	private void addRow( final boolean[] keyFilter, final String[] keyVals, final int keyCount
			, final boolean[] sumFilter, final BigDecimal[] sumVals, final int aRow ) {
		String[] columnValues = new String[numberOfColumns];
		int sc = 0;
		int kc = 0;
		for( int column=0; column<numberOfColumns; column++ ) {
			String val = "";
			if( keyFilter[column] ) {
				val = keyVals[kc++];
			}
			if( sumFilter[column] ) {
				val = FORMAT.format( sumVals[sc++] );
			}
			if( column == rowCountColumn ) {
				val = String.valueOf( keyCount );
			}
			columnValues[column] = val;
		}

		if( aRow < 0 ) {
			addColumnValues( columnValues );
		}
		else {
			addValues( columnValues, aRow, false );
		}
	}

	/**
	 * キーの値配列、集計値の配列を引数として、追加行を生成し、DBTableModelに追加します。
	 * キー、及び集計値がDBTableModel上のどのカラムに位置するかは、キーフィルタ、集計フィルタで指定します。
	 * 
	 * @og.rev 5.6.1.0 (2013/02/01) doubleをbigDecimal
	 *
	 * @param keyFilter キーフィルタ
	 * @param keyVals キーの値配列
	 * @param keyCount 集計した行のカウント
	 * @param sumFilter 集計フィルタ
	 * @param sumVals 集計値配列
	 */
	private void addRow( final boolean[] keyFilter, final String[] keyVals, final int keyCount
			, final boolean[] sumFilter, final BigDecimal[] sumVals ) {
		addRow( keyFilter, keyVals, keyCount, sumFilter, sumVals, -1 );
	}

	/**
	 * キーの値、集計値の配列を引数として、追加行を生成し、DBTableModelに追加します。
	 * キー、及び集計値がDBTableModel上のどのカラムに位置するかは、キーフィルタ、集計フィルタで指定します。
	 * 
	 * @og.rev 5.6.1.0 (2013/02/01) doubleをbigDecimalに
	 *
	 * @param keyFilter キーフィルタ
	 * @param keyVal キーの値
	 * @param keyCount 集計した行のカウント
	 * @param sumFilter 集計フィルタ
	 * @param sumVals 集計値配列
	 * @param aRow 挿入する行番号
	 */
	private void addRow( final boolean[] keyFilter, final String keyVal, final int keyCount
			, final boolean[] sumFilter, final BigDecimal[] sumVals, final int aRow ) {
		final List<String> tmp = new ArrayList<String>();
		for( int column=0; column<numberOfColumns; column++ ) {
			if( keyFilter[column] ) {
				tmp.add( keyVal );
			}
		}
		addRow( keyFilter, tmp.toArray( new String[tmp.size()] ), keyCount, sumFilter, sumVals, aRow );
	}
}
