/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import static org.opengion.fukurou.util.StringUtil.nval;

import java.io.File;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.util.Locale;
import java.util.Map;
import java.util.Arrays;

import org.opengion.fukurou.business.ArrayTableModel;
import org.opengion.fukurou.business.BizLogicHelper;
import org.opengion.fukurou.db.Transaction;
import org.opengion.fukurou.db.TransactionReal;
import org.opengion.fukurou.util.ErrMsg;
import org.opengion.fukurou.util.ErrorMessage;
import org.opengion.fukurou.util.HybsLoader;
import org.opengion.fukurou.util.HybsLoaderConfig;
import org.opengion.fukurou.util.HybsLoaderFactory;
import org.opengion.fukurou.util.StringUtil;
import org.opengion.fukurou.util.ToString;						// 6.1.1.0 (2015/01/17)
import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.db.DBTableModel;

/**
 * 業務ロジックを呼び出すためのタグです。
 *
 * logics属性に呼び出す業務ロジックのクラス名を記述します。
 * このタグでは、複数の業務ロジックを1度に呼び出すことができ、
 * DB接続のcommit,rollbackは一番最後に、1度のみ実行されます。
 * 各業務ロジックは、記述した順番に呼び出されます。
 *
 * 業務ロジックは、{@link org.opengion.fukurou.business.BizLogicHelper}の
 * 実装クラス、または、この実装クラスを継承したサブクラスである必要があります。
 *
 * 業務ロジッククラスについては、ホットデプロイ機能により、動的コンパイル、クラスロードが
 * 行われます。
 *
 * 業務ロジックのソースディレクトリは、システムリソースの BIZLOGIC_SRC_PATH で定義されます。
 * また、同様にクラスディレクトリは、システムリソースの BIZLOGIC_CLASS_PATH で定義されます。
 * さらに、システムリソースの BIZLOGIC_HOTDEPLOY を false に設定することで、動的コンパイル
 * 、クラスロードを行わなくすることもできます。
 * この場合、予めコンパイルされたクラスを、初回呼び出し時に1回のみロードされます。
 *
 * SystemData の USE_SQL_INJECTION_CHECK が true か、quotCheck 属性が true の場合は、
 * ＳＱＬインジェクション対策用のクォーティションチェックを行います。リクエスト引数に
 * クォーティション(')が含まれると、エラーになります。
 * 同様にUSE_XSS_CHECKがtrueか、xssCheck属性がtrueの場合は、
 * クロスサイトススクリプティング(XSS)対策のためless/greater than signのチェックを行います。
 *
 * ※ このタグは、Transaction タグの対象です。
 *
 * @og.formSample
 * ●形式：
 *       ・&lt;og:bizLog
 *             logics       = "業務ロジックのクラス名"
 *             command      = "ENTRY"
 *             scope        = "session"
 *             dbid         = "DEFAULT"
 *             tableId      = "DEFAULT"
 *             selectedAll  = "false"
 *             modifyType   = "A"
 *             keys         = "SYSTEM_ID"
 *             vals         = "{&#064;SYSTEM_ID}"
 *             stopError    = "true"
 *             quotCheck    = "true"
 *             xssCheck     = "true"
 *             debug        = "false"
 *        /&gt;
 * ●body：なし
 *
 * ●Tag定義：
 *   &lt;og:bizLogic
 *       logics           ○【TAG】実行する業務ロジック名を指定します(必須)。
 *       command            【TAG】コマンドをセットします(初期値:ENTRY)
 *       scope              【TAG】キャッシュする場合のスコープ[request/page/session/applicaton]を指定します(初期値:session)
 *       dbid               【TAG】(通常は使いません)Queryオブジェクトを作成する時のDB接続IDを指定します(初期値:null)
 *       tableId            【TAG】(通常は使いません)結果のDBTableModelを、sessionに登録するときのキーを指定します
 *       selectedAll        【TAG】データを全件選択済みとして処理するかどうか[true/false]を指定します(初期値:false)
 *       modifyType         【TAG】DB検索時の モディファイタイプを指定します[A:追加/C:更新/D:削除]
 *       keys               【TAG】リンク先に渡すキーをCSV形式で複数指定します
 *       vals               【TAG】リンク先に渡す値をCSV形式で複数指定します
 *       stopError          【TAG】処理エラーの時に処理を中止するかどうか[true/false]を設定します(初期値:true)
 *       quotCheck          【TAG】リクエスト情報の クォーティション(') 存在チェックを実施するかどうか[true/false]を設定します (初期値:USE_SQL_INJECTION_CHECK[=true])
 *       xssCheck           【TAG】リクエスト情報の HTMLTag開始/終了文字(&gt;&lt;) 存在チェックを実施するかどうか[true/false]を設定します (初期値:USE_XSS_CHECK[=true])
 *       multi              【TAG】vals属性でパラメーターを取得する際、複数件存在する場合に、値を連結するかどうかを指定します(初期値:false)
 *       debug              【TAG】デバッグ情報を出力するかどうか[true/false]を指定します(初期値:false)
 *   /&gt;
 *
 * ●使用例
 *     &lt;!-- 業務ロジックの呼び出しを行います --&gt;
 *     &lt;og:bizLogic logics="org.opengion.logic.gf9110.BizLogic_0001" keys="SYSTEM_ID" vals="{&#064;MEM.SYSTEM_ID}" /&gt;
 *
 * @og.rev 5.1.1.0 (2009/12/01) 新規作成
 * @og.group 業務ロジック
 *
 * @version 5.0
 * @author Hiroki Nakamura
 * @since JDK1.6,
 */
public class BizLogicTag extends CommonTagSupport {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "567020130727" ;
	private static final long serialVersionUID = 567020130727L ;

	/** command 引数に渡す事の出来る コマンド  エントリー {@value} */
	public static final String CMD_ENTRY		= "ENTRY" ;		// 5.1.9.0 (2010/08/01)
	private static final String ERR_MSG_ID	 	= HybsSystem.ERR_MSG_KEY;

	private 			String		command		= CMD_ENTRY;
	private				String[]	logics		;
	private   			String		dbid		;
	private transient	DBTableModel table		;
	private 			String		tableId		= HybsSystem.TBL_MDL_KEY;
	private 			boolean		selectedAll	;
	private				String		modifyType	;
	private				String[]	keys		;
	private				String[]	vals		;

	private				boolean		stopError	= true;
	private				boolean		quotCheck	= HybsSystem.sysBool( "USE_SQL_INJECTION_CHECK" );	// 4.0.0 (2005/08/31)
	private				boolean		xssCheck	= HybsSystem.sysBool( "USE_XSS_CHECK" );	// 5.0.0.2 (2009/09/15)

	private transient	ErrorMessage errMessage ;
	private				int		errCode	 		= ErrorMessage.OK;
	private				int		executeCount	= -1;
	private				ArrayTableModel	arrTable;
	private				HybsLoader		loader	;

	private				String	srcDir			= HybsSystem.sys( "REAL_PATH" ) + HybsSystem.sys( "BIZLOGIC_SRC_PATH" );
	private				String	classDir		= HybsSystem.sys( "REAL_PATH" ) + HybsSystem.sys( "BIZLOGIC_CLASS_PATH" );
	private				boolean	isAutoCompile	= HybsSystem.sysBool( "BIZLOGIC_AUTO_COMPILE" );
	private				boolean	isHotDeploy		= HybsSystem.sysBool( "BIZLOGIC_HOT_DEPLOY" );
	private				boolean	isMulti			;			// 5.1.8.0 (2010/07/01) 追加

	private	static final String	CLASS_PATH;

	// HotDeploy機能を使用する場合に、Javaクラスをコンパイルするためのクラスパスを設定します。
	// 対象となるクラスパスは、WEB-INF/classes 及び WEB-INF/lib/*.jar です。

	static {
		final StringBuilder sb = new StringBuilder( BUFFER_MIDDLE )
			.append( '.' ).append( File.pathSeparatorChar );
		final File lib = new File( HybsSystem.sys( "REAL_PATH" ) + "WEB-INF" + File.separator + "lib" );
		final File[] libFiles = lib.listFiles();
		for( int i=0; i<libFiles.length; i++ ) {
			// 5.1.1.2 (2009/12/10) File.pathSeparatorCharを使用
			// 5.1.8.0 (2010/07/01) libの検索パスの不具合対応
			sb.append( libFiles[i].getAbsolutePath() ).append( File.pathSeparatorChar );
		}
		sb.append( HybsSystem.sys( "REAL_PATH" ) + "WEB-INF" + File.separator + "classes" )
			.append( File.pathSeparatorChar )
			// 5.1.8.0 (2010/07/01) bizの下も検索パスに追加
			.append( HybsSystem.sys( "REAL_PATH" ) + HybsSystem.sys( "BIZLOGIC_CLASS_PATH" ) ).append( File.pathSeparatorChar );

		CLASS_PATH = sb.toString();
	}

	/**
	 * Taglibの開始タグが見つかったときに処理する doStartTag() を オーバーライドします。
	 *
	 * @og.rev 5.3.4.0 (2011/04/01) command=ENTRY以外ではDBTableModelの処理を行わない。
	 *
	 * @return	後続処理の指示(SKIP_BODY)
	 */
	@Override
	public int doStartTag() {
		// 5.3.4.0 (2011/04/01)
		if( CMD_ENTRY.equals( command ) ) {
			startQueryTransaction( tableId );
		}

		return SKIP_BODY ;
	}

	/**
	 * Taglibの終了タグが見つかったときに処理する doEndTag() を オーバーライドします。
	 *
	 * @og.rev 5.1.8.0 (2010/07/01) isMulti対応
	 * @og.rev 5.3.4.0 (2011/04/01) command=ENTRY以外ではDBTableModelの処理を行わない。
	 *
	 * @return	後続処理の指示
	 */
	@Override
	public int doEndTag() {
		debugPrint();

		// 5.3.4.0 (2011/04/01)
		useQuotCheck( quotCheck );
		useXssCheck( xssCheck );

		makeVals();
		execute();

		final String err = TaglibUtil.makeHTMLErrorTable( errMessage,getResource() );
		if( err != null && err.length() > 0 ) {
			setSessionAttribute( ERR_MSG_ID,errMessage );
		}

		if( table != null && ! commitTableObject( tableId, table ) ) {
				jspPrint( "BizLoicTag 処理が割り込まれました。DBTableModel は登録しません。" );
				return SKIP_PAGE;
		}

		jspPrint( err );

		if( errCode >= ErrorMessage.NG && stopError )  {
			return SKIP_PAGE;
		}
		else {
			return EVAL_PAGE;
		}
	}

	/**
	 * タグリブオブジェクトをリリースします。
	 * キャッシュされて再利用されるので、フィールドの初期設定を行います。
	 *
	 * @og.rev 5.1.8.0 (2010/07/01) isMultiを追加
	 * @og.rev 5.1.9.0 (2010/08/01) Transaction 対応
	 */
	@Override
	protected void release2() {
		super.release2();
		command			= CMD_ENTRY;
		logics			= null;
		dbid			= null;
		table 			= null;
		tableId			= HybsSystem.TBL_MDL_KEY;
		selectedAll		= false;
		modifyType      = null;
		keys 			= null;
		vals 			= null;
		stopError		= true;
		quotCheck		= HybsSystem.sysBool( "USE_SQL_INJECTION_CHECK" );
		xssCheck		= HybsSystem.sysBool( "USE_XSS_CHECK" );
		errMessage		= null;
		errCode			= ErrorMessage.OK;
		executeCount	= -1;
		arrTable		= null;
		loader			= null;
		srcDir			= HybsSystem.sys( "REAL_PATH" ) + HybsSystem.sys( "BIZLOGIC_SRC_PATH" );
		classDir		= HybsSystem.sys( "REAL_PATH" ) + HybsSystem.sys( "BIZLOGIC_CLASS_PATH" );
		isAutoCompile	= HybsSystem.sysBool( "BIZLOGIC_AUTO_COMPILE" );
		isHotDeploy		= HybsSystem.sysBool( "BIZLOGIC_HOT_DEPLOY" );
		isMulti			= false;	// 5.1.8.0 (2010/07/01) 追加
	}

	/**
	 * 業務ロジックを実行します。
	 *
	 * @og.rev 5.1.9.0 (2010/08/01) Transaction 対応
	 * @og.rev 5.3.4.0 (2011/04/01) command=ENTRY以外ではDBTableModelの処理を行わない。
	 * @og.rev 5.3.7.0 (2011/07/01) TransactionReal の引数変更 、Transaction対応で、close処理を入れる。
	 * @og.rev 5.6.0.3 (2012/01/24) arrTable に変更された値を、table に書き戻す処理を追加
	 */
	private void execute() {
		int[] rowNos = new int[0];

		// 5.3.4.0 (2011/04/01)
		if( CMD_ENTRY.equals( command ) ) {
			table = (DBTableModel)getObject( tableId );
		}

		if( table != null ) {
 			rowNos = getParameterRows();
			String[][] tblVals = new String[rowNos.length][table.getColumnCount()];
			String[] modTypes = new String[rowNos.length];
			for( int i=0; i<rowNos.length; i++ ) {
				tblVals[i] = table.getValues( rowNos[i] );
				modTypes[i] = table.getModifyType( rowNos[i] );
			}
			arrTable = new ArrayTableModel( table.getNames(), tblVals, modTypes );
		}

		// 5.1.9.0 (2010/08/01) Transaction 対応
		Transaction tran = null;
		// 5.3.7.0 (2011/07/01) Transaction対応で、close処理を入れる。
		try {
			final TransactionTag tranTag = (TransactionTag)findAncestorWithClass( this,TransactionTag.class );
			if( tranTag == null ) {
	//			tran = new TransactionReal( dbid,getApplicationInfo() );
				tran = new TransactionReal( getApplicationInfo() );		// 5.3.7.0 (2011/07/01) 引数変更
			}
			else {
				tran = tranTag.getTransaction();
			}
	//		conn = ConnectionFactory.connection( dbid, null );
			errMessage = new ErrorMessage();
			loader = HybsLoaderFactory.getLoader(
						new HybsLoaderConfig( srcDir, classDir, isAutoCompile, isHotDeploy, CLASS_PATH )
						);

			boolean rtn = false;
			for( int i=0; i<logics.length; i++ ) {
	//			BizLogic logic = (BizLogic)loader.newInstance( logics[i] );
	//			rtn = call( logic );
	//			rtn = call( logics[i] );
				rtn = call( logics[i] , tran );		// 5.1.9.0 (2010/08/01) Transaction 対応
				if( !rtn ) { break; }
			}

			// 5.6.0.3 (2012/01/24) arrTable に変更された値を、table に書き戻す処理
			if( arrTable != null ) {
				final Map<Integer,String[]> valMap = arrTable.getModifyVals();
				if( valMap != null ) {
					for( final Map.Entry<Integer, String[]> entr : valMap.entrySet()) {
						final int seq = entr.getKey();					// intValue() は省略できる。
						final String[] vals = entr.getValue();
						table.setValues( vals , rowNos[seq] );		// DBTableModel へ書き戻し。元の行番号に変換が必要。
					}
				}
			}

			executeCount = rowNos.length;
			errCode = errMessage.getKekka();
			setRequestAttribute( "DB.COUNT"   , String.valueOf( executeCount ) );
			setRequestAttribute( "DB.ERR_CODE", String.valueOf( errCode ) );

			if( errCode < ErrorMessage.NG ) {
	//			Closer.commit( conn );
				tran.commit();				// 5.1.9.0 (2010/08/01) Transaction 対応

				if( table != null && rowNos.length > 0 ) {
					for( int j=rowNos.length-1; j>=0; j-- ) {
						final int row = rowNos[j];
						if( DBTableModel.DELETE_TYPE.equals( table.getModifyType( row ) ) ) {
							table.removeValue( row );
						}
						else {
							table.resetModify( row );
						}
					}
				}
			}
			else {
	//			Closer.rollback( conn );
				tran.rollback();				// 5.1.9.0 (2010/08/01) Transaction 対応
			}
	//		ConnectionFactory.close( conn, dbid );
		}
		finally {
			if( tran != null ) { tran.close(); }
		}

		// エラーメッセージの行番号を元の選択行に戻します。
		final ErrMsg[] errs = errMessage.toArray();
		final ErrorMessage errMsgTmp = new ErrorMessage();
		for( int i=0; i<errs.length; i++ ) {
			if( table != null && rowNos.length > 0 ) {
				errMsgTmp.addMessage( errs[i].copy( rowNos[errs[i].getNo()] + 1 ) );
			}
			else {
				errMsgTmp.addMessage( errs[i].copy( errs[i].getNo() + 1 ) );
			}
		}
		errMessage = errMsgTmp;
	}

	/**
	 * 業務ロジックをCallします。
	 *
	 * @og.rev 5.1.9.0 (2010/08/01) Transaction 対応
	 * @og.rev 5.1.9.0 (2010/08/01) DBIDをセット、ConnectonではなくTransactionを渡すように変更
	 * @og.rev 5.6.7.0 (2013/07/27) Exception を throw するとき、一旦、errMsg 変数にセットします。
	 *
	 * @param	logicName	業務ロジック名
	 * @param	tran		ランザクションオブジェクト
	 *
	 * @return 業務ロジックの呼び出しが成功したかどうか
	 */
	private boolean call( final String logicName , final Transaction tran ) {
		final BizLogicHelper logicHp = new BizLogicHelper( logicName, loader );

		if( logicHp.isRequireTable() ) {
			if( arrTable == null ) {
				// 5.6.7.0 (2013/07/27) Exception を throw するとき、一旦、errMsg 変数にセットします。
				final String errMsg = "TableModelが存在しません。logic=[" + logicName + "]" ;
				throw new HybsSystemException( errMsg );
			}
			else if( arrTable.getRowCount() == 0 ) { return true; } // 0件のときは呼び出ししない
			else { logicHp.setTable( arrTable ); }
		}
		logicHp.setTransaction( tran );
		logicHp.setDbid( dbid );		// 5.1.9.0 (2010/08/01) DBIDをセット
		logicHp.setKeys( keys );
		logicHp.setVals( vals );
		logicHp.setUserId( getUser().getUserID() );
		logicHp.setParentPgId( getGUIInfoAttri( "KEY" ) );
		logicHp.setLoader( loader );
		if( isDebug() ) { logicHp.setDebug(); }

		boolean rtn = false;
		try {
			rtn = logicHp.exec();
		}
		catch ( Throwable th ) {
			tran.rollback();		// 5.1.9.0 (2010/08/01) Transaction 対応
			// 5.6.7.0 (2013/07/27) Exception を throw するとき、一旦、errMsg 変数にセットします。
			final String errMsg = "業務ロジックの処理中にエラーが発生しました。" + th.getMessage() ;
			throw new HybsSystemException( errMsg,th );
		}
		errMessage.append( logicHp.getErrMsg() );

		setRequestAttribute( logicHp.getClass().getSimpleName() + ".RETURN"   , logicHp.getReturn() );
		setRequestAttribute( "RETURN", logicHp.getReturn() );

		if( isDebug() ) { jspPrint( logicHp.getDebugMsg() ); }

		return rtn;
	}

	/**
	 * Valsの配列を生成します。
	 *
	 * @og.rev 5.1.8.0 (2010/07/01) 新規作成-
	 * @og.rev 5.6.7.0 (2013/07/27) Exception を throw するとき、一旦、errMsg 変数にセットします。
	 */
	private void makeVals() {
		if( keys != null && keys.length > 0 ) {
			final boolean isSetVal = vals != null && vals.length > 0 ;
			if( isSetVal ) {
				if( keys.length != vals.length ) {
					// 5.6.7.0 (2013/07/27) Exception を throw するとき、一旦、errMsg 変数にセットします。
					final String errMsg = "keysとValsの個数が異なります。"  + CR
								+ "  keys=" + Arrays.toString( keys ) + CR
								+ "  vals=" + Arrays.toString( vals ) ;
					throw new HybsSystemException( errMsg );
				}
			}
			else {
				vals = new String[keys.length];
			}
			for( int i=0; i<keys.length; i++ ) {
				if( isSetVal ) {
					if( isMulti )	{ vals[i] = StringUtil.array2csv( getRequestParameterValues( vals[i] ) );}
					else			{ vals[i] = getRequestParameter( vals[i] ); }
				}
				else {
					if( isMulti )	{ vals[i] = StringUtil.array2csv( getRequestValues( keys[i] ) );}
					else			{ vals[i] = getRequestValue( keys[i] ); }
				}
			}
		}
	}

	/**
	 * 【TAG】コマンドをセットします(初期値:ENTRY)。
	 *
	 * @og.tag
	 * command=ENTRY以外ではDBTableModelの処理を行いません。
	 * コマンドは,HTMLから[get/post]指定されますので,CMD_xxx で設定される
	 * フィールド定数値のいづれかを、指定できます。
	 *
	 * @param	cmd コマンド (public static final 宣言されている文字列)
	 * @see		<a href="../../../../constant-values.html#org.opengion.hayabusa.taglib.BizLogicTag.CMD_ENTRY">コマンド定数</a>
	 */
	public void setCommand( final String cmd ) {
		final String cmd2 = getRequestParameter( cmd );
		if( cmd2 != null && cmd2.length() >= 0 ) { command = cmd2.toUpperCase(Locale.JAPAN); }
	}

	/**
	 * 【TAG】実行する業務ロジック名を指定します。
	 *
	 * @og.tag
	 * 実行する業務ロジック名を指定します。業務ロジック名は、クラス名を指定します。
	 * クラス名については、クラス自身の名称のみを指定することができます。
	 * (パッケージ名を含めた完全な形のクラス名を指定することもできます)
	 * また、CSV形式で、複数指定することもできます。
	 * この場合、指定した順番に処理されます。
	 *
	 * @param	lgs 業務ロジック名
	 */
	public void setLogics( final String lgs ) {
		logics = getCSVParameter( lgs );
	}

	/**
	 * 【TAG】(通常は使いません)Queryオブジェクトを作成する時のDB接続IDを指定します(初期値:null)。
	 *
	 * @og.tag
	 * Queryオブジェクトを作成する時のDB接続IDを指定します。
	 * これは、システムリソースで、DEFAULT_DB_URL 等で指定している データベース接続先
	 * 情報に、XX_DB_URL を定義することで、 dbid="XX" とすると、この 接続先を使用して
	 * データベースにアクセスできます。
	 *
	 * @param	id データベース接続ID
	 */
	public void setDbid( final String id ) {
		dbid = nval( getRequestParameter( id ),dbid );
	}

	/**
	 * 【TAG】(通常は使いません)結果のDBTableModelを、sessionに登録するときのキーを指定します
	 *		(初期値:HybsSystem#TBL_MDL_KEY[={@og.value org.opengion.hayabusa.common.HybsSystem#TBL_MDL_KEY}])。
	 *
	 * @og.tag
	 * 検索結果より、DBTableModelオブジェクトを作成します。これを、下流のviewタグ等に
	 * 渡す場合に、通常は、session を利用します。その場合の登録キーです。
	 * query タグを同時に実行して、結果を求める場合、同一メモリに配置される為、
	 * この tableId 属性を利用して、メモリ空間を分けます。
	 *		(初期値:HybsSystem#TBL_MDL_KEY[={@og.value org.opengion.hayabusa.common.HybsSystem#TBL_MDL_KEY}])。
	 *
	 * @param	id テーブルID (sessionに登録する時のID)
	 */
	public void setTableId( final String id ) {
		tableId   = nval( getRequestParameter( id ),tableId );
	}

	/**
	 * 【TAG】データを全件選択済みとして処理するかどうか[true/false]を指定します(初期値:false)。
	 *
	 * @og.tag
	 * 全てのデータを選択済みデータとして扱って処理します。
	 * 全件処理する場合に、(true/false)を指定します。
	 * 初期値は false です。
	 *
	 * @param  all データを全件選択済み [true:全件選択済み/false:通常]
	 */
	public void setSelectedAll( final String all ) {
		selectedAll = nval( getRequestParameter( all ),selectedAll );
	}

	/**
	 * 【TAG】DB検索時の モディファイタイプを指定します[A:追加/C:更新/D:削除]。
	 *
	 * @og.tag
	 * DB検索時に、そのデータをA(追加)、C(更新)、D(削除)のモディファイタイプを
	 * つけた状態にします。
	 * その状態で、そのまま、update する事が可能になります。
	 *
	 * @param   type モディファイタイプ [A:追加/C:更新/D:削除]
	 */
	public void setModifyType( final String type ) {
		modifyType = nval( getRequestParameter( type ),modifyType );
	}

	/**
	 * 【TAG】リンク先に渡すキーをCSV形式で複数指定します。
	 *
	 * @og.tag リンク先に渡すキーを指定します。
	 * Keysだけを指定して、Valsを指定しない場合、Keysで指定された項目名に対応するパラメーターを取得し、
	 * Valsとして使用します。
	 *
	 * @og.rev 3.5.6.2 (2004/07/05) CommonTagSupport#getCSVParameter を使用
	 *
	 * @param	key リンク先に渡すキー(CSV形式)
	 */
	public void setKeys( final String key ) {
		keys = getCSVParameter( getRequestParameter( key ) );
	}

	/**
	 * 【TAG】リンク先に渡す値をCSV形式で複数指定します。
	 *
	 * @og.tag リンク先に渡す値を指定します。
	 * Keysだけを指定して、Valsを指定しない場合、Keysで指定された項目名に対応するパラメーターを取得し、
	 * Valsとして使用します。
	 * 分解方法は、CSV変数を先に分解してから、getRequestParameter で値を取得します。
	 * こうしないとデータ自身にカンマを持っている場合に分解をミスる為です。
	 *
	 * @og.rev 3.5.6.2 (2004/07/05) CommonTagSupport#getCSVParameter を使用
	 * @og.rev 5.1.8.0 (2010/07/01) isMuitl対応
	 *
	 * @param	val リンク先に渡す値(CSV形式)
	 */
	public void setVals( final String val ) {
		vals = StringUtil.csv2Array( val );
	}

	/**
	 * 【TAG】処理エラーの時に処理を中止するかどうか[true/false]を設定します(初期値:true)。
	 *
	 * @og.tag
	 * false(中止しない)に設定する場合、後続処理では、{&#064;DB.ERR_CODE}の値により、
	 * PLSQL/SQLの異常/正常終了によって分岐処理は可能となります。
	 * 初期値は、true(中止する)です。
	 *
	 * @param   flag  処理の中止 [true:中止する/false:中止しない]
	 */
	public void setStopError( final String flag ) {
		stopError = nval( getRequestParameter( flag ),stopError );
	}

	/**
	 * 【TAG】リクエスト情報の クォーティション(') 存在チェックを実施するかどうか[true/false]を設定します
	 *		(初期値:USE_SQL_INJECTION_CHECK[={@og.value org.opengion.hayabusa.common.SystemData#USE_SQL_INJECTION_CHECK}])。
	 *
	 * @og.tag
	 * ＳＱＬインジェクション対策の一つとして、暫定的ではありますが、SQLのパラメータに
	 * 渡す文字列にクォーティション(') を許さない設定にすれば、ある程度は防止できます。
	 * 数字タイプの引数には、 or 5=5 などのクォーティションを使用しないコードを埋めても、
	 * 数字チェックで検出可能です。文字タイプの場合は、必ず (')をはずして、
	 * ' or 'A' like 'A のような形式になる為、(')チェックだけでも有効です。
	 * (') が含まれていたエラーにする(true)／かノーチェックか(false)を指定します。
	 * (初期値:システム定数のUSE_SQL_INJECTION_CHECK[={@og.value org.opengion.hayabusa.common.SystemData#USE_SQL_INJECTION_CHECK}])。
	 *
	 * @param   flag クォーティションチェック [true:する/それ以外:しない]
	 * @see		org.opengion.hayabusa.common.SystemData#USE_SQL_INJECTION_CHECK
	 */
	public void setQuotCheck( final String flag ) {
		quotCheck = nval( getRequestParameter( flag ),quotCheck );
	}

	/**
	 * 【TAG】リクエスト情報の HTMLTag開始/終了文字(&gt;&lt;) 存在チェックを実施するかどうか[true/false]を設定します
	 *		(初期値:USE_XSS_CHECK[={@og.value org.opengion.hayabusa.common.SystemData#USE_XSS_CHECK}])。
	 *
	 * @og.tag
	 * クロスサイトスクリプティング(XSS)対策の一環としてless/greater than signについてのチェックを行います。
	 * (&gt;&lt;) が含まれていたエラーにする(true)／かノーチェックか(false)を指定します。
	 * (初期値:システム定数のUSE_XSS_CHECK[={@og.value org.opengion.hayabusa.common.SystemData#USE_XSS_CHECK}])。
	 *
	 * @param	flag	XSSチェックする [true:チェックする/false:しない]
	 * @see		org.opengion.hayabusa.common.SystemData#USE_XSS_CHECK
	 */
	public void setXssCheck( final String flag ) {
		xssCheck = nval( getRequestParameter( flag ),xssCheck );
	}

	/**
	 * 【TAG】vals属性でパラメーターを取得する際、複数件存在する場合に、値を連結するかどうかを指定します(初期値:false)。
	 *
	 * @og.tag
	 * この属性がtrueに指定された場合、パラメーターが複数存在する場合に、カンマで連結します。
	 * 初期値は、false(連結しない)です。
	 *
	 * @og.rev 5.1.8.0 (2010/07/01) 新規作成
	 *
	 * @param	flag	値連結 [true:する/false:しない]
	 */
	public void setMulti( final String flag ) {
		 isMulti = nval( getRequestParameter( flag ),isMulti );
	}

	/**
	 * 表示データの HybsSystem.ROW_SEL_KEY を元に、選ばれた 行を処理の対象とします。
	 *
	 * @return	選択行の配列
	 */
	@Override
	protected int[] getParameterRows() {
		final int[] rowNo ;
		if( selectedAll ) {
			final int rowCnt = table.getRowCount();
			rowNo = new int[ rowCnt ];
			for( int i=0; i<rowCnt; i++ ) {
				rowNo[i] = i;
			}
		} else {
			rowNo = super.getParameterRows();
		}
		return rowNo ;
	}

	/**
	 * シリアライズ用のカスタムシリアライズ書き込みメソッド。
	 *
	 * @serialData 一部のオブジェクト(DBTableModel,ErrorMessage)は、シリアライズされません。
	 *
	 * @param	strm	ObjectOutputStreamオブジェクト
	 * @throws IOException	入出力エラーが発生した場合
	 */
	private void writeObject( final ObjectOutputStream strm ) throws IOException {
		strm.defaultWriteObject();
	}

	/**
	 * シリアライズ用のカスタムシリアライズ読み込みメソッド
	 *
	 * ここでは、transient 宣言された内部変数の内、初期化が必要なフィールドのみ設定します。
	 *
	 * @serialData 一部のオブジェクト(DBTableModel,ErrorMessage)は、読み込まれません。
	 *
	 * @param	strm	ObjectInputStreamオブジェクト
	 * @see #release2()
	 * @throws IOException	シリアライズに関する入出力エラーが発生した場合
	 * @throws ClassNotFoundException	クラスを見つけることができなかった場合
	 */
	private void readObject( final ObjectInputStream strm ) throws IOException , ClassNotFoundException {
		strm.defaultReadObject();
	}

	/**
	 * このオブジェクトの文字列表現を返します。
	 * 基本的にデバッグ目的に使用します。
	 *
	 * @return このクラスの文字列表現
	 * @og.rtnNotNull
	 */
	@Override
	public String toString() {
		return ToString.title( this.getClass().getName() )
				.println( "VERSION"			,VERSION		)
				.println( "command"			,command		)
				.println( "logics"			,logics			)
				.println( "dbid"			,dbid			)
				.println( "tableId"			,tableId		)
				.println( "selectedAll"		,selectedAll	)
				.println( "modifyType"		,modifyType		)
				.println( "keys"			,keys			)
				.println( "vals"			,vals			)
				.println( "stopError"		,stopError		)
				.println( "quotCheck"		,quotCheck		)
				.println( "xssCheck"		,xssCheck		)
				.println( "executeCount"	,executeCount	)
				.println( "errCode"	 		,errCode	 	)
				.println( "Other..."		,getAttributes().getAttribute() )
				.fixForm().toString() ;
	}
}
