/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.db.DBColumn;
import org.opengion.hayabusa.db.DBEventColumn;
import org.opengion.hayabusa.db.DBLastSql;
import org.opengion.hayabusa.db.DBTableModel;
import org.opengion.hayabusa.resource.GUIInfo;
import org.opengion.hayabusa.resource.LabelInterface;
import org.opengion.hayabusa.resource.ResourceFactory;
import org.opengion.hayabusa.resource.ResourceManager;
import org.opengion.hayabusa.resource.UserInfo;

import org.opengion.fukurou.util.HybsConst ;		// 6.1.0.0 (2014/12/26)
import org.opengion.fukurou.db.DBFunctionName;
import org.opengion.fukurou.util.ApplicationInfo;
import org.opengion.fukurou.util.Attributes;
import org.opengion.fukurou.util.StringUtil;
import org.opengion.fukurou.util.HybsDateUtil;
import org.opengion.fukurou.util.ToString;						// 6.1.1.0 (2015/01/17)
import static org.opengion.fukurou.util.StringUtil.nval;

import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.Arrays;
import java.util.Collections;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Locale;
import java.util.Map;

import javax.servlet.ServletContext;
import javax.servlet.ServletRequest;
import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;
import javax.servlet.jsp.JspWriter;
import javax.servlet.jsp.tagext.BodyContent;
import javax.servlet.jsp.tagext.BodyTagSupport;
import javax.servlet.jsp.tagext.TryCatchFinally;

/**
 * TagSupport から継承されたサブクラスです。
 *
 * 汎用属性 のsetterメソッドと、Attributes オブジェクトを持っています。
 * それ以外に、{&#064;XXXX} 変数の対応と、lang属性のメソッドも用意しています。
 *
 * language 属性は、個々のリソースのロケールを指定できます。通常は、
 * ユーザー情報の lang 属性をデフォルトで使用し、セットされていない場合は、
 * リクエスト情報のロケールから取得します。
 *
 * 以下に、このメソッド内で定義される属性を記述します。
 *
 *       language           【TAG】タグ内部で使用する言語コード[ja/en/zh/…]を指定します
 *       lbl                【TAG】ラベルリソースのラベルIDを指定します
 *       debug              【TAG】デバッグ情報を出力するかどうか[true/false]を指定します(初期値:false)
 *       caseKey            【TAG】このタグ自体を利用するかどうかの条件キーを指定します(初期値:null)
 *       caseVal            【TAG】このタグ自体を利用するかどうかの条件値を指定します(初期値:null)
 *       caseNN             【TAG】指定の値が、null/ゼロ文字列 でない場合(Not Null=NN)は、このタグは使用されます(初期値:true)
 *       caseNull           【TAG】指定の値が、null/ゼロ文字列 の場合は、このタグは使用されます(初期値:true)
 *       scope              【TAG】キャッシュする場合のスコープ(request,session)を指定します(初期値:request)
 *
 * 各属性は、{&#064;XXXX} 変数が使用できます。
 * これは、ServletRequest から、XXXX をキーに値を取り出し,この変数に
 * 割り当てます。つまり、このXXXXをキーにリクエストすれば、
 * この変数に値をセットすることができます。
 *
 * http://localhost/query.jsp?KEY1=VLA1&amp;KEY2=VAL2
 *
 * のようなリクエストで、{&#064;KEY1} とすれば、 VAL1 がセットされます。
 *
 * このタグは、ラベル部分と入力フィールド部分がテーブルタグの&lt;td&gt;
 * により左右に分割されます。HTML 表示時は、前後に&lt;tr&gt;タグで囲って,
 * 整形できます。
 *
 * @og.group 画面制御
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
class CommonTagSupport extends BodyTagSupport implements TryCatchFinally {
	private static final long serialVersionUID = 574220140320L ;	// 5.7.4.2 (2014/03/20)

	/** システムの改行コードを設定します。*/
	protected static final String CR		 = HybsConst.CR;			// 6.1.0.0 (2014/12/26) refactoring
	/** HTMLの改行コード(&lt;br /&gt; + CR)を設定します。*/
	protected static final String BR		 = HybsConst.BR;			// 6.1.0.0 (2014/12/26) refactoring
	/** StringBilderなどの初期値を設定します。	{@value} */
	protected static final int BUFFER_MIDDLE = HybsConst.BUFFER_MIDDLE;	// 6.1.0.0 (2014/12/26) refactoring

	private transient Attributes		attri		= new Attributes();
	private transient ResourceManager	resource	;
	private transient UserInfo			userInfo	;
	private transient GUIInfo			guiInfo 	;
	private transient HttpSession		session 	;
	private transient ServletRequest	request 	;
	private transient Map<String,String[]>	requestCache;		// 3.5.6.2 (2004/07/05)
	private transient LabelInterface	msglbl		;			// 4.0.0 (2005/01/31)

	private String				language	;
	private boolean				debugFlag	;				// 3.5.5.3 (2004/04/09)
	private boolean				isReqNull	;
	private boolean				quotCheck	;				// 4.0.0 (2005/08/31)
	private String				scope		= "session";	// "request","page","session","applicaton"
	// 3.1.7.0 (2003/05/02) value値の使用可否を指定する、useValue 属性を追加。
	private Long				startTransaction;			// 3.6.0.8 (2004/11/19)
	private	int[]				rowNo		;				// 4.0.0 (2005/01/31)
	private boolean				xssCheck	= HybsSystem.sysBool( "USE_XSS_CHECK" ); // 5.0.0.2 (2009/09/15)
	private boolean				useTrans	;				// 5.1.6.0 (2010/05/01)

	private String				caseKey		;				// 5.2.2.0 (2010/11/01) 新規追加
	private String				caseVal		;				// 5.2.2.0 (2010/11/01) 新規追加
	private boolean				caseNN		= true;			// 5.6.7.0 (2013/07/27) 新規追加
	private boolean				caseNull	= true;			// 5.6.8.0 (2013/09/06) 新規追加

	private boolean				isSanitized	;				// 5.7.4.2 (2014/03/20) 新規追加

	/**
	 * Taglibの開始タグが見つかったときに処理する doStartTag() を オーバーライドします。
	 *
	 * @return	後続処理の指示(SKIP_BODY)
	 */
	@Override
	public int doStartTag() {
		return SKIP_BODY ;				// Body を評価しない
	//	return( EVAL_BODY_INCLUDE );	// Body インクルード( extends TagSupport 時)
	//	return EVAL_BODY_BUFFERED ;	// Body を評価する。( extends BodyTagSupport 時)
	}

	/**
	 * Taglibのタグ本体を処理する doAfterBody() を オーバーライドします。
	 *
	 * @return	後続処理の指示(SKIP_BODY)
	 */
	@Override
	public int doAfterBody() {
		return SKIP_BODY ;				// Body を評価しない
	//	return( EVAL_BODY_AGAIN );		// ボディーを再評価( extends TagSupport 時)
	//	return EVAL_BODY_BUFFERED ;	// ボディーを再評価( extends BodyTagSupport 時)
	}

	/**
	 * Taglibの終了タグが見つかったときに処理する doEndTag() を オーバーライドします。
	 *
	 * @og.rev 3.1.1.2 (2003/04/04) Tomcat4.1 対応。release2() を doEndTag()で呼ぶ。
	 *
	 * @return	後続処理の指示
	 */
	@Override
	public int doEndTag() {
		debugPrint();		// 4.0.0 (2005/02/28)

		return EVAL_PAGE ;		// ページの残りを評価する。( extends TagSupport 時)
	//	return(SKIP_PAGE);		// ページの残りの処理を行わない。
	}

	/**
	 * タグの処理中(セッターメソッドを除く)の例外を全て受け取ります。
	 *
	 * タグの中のボディ部の評価中、または Tag.doStartTag(), Tag.doEndTag(),
	 * IterationTag.doAfterBody(), BodyTag.doInitBody() のいずれもの
	 * メソッドの中で、Throwableが投げられたときに呼び出されます。
	 *
	 * このメソッドはセッターメソッドの中でThrowableが起きた場合は呼び出されません。
	 *
	 * @og.rev 3.5.0.0 (2003/09/17) TryCatchFinally インターフェースを適用。
	 *
	 * @param	th	このタグを通過してきたThrowableな例外
	 */
	@Override
	public void doCatch(final Throwable th) throws Throwable {
		throw th;
	}

	/**
	 * タグの処理毎の、doEndTag()の後で呼び出されます。
	 *
	 * Tag,IterationTag,BodyTagを実装した全てのクラスの doEndTag()の
	 * 後で呼び出されます。 このメソッドはタグのボディ部や Tag.doStartTag(),
	 * Tag.doEndTag(), IterationTag.doAfterBody() ,BodyTag.doInitBody()の
	 * すべてのメソッドで例外が発生した後でも呼び出されます。
	 *
	 * このメソッドはセッターメソッドの中でThrowableが起きた場合は呼び出されません。
	 *
	 * このメソッドからは例外を投げるべきではありません。
	 * このメソッドは呼び出し毎のデータの整合性をとることとリソース管理の
	 * 動作をさせることを意図しています。
	 *
	 * @og.rev 3.5.0.0 (2003/09/17) TryCatchFinally インターフェースを適用。
	 *
	 */
	@Override
	public void doFinally() {
		release2();
	}

	/**
	 * タグリブオブジェクトをリリースします。
	 * キャッシュされて再利用されるので、フィールドの初期設定を行います。
	 *
	 * @og.rev 2.0.0.4 (2002/09/27) カスタムタグの release() メソッドを、追加
	 * @og.rev 3.1.1.2 (2003/04/04) Tomcat4.1 対応。release2() を doEndTag()で呼ぶ。
	 * @og.rev 3.1.1.2 (2003/04/04) Tomcat4.1 対応。HybsRequestWrapper 廃止。直接 Mapでキャッシュする。
	 * @og.rev 3.1.3.0 (2003/04/10) エンコード情報の取得を廃止する。
	 * @og.rev 3.1.7.0 (2003/05/02) value値の使用可否を指定する、useValue 属性を追加。
	 * @og.rev 3.5.5.3 (2004/04/09) debugFlag を、String ではなく、boolean 型に変更
	 * @og.rev 3.6.0.8 (2004/11/19) startTransaction 属性を追加
	 * @og.rev 3.8.0.2 (2005/07/11) rightNow 属性を追加
	 * @og.rev 5.0.0.2 (2009/09/15) XSS対応
	 * @og.rev 5.1.6.0 (2010/05/01) DBLastSQL周りの実装見直し
	 * @og.rev 5.1.8.0 (2010/07/01) isNullSet 属性 廃止にともなう、useValue 属性廃止
	 * @og.rev 5.2.2.0 (2010/11/01) caseKey、caseVal 属性の追加
	 * @og.rev 5.3.2.0 (2011/02/01) paramNames 属性の追加
	 * @og.rev 5.6.7.0 (2013/07/27) caseNN 属性の追加
	 * @og.rev 5.6.8.0 (2013/09/06) caseNull 属性の追加
	 * @og.rev 5.7.4.1 (2014/03/14) rightNow 属性 廃止
	 * @og.rev 5.7.4.1 (2014/03/14) isSanitized 属性の追加
	 */
	protected void release2() {
		language	= null;
		attri		= new Attributes();
		resource	= null;
		debugFlag	= false;		// 3.5.5.3 (2004/04/09)
		userInfo	= null;
		guiInfo 	= null;
		session 	= null;
		request 	= null;
		isReqNull	= false;
		scope		= "session";	// "request","page","session","applicaton"
		requestCache = null;
		startTransaction = null;	// 3.6.0.8 (2004/11/19)
		rowNo		= null;			// 4.0.0 (2005/01/31)
		msglbl		= null;			// 4.0.0 (2005/01/31)
		quotCheck	= false;		// 4.0.0 (2005/08/31)
		xssCheck	= HybsSystem.sysBool( "USE_XSS_CHECK" ); // 5.0.0.2 (2009/09/15)
		useTrans	= false;		// 5.1.6.0 (2010/05/01)
		caseKey		= null;			// 5.2.2.0 (2010/11/01)
		caseVal		= null;			// 5.2.2.0 (2010/11/01)
		caseNN		= true;			// 5.6.7.0 (2013/07/27) 新規追加
		caseNull	= true;			// 5.6.8.0 (2013/09/06) 新規追加
		isSanitized	= false;		// 5.7.4.2 (2014/03/20) 新規追加。一応入れておくが都度、初期化しています。
	}

	/**
	 * 【TAG】タグ内部で使用する言語コード[ja/en/zh/…]を指定します。
	 *
	 * @og.tag
	 * この言語コードに基づいて、表示のラベルをリソースから作成します。
	 *
	 * @param   lang 言語コード [ja/en/zh/…]
	 * @see		#getLanguage()
	 */
	public void setLanguage( final String lang ) {
		language = getRequestParameter( lang );
	}

	/**
	 * 言語コード[ja/en/zh/…]を取得します。
	 *
	 * 言語コードが、セットされている場合は，設定値を優先します。
	 * 設定されていない場合は、ログインユーザーの持つLANG属性を、それが null の場合は、
	 * 実行環境のリクエストの言語を返します。
	 *
	 * @og.rev 2.1.1.0 (2002/11/08) セッション情報から取得できない場合に、クライアントの
	 *                      リクエスト情報のロケールから取得する処理を追加
	 * @og.rev 2.2.0.0 (2002/12/17) セッション情報から取得するのではなく、ユーザー情報より
	 *                      取得するように変更。そこにない場合は、リクエスト情報の
	 *                      ロケールから取得する
	 *
	 * @og.rev 6.0.2.5 (2014/10/31) 初期エラー発生時は、ユーザーも取得できないので、null でも返す。
	 *
	 * @return   言語コード[ja/en/zh/…]
	 * @see		#setLanguage( String )
	 */
	protected String getLanguage() {
		// 6.0.2.5 (2014/10/31) 初期エラー発生時は、ユーザーも取得できないので、null でも返す。
		try {
			if( language == null ) {
				language = getUser().getLang();
				if( language == null ) {
					language = getRequest().getLocale().getLanguage();
				}
			}
		}
		catch( RuntimeException ex ) {
			final String errMsg = "言語コードを取得できませんでした。"
						+ ex.getMessage() ;
			System.err.println( errMsg );
//			ex.printStackTrace();
		}

		return language;

//		if( language != null ) {
//			return language ;
//		}

//	final String errMsg = "言語コードがセットされていません。" ;
//		throw new HybsSystemException( errMsg );
	}

	/**
	 * 【TAG】ラベルリソースのラベルIDを指定します。
	 *
	 * @og.tag
	 * ラベルを変更するときに、lbl属性を使います。
	 *
	 * ラベルID は、所定の language に基づく ResourceManager の
	 * getLabelData( id ) を呼び出し、その結果のLabelInterfaceを使用します。
	 * getMsglbl() で取り出せます。
	 * ラベルとメッセージは統一されました。
	 *
	 * @og.rev 4.0.0.0 (2005/01/31) label 変数は、生データを保管するように変更。
	 *
	 * @param   lbl ラベルID
	 * @see		#getMsglbl()
	 */
	public void setLbl( final String lbl ) {
		msglbl = (LabelInterface)getResource().getLabelData( getRequestParameter( lbl ) ) ;		// 4.0.0 (2005/01/31)
	}

	/**
	 * 【TAG】デバッグ情報を出力するかどうか[true/false]を指定します(初期値:false)。
	 *
	 * @og.tag
	 * デバッグ情報を [true:出力する/false:しない]を指定します。
	 * 出力形式自体は、個々のタグによって異なります。
	 *
	 * @og.rev 3.5.5.3 (2004/04/09) debugFlag を、String ではなく、boolean 型に変更
	 *
	 * @param   flag  デバッグ [true:出力する/それ以外:しない]
	 */
	public void setDebug( final String flag ) {
		debugFlag = nval( getRequestParameter( flag ),debugFlag );
	}

	/**
	 * 【TAG】キャッシュする場合のスコープ[request/page/session/applicaton]を指定します(初期値:session)。
	 *
	 * @og.tag
	 * "request","page","session","applicaton" が指定できます。
	 * JSPのスコープは、「変数の有効範囲」を表すもので、フレームワーク上では、
	 * 主にテーブルモデルを管理するスコープを指します。
	 *
     * <table border="1" frame="box" rules="all" >
     *   <caption>テーブルモデルを管理するスコープ変数の有効範囲</caption>
     *   <tr><th>スコープ	</th><th>変数の有効範囲</th><th>説明</th></tr>
     *   <tr><td>page		</td><td>JSPページ内</td>
     *       <td>そのJSPページ内のみで有効です。フレームワーク的には、JSPページにまたがる処理が多いため、ほとんど使う機会はありません。</td></tr>
     *   <tr><td>request	</td><td>HTTPリクエスト</td>
     *       <td>リクエストの一連の処理期間中に有効な変数で、メモリに多くの情報を残したくない場合に利用します。検索系やポップアップのJSP画面等に利用します。</td></tr>
     *   <tr><td>session	</td><td>HTTPセッション</td>
     *       <td>初期設定されているスコープで、ログインユーザー単位にログアウトまで保持されます。
     *           内部的には、同じキーワード（tableId）で管理しているため、検索都度、破棄されます。
     *           （ガーベジコレクションにて破棄されるのを待ちます。）</td></tr>
     *   <tr><td>application</td><td>Webアプリケーション</td>
     *       <td>ユーザー間で共有する場合のスコープになります。JSP画面の開発では、まず使うことはありません。</td></tr>
     * </table>
     *
	 * @param  scp スコープ [request/page/session/applicaton]
	 * @see		#getScope()
	 */
	public void setScope( final String scp ) {
		scope = nval( getRequestParameter( scp ),scope );
	}

	/**
	 * キャッシュする場合のスコープ[request/page/session/applicaton]を返します。
	 *
	 * "request","page","session","applicaton" があります。
	 *
	 * @og.rev 3.5.5.8 (2004/05/20) 新規追加
	 *
	 * @return   スコープ[request/page/session/applicaton]
	 * @see		#setScope( String )
	 */
	public String getScope() {
		return scope ;
	}

	/**
	 * 【TAG】このタグ自体を利用するかどうかの条件キーを指定します(初期値:null)。
	 *
	 * @og.tag
	 * caseKey.matches( caseVal ) の値が、true の場合は、このタグは使用されます。
	 * false の場合は、このタグは使用されません。
	 *
	 * caseKey , caseVal ともに null の場合は、true です。どちらかが、非null の場合は、
	 * もう片方も 非null で、かつ、caseKey.matches( caseVal ) が成立する必要があります。
	 * この属性は、caseKey , caseVal , caseNN , caseNull とともに useTag() の判定で使用されます。
	 *
	 * @og.rev 5.2.2.0 (2010/11/01) 新規追加
	 * @og.rev 5.6.3.3 (2013/04/19) 条件変更のためのコメント修正
	 *
	 * @param	ckey 条件キー (何も指定されない場合は、使用すると判断)
	 * @see		#setCaseVal( String )
	 * @see		#useTag()
	 */
	public void setCaseKey( final String ckey ) {
		caseKey = nval( getRequestParameter( ckey ),caseKey );
	}

	/**
	 * 【TAG】このタグ自体を利用するかどうかの条件値を指定します(初期値:null)。
	 *
	 * @og.tag
	 * caseKey.matches( caseVal ) の値が、true の場合は、このタグは使用されます。
	 * false の場合は、このタグは使用されません。
	 *
	 * caseKey , caseVal ともに null の場合は、true です。どちらかが、非null の場合は、
	 * もう片方も 非null で、かつ、caseKey.matches( caseVal ) が成立する必要があります。
	 * この属性は、caseKey , caseVal , caseNN , caseNull とともに useTag() の判定で使用されます。
	 *
	 * @og.rev 5.2.2.0 (2010/11/01) 新規追加
	 * @og.rev 5.6.3.3 (2013/04/19) 条件変更のためのコメント修正
	 *
	 * @param	cval 条件値 (何も指定されない場合は、使用すると判断)
	 * @see		#setCaseKey( String )
	 * @see		#useTag()
	 */
	public void setCaseVal( final String cval ) {
		caseVal = nval( getRequestParameter( cval ),caseVal );
	}

	/**
	 * 【TAG】このタグ自体を利用するかどうかの条件として、NotNullかどうか判定します(初期値:true)。
	 *
	 * @og.tag
	 * この値が、null/ゼロ文字列 でない場合(Not Null=NN)は、このタグは使用されます。
	 * null/ゼロ文字列 の場合は、このタグは使用されません。
	 * 何も指定しない場合は、使用されます。
	 * caseNull と逆の動きをします。
	 * {&#064;XXXX} で、指定した場合は、値が設定されなければ、使用されません。
	 *
	 * この属性は、caseKey , caseVal , caseNN , caseNull とともに useTag() の判定で使用されます。
	 *
	 * @og.rev 5.6.7.0 (2013/07/27) 新規追加
	 *
	 * @param	cnn NotNull判定値 (何も指定されない場合は、使用すると判断)
	 * @see		#setCaseVal( String )
	 * @see		#useTag()
	 */
	public void setCaseNN( final String cnn ) {
		final String tempNN = nval( getRequestParameter( cnn ),null );

		caseNN = tempNN != null && !tempNN.isEmpty() ;
	}

	/**
	 * 【TAG】このタグ自体を利用するかどうかの条件として、Nullかどうか判定します(初期値:true)。
	 *
	 * @og.tag
	 * この値が、null/ゼロ文字列 の場合は、このタグは使用されます。
	 * null/ゼロ文字列 でない場合は、このタグは使用されません。
	 * 何も指定しない場合は、使用されます。
	 * caseNN と逆の動きをします。
	 * {&#064;XXXX} で、指定した場合は、値が設定されていなければ、使用されます。
	 *
	 * この属性は、caseKey , caseVal , caseNN , caseNull とともに useTag() の判定で使用されます。
	 *
	 * @og.rev 5.6.8.0 (2013/09/06) 新規追加
	 *
	 * @param	cnul Null判定値 (何も指定されない場合は、使用すると判断)
	 * @see		#setCaseVal( String )
	 * @see		#useTag()
	 */
	public void setCaseNull( final String cnul ) {
		final String tempNull = nval( getRequestParameter( cnul ),null );

		caseNull = tempNull == null || tempNull.isEmpty() ;
	}

	/**
	 * このタグ自体を利用するかどうかの条件判定を行います。
	 *
	 * caseNN &amp;&amp; caseNull &amp;&amp;
	 *    ( (caseKey == null &amp;&amp; caseVal == null) ||
	 *		(caseKey != null &amp;&amp; caseVal != null &amp;&amp; caseKey.matches( caseVal )))
	 * の結果を返します。
	 *
	 * これは、タグ本体に、条件式を登録できる機能です。必要なタグには、tld ファイルで、
	 * caseKey 、caseVal 、caseNN 、caseNull属性が使用できるように、設定します。
	 * 各タグを、equals タグで括る方法では、ソースの見通しが悪くなるため、
	 * ある程度タグ自身に判定機能を設けることで対応できるようにしました。
	 * ただ、本来、JSP 側にロジックを持ち込むのはよくないので、利用に関しては、
	 * 慎重にお願いします。
	 *
	 * @og.rev 5.2.2.0 (2010/11/01) 新規追加
	 * @og.rev 5.6.3.3 (2013/04/19) 条件変更(caseKey と caseVal が 両方とも null の場合のみ true)
	 * @og.rev 5.6.7.0 (2013/07/27) caseNN 属性追加。先のcaseKey、caseVal 条件と、AND 結合になります。
	 * @og.rev 5.6.8.0 (2013/09/06) caseNull 属性追加。先の条件と、AND 結合になります。
	 *
	 * @return	このタグ自体を利用するかどうか(true:利用する/false:利用しない)
	 * @see		#setCaseVal( String )
	 * @see		#setCaseKey( String )
	 * @see		#setCaseNN( String )
	 */
	protected boolean useTag() {
		return caseNN && caseNull &&
					( (caseKey == null && caseVal == null) ||
					  (caseKey != null && caseVal != null && caseKey.matches( caseVal ))) ;
	}

	/**
	 * (通常使いません)タグで処理される処理がメインとなるトランザクション処理かどうかを指定します(初期値:false)。
	 *
	 * この値は、ファイルダウンロード処理に影響します。この値がtrueに指定された時にcommitされたDBTableModelが
	 * ファイルダウンロードの対象の表になります。
	 *
	 * このパラメーターは、通常、各タグにより実装され、ユーザーが指定する必要はありません。
	 * 但し、1つのJSP内でDBTableModelが複数生成される場合に、前に処理したDBTableModelについてファイルダウンロードをさせたい
	 * 場合は、後ろでDBTableModelを生成するタグで、明示的にこの値をfalseに指定することで、ファイルダウンロード処理の対象から
	 * 除外することができます。
	 *
	 * @og.rev 5.1.6.0 (2010/05/01) 新規作成
	 *
	 * @param  flag メイントランザクション処理かどうか [true:メイン/false:その他]
	 */
	protected void useMainTrans( final boolean flag ) {
		useTrans = flag;
	}

	/**
	 * メッセージラベル(msglbl)を取得します。
	 *
	 * メッセージラベルは、lbl属性で登録された値を、
	 * それぞれのリソースに応じて各言語に変換した結果を格納しています。
	 *
	 * @og.rev 4.0.0.0 (2005/01/31) msglbl は、LabelInterface オブジェクトを利用
	 *
	 * @return   メッセージラベル
	 * @see		#setLbl( String )
	 */
	protected String getMsglbl() {
		String rtn = null;

		if( msglbl != null ) { rtn = msglbl.getLabel(); }

		return rtn ;
	}

	/**
	 * メッセージラベル(msglbl)のチップス表記を取得します。
	 *
	 * メッセージラベルは、lbl属性で登録された値を、
	 * それぞれのリソースに応じて各言語に変換した結果を格納しています。
	 * 別途、title属性を指定している場合は、置き換えます。
	 *
	 * @og.rev 4.0.0.0 (2005/01/31) msglbl は、LabelInterface オブジェクトを利用
	 *
	 * @return   メッセージラベル
	 */
	protected String getLongLabel() {
		String rtn = null;

		if( msglbl != null ) {
			rtn = msglbl.getLongLabel( get( "title" ) );
		}

		return rtn ;
	}

	/**
	 * メッセージラベル(LabelInterface)を取得します。
	 *
	 * メッセージラベルは、lbl属性で登録された値を、
	 * それぞれのリソースに応じて各言語に変換した結果を格納しています。
	 *
	 * @return   メッセージラベル
	 */
	protected LabelInterface getLabelInterface() {
		return msglbl ;
	}

	/**
	 * ResourceManager を取得します。
	 *
	 * ページスコープ にセットされた ResourceManager があれば、それを、
	 * なければ、language 属性よりデフォルト ResourceManager を構築します。
	 * LOCAL_RES_KEY で管理される ResourceManager は、LocalResourceTag で
	 * 登録されたリソースです。これは、ローカルリソース情報として、使用されます。
	 *
	 * @return	ResourceManagerオブジェクト
	 */
	protected ResourceManager getResource() {
		if( resource == null ) {
			resource = (ResourceManager)pageContext.getAttribute( HybsSystem.LOCAL_RES_KEY );
			if( resource == null ) {
				resource = ResourceFactory.newInstance( getLanguage() );
			}
		}
		return resource;
	}

	/**
	 * デバッグ状態 を取得します。
	 *
	 * setDebug( String )で登録します。
	 * 初期値は、false です。
	 *
	 * @og.rev 3.5.5.3 (2004/04/09) getDebug() から、メソッド名変更
	 *
	 * @return   true(デバッグ状態)/ false(通常)
	 */
	protected boolean isDebug() {
		return debugFlag ;
	}

	/**
	 * ラベル文字列を返します。
	 *
	 * これは、{&#064;LBL.XXXX %Y} 引数処理をおこなうための、サポートメソッドです。
	 * 引数は、"XXXX %Y" という状態で受け取ります。(LBL. ははずした形)
	 * ラベルには、通常のラベル(Label)以外に、Short,Tips,Description,RawShortLabel の情報を持っています。
	 * {&#064;LBL.XXXX %Y} の Y に、先のLabel,Short,Tips,Description,RawShortLabel,CodeData の頭文字('L','S','T','D','R','C')を
	 * 指定することで、それぞれの状態を取得することが可能になります。
	 * Y を指定しない場合({&#064;LBL.XXXX}) は、'L' が指定された場合と同じ効果があります。
	 * Y は、先頭文字１文字で判断していますので、{&#064;LBL.XXXX %Short}と記述できます。
	 * Y 引数には、&#064;で始まるリクエスト引数が指定できます。例えば、{&#064;LBL.XXXX &#064;ZZ}
	 * とすると、ZZ のリクエスト引数の値が Y に適用されます。
	 * ラベルキーそのものをパラメータ化することが出来ます。
	 * これは、{&#064;LBL.&#064;XXXX}という形式になります。引数は、先の説明と同じです。
	 * この場合は、XXXX をキーにリクエスト引数の値が、ラベルリソースのキーになります。
	 *
	 * @og.rev 4.0.0.0 (2007/10/17) メッセージリソース統合に伴い、{&#064;LBL.XXXX Y}⇒{&#064;LBL.XXXX %Y}
	 * @og.rev 5.4.0.1 (2011/11/01) ラベル形式('L','S','T','D') に、R(RawShortLabel) を追加
	 * @og.rev 5.5.7.2 (2012/10/09) ラベル形式('L','S','T','D','R') に、C(CodeData) を追加
	 *
	 * @param    lbl ラベルのキー
	 *
	 * @return   ラベル文字列
	 */
	protected String getLabel( final String lbl ) {

		String key = lbl ;
		String val = null;

		final int spc = lbl.indexOf( ' ' );		// " " があるかどうか
		if( spc > 0 ) {
			key = lbl.substring( 0,spc );
			if( key.charAt(0) == '@' ) { key = getRequestValue( key.substring( 1 ) ); }

			// リクエスト引数が指定された場合
			char ch  = lbl.length() > spc+1 ? lbl.toUpperCase( Locale.JAPAN ).charAt( spc+1 ) : ' ';	// Label,Short,Tips,Description
			char ch2 = lbl.length() > spc+2 ? lbl.toUpperCase( Locale.JAPAN ).charAt( spc+2 ) : ' ';	// Label,Short,Tips,Description
			if( ch == '@' ) {
				final String tmp = getRequestValue( lbl.substring( spc+2 ) );
				if( tmp != null && tmp.length() > 0 ) {
					ch  = tmp.toUpperCase( Locale.JAPAN ).charAt( 0 );
					ch2 = tmp.length() > 1 ? tmp.toUpperCase( Locale.JAPAN ).charAt( 1 ) : ' ';
				}
			}
			// 4.0.0.0 (2007/10/19)
			if( ch == '%' ) {
				switch( ch2 ) {
					case 'L': val = getResource().getLabel( key ); break;
					case 'S': val = getResource().getLabelData( key ).getShortLabel(); break;
					case 'T': val = getResource().getLabelData( key ).getLongLabel(); break;
					case 'D': val = getResource().getLabelData( key ).getDescription(); break;
					case 'R': val = getResource().getLabelData( key ).getRawShortLabel(); break;	// 5.4.0.1 (2011/11/01)
					case 'C': val = getResource().getLabelData( key + "." + getRequestValue( key ) ).getShortLabel(); break;	// 5.5.7.2 (2012/10/09)
					default : break;
				}
			}
			else if( ch != ' ' ) {
				String[] msgVals = StringUtil.csv2Array( lbl.substring( spc+1 ),' ' );
				for( int i=0; i<msgVals.length; i++ ) {
					// リクエスト文字パラメータ時の処理。その他は、ラベル文字は処理不要。
					if( msgVals[i].startsWith( "@" ) ) {
						msgVals[i] = getRequestValue( msgVals[i].substring( 1 ) );
					}
				}
				val = getResource().getLabel( key,msgVals );
			}
		}
		else {
			if( key.charAt(0) == '@' ) { key = getRequestValue( key.substring( 1 ) ); }
		}

		if( val == null ) { val = getResource().getLabel( key ); }
		return val;
	}

	/**
	 * DBColumn オブジェクトを返します。
	 *
	 * これは、キーを元に DBColumnオブジェクトをカラムリソースの
	 * 定義ファイルより取得して、リソースマネージャで管理します。
	 *
	 * @param	key	オブジェクトのキー
	 *
	 * @return	DBColumnオブジェクト
	 */
	protected DBColumn getDBColumn( final String key ) {
		return getResource().makeDBColumn( key ) ;
	}

	/**
	 * 内部の Attributes オブジェクトに、属性値をセットします。
	 *
	 * 同じキーの値が登録されていた場合は、置き換えられます。
	 *
	 * @param   key   キー
	 * @param   value 属性値
	 * @see		#add( String , String )
	 */
	protected void set( final String key, final String value ) {
		attri.set( key,value );
	}

	/**
	 * 内部の Attributes オブジェクトに、属性値を追加します。
	 *
	 * ここでは、すでに同じキーが登録されている場合は、その値に、
	 * 標準セパレータ(スペース)を追加して、文字列結合します。
	 * たとえば、class 属性などは、値をスペースで追加する事で、
	 * CSS で処理することができます。
	 *
	 * @og.rev 4.0.0.0 (2007/05/18) 新規追加
	 *
	 * @param   key   キー
	 * @param   value 属性値
	 * @see		#add( String , String , String )
	 * @see		#set( String , String )
	 */
	protected void add( final String key, final String value ) {
		attri.add( key,value );
	}

	/**
	 * 内部の Attributes オブジェクトに、属性値を追加します。
	 *
	 * ここでは、すでに同じキーが登録されている場合は、その値に、
	 * 引数のセパレータを追加して、文字列結合します。
	 *
	 * @og.rev 3.5.0.0 (2003/09/17) 新規追加
	 * @og.rev 3.5.5.9 (2004/06/07) セパレータ引数付きのメソッドに変更
	 *
	 * @param   key   キー
	 * @param   value 属性値
	 * @param   sepa  セパレータ
	 * @see		#add( String , String )
	 */
	protected void add( final String key, final String value, final String sepa ) {
		attri.add( key,value,sepa );
	}

	/**
	 * 内部の Attributes オブジェクトから、属性値を取得します。
	 *
	 * @param	key	キー
	 *
	 * @return	属性値
	 * @see		#set( String , String )
	 */
	protected String get( final String key ) {
		return attri.get( key );
	}

	/**
	 * 属性オブジェクトの取得。
	 *
	 * Attributes オブジェクトを取得します。
	 *
	 * @return	Attributesオブジェクト
	 */
	protected Attributes getAttributes() {
		return attri;
	}

	/**
	 * {&#064;XXXX} 形式の文字列から XXXX をキーとして ServletRequest から getParameter で値を取り出します。
	 *
	 * 他の文字列に混在した {&#064;XXXX} 文字を変換可能です。
	 * ただし、処理の簡素化のため、上記形式以外は変換いたしません。
	 * エラー例)× { &#064;XXXX }、{&#064; XXXX }、{&#064;XXXX&#064;yyyy}、{&#064;XXXX{&#064;yyyy}}
	 * また、"{&#064;" を通常の記述で使うことは無いと考え、エスケープも用意して
	 * いません。よって、"{&#064;" のパターンが見つかって,"}" で閉じられていない
	 * 場合は,エラーとして、HybsSystemException を throw します。
	 *
	 * @og.rev 3.8.0.4 (2005/08/08) {} の処理方法見直し。連続処理、単体処理を可能にします。
	 *
	 * @param   key リクエストのキー
	 *
	 * @return  リクエストの値
	 * @og.rtnNotNull
	 */
	protected String getRequestParameter( final String key ) {
		isReqNull = false;

		if( key == null ) { isReqNull = true; return ""; }
		int index = key.indexOf( "{@" );
		if( index < 0 ) { return key; }

		// 変数が "{@XXXX}" の場合を優先的に検索。
		// これにより多くのパターンで、StringTokenizer による
		// 文字列操作を行う必要がなくなります。
		if( index == 0 &&
			( key.indexOf( '}' ) == key.lastIndexOf( '}' ) ) &&
			( key.charAt(key.length()-1) == '}' ) ) {
				return getRequestValue( key.substring( 2,key.length()-1 ) );
		}

		// 3.8.0.4 (2005/08/08) {} の処理方法見直し。連続処理、単体処理を可能にします。
		final StringBuilder rtn = new StringBuilder( BUFFER_MIDDLE );
		int start = 0;
		while( index >= 0 ) {
			final int end = key.indexOf( '}',index );
			if( end < 0 ) {
				final String errMsg = "{@ と } との対応関係がずれています。" + CR
							+ "key=[" + key + "] : index=" + index ;
				throw new HybsSystemException( errMsg );
			}

			// {@ より前方の文字列を追加
			if( index > 0 ) { rtn.append( key.substring( start,index ) ); }

			// {@XXXX} の XXXX部分を処理
			final String val = getRequestValue( key.substring( index+2,end ) );
			if( val != null ) { rtn.append( val ); }

			start = end+1 ;
			index = key.indexOf( "{@",start );
		}
		rtn.append( key.substring( start ) );

		return rtn.toString();
	}

	/**
	 * {&#064;XXX.YYYY} 形式の文字列から値を取得します。
	 * 予約語のみ処理をし、それ以外は{&#064;xxx}のままとします。
	 *
	 * 他の文字列に混在した {&#064;XXXX} 文字を変換可能です。
	 * ただし、処理の簡素化のため、上記形式以外は変換いたしません。
	 * エラー例）× { &#064;XXXX }、{&#064; XXXX }、{&#064;XXXX&#064;yyyy}、{&#064;XXXX{&#064;yyyy}}
	 * また、"{&#064;" を通常の記述で使うことは無いと考え、エスケープも用意して
	 * いません。よって、"{&#064;" のパターンが見つかって,"}" で閉じられていない
	 * 場合は,エラーとして、HybsSystemException を throw します。
	 *
	 * @og.rev 5.5.4.0 (2012/07/02) 新規作成
	 *
	 * @param   key リクエストのキー
	 *
	 * @return  リクエストの値
	 * @og.rtnNotNull
	 */
	protected String getReservedParameter( final String key ) {
		isReqNull = false;

		if( key == null ) { isReqNull = true; return ""; }
		int index = key.indexOf( "{@" );
		if( index < 0 ) { return key; }

		// 変数が "{@XXXX}" の場合を優先的に検索。
		// これにより多くのパターンで、StringTokenizer による
		// 文字列操作を行う必要がなくなります。
		if( index == 0 &&
			( key.indexOf( '}' ) == key.lastIndexOf( '}' ) ) &&
			( key.charAt(key.length()-1) == '}' ) ) {
				return getReservedValue( key.substring( 2,key.length()-1 ) );
		}

		// 3.8.0.4 (2005/08/08) {} の処理方法見直し。連続処理、単体処理を可能にします。
		final StringBuilder rtn = new StringBuilder( BUFFER_MIDDLE );
		int start = 0;
		while( index >= 0 ) {
			final int end = key.indexOf( '}',index );
			if( end < 0 ) {
				final String errMsg = "{@ と } との対応関係がずれています。" + CR
							+ "key=[" + key + "] : index=" + index ;
				throw new HybsSystemException( errMsg );
			}

			// {@ より前方の文字列を追加
			if( index > 0 ) { rtn.append( key.substring( start,index ) ); }

			// {@XXXX} の XXXX部分を処理
			final String val = getReservedValue( key.substring( index+2,end ) );
			if( val != null ) { rtn.append( val ); }

			start = end+1 ;
			index = key.indexOf( "{@",start );
		}
		rtn.append( key.substring( start ) );

		return rtn.toString();
	}

	/**
	 * {&#064;XXXX} 形式の文字列から XXXX をキーとして ServletRequest から getParameterValues で値を取り出します。
	 *
	 * これは、複数(配列)でリクエストを取り出すことが可能です。
	 * そのため､他の文字列に混在させて変換することができません。
	 * "{&#064;XXXX}" 形式 からのみの変換となります。
	 *
	 * @og.rev 3.6.0.0 (2004/09/22) キーがnull のときにnullではなく長さ０の配列を返します。
	 *
	 * @param   key リクエストのキー
	 *
	 * @return  リクエストの値
	 * @og.rtnNotNull
	 */
	protected String[] getRequestParameterValues( final String key ) {
		if( key == null ) { return new String[0]; }		// 3.6.0.0 (2004/09/22)
		final int index = key.indexOf( "{@" );
		if( index < 0 ) { return StringUtil.csv2Array( key ); }

		if( index == 0 && key.charAt( key.length()-1 ) == '}' ) {
			return getRequestValues( key.substring( 2,key.length()-1 ) );
		}

		final String errMsg = "引数の形式が異なります。 [" + key + "]" ;
		throw new HybsSystemException( errMsg );
	}

	/**
	 * 引数 in が、引数 check の文字列の中に存在すれば､ true を、存在しなければ､false を返します。
	 *
	 * check は、 非null のString を、in は、null でも構いません。
	 *
	 * @param    in    チェックする文字列
	 * @param    check チェック用の基本文字列
	 *
	 * @return   存在する true ／  存在しない false
	 */
	protected boolean check( final String in, final String check ) {
		if( in == null ) { return false; }
		return check.indexOf( in ) >= 0 ;
	}

	/**
	 * 引数 in が、引数 check の文字列配列の中に存在すれば､ true を、存在しなければ､false を返します。
	 *
	 * check は、 String配列 を、in は、null でも構いません。
	 *
	 * @og.rev 2.1.0.3 (2002/11/08) 文字列配列を引数に取るメソッドを追加
	 *
	 * @param    in    チェックする文字列
	 * @param    check チェック用の基本文字列配列(可変長引数)
	 *
	 * @return   存在する true ／  存在しない false
	 */
//	protected boolean check( final String in, final String[] check ) {
	protected boolean check( final String in, final String... check ) {
//		if( in == null || check == null ) { return false; }
//		for( int i=0; i<check.length; i++ ) {
//			if( in.equals( check[i] ) ) { return true; }
//		}
//		return false ;
		// 6.1.1.0 (2015/01/17) 可変長引数でもnullは来る。
		if( in != null && check != null ) {
			for( int i=0; i<check.length; i++ ) {
				if( in.equals( check[i] ) ) { return true; }
			}
		}
		return false ;
	}

	/**
	 * ユーザーオブジェクトが持っている内部情報を取得します。
	 *
	 * これは、UserInfo#getAttribute( String ) で取得される情報です。
	 * ユーザーパラメータとは異なります。
	 *
	 * @og.rev 2.2.0.0 (2002/12/17) 中国語(国際化)対応 getUser() を使用するように変更
	 *
	 * @param    user ユーザー情報を取り出すキー
	 *
	 * @return   ユーザー情報文字列
	 */
	protected String getUserInfo( final String user ) {
		if( user == null ) { return null; }

		String key = user ;
		if( key.charAt(0) == '@' ) {
			key = getRequestValue( key.substring( 1 ) );
		}

		return getUser().getAttribute( key );
	}

	/**
	 * ユーザーオブジェクトが持っているEditの内部情報を取得します。
	 *
	 * これは、UserInfo#getSelectedEdit( String ) で取得される情報です。
	 * ユーザーパラメータとは異なります。
	 *
	 * @og.rev 5.8.2.3 (2014/12/27) 新規作成
	 *
	 * @param    guikey 取り出す画面ID
	 *
	 * @return   ユーザー情報文字列
	 */
	protected String getUserEditInfo( final String guikey ) {
		if( guikey == null ) { return null; }

		String key = guikey ;
		if( key.charAt(0) == '@' ) {
			key = getRequestValue( key.substring( 1 ) );
		}

		return getUser().getSelectedEdit( key );
	}

	/**
	 * ユーザー情報を設定します。
	 *
	 * 初めての場合は、session 登録項目 の HybsSystem#USERINFO_KEY キー の値で
	 * 取得します。
	 * save属性は、GE20(ユーザー定数)に情報を保存するかどうかを指定します。
	 *
	 * @og.rev 2.1.1.4 (2002/11/25) ユーザー情報をセットするメソッドを追加
	 * @og.rev 2.2.0.0 (2002/12/17) 中国語(国際化)対応 getUser() を使用するように変更
	 * @og.rev 4.3.4.0 (2008/12/01) GE20(ユーザー定数)へ登録するかのフラグを追加
	 *
	 * @param    key ユーザー情報をセットするキー
	 * @param    value ユーザー情報文字列
	 * @param    save 情報保存 [true:保存/false:保存しない]
	 */
	protected void setUserInfo( final String key,final String value, final boolean save ) {
		if( key != null ) {
			getUser().setAttribute( key, value, save );
		}
	}

	/**
	 * ユーザー情報オブジェクトを取得します。
	 *
	 * 初めての場合は、session 登録項目 の HybsSystem#USERINFO_KEY キー の値で
	 * 取得します。
	 *
	 * @og.rev 2.2.0.0 (2002/12/17) 中国語(国際化)対応 新規追加
	 * @og.rev 3.6.0.0 (2004/09/17) private ⇒ protected 化します。
	 *
	 * @return   ユーザー情報オブジェクト
	 * @og.rtnNotNull
	 */
	protected UserInfo getUser() {
		if( userInfo == null ) {
			userInfo = (UserInfo)getSessionAttribute( HybsSystem.USERINFO_KEY );
		}
		if( userInfo == null ) {
			final String errMsg = "ユーザーオブジェクトが存在しません。";
			throw new HybsSystemException( errMsg );
		}
		return userInfo ;
	}

	/**
	 * 画面情報(GUIInfo)を取得します。
	 *
	 * これは、session 登録項目 の HybsSystem#GUIMAP_KEY キー の値で
	 * 登録された MAP を取り出し、そこから取得します。
	 * 画面情報は、ログインユーザー毎に個別に持っています。
	 *
	 * @og.rev 4.0.0.0 (2005/01/31) GUIInfo が存在しない場合も処理を続けます。
	 *
	 * @param    gamenId 画面ID
	 *
	 * @return   画面情報(GUIInfo)
	 */
	protected GUIInfo getGUIInfo( final String gamenId ) {
		return getUser().getGUIInfo( gamenId );
	}

	/**
	 * 画面情報(GUIInfo)の属性値を取得します。
	 *
	 * これは、{&#064;GUI.XXXX ID} 引数処理をおこなうための、サポートメソッドです。
	 * 引数は、"XXXX ID" という状態で受け取ります。(GUI. ははずした形)
	 * XXXXには、画面情報(GUIInfo)の属性キーを指定します。IDが指定されない場合は、
	 * 実行中の自分自身の画面が指定されたことになります。
	 * これは、session 登録項目 の HybsSystem#GUIINFO_KEY キー の値で取得します。
	 * この値は,画面が呼び出される毎に毎回設定されており、リクエスト毎に
	 * 所得し直す必要があります。
	 *
	 * ID に、画面IDを指定した場合は、&#064; 指定によるリクエスト引数の値を適用できます。
	 * {&#064;GUI.&#064;XXXX ID} や、{&#064;GUI.XXXX &#064;ID} です。(同時指定も可能)
	 *
	 * @og.rev 3.6.0.6 (2004/10/22) GUIInfo が存在しない場合も処理を続けます。
	 * @og.rev 4.0.0.0 (2004/11/30) 画面ID引数や、リクエスト引数の使用を可能にします。
	 *
	 * @param    attkey 画面情報を取り出すキー
	 *
	 * @return   画面情報文字列
	 */
	protected String getGUIInfoAttri( final String attkey ) {
		if( attkey == null ) { return null; }

		String  key = attkey ;
		final GUIInfo gui ;

		final int spc = key.indexOf( ' ' );		// " " があるかどうか
		if( spc > 0 ) {
			key = attkey.substring( 0,spc );
			String id = attkey.substring( spc+1 );
			if( id.charAt(0) == '@' ) { id = getRequestValue( id.substring( 1 ) ); }

			gui = getUser().getGUIInfo( id );
		}
		else {
			if( guiInfo == null ) {
				guiInfo = (GUIInfo)getSessionAttribute( HybsSystem.GUIINFO_KEY );
			}
			gui = guiInfo;
		}
		if( gui == null ) { return "Not Found[" + attkey + "]"; }
		if( key.charAt(0) == '@' ) { key = getRequestValue( key.substring( 1 ) ); }

		return gui.getAttribute( key );
	}

	/**
	 * {&#064;NVL.XXX 値} で、XXXが NULLの場合、値を返します。
	 *
	 * ORACLE等の COALESCE ( expr_list ) 処理に近い結果を返します。
	 * NVL ( expr1 , expr2 ) は、expr1 が NULL の場合、expr2 を返しますが、
	 * COALESCE は、第一引数が NULLなら、次の引数、それも NULL なら、さらに次と
	 * 引数リストを順次処理していきます。
	 * それと同じく、XXX が NULL なら、その次・・と順次評価していき、NULL でない
	 * 値が返ってきたところで、その値を返します。
	 * これは、{&#064;NVL.XXX &#064;YYY ZZZ ･･･} 形式を処理します。
	 * これは、パラメータ XXX が NULLの場合、次の値を使います。(&#064;YYY) 
	 * &#064;YYY は、YYY パラメータの事で、これも NULL の場合は、ZZZ を使います。
	 * 最後まで NULL の場合は、 ゼロ文字列が返されます。
	 *
	 * @og.rev 5.6.4.0 (2013/05/02) 新規追加
	 *
	 * @param    attkey NVL情報を取り出すパラメータ
	 *
	 * @return   NULL以外の値が出てくるまで、順次評価した結果
	 */
	protected String getNVLAttri( final String attkey ) {
		if( attkey == null ) { return null; }

		final String[]  keys = attkey.split( " " );			// スペースで、パラメータを分解する。
		String    val  = getRequestValue( keys[0] );	// 第一パラメータは、そのままのキーで検索

		// val が null の間は、チェックする。
		if( val == null || val.isEmpty() ) {
			for( int i=1; i<keys.length; i++ ) {
				val = keys[i];
				// 先頭が @ の場合は、リクエスト変数のキーとして、値を判定
				if( val.charAt(0) == '@' ) { val = getRequestValue( val.substring( 1 ) ); }
				if( val != null && val.length() > 0 ) { break; }
			}
		}

		if( val == null ) { val = ""; }		// NULL の場合は、 ゼロ文字列を返す。

		return val;
	}

	/**
	 * {&#064;LAST.XXX} で、XXXが 最後に使われた値を返します。
	 *
	 * XXX は、command="NEW" でセットされたリクエスト値です。通常、{&#064;MEM.XXX} は
	 * 画面単位に、既存のキャッシュから値を取り出しますが、{&#064;LAST.XXX} は、
	 * 画面に関係なく、ユーザー単位に管理しています。
	 * また、値は、データベース(GE20)に保管されますので、次回ログイン時にも有効です。
	 * この処理が呼び出されたときに、リクエスト変数に、XXXX が存在した場合は、そちらを優先に
	 * 使います。その場合は、command に関係なく、値を設定しておきます。
	 *
	 * command="NEW"の場合のリクエスト変数の値の設定は、RequestCacheTag で行います。
	 * 
	 * ※ データベースには、画面アクセス情報のセーブ時に行われます。
	 * valueタグのsave属性の様に、リアルタイムではありませんので、Tomcatが
	 * 異常終了した場合は、セーブされません。
	 *
	 * @og.rev 5.6.8.1 (2013/09/13) 新規追加
	 *
	 * @param    attkey 最後に使われた値をを取り出すパラメータ
	 *
	 * @return   XXXが 最後に使われた値
	 * @see		org.opengion.hayabusa.taglib.RequestCacheTag#commandExec( String ,ServletRequest )
	 * @see		org.opengion.hayabusa.resource.UserInfo#getLastRequestValue( String )
	 */
	protected String getLASTAttri( final String attkey ) {
		if( attkey == null ) { return null; }

		// 最新のリクエスト変数をチェック
		final String[] vals = getRequest().getParameterValues( attkey );

		String val = null;
		if( vals == null ) {
			val = getUser().getLastRequestValue( attkey );		// なければ、取得
		}
		else {
			for( int i=0; i<vals.length; i++ ) {
				val = vals[i];
				if( ! "0".equals( val ) ) { break; }			// チェックボックス対応
			}
			getUser().setLastRequestValue( attkey,val );		// あれば、最新の値をセット
		}

		return val ;
	}

	/**
	 * 予約語に関する情報の文字列を取得します。
	 *
	 * @og.rev 5.5.4.0 (2012/07/02) 予約語部分のみ分離
	 * @og.rev 5.6.4.0 (2013/05/02) NVL 追加
	 * @og.rev 5.6.8.1 (2013/09/13) LAST 追加
	 * @og.rev 5.8.2.3 (2014/12/27) USEREDIT追加
	 *
	 * @param    key キー
	 *
	 * @return   リクエスト情報の文字列
	 * @og.rtnNotNull
	 */
	protected String getReservedValue( final String key ) {
		if( key == null ) { isReqNull = true; return ""; }		// 3.4.0.3 (2003/09/10)

		String rtn ;
		final int adrs = key.indexOf( '.' );
		if( adrs > 0 ) {
			final String subKey = key.substring( adrs+1 );
			if( key.startsWith( "USER." ) ) {
				rtn = getUserInfo( subKey );
			}
			else if( key.startsWith( "USEREDIT." ) ) {
				rtn = getUserEditInfo( subKey );	// 5.8.2.3 (2014/12/27)
			}
			else if( key.startsWith( "GUI." ) ) {
				rtn = getGUIInfoAttri( subKey );	// 4.0.0 (2005/01/31)
			}
			else if( key.startsWith( "SYS." ) ) {
				rtn = sys( subKey );		// 3.5.6.6 (2004/08/23)
			}
			else if( key.startsWith( "SESSION." ) ) {		// 3.5.5.3 (2004/04/09)
				rtn = String.valueOf( getSessionAttribute( subKey ) );
			}
			// 3.4.0.3 (2003/09/10) MEM.XXXX で、REQUEST_CACHE の値を取得できるように修正。
			else if( key.startsWith( "MEM." ) ) {
				// 3.5.4.7 (2004/02/06) getRequestCacheData を使用するように修正
				rtn = getRequestCacheData( subKey );
			}
			// 3.8.0.2 (2005/07/11) MSG.XXXX で、メッセージリソースの値を取得できるように追加。
			// 3.8.0.2 (2005/07/11) LBL.XXXX で、ラベルリソースの値を取得できるように追加。
			else if( key.startsWith( "LBL." ) ) {
				rtn = getLabel( subKey );
			}
			// 3.8.0.2 (2005/07/11) DATE.XXXX で、日付関係の値を取得できるように追加。
			else if( key.startsWith( "DATE." ) ) {
				rtn = getDateFormat( subKey );
			}
			// 3.8.0.1 (2005/06/17) NVAR.XXXX で、getUnicodeEscape 変換() を行います。
			// NVAR. を取り除いた XXXX で再度、リクエスト値を取得し、それを Escape変換します。
			else if( key.startsWith( "NVAR." ) ) {
				rtn = StringUtil.getUnicodeEscape( getRequestValue( subKey ) );
			}
			// 4.3.6.0 (2009/04/01) メールモジュール用の予約語
			else if( key.startsWith( "MAIL." ) ) {
				rtn = ( String )getSessionAttribute( key );
			}
			// 4.3.7.0 (2009/06/01) DB関数名の取得
			else if( key.startsWith( "DBF." ) ) {
				rtn = getDBFunctionName( subKey );
			}
			// 4.4.0.0 (2009/08/02) データロールに基づく条件式の取得
			else if( key.startsWith( "SEC." ) ) {
				rtn = getDataCondition( subKey );
			}
			// 5.3.9.0 (2011/09/01) URLエンコード変換
			else if( key.startsWith( "URL." ) ) {
				rtn = StringUtil.urlEncode( getRequestValue( subKey ) );
			}
			// 5.5.1.3 (2012/04/09) エスケープ変換
			else if( key.startsWith( "ESC." ) ) {
				rtn = StringUtil.htmlFilter( getRequestValue(subKey,false) );
			}
	 		// 5.6.4.0 (2013/05/02) NVL 追加
			else if( key.startsWith( "NVL." ) ) {
				rtn = getNVLAttri( subKey );
			}
	 		// 5.6.8.1 (2013/09/13) LAST 追加
			else if( key.startsWith( "LAST." ) ) {
				rtn = getLASTAttri( subKey );
			}
			// 4.0.0 (2007/06/12) DB.XXXX は、直接取り出すように変更します。
			else { // 4.0.0.0 (2007/11/16)
				rtn = (String)getRequestAttribute( key );		// ※ 取り出しは、key で
			}
		}
		else{
			rtn = "{@" + key + "}"; // 予約語以外は括弧を付けて書き戻します。
		}
		return rtn;
	}

	/**
	 * リクエスト情報の文字列を取得します。
	 *
	 * @og.rev 5.0.0.2 (2009/09/15) XSS対策
	 *
	 * @param    key キー
	 *
	 * @return   リクエスト情報の文字列
	 */
	protected String getRequestValue( final String key ) {
		return getRequestValue( key, xssCheck);
	}

	/**
	 * リクエスト情報の文字列を取得します。
	 *
	 * @og.rev 2.2.0.0 (2002/12/17) 中国語(国際化)対応 エンコードの取得方法変更
	 * @og.rev 3.0.0.0 (2002/12/25) StringUtil#changeString 廃止
	 * @og.rev 3.0.0.0 (2002/12/25) ValueTag追加の為、指定の scope の Attributeより取得
	 * @og.rev 3.1.0.1 (2003/03/26) Valueタグの値と、request情報の値の所得優先順位を、request が優先されるように変更。
	 * @og.rev 3.1.1.2 (2003/04/04) Tomcat4.1 対応。HybsRequestWrapper 廃止。直接 Mapでキャッシュする。
	 * @og.rev 3.1.5.0 (2003/04/22) SYS.XXXX で、システムパラメータ の値を取得できるように修正。
	 * @og.rev 3.1.7.0 (2003/05/02) リクエスト情報の取得順序を、Request、キャッシュ、Value の順に変更。
	 * @og.rev 3.1.7.0 (2003/05/02) value値の使用可否を指定する、useValue 属性を追加。
	 * @og.rev 3.4.0.3 (2003/09/10) MEM.XXXX で、REQUEST_CACHE の値を取得できるように修正。
	 * @og.rev 3.5.4.7 (2004/02/06) getRequestCacheData を使用するように修正
	 * @og.rev 3.5.5.3 (2004/04/09) {&#064;SESSION.XXXX} で、session.getAttribute( "XXXX" ) の値を取得するように修正
	 * @og.rev 3.5.6.6 (2004/08/23) SYS.XXXX の処理を getSystemParameter( String key ) メソッドへ移動
	 * @og.rev 3.8.0.1 (2005/06/17) NVAR.XXXX で、getUnicodeEscape 変換() を行います。
	 * @og.rev 3.8.0.2 (2005/07/11) MSG.XXXX , LBL.XXXX の処理を追加
	 * @og.rev 3.8.0.2 (2005/07/11) チェックボックス対応で、重複リクエストに対応させます。
	 * @og.rev 3.8.8.8 (2007/05/11) 重複リクエスト処理の場所を移動。リクエストのみ対象とする。
	 * @og.rev 4.0.0.0 (2005/08/31) quotCheck によるＳＱＬインジェクション対策
	 * @og.rev 4.0.0.0 (2005/08/31) getSystemParameter を sys に名称変更
	 * @og.rev 4.0.0.0 (2007/04/02) Valueタグの値と、キャッシュでは、Valueタグの値を優先するように変更
	 * @og.rev 4.0.0.0 (2007/11/16) "."付きのパラメータのエラー処理をなくし、getRequestAttributeで取得する。
	 * @og.rev 4.3.0.0 (2008/07/04) DB.XXXX は、必ずStringオブジェクトとし、String.valueOf しない。
	 * @og.rev 4.3.6.0 (2009/04/01) メールモジュール用の予約語MAIL.XXXXの取得対応
	 * @og.rev 4.4.0.0 (2009/08/02) データロール対応(SEC.xxxの取得対応)
	 * @og.rev 5.0.0.2 (2009/09/15) XSS対策用にメソッドにフラグを追加
	 * @og.rev 5.1.8.0 (2010/07/01) isNullSet 属性 廃止にともなう、useValue 属性廃止
	 * @og.rev 5.3.9.0 (2011/09/01) URL.XXXX処理を追加
	 * @og.rev 5.5.1.3 (2012/04/09) ESC.XXXX処理を追加
	 * @og.rev 5.5.4.0 (2012/07/01) 予約語の処理を分離
	 * @og.rev 5.7.4.2 (2014/03/20) サニタイズ処理は、getSanitizedBodyString() ではなく、ここで行います。
	 *
	 * @param	key	キー
	 * @param	xssCheckFlg	XSS対策用
	 *
	 * @return   リクエスト情報の文字列
	 * @og.rtnNotNull
	 */
	protected String getRequestValue( final String key, final boolean xssCheckFlg ) {
		if( key == null ) { isReqNull = true; return ""; }		// 3.4.0.3 (2003/09/10)

		String rtn ;
		final int adrs = key.indexOf( '.' );

		if( adrs > 0 ) {
			rtn = getReservedValue( key ); // 5.5.4.0 (2012/07/02)
		}
		else {
			rtn = getRequest().getParameter( key );

			// 5.7.4.2 (2014/03/20) サニタイズ処理は、getSanitizedBodyString() ではなく、ここで行います。
			// 6.0.0.1 (2014/04/25) These nested if statements could be combined
			if( isSanitized && rtn != null && rtn.indexOf( '[' ) >= 0 ) {
				rtn = rtn.replace( "[", "\\]\\" );
			}

			// 5.0.0.2 (2009/09/15) tagCheck によるthan signチェック Parameterのみにかけるためこの位置
			if( rtn != null && rtn.length() > 0 && xssCheckFlg && ( rtn.indexOf( '<' ) >= 0 || rtn.indexOf( '>' ) >= 0 ) ) {
				final String errMsg = "リクエスト引数に Less/Greater than sign(<,>)を含むことは出来ません。" + CR
							+ " 処理が正しい場合は、xssCheck 属性を false にセットしてください。" + CR
							+ " key=[" + key + "]"
							+ " val=[" + rtn + "]"
							+ " tag=[" + getTagName() + "]" ;
				throw new HybsSystemException( errMsg );
			}

			// 3.8.8.8 (2007/05/11) 重複リクエスト処理の場所を移動。リクエストのみ対象とする。
			// 3.8.0.2 (2005/07/11) チェックボックス対応で、重複リクエストに対応させます。
			// {@XXXX} で、値が"0"の場合、複数リクエストを確認して、"1"が含まれていれば、"1"とします。
			if( "0".equals(rtn) ) {
				final boolean backFlag = isReqNull ;
				final String[] vals = getRequestValues( key );
				if( vals != null && vals.length > 1 ) {
					for( int i=0; i<vals.length; i++ ) {
						if( "1".equals( vals[i] ) ) { rtn = "1"; break; }
					}
				}
				isReqNull = backFlag;	// 3.8.8.8 (2007/05/11) getRequestValues での NULLセット解除
			}

			// 3.1.0.1 (2003/03/26) Valueタグの値と、request情報の値の取得優先順位を、
			// request が優先されるように変更。
			if( ( rtn == null || rtn.isEmpty() ) && requestCache != null ) {
				final String[] str = requestCache.get( key );
				if( str != null && str.length > 0 ) {
					rtn = str[0];
				}
			}
			// 5.1.8.0 (2010/07/01) isNullSet 属性 廃止にともなう、useValue 属性廃止
			if( rtn == null || rtn.isEmpty() ) {
				final Object obj = pageContext.findAttribute( key );
				if( obj != null ) {
					rtn = obj.toString();
				}
			}
		}
		if( rtn == null || rtn.isEmpty() ) {
			isReqNull = true;
			rtn    = "";
		}
		// 4.0.0 (2005/08/31) quotCheck によるＳＱＬインジェクション対策
//		else if( quotCheck && rtn.indexOf( "'" ) >= 0 && !key.startsWith( "SEC." ) ) {
		else if( quotCheck && rtn.indexOf( '\'' ) >= 0 && !key.startsWith( "SEC." ) ) {		// 6.0.2.5 (2014/10/31) refactoring
			final String errMsg = "リクエスト引数に、クォーティション(')を含むことは出来ません。" + CR
						+ " 処理が正しい場合は、quotCheck 属性を false にセットしてください。" + CR
						+ " key=[" + key + "]"
						+ " val=[" + rtn + "]"
						+ " tag=[" + getTagName() + "]" ;
			throw new HybsSystemException( errMsg );
		}

		return rtn ;
	}

	/**
	 * リクエスト情報の文字列を取得します。
	 *
	 * @og.rev 2.2.0.0 (2002/12/17) 中国語(国際化)対応 エンコードの取得方法変更
	 * @og.rev 3.0.0.0 (2002/12/25) StringUtil#changeString 廃止
	 * @og.rev 3.1.8.0 (2003/05/16) RequestCache データをリクエスト配列情報にも適用する。
	 * @og.rev 5.3.8.0 (2011/08/01) Attribute等からも値が取得できるようにする。
	 *
	 * @param    key キー
	 *
	 * @return   リクエスト情報の文字列
	 */
	protected String[] getRequestValues( final String key ) {
		String[] rtn = getRequest().getParameterValues( key );

		// 3.1.8.0 (2003/05/16) RequestCache データをリクエスト配列情報にも適用する。
		if( ( rtn == null || rtn.length == 0 ) && requestCache != null ) {
			rtn =requestCache.get( key );
		}

		// 5.3.8.0 (2011/08/01) Attribute等からも値が取得できるようにする。
		if( rtn == null || rtn.length == 0 ) {
			final String tmp = getRequestValue( key );
			if( tmp != null && tmp.length() > 0 ) {
				rtn = new String[]{ tmp };
			}
		}

		if( rtn == null || rtn.length == 0 ) { isReqNull = true; }
		return rtn ;
	}

	/**
	 * リクエスト情報の文字列のキー集合を取得します。
	 *
	 * @og.rev 5.3.2.0 (2011/02/01) パラメーターの外部指定対応
	 *
	 * @return   リクエスト情報の文字列のキー集合
	 */
	protected Enumeration<?> getParameterNames() {		// 4.3.3.6 (2008/11/15) Generics警告対応
		final String[] names = (String[])getRequestAttribute( HybsSystem.PARAM_NAMES_KEY );
		return names == null ? getRequest().getParameterNames() : Collections.enumeration( Arrays.asList( names ) ) ;
	}

	/**
	 * リクエスト情報の文字列のキー集合をセットします。
	 *
	 * @og.rev 5.3.2.0 (2011/02/01) パラメーターの外部指定対応
	 *
	 * @param names リクエスト情報の文字列のキー配列(可変長引数)
	 */
//	protected void setParameterNames( final String[] names ) {
	protected void setParameterNames( final String... names ) {
		setRequestAttribute( HybsSystem.PARAM_NAMES_KEY, names );
	}

	/**
	 * リクエスト情報の文字列に NULL が存在していたかどうかを取得します。
	 *
	 * これは、getRequestParameter( String ) の呼出し毎に設定されます。
	 * つまり、上記メソッドの実行直後の値を取り出す必要があります。
	 * NULL が含まれていた(true)／含まれていなかった。(false)
	 *
	 * @return	NULLが含まれていた(true)／含まれていなかった。(false)
	 */
	protected boolean isNull() {
		return isReqNull;
	}

	/**
	 * セッションに登録されているオブジェクトを取得します。
	 *
	 * @param   key キー
	 *
	 * @return   セッションに登録されているオブジェクト
	 */
	protected Object getSessionAttribute( final String key ) {
		if( session == null ) { session = pageContext.getSession(); }
		return session.getAttribute( key );
	}

	/**
	 * セッションに 指定のキーでオブジェクトをセットします。
	 *
	 * @param   key キー
	 * @param   object セッションに登録するオブジェクト
	 */
	protected void setSessionAttribute( final String key ,final Object object ) {
		if( session == null ) { session = pageContext.getSession(); }
		session.setAttribute( key,object );
	}

	/**
	 * セッションに指定のキーで登録されているオブジェクトを 削除します。
	 *
	 * @param   key キー
	 */
	protected void removeSessionAttribute( final String key ) {
		if( session == null ) { session = pageContext.getSession(); }
		session.removeAttribute( key );
	}

	/**
	 * リクエストに登録されているオブジェクトを取得します。
	 *
	 * @param   key キー
	 *
	 * @return   リクエストンに登録されているオブジェクト
	 */
	protected Object getRequestAttribute( final String key ) {
		return getRequest().getAttribute( key );
	}

	/**
	 * リクエストに 指定のキーでオブジェクトをセットします。
	 *
	 * @param   key キー
	 * @param   object リクエストに登録するオブジェクト
	 */
	protected void setRequestAttribute( final String key ,final Object object ) {
		getRequest().setAttribute( key,object );
	}

	/**
	 * リクエストに指定のキーで登録されているオブジェクトを 削除します。
	 *
	 * @param   key キー
	 */
	protected void removeRequestAttribute( final String key ) {
		getRequest().removeAttribute( key );
	}

	/**
	 * コンテキスト(applicaton)に登録されているオブジェクトを取得します。
	 *
	 * scope属性に、"applicaton" が指定された場合に、実行されます。
	 *
	 * @og.rev 3.0.0.0 (2002/12/25) scope="applicaton" 指定の追加
	 *
	 * @param   key キー
	 *
	 * @return   コンテキスト(applicaton)に登録されているオブジェクト
	 */
	protected Object getContextAttribute( final String key ) {
		final ServletContext applicaton = pageContext.getServletContext();
		return applicaton.getAttribute( key );
	}

	/**
	 * コンテキスト(applicaton)指定のキーでオブジェクトをセットします。
	 *
	 * scope属性に、"applicaton" が指定された場合に、実行されます。
	 *
	 * @og.rev 3.0.0.0 (2002/12/25) scope="applicaton" 指定の追加
	 *
	 * @param   key キー
	 * @param   object コンテキスト(applicaton)に登録するオブジェクト
	 */
	protected void setContextAttribute( final String key ,final Object object ) {
		final ServletContext applicaton = pageContext.getServletContext();
		applicaton.setAttribute( key,object );
	}

	/**
	 * コンテキスト(applicaton)指定のキーで登録されているオブジェクトを 削除します。
	 *
	 * scope属性に、"applicaton" が指定された場合に、実行されます。
	 *
	 * @og.rev 3.0.0.0 (2002/12/25) scope="applicaton" 指定の追加
	 *
	 * @param   key キー
	 */
	protected void removeContextAttribute( final String key ) {
		final ServletContext applicaton = pageContext.getServletContext();
		applicaton.removeAttribute( key );
	}

	/**
	 * アプリケーションサーバーのコンテキストパスのURLを返します。
	 *
	 * @return   コンテキストパス
	 */
	protected String getContextPath() {
		return ((HttpServletRequest)getRequest()).getContextPath();
	}

	/**
	 * スコープに応じて登録されているオブジェクトを取得します。
	 *
	 * @og.rev 3.0.0.0 (2002/12/25) scope="applicaton" 指定の追加
	 *
	 * @param   key キー
	 *
	 * @return   スコープに応じて登録されているオブジェクト
	 */
	protected Object getObject( final String key ) {
		if( "session".equals( scope ) ) { return getSessionAttribute( key ); }
		else if( "request".equals( scope ) ) { return getRequestAttribute( key ); }
		else if( "applicaton".equals( scope ) ) { return getContextAttribute( key ); }
		else {
			final String errMsg = "このスコープはサポートされていません。[" + scope + "]";
			throw new IllegalArgumentException( errMsg );
		}
	}

	/**
	 * スコープに応じて登録されているオブジェクトを指定のキーでセットします。
	 *
	 * @og.rev 3.0.0.0 (2002/12/25) scope="applicaton" 指定の追加
	 *
	 * @param   key キー
	 * @param   object リクエストに登録するオブジェクト
	 * @see     #setObject( String ,Object ,String )
	 */
	protected void setObject( final String key ,final Object object ) {
		setObject( key,object,scope );
	}

	/**
	 * スコープに応じて登録されているオブジェクトを指定のキーでセットします。
	 *
	 * 引数にスコープを指定します。スコープが null の場合は、オリジナルの
	 * スコープを使用します。
	 *
	 * @og.rev 5.2.2.0 (2010/11/01) 新規追加
	 *
	 * @param   key キー
	 * @param   object リクエストに登録するオブジェクト
	 * @param   scp スコープ
	 * @see     #setObject( String ,Object )
	 */
	protected void setObject( final String key ,final Object object ,final String scp ) {
		final String inScp = (scp == null) ? scope : scp ;

		if(      "session".equals( inScp	) ) { setSessionAttribute( key,object ); }
		else if( "request".equals( inScp	) ) { setRequestAttribute( key,object ); }
		else if( "applicaton".equals( inScp ) ) { setContextAttribute( key,object ); }
		else {
			final String errMsg = "このスコープはサポートされていません。[" + inScp + "]";
			throw new IllegalArgumentException( errMsg );
		}
	}

	/**
	 * スコープに応じて登録されているオブジェクトを指定のキーで削除します。
	 *
	 * @og.rev 3.0.0.0 (2002/12/25) scope="applicaton" 指定の追加
	 *
	 * @param   key キー
	 */
	protected void removeObject( final String key ) {
		if( "session".equals( scope ) ) { removeSessionAttribute( key ); }
		else if( "request".equals( scope ) ) { removeRequestAttribute( key ); }
		else if( "applicaton".equals( scope ) ) { removeContextAttribute( key ); }
		else {
			final String errMsg = "このスコープはサポートされていません。[" + scope + "]";
			throw new IllegalArgumentException( errMsg );
		}
	}

	/**
	 * リクエストオブジェクトを取得します。
	 *
	 * @og.rev 2.2.0.0 (2002/12/17) 中国語(国際化)対応 エンコードの取得方法変更
	 * @og.rev 2.2.0.0 (2002/12/17) 文字化け対策 setCharacterEncoding が効いていないので削除
	 * @og.rev 3.1.1.2 (2003/04/04) Tomcat4.1 対応。HybsRequestWrapper 廃止。直接 Mapでキャッシュする。
	 * @og.rev 3.4.0.3 (2003/09/10) 冗長な個所や、無意味な個所を、等価な内容のロジックに置き換える。
	 * @og.rev 3.5.5.0 (2004/03/12) command=RESET 時にも、キャッシュを取り出すように変更します。
	 *
	 * @return   リクエストオブジェクト
	 */
	@SuppressWarnings(value={"unchecked"})
	protected ServletRequest getRequest() {
		if( request == null ) {
			request = pageContext.getRequest();
			// リクエストキャッシュ機能
			final String cmd =request.getParameter( "command" );
			if( "RENEW".equals( cmd ) || "RESET".equals( cmd ) ) {	// 3.5.5.0
				requestCache = (Map<String,String[]>)getSessionAttribute( HybsSystem.REQ_CACHE_KEY );
			}
		}
		return request;
	}

	/**
	 * BodyContent オブジェクトを取得して、ボディの内容を取得します。
	 *
	 * 処理としては、getRequestParameter() によるパラメータ処理も含みます。
	 * このメソッドは、必ず doAfterBody() から呼び出してください。それ以外(例えば、
	 * doEndTag()等)では、すでに Body情報が破棄/再利用されている可能性があり、
	 * 正常に動作しなくなる可能性があります。
	 *
	 * @og.rev 3.1.1.0 (2003/03/28) BodyContent オブジェクトを取得して、ボディの内容を取得する処理を追加
	 *
	 * @return   ボディ文字列
	 */
	protected String getBodyString() {
		final BodyContent body = getBodyContent();
		return getRequestParameter( body.getString() );
	}

	/**
	 * BodyContent オブジェクトを取得して、ボディの内容を取得します。
	 *
	 * {&#064;XXXX}を変換しない生のBODY文を返します
	 *
	 * @og.rev 4.3.6.0 (2009/04/01) 新規作成
	 *
	 * @return   ボディ文字列
	 */
	protected String getBodyRawString() {
		final BodyContent body = getBodyContent();
		return body.getString();
	}

	/**
	 * BodyContent オブジェクトを取得して、ボディの内容を取得します。
	 *
	 * {&#064;XXXX}の変換を行いますが、その変換結果に、"["が含まれる場合は、
	 * "\\]\\"に変換して、フォーマット処理されないようにサニタイズします。
	 *
	 * @og.rev 5.1.7.0 (2010/06/01) 新規作成
	 * @og.rev 5.7.4.2 (2014/03/20) サニタイズ処理を、標準の処理で行う。
	 *
	 * @return   ボディ文字列
	 */
	protected String getSanitizedBodyString() {
		isSanitized = true;
		final String rtn = getBodyString();
		isSanitized = false;				// 一連の処理の中だけ、有効とします。

		return rtn;
	}

	/**
	 * JspWriter を使用した画面出力です。
	 *
	 * @param msg 画面に出力する文字列
	 */
	protected void jspPrint( final String msg ) {
		if( msg == null ) { return ; }
		try {
			final JspWriter out = pageContext.getOut();
			out.print( msg );
		} catch(IOException ex) {
			final String errMsg = "画面出力時の PageContext の取得時にエラーが発生しました。"
							+ ex.getMessage();			// 5.1.8.0 (2010/07/01) errMsg 修正
			throw new HybsSystemException( errMsg,ex );		// 3.5.5.4 (2004/04/15) 引数の並び順変更
		}
	}

	/**
	 * デバッグ用の文字列を JspWriter を使用して画面に出力します。
	 * このメソッドは、debugFlag=true の場合のみ動作します。
	 *
	 * 出力内容は，各オブジェクトの toString() 文字列です。
	 *
	 * @og.rev 4.0.0.0 (2005/02/28) debugFlag の条件式を追加。
	 * @og.rev 4.0.0.0 (2005/02/28) 簡易リファレンスへのリンクを追加。
	 */
	protected void debugPrint() {
		if( debugFlag ) {
			try {
				final JspWriter out = pageContext.getOut();
				out.println( getDocumentLink() );		// 4.0.0 (2005/02/28)
				out.println( "<pre>" );
				out.println( toString() );
				out.println( "</pre>" );
			} catch(IOException ex) {
				final String errMsg = "デバッグ画面出力時の PageContext の取得時にエラーが発生しました。"
							+ ex.getMessage();			// 5.1.8.0 (2010/07/01) errMsg 修正
				throw new HybsSystemException( errMsg,ex );		// 3.5.5.4 (2004/04/15) 引数の並び順変更
			}
		}
	}

	/**
	 * GAMENID付のリクエストキャッシュ情報を取り出します。
	 *
	 * @og.rev 3.5.4.7 (2004/02/06) 新規作成
	 *
	 * @param key リクエストキャッシュのキー情報
	 *
	 * @return リクエスト情報(存在しない場合は、null)
	 */
	protected String getRequestCacheData( final String key ) {
		String rtn = null;

		final String memKey = HybsSystem.REQ_CACHE_KEY + getGUIInfoAttri( "KEY" );	// 4.0.0 (2005/01/31)
		final Map<?,?> mem = (Map<?,?>)getSessionAttribute( memKey );		// 4.3.3.6 (2008/11/15) Generics警告対応

		if( mem != null ) {
			final String[] vals = (String[])mem.get( key );
			if( vals != null && vals.length > 0 ) {
				rtn = vals[0];
			}
		}
		return rtn ;
	}

	/**
	 * GAMENID付のリクエストキャッシュ情報を取り出します。
	 *
	 * @og.rev 3.5.4.7 (2004/02/06) 新規作成
	 *
	 * @param key  リクエストキャッシュのキー情報
	 * @param value リクエストキャッシュに登録する値
	 */
	@SuppressWarnings(value={"unchecked"})
	protected void setRequestCacheData( final String key,final String value ) {
		final String[] vals = new String[] { value } ;

		final String memKey = HybsSystem.REQ_CACHE_KEY + getGUIInfoAttri( "KEY" );	// 4.0.0 (2005/01/31)
		final Map<String,String[]> mem = (Map<String,String[]>)getSessionAttribute( memKey );
		if( mem != null ) {
			mem.put( key,vals );
		}
	}

	/**
	 * CSV形式引数(CSV引数)を配列に分解して返します。
	 *
	 * CSV形式引数(CSV引数)で複数指定されたリクエストパラメータを
	 * 文字列配列に分解して、返します。
	 * 引数は、{&#064;XXXX} 変数も使用できます。
	 * 分解方法は、CSV変数を先に分解してから、getRequestParameter で値を取得します。
	 * こうしないとデータ自身にカンマを持っている場合に分解をミスる為です。
	 *
	 * @og.rev 3.5.6.2 (2004/07/05) 新規作成
	 *
	 * @param	csvKey 引数(CSV形式)
	 *
	 * @return	配列に分解されたリクエストパラメータ値
	 */
	protected String[] getCSVParameter( final String csvKey ) {
		final String[] keys = StringUtil.csv2Array( csvKey );
		String[] vals = new String[keys.length];
		for( int i=0; i<keys.length; i++ ) {
			vals[i] = getRequestParameter( keys[i] ) ;
		}
		return vals ;
	}

	/**
	 * リクエスト変数 {&#064;SYS.XXXX} に対する値の取得を行います。
	 *
	 * 本来は、システムパラメータ の値を取得できますが、
	 * システム的な共有値も取得できるように機能追加しています。
	 * また、ユーザー個別にシステムパラメータを変更できます。この取得は、
	 * システムパラメータとして取得します。(値はユーザー個別値を返します。)
	 * ここで、引数に、&#064;変数が使用できます。具体的には、{&#064;SYS.&#064;XXXX}
	 * で、&#064;XXXX をリクエスト変数から取得した値を利用します。
	 * この中で、&#064;GUIID だけが、さらに特殊で、実行中の画面IDを割り当てます。
	 * この &#064;GUIID は、ここまでの文字列を画面IDに置き換えるとともに、それ以降の
	 * 文字列を、画面IDに連結させます。
	 * {&#064;SYS.&#064;GUIID_XXXX} ⇒ 画面ID_XXXX 文字列で、システムパラメータ の値を取得します。
	 *
	 *	SERVER_NAME     このTomcatが実行しているサーバー名             localhost 等
	 *	SERVER_URL      Portも含むURLアドレス                          http://localhost:8823/
	 *	CONTEXT_URL     実行しているコンテキストのURLアドレス          http://localhost:8823/dbdef2/
	 *	REAL_PATH       / ルートに対応する物理ディレクトリ             d:/webapps/dbdef2/ 等
	 *	CONTEXT_NAME    コンテキスト名(webapps 直下の仮想フォルダ名)   dbdef 等
	 *	DATE            YMDH とほぼ同じですが、'yyyy/MM/dd HH:mm:ss' の形式で取得できます。
	 *	HOSTNAME		スペース区切りで指定したIPアドレスからホスト名を逆引きします
	 *	任意            ユーザーパラメータ(GE16) の値/システムパラメータ(GE12)の値を取得
	 *	  &#064;GUIID_XXXX  既存の画面IDに、_XXXX を追加した文字列
	 *	  &#064;XXXX        XXXX でリクエスト変数から取得した文字列
	 *	  XXXX              XXXX の文字列
	 *
	 * @og.rev 3.5.6.6 (2004/08/23) 新規作成
	 * @og.rev 3.7.0.3 (2005/03/01) クッキー取得機能を追加
	 * @og.rev 4.0.0.0 (2005/11/30) ユーザーパラメータは、システムパラメータとして取得します。
	 * @og.rev 5.1.6.0 (2010/05/01) システムパラメータに、&#064;GUIID という特殊パラメータが使用できるように対応します。
	 * @og.rev 5.6.6.2 (2013/07/19) SYS.HOSTNAMEに対応します。
	 *
	 * @param	key	{&#064;SYS.XXXX}のXXXX部分の文字列(キー)
	 *
	 * @return   キーに対する値。なければ、null
	 */
	protected String sys( final String key ) {
		final String rtn;

		if( key.startsWith( "COOKIE." ) ) {		// 3.7.0.3 (2005/03/01)
			rtn = getCookie( key.substring( "COOKIE.".length() ) );
		}
		else if( key.startsWith( "DATE" ) ) {
			final int idx = key.indexOf( ' ' );
			if( idx >= 0 ) {
				rtn = HybsSystem.getDate( key.substring( idx+1 ) );
			}
			else {
				rtn = HybsSystem.getDate();
			}
		}
		else if( key.startsWith( "HOSTNAME" ) ) { // 5.6.6.2 (2013/07/19)
			final int idx = key.indexOf( ' ' );
			if( idx >= 0 ) {
				final String key2 = key.substring( idx+1 ) ;
				if( key2.startsWith( "@" ) ) {
					rtn = getHostName( getRequestValue( key2.substring( 1 ) ) );
				}
				else{
					rtn = getHostName( key2 );
				}
			}
			else{
				rtn = getUser().getParameter( key );
			}
		}
		// 5.1.6.0 (2010/05/01) {@SYS.@GUIID_XXXX} パラメータ対応
		else if( key.startsWith( "@GUIID" ) ) {
			final String key2 = getGUIInfoAttri( "ID" ) +  key.substring( "@GUIID".length() );
			rtn = getUser().getParameter( key2 );
		}
		// 5.1.6.0 (2010/05/01) {@SYS.@XXXX} パラメータ対応
		else if( key.startsWith( "@" ) ) {
			final String key2 = getRequestValue( key.substring( 1 ) );
			rtn = getUser().getParameter( key2 );
		}
		else {
			rtn = getUser().getParameter( key );
		}

		return rtn ;
	}

	/**
	 * システムパラメータの値を、boolean 型に変換して返します。
	 *
	 * 本来は、システムパラメータ の値を取得できますが、
	 * システム的な共有値も取得できるように機能追加しています。
	 * また、ユーザー個別にシステムパラメータを変更できます。この取得は、
	 * システムパラメータとして取得します。(値はユーザー個別値を返します。)
	 *
	 * @og.rev 4.0.0.0 (2005/11/30) 新規追加
	 *
	 * @param	key システム設定キー
	 *
	 * @return	システム設定値(boolean型)
	 */
	protected boolean sysBool( final String key ) {
//		return Boolean.valueOf( sys( key ) ).booleanValue();
		return Boolean.parseBoolean( sys( key ) );		// 6.1.0.0 (2014/12/26) refactoring
	}

	/**
	 * システムパラメータの値を、int 型に変換して返します。
	 *
	 * 本来は、システムパラメータ の値を取得できますが、
	 * システム的な共有値も取得できるように機能追加しています。
	 * また、ユーザー個別にシステムパラメータを変更できます。この取得は、
	 * システムパラメータとして取得します。(値はユーザー個別値を返します。)
	 *
	 * ※ システムパラメータの値が数字でない場合、HybsSystemException が throw されます。
	 * ※ キーの値が nullの場合、HybsSystemException が throw されます。
	 *
	 * @og.rev 4.0.0.0 (2005/11/30) 新規追加
	 *
	 * @param	key システム設定キー
	 *
	 * @return	システム設定値(int型)
	 */
	protected int sysInt( final String key ) {
		String tmp = null;
		int rtn ;
		try {
			tmp = sys( key );
			rtn = Integer.parseInt( tmp );
		}
		catch( NumberFormatException ex ) {
			final String errMsg = "システムパラメータの値が数字ではありません。" + CR
					+ "  Resource key=[" + key + "] val=[" + tmp + "]"  ;
			throw new HybsSystemException( errMsg,ex );		// 3.5.5.4 (2004/04/15) 引数の並び順変更
		}
		catch( IllegalArgumentException ex ) {
			final String errMsg = "キーの値が null です。key=[" + key + "] val=[" + tmp + "]";
			throw new HybsSystemException( errMsg,ex );		// 3.5.5.4 (2004/04/15) 引数の並び順変更
		}

		return rtn;
	}

	/**
	 * session に、処理開始時刻を設定します。
	 * これは、DBTableModel を登録する場合に、一連の処理が連続であるかどうかを
	 * 判断する時に使用します。
	 * 処理が一連でない(start 時のタイムスタンプが書き換えられている)場合は、
	 * DBTableModel の登録処理を行いません。
	 * なお、判断処理を行うのは、scope が session の場合のみです。
	 * 判定は、commitTableObject( String ,DBTableModel ) で行います。
	 *
	 * @og.rev 3.6.0.8 (2004/11/19) 新規追加
	 * @og.rev 4.3.0.0 (2008/07/04) fileUD 対応。
	 * @og.rev 5.1.6.0 (2010/05/01) DBLastSqlの処理は、DBTableModelが新規作成された処理でのみ行う。
	 *
	 * @param   tableId キー
	 * @see     #commitTableObject( String ,DBTableModel )
	 */
	protected void startQueryTransaction( final String tableId ) {
		if( "session".equals( scope ) ) {
			startTransaction = Long.valueOf( System.currentTimeMillis() );
			setSessionAttribute( tableId+"_TRANSACTION", startTransaction );
		}

		// 5.1.6.0 (2010/05/01) DBLastSqlの処理の見直し
		if( useTrans ) {
			// 4.3.0.0 (2008/07/04) fileUD 対応
			removeSessionAttribute( HybsSystem.DB_LAST_SQL_KEY );	// 無条件削除
		}
	}

	/**
	 * スコープに応じて登録されている DBTableModel を指定のキーでセットします。
	 * これは、startQueryTransaction( String ) でセッションに登録した処理開始時刻と、
	 * このオブジェクト自身が持っている(セッションに登録した開始時刻そのもの)を
	 * 比較し、異なる場合は、DBTableModel の登録を行いません。
	 * これにより、検索処理の開始順にしか登録しないようなロジックを入れています。
	 * 検索処理時間が掛かるSQLを実行した場合、先に検索した結果があとから登録される
	 * ケースがあるためです。
	 * また、判断処理を行うのは、scope が session の場合のみです。
	 *
	 * @og.rev 3.6.0.8 (2004/11/19) 新規追加
	 * @og.rev 3.8.1.1 (2005/11/21) ExcelOut の整合性を取る為の仕掛け
	 * @og.rev 4.3.0.0 (2008/07/04) fileUD 対応。
	 * @og.rev 5.1.6.0 (2010/05/01) DBLastSqlの処理は、DBTableModelが新規作成された処理でのみ行う。
	 *
	 * @param   tableId キー
	 * @param   table   登録するDBTableModelオブジェクト
	 *
	 * @return  正常に登録(true) / 一連でないため登録していない(false)
	 * @see     #startQueryTransaction( String )
	 */
	protected boolean commitTableObject( final String tableId ,final DBTableModel table ) {
		// 登録しないケースをピックアップします。
		if( "session".equals( scope ) ) {
			final String key = tableId+"_TRANSACTION";
			final Long endTime = (Long)getSessionAttribute( key );
			removeSessionAttribute( key );
			if( endTime == null ||
				startTransaction == null ||
				endTime.compareTo( startTransaction ) != 0 ) {
					final String msg = "CommonTagSupport Query処理が割り込まれました。DBTableModel は登録しません。"
								+ "[" + getUser().getUserID() + "],"
								+ "[" + getGUIInfoAttri( "KEY" ) + "]" 	// 4.0.0 (2005/01/31)
								+ "[" + startTransaction + "]" 				// 4.0.0 (2005/01/31)
								+ "[" + endTime + "]"; 						// 4.0.0 (2005/01/31)
					System.out.println( msg );
					return false;
			}
			// 3.8.1.1 (2005/11/21) ExcelOut の整合性を取る為の仕掛け
			if( table != null && HybsSystem.TBL_MDL_KEY.equals( tableId ) ) {
				final String consisKey = table.getConsistencyKey();
				setSessionAttribute( HybsSystem.TBL_MDL_CONKEY,consisKey );
			}
		}

		// 4.3.0.0 (2008/07/04) fileUD 対応
		// 5.1.6.0 (2010/05/01) DBLastSqlの処理の見直し
		if( useTrans && table != null ) {
			final String guikey = getGUIInfoAttri( "KEY" );
			final DBLastSql lastSql = new DBLastSql( scope,guikey,table.isOverflow(),tableId );
			setSessionAttribute( HybsSystem.DB_LAST_SQL_KEY,lastSql );
		}

		setObject( tableId,table );
		return true;
	}

	/**
	 * 表示データの HybsSystem.ROW_SEL_KEY を元に、選ばれた 行番号の
	 * 配列を返します。
	 * 配列情報は、行番号でソートされて返されます。
	 * なにも選ばれていない場合は、サイズ０の配列を返します。
	 *
	 * @og.rev 4.0.0.0 (2005/01/31) 新規追加
	 *
	 * @return	(選ばれていない場合は、サイズ０の配列を返す)
	 * @og.rtnNotNull
	 */
	protected int[] getParameterRows() {
		if( rowNo != null ) { return rowNo; }

		rowNo = (int[])getRequestAttribute( HybsSystem.ROW_SEL_KEY );
		if( rowNo != null ) { return rowNo; }

		final String[] selected = getRequestValues( HybsSystem.ROW_SEL_KEY ) ;
		if( selected != null && selected.length != 0 ) {
			rowNo = new int[ selected.length ];
			for( int i=0; i<selected.length; i++ ) {
				if( selected[i] != null && selected[i].length() > 0 ) {
					rowNo[i] = Integer.parseInt( selected[i] );
				}
			}
			Arrays.sort( rowNo );
		}
		else {
			rowNo = new int[0];
		}

		return rowNo;
	}

	/**
	 * 表示データの HybsSystem.ROW_SEL_KEY に対して、選ばれた 行番号の
	 * 配列を設定します。
	 * ここで設定した選択配列は、getParameterRows() メソッドで取得する場合、優先されます。
	 *
	 * @og.rev 4.0.0.0 (2005/01/31) 新規追加
	 *
	 * @param	rowNo	行番号配列(可変長引数)
	 */
//	protected void setParameterRows( final int[] rowNo ) {
	protected void setParameterRows( final int... rowNo ) {
		setRequestAttribute(  HybsSystem.ROW_SEL_KEY , rowNo );
	}

	/**
	 * 指定のクッキーをセットします。
	 * これは、従来，各Taglibパッケージで使用していました、ErrorMessage オブジェクトを、
	 * HTMLテーブル形式で表示する為に、DBUtilクラスや、ViewFormクラスなど、複数のクラスを
	 * 複雑に組み合わせて使用していましたが、一つの static メソッドにまとめたものです。
	 *
	 * @og.rev 3.7.0.3 (2005/03/01) 新規登録
	 *
	 * @param   key		クッキーのキー
	 * @param   value	クッキーの設定値
	 * @param	maxage	最長存続期間を秒単位で設定 (負の値は Cookie を保存しない、 0 なら Cookie を削除する)
	 */
	protected void setCookie( final String key,final String value,final int maxage ) {
		final HttpServletResponse res = (HttpServletResponse)pageContext.getResponse();
		final Cookie ck = new Cookie( key, value );
		ck.setMaxAge( maxage );		// 有効秒
		res.addCookie( ck );
	}

	/**
	 * 指定のクッキーを取得します。
	 * 見つからない場合は、null を返します。
	 *
	 * @og.rev 3.7.0.3 (2005/03/01) 新規登録
	 *
	 * @param	key 	クッキーのキー
	 *
	 * @return	クッキーの設定値
	 */
	protected String getCookie( final String key ) {
		final HttpServletRequest req = (HttpServletRequest)pageContext.getRequest();
		final Cookie[] cks = req.getCookies();

		String val = null;
		for( int i=0; i<cks.length; i++ ) {
			final Cookie ck = cks[i];
			if( ck.getName().equals( key ) ) {
				val = ck.getValue();
				break;
			}
		}
		return val ;
	}

	/**
	 * リクエスト情報の クォーティション(') 存在チェックを実施するかどうか[true/false]を設定します
	 *		(初期値:USE_SQL_INJECTION_CHECK[={@og.value org.opengion.hayabusa.common.SystemData#USE_SQL_INJECTION_CHECK}])。
	 *
	 * ＳＱＬインジェクション対策の一つとして、暫定的ではありますが、SQLのパラメータに
	 * 渡す文字列にクォーティション(') を許さない設定にすれば、ある程度は防止できます。
	 * 数字タイプの引数には、 or 5=5 などのクォーティションを使用しないコードを埋めても、
	 * 数字チェックで検出可能です。文字タイプの場合は、必ず (')をはずして、
	 * ' or 'A' like 'A のような形式になる為、(')チェックだけでも有効です。
	 * (') が含まれていたエラーにする(true)／かノーチェックか(false)を指定します。
	 * (初期値:システム定数のUSE_SQL_INJECTION_CHECK[={@og.value org.opengion.hayabusa.common.SystemData#USE_SQL_INJECTION_CHECK}])。
	 *
	 * @og.rev 4.0.0.0 (2005/08/31) 新規追加
	 *
	 * @param	flag	クォーティションチェック  [true:する/:falseしない]
	 * @see		org.opengion.hayabusa.common.SystemData#USE_SQL_INJECTION_CHECK
	 */
	protected void useQuotCheck( final boolean flag ) {
		quotCheck = flag;
	}

	/**
	 * リクエスト情報の HTMLTag開始/終了文字(&gt;&lt;) 存在チェックを実施するかどうか[true/false]を設定します
	 *		(初期値:USE_XSS_CHECK[={@og.value org.opengion.hayabusa.common.SystemData#USE_XSS_CHECK}])。
	 *
	 * クロスサイトスクリプティング(XSS)対策の一環としてless/greater than signについてのチェックを行います。
	 * (&gt;&lt;) が含まれていたエラーにする(true)／かノーチェックか(false)を指定します。
	 * 現在の実装としてはリクエストパラメータのみチェックして、attributesに対しては行いません。
	 * (初期値:システム定数のUSE_XSS_CHECK[={@og.value org.opengion.hayabusa.common.SystemData#USE_XSS_CHECK}])。
	 *
	 * @og.rev 5.0.0.2 (2009/09/15) 新規追加
	 *
	 * @param	flag	XSSチェック [true:する/false:しない]
	 * @see		org.opengion.hayabusa.common.SystemData#USE_XSS_CHECK
	 */
	protected void useXssCheck( final boolean flag ) {
		xssCheck = flag;
	}

	/**
	 * 日付関係の情報を簡易的に取り出す処理を行います。
	 *
	 * これは、{&#064;DATE.XXXX AA BB CC} 引数処理をおこなうための、サポートメソッドです。
	 * XXXX は結果のフォーマット、AA が基準時刻で省略した場合は、現在時刻が利用されます。
	 * BB 引数は、日付についての加減算処理を行うためのコマンドです。
	 * CC 引数は、BB引数のコマンドに付属するパラメータです。加減算処理の数値を指定できます。
	 * AA,BB,CC 引数については、先頭に、@ を付ける事で、リクエスト変数が使用できます。
	 *
	 * 引数は、"XXXX AA BB" という状態で受け取ります。(DATE. ははずした形)
	 * "XXXX" は、日付処理を行うキー文字列で予約語になっています。
	 * ・YMD　　：８文字の4-2-2年月日データ(yyyyMMdd)を扱います。
	 * ・Y2MD　 ：６文字の2-2-2年月日データ(yyMMdd)を扱います。
	 * ・YM　　 ：６文字の4-2年月データ(yyyyMM)を扱います。
	 * ・HMS　　：６文字の2-2-2時分秒データ(HHmmss)を扱います。
	 * ・YMDHMS ：１４文字の4-2-2-2-2-2年月日時分秒データ(yyyyMMddHHmmss)を扱います。
	 * ・EEE　　：曜日をデフォルトロケール(EEE)で表示します。
	 *
	 * F付きは、フォーマットされた日付を返します。
	 * ・YMDF　 ：１０文字の日付表現(yyyy/MM/dd)を扱います。
	 * ・Y2MDF　：８文字の日付表現(yy/MM/dd)を扱います。
	 * ・YMF　　：７文字の日付表現(yyyy/MM)を扱います。
	 * ・HMSF　 ：８文字の時刻表現(HH:mm:ss)を扱います。
	 * ・YMDHMSF：１９文字の日付表現(yyyy/MM/dd HH:mm:ss)を扱います。
	 * ・MDF　　：５文字の月日表現(MM/dd)を扱います。
	 * ・MDEF　 ：５文字＋曜日の月日表現(MM/dd(EEE))を扱います。
	 * ・MD2F　 ：和暦の月日表現(MM月dd日)を扱います。(5.5.5.2 追加)
	 * ・GYMDF　：和暦の年月日表現(GGGGyyyy年MM月dd日)を扱います。
	 * ・G2YMDF ：和暦の日付表現(Gyyyy/MM/dd)を扱います。
	 * ・GYMF　 ：和暦の年月表現(GGGGyyyy年MM月)を扱います。
	 * ・GYF　　：和暦の年表現(GGGGyyyy)を扱います。
	 *
	 * なお、上記以外のフォーマットを指定する場合は、XXXX部分に直接記述できます。(5.5.5.2 追加)
	 * ただし、スペースで分解するため、フォーマットにスペースを含む場合は、ダブルコーテーション等で
	 * くくる等の処理が必要です。基本的には、自由フォーマットは、エラーチェックがない為、使わないでください。
	 *
	 * 第二引数 AA は、基準となる日付を、yyyyMMdd形式で指定します。nullの場合は、現在時刻を使用します。
	 * 指定できる日付は、yyyyMMdd形式を推奨しますが、'/' , '-' , ' ' , ':' を削除して使います。
	 * ６桁の場合は、yyyyMM + 01 とし、８ケタの場合は、yyyyMMdd とし、14ケタ以上の場合は、前半１４文字を
	 * yyyyMMddHHmmss として処理します。それ以外の桁数の場合は、エラーになります。
	 * たとえば、"2012/09/05 16:52:36" のようなフォーマットデータの場合、'/' , '-' , ' ' , ':' を削除して
	 * "20120905165236" に変換後、日付オブジェクトに変換されます。
	 *
	 * AA には、数字で始まる(20050701など)実日付と&#064;で始まるパラメータが使用できます。
	 * &#064;AA と記述することで、{&#064;AA}で指定する場合と同様のリクエストパラメータが使用できます。
	 * このパラメータの値の解析結果が、null の場合は、現在時刻が使用されます。
	 * 数字以外の場合は、省略されたと判断して、コマンド（BB引数）として使用されます。
	 *
	 * BB 引数は、日付についての加減算処理を行います。
	 * 省略すると、なにも加減算処理を行いません。
	 * この引数もパラメータ(&#064;BB)指定で、リクエストパラメータが使用できます。
	 * 加減算処理のﾊﾟﾗﾒｰﾀが使用できるのは、"H" , "D" , "M" の１文字パラメータの場合のみです。
	 * それ以外のコマンドで、加減算処理する場合は、独立した CC 引数 を使用してください。
	 * ・SD ：当月の最初の日付にセットします。(当月１日)。CC引数は、-N:N月前、0:当月(=SD)、N:N月後、-1:BSD と同じ、1:ASD と同じ
	 * ・ED ：当月の最後の日付にセットします。(当月月末)。CC引数は、-N:N月前、0:当月(=ED)、N:N月後、-1:BED と同じ、1:AED と同じ
	 * ・SW ：日付処理の週初め(月曜日)にセットします。日付は当日より前に移動します。CC引数は、-N:N週前、0:今週(=SW)、N:N週後
	 * ・EW ：日付処理の週末(日曜日)にセットします。日付は当日より後ろに移動します。CC引数は、-N:N週前、0:今週(=EW)、N:N週後
	 * ・H1 ～ HXXX ：時を指定の分だけ進めます。H1なら１時間後、H24 なら２４時間後(5.5.5.6 (2012/08/31) 追加)
	 * ・D1 ～ DXXX ：日を指定の分だけ進めます。D1なら翌日、D200 なら200日後
	 * ・M1 ～ MXXX ：月を指定の分だけ進めます。M1なら翌月、M6 なら半年後
	 * ・BSD ：(有閑)先月の最初の日付にセットします。(先月１日)(5.5.5.2 追加)。SD -1 と同等
	 * ・BED ：(有閑)先月の最後の日付にセットします。(先月月末)(5.5.5.2 追加)。ED -1 と同等
	 * ・ASD ：(有閑)翌月の最初の日付にセットします。(翌月１日)(5.5.5.2 追加)。SD 1  と同等
	 * ・AED ：(有閑)翌月の最後の日付にセットします。(翌月月末)(5.5.5.2 追加)。ED 1  と同等
	 *
	 * CC 引数は、特別な処理で、BB 引数に対して、加算、減算のための数字を指定できます。(5.7.4.1 (2014/03/14) 追加)
	 * 従来は、BB 引数が、"H" , "D" , "M" の １文字パラメータの場合のみ利用可能でした。
	 * これは、"H15" と指定するのと、"H" "15" と指定するのと同じ意味になります。
	 * 異なるのは、CC 引数も、(&#064;CC)指定で、リクエストパラメータが使用できます。
	 * 従来は、文字列として結合された状態でしか、BB 引数を渡せませんでしたが、この、CC 引数の
	 * 追加で、日付の加減算を、パラメータ指定できるようになります。
	 * 数字以外の文字が指定されたり、パラメータの解析結果が NULL の場合には、BB引数自体も無視されます。
	 * 注意点は、各 BB 引数に応じて、数字の意味が異なるという事です。
	 *
	 * HXXX,DXXX,MXXX 形式に、CC 引数を付けた場合は、XXX にさらに加算されます。
	 * prmB に、数字を使用した場合、(コマンドでない場合)にも、CC 引数は、加算されます。
	 *
	 * @og.rev 3.8.0.2 (2005/07/11) 新規追加
	 * @og.rev 5.4.0.1 (2011/11/01) 日付処理の機能追加(BB 引数に、リクエストパラメータ対応)
	 * @og.rev 5.5.0.2 (2012/03/09) 和暦対応
	 * @og.rev 5.5.5.2 (2012/08/18) XXXXフォーマット追加、自由フォーマット対応、BB引数追加、/,-削除機能追加、SM,EM廃止
	 * @og.rev 5.5.5.6 (2012/08/31) H1 ～ HXXX 追加。時間の加算を指定できる。
	 * @og.rev 5.5.7.2 (2012/10/09) HybsDateUtil を利用するように修正します。
	 * @og.rev 5.5.8.2 (2012/11/09) prmA の判定に、null と ゼロ文字列を判定する。
	 * @og.rev 5.6.0.1 (2013/01/11) 5.5.7.2でyyyyMMddしか取っていないため、HHmmssを追加します
	 * @og.rev 5.7.4.1 (2014/03/14) AA 引数の@解析後のコマンド判定方法を、８ケタ以下から先頭が数字以外に変更します。
	 * @og.rev 5.7.4.1 (2014/03/14) CC 引数を、"H" , "D" , "M" 以外でも使用できるように拡張します。
	 *
	 * @param   value	パラメータ
	 *
	 * @return   メッセージ情報
	 * @see		org.opengion.fukurou.util.HybsDateUtil#getDateFormat( String,String,String,int )
	 */
	protected String getDateFormat( final String value ) {
		// {@DATE.XXXX AA BB CC} を分割
		final String[] vals = StringUtil.csv2Array( value,' ' );		// ダブルクオート内は保持される。

		final String key = vals[0] ;

		// 5.7.4.1 (2014/03/14) 初期化時に、vals を設定しておきます。
		String prmA = (vals.length >= 2) ? vals[1] : null ;
		String prmB = (vals.length >= 3) ? vals[2] : null ;
		String prmC = (vals.length >= 4) ? vals[vals.length-1] : null ;		// 互換性。最後の値が、CC引数

//		if( prmA != null && prmA.startsWith( "@" ) ) {
		if( prmA != null && prmA.length() > 0 && prmA.charAt(0) == '@' ) {	// 6.1.0.0 (2014/12/26) refactoring
			prmA = getRequestValue( prmA.substring(1) );
		}

//		if( prmB != null && prmB.startsWith( "@" ) ) {
		if( prmB != null && prmB.length() > 0 && prmB.charAt(0) == '@' ) {	// 6.1.0.0 (2014/12/26) refactoring
			prmB = getRequestValue( prmB.substring(1) );
		}

//		if( prmC != null && prmC.startsWith( "@" ) ) {
		if( prmC != null && prmC.length() > 0 && prmC.charAt(0) == '@' ) {	// 6.1.0.0 (2014/12/26) refactoring
			prmC = getRequestValue( prmC.substring(1) );
		}

		// 5.7.4.1 (2014/03/14) AA 引数の@解析後のコマンド判定方法を、８ケタ以下から先頭が数字以外に変更します。
		if( prmA != null && prmA.length() > 0 ) {
			final char chA = prmA.charAt(0);
			if( chA < '0' || chA > '9' ) {		// 先頭が、数字以外の場合は、コマンドなので、一つずつずらす。
				prmC = prmB;
				prmB = prmA;
				prmA = null;
			}
		}

		// 5.7.4.1 (2014/03/14) CC 引数を、"H" , "D" , "M" 以外でも使用できるように拡張します。
		int intC = 0;
		if( prmC != null && prmC.length() > 0 ) {
			try { 
				intC = Integer.parseInt( prmC );
			}
			catch( NumberFormatException ex ) {
				final String errMsg = "CC引数が数字ではありません。value=[" + value + "]" 
								+ ex.getMessage() ;
				System.err.println( errMsg );
			}
		}

		// prmA が null か、isEmpty() の場合は、現在時刻が使用される。
		return HybsDateUtil.getDateFormat( key,prmA,prmB,intC );	// 5.7.4.1 (2014/03/14) CC 引数を拡張します。
	}

	/**
	 * debug や エラー時に参考にする、簡易リファレンスへのリンクを作成します。
	 * リンクを行うタグの名称は、getTagName() メソッドより取得します。
	 *
	 * @og.rev 4.0.0.0 (2005/01/31) 新規追加
	 * @og.rev 4.2.1.0 (2008/04/11) URLを相対パスに変更
	 *
	 * @return  簡易リファレンスへのリンク
	 * @og.rtnNotNull
	 * @see  #getTagName()
	 */
	protected String getDocumentLink() {
		final String name    = getTagName();

	//	try {
	//		Field fld = getClass().getDeclaredField( "VERSION" ) ;
	//		version = (String)fld.get( null );
	//	}
	//	catch( Exception ex ) {
	//		version = ex.toString();
	//	}

		// org.opengion.hayabusa.taglib.AbcdTag というクラス名より、abcd を取り出す。
		final StringBuilder buf = new StringBuilder( BUFFER_MIDDLE )
			.append( "<br /><a href=\"../common/documents.html#VAL_" )
			.append( name )
			.append( "\" target=\"_brank\" >簡易リファレンス[" )
			.append( name )
			.append( "] " )
		//	.append( version )
			.append( "</a>" ).append( BR );

		return buf.toString();
	}

	/**
	 * タグの名称を、返します。
	 * これは、debug や エラー時に参考にする、簡易リファレンスへのリンクを作成する場合に
	 * 使用します。
	 * 通常は、org.opengion.hayabusa.taglib.AbcdTag という自分自身のクラス名より、
	 * abcd の部分を取り出し、返します。
	 * クラス名とタグ名が、上記変換ルールと異なる場合は、このメソッドを
	 * 使用して、直接 abcd の部分に相当する文字列を返すようにしてください。
	 *
	 * @og.rev 4.0.0.0 (2005/01/31) 新規追加
	 *
	 * @return  タグの名称
	 * @og.rtnNotNull
	 * @see  #getDocumentLink()
	 */
	protected String getTagName() {
		final String name = getClass().getName();
		final int adrs = name.lastIndexOf('.');

		// org.opengion.hayabusa.taglib.AbcdTag というクラス名より、abcd を取り出す。
		return name.substring( adrs+1,adrs+2 ).toLowerCase(Locale.JAPAN)
				+ name.substring( adrs+2,name.length()-3 ) ;
	}

	/**
	 * リクエストに対して、画面遷移なしモードを有効にします[true/false]。
	 * この情報は画面IDをキーにセッションに保存されるため、
	 * 各タグで共有することができます。
	 *
	 * @og.rev 4.3.3.0 (2008/10/01) 新規作成
	 * @og.rev 4.3.8.0 (2009/08/01) リクエストからセッションに変更(名称も変更)
	 * @og.rev 5.1.3.0 (2010/02/01) noTransition、ajaxSubmitのコントロールは、requestで行う。
	 *
	 * @param	flag	画面遷移なしモード [true:有効/false:無効]
	 */
	protected void setNoTransitionRequest( final boolean flag ) {
		setRequestAttribute( HybsSystem.NO_TRANSITION_MODE_KEY, String.valueOf( flag ));
	}

	/**
	 * リクエストで画面遷移なしモードが有効になっているかを返します。
	 * この情報はセッションから画面IDをキーに取得します。
	 * セッションに情報が設定されていない(#setNoTransition()が呼ばれていない)場合は、
	 * falseを返します。
	 *
	 * @og.rev 4.3.3.0 (2008/10/01) 新規作成
	 * @og.rev 4.3.8.0 (2009/08/01) リクエストからセッションに変更
	 * @og.rev 5.1.3.0 (2010/02/01) noTransition、ajaxSubmitのコントロールは、requestで行う。
	 *
	 * @return 画面遷移なしモードが有効
	 */
	protected boolean isNoTransitionRequest() {
		return nval( (String)getRequestAttribute( HybsSystem.NO_TRANSITION_MODE_KEY ), false );
	}

	/**
	 * リクエストに対して、AjaxSubmitモードを使用するかどうか指定します[true/false]。
	 * この情報は画面IDをキーにセッションに保存されるため、
	 * 各タグで共有することができます。
	 *
	 * @og.rev 4.3.8.0 (2009/08/01) 新規作成
	 * @og.rev 5.1.3.0 (2010/02/01) noTransition、ajaxSubmitのコントロールは、requestで行う。
	 *
	 * @param	flag	AjaxSubmitモード [true:使用する/false:使用しない]
	 */
	protected void setAjaxSubmitRequest( final boolean flag ) {
		setRequestAttribute( HybsSystem.USE_AJAX_SUBMIT_KEY, String.valueOf( flag ));
	}

	/**
	 * リクエストでAjaxSubmitモードが有効になっているかを返します。
	 * この情報はセッションから画面IDをキーに取得します。
	 * セッションに情報が設定されていない(#setNoTransition()が呼ばれていない)場合は、
	 * falseを返します。
	 *
	 * @og.rev 4.3.8.0 (2009/08/01) 新規作成
	 * @og.rev 5.1.3.0 (2010/02/01) noTransition、ajaxSubmitのコントロールは、requestで行う。
	 *
	 * @return AjaxSubmitモードが有効
	 */
	protected boolean isAjaxSubmitRequest() {
		return nval( (String)getRequestAttribute( HybsSystem.USE_AJAX_SUBMIT_KEY ), false );
	}

	/**
	 * シリアライズ用のカスタムシリアライズ書き込みメソッド。
	 *
	 * @og.rev 4.0.0.0 (2006/09/31) 新規追加
	 * @serialData 一部のオブジェクトは、シリアライズされません。
	 *
	 * @param	strm	ObjectOutputStreamオブジェクト
	 */
	private void writeObject( final ObjectOutputStream strm ) throws IOException {
		strm.defaultWriteObject();
	}

	/**
	 * シリアライズ用のカスタムシリアライズ読み込みメソッド
	 *
	 * ここでは、transient 宣言された内部変数の内、初期化が必要なフィールドのみ設定します。
	 *
	 * @og.rev 4.0.0.0 (2006/09/31) 新規追加
	 * @serialData 一部のオブジェクトは、シリアライズされません。
	 *
	 * @param	strm	ObjectInputStreamオブジェクト
	 * @see #release2()
	 */
	private void readObject( final ObjectInputStream strm ) throws IOException , ClassNotFoundException {
		strm.defaultReadObject();
		attri = new Attributes();
	}

	/**
	 * アクセスログ取得の為,ApplicationInfoオブジェクトを返します。
	 * 見つからない場合は、null が返ります。(暫定対応)
	 *
	 * @og.rev 3.8.7.0 (2006/12/15) 新規追加
	 *
	 * @return	アクセスログ取得の為の管理オブジェクト
	 */
	protected ApplicationInfo getApplicationInfo() {
		final String gamenId = getGUIInfoAttri( "KEY" );
		final String jspId   = (String)getSessionAttribute( "JSPID" );

		return getUser().getApplicationInfo( gamenId,jspId );
	}

	/**
	 * イベントカラムの実行に必要なカラム情報をマップに登録します。
	 *
	 * @og.rev 5.1.7.0 (2010/06/01) 動的プルダウン実装見直し
	 *
	 * @param col DBカラム
	 */
	protected void addEventColumn( final DBColumn col ) {
		addEventColumn( col.getName(), col.getEventColumn(), col.getEventURL()
						, col.getRenderer(), col.getEditor(), col.getRawRendParam(), col.getRawEditParam() );
	}

	/**
	 * イベントカラムの実行に必要なカラム情報をマップに登録します。
	 *
	 * @og.rev 5.1.7.0 (2010/06/01) 動的プルダウン実装見直し
	 *
	 * @param name カラム名
	 * @param evCol イベントカラム名
	 * @param url イベントURL
	 * @param renderer カラムのレンデラー
	 * @param editor カラムのエディター
	 * @param rendParam カラムの表示パラメーター
	 * @param editParam カラムの編集パラメーター
	 */
	@SuppressWarnings(value={"unchecked"})
	protected void addEventColumn( final String name, final String evCol, final String url, final String renderer, final String editor, final String rendParam, final String editParam ) {
		if( evCol == null || evCol.isEmpty() ){ return; }

		String key = (String)getRequestAttribute( HybsSystem.USE_EVENT_COLUMN_KEY );
		if( key == null || key.isEmpty() ) {
			key = HybsSystem.USE_EVENT_COLUMN_KEY + System.currentTimeMillis();
			setRequestAttribute( HybsSystem.USE_EVENT_COLUMN_KEY, key );
		}

		Map<String, DBEventColumn> evColMap = (Map<String, DBEventColumn>)( getSessionAttribute( key ) );
		if( evColMap == null ){
			evColMap = new HashMap<String,DBEventColumn>();
		}
		if( evColMap.get( name ) == null ) {
			evColMap.put( name, new DBEventColumn( name, evCol, url, renderer, editor, rendParam, editParam ) );
		}
		setSessionAttribute( key, evColMap );
	}

	/**
	 * 各データベースに対応するファンクション名を返します。
	 *
	 * @og.rev 4.3.7.0 (2009/06/01) 新規作成
	 * @og.rev 5.1.4.0 (2010/03/01) データベース名 でなく、DBID 名で検索するようにします。
	 *
	 * @param   key ファンクション名(定義文字)
	 *
	 * @return  実ファンクション名
	 */
	protected String getDBFunctionName( final String key ) {
		final int idx = key.indexOf( ' ' );
		String func = null;
		String dbid = null;
		if( idx >= 0 ) {
			func = key.substring( 0, idx );
			dbid = key.substring( idx+1, key.length() );
			if( dbid.length() > 0 && dbid.charAt(0) == '@' ) { dbid = getRequestValue( dbid.substring( 1 ) ); }
		}
		else {
			func = key;
		}
		// 5.1.4.0 (2010/03/01) データベース名 でなく、DBID 名で検索する。
		return DBFunctionName.getFunctionName( func, dbid );
	}

	/**
	 * データロールの設定に基づき、キー(カラム名)に対応する条件式を返します。
	 *
	 * @og.rev 4.4.0.0 (2009/08/02) 新規作成
	 *
	 * @param   key カラム名
	 *
	 * @return  条件式
	 */
	protected String getDataCondition( final String key ) {
		return getUser().getDataCondition( key );
	}

	/**
	 * 与えたIPアドレスからホスト名を取得して返します。
	 * 取得できなかった場合は空文字を返します。
	 * IPアドレスが解決できない場合や、セキュリティマネージャーで
	 * 許可されていない場合はエラーを返します。
	 *
	 * @og.rev 5.6.6.2 (2013/07/19) 新規作成
	 *
	 * @param   ip IPアドレス
	 *
	 * @return  ホスト名（取得できない場合は空文字）
	 */
	protected String getHostName( final String ip ) {
		String hostname = "";
		try{
			hostname =  InetAddress.getByName(ip).getHostName();
		}
		catch(UnknownHostException ex){
			final String errMsg = "IPアドレスが解決できません。"
				+ ex.getMessage();			
			throw new HybsSystemException( errMsg,ex );		
		}
		catch(SecurityException ex){
			final String errMsg = "この操作はセキュリティ上許可されていません。"
				+ ex.getMessage();			
			throw new HybsSystemException( errMsg,ex );
		}
		return hostname;
	}

	/**
	 * このオブジェクトの文字列表現を返します。
	 * 基本的にデバッグ目的に使用します。
	 *
	 * @return このクラスの文字列表現
	 * @og.rtnNotNull
	 */
	@Override
	public String toString() {
		return ToString.title( this.getClass().getName() )
				.println( "Other..."	,getAttributes().getAttribute() )
				.fixForm().toString() ;
	}
}
