/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.resource.GUIInfo;
import org.opengion.fukurou.db.Transaction;
import org.opengion.fukurou.db.TransactionReal;
import org.opengion.fukurou.util.StringUtil;
import org.opengion.fukurou.util.ToString;						// 6.1.1.0 (2015/01/17)
import org.opengion.fukurou.xml.XMLFileLoader;				// 6.0.0.0 (2014/04/11) XMLFileLoader を使う様に変更

import static org.opengion.fukurou.util.StringUtil.nval ;

import java.sql.Connection;

import java.io.File;
import java.io.StringWriter;								// 6.0.0.0 (2014/04/11) XMLFileLoader に渡す Log
import java.util.Map;
import java.util.HashMap;
import java.util.Arrays;									// 6.0.0.0 (2014/04/11) keys,vals のエラーメッセージ作成用

/**
 * 指定の拡張XDK形式ファイルを直接データベースに登録するデータ入力タグです。
 *
 * このクラスは、オラクル XDKの oracle.xml.sql.dml.OracleXMLSave クラスと
 * ほぼ同様の目的で使用できる org.opengion.fukurou.xml.HybsXMLSave のラッパークラスです。
 * 
 * 拡張XDK形式のXMLファイルを読み込み、データベースに INSERT します。
 * 拡張XDK形式の元となる オラクル XDK(Oracle XML Developer's Kit)については、以下の
 * リンクを参照願います。
 * <a href="http://otn.oracle.co.jp/software/tech/xml/xdk/index.html" target="_blank" >
 * XDK(Oracle XML Developer's Kit)</a>
 *
 * このタグでは、keys,vals を登録することにより、XMLファイルに存在しないカラムを
 * 追加したり、XMLファイルの情報を書き換えることが可能になります。
 * 例えば、登録日や、登録者、または、テンプレートより各システムID毎に
 * 登録するなどです。
 *
 * 拡張XDK形式とは、ROW 以外に、SQL処理用タグ(EXEC_SQL)を持つ XML ファイルです。
 * また、登録するテーブル(table)を ROWSETタグの属性情報として付与することができます。
 * (大文字小文字に注意)
 * これは、オラクルXDKで処理する場合、無視されますので、同様に扱うことが出来ます。
 * この、EXEC_SQL は、それそれの XMLデータをデータベースに登録する際に、
 * SQL処理を自動的に流す為の、SQL文を記載します。
 * この処理は、イベント毎に実行される為、その配置順は重要です。
 * このタグは、複数記述することも出来ますが、BODY部には、１つのSQL文のみ記述します。
 *
 * 6.0.0.0 (2014/04/11)
 *   指定のファイルがフォルダの場合は、以下のファイルすべて(拡張子はxml)を対象に読込-登録します。
 *   また、拡張子が、zip の場合は、zip内部の xml ファイルを読込-登録します。
 *
 * ※ このタグは、Transaction タグの対象です。
 *
 *   &lt;ROWSET tableName="XX" &gt;
 *       &lt;EXEC_SQL&gt;                    最初に記載して、初期処理(データクリア等)を実行させる。
 *           delete from GEXX where YYYYY
 *       &lt;/EXEC_SQL&gt;
 *       &lt;MERGE_SQL&gt;                   このSQL文で UPDATEして、結果が０件ならINSERTを行います。
 *           update GEXX set AA=[AA] , BB=[BB] where CC=[CC]
 *       &lt;/MERGE_SQL&gt;
 *       &lt;ROW num="1"&gt;
 *           &lt;カラム1&gt;値1&lt;/カラム1&gt;
 *             ･･･
 *           &lt;カラムn&gt;値n&lt;/カラムn&gt;
 *       &lt;/ROW&gt;
 *        ･･･
 *       &lt;ROW num="n"&gt;
 *          ･･･
 *       &lt;/ROW&gt;
 *       &lt;EXEC_SQL&gt;                    最後に記載して、項目の設定(整合性登録)を行う。
 *           update GEXX set AA='XX' , BB='XX' where YYYYY
 *       &lt;/EXEC_SQL&gt;
 *   &lt;ROWSET&gt;
 *
 * @og.formSample
 * ●形式：&lt;og:directXMLSave fileURL="[･･･]" ･･･ /&gt;
 * ●body：なし
 *
 * ●Tag定義：
 *   &lt;og:directXMLSave
 *       dbid               【TAG】(通常は使いません)検索時のDB接続IDを指定します(初期値:DEFAULT)
 *       fileURL            【TAG】読み取り元ディレクトリ名を指定します (初期値:FILE_URL[=filetemp/])
 *       filename           【TAG】ファイルを作成するときのファイル名をセットします (初期値:null)
 *       displayMsg         【TAG】query の結果を画面上に表示するメッセージIDを指定します(初期値:MSG0003[ファイルの登録が完了しました。])
 *       keys               【TAG】XMLファイルを読み取った後で指定するキーをCSV形式で複数指定します
 *       vals               【TAG】XMLファイルを読み取った後で指定する値をCSV形式で複数指定します
 *       useTimeStamp       【TAG】XMLファイルの読み取りで、タイムスタンプ管理を行うかどうか[true:行う/false:行わない]指定します(初期値:false)
 *       caseKey            【TAG】このタグ自体を利用するかどうかの条件キーを指定します(初期値:null)
 *       caseVal            【TAG】このタグ自体を利用するかどうかの条件値を指定します(初期値:null)
 *       caseNN             【TAG】指定の値が、null/ゼロ文字列 でない場合(Not Null=NN)は、このタグは使用されます(初期値:true)
 *       caseNull           【TAG】指定の値が、null/ゼロ文字列 の場合は、このタグは使用されます(初期値:true)
 *       debug              【TAG】デバッグ情報を出力するかどうか[true/false]を指定します(初期値:false)
 *   /&gt;
 *
 * ●使用例
 *     &lt;og:directXMLSave
 *         dbid         = "ORCL"                接続データベースID(初期値:DEFAULT)
 *         fileURL      = "{&#064;USER.ID}"     読み取り元ディレクトリ名
 *         filename     = "{&#064;filename}"    読み取り元ファイル名
 *         displayMsg   = "MSG0003"             登録完了後のメッセージ
 *     /&gt;
 *
 * @og.group ファイル入力
 * @og.rev 4.0.0.0 (2007/03/08) 新規追加
 * @og.rev 6.0.0.0 (2014/04/11) 単体ファイル以外（フォルダ、ZIPファイル）への対応
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class DirectXMLSaveTag extends CommonTagSupport {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "6.0.2.0 (2014/09/19)" ;

	private static final long serialVersionUID = 602020140919L ;

	private String	dbid		;
	private String	fileURL		= HybsSystem.sys( "FILE_URL" );			// 4.0.0 (2005/01/31)
	private String	filename	;										// 6.0.0.0 (2014/04/11) 初期値:null
	private String	displayMsg	= "MSG0003";	// ファイルの登録が完了しました。
	private String[]	keys	;
	private String[]	vals	;
	private long	dyStart		;				// 実行時間測定用のDIV要素を出力します。
	private boolean	useTimeStamp;				// 6.0.2.0 (2014/09/19) タイムスタンプ管理を行うかどうか(true:行う/false:行わない)

	/**
	 * Taglibの開始タグが見つかったときに処理する doStartTag() を オーバーライドします。
	 *
	 * @og.rev 5.6.6.1 (2013/07/12) caseKey 、caseVal 属性対応
	 *
	 * @return	後続処理の指示( SKIP_BODY )
	 */
	@Override
	public int doStartTag() {
		// 5.6.6.1 (2013/07/12) caseKey 、caseVal 属性対応
		if( useTag() ) {
			dyStart = System.currentTimeMillis();		// 時間測定用
		}
		return SKIP_BODY ;	// Body を評価しない
	}

	/**
	 * Taglibの終了タグが見つかったときに処理する doEndTag() を オーバーライドします。
	 *
	 * @og.rev 4.0.0.0 (2007/10/18) メッセージリソース統合( getResource().getMessage ⇒ getResource().getLabel )
	 * @og.rev 4.0.0.1 (2007/12/03) try ～ catch ～ finally をきちんと行う。
	 * @og.rev 5.1.9.0 (2010/08/01) Transaction 対応
	 * @og.rev 5.3.7.0 (2011/07/01) TransactionReal の引数変更
	 * @og.rev 5.5.2.6 (2012/05/25) findbugs対応。例外経路で null 値を利用することが保証されています。
	 * @og.rev 5.6.6.1 (2013/07/12) caseKey 、caseVal 属性対応
	 * @og.rev 5.6.7.0 (2013/07/27) DDL（データ定義言語：Data Definition Language）の処理件数追加
	 * @og.rev 6.0.2.0 (2014/09/19) useTimeStamp 属性追加。タイムスタンプ管理を行うかどうか
	 *
	 * @return	後続処理の指示
	 */
	@Override
	public int doEndTag() {
		debugPrint();		// 4.0.0 (2005/02/28)
		// 5.6.6.1 (2013/07/12) caseKey 、caseVal 属性対応
		if( !useTag() ) { return EVAL_PAGE ; }

		final StringWriter logW = new StringWriter();		// 6.0.0.0 (2014/04/11) XMLFileLoader で Logをセット

		// 6.0.0.0 (2014/04/11) XMLFileLoader に渡す 読み取り開始ファイルオブジェクト。
		final String directory = HybsSystem.url2dir( fileURL );
		final File loadFile = ( filename != null ) ? new File( directory,filename ) : new File( directory );

		final int insCnt ;				// 追加数だけ記録する。
		boolean errFlag = true;
		Transaction tran = null;		// 5.1.9.0 (2010/08/01) Transaction 対応
		try {
			// 5.1.9.0 (2010/08/01) Transaction 対応
			final TransactionTag tranTag = (TransactionTag)findAncestorWithClass( this,TransactionTag.class );
			if( tranTag == null ) {
				tran = new TransactionReal( getApplicationInfo() );		// 5.3.7.0 (2011/07/01) 引数変更
			}
			else {
				tran = tranTag.getTransaction();
			}

			final Connection conn = tran.getConnection( dbid );		// 5.1.9.0 (2010/08/01) Transaction 対応

			// 6.0.0.0 (2014/04/11) フォルダ一括登録ができるようにします。
	 		// 6.0.2.0 (2014/09/19) useTimeStamp 属性追加。タイムスタンプ管理を行うかどうか
//			XMLFileLoader loader = new XMLFileLoader( conn,false );			// コネクションとuseTimeStamp=false 指定
			final XMLFileLoader loader = new XMLFileLoader( conn,useTimeStamp );	// コネクションとuseTimeStamp 指定
			if( keys != null ) { loader.setAfterMap( getAfterMap() ); }
			loader.setLogWriter( logW );

			loader.loadXMLFiles( loadFile );

			final int[] crudCnt = loader.getCRUDCount();	// 実行結果のカウント数
			insCnt = crudCnt[XMLFileLoader.INS];

			tran.commit();			// 5.1.9.0 (2010/08/01) Transaction 対応
			errFlag = false;		// エラーではない
		}
		catch( Throwable ex ) {
			if( tran != null ) {		// 5.5.2.6 (2012/05/25) findbugs対応
				tran.rollback();		// 5.1.9.0 (2010/08/01) Transaction 対応
			}
			throw new HybsSystemException( ex );
		}
		finally {
			if( tran != null ) {		// 5.5.2.6 (2012/05/25) findbugs対応
				tran.close( errFlag );
			}
		}

		// 実行件数の表示
		// 4.0.0 (2005/11/30) 出力順の変更。一番最初に出力します。
		if( displayMsg != null && displayMsg.length() > 0 ) {
			final StringBuilder buf = new StringBuilder( BUFFER_MIDDLE )
				.append( "<pre>" )
				.append( logW.toString() )							// 6.0.0.0 (2014/04/11) XMLFileLoader で Logをセット
				.append( CR )
				.append( HybsSystem.getDate() ).append( "  " )
				.append( getResource().getLabel( displayMsg ) )
				.append( CR )
				.append( "</pre>" );

			jspPrint( buf.toString() );
		}

		// 時間測定用の DIV 要素を出力
		final long dyTime = System.currentTimeMillis()-dyStart;
		jspPrint( "<div id=\"queryTime\" value=\"" + (dyTime) + "\"></div>" );	// 3.5.6.3 (2004/07/12)

		// 4.0.0 (2005/01/31) セキュリティチェック(データアクセス件数登録)
		final GUIInfo guiInfo = (GUIInfo)getSessionAttribute( HybsSystem.GUIINFO_KEY );
		if( guiInfo != null ) { guiInfo.addWriteCount( insCnt,dyTime,loadFile.getPath() ); }

		return EVAL_PAGE ;
	}

	/**
	 * タグリブオブジェクトをリリースします。
	 * キャッシュされて再利用されるので、フィールドの初期設定を行います。
	 *
	 * @og.rev 4.0.0.0 (2007/10/10) dbid の初期値を、"DEFAULT" から null に変更
	 * @og.rev 6.0.0.0 (2014/04/11) filename の初期値を、システムプロパティーのFILE_FILENAME から null に変更
	 * @og.rev 6.0.0.0 (2014/04/11) displayMsg の初期値を、MSG0040 から MSG0003 に変更
	 * @og.rev 6.0.2.0 (2014/09/19) useTimeStamp 属性追加。タイムスタンプ管理を行うかどうか
	 */
	@Override
	protected void release2() {
		super.release2();
		dbid		= null;
		fileURL		= HybsSystem.sys( "FILE_URL" );			// 4.0.0 (2005/01/31)
		filename	= null;									// 6.0.0.0 (2014/04/11) 初期値:null
		displayMsg	= "MSG0003";							// ファイルの登録が完了しました。
		keys		= null;
		vals		= null;
		useTimeStamp= false;								// 6.0.2.0 (2014/09/19) タイムスタンプ管理を行うかどうか
	}

	/**
	 * XMLファイルを読み取った後で指定するカラムと値のペア(マップ)情報を作成します。
	 *
	 * このカラムと値のペアのマップは、オブジェクト構築後に設定される為、
	 * XMLファイルのキーの存在に関係なく、Mapのキーと値が使用されます。(Map優先)
	 * key が null や ゼロ文字列の場合は、Map に追加しません。
	 *
	 * @og.rev 5.6.6.1 (2013/07/12) key が null や ゼロ文字列の場合は、Map に追加しません。
	 * @og.rev 6.0.0.0 (2014/04/11) keys と vals の個数チェックを追加
	 *
	 * @return	カラムと値のペア(マップ)情報
	 */
	private Map<String,String> getAfterMap() {
		// 6.0.2.4 (2014/10/17) NP:null 値を利用している可能性があります。
		if( keys == null || vals == null ) { return null; }		// 追加しないケース

		// 6.0.0.0 (2014/04/11) keys と vals の個数チェックを追加
//		if( keys != null && keys.length != vals.length ) {
		if( keys.length != vals.length ) {
			final String errMsg = "keys と vals の個数が異なります。"
								+ " keys=" + Arrays.toString( keys )
								+ " vals=" + Arrays.toString( vals ) ;
			throw new HybsSystemException( errMsg );
		}

		final Map<String,String> map = new HashMap<String,String>();
		for( int i=0; i<keys.length; i++ ) {
			if( keys[i] != null && keys[i].length() > 0 ) {		// 5.6.6.1 (2013/07/12)
				map.put( keys[i],vals[i] );
			}
		}
		return map ;
	}

	/**
	 * 【TAG】(通常は使いません)検索時のDB接続IDを指定します(初期値:DEFAULT)。
	 *
	 * @og.tag
	 *   検索時のDB接続IDを指定します。初期値は、DEFAULT です。
	 *
	 * @param	id データベース接続ID
	 */
	public void setDbid( final String id ) {
		dbid = nval( getRequestParameter( id ),dbid );
	}

	/**
	 * 【TAG】読み取り元ディレクトリ名を指定します
	 *		(初期値:FILE_URL[={@og.value org.opengion.hayabusa.common.SystemData#FILE_URL}])。
	 *
	 * @og.tag
	 * この属性で指定されるディレクトリより、ファイルを読み取ります。
	 * 指定方法は、通常の fileURL 属性と同様に、先頭が、'/' (UNIX) または、２文字目が、
	 * ":" (Windows)の場合は、指定のURLそのままのディレクトリに、そうでない場合は、
	 * fileURL = "{&#064;USER.ID}" と指定すると、FILE_URL 属性で指定のフォルダの下に、
	 * さらに、各個人ID別のフォルダの下より、読み取ります。
	 * (初期値:システム定数のFILE_URL[={@og.value org.opengion.hayabusa.common.SystemData#FILE_URL}])。
	 *
	 * @og.rev 4.0.0.0 (2007/11/20) 指定されたディレクトリ名の最後が"\"or"/"で終わっていない場合に、"/"を付加する。
	 *
	 * @param	url ファイルURL
	 * @see		org.opengion.hayabusa.common.SystemData#FILE_URL
	 */
	public void setFileURL( final String url ) {
		String furl = nval( getRequestParameter( url ),null );
		if( furl != null ) {
			final char ch = furl.charAt( furl.length()-1 );
			if( ch != '/' && ch != '\\' ) { furl = furl + "/"; }
			fileURL = StringUtil.urlAppend( fileURL,furl );
		}
	}

	/**
	 * 【TAG】ファイルを作成するときのファイル名をセットします(初期値:null)。
	 *
	 * @og.tag
	 * ファイルを作成するときのファイル名をセットします。
	 * (初期値:null)。
	 *
	 * @og.rev 6.0.0.0 (2014/04/11) filename の初期値を、システムプロパティーのFILE_FILENAME から null に変更
	 *
	 * @param   fname ファイル名
	 */
	public void setFilename( final String fname ) {
		filename = nval( getRequestParameter( fname ),filename );
	}

	/**
	 * 【TAG】query の結果を画面上に表示するメッセージIDを指定します(初期値:MSG0003[ファイルの登録が完了しました])。
	 *
	 * @og.tag
	 * ここでは、検索結果の件数や登録された件数をまず出力し、
	 * その次に、ここで指定したメッセージをリソースから取得して
	 * 表示します。
	 * 表示させたくない場合は, displayMsg = "" をセットしてください。
	 * 初期値は、検索件数を表示します。
	 * ※ この属性には、リクエスト変数({&#064;XXXX})は使用できません。
	 *
	 * @param	id 処理結果表示メッセージID
	 */
	public void setDisplayMsg( final String id ) {
		if( id != null ) { displayMsg = id; }
	}

	/**
	 * 【TAG】XMLファイルを読み取った後で指定するキーをCSV形式で複数指定します。
	 *
	 * @og.tag
	 * XMLファイルを読み取った後で、データを変更できます。
	 * 変更するカラム名(キー)をCSV形式で指定します。
	 * XMLファイルにキーが存在していた場合は、vals で指定の値に書き換えます。
	 * キーが存在していない場合は、ここで指定するキーと値が、データとして
	 * 追加されます。
	 * 例えば、登録日や、登録者、または、テンプレートより各システムID毎に
	 * 登録するなどの使い方を想定しています。
	 * 分解方法は、CSV変数を先に分解してから、getRequestParameter で値を取得します。
	 * こうしないとデータ自身にカンマを持っている場合に分解をミスる為です。
	 *
	 * @param	key リンク先に渡すキー
	 * @see		#setVals( String )
	 */
	public void setKeys( final String key ) {
		keys = getCSVParameter( key );
	}

	/**
	 * 【TAG】XMLファイルを読み取った後で指定する値をCSV形式で複数指定します。
	 *
	 * @og.tag
	 * XMLファイルを読み取った後で、データを変更できます。
	 * 変更する値をCSV形式で指定します。
	 * XMLファイルにキーが存在していた場合は、vals で指定の値に書き換えます。
	 * キーが存在していない場合は、ここで指定するキーと値が、データとして
	 * 追加されます。
	 * 例えば、登録日や、登録者、または、テンプレートより各システムID毎に
	 * 登録するなどの使い方を想定しています。
	 * 分解方法は、CSV変数を先に分解してから、getRequestParameter で値を取得します。
	 * こうしないとデータ自身にカンマを持っている場合に分解をミスる為です。
	 *
	 * @param	val keys属性に対応する値
	 * @see		#setKeys( String )
	 */
	public void setVals( final String val ) {
		vals = getCSVParameter( val );
	}

	/**
	 * 【TAG】タイムスタンプ管理を行うかどうか[true:行う/false:行わない]を指定します(初期値:false)。
	 *
	 * @og.tag
	 * useTimeStamp=true の場合、登録の実行有無の判断は、ファイルの更新時刻より判断します。
	 * これは、読み取りファイルの更新時刻が、０でない場合、読み取りを行います。
	 * 読み取りが完了した場合は、更新時刻を ０ に設定します。
	 * つまり、一度しか読み込まないように制御できます。
	 * useTimeStamp=false の場合は、無条件に読み込みます。
	 *
	 * @og.rev 6.0.2.0 (2014/09/19) 新規追加
	 *
	 * @param	flag タイムスタンプ管理 [true:行う/false:行わない]
	 */
	public void setUseTimeStamp( final String flag ) {
		useTimeStamp = nval( getRequestParameter( flag ),useTimeStamp );
	}

	/**
	 * このオブジェクトの文字列表現を返します。
	 * 基本的にデバッグ目的に使用します。
	 *
	 * @return このクラスの文字列表現
	 * @og.rtnNotNull
	 */
	@Override
	public String toString() {
		return ToString.title( this.getClass().getName() )
				.println( "VERSION"			,VERSION		)
				.println( "dbid"			,dbid			)
				.println( "fileURL"			,fileURL		)
				.println( "filename"		,filename		)
				.println( "displayMsg"		,displayMsg		)
				.println( "dyStart"			,dyStart		)
				.println( "Other..."		,getAttributes().getAttribute() )
				.fixForm().toString() ;
	}
}
