/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.fukurou.util.StringUtil;
import org.opengion.fukurou.util.ToString;						// 6.1.1.0 (2015/01/17)

import static org.opengion.fukurou.util.StringUtil.nval ;

/**
 * Where句を作成するための条件を指定します。
 *
 * このタグのvalue 値に、{&#064;XXXX} 変数が含まれている場合、そのリクエスト値が
 * ない場合は、このタグそのものがなにも出力しません。(つまり条件から消えます。)
 * startKeyは、value を連結する場合の頭に置かれる文字列で、where句の最初には表示されず、
 * それ以降について、表示されます。(つまり、where VALUE1 and VALUE2 and VALUE3 … です。)
 * startKey の初期値は、"and" です。
 * multi は、{&#064;XXXX} 変数に、値が複数含まれている場合の処理を規定します。
 * 複数の値とは、同一nameでチェックボックス指定や、メニューでの複数指定した場合、
 * リクエストが配列で送られます。multi="true" とすると、'xx1','xx2','xx3', ･･･ という
 * 形式に変換されます。
 * 具体的には、"where PN in ( {&#064;PN} )" という文字列に対して、
 * "where PN in ( 'xx1','xx2','xx3' )" を作成することができます。
 * multi の初期値は、"false" です。
 * SystemData の USE_SQL_INJECTION_CHECK が true か、quotCheck 属性が true の場合は、
 * ＳＱＬインジェクション対策用のクォーティションチェックを行います。リクエスト引数に
 * クォーティション(')が含まれると、エラーになります。
 * 同様にUSE_XSS_CHECKがtrueか、xssCheck属性がtrueの場合は、
 * クロスサイトススクリプティング(XSS)対策のためless/greater than signのチェックを行います。
 *
 * 各属性は、{&#064;XXXX} 変数が使用できます。
 * これは、ServletRequest から、XXXX をキーに値を取り出し,この変数に割り当てます。
 * つまり、このXXXXをキーにリクエストすれば、この変数に値をセットすることができます。
 *
 * @og.formSample
 * ●形式：&lt;og:and startKey="[and|or|…]" value="…" multi="[false|true]" /&gt;
 * ●body：あり(EVAL_BODY_BUFFERED:BODYを評価し、{&#064;XXXX} を解析します)
 *
 * ●Tag定義：
 *   &lt;og:and
 *       startKey           【TAG】SQL条件句の最初の演算子を指定します(初期値:and)
 *       value              【TAG】条件の値を セットします
 *       multi              【TAG】複数の引数に対して処理するかどうか[true/false]を設定します(初期値:false)
 *       separator          【TAG】multi アクション時の文字列を分割する項目区切り文字をセットします
 *       instrVals          【TAG】スペースで区切られた複数の値すべてを含む条件を作成します
 *       instrType          【TAG】instrValsで複数の値を条件にする際の方法を指定します(初期値:and)
 *       quotCheck          【TAG】リクエスト情報の クォーティション(') 存在チェックを実施するかどうか[true/false]を設定します (初期値:USE_SQL_INJECTION_CHECK[=true])
 *       xssCheck           【TAG】リクエスト情報の HTMLTag開始/終了文字(&gt;&lt;) 存在チェックを実施するかどうか[true/false]を設定します (初期値:USE_XSS_CHECK[=true])
 *       caseKey            【TAG】このタグ自体を利用するかどうかの条件キーを指定します(初期値:null)
 *       caseVal            【TAG】このタグ自体を利用するかどうかの条件値を指定します(初期値:null)
 *       caseNN             【TAG】指定の値が、null/ゼロ文字列 でない場合(Not Null=NN)は、このタグは使用されます(初期値:true)
 *       caseNull           【TAG】指定の値が、null/ゼロ文字列 の場合は、このタグは使用されます(初期値:true)
 *       debug              【TAG】デバッグ情報を出力するかどうか[true/false]を指定します(初期値:false)
 *   &gt;   ... Body ...
 *   &lt;/og:and&gt;
 *
 * ●使用例
 *     &lt;og:query command="NEW"&gt;
 *             select PN,YOBI,NMEN,HINM from XX01
 *         &lt;og:where&gt;
 *             &lt;og:and value="PN   =    '{&#064;PN}'"    /&gt;
 *             &lt;og:and value="YOBI like '{&#064;YOBI}%'" /&gt;
 *         &lt;/og:where&gt;
 *             order by PN
 *     &lt;/og:query&gt;
 *
 *          ・検索条件が入力された時(PN=AAA , YOBI=BBB)
 *            作成されるSQL文⇒select PN,YOBI,NMEN,HINM from XX01 where PN = 'AAA' and YOBI like 'BBB%' order by PN
 *
 *          ・検索条件が片方入力されなかった時(PNがNULLのとき, YOBI=BBB)
 *            作成されるSQL文⇒select PN,YOBI,NMEN,HINM from XX01 where YOBI like 'BBB%' order by PN
 *
 *          ・検索条件が入力されなかった時(PNがNULL, YOBIがNULL) WHERE句がなくなる。
 *            作成されるSQL文⇒select PN,YOBI,NMEN,HINM from XX01 order by PN
 *
 *        注意:WhereTagを使わない場合に、検索条件が入力されなかった場合は、下記のようになります。
 *            select PN,YOBI,NMEN,HINM from XX01 where PN = '' and YOBI like '%' order by PN
 *
 *    --------------------------------------------------------------------------------------------------------------
 *
 *     &lt;og:query command="NEW"&gt;
 *             select PN,YOBI,NMEN,HINM from XX01 where PN="11111"
 *         &lt;og:where startKey="and"&gt;
 *             &lt;og:and value="YOBI in   ({&#064;YOBI})" multi="true" /&gt;
 *             &lt;og:and value="HINM like '{&#064;HINM}%'"             /&gt;
 *         &lt;/og:where&gt;
 *             order by PN
 *     &lt;/og:query&gt;
 *
 *          ・YOBI を複数選択し、in で検索する時(YOBI=AA,BB,CC を選択)
 *            作成されるSQL文⇒select PN,YOBI,NMEN,HINM from XX01 where PN = '11111'
 *                             and YOBI in ( 'AA','BB','CC' ) and HINM like 'BBB%' order by PN
 *
 * @og.group 画面部品
 *
 * @version  4.0
 * @author	 Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class SqlAndTag extends CommonTagSupport {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "6.1.1.0 (2015/01/17)" ;

	private static final long serialVersionUID = 611020150117L ;

	private String	startKey	= "and";
	private String	value		= "";
	private String	instrVals	;			// 3.8.8.1 (2007/01/06)
	private String	instrType	= "and";	// 5.4.1.0 (2011/11/01)
	private boolean	multi		;
	private boolean	quotCheck	= HybsSystem.sysBool( "USE_SQL_INJECTION_CHECK" );	// 4.0.0 (2005/08/31)
	private boolean	xssCheck	= HybsSystem.sysBool( "USE_XSS_CHECK" );			// 5.0.0.2 (2009/09/15)

	private boolean	allNull	 	;			// 5.0.0.2 (2009/09/15)
	private boolean	localReq	;			// 6.1.1.0 (2015/01/17) このクラスの getRequestValue を呼び出すキー

	private String	separator	;			// 5.2.2.0 (2010/11/01) 項目区切り文字

	/**
	 * Taglibの開始タグが見つかったときに処理する doStartTag() を オーバーライドします。
	 *
	 * @og.rev 4.0.0.0 (2006/12/05) BODY 部の値を value に使用する機能追加
	 * @og.rev 4.0.0.0 (2005/08/31) useQuotCheck() によるＳＱＬインジェクション対策
	 * @og.rev 5.0.0.2 (2009/09/15) XSS対策
	 * @og.rev 5.2.2.0 (2010/11/01) caseKey 、caseVal 属性対応
	 * @og.rev 6.1.1.0 (2015/01/17) localReq変数を使う事で、RequestParameter処理を制御します。
	 *
	 * @return	後続処理の指示
	 */
	@Override
	public int doStartTag() {
		// 5.2.2.0 (2010/11/01) caseKey 、caseVal 属性対応
		if( useTag() ) {
			useQuotCheck( quotCheck );
			// 5.0.0.2 (2009/09/15) XSS対策
			useXssCheck( xssCheck );

			localReq = multi;						// 6.1.1.0 (2015/01/17) 内部の getRequestValue を呼び出すキー
			value = getRequestParameter( value );

			if( value == null || value.isEmpty() ) {
				return EVAL_BODY_BUFFERED ;	// Body を評価する。( extends BodyTagSupport 時)
			}

	//		if( value != null && value.length() > 0 ) {
	//			return( SKIP_BODY );			// Body を評価しない
	//		}
	//		else {
	//			return EVAL_BODY_BUFFERED ;	// Body を評価する。( extends BodyTagSupport 時)
	//		}
		}
		return SKIP_BODY ;			// Body を評価しない
	}

	/**
	 * Taglibのタグ本体を処理する doAfterBody() を オーバーライドします。
	 *
	 * @og.rev 4.0.0.0 (2006/12/05) BODY 部の値を value に使用する機能追加
	 * @og.rev 6.1.1.0 (2015/01/17) localReq変数を使う事で、RequestParameter処理を制御します。
	 *
	 * @return	後続処理の指示(SKIP_BODY)
	 */
	@Override
	public int doAfterBody() {
		localReq = multi;						// 6.1.1.0 (2015/01/17) 内部の getRequestValue を呼び出すキー
		value = getBodyString();
		return SKIP_BODY ;
	}

	/**
	 * Taglibの終了タグが見つかったときに処理する doEndTag() を オーバーライドします。
	 *
	 * @og.rev 3.1.1.2 (2003/04/04) Tomcat4.1 対応。release2() を doEndTag()で呼ぶ。
	 * @og.rev 3.8.8.1 (2007/01/06) makeInstrVals を加味する。
	 * @og.rev 5.0.0.2 (2009/09/15) multi時のallNull対応
	 * @og.rev 5.1.9.0 (2010/08/01) matchKey 、matchVal 対応 ⇒ 5.2.2.0 (2010/11/01) 廃止
	 * @og.rev 5.2.2.0 (2010/11/01) caseKey 、caseVal 属性対応
	 *
	 * @return	後続処理の指示
	 */
	@Override
	public int doEndTag() {
		debugPrint();		// 4.0.0 (2005/02/28)
		// 5.2.2.0 (2010/11/01) caseKey 、caseVal 属性対応
		if( useTag() ) {
			final SqlWhereTag where = (SqlWhereTag)findAncestorWithClass( this,SqlWhereTag.class );
			if( where == null ) {
				final String errMsg = "<b>" + getTagName() + "タグは、where タグの内部におく必要があります。</b>";
				throw new HybsSystemException( errMsg );
			}

			// 5.1.9.0 (2010/08/01) matchKey 、matchVal 対応 ⇒ 5.2.2.0 (2010/11/01) 廃止
			if( ! isNull() && ! allNull ) {			// 5.2.2.0 (2010/11/01)
				value = makeInstrVals( instrVals,instrType,value );	// 5.4.1.0 (2011/11/01)
				if( value != null ) {
					set( "keyWord", startKey );
					set( "value"  , value );
					where.setAttributes( getAttributes() );
				}
			}
		}
		return EVAL_PAGE ;
	}

	/**
	 * タグリブオブジェクトをリリースします。
	 * キャッシュされて再利用されるので、フィールドの初期設定を行います。
	 *
	 * @og.rev 2.0.0.4 (2002/09/27) カスタムタグの release() メソッドを、追加
	 * @og.rev 3.1.1.2 (2003/04/04) Tomcat4.1 対応。release2() を doEndTag()で呼ぶ。
	 * @og.rev 3.8.8.1 (2007/01/06) instrVals 属性追加
	 * @og.rev 4.0.0.0 (2005/08/31) quotCheck 属性の追加
	 * @og.rev 5.0.0.2 (2009/09/15) XSS対応
	 * @og.rev 5.0.0.2 (2009/09/15) multi時のallNull対応
	 * @og.rev 5.1.9.0 (2010/08/01) matchKey、matchVal 属性の追加
	 * @og.rev 5.2.2.0 (2010/11/01) separator , isMatch 属性の追加
	 * @og.rev 5.2.2.0 (2010/11/01) matchKey、matchVal 属性廃止(caseKey,caseVal属性を使用してください。)
	 * @og.rev 5.4.1.0 (2011/11/01) instrType属性追加
	 * @og.rev 6.1.1.0 (2015/01/17) localReq変数を使う事で、RequestParameter処理を制御します。
	 */
	@Override
	protected void release2() {
		super.release2();
		startKey	= "and";
		value		= "";
		instrVals	= null;		// 3.8.8.1 (2007/01/06)
		instrType	= "and";	// 5.4.1.0 (2011/11/01)
		multi		= false;
		quotCheck	= HybsSystem.sysBool( "USE_SQL_INJECTION_CHECK" );	// 4.0.0 (2005/08/31)
		xssCheck	= HybsSystem.sysBool( "USE_XSS_CHECK" );			// 5.0.0.2 (2009/09/15)
		allNull	 	= false;	// 5.0.0.2 (2009/09/15)
		separator	= null;		// 5.2.2.0 (2010/11/01) 項目区切り文字
		localReq	= false;	// 6.1.1.0 (2015/01/17) このクラスの getRequestValue を呼び出すキー
	}

	/**
	 * リクエスト情報の文字列を取得します。
	 *
	 * これは、通常のgetRequestParameter 処理の中で呼ばれる getRequestValue を
	 * オーバーライトしています。
	 *
	 * @og.rev 5.0.0.2 (2009/09/15) valuesの全NULL/空文字をisNull扱いにする
	 * @og.rev 5.3.8.0 (2011/08/01) Attribute等からも値が取得できるようにする。の対応時の特殊処理
	 * @og.rev 6.1.1.0 (2015/01/17) localReq変数を使う事で、RequestParameter処理を制御します。
	 *
	 * @param    key キー
	 *
	 * @return   リクエスト情報の文字列
	 */
	@Override
	protected String getRequestValue( final String key ) {
		String rtn = "";

//		if( multi ) {
		if( localReq ) {		// 6.1.1.0 (2015/01/17) localReq変数を使う
			// 5.3.8.0 (2011/08/01) getRequestValues の中で、getRequestValue を呼び出すためこのままでは
			// 再帰呼び出しが永遠に続くので、２回目以降は、再帰しないように、強制的に multi の値を書き換えます。
//			multi = false;		// 5.3.8.0 (2011/08/01) 再帰しないように、強制的に値を書き換え
			localReq = false;	// 6.1.1.0 (2015/01/17) 再帰しないように、localReq変数の値を書き換え
			final String[] array = getRequestValues( key );
			allNull = true; // 5.0.0.2 (2009/09/15) arrayの内容が全てnull/空文字か
			if( ! isNull() ) {
				// 5.0.0.2 (2009/09/15) 全てnull/空文字の場合はnullと扱い
				for( int i=0; i<array.length; i++ ) {
//					if( array[i] != null && array[i].length() > 0 ) {
					if( array[i] != null && !array[i].isEmpty() ) {
						allNull = false;
						break;
					}
				}
				if( ! allNull ){
					rtn = makeCSVvalue( array );
				}
			}
//			multi = true;	// 5.3.8.0 (2011/08/01) 強制的に書き換えた値を元に戻す。
		}
		else {
			rtn = super.getRequestValue( key );
		}
		return rtn ;
	}

	/**
	 * 複数の値を 'xx1','xx2','xx3', ･･･ という形式に変換します。
	 *
	 * この処理は、in などで使用するためのリクエストを配列で受け取って処理
	 * する場合の文字列を加工します。
	 *
	 * @og.rev 5.2.2.0 (2010/11/01) separator 対応
	 * @og.rev 6.1.1.0 (2015/01/17) 引数が、null や空文字列の場合は、処理しません。
	 *
	 * @param	array	元の文字列配列(可変長引数)
	 *
	 * @return  連結後の文字列
	 * @og.rtnNotNull
	 */
//	private String makeCSVvalue( final String[] array ) {
	private String makeCSVvalue( final String... array ) {
		if( array == null || array.length == 0 ) {
			final String errMsg = "array 引数に、null や、サイズゼロの配列は使用できません。";
			throw new HybsSystemException( errMsg );
		}

		final StringBuilder buf = new StringBuilder( BUFFER_MIDDLE );

		if( separator != null ) {
			for(int i=0; i < array.length; i++) {
				if( array[i] != null && !array[i].isEmpty() ) {			// 6.1.1.0 (2015/01/17)
					final String[] ary = array[i].split( separator );
					for( int j=0; j<ary.length; j++ ) {
						if( ary[j] != null && !ary[j].isEmpty() ) {		// 6.1.1.0 (2015/01/17)
//							buf.append( "'" );
							buf.append( '\'' ).append( ary[j] ).append( "'," );
						}
					}
				}
			}
			buf.deleteCharAt( buf.length()-1 );		// 最後の ピリオドを削除する。
		}
		else {
			for(int i=0; i < array.length; i++) {
				if( array[i] != null && !array[i].isEmpty() ) {			// 6.1.1.0 (2015/01/17)
//					buf.append( "'" );
					buf.append( '\'' ).append( array[i] ).append( "'," );
				}
			}
			buf.deleteCharAt( buf.length()-1 );		// 最後の ピリオドを削除する。
		}

		return buf.toString();
	}

	/**
	 * スペースで区切られた複数の値を and 接続で連結します。
	 *
	 * value="CLM" instrVals="ABC DEF GHI" と指定すると、
	 * value="CLM LIKE '%ABC%' AND CLM LIKE '%DEF%'  AND CLM LIKE '%GHI%' "
	 * という文字列を作成します。
	 * 個別にLIKE検索項目を AND 連結する為、現れる場所に依存しません。
	 * 逆に、現れる順序を指定する場合は、ABC%DEF の様に指定可能です。
	 * ただし、columnMarker の instrVals で、複数文字のマーカーを行う場合、
	 * ABC%DEF という文字列は、オリジナルでないので、マークアップされません。
	 *
	 * @og.rev 5.4.1.0 (2011/11/01) instrType属性対応
	 * @og.rev 5.5.1.1 (2012/04/06) notin対応
	 * @og.rev 6.1.1.0 (2015/01/17) localReq変数を使う事で、RequestParameter処理を制御します。
	 * @og.rev 6.1.1.0 (2015/01/17) 分割キーをseparatorで指定可能とします。
	 *
	 * @param	instrVals 	繰返し処理を行う値
	 * @param	instrType 	連結方法
	 * @param	value   	繰返し処理を行うvalue
	 *
	 * @return  連結後の文字列
	 * @see		#setInstrVals( String )
	 * @see		ColumnMarkerTag#setInstrVals( String )
	 */
	private String makeInstrVals( final String instrVals, final String instrType , final String value ) {
		if( instrVals == null || instrVals.isEmpty() ) { return value; }

		localReq = multi;						// 6.1.1.0 (2015/01/17) 内部の getRequestValue を呼び出すキー
		final String reqVals = nval( getRequestParameter( instrVals ),null );
		if( reqVals == null || reqVals.isEmpty() ) { return null; }

		final String[] vals ;
		if( multi ) {
			// multi のときは、makeCSVvalue で加工された値になっている。
			vals = StringUtil.csv2Array( reqVals,',' );
			// 前後の ' はずし
			for( int i=0; i<vals.length; i++ ) {
				vals[i] = vals[i].substring( 1,vals[i].length()-1 );
			}
		}
		else {
//			vals = StringUtil.csv2Array( reqVals,' ' );
			vals = reqVals.split( separator == null ? " " : separator );	// 6.1.1.0 (2015/01/17) 分割キーを指定
		}

		if( vals == null || vals.length == 0 ) { return null; }

		char instType = instrType != null && instrType.length() > 0 ? instrType.charAt(0) : 'X' ;

		final StringBuilder buf = new StringBuilder( BUFFER_MIDDLE )
			.append( " (" );
		// 条件:and ⇒ 各値をandのlike条件で結合(%あり)
//		if( instrType.startsWith( "a" ) ) {
		if( 'a' == instType || 'A' == instType ) {			// 6.1.1.0 (2015/01/17) 大文字対応
			buf.append( value ).append( " LIKE '%" ).append( vals[0] ).append( "%'" );
			for(int i=1; i<vals.length; i++) {
				buf.append( " and " ).append( value ).append( " LIKE '%" ).append( vals[i] ).append( "%'" );
			}
//			for(int i=0; i<vals.length; i++) {
//				if( i > 0 ) { buf.append( " and " ); }
//				buf.append( value ).append( " LIKE '%" ).append( vals[i] ).append( "%'" );
//			}
		}
		// 条件:or ⇒ 各値をorのlike条件で結合(%あり)
//		else if( instrType.startsWith( "o" ) ) {
		else if( 'o' == instType || 'O' == instType ) {			// 6.1.1.0 (2015/01/17) 大文字対応
			buf.append( value ).append( " LIKE '%" ).append( vals[0] ).append( "%'" );
			for(int i=1; i<vals.length; i++) {
				buf.append( " or " ).append( value ).append( " LIKE '%" ).append( vals[i] ).append( "%'" );
			}
//			for(int i=0; i<vals.length; i++) {
//				if( i > 0 ) { buf.append( " or " ); }
//				buf.append( value ).append( " LIKE '%" ).append( vals[i] ).append( "%'" );
//			}
		}
		// 条件:in ⇒ 各値をorのlike条件で結合(%なし)
//		else if( instrType.startsWith( "i" ) ) {
		else if( 'i' == instType || 'I' == instType ) {			// 6.1.1.0 (2015/01/17) 大文字対応
			buf.append( value ).append( " LIKE '" ).append( vals[0] ).append( '\'' );
			for(int i=1; i<vals.length; i++) {
				buf.append( " or " ).append( value ).append( " LIKE '" ).append( vals[i] ).append( '\'' );
			}
//			for(int i=0; i<vals.length; i++) {
//				if( i > 0 ) { buf.append( " or " ); }
//				buf.append( value ).append( " LIKE '" ).append( vals[i] ).append( '\'' );
//			}
		}
		// 条件:notin ⇒ 各値をandのnot like条件で結合(%なし) 5.5.1.1(2012/04/05)
//		else if( instrType.startsWith( "n" ) ) {
		else if( 'n' == instType || 'N' == instType ) {			// 6.1.1.0 (2015/01/17) 大文字対応
			buf.append( value ).append( " NOT LIKE '" ).append( vals[0] ).append( '\'' );
			for(int i=1; i<vals.length; i++) {
				buf.append( " and " ).append( value ).append( " NOT LIKE '" ).append( vals[i] ).append( '\'' );
			}
//			for(int i=0; i<vals.length; i++) {
//				if( i > 0 ) { buf.append( " and " ); }
//				buf.append( value ).append( " NOT LIKE '" ).append( vals[i] ).append( '\'' );
//			}
		}
		else {
			final String errMsg = "instrTypeには、'and','or','in','notin'のいずれかを指定して下さい。" + CR +
										" instrType=[" + instrType + "]";
			throw new HybsSystemException( errMsg );
		}
		buf.append( ") " );

		return buf.toString();
	}

	/**
	 * 【TAG】SQL条件句の最初の演算子を指定します(初期値:and)。
	 *
	 * @og.tag
	 * value を連結する場合の頭に置かれる文字列で、where句の最初には表示されず、
	 * それ以降について、表示されます。
	 * (つまり、where VALUE1 and VALUE2 and VALUE3 … です。)
	 * startKey の初期値は、"and" です。
	 *
	 * @og.rev 6.1.1.0 (2015/01/17) localReq変数を使う事で、RequestParameter処理を制御します。
	 *
	 * @param	skey 条件句の最初の演算子
	 */
	public void setStartKey( final String skey ) {
//		if( skey != null && skey.length() > 0 ) { startKey = skey; }
		startKey = nval( getRequestParameter( skey ),startKey );		// 6.1.1.0 (2015/01/17)
	}

	/**
	 * 【TAG】条件の値を セットします。
	 *
	 * @og.tag
	 * 条件値に、{&#064;XXXX} 変数が含まれている場合、そのリクエスト値がない場合は、
	 * このタグそのものがなにも出力しません。(つまり条件から消えます。)
	 * BODY 部に記述することが可能です。その場合は、条件属性になにも設定できません。
	 *
	 * @param	val 条件値
	 */
	public void setValue( final String val ) {
		value = val;
	}

	/**
	 * 【TAG】スペースで区切られた複数の値すべてを含む条件を作成します。
	 *
	 * @og.tag
	 * value="CLM" instrVals="ABC DEF GHI" と指定すると、
	 * value="CLM LIKE '%ABC%' AND CLM LIKE '%DEF%'  AND CLM LIKE '%GHI%' "
	 * という文字列を作成します。
	 * 通常の、value="CLM LIKE '%ABC%DEF%'" の指定方法では、ABCとDEFの
	 * 順番が固定化されますが、instrVals を用いた方法では、個別指定が可能です。
	 *
	 * これは、instrVals に指定した引数に対して、スペース区切り(separatorで変更可)
	 * で分割し、前方の value に複数のAND検索(instrTypeで変更可)を同時に指定できる
	 * ため、現れる場所に依存しません。
	 * 逆に、現れる順序を指定する場合は、ABC%DEF の様に指定可能です。
	 * ただし、columnMarker の instrVals で、複数文字のマーカーを行う場合、
	 * ABC%DEF という文字列は、オリジナルでないので、マークアップされません。
	 * ※instrType属性の指定により条件の生成方法を変更することができます。
	 *   詳細については、instrType属性のドキュメントを参照下さい。
	 *
	 * @param	val 複合条件作成のための設定値
	 * @see		#setInstrType
	 * @see		ColumnMarkerTag#setInstrVals( String )
	 */
	public void setInstrVals( final String val ) {
		instrVals = val;
	}

	/**
	 * 【TAG】instrValsで複数の値を条件にする際の方法を指定します(初期値:and)。
	 *
	 * @og.tag
	 * 通常、instrValsに指定された値は、スペース区切りで分割した各値を
	 * LIKE条件としてand結合します。
	 * しかし、instrType属性を変更することで、この条件式の生成方法を変更
	 * することができます。
	 * 具体的には、以下の通りです。
	 * ①instrTypeに"and"が指定されている場合(初期値)
	 *   タグの記述 : value="CLM" instrVals="ABC DEF GHI"
	 *   生成文字列 :       "( CLM LIKE '%ABC%' AND CLM LIKE '%DEF%' AND CLM LIKE '%GHI%' )"
	 * ②instrTypeに"or"が指定されている場合
	 *   タグの記述 : value="CLM" instrVals="ABC DEF GHI"
	 *   生成文字列 :       "( CLM LIKE '%ABC%' OR CLM LIKE '%DEF%' OR CLM LIKE '%GHI%' )"
	 * ③instrTypeに"in"が指定されている場合
	 *   タグの記述 : value="CLM" instrVals="ABC DEF GHI"
	 *   生成文字列 :       "( CLM LIKE 'ABC' OR CLM LIKE 'DEF5' OR CLM LIKE 'GHI' )"
	 * ④instrTypeに"notin"が指定されている場合
	 * 	 タグの記述 : value="CLM" instrVals="ABC DEF GHI"
	 *   生成文字列 :       "( CLM NOT LIKE 'ABC' AND CLM NOT LIKE 'DEF5' AND CLM NOT LIKE 'GHI' )"
	 * ※この属性を指定しない場合は、①のLIKE条件でのand結合となります。
	 * ※③④について、LIKE条件で%を自動付加しないことにより、画面からの入力値に応じて、
	 *   前方一致、後方一致、前後方一致の制御を行うことができます。
	 *
	 * @og.rev 5.5.1.1 (2012/04/06) notin対応(コメント修正)
	 * @og.rev 6.1.1.0 (2015/01/17) 初期値指定のコーディングミス修正
	 *
	 * @param	tp 条件方法 [and/or/in/notin]
	 * @see		#setInstrVals( String )
	 */
	public void setInstrType( final String tp ) {
//		instrType = nval( getRequestParameter( tp ),tp );
		instrType = nval( getRequestParameter( tp ),instrType );		// 6.1.1.0 (2015/01/17)
	}

	/**
	 * 【TAG】複数の引数に対して処理するかどうか[true/false]を設定します(初期値:false)。
	 *
	 * @og.tag
	 * {&#064;XXXX} 変数に、値が複数含まれている場合の処理を規定します。
	 * multi="true" に設定すると、複数の引数は、'xx1','xx2','xx3', ･･･ という
	 * 形式に変換します。
	 * where 条件で言うと、 "where PN in ( {&#064;PN} )" という文字列に対して、
	 * "where PN in ( 'xx1','xx2','xx3' )" を作成することになります。
	 * 初期値は、 false (マルチ変換しない) です。
	 *
	 * @og.rev 6.1.1.0 (2015/01/17) localReq変数を使う事で、RequestParameter処理を制御します。
	 *
	 * @param   flag マルチ変換 [true:する/それ以外:しない]
	 * @see		#setSeparator( String )
	 */
	public void setMulti( final String flag ) {
//		multi = nval( flag,multi );
		multi = nval( getRequestParameter( flag ),multi );
	}

	/**
	 * 【TAG】multi アクション/instrVals 時の文字列を分割する項目区切り文字をセットします。
	 *
	 * @og.tag
	 * multi="true" の場合、複数のリクエストを連結して、in 句で問合せを行う文字列を
	 * 作成しますが、separator を指定すると、さらに、separator で文字列を分割して、
	 * in 句の引数を構築します。
	 * これは、instrVals を指定した場合にも、同様に分解します。
	 * 具体的には、分割後の文字列が、複数の個々のリクエスト変数と同じ形式に加工されます。
	 * String#split( separator ) で、分解するため、正規表現が使用できます。
	 *
	 * 何も指定しない場合は、multi アクション時は、分割処理は行いません。
	 * instrVals 時は、スペースで分解処理します。
	 *
	 * @og.rev 5.2.2.0 (2010/11/01) 新規追加
	 * @og.rev 6.1.1.0 (2015/01/17) コメント修正。separatorは、正規表現が使用できます。
	 *
	 * @param   sepa 項目区切り文字(正規表現)
	 * @see		#setMulti( String )
	 */
	public void setSeparator( final String sepa ) {
		separator = nval( getRequestParameter( sepa ),separator );
	}

	/**
	 * 【TAG】リクエスト情報の クォーティション(') 存在チェックを実施するかどうか[true/false]を設定します
	 *		(初期値:USE_SQL_INJECTION_CHECK[={@og.value org.opengion.hayabusa.common.SystemData#USE_SQL_INJECTION_CHECK}])。
	 *
	 * @og.tag
	 * ＳＱＬインジェクション対策の一つとして、暫定的ではありますが、SQLのパラメータに
	 * 渡す文字列にクォーティション(') を許さない設定にすれば、ある程度は防止できます。
	 * 数字タイプの引数には、 or 5=5 などのクォーティションを使用しないコードを埋めても、
	 * 数字チェックで検出可能です。文字タイプの場合は、必ず (')をはずして、
	 * ' or 'A' like 'A のような形式になる為、(')チェックだけでも有効です。
	 * (') が含まれていたエラーにする(true)／かノーチェックか(false)を指定します。
	 * (初期値:システム定数のUSE_SQL_INJECTION_CHECK[={@og.value org.opengion.hayabusa.common.SystemData#USE_SQL_INJECTION_CHECK}])。
	 *
	 * @og.rev 4.0.0.0 (2005/08/31) 新規追加
	 *
	 * @param   flag クォーティションチェック [true:する/それ以外:しない]
	 * @see		org.opengion.hayabusa.common.SystemData#USE_SQL_INJECTION_CHECK
	 */
	public void setQuotCheck( final String flag ) {
		quotCheck = nval( getRequestParameter( flag ),quotCheck );
	}

	/**
	 * 【TAG】リクエスト情報の HTMLTag開始/終了文字(&gt;&lt;) 存在チェックを実施するかどうか[true/false]を設定します
	 *		(初期値:USE_XSS_CHECK[={@og.value org.opengion.hayabusa.common.SystemData#USE_XSS_CHECK}])。
	 *
	 * @og.tag
	 * クロスサイトスクリプティング(XSS)対策の一環としてless/greater than signについてのチェックを行います。
	 * (&gt;&lt;) が含まれていたエラーにする(true)／かノーチェックか(false)を指定します。
	 * (初期値:システム定数のUSE_XSS_CHECK[={@og.value org.opengion.hayabusa.common.SystemData#USE_XSS_CHECK}])。
	 *
	 * @og.rev 5.0.0.2 (2009/09/15) 新規追加
	 *
	 * @param	flag	XSSチェック [true:する/false:しない]
	 * @see		org.opengion.hayabusa.common.SystemData#USE_XSS_CHECK
	 */
	public void setXssCheck( final String flag ) {
		xssCheck = nval( getRequestParameter( flag ),xssCheck );
	}

	/**
	 * タグの名称を、返します。
	 * 自分自身のクラス名より、自動的に取り出せないため、このメソッドをオーバーライドします。
	 *
	 * @og.rev 4.0.0.0 (2005/01/31) 新規追加
	 *
	 * @return  タグの名称
	 * @og.rtnNotNull
	 */
	@Override
	protected String getTagName() {
		return "and" ;
	}

	/**
	 * このオブジェクトの文字列表現を返します。
	 * 基本的にデバッグ目的に使用します。
	 *
	 * @return このクラスの文字列表現
	 * @og.rtnNotNull
	 */
	@Override
	public String toString() {
		return ToString.title( this.getClass().getName() )
				.println( "VERSION"			,VERSION	)
				.println( "startKey"		,startKey	)
				.println( "value"			,value		)
				.println( "instrVals"		,instrVals	)
				.println( "multi"			,multi		)
				.println( "quotCheck"		,quotCheck	)
				.println( "Other..."	,getAttributes().getAttribute() )
				.fixForm().toString() ;
	}
}
