/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.db.DBTableModel;
import org.opengion.hayabusa.io.TableWriter;
import org.opengion.fukurou.util.FileUtil;
import org.opengion.fukurou.util.HybsEntry;
import org.opengion.fukurou.util.Closer ;
import org.opengion.fukurou.util.StringUtil ;
import org.opengion.fukurou.util.ToString;						// 6.1.1.0 (2015/01/17)

import static org.opengion.fukurou.util.StringUtil.nval ;

import javax.servlet.ServletResponse ;
import javax.servlet.http.HttpServletResponse ;
import javax.servlet.jsp.JspWriter ;

import java.util.List;
import java.util.ArrayList;
import java.util.Locale;
import java.io.File;
import java.io.OutputStream;
import java.io.PrintWriter;
import java.io.FileOutputStream;
import java.io.BufferedOutputStream;				// 6.0.4.0 (2014/11/28)
import java.io.ByteArrayOutputStream;
import java.io.ObjectOutputStream;
import java.io.ObjectInputStream;
import java.io.IOException;
import java.io.InputStream;							// 6.0.4.0 (2014/11/28)
import java.io.FileInputStream;						// 6.0.4.0 (2014/11/28)
import java.io.BufferedInputStream;					// 6.0.4.0 (2014/11/28)

// import java.util.zip.GZIPOutputStream;
// import java.util.zip.ZipOutputStream;
// import java.util.zip.ZipEntry;
import org.apache.commons.compress.archivers.zip.ZipArchiveEntry;			// 6.0.4.0 (2014/11/28)
import org.apache.commons.compress.archivers.zip.ZipArchiveOutputStream;	// 6.0.4.0 (2014/11/28)
import org.apache.commons.compress.utils.IOUtils;							// 6.0.4.0 (2014/11/28)

/**
 * DBTableModelオブジェクトを指定のファイルに出力するタグです。
 *
 * データ(DBTableModel)と、コントローラ(WriteTableTagForm)を与えて、
 * 外部からコントロールすることで、各種形式で データ(DBTableModel)を表示できます。
 * writeTableタグ に対して、コマンドを与えることにより、内部のコントローラの
 * 実装に対応した、 形式でデータを作成します。
 *
 * @og.formSample
 * ●形式：
 *     &lt;og:writeTable
 *         fileURL   = "{&#064;USER.ID}"    保存先ディレクトリ名
 *         filename  = "{&#064;filename}"   保存ファイル名
 *         encode    = "UnicodeLittle"      保存ファイルエンコード名
 *     /&gt;
 *
 * ●body：あり(EVAL_BODY_BUFFERED:BODYを評価し、{&#064;XXXX} を解析します)
 *
 * ●Tag定義：
 *   &lt;og:writeTable
 *       writerClass        【TAG】実際に書き出すクラス名の略称(TableWriter_**** クラスの ****)を指定します(初期値:Default)
 *       separator          【TAG】可変長ファイルを作成するときの項目区切り文字をセットします(初期値:TableWriter#TAB_SEPARATOR[= ])
 *       headerSequence     【TAG】DBTableModelの出力順(LNSCD など)をセットします
 *       fileURL            【TAG】ファイル保存先ディレクトリ名を指定します(初期値:FILE_URL[={@og.value org.opengion.hayabusa.common.SystemData#FILE_URL}])
 *       filename           【TAG】ファイルを作成するときのファイル名をセットします(初期値:FILE_FILENAME[={@og.value org.opengion.hayabusa.common.SystemData#FILE_FILENAME}])
 *       beforeDelete       【TAG】事前にファイルを削除するかどうかをセットします (初期値:false)
 *       encode             【TAG】ファイルを作成するときのファイルエンコーディング名をセットします(初期値:FILE_ENCODE[={@og.value org.opengion.hayabusa.common.SystemData#FILE_ENCODE}])
 *       fileAppend         【TAG】追加モードで書き込むかどうか[true/false]を指定します(初期値:false[新規モード])
 *       direct             【廃止】結果をダイレクトにEXCEL起動するかどうか[true/false]を指定します(初期値:false[ファイル])
 *       zip                【TAG】結果をファイルに出力するときに、ZIPで圧縮するかどうか[true/false]を指定します(初期値:false)
 *       zipFilename        【TAG】ZIPファイルを作成するときのZIPファイル名をセットします(初期値:filename + ".zip")
 *       tableId            【TAG】(通常使いません)sessionから所得する DBTableModelオブジェクトの ID
 *       scope              【TAG】キャッシュする場合のスコープ[request/page/session/applicaton]を指定します(初期値:session)
 *       disposition        【TAG】Content-Disposition (inline:埋め込む/attachment:ダイアログを起動)を指定します(初期値:attachment)
 *       useNumber          【TAG】行番号情報を、出力する/しない[true/false]を指定します(初期値:true)
 *       columns            【TAG】書き込み先ファイルのカラム列を、外部(タグ)よりCSV形式で指定します
 *       omitNames          【TAG】書き込み対象外のカラム列を、外部(タグ)よりCSV形式で指定します
 *       useRenderer        【TAG】書込処理でコードリソースのラベル変換を行うかどうかを指定します
 *                                      (初期値:USE_TABLE_WRITER_RENDERER[={@og.value org.opengion.hayabusa.common.SystemData#USE_TABLE_WRITER_RENDERER}])
 *       selectedAll        【TAG】チェック行のみを対象にするかを指定します(初期値:true)
 *   ※  sheetName          【TAG】EXCELファイルを書き出すときのシート名を設定します
 *   ※  refFileURL         【TAG】EXCEL雛型参照ファイルのディレクトリ名をセットします
 *   ※  refFilename        【TAG】EXCEL雛型参考ファイル名をセットします
 *   ※  refSheetName       【TAG】EXCEL雛型参考ファイルのシート名を設定します
 *   ※  fontName           【TAG】EXCEL出力時のデフォルトフォント名を設定します
 *                                      (初期値:TABLE_WRITER_DEFAULT_FONT_NAME[={@og.value org.opengion.hayabusa.common.SystemData#TABLE_WRITER_DEFAULT_FONT_NAME}])
 *   ※  fontPoint          【TAG】EXCEL出力時のデフォルトフォントポイント数を設定します
 *                                      (初期値:TABLE_WRITER_DEFAULT_FONT_POINTS[={@og.value org.opengion.hayabusa.common.SystemData#TABLE_WRITER_DEFAULT_FONT_POINTS}])
 *   ※  skipRowCount       【TAG】(通常は使いません)EXCEL出力時のデータの書き込み開始位置を設定します
 *   ※  useCellStyle       【TAG】EXCEL出力時に、データを書き込んだ範囲に罫線を入れるかどうかを指定します(初期値:false)
 *   ※  useAutoCellSize    【TAG】EXCEL出力時に、セルの幅をデータの幅に自動的に合わせるかどうかを指定します(初期値:false)
 *   ※  useActiveWorkbook  【TAG】EXCEL出力時に、セルの有効範囲を設定するかどうかを指定します(初期値:false)
 *   ※  pageBreakColumn    【TAG】EXCEL出力時に、シート変更するキーとなるカラム名を指定します(このカラムの値がシート名になります)
 *   ※  fileBreakColumn    【TAG】EXCEL出力時に、ファイル名を変更するキーとなるカラム名を指定します(このカラムの値がファイル名になります)
 *   ※  hyperLinkColumn    【TAG】EXCEL出力時に、Hyperlinkを作成するキーとなるカラム名と値となるカラム名を指定します
 *   ※  addTitleSheet      【TAG】EXCEL出力時に、存在するSheet名一覧を作成する場合に、そのSheet名を指定します。
 *       language           【TAG】タグ内部で使用する言語コード[ja/en/zh/…]を指定します
 *       caseKey            【TAG】このタグ自体を利用するかどうかの条件キーを指定します(初期値:null) 5.7.7.2 (2014/06/20)
 *       caseVal            【TAG】このタグ自体を利用するかどうかの条件値を指定します(初期値:null) 5.7.7.2 (2014/06/20)
 *       caseNN             【TAG】指定の値が、null/ゼロ文字列 でない場合(Not Null=NN)は、このタグは使用されます(初期値:true) 5.7.7.2 (2014/06/20)
 *       caseNull           【TAG】指定の値が、null/ゼロ文字列 の場合は、このタグは使用されます(初期値:true) 5.7.7.2 (2014/06/20)
 *       debug              【TAG】デバッグ情報を出力するかどうか[true/false]を指定します(初期値:false)
 *   &gt;   ... Body ...
 *   &lt;/og:writeTable&gt;
 *
 * ●使用例
 *     &lt;og:writeTable ･････ &gt;
 *         &lt;og:writeTableParam
 *             key  = "Tablename"  value="GE12"
 *         /&gt;
 *         &lt;og:writeTableParam
 *             key  = "First"             First:最初に登録
 *         &gt;
 *                      insert into GE12bk
 *                         select * from GE12
 *                         where SYSTEM_ID='**'
 *         &lt;/og:writeTableParam
 *         &lt;og:writeTableParam
 *             key  = "First"             First:の２番目に登録
 *         &gt;
 *              delete from GE12 where SYSTEM_ID='**' and KBSAKU='0'
 *         &lt;/og:writeTableParam
 *         &lt;og:writeTableParam
 *             key  = "Last"              Last:最後に登録
 *         &gt;
 *              update GE12 set XXXX='YYYY' where SYSTEM_ID='**' and KBSAKU='0'
 *         &lt;/og:writeTableParam
 *     &lt;/og:writeTableParam
 *
 * @og.group ファイル出力
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class WriteTableTag extends CommonTagSupport {
	//* このプログラムのVERSION文字列を設定します	{@value} */
	private static final String VERSION = "6.1.0.0 (2014/12/26)" ;

	private static final long serialVersionUID = 610020141226L ;

	private final String  BASE_URL	= HybsSystem.sys( "FILE_URL" );

	private String	separator		= TableWriter.TAB_SEPARATOR;	 // 項目区切り文字
	private String	headerSequence	;					// ヘッダー項目の並び順
	private String	fileURL 		= BASE_URL;
	private String	filename		= HybsSystem.sys( "FILE_FILENAME" );   // ファイル名
	private String	sheetName		= "Sheet1" ;		// 3.5.4.2 (2003/12/15)
	private String	refFileURL		= BASE_URL;			// 3.5.4.3 (2004/01/05)
	private String	refFilename 	;					// 3.5.4.3 (2004/01/05)
	private String	refSheetName	;					// 3.5.4.3 (2004/01/05)
	private String	fontName		= HybsSystem.sys( "TABLE_WRITER_DEFAULT_FONT_NAME" );			// 3.8.5.3 (2006/08/07)
	private String	fontPoint	 	= HybsSystem.sys( "TABLE_WRITER_DEFAULT_FONT_POINTS" );			// 3.8.5.3 (2006/08/07)
	private String	encode			= HybsSystem.sys( "FILE_ENCODE" );	// ファイルエンコーディング  "DEFAULT","JISAutoDetect" ,"JIS", "EUC_JP", "MS932", "SJIS" , "Windows-31J" , "Shift_JIS"
	private String	writerClass 	= HybsSystem.sys( "TABLE_WRITER_DEFAULT_CLASS" );		// 3.8.5.3 (2006/08/07)
	private boolean	fileAppend		;					// ファイルをAPPENDモードで出力するか
	private boolean	direct			;
	private boolean	zip 			;
	private String  zipFilename		;					// 6.0.4.0 (2014/11/28) Zipファイル名を外部から与えられるように修正
	private String  disposition     = "attachment";		// 3.8.0.9 (2005/10/17) Content-Disposition 初期値変更 inline ⇒ attachment

	private transient DBTableModel table ;
	private String	tableId 		= HybsSystem.TBL_MDL_KEY ;
	private String	directory		;					// 3.8.6.0 (2006/08/23)

	private transient List<HybsEntry> param ;			// 4.0.0 (2005/02/28)

	private boolean	useNumber		= true;				// 3.7.0.2 (2005/02/14)
	// 4.0.0 (2005/12/31) 外部よりカラム列(CSV形式)を指定できるようにする。
	private String		columns		;
	// 6.1.0.0 (2014/12/26) 書き込み対象外のカラム列を、外部(タグ)より指定する。
	private String		omitNames	;					// 6.1.0.0 (2014/12/26)

	// 5.2.1.0 (2010/10/01) 書込処理でコードリソースのラベル変換を行うかどうか
	private boolean	useRenderer		= HybsSystem.sysBool( "USE_TABLE_WRITER_RENDERER" );	// 5.2.1.0 (2010/10/01)

	// 5.5.5.5 (2012/08/24) チェック対応
	private boolean selectedAll		= true;				// 5.5.5.5 (2012/08/24)

	// 5.5.9.1 (2012/12/07) 事前にファイルを削除するかどうか。互換性のため、false に設定。推奨は、true
	private boolean	beforeDelete	;					// 5.5.9.1 (2012/12/07) 事前にファイルを削除するかどうか。

	private int		skipRowCount	;					// 5.7.9.0 (2014/08/08) データの書き込み開始位置設定

	// 6.0.2.0 (2014/09/19) EXCEL関係機能追加
	private boolean	useCellStyle		;			// EXCEL出力時に、データを書き込んだ範囲に罫線を入れるかどうかを指定します
	private boolean	useAutoCellSize		;			// EXCEL出力時に、セルの幅をデータの幅に自動的に合わせるかどうかを指定します
	private boolean	useActiveWorkbook	;			// EXCEL出力時に、セルの有効範囲を設定するかどうかを指定します
	private String	pageBreakColumn		;			// EXCEL出力時に、シート変更するキーとなるカラム名を指定します(このカラムの値がシート名になります)
	private String	fileBreakColumn		;			// EXCEL出力時に、ファイル名を変更するキーとなるカラム名を指定します(このカラムの値がファイル名になります)
	private String	hyperLinkColumn		;			// EXCEL出力時に、Hyperlinkを作成するキーとなるカラム名と値となるカラム名を指定します
	private String	addTitleSheet		;			// EXCEL出力時に、存在するSheet名一覧を作成する場合に、そのSheet名を指定します。

	/**
	 * Taglibの開始タグが見つかったときに処理する doStartTag() を オーバーライドします。
	 *
	 * @og.rev 5.5.9.1 (2012/12/07) 事前にファイルを削除するかどうか（beforeDelete）フラグを追加
	 * @og.rev 5.7.7.2 (2014/06/20) caseKey,caseVal,caseNN,caseNull 属性を追加
	 * @og.rev 6.0.4.0 (2014/11/28) Zipファイル名を、内部変数化。(将来的に引数化を容易にするため)
	 *
	 * @return	後続処理の指示( EVAL_BODY_BUFFERED )
	 */
	@Override
	public int doStartTag() {
		// 5.7.7.2 (2014/06/20) caseKey,caseVal,caseNN,caseNull 属性を追加
		if( !useTag() ) { return SKIP_BODY ; }

		directory = HybsSystem.url2dir( fileURL );		// 5.5.9.1 (2012/12/07) doEndTag() から移動
		if( zip && zipFilename == null ) { zipFilename = filename + ".zip"; }		// 6.0.4.0 (2014/11/28)

		// 5.5.9.1 (2012/12/07) 事前にファイルを削除するかどうか
		if( beforeDelete && !fileAppend ) {
			File delFile = null;
			if( zip ) {
//				delFile = new File( directory,filename + ".zip" );
				delFile = new File( directory,zipFilename );				// 6.0.4.0 (2014/11/28)
			}
			else	  {
				delFile = new File( directory,filename );
			}
			// 6.0.2.4 (2014/10/17) RV  java.io.File.delete() の例外的戻り値を無視しています。 
//			if( delFile.exists() ) { delFile.delete(); }
			if( delFile.exists() ) {
				if( !delFile.delete() ) {
					final String errMsg = "事前にファイルを削除するのに失敗しました。filename=[" + delFile +"]" ;
					System.err.println( errMsg );
				}
			}
		}

		return EVAL_BODY_BUFFERED ;	// Body を評価する。( extends BodyTagSupport 時)
	}

	/**
	 * Taglibの終了タグが見つかったときに処理する doEndTag() を オーバーライドします。
	 *
	 * @og.rev 2.1.3.1 (2002/12/13) ContentType を、x-msexcel から vnd.ms-excel に変更した。
	 * @og.rev 3.1.1.2 (2003/04/04) Tomcat4.1 対応。release2() を doEndTag()で呼ぶ。
	 * @og.rev 3.1.3.0 (2003/04/10) FILE_ENCODE から、エンコード情報を取得する。
	 * @og.rev 3.1.5.0 (2003/04/20) Content-Disposition を外部から指定できるように変更。
	 * @og.rev 3.2.0.0 (2003/05/22) GZIPOutputStream を使用していたが、ZipOutputStream に変更。
	 * @og.rev 3.5.4.1 (2003/12/01) 引数の PrintWriter を、OutputStream に変更。
	 * @og.rev 3.5.4.3 (2004/01/05) 引数を、 PrintWriter に戻す。
	 * @og.rev 3.6.0.2 (2004/10/04) EXCELダイレクト出力時にファイルエンドまで出力されない不具合対策
	 * @og.rev 3.8.0.9 (2005/10/17) disposition属性を使用できるように変更します。
	 * @og.rev 3.8.6.0 (2006/08/23) フォルダがない場合は、複数階層分のフォルダを自動で作成します。
	 * @og.rev 4.3.4.3 (2008/12/22) Excel出力の判別方法を前方一致に変更
	 * @og.rev 5.1.6.0 (2010/05/01) 画面帳票作成機能対応
	 * @og.rev 5.5.2.0 (2012/05/01) ZIP時にもencodeが有効にする
	 * @og.rev 5.5.5.5 (2012/08/24) selectedAllの処理を追加
	 * @og.rev 5.5.9.1 (2012/12/07) beforeDeleteフラグの追加に伴う、ロジック変更
	 * @og.rev 5.7.7.2 (2014/06/20) caseKey,caseVal,caseNN,caseNull 属性を追加
	 * @og.rev 6.0.4.0 (2014/11/28) Zip処理を、ZipOutputStream → ZipArchiveOutputStream に変更
	 *
	 * @return	後続処理の指示
	 */
	@Override
	public int doEndTag() {
		debugPrint();		// 4.0.0 (2005/02/28)

		// 5.7.7.2 (2014/06/20) caseKey,caseVal,caseNN,caseNull 属性を追加
		if( !useTag() ) { return EVAL_PAGE ; }

		table = (DBTableModel)getObject( tableId );
		if( ! selectedAll ){		// 5.5.5.5 (2012/08/24) 
			table = getSelRowsTable( table );
		}
		if( table != null && table.getRowCount() > 0 ) {

			final File dir = new File(directory);
			if( ! dir.exists() && ! dir.mkdirs() ) {
				final String errMsg = "ディレクトリの作成に失敗しました。[" + directory + "]";
				throw new HybsSystemException( errMsg );
			}

			final boolean isExcel =  writerClass.toUpperCase(Locale.JAPAN).startsWith("EXCEL")
							|| writerClass.toUpperCase(Locale.JAPAN).startsWith("CALC");	// 6.0.4.0 (2014/11/28)

			PrintWriter pw = null;
			ByteArrayOutputStream bt = null ;
			try {
				if( direct ) {
					// 6.0.4.0 (2014/11/28) 共通処理を、if( zip ) の前後でまとめます。
					bt = new ByteArrayOutputStream();
					pw = new PrintWriter( bt );
					create( pw ) ;
					pw.flush();				// 3.6.0.2 (2004/10/04)
					final ServletResponse response = pageContext.getResponse() ;
					if( zip ) {
//						bt = new ByteArrayOutputStream();
//						pw = new PrintWriter( bt );
//						create( pw ) ;
//						pw.flush();				// 3.6.0.2 (2004/10/04)
//						ServletResponse response = pageContext.getResponse() ;
						((HttpServletResponse)response).addHeader( "Content-Encoding","gzip" );

//						GZIPOutputStream gzip = new GZIPOutputStream(bt);
						final ZipArchiveOutputStream gzip = new ZipArchiveOutputStream(bt);	// 6.0.4.0 (2014/11/28)
						gzip.finish() ;
						response.setContentLength( bt.size() );

						gzip.closeArchiveEntry();	// 6.0.4.0 (2014/11/28)
						gzip.finish() ;				// 6.0.4.0 (2014/11/28)
						Closer.ioClose( gzip );		// 6.0.4.0 (2014/11/28)

//						JspWriter out = pageContext.getOut();
//						out.clear();
//						out.print( bt );
//						out.flush();			// 3.6.0.2 (2004/10/04)
					}
					else {
//						bt = new ByteArrayOutputStream();
//						pw = new PrintWriter( bt );
//						create( pw ) ;
//						pw.flush();				// 3.6.0.2 (2004/10/04)
//						ServletResponse response = pageContext.getResponse() ;
						response.setContentType( "application/vnd.ms-excel; charset=" + encode );
	//					response.setContentType( "application/x-msexcel; charset=" + encode );
	//					response.setContentType( "text/comma-separated-values ; charset=" + encode );
	//					((HttpServletResponse)response).setHeader( "contentType","application/x-msexcel; charset=Shift_JIS" );
	//					((HttpServletResponse)response).setHeader( "Content-Disposition","attachment; filename=\"" + filename + "\"" );
	//					((HttpServletResponse)response).setHeader( "Content-Disposition","inline; filename=\"" + filename + "\"" );
						// 3.8.0.9 (2005/10/17) disposition属性を使用できるように変更します。
						((HttpServletResponse)response).setHeader( "Content-Disposition",disposition + "; filename=\"" + filename + "\"" );

//						JspWriter out = pageContext.getOut();
//						out.clear();
//						out.print( bt );
//						out.flush();			// 3.6.0.2 (2004/10/04)
					}
					final JspWriter out = pageContext.getOut();
					out.clear();
					out.print( bt );
					out.flush();			// 3.6.0.2 (2004/10/04)
				}
				else {
					// 3.2.0.0 (2003/05/22) GZIPOutputStream を使用していたが、ZipOutputStream に変更。
					if( zip ) {
//						bt = new ByteArrayOutputStream();
//						if( writerClass.toUpperCase(Locale.JAPAN).startsWith("EXCEL")	// 5.5.2.0(2012/05/01)
//								|| writerClass.toUpperCase(Locale.JAPAN).startsWith("CALC") ) {
						InputStream is = null;			// 6.0.4.0 (2014/11/28)
						if( isExcel ) {					// 6.0.4.0 (2014/11/28)
//							pw = new PrintWriter( bt );
							create( null );				// 6.0.4.0 (2014/11/28)
						}
						else{
							bt = new ByteArrayOutputStream();
							pw = getPrintWriter( bt );	// 6.0.4.0 (2014/11/28)
							create( pw ) ;				// 6.0.4.0 (2014/11/28)
							pw.flush();					// 6.0.4.0 (2014/11/28)
						}
//						create( pw ) ;
//						pw.flush();				// 3.6.0.2 (2004/10/04)

						// 6.0.4.0 (2014/11/28) Zip処理を、ZipOutputStream → ZipArchiveOutputStream に変更
//						ZipOutputStream gzip = null;
						ZipArchiveOutputStream gzip = null;			// 6.0.4.0 (2014/11/28)
						try {
//							gzip = new ZipOutputStream(
//											new FileOutputStream(
//												StringUtil.urlAppend( directory,filename + ".zip" )));
//							gzip.putNextEntry( new ZipEntry( filename ) );
							gzip = new ZipArchiveOutputStream(
										new BufferedOutputStream (
											new FileOutputStream (
												new File( directory,zipFilename ))));	// 6.0.4.0 (2014/11/28)
							gzip.setEncoding( "Windows-31J" );
							gzip.putArchiveEntry( new ZipArchiveEntry( filename ) );
							// 6.0.4.0 (2014/11/28) Excel処理とText処理が異なる。
							// bt.writeTo( gzip );
							if( isExcel ) {
								// 6.0.4.0 (2014/11/28) Excel の場合は、直接書き込むのではなく、ファイルを読み込んで Zip化する。
								final File xlsTemp = new File( directory,filename );
								is = new BufferedInputStream( new FileInputStream( xlsTemp ) );
								IOUtils.copy( is,gzip );
								Closer.ioClose( is );
								if( xlsTemp.exists() && !xlsTemp.delete() ) {
									final String errMsg = "ExcelをZip化する時のTempファイルを削除するのに失敗しました。filename=[" + xlsTemp +"]" ;
									System.err.println( errMsg );
								}
							}
							else {
								bt.writeTo( gzip );
							}
							gzip.closeArchiveEntry();				// 6.0.4.0 (2014/11/28)
							gzip.finish() ;
						}
						finally {
							Closer.ioClose( gzip );		// 4.0.0.0 (2006/01/31) close 処理時の IOException を無視
							Closer.ioClose( is );		// 6.0.4.0 (2014/11/28) is は、null でも処理する。
						}
					}
					else {
//						if( "Excel".equalsIgnoreCase( writerClass ) ) {
//						if( writerClass.toUpperCase(Locale.JAPAN).startsWith("EXCEL") // 4.3.4.3 (2008/12/22)
//							||  writerClass.toUpperCase(Locale.JAPAN).startsWith("CALC") ) { // 5.1.6.0 (2010/05/01)
						if( isExcel ) {					// 6.0.4.0 (2014/11/28)
							create( null );
						}
						else {
							pw = getPrintWriter();
							create( pw );
							pw.flush();				// 3.6.0.2 (2004/10/04)
						}
					}
				}
			}
			catch( IOException ex ) {
				final String errMsg = "テーブル出力時に、IOエラーが発生しました。" + toString();
				throw new HybsSystemException( errMsg,ex );		// 3.5.5.4 (2004/04/15) 引数の並び順変更
			}
			finally {
				Closer.ioClose( pw );		// 4.0.0 (2006/01/31) close 処理時の IOException を無視
				Closer.ioClose( bt );		// 4.0.0 (2006/01/31) close 処理時の IOException を無視
			}
		}

		return EVAL_PAGE ;
	}

	/**
	 * タグリブオブジェクトをリリースします。
	 * キャッシュされて再利用されるので、フィールドの初期設定を行います。
	 *
	 * @og.rev 2.0.0.4 (2002/09/27) カスタムタグの release() メソッドを、追加
	 * @og.rev 3.1.1.2 (2003/04/04) Tomcat4.1 対応。release2() を doEndTag()で呼ぶ。
	 * @og.rev 3.1.3.0 (2003/04/10) FILE_ENCODE から、エンコード情報を取得する。
	 * @og.rev 3.1.5.0 (2003/04/20) Content-Disposition を外部から指定できるように変更。
	 * @og.rev 3.5.4.2 (2003/12/15) EXCELのシート名を指定できるように変更。
	 * @og.rev 3.5.4.3 (2004/01/05) EXCEL雛型参照ファイルのURL,filenameを追加します。
	 * @og.rev 3.8.5.3 (2006/08/07) writerClass 属性の初期値をシステムリソースより取得します。
	 * @og.rev 3.8.5.3 (2006/08/07) fontName 属性を追加します。
	 * @og.rev 3.8.5.3 (2006/08/07) fontPoint 属性を追加します。
	 * @og.rev 3.8.6.0 (2006/08/23) directory 属性を追加します。
	 * @og.rev 4.0.0.0 (2005/02/28) XMLファイルにセットする、param を追加。
	 * @og.rev 5.2.1.0 (2010/10/01) 書込処理でコードリソースのラベル変換を行うかどうか useRenderer 属性追加
	 * @og.rev 5.5.5.5 (2012/08/24) selectedAll属性を追加
	 * @og.rev 5.5.9.1 (2012/12/07) 事前にファイルを削除するかどうか。互換性のため、false に設定。推奨は、true
	 * @og.rev 5.7.9.0 (2014/08/08) データの書き込み開始位置(skipRowCount)を設定
	 * @og.rev 6.0.2.0 (2014/09/19) EXCEL関係機能追加(useCellStyle,useAutoCellSize,useActiveWorkbook,
	 * 								pageBreakColumn,fileBreakColumn,hyperLinkColumn,addTitleSheet)
	 * @og.rev 6.0.4.0 (2014/11/28) zipFilename属性を追加。
	 * @og.rev 6.1.0.0 (2014/12/26) omitNames 属性を追加
	 *
	 */
	@Override
	protected void release2() {
		super.release2();
		separator		= TableWriter.TAB_SEPARATOR;   // 項目区切り文字
		headerSequence	= null;				// ヘッダー項目の並び順
		fileURL 		= BASE_URL;
		filename		= HybsSystem.sys( "FILE_FILENAME" );   // ファイル名
		sheetName		= "Sheet1";			// 3.5.4.2 (2003/12/15)
		beforeDelete	= false;			// 5.5.9.1 (2012/12/07)
		refFileURL		= BASE_URL;			// 3.5.4.3 (2004/01/05)
		refFilename 	= null;				// 3.5.4.3 (2004/01/05)
		refSheetName	= null;				// 3.5.4.2 (2003/12/15)
		fontName		= HybsSystem.sys( "TABLE_WRITER_DEFAULT_FONT_NAME" );			// 3.8.5.3 (2006/08/07)
		fontPoint		= HybsSystem.sys( "TABLE_WRITER_DEFAULT_FONT_POINTS" );			// 3.8.5.3 (2006/08/07)
		encode			= HybsSystem.sys( "FILE_ENCODE" );	 // ファイルエンコーディング  "DEFAULT","JISAutoDetect" ,"JIS", "EUC_JP", "MS932", "SJIS" , "Windows-31J" , "Shift_JIS"
		writerClass 	= HybsSystem.sys( "TABLE_WRITER_DEFAULT_CLASS" );		// 3.8.5.3 (2006/08/07)
		fileAppend		= false;			// ファイルをAPPENDモードで出力するか
		direct			= false;
		zip 			= false;
		zipFilename		= null;				// 6.0.4.0 (2014/11/28)
		table			= null;
		tableId 		= HybsSystem.TBL_MDL_KEY ;
		disposition 	= "attachment";		// 3.8.0.9 (2005/10/17) 初期値変更 inline ⇒ attachment
		param			= null ;			// 4.0.0 (2005/01/31)
		useNumber		= true;				// 3.7.0.2 (2005/02/14)
		skipRowCount	= 0;				// 5.7.9.0 (2014/08/08) データの書き込み開始位置を設定
		directory		= null;				// 3.8.6.0 (2006/08/23)
		columns			= null;				// 4.0.0 (2005/12/31)
		omitNames		= null;				// 6.1.0.0 (2014/12/26)
		useRenderer		= HybsSystem.sysBool( "USE_TABLE_WRITER_RENDERER" );	// 5.2.1.0 (2010/10/01)
		selectedAll		= true;				// 5.5.5.5 (2012/08/24)
		// 6.0.2.0 (2014/09/19) EXCEL関係機能追加
		useCellStyle		= false;		// EXCEL出力時に、データを書き込んだ範囲に罫線を入れるかどうかを指定します
		useAutoCellSize		= false;		// EXCEL出力時に、セルの幅をデータの幅に自動的に合わせるかどうかを指定します
		useActiveWorkbook	= false;		// EXCEL出力時に、セルの有効範囲を設定するかどうかを指定します
		pageBreakColumn		= null;			// EXCEL出力時に、シート変更するキーとなるカラム名を指定します(このカラムの値がシート名になります)
		fileBreakColumn		= null;			// EXCEL出力時に、ファイル名を変更するキーとなるカラム名を指定します(このカラムの値がファイル名になります)
		hyperLinkColumn		= null;			// EXCEL出力時に、Hyperlinkを作成するキーとなるカラム名と値となるカラム名を指定します
		addTitleSheet		= null;			// EXCEL出力時に、存在するSheet名一覧を作成する場合に、そのSheet名を指定します。
	}

	/**
	 * TableWriter の実オブジェクトを生成して，OutputStream に書き込みます。
	 *
	 * @og.rev 3.5.4.1 (2003/12/01) 引数の PrintWriter を、OutputStream に変更。
	 * @og.rev 3.5.4.2 (2003/12/15) lang 引数も table と同時に設定します。
	 * @og.rev 3.5.4.2 (2003/12/15) TableWriter のサブクラス名変更。
	 * @og.rev 3.5.4.3 (2004/01/05) EXCEL雛型参考ファイルを指定するように変更。
	 * @og.rev 3.5.4.3 (2004/01/05) 引数を、 PrintWriter に戻す。
	 * @og.rev 3.5.4.5 (2004/01/23) 文字エンコードを渡すように変更する。
	 * @og.rev 3.5.6.0 (2004/06/18) 各種プラグイン関連付け設定を、システムパラメータ に記述します。
	 * @og.rev 3.8.5.3 (2006/08/07) EXCEL時に fontName と fontPoint 属性をセットします。
	 * @og.rev 4.0.0.0 (2005/01/31) キーの指定を、TableWriter. から、TableWriter_ に変更します。
	 * @og.rev 4.0.0.0 (2005/12/31) 外部よりカラム列(CSV形式)を指定できるようにする。
	 * @og.rev 5.2.1.0 (2010/10/01) 書込処理でコードリソースのラベル変換を行うかどうか設定 useRenderer 属性追加
	 * @og.rev 5.7.9.0 (2014/08/08) データの書き込み開始位置(skipRowCount)を設定
	 * @og.rev 6.0.2.0 (2014/09/19) EXCEL関係機能追加(useCellStyle,useAutoCellSize,useActiveWorkbook,pageBreakColumn,fileBreakColumn)
	 * @og.rev 6.0.4.0 (2014/11/28) NullPointerException が発生するので、事前にチェックします。
	 * @og.rev 6.1.0.0 (2014/12/26) デバッグ情報を出力するかどうかを指定
	 * @og.rev 6.1.0.0 (2014/12/26) omitNames 属性を追加
	 *
	 * @param   out PrintWriterオブジェクト
	 */
	private void create( final PrintWriter out )  {
		// 5.2.1.0 (2010/10/01) writerClass で Renderer系の場合は、useRendererフラグに変更する。
		if( "ExcelRenderer".equalsIgnoreCase( writerClass ) ) {
			writerClass = "Excel" ;
			useRenderer = true;
		}
		else if( "Renderer".equalsIgnoreCase( writerClass ) ) {
			writerClass = "Default" ;
			useRenderer = true;
		}

		final String className = HybsSystem.sys( "TableWriter_" + writerClass ) ;		// 4.0.0 (2005/01/31)

		// NullPointerException が発生するので、事前にチェックします。
		if( className == null ) {
			final String errMsg = "TableWriter クラスが見つかりません。class=" + "TableWriter_" + writerClass;
			throw new RuntimeException( errMsg );
		}

		final TableWriter writer = (TableWriter)HybsSystem.newInstance( className );	// 3.5.5.3 (2004/04/09)

		writer.setDBTableModel( table,getLanguage() );
		writer.setSeparator( separator );
		writer.setHeaderSequence( headerSequence );
		writer.setEncode( encode );					// 3.5.4.5 (2004/01/23)
		writer.setAppend( fileAppend );				// 3.5.4.2 (2003/12/15)
		if( param != null ) { writer.setParam( param ); }	// 4.0.0 (2005/01/31)
		writer.setUseNumber( useNumber );			// 3.7.0.2 (2005/02/14)
	//	if( columns == null ) {
	//		columns = (String)getSessionAttribute( HybsSystem.TBL_WRITER_CLMS );
	//		removeSessionAttribute( HybsSystem.TBL_WRITER_CLMS );
	//	}
		writer.setColumns( columns );				// 4.0.0 (2005/12/31)
		writer.setOmitNames( omitNames );			// 6.1.0.0 (2014/12/26)
		writer.setUseRenderer( useRenderer );		// 5.2.1.0 (2010/10/01)
		writer.setDebug( isDebug() );				// 6.1.0.0 (2014/12/26) デバッグ情報を出力するかどうかを指定
		if( writer.isExcel() ) {					// 3.5.4.3 (2004/01/05)
			writer.setSheetName( sheetName );
//			writer.setFilename( StringUtil.urlAppend( directory , filename ) );
			writer.setFilename( directory , filename );					// 6.0.2.0 (2014/09/19) EXCEL関係機能追加
			if( refFileURL != null && refFilename != null ) {
				writer.setRefFilename( StringUtil.urlAppend( HybsSystem.url2dir(refFileURL) , refFilename ) );
			}
			if( refSheetName != null && refSheetName.length() > 0 ) {	// 5.7.9.0 (2014/08/08) nullチェック
				writer.setRefSheetName( refSheetName );
			}
//			writer.setFontName( nval( fontName,null ) );				// 3.8.5.3 (2006/08/07)
			writer.setFontName( fontName )	;							// 5.7.9.0 (2014/08/08) nvalはずし
			if( fontPoint != null && fontPoint.length() > 0 ) {
				writer.setFontPoint( Short.parseShort( fontPoint ) );	// 3.8.5.3 (2006/08/07)
			}
			writer.setSkipRowCount( skipRowCount );						// 5.7.9.0 (2014/08/08)

			writer.setUseCellStyle( useCellStyle );						// 6.0.2.0 (2014/09/19) EXCEL関係機能追加
			writer.setUseAutoCellSize( useAutoCellSize );				// 6.0.2.0 (2014/09/19) EXCEL関係機能追加
			writer.setUseActiveWorkbook( useActiveWorkbook );			// 6.0.2.0 (2014/09/19) EXCEL関係機能追加
			writer.setPageBreakColumn( pageBreakColumn );				// 6.0.2.0 (2014/09/19) EXCEL関係機能追加
			writer.setFileBreakColumn( fileBreakColumn );				// 6.0.2.0 (2014/09/19) EXCEL関係機能追加
			writer.setHyperLinkColumn( hyperLinkColumn );				// 6.0.2.0 (2014/09/19) EXCEL関係機能追加
			writer.setAddTitleSheet( addTitleSheet );					// 6.0.2.0 (2014/09/19) EXCEL関係機能追加

			writer.writeDBTable();
		}
		else {
			writer.writeDBTable( out );
		}
	}

	/**
	 * PrintWriter を取得します。
	 *
	 * ここでは、一般的なファイル出力を考慮した PrintWriter を作成します。
	 *
	 * @og.rev 2.2.0.0 (2002/12/17) 中国語(国際化)対応 エンコードの取得方法変更
	 * @og.rev 3.1.3.0 (2003/04/10) FILE_ENCODE から、エンコード情報を取得する。
	 * @og.rev 3.5.4.1 (2003/12/01) PrintWriter を、OutputStream に変更。
	 * @og.rev 3.5.5.9 (2004/06/07) FileUtil.getBufferedReader を使用
	 * @og.rev 3.7.1.1 (2005/05/23) フォルダがない場合は、複数階層分のフォルダを自動で作成します。
	 * @og.rev 3.8.0.0 (2005/06/07) FileUtil#getPrintWriter を利用。
	 *
	 * @return	ファイル書き出し用のPrintWriterオブジェクト
	 */
	private PrintWriter getPrintWriter() {
		if( filename == null ) {
			final String errMsg = "ファイル名がセットされていません。";
			throw new HybsSystemException( errMsg );
		}

		// ※ 注意 StringUtil.urlAppend を組み込んでいる意図が不明。一旦削除していますが、注意
	 	// 3.8.0.0 (2005/06/07) FileUtil#getPrintWriter を利用。
	//	out = FileUtil.getPrintWriter( StringUtil.urlAppend( directory,filename ),fileAppend,encode);
		final PrintWriter out = FileUtil.getPrintWriter( new File( directory,filename ),encode,fileAppend );

		return out ;
	}

	/**
	 * PrintWriter を取得します。
	 *
	 * 引数で指定したOutputStreamを利用して出力します。
	 *
	 * @og.rev 5.5.2.0 (2012/05/01) 新規作成
	 *
	 * @param	os	OutputStreamオブジェクト
	 *
	 * @return	指定したstreamに書き出すPrintWriter
	 */
	private PrintWriter getPrintWriter( final OutputStream os ) {
		final PrintWriter out = FileUtil.getPrintWriter( os,encode );

		return out ;
	}

	/**
	 * チェック分のテーブルモデルを再構成します。
	 *
	 * @og.rev 5.5.5.5 (2012/08/24) 新規作成
	 * @og.rev 5.5.9.1 (2012/12/07) 内部処理変更
	 *
	 * @param	oldTable	コピー元テーブルモデル
	 *
	 * @return	チェックした分だけで再構成したテーブルモデル
	 */
	private DBTableModel getSelRowsTable( final DBTableModel oldTable ) {
		if( oldTable == null ) { return null; }		// 5.5.9.1 (2012/12/07) 引数のDBTableModelが null の場合の処理。

		final DBTableModel newTable = oldTable.newModel();
		final int[] rowNo = getParameterRows();
		final int rowCount = rowNo.length ;

		for( int i=0; i<rowCount; i++ ) {
			newTable.addValues( oldTable.getValues(rowNo[i]), i );
		}
		return newTable;
	}

	/**
	 * 【TAG】(通常は使いません)結果のDBTableModelを、sessionに登録するときのキーを指定します
	 *		(初期値:HybsSystem#TBL_MDL_KEY[={@og.value org.opengion.hayabusa.common.HybsSystem#TBL_MDL_KEY}])。
	 *
	 * @og.tag
	 * 検索結果より、DBTableModelオブジェクトを作成します。これを、下流のviewタグ等に
	 * 渡す場合に、通常は、session を利用します。その場合の登録キーです。
	 * query タグを同時に実行して、結果を求める場合、同一メモリに配置される為、
	 * この tableId 属性を利用して、メモリ空間を分けます。
	 *		(初期値:HybsSystem#TBL_MDL_KEY[={@og.value org.opengion.hayabusa.common.HybsSystem#TBL_MDL_KEY}])。
	 *
	 * @param	id テーブルID (sessionに登録する時のID)
	 */
	public void setTableId( final String id ) {
		tableId = nval( getRequestParameter( id ), tableId );
	}

	/**
	 * 【TAG】可変長ファイルを作成するときの項目区切り文字をセットします
	 *		(初期値:TableWriter#TAB_SEPARATOR[={@og.value TableWriter#TAB_SEPARATOR}])。
	 *
	 * @og.tag 可変長ファイルを作成するときの項目区切り文字をセットします。
	 * (初期値:TableWriter#TAB_SEPARATOR[={@og.value TableWriter#TAB_SEPARATOR}])。
	 *
	 * @param   separator 項目区切り文字
	 * @see		org.opengion.hayabusa.io.TableWriter#TAB_SEPARATOR
	 */
	public void setSeparator( final String separator ) {
		this.separator = nval( getRequestParameter( separator ),this.separator );
	}

	/**
	 * 【TAG】DBTableModelの出力順(LNSCD など)をセットします。
	 *
	 * @og.tag
	 * Label,Name,Size,Class,Data の各フィールドの頭文字のアルファベットで
	 * 出力順を設定します。
	 *
	 * @param   hs 出力順 (LNSCD など)
	 */
	public void setHeaderSequence( final String hs ) {
		final String[] headers = getRequestParameterValues( hs );
		if( headers != null ) {
			final StringBuilder buf = new StringBuilder( BUFFER_MIDDLE );
			for( int i=0; i<headers.length; i++ ) {
				if( headers[i] != null ) { buf.append( headers[i] ); }
			}
			headerSequence = buf.toString();
		}
	}

	/**
	 * 【TAG】ファイル保存先ディレクトリ名を指定します
	 *		(初期値:FILE_URL[={@og.value org.opengion.hayabusa.common.SystemData#FILE_URL}])。
	 *
	 * @og.tag
	 * この属性で指定されるディレクトリに、ファイルをセーブします。
	 * 指定方法は、通常の fileURL 属性と同様に、先頭が、'/' (UNIX) または、２文字目が、
	 * ":" (Windows)の場合は、指定のURLそのままのディレクトリに、そうでない場合は、
	 * システムパラメータ の FILE_URL 属性で指定のフォルダの下に、作成されます。
	 * fileURL = "{&#064;USER.ID}" と指定すると、FILE_URL 属性で指定のフォルダの下に、
	 * さらに、各個人ID別のフォルダを作成して、そこにセーブします。
	 *
	 * @og.rev 3.5.4.3 (2004/01/05) 内部処理を、makeFileURL に移動。
	 * @og.rev 4.0.0.0 (2005/01/31) StringUtil.urlAppend メソッドの利用
	 * @og.rev 4.0.0.0 (2007/11/20) 指定されたディレクトリ名の最後が"\"or"/"で終わっていない場合に、"/"を付加する。
	 *
	 * @param	url 保存先ディレクトリ名
	 */
	public void setFileURL( final String url ) {
		String furl = nval( getRequestParameter( url ),null );
		if( furl != null ) {
			final char ch = furl.charAt( furl.length()-1 );
			if( ch != '/' && ch != '\\' ) { furl = furl + "/"; }
			fileURL = StringUtil.urlAppend( fileURL,furl );
		}
	}

	/**
	 * 【TAG】ファイルを作成するときのファイル名をセットします
	 *		(初期値:FILE_FILENAME[={@og.value org.opengion.hayabusa.common.SystemData#FILE_FILENAME}])。
	 *
	 * @og.tag ファイルを作成するときのファイル名をセットします。
	 * (初期値:システム定数のFILE_ENCODE[={@og.value org.opengion.hayabusa.common.SystemData#FILE_FILENAME}])。
	 *
	 * @param   filename ファイル名
	 * @see		org.opengion.hayabusa.common.SystemData#FILE_FILENAME
	 */
	public void setFilename( final String filename ) {
		this.filename = nval( getRequestParameter( filename ),this.filename );
	}

	/**
	 * 【TAG】EXCELファイルを書き出すときのシート名を設定します(初期値:Sheet1)。
	 *
	 * @og.tag
	 * EXCELファイルを書き出す時に、シート名を指定します。これにより、複数の形式の
	 * 異なるデータを順次書き出したり(appendモードを併用)することや、シートを指定して
	 * 新規にEXCELを作成することが可能になります。
	 * 初期値は、Sheet1 です。
	 *
	 * @og.rev 3.5.4.2 (2003/12/15) 新規追加
	 *
	 * @param   sheet シート名
	 */
	public void setSheetName( final String sheet ) {
		sheetName = nval( getRequestParameter( sheet ),sheetName );
	}

	/**
	 * 【TAG】ファイルを作成するときのファイルエンコーディング名をセットします
	 *		(初期値:FILE_ENCODE[={@og.value org.opengion.hayabusa.common.SystemData#FILE_ENCODE}])。
	 *
	 * @og.tag
	 * "DEFAULT","JISAutoDetect" ,"JIS", "EUC_JP", "MS932", "SJIS" , "Windows-31J" , "Shift_JIS"
	 * (初期値:システム定数のFILE_ENCODE[={@og.value org.opengion.hayabusa.common.SystemData#FILE_ENCODE}])。
	 *
	 * @og.rev 2.2.0.0 (2002/12/17) 中国語(国際化)対応 エンコードの取得方法変更
	 * @og.rev 3.1.3.0 (2003/04/10) FILE_ENCODE から、エンコード情報を取得する。
	 *
	 * @param   enc エンコーディング名 (例:DEFAULT,JISAutoDetect,JIS,EUC_JP,MS932,SJIS,Windows-31J,Shift_JIS)
	 * @see     <a href="http://www.iana.org/assignments/character-sets">IANA Charset Registry</a>
	 * @see		org.opengion.hayabusa.common.SystemData#FILE_ENCODE
	 */
	public void setEncode( final String enc ) {
		encode = nval( getRequestParameter( enc ),encode );
	}

	/**
	 * 【TAG】実際に書き出すクラス名の略称({@og.doc03Link writerClass TableWriter_**** クラスの ****})を指定します(初期値:Default)。
	 *
	 * @og.tag
	 * 実際に書き出すクラス名(の略称)をセットします。
	 * org.opengion.hayabusa.io.TableWriter インターフェースを継承したサブクラスです。
	 * これは、org.opengion.plugin.io 以下の TableWriter_**** クラスの **** を与えます。
	 * 属性クラス定義の {@link org.opengion.hayabusa.io.TableWriter TableWriter} を参照願います。
	 * {@og.doc03Link writerClass TableWriter_**** クラス}
	 *
	 * ExcelRenderer と、Renderer は、特別な名前で、useRenderer 属性に、"true" をセットするのと同じ効果があります。
	 * 実際は、ExcelRenderer ⇒ Excel 、Renderer  ⇒ Default のクラスが呼ばれます。
	 *
	 * @param   writerClass クラス名(の略称)
	 * @see		org.opengion.hayabusa.io.TableWriter  TableWriterのサブクラス
	 */
	public void setWriterClass( final String writerClass ) {
		this.writerClass = nval( getRequestParameter( writerClass ),this.writerClass );
	}

	/**
	 * 【TAG】追加モードで書き込むかどうか[true/false]を指定します(初期値:false[新規モード])。
	 *
	 * @og.tag DBTableModelのファイルを、追加モードで書き込むかどうかをセットします。
	 *
	 * @param   flag 追加モードか [true:追加モード/それ以外:新規モード]
	 */
	public void setFileAppend( final String flag ) {
		fileAppend = nval( getRequestParameter( flag ),fileAppend );
	}

	/**
	 * 【廃止】結果をダイレクトにEXCEL起動するかどうか[true/false]を指定します(初期値:false[ファイル])。
	 *
	 * @og.tag 結果をダイレクトに EXCEL ファイルとして出力するかどうかをセットします。
	 *
	 * @param  flag ダイレクト起動か [true:ダイレクト/それ以外:ファイル]
	 * @deprecated 結果をダイレクトするとエンコード等でおかしな動きをする可能性がある。
	 */
	@Deprecated public void setDirect( final String flag ) {
		direct = nval( getRequestParameter( flag ),direct );
	}

	/**
	 * 【TAG】結果をファイルに出力するときに、ZIPで圧縮するかどうか[true/false]を指定します(初期値:false)。
	 *
	 * @og.tag
	 * 初期値は、圧縮しないです。
	 *
	 * @param  flag ZIPで圧縮する [true:圧縮する/それ以外:しない]
	 */
	public void setZip( final String flag ) {
		zip = nval( getRequestParameter( flag ),zip );
	}

	/**
	 * 【TAG】ZIPファイルを作成するときのZIPファイル名をセットします(初期値:filename + ".zip")。
	 *
	 * @og.tag
	 * zip 属性に、true を指定した場合に、ZIPファイル化します。その場合のファイル名を指定します。
	 * なにも指定しない場合は、filename + ".zip" になります。
	 *
	 * @og.rev 6.0.4.0 (2014/11/28) Zipファイル名を外部から与えられるように修正
	 *
	 * @param   zipFile ZIPファイル名
	 * @see #setZip( String )
	 */
	public void setZipFilename( final String zipFile ) {
		zipFilename = nval( getRequestParameter( zipFile ),zipFilename );
	}

	/**
	 * 【TAG】Content-Disposition [inline:埋め込む/attachment:ダイアログを起動]を指定します(初期値:attachment)。
	 *
	 * @og.tag
	 * Content-Disposition を指定することで、ブラウザにアプリケーションデータを
	 * 埋め込むのか(inline)、セーブ/オープンのダイアログを起動するのか(attachment)
	 * 指定します。
	 * この属性が有効なのは、direct="true" で、zip="false"(初期値)のときのみです。
	 * 値は、inline 、attachment が指定できます。
	 * 初期値は、attachment です。
	 * 注意：動作検証しましたが、inline と attachment で動作に差が現れませんでした。
	 *
	 * @og.rev 3.1.5.0 (2003/04/20) Content-Disposition を外部から指定できるように変更。
	 * @og.rev 3.8.0.9 (2005/10/17) Content-Disposition 初期値変更 inline ⇒ attachment
	 *
	 * @param  pos 埋め込みするかどうか [inline:埋め込む/attachment:ダイアログを起動]
	 */
	public void setDisposition( final String pos ) {
		disposition = nval( getRequestParameter( pos ),disposition );
		if( !"inline".equals( disposition ) && !"attachment".equals( disposition ) ) {
			final String errMsg = "disposition には、inline か attachment 以外は指定できません。"
						+ "disposition=[" + disposition + "]" ;
			throw new HybsSystemException( errMsg );
		}
	}

	/**
	 * 【TAG】EXCEL雛型参照ファイルのディレクトリ名をセットします
	 * 		(初期値:FILE_URL[={@og.value org.opengion.hayabusa.common.SystemData#FILE_URL}])。
	 *
	 * @og.tag
	 * この属性で指定されるディレクトリに、ファイルをセーブします。
	 * 指定方法は、通常の fileURL 属性と同様に、先頭が、'/' (UNIX) または、２文字目が、
	 * ":" (Windows)の場合は、指定のURLそのままのディレクトリに、そうでない場合は、
	 * システムパラメータ の FILE_URL 属性で指定のフォルダの下に、作成されます。
	 * fileURL = "{&#064;USER.ID}" と指定すると、FILE_URL 属性で指定のフォルダの下に、
	 * さらに、各個人ID別のフォルダを作成して、そこにセーブします。
	 *
	 * @og.rev 3.5.4.3 (2004/01/05) 新規作成
	 * @og.rev 4.0.0.0 (2005/01/31) StringUtil.urlAppend メソッドの利用
	 * @og.rev 4.0.0.0 (2007/11/20) 指定されたディレクトリ名の最後が"\"or"/"で終わっていない場合に、"/"を付加する。
	 *
	 * @param	url 雛型ﾃﾞｨﾚｸﾄﾘ名
	 */
	public void setRefFileURL( final String url ) {
		String furl = nval( getRequestParameter( url ),null );
		if( furl != null ) {
			final char ch = furl.charAt( furl.length()-1 );
			if( ch != '/' && ch != '\\' ) { furl = furl + "/"; }
			refFileURL = StringUtil.urlAppend( refFileURL,furl );
		}
	}

	/**
	 * 【TAG】EXCEL雛型参考ファイル名をセットします。
	 *
	 * @og.tag
	 * 内部処理的には、雛形ファイルの雛形シートをメモリ上に構築し、
	 * そのシートのセルに、データを追加していきます。
	 * 最後には、雛形シートとして存在しない部分は、すべて削除します。
	 * 雛形シートを使わない場合は、新規シートに追加していくことになります。
	 * appendモードの場合は、オリジナルが雛形ファイルになります。
	 * 雛形ファイルの指定は、無効になります。
	 * ※ 互換性の為、同じ名前の設定は有効です。
	 *
	 * ※ Excel2007形式(XSSF)では、行全体を指定した色や罫線は、
	 * セルとして認識されません。範囲指定で、罫線や色付けを行う必要があります。
	 *
	 * @og.rev 3.5.4.3 (2004/01/05) 新規作成
	 *
	 * @param   filename 雛型ﾌｧｲﾙ名
	 */
	public void setRefFilename( final String filename ) {
		refFilename = nval( getRequestParameter( filename ),refFilename );
	}

	/**
	 * 【TAG】EXCEL雛型参考ファイルのシート名を設定します。
	 *
	 * @og.tag
	 * EXCELファイルを書き出す時に、雛型として参照するシート名を指定します。
	 * これにより、複数の形式の異なるデータを順次書き出したり(appendモードを併用)する
	 * ことや、シートを指定して新規にEXCELを作成する場合にフォームを設定する事が可能になります。
	 * 初期値は、null(第一シート) です。
	 *
	 * @og.rev 3.5.4.3 (2004/01/05) 新規追加
	 *
	 * @param   sheet 雛型ｼｰﾄ名
	 */
	public void setRefSheetName( final String sheet ) {
		refSheetName = nval( getRequestParameter( sheet ),refSheetName );
	}

	/**
	 * 【TAG】EXCEL出力時のデフォルトフォント名を設定します
	 *		(初期値:TABLE_WRITER_DEFAULT_FONT_NAME[={@og.value org.opengion.hayabusa.common.SystemData#TABLE_WRITER_DEFAULT_FONT_NAME}])。
	 *
	 * @og.tag
	 * これは、EXCEL追加機能として実装されています。
	 *
	 * EXCELファイルを書き出す時に、デフォルトフォント名を指定します。
	 * フォント名は、EXCELのフォント名をそのまま使用してください。
	 * 内部的に、POI の org.apache.poi.hssf.usermodel.HSSFFont#setFontName( String )
	 * に設定されます。
	 * (初期値:システム定数のTABLE_WRITER_DEFAULT_FONT_NAME[={@og.value org.opengion.hayabusa.common.SystemData#TABLE_WRITER_DEFAULT_FONT_NAME}])。
	 *
	 * @og.rev 3.8.5.3 (2006/08/07) 新規追加
	 *
	 * @param   name デフォルトフォント名
	 * @see		org.opengion.hayabusa.common.SystemData#TABLE_WRITER_DEFAULT_FONT_NAME
	 */
	public void setFontName( final String name ) {
		fontName = nval( getRequestParameter( name ),fontName );
	}

	/**
	 * 【TAG】EXCEL出力時のデフォルトフォントポイント数を設定します
	 *		(初期値:TABLE_WRITER_DEFAULT_FONT_POINTS[={@og.value org.opengion.hayabusa.common.SystemData#TABLE_WRITER_DEFAULT_FONT_POINTS}])。
	 *
	 * @og.tag
	 * これは、EXCEL追加機能として実装されています。
	 *
	 * EXCELファイルを書き出す時に、デフォルトポイント数を指定します。
	 * 内部的に、POI の org.apache.poi.hssf.usermodel.HSSFFont#setFontHeightInPoints( short )
	 * に設定されます。
	 * (初期値:システム定数のTABLE_WRITER_DEFAULT_FONT_POINTS[={@og.value org.opengion.hayabusa.common.SystemData#TABLE_WRITER_DEFAULT_FONT_POINTS}])。
	 *
	 * @og.rev 3.8.5.3 (2006/08/07) 新規追加
	 *
	 * @param   point フォントポイント数
	 * @see		org.opengion.hayabusa.common.SystemData#TABLE_WRITER_DEFAULT_FONT_POINTS
	 */
	public void setFontPoint( final String point ) {
		fontPoint = nval( getRequestParameter( point ),fontPoint );
	}

	/**
	 * 【TAG】ファイル出力時に、行番号情報を、出力する/しない[true/false]を指定します(初期値:true)。
	 *
	 * @og.tag
	 * 通常のフォーマットでは、各行の先頭に行番号を出力します。
	 * これは、#NAME 属性を使用する場合には、必ず出力する必要があります。
	 * (#NAME 属性は、読み取り時にあれば、自動的にカラムに割り当てられます。)
	 * この、先頭の行番号が不要な場合(つまり、他のシステムへのデータ出力等)
	 * の為に出力する場合に、false を設定することで、行番号列を出力しない
	 * ようにできます。
	 * 初期値は、true(出力する) です。
	 *
	 * @og.rev 3.7.0.2 (2005/02/14) 新規追加
	 *
	 * @param   useNo 行番号情報 [true:出力する/false:しない]
	 */
	public void setUseNumber( final String useNo ) {
		useNumber = nval( getRequestParameter( useNo ),useNumber );
	}

	/**
	 * 【TAG】書き込み先ファイルのカラム列を、外部(タグ)よりCSV形式で指定します。
	 *
	 * @og.tag
	 * この指定により、書き込むカラムを限定的に使用することが可能です。
	 * カラム名は、検索したDBTableModel上に含まれる名前で指定する必要があります。
	 * 別名で書き込みたい場合は、予めその名前で検索しておく必要があります。
	 * カラム名を指定しない場合(例えば ,,, など)は、カラムを書き飛ばします。
	 *
	 * @og.rev 4.0.0.0 (2005/12/31) 新規作成
	 *
	 * @param   clms 書込カラム列 (CSV形式)
	 */
	public void setColumns( final String clms ) {
		columns = nval( getRequestParameter( clms ),columns );
	}

	/**
	 * 【TAG】書き込み対象外のカラム列を、外部(タグ)よりCSV形式で指定します。
	 *
	 * @og.tag
	 * 指定するカラム名に対して、書き込み処理を行いません。
	 * ここで指定するカラム名は、検索したDBTableModel上に含まれる必要はありません。
	 * その場合は、ここでの指定は無視されます。
	 *
	 * @og.rev 6.1.0.0 (2014/12/26) omitNames 属性を追加
	 *
	 * @param   clms 書込対象外カラム列 (CSV形式)
	 */
	public void setOmitNames( final String clms ) {
		omitNames = nval( getRequestParameter( clms ),omitNames );
	}

	/**
	 * 【TAG】書込処理でコードリソースのラベル変換を行うかどうかを指定します
	 *		(初期値:USE_TABLE_WRITER_RENDERER[={@og.value org.opengion.hayabusa.common.SystemData#USE_TABLE_WRITER_RENDERER}])。
	 *
	 * @og.tag
	 * コードリソースをそのままの値で出力すると、数字や記号になり何が書かれているのか
	 * 不明になります。
	 * これは、コードリソースをラベルに変換して出力するかどうかを指定します。
	 * 当然、コードはユニークですが、ラベルはユニークになるかどうか保障はされていませんので
	 * TableReader 系で読み込む場合には、リスクが発生します。
	 * また、TableReader 系で読み込む場合にも、ラベルからコードを求める逆変換を行うように、
	 * setUseRenderer メソッドで指定する必要があります。
	 *
	 * 従来は、TableWriter 系に、TableWriter_Renderer 系のクラスを作って対応していましたが、
	 * このメソッドの属性値のフラグで、制御します。
	 * (初期値:システム定数のUSE_TABLE_WRITER_RENDERER[={@og.value org.opengion.hayabusa.common.SystemData#USE_TABLE_WRITER_RENDERER}])。
	 *
	 * @og.rev 5.2.1.0 (2010/10/01) 新規作成
	 *
	 * @param  flag ラベル変換可否 [true:する/false:しない]
	 * @see		org.opengion.hayabusa.common.SystemData#USE_TABLE_WRITER_RENDERER
	 */
	public void setUseRenderer( final String flag ) {
		useRenderer = nval( getRequestParameter( flag ),useRenderer );
	}

	/**
	 * 【TAG】(通常は使いません)EXCEL出力時のデータの書き込み開始行番号を設定します(初期値:0)。
	 *
	 * @og.tag
	 * EXCELのデータの書き込みの開始行番号を指定します。
	 * 属性名は、行を飛ばす処理ということで、readTable タグと同じ名称です。
	 * ファイルの先頭行が、０行としてカウントしますので、設定値は、読み飛ばす
	 * 件数になります。(１と指定すると、１行読み飛ばし、２行目から読み込みます。)
	 * 行の読み飛ばしと、カラムの読み飛ばし(columns)、refFileURL、refFilename、
	 * refSheetName とともに使用すれば、ある程度のレイアウト設定が可能です。
	 * 現時点では、Excel の場合のみ有効です。
	 *
	 * @og.rev 5.7.9.0 (2014/08/08) 新規作成
	 *
	 * @param	count 書き込み開始行番号
	 */
	public void setSkipRowCount( final String count ) {
		skipRowCount = nval( getRequestParameter( count ),skipRowCount );
	}

	/**
	 * 【TAG】EXCEL出力時に、データを書き込んだ範囲に罫線を入れるかどうかを指定します(初期値:false)。
	 *
	 * @og.tag
	 * データを書き込んでEXCELを作成しても、ノーマルのセルに値がセットされている
	 * だけなので、ある程度加工が必要です。
	 * そこで、データのセットされたセルに罫線を入れることで、それなりのデータが
	 * 出力された感じになります。
	 * この設定と、useAutoCellSize="true" で、セルの幅を自動調整すれば、見栄えが良くなります。
	 * なお、この機能は、TableWriter_Excel のみに実装します。
	 *
	 * @og.rev 6.0.2.0 (2014/09/19) 新規作成
	 *
	 * @param	useStyle 罫線追記 [true:する/false:しない]
	 * @see		#setUseAutoCellSize( String )
	 */
	public void setUseCellStyle( final String useStyle ) {
		useCellStyle = nval( getRequestParameter( useStyle ),useCellStyle );
	}

	/**
	 * 【TAG】EXCEL出力時に、セルの幅をデータの幅に自動的に合わせるかどうかを指定します(初期値:false)。
	 *
	 * @og.tag
	 * データを書き込んでEXCELを作成しても、ノーマルのセルに値がセットされている
	 * だけなので、ある程度加工が必要です。
	 * そこで、データのセットされたセルの幅を自動調整することで、それなりのデータが
	 * 出力された感じになります。
	 * この設定と、useCellStyle="true" で、セルの罫線を自動設定すれば、見栄えが良くなります。
	 * なお、この機能は、TableWriter_Excel のみに実装します。
	 *
	 * @og.rev 6.0.2.0 (2014/09/19) 新規作成
	 *
	 * @param	useAutoSize 幅の自動的調整 [true:する/false:しない]
	 * @see		#setUseCellStyle( String )
	 */
	public void setUseAutoCellSize( final String useAutoSize ) {
		useAutoCellSize = nval( getRequestParameter( useAutoSize ),useAutoCellSize );
	}

	/**
	 * 【TAG】EXCEL出力時に、セルの有効範囲を設定するかどうかを指定します(初期値:false)。
	 *
	 * @og.tag
	 * セルの有効範囲というのは、EXCELでの 空行、空列の存在しない範囲を指します(初期値:false)。
	 * 通常、空行でも、データとして残っている場合は、EXCELのセルオブジェクトは存在します。
	 * ここで、useActiveWorkbook="true" とすると、空行、空列を削除します。
	 * 
	 * 雛形を使用した場合は、データより多めに設定した計算などは、この処理で
	 * 削除されますので、データサイズにフィットさせることができます。
	 * なお、この機能は、TableWriter_Excel のみに実装します。
	 *
	 * @og.rev 6.0.2.0 (2014/09/19) 新規作成
	 *
	 * @param	useActWB 有効範囲の設定 [true:する/false:しない]
	 */
	public void setUseActiveWorkbook( final String useActWB ) {
		useActiveWorkbook = nval( getRequestParameter( useActWB ),useActiveWorkbook );
	}

	/**
	 * 【TAG】EXCEL出力時に、シート変更するキーとなるカラム名を指定します(このカラムの値がシート名になります)。
	 *
	 * @og.tag
	 * EXCEL帳票では、帳票雛形に、PAGE_BRAKE キーを設定しましたが、TableWriterでは、
	 * メモリ上のカラムの値が変更したときに、シート変更させることができます。
	 * このカラムの値がキーブレイクすると、新しいシートに書き出し始めます。
	 * シート名は、このカラムの値（キーブレイクする値）です。
	 * 
	 * 雛形ファイルを使用する場合、雛形シートもキーブレイクに伴って、＋１されます。
	 * つまり、雛形シートとデータシートは同時に変更されます。
	 * ただし、雛形シートは、最後の雛形シートで止まります。
	 * これは、雛形シートにヘッダー雛形とボディ雛形を用意しておき、最初のキーブレイクで
	 * ヘッダーからボディの書き込みにチェンジするイメージで使用できます。
	 * なお、この機能は、TableWriter_Excel のみに実装します。
	 *
	 * @og.rev 6.0.2.0 (2014/09/19) 新規作成
	 *
	 * @param	pbColumn シート変更カラム名
	 * @see		#setFileBreakColumn( String )
	 */
	public void setPageBreakColumn( final String pbColumn ) {
		pageBreakColumn = nval( getRequestParameter( pbColumn ),pageBreakColumn );
	}

	/**
	 * 【TAG】EXCEL出力時に、ファイル名を変更するキーとなるカラム名を指定します(このカラムの値がファイル名になります)。
	 *
	 * @og.tag
	 * EXCEL帳票では、メモリ上のカラムの値が変更したときに、ファイル名を変更することができます。
	 * このカラムの値がキーブレイクすると、新しいファイルに書き出し始めます。
	 * ファイル名は、このカラムの値（キーブレイクする値）＋ 元の出力ファイル名の拡張子(.xlsなど)です。
	 * この設定を使用する場合は、出力ファイル名は無視されますが、拡張子だけは使用されます。
	 * 
	 * 雛形ファイルを使用する場合、雛形ファイルもキーブレイクに伴って、再利用されます。
	 * 例えば、pageBreakColumn と併用する場合、キーブレイクで雛形シートも最初から適用になります。
	 * なお、この機能は、TableWriter_Excel のみに実装します。
	 *
	 * @og.rev 6.0.2.0 (2014/09/19) 新規作成
	 *
	 * @param	fbColumn ファイル名変更カラム名
	 * @see		#setPageBreakColumn( String )
	 */
	public void setFileBreakColumn( final String fbColumn ) {
		fileBreakColumn = nval( getRequestParameter( fbColumn ),fileBreakColumn );
	}

	/**
	 * 【TAG】EXCEL出力時に、Hyperlinkを作成するキーとなる「カラム名:値」形式のカラム名を指定します。
	 *
	 * @og.tag
	 * ここで、作成するハイパーリンクは、EXCELのシートに対するハイパーリンクです。
	 * それ以外のリンク(本当のURLやファイル等)のリンクは(今は)作成できません。
	 * ハイパーリンクを作成するには、①作成するカラム と ②作成する値 が必要です。
	 * このメソッドで設定するのは、「①:②」という形式でカラム名を指定します。
	 * ②がなければ、①と同じとします。
	 * ②の値のシートの存在有無は、無視します。ハイパーリンクを作成するシートを作成する前に
	 * ハイパーリンクを作成するケースが存在します。
	 * (例えば、各シートへのリンクを持った一覧を作成してから、明細の各シートを作成する様なケース)
	 * なお、この機能は、TableWriter_Excel のみに実装します。
	 *
	 * @og.rev 6.0.2.0 (2014/09/19) 新規作成
	 *
	 * @param	hlColumn Hyperlink作成カラム名 (「カラム名:値」形式)
	 */
	public void setHyperLinkColumn( final String hlColumn ) {
		hyperLinkColumn = nval( getRequestParameter( hlColumn ),hyperLinkColumn );
	}

	/**
	 * 【TAG】EXCEL出力時に、Sheet一覧を先頭Sheetに作成する場合のSheet名を指定します。
	 *
	 * @og.tag
	 * これは、Workbook に含まれる Sheet 一覧を作成する場合に、利用可能です。
	 * なお、この機能は、TableWriter_Excel のみに実装します。
	 *
	 * @og.rev 6.0.2.0 (2014/09/19) 新規作成
	 *
	 * @param	sheetName タイトルシート名
	 */
	public void setAddTitleSheet( final String sheetName ) {
		addTitleSheet = nval( getRequestParameter( sheetName ),addTitleSheet );
	}

	/**
	 * 【TAG】チェック行のみを対象にするかを指定します(初期値:true)。
	 *
	 * @og.tag
	 * falseを指定する事で、前画面でチェックした行のみを出力します。
	 * 初期値は全ての行を出力(true)です。
	 * 出力されるファイルの行番号は、1から振りなおされます。
	 *
	 * @og.rev 5.5.5.5 (2012/08/24) 新規作成
	 *
	 * @param   flag 対象にするか [true:全件/false:対象のみ]
	 */
	public void setSelectedAll( final String flag ) {
		selectedAll = nval( getRequestParameter( flag ),selectedAll );		// 5.5.9.1 (2012/12/07) バグ対応
	}

	/**
	 * 【TAG】事前にファイルを削除するかどうかを指定します(初期値:false)。
	 *
	 * @og.tag
	 * 出力元の DBTableModel が、存在しないか、０件の場合、過去に作成したファイルが
	 * そのまま、残っていると、不要なトラブルのもととなる為、実行前に削除するか
	 * どうかを指定できます。
	 * なお、fileAppend="true"（追加モードで書き込むを指定)時には、データが０件でも
	 * 事前削除は行いません。
	 * 初期値は互換性のため削除しない(false)です。
	 *
	 * @og.rev 5.5.9.1 (2012/12/07) 新規作成
	 *
	 * @param   flag 事前削除するかどうか [true:削除する/false:削除しない]
	 */
	public void setBeforeDelete( final String flag ) {
		beforeDelete = nval( getRequestParameter( flag ),beforeDelete );
	}

	/**
	 * WriteTable オブジェクトに渡すパラメータオブジェクトをセットします。
	 *
	 * WriteTableParamTag クラスよりセットされます。
	 *
	 * @og.rev 4.0.0.0 (2005/02/28) 新規追加
	 *
	 * @param   entry HybsEntryオブジェクト
	 */
	protected void addParam( final HybsEntry entry ) {
		if( param == null ) { param = new ArrayList<HybsEntry>(); }
		param.add( entry );
	}

	/**
	 * シリアライズ用のカスタムシリアライズ書き込みメソッド。
	 *
	 * @og.rev 4.0.0.0 (2006/09/31) 新規追加
	 * @serialData 一部のオブジェクトは、シリアライズされません。
	 *
	 * @param	strm	ObjectOutputStreamオブジェクト
	 * @throws IOException	入出力エラーが発生した場合
	 */
	private void writeObject( final ObjectOutputStream strm ) throws IOException {
		strm.defaultWriteObject();
	}

	/**
	 * シリアライズ用のカスタムシリアライズ読み込みメソッド
	 *
	 * ここでは、transient 宣言された内部変数の内、初期化が必要なフィールドのみ設定します。
	 *
	 * @og.rev 4.0.0.0 (2006/09/31) 新規追加
	 * @serialData 一部のオブジェクトは、シリアライズされません。
	 *
	 * @param	strm	ObjectInputStreamオブジェクト
	 * @see #release2()
	 * @throws IOException	シリアライズに関する入出力エラーが発生した場合
	 * @throws ClassNotFoundException	クラスを見つけることができなかった場合
	 */
	private void readObject( final ObjectInputStream strm ) throws IOException , ClassNotFoundException {
		strm.defaultReadObject();
	}

	/**
	 * このオブジェクトの文字列表現を返します。
	 * 基本的にデバッグ目的に使用します。
	 *
	 * @return このクラスの文字列表現
	 * @og.rtnNotNull
	 */
	@Override
	public String toString() {
		return ToString.title( this.getClass().getName() )
				.println( "VERSION"			,VERSION		)
				.println( "BASE_URL"		,BASE_URL		)
				.println( "separator"		,separator		)
				.println( "headerSequence"	,headerSequence	)
				.println( "fileURL"			,fileURL 		)
				.println( "filename"		,filename		)
				.println( "refFileURL"		,refFileURL		)
				.println( "refFilename"		,refFilename 	)
				.println( "encode"			,encode			)
				.println( "writerClass"		,writerClass 	)
				.println( "fileAppend"		,fileAppend		)
				.println( "direct"			,direct			)
				.println( "zip"				,zip 			)
				.println( "zipFilename"		,zipFilename	)		// 6.0.4.0 (2014/11/28)
				.println( "disposition"		,disposition	)
				.println( "tableId"			,tableId 		)
				.println( "sheetName"		,sheetName		)
				.println( "refSheetName"	,refSheetName	)
				.println( "useNumber"		,useNumber		)
				.println( "Other..."	,getAttributes().getAttribute() )
				.fixForm().toString() ;
	}
}
