/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.calendar;

import org.opengion.hayabusa.resource.CalendarQuery;
import org.opengion.hayabusa.common.HybsSystemException;

/**
 * カレンダＤＢ(TP637)の検索QUERYを定義したクラスです。
 *
 * QUERY は、このオブジェクトを、toString() して求めることとします。
 * 本来は、これらのクラスの共通インターフェースを作成して、getQuery() などのメソッドを
 * 介して取得すべきですが、Object の共通クラスを利用することとします。
 *
 * @og.rev 3.6.0.0 (2004/09/17) 新規作成
 * @og.group リソース管理
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public final class CalendarQuery_TP637 implements CalendarQuery {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "4.0.0.0 (2005/08/31)" ;

	/** カレンダＤＢの読み込みのクエリー(TP637) */
	public static final String QUERY =
				"select YM,DAY1,DAY2,DAY3,DAY4,DAY5,DAY6,DAY7,DAY8,DAY9,DAY10,"
				+ "DAY11,DAY12,DAY13,DAY14,DAY15,DAY16,DAY17,DAY18,DAY19,DAY20,"
				+ "DAY21,DAY22,DAY23,DAY24,DAY25,DAY26,DAY27,DAY28,DAY29,DAY30,DAY31"
				+ " from TP637 where CDK=? and WC=? and WS=? and KBCAL=? and KBREC='1'"
				+ " order by YM" ;

	/**
	 * ４つの引数を受け取り、整合性チェックを行います。
	 * 引数は、各クラスによって使用するカラム名(意味)が異なります。
	 * また、すべての引数をチェックするのではなく、クラス毎に、チェックする
	 * カラムの数は、異なります。
	 * 引数が正しくない場合は、HybsSystemException を発行します。
	 *
	 * @param	arg1  	データベース検索時の第１引数(CDK:生産工場)
	 * @param	arg2  	データベース検索時の第２引数(WC:ＷＣ)
	 * @param	arg3  	データベース検索時の第３引数(WS:構成)
	 * @param	arg4  	データベース検索時の第４引数(KBCAL:カレンダ区分)
	 *
	 * @return	入力パラメータに応じた配列文字列(cdk,wc,ws,kbcal)
	 * @og.rtnNotNull
	 * @throws HybsSystemException CLNBK(カレンダ区分) が設定されていない場合。
	 */
	public String[] checkArgment( final String arg1,final String arg2,final String arg3,final String arg4 ) {
		final String cdk   = arg1 ;
		String wc    = arg2 ;
		String ws    = arg3 ;
		final String kbcal = arg4 ;

		if( kbcal == null || kbcal.isEmpty() ) {
			final String errMsg = "CalendarQuery_TP637 クラスの第４引数 "
						+ "KBCAL(カレンダ区分)が指定されていません。"
						+ "CDK=" + cdk + ", WC=" + wc + ", WS=" + ws + ", KBCAL=" + kbcal ;
			throw new HybsSystemException( errMsg );
		}

		// TP637 ＤＢ定義書 参照
		switch( kbcal.charAt(0) ) {
			case '1': wc = cdk;  ws = cdk ; break ;
			case '2': wc = cdk;  ws = wc  ; break ;
			default : break ;
		}

		if( cdk == null || cdk.isEmpty() ||
			wc == null || wc.isEmpty() ||
			ws == null || ws.isEmpty() ) {
				final String errMsg = "CalendarQuery_TP637 クラスの引数キーの、"
							+ "CDK,WC,WS が、設定されていません。"
							+ "CDK=" + cdk + ", WC=" + wc + ", WS=" + ws + ", KBCAL=" + kbcal ;
				throw new HybsSystemException( errMsg );
		}

		return new String[] { cdk,wc,ws,kbcal } ;
	}

	/**
	 * データベース検索の為の Select 文を返します。
	 * 引数リストとともに、使用します。
	 *
	 * @return データベース検索の為の Select 文
	 * @og.rtnNotNull
	 */
	public String getQuery() {
		return QUERY;
	}

	/**
	 * データベースの持ち方を指定します。
	 * 持ち方がフラット(横持ち=1～31の日付をカラムで持つ)の場合、trueを返します。
	 * 縦持ち(日付単位で、行情報として持つ)場合は、false です。
	 *
	 * @return DBの持ち方がフラット(横持ち=1～31の日付をカラムで持つ)の場合、true
	 *
	 */
	public boolean isFlatTable() {
		return true;
	}
}
