/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.column;

import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.db.AbstractEditor;
import org.opengion.hayabusa.db.CellEditor;
import org.opengion.hayabusa.db.DBColumn;
import org.opengion.hayabusa.db.Selection;
import org.opengion.hayabusa.db.SelectionFactory;			// 6.0.4.0 (2014/11/28)
import org.opengion.fukurou.util.XHTMLTag;
import org.opengion.fukurou.util.Attributes;
import org.opengion.fukurou.util.TagBuffer;

import static org.opengion.fukurou.util.StringUtil.isNull;				// 6.1.1.0 (2015/01/17)

/**
 * INMENU エディターは、コードリソースに対応したプルダウンメニューと、
 * テキストフィールドによる入力の両方をサポートする、編集に使用するクラスです。
 *
 * JavaScript によりテキストフィールドとメニュー(コンボボックス)を重ね合わせて
 * 表示しておき、メニューで選択した値を、テキストフィールドに設定しています。
 * このエディタを使用するには、jsp/common/inputMenu.js を予め使用できるように
 * 設定しておく必要があります。
 *
 * このエディタはeventColumnに対応していません。
 *
 *  カラムの表示に必要な属性は, DBColumn オブジェクト より取り出します。
 * このクラスは、DBColumn オブジェクト毎に１つ作成されます。
 *
 * @og.rev 3.5.6.2 (2004/07/05) 新規作成
 * @og.group データ編集
 *
 * @version  4.0
 * @author	 Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class Editor_INMENU extends AbstractEditor {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "6.0.4.0 (2014/11/28)" ;

	private static final String SEL1 = "<script type=\"text/javascript\">makeInputMenu('" ;
	private static final String SEL2 = "');</script>" ;

	/** セレクションオブジェクト */
//	protected Selection	selection ;
	private final Selection	selection	;		// 6.0.4.0 (2014/11/28) サブクラスからの参照を廃止
	private final boolean addNoValue	;
	private final boolean seqFlag		;		// 3.6.0.6 (2004/10/22)
	private final TagBuffer selTagBuffer = new TagBuffer() ;

	private String errMsg	;					// 6.0.4.0 (2014/11/28)

	/**
	 * デフォルトコンストラクター。
	 * このコンストラクターで、基本オブジェクトを作成します。
	 *
	 * @og.rev 3.6.0.6 (2004/10/22) シーケンスアクセス機能(seqFlag)を追加します
	 */
	public Editor_INMENU() {
		// 4.3.4.4 (2009/01/01)
		selection  = null;
		addNoValue = false;		// 3.5.5.7 (2004/05/10)
		seqFlag    = false;		// 3.6.0.6 (2004/10/22)
	}

	/**
	 * コンストラクター。
	 *
	 * @og.rev 3.6.0.6 (2004/10/22) シーケンスアクセス機能(seqFlag)を追加します
	 * @og.rev 4.0.0.0 (2005/01/31) SelectionFactory ではなく、直接 Selection_CODE を作成。
	 * @og.rev 4.0.0.0 (2006/11/24) TextField分の属性設定
	 * @og.rev 4.0.0.0 (2007/11/07) SelectionオブジェクトをDBColumnから取得
	 * @og.rev 5.6.3.0 (2013/04/01) プルダウンのonChangeの設定場所を変更
	 * @og.rev 6.0.4.0 (2014/11/28) selection が null の場合、警告表示します。
	 *
	 * @param	clm	DBColumnオブジェクト
	 */
	protected Editor_INMENU( final DBColumn clm ) {
		super( clm );
		tagBuffer.add( XHTMLTag.inputAttri( attributes ) );

		addNoValue = clm.isAddNoValue() ;		// 3.5.5.7 (2004/05/10)
		seqFlag    = "SEQ".equals( clm.getEditorParam() );	// 3.6.0.6 (2004/10/22)

		final String	disabled = clm.isWritable() ? null : "disabled" ;

		// 6.1.1.0 (2015/01/17) Attributesの連結記述
		final Attributes selAttri = new Attributes()
				.set( "disabled" ,disabled )
				.set( clm.getEditorAttributes() )			// #addAttributes( Attributes ) の代替え
				.set( "onChange" ,"selChanged(this);" );	// INMENU 特有のJavaScript 5.6.3.0 (2013/04/01) 場所移動

//		final Attributes selAttri = new Attributes();
//		selAttri.set( "disabled" ,disabled );
//
//		selAttri.addAttributes( clm.getEditorAttributes() );
//		selAttri.set( "onChange" ,"selChanged(this);" );	// INMENU 特有のJavaScript 5.6.3.0 (2013/04/01) 場所移動

		// 6.1.1.0 (2015/01/17) TagBufferの連結記述
		selTagBuffer.add( XHTMLTag.selectAttri( selAttri ) )
					.add( selAttri.get( "optionAttributes" ) );				// 6.0.4.0 (2014/11/28)

//		selTagBuffer.add( XHTMLTag.selectAttri( selAttri ) );
//
//		final String optAttr = selAttri.get( "optionAttributes" );	// 6.0.4.0 (2014/11/28)
//		selTagBuffer.add( optAttr );								// 6.0.4.0 (2014/11/28)

		// 6.0.4.0 (2014/11/28) selection は、Column から取得するのではなく、Factory で作成する。
//		selection = clm.getSelection();							// 4.0.0.0 (2007/11/07)
		selection = SelectionFactory.newSelection( "MENU",clm.getCodeData() );

		// 6.0.4.0 (2014/11/28) selection が null の場合、警告表示します。
		if( selection == null ) {
			errMsg = "codeData が未設定です。"
								+ " name="  + name
								+ " label=" + clm.getLabel()
								+ " editType="  + clm.getEditor() ;
			System.out.println( errMsg );
		}
	}

	/**
	 * 各オブジェクトから自分のインスタンスを返します。
	 * 自分自身をキャッシュするのか、新たに作成するのかは、各サブクラスの実装に
	 * まかされます。
	 *
	 * @param	clm	DBColumnオブジェクト
	 *
	 * @return	CellEditorオブジェクト
	 * @og.rtnNotNull
	 */
	public CellEditor newInstance( final DBColumn clm ) {
		return new Editor_INMENU( clm );
	}

	/**
	 * データの編集用文字列を返します。
	 *
	 * @og.rev 3.6.0.6 (2004/10/22) シーケンスアクセス機能(seqFlag)を追加します
	 * @og.rev 3.8.5.3 (2006/06/30) 位置を絶対位置指定(position:absolute;)
	 * @og.rev 4.3.7.2 (2009/06/15) 属性でidが出力される場合は、idを出力しない
	 * @og.rev 6.0.4.0 (2014/11/28) selection が null の場合、INMENU は作成しません。
	 *
	 * @param	value 入力値
	 *
	 * @return	データの編集用文字列
	 * @og.rtnNotNull
	 */
	@Override
	public String getValue( final String value ) {
		// input タグの作成
		// 6.1.1.0 (2015/01/17) TagBufferの連結記述
		final String intag = new TagBuffer( "input" )
						.add( "name"	, name )
						.add( "id"		, name , isNull( attributes.get( "id" ) ) )		// 4.3.7.2 (2009/06/15)
						.add( "value"	, value )
						.add( "size"	, size1 )
						.add( tagBuffer.makeTag() )
						.makeTag();

		// 6.0.4.0 (2014/11/28) selection が null の場合、INMENU は作成しません。
		if( selection == null ) {
			return intag + CR;
		}

//		// input タグの作成
//		final TagBuffer intag = new TagBuffer( "input" );
//		intag.add( "name"    , name );
////		if( attributes.get( "id" ) == null || attributes.get( "id" ).length() == 0 ) { // 4.3.7.2 (2009/06/15)
//		if( attributes.get( "id" ) == null || attributes.get( "id" ).isEmpty() ) { // 4.3.7.2 (2009/06/15)
//			intag.add( "id"      , name );		// INMENU 特有のJavaScript用のキー
//		}
//		intag.add( "value"   , value );
//		intag.add( "size"    , size1 );
//		intag.add( tagBuffer.makeTag() );
////		intag.add( optAttr );
//
//		// 6.0.4.0 (2014/11/28) selection が null の場合、INMENU は作成しません。
//		if( selection == null ) {
//			return intag.makeTag() + CR;
//		}

		// select タグの作成
		// 6.1.1.0 (2015/01/17) TagBufferの連結記述
		final String seltag = new TagBuffer( "select" )
						.add( "id"		, name + ".sel" )			// INMENU 特有のJavaScript用のキー
						.add( "style"	, "position:absolute;" )	// 3.8.5.3 (2006/06/30) 位置を絶対位置指定
						.add( selTagBuffer.makeTag() )
						.addBody( Selection.NO_VALUE_OPTION , addNoValue )					// 5.5.1.0 (2012/04/03)
						.addBody( selection.getOption( value,seqFlag ) )
						.makeTag();

		return intag + CR + seltag + CR + SEL1 + name + SEL2;

//		// select タグの作成
//		final TagBuffer seltag = new TagBuffer( "select" );
//		seltag.add( "id"      , name + ".sel" );	// INMENU 特有のJavaScript用のキー
//		seltag.add( "style"   , "position:absolute;" );	// 3.8.5.3 (2006/06/30) 位置を絶対位置指定
//		seltag.add( selTagBuffer.makeTag() );
////		seltag.add( optAttr );		// 3.5.5.8 (2004/05/20)
//
//		if( addNoValue ) {
//			seltag.setBody( Selection.NO_VALUE_OPTION + selection.getOption( value,seqFlag ) );
//		}
//		else {
//			seltag.setBody( selection.getOption( value,seqFlag ) );
//		}
//
//		return intag.makeTag() + CR +
//				seltag.makeTag() + CR +
//				SEL1 + name + SEL2;
	}

	/**
	 * name属性を変えた、データ表示/編集用のHTML文字列を作成します。
	 * テーブル上の name に 行番号を付加して、名前_行番号 で登録するキーを作成し,
	 * リクエスト情報を１つ毎のフィールドで処理できます。
	 *
	 * @og.rev 3.6.0.6 (2004/10/22) シーケンスアクセス機能(seqFlag)を追加します
	 * @og.rev 3.8.5.1 (2006/04/28) makeInputMenu 呼び出し時の引数記述ミスを修正
	 * @og.rev 3.8.5.3 (2006/06/30) 位置を絶対位置指定(position:absolute;)
	 * @og.rev 4.3.7.2 (2009/06/15) 属性でidが出力される場合は、idを出力しない
	 * @og.rev 6.0.4.0 (2014/11/28) selection が null の場合、INMENU は作成しません。
	 *
	 * @param	row   行番号
	 * @param	value 入力値
	 *
	 * @return	データ表示/編集用の文字列
	 * @og.rtnNotNull
	 */
	@Override
	public String getValue( final int row,final String value ) {

		final String name2 = name + HybsSystem.JOINT_STRING + row ;

		// input タグの作成
		// 6.1.1.0 (2015/01/17) TagBufferの連結記述
		final String intag = new TagBuffer( "input" )
						.add( "name"    , name2 )
						.add( "id"		, name2 , isNull( attributes.get( "id" ) ) )		// INMENU 特有のJavaScript用のキー
						.add( "value"   , value )
						.add( "size"    , size2 )
						.add( tagBuffer.makeTag() )
						.makeTag( row,value );

		// 6.0.4.0 (2014/11/28) selection が null の場合、INMENU は作成しません。
		if( selection == null ) {
			return intag + CR;
		}

//		final TagBuffer intag = new TagBuffer( "input" );
//		intag.add( "name"    , name2 );
////		if( attributes.get( "id" ) == null || attributes.get( "id" ).length() == 0 ) { // 4.3.7.2 (2009/06/15)
//		if( attributes.get( "id" ) == null || attributes.get( "id" ).isEmpty() ) { // 4.3.7.2 (2009/06/15)
//			intag.add( "id"      , name2 );		// INMENU 特有のJavaScript用のキー
//		}
//		intag.add( "value"   , value );
//		intag.add( "size"    , size2 );
//		intag.add( tagBuffer.makeTag() );
////		intag.add( optAttr );
//
//		// 6.0.4.0 (2014/11/28) selection が null の場合、INMENU は作成しません。
//		if( selection == null ) {
//			return intag.makeTag( row,value ) + CR;
//		}

		// select タグの作成
		// 6.1.1.0 (2015/01/17) TagBufferの連結記述
		final String seltag = new TagBuffer( "select" )
						.add( "id"		, name2 + ".sel" )			// INMENU 特有のJavaScript用のキー
						.add( "style"	, "position:absolute;" )	// 3.8.5.3 (2006/06/30) 位置を絶対位置指定
						.add( selTagBuffer.makeTag() )
						.addBody( Selection.NO_VALUE_OPTION , addNoValue )					// 5.5.1.0 (2012/04/03)
						.addBody( selection.getOption( value,seqFlag ) )
						.makeTag( row,value );

		return intag + CR + seltag + CR + SEL1 + name2 + SEL2;

//		// select タグの作成
//		final TagBuffer seltag = new TagBuffer( "select" );
//		seltag.add( "id"      , name2 + ".sel" );		// INMENU 特有のJavaScript用のキー
//		seltag.add( "style"   , "position:absolute;" );	// 3.8.5.3 (2006/06/30) 位置を絶対位置指定
//		seltag.add( selTagBuffer.makeTag() );
//
//		if( addNoValue ) {
//			seltag.setBody( Selection.NO_VALUE_OPTION + selection.getOption( value,seqFlag ) );
//		}
//		else {
//			seltag.setBody( selection.getOption( value,seqFlag ) );
//		}

//		return intag.makeTag( row,value ) + CR +
//				seltag.makeTag( row,value ) + CR +
//				SEL1 + name2 + SEL2;
	}
}
