/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.column;

import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.db.AbstractEditor;
import org.opengion.hayabusa.db.CellEditor;
import org.opengion.hayabusa.db.DBColumn;
import org.opengion.fukurou.util.StringFormat;
import org.opengion.fukurou.util.XHTMLTag;
import org.opengion.fukurou.util.ApplicationInfo;
import org.opengion.fukurou.db.DBUtil;

/**
 * QUERY エディターは、編集パラメータで指定された SQL文の実行結果をテキストエリアに表示する
 * クラスで、元のValue を、$1 として使用可能です。
 * 又、$Cで自身のカラム名が使用可能です。
 *
 *  カラムの表示に必要な属性は, DBColumn オブジェクト より取り出します。
 * このクラスは、DBColumn オブジェクト毎に１つ作成されます。
 *
 * @og.rev 4.0.0.0 (2006/04/01) 新規追加
 * @og.group データ編集
 *
 * @version  4.0
 * @author	 Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class Editor_QUERY extends AbstractEditor {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "5.7.9.0 (2014/08/08)" ;

	private final String query ;
	private final String dbid ;

	/** コネクションにアプリケーション情報を追記するかどうか指定 */
	public static final boolean USE_DB_APPLICATION_INFO  = HybsSystem.sysBool( "USE_DB_APPLICATION_INFO" ) ;

	// 3.8.7.0 (2006/12/15) アクセスログ取得の為,ApplicationInfoオブジェクトを設定
	private final ApplicationInfo appInfo;
	private static final String SYSTEM_ID  = HybsSystem.sys( "SYSTEM_ID" ) ;

	/**
	 * デフォルトコンストラクター。
	 * このコンストラクターで、基本オブジェクトを作成します。
	 *
	 * @og.rev 3.8.7.0 (2006/12/15) アクセスログ取得の為,ApplicationInfoオブジェクトを設定
	 */
	public Editor_QUERY() {
		// 4.3.4.4 (2009/01/01)
		query   = null;
		dbid	= null;
		appInfo = makeApplicationInfo( null );
		name	= null; // 4.3.4.0 (2008/12/01)
	}

	/**
	 * コンストラクター。
	 *
	 * @og.rev 3.8.7.0 (2006/12/15) アクセスログ取得の為,ApplicationInfoオブジェクトを設定
	 *
	 * @param	clm	DBColumnオブジェクト
	 */
	private Editor_QUERY( final DBColumn clm ) {
		super( clm );
		query = clm.getEditorParam();
		dbid  = clm.getDbid();
		tagBuffer.add( XHTMLTag.inputAttri( attributes ) );
		appInfo = makeApplicationInfo( clm.getName() );
		name  = clm.getName(); //4.3.4.0 (2008/12/01)
	}

	/**
	 * アクセスログ取得の為,ApplicationInfoオブジェクトを作成します。
	 *
	 * @og.rev 3.8.7.0 (2006/12/15) 新規作成
	 *
	 * @param	name	オブジェクト
	 *
	 * @return ApplicationInfoオブジェクト
	 */
	private ApplicationInfo makeApplicationInfo( final String name ) {
		ApplicationInfo infoTemp = null;

		// 3.8.7.0 (2006/12/15) アクセスログ取得の為,ApplicationInfoオブジェクトを設定
		if( USE_DB_APPLICATION_INFO ) {
			infoTemp = new ApplicationInfo();
			// ユーザーID,IPアドレス,ホスト名
			infoTemp.setClientInfo( SYSTEM_ID,HybsSystem.HOST_ADRS,HybsSystem.HOST_NAME );
			// 画面ID,操作,プログラムID
			infoTemp.setModuleInfo( "Editor_QUERY",null,name );
		}

		return infoTemp ;
	}

	/**
	 * 各オブジェクトから自分のインスタンスを返します。
	 * 自分自身をキャッシュするのか、新たに作成するのかは、各サブクラスの実装に
	 * まかされます。
	 *
	 * @param	clm	DBColumnオブジェクト
	 *
	 * @return	CellEditorオブジェクト
	 * @og.rtnNotNull
	 */
	public CellEditor newInstance( final DBColumn clm ) {
		return new Editor_QUERY( clm );
	}

	/**
	 * データの編集用文字列を返します。
	 *
	 * @og.rev 3.8.7.0 (2006/12/15) アクセスログ取得の為,ApplicationInfoオブジェクトを設定
	 * @og.rev 4.3.4.0 (2008/12/01) $C対応
	 * @og.rev 5.7.9.0 (2014/08/08) DBUtil.dbExecute 実行時エラーを回避
	 *
	 * @param	value 入力値
	 *
	 * @return	データの編集用文字列
	 */
	@Override
	public String getValue( final String value ) {
		// StringFormat format = new StringFormat( query,value );
		final StringFormat format = new StringFormat( query,value, name ); // 4.3.4.0 (2008/12/01)
		final String str = format.format();

		// 5.7.9.0 (2014/08/08) DBUtil.dbExecute 実行時エラーを回避
//		String[][] rtn = DBUtil.dbExecute( str,null,appInfo,dbid );
//		return super.getValue( rtn[0][0] );

		String rtnVal = (value == null) ? "" : value ;		// 本来、QUERYで変換すべきだが、元の値を設定する。
		try {
			final String[][] rtn = DBUtil.dbExecute( str,null,appInfo,dbid );
			rtnVal = ( rtn == null || rtn[0] == null || rtn[0][0] == null ) ? "" : rtn[0][0];
		}
		catch( RuntimeException ex ) {
			final String errMsg = "SQL文の処理に失敗しました。" + CR
								+ "query=" + query + " , inValue=" + value + " , name=" + name
								+ CR
								+ ex.getMessage() ;
			System.err.println( errMsg );
		}

		return super.getValue( rtnVal );
	}

	/**
	 * name属性を変えた、データ表示/編集用のHTML文字列を作成します。
	 * テーブル上の name に 行番号を付加して、名前_行番号 で登録するキーを作成し,
	 * リクエスト情報を１つ毎のフィールドで処理できます。
	 *
	 * @og.rev 3.8.7.0 (2006/12/15) アクセスログ取得の為,ApplicationInfoオブジェクトを設定
	 * @og.rev 4.3.4.0 (2008/12/01) $C対応
	 * @og.rev 5.7.9.0 (2014/08/08) DBUtil.dbExecute 実行時エラーを回避
	 *
	 * @param	row   行番号
	 * @param	value 入力値
	 *
	 * @return	データ表示/編集用の文字列
	 */
	@Override
	public String getValue( final int row,final String value ) {
		// StringFormat format = new StringFormat( query,value );
		final StringFormat format = new StringFormat( query,value,name ); // 4.3.4.0 (2008/12/01)
		final String str = format.format();

		// 5.7.9.0 (2014/08/08) DBUtil.dbExecute 実行時エラーを回避
//		String[][] rtn = DBUtil.dbExecute( str,null,appInfo,dbid );
//		return super.getValue( row,rtn[0][0] );

		String rtnVal = (value == null) ? "" : value ;		// 本来、QUERYで変換すべきだが、元の値を設定する。
		try {
			final String[][] rtn = DBUtil.dbExecute( str,null,appInfo,dbid );
			rtnVal = ( rtn == null || rtn[0] == null || rtn[0][0] == null ) ? "" : rtn[0][0];
		}
		catch( RuntimeException ex ) {
			final String errMsg = "SQL文の処理に失敗しました。" + CR
								+ "row=" + row + " , query=" + query + " , inValue=" + value + " , name=" + name
								+ CR
								+ ex.getMessage() ;
			System.err.println( errMsg );
		}

		return super.getValue( row,rtnVal );
	}
}
