/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.column;

import org.opengion.hayabusa.db.AbstractRenderer;
import org.opengion.hayabusa.db.CellRenderer;
import org.opengion.hayabusa.db.DBColumn;

// import org.opengion.fukurou.util.StringUtil;		// 6.0.2.4 (2014/10/17)

/**
 * HHM レンデラーは、カラムのデータを時：分に分けて表示する場合に
 * 使用するクラスです。
 * HMレンデラーと異なり、元の値は、「分」であり、それを、時間:分 に
 * 分解表示します。
 *
 * カラムの表示に必要な属性は, DBColumn オブジェクト より取り出します。
 * このクラスは、DBColumn オブジェクト毎に１つ作成されます。
 *
 * @og.group データ表示
 * @og.rev 6.0.2.3 (2014/10/10) 新規作成
 *
 * @version  5.0
 * @author	 Kazuhiko Hasegawa
 * @since    JDK6.0,
 */
public class Renderer_HHM extends AbstractRenderer {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "6.0.4.0 (2014/11/28)" ;

	private static final CellRenderer dbCell = new Renderer_HHM() ; // 5.6.5.1 (2013/06/14)

	/**
	 * 各オブジェクトから自分のインスタンスを返します。
	 * 自分自身をキャッシュするのか、新たに作成するのかは、各サブクラスの実装に
	 * まかされます。
	 *
	 * @param	clm	DBColumnオブジェクト
	 *
	 * @return	CellRendererオブジェクト
	 * @og.rtnNotNull
	 */
	public CellRenderer newInstance( final DBColumn clm ) {
		return dbCell;
	}

	/**
	 * データの表示用文字列を返します。
	 * ここでは、null か ゼロ文字列の場合は、ゼロ文字列を返します。
	 * HMレンデラーと異なり、元の値は、「分」であり、それを、時間:分 に
	 * 分解表示します。
	 * 数値化して、60 で割り算しますので、数値化できない場合は、エラーになります。
	 *
	 * @og.rev 6.0.2.3 (2014/10/10) 新規追加
	 * @og.rev 6.0.2.4 (2014/10/17) 分が、１桁の場合、XX:X 表記になってしまうバグを修正
	 * @og.rev 6.0.4.0 (2014/11/28) ロジックの共通化
	 *
	 * @param	value 入力値
	 *
	 * @return	データの表示用文字列
	 */
	@Override
	public String getValue( final String value ) {
		return getValue( value , true );

//		if( value == null || value.length() == 0 ) { return ""; }
//
//		String rtn ;
////		int  hh = 0;
////		int  mm = 0;
//		try {
//			int val = Integer.parseInt( value );
////			rtn = val / 60 + ":" + val % 60 ;			// 時:分
//			rtn = val / 60 + ":" + StringUtil.intFill( val % 60 , 2 ) ;			// 6.0.2.4 (2014/10/17)
//		}
//		catch( NumberFormatException ex ) {
//			rtn = "<span class=\"error\">" + value + "</span>";
//		}
//
//		return rtn;
	}

	/**
	 * データ出力用の文字列を作成します。
	 * ファイル等に出力する形式を想定しますので、HTMLタグを含まない
	 * データを返します。
	 * 基本は、#getValue( String ) をそのまま返します。
	 *
	 * @og.rev 6.0.4.0 (2014/11/28) データ出力用のレンデラー
	 *
	 * @param   value 入力値
	 *
	 * @return  データ出力用の文字列
	 * @see		#getValue( String )
	 */
	@Override
	public String getWriteValue( final String value ) {
		return getValue( value , false );
	}

	/**
	 * データ表示用/出力用の文字列を作成します。
	 * 第二引数の isView == true で、データ表示用文字列を、false で
	 * データ出力用の文字列を作成します。
	 * 処理の共通化を行うためのメソッドです。
	 *
	 * @og.rev 6.0.4.0 (2014/11/28) ロジックの共通化
	 *
	 * @param   value  入力値
	 * @param   isView データ表示用かどうか(true:表示用/false:出力用)
	 *
	 * @return  データ表示用/出力用の文字列
	 * @og.rtnNotNull
	 * @see		#getValue( String )
	 */
	private String getValue( final String value , final boolean isView ) {
//		if( value == null || value.length() == 0 ) { return ""; }
		if( value == null || value.isEmpty() ) { return ""; }

		String rtn = value;			// 初期値:isView = false の時の値
		try {
			final int val = Integer.parseInt( value );
			final int mm  = val % 60;

			rtn = val/60 + ":" + (mm<10 ? "0" + mm : mm) ;			// 時:分
		}
		catch( NumberFormatException ex ) {
			if( isView ) {
				rtn = "<span class=\"error\">" + value + "</span>";
			}
		}

		return rtn;
	}
}
