/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.view;

// import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.fukurou.util.StringUtil;

/**
 * 行と列を入れ替えて表示する、テーブル回転表示クラスです。
 *
 * このクラスは、表示のみ実行可能です。旧ヘッダー部分は、第一カラムに表示されます。
 * このビューでは、行と列が入れ替わって表示している為、登録はできません。
 *
 * AbstractViewForm により、setter/getterメソッドのデフォルト実装を提供しています。
 * 各HTMLのタグに必要な setter/getterメソッドのみ，追加定義しています。
 *
 * AbstractViewForm を継承している為,ロケールに応じたラベルを出力させる事が出来ます。
 *
 * @og.rev 3.5.3.0 (2003/10/27) 新規作成
 * @og.group 画面表示
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class ViewForm_HTMLRotationTable extends ViewForm_HTMLTable  {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "5.1.6.0 (2010/05/01)" ;

	private boolean	 noClass ;

	/**
	 * DBTableModel から HTML文字列を作成して返します。
	 * startNo(表示開始位置)から、pageSize(表示件数)までのView文字列を作成します。
	 * 表示残りデータが pageSize 以下の場合は,残りのデータをすべて出力します。
	 *
	 * @og.rev 3.5.3.1 (2003/10/31) BgColorCycleClass の設定不具合修正。
	 * @og.rev 3.5.6.4 (2004/07/16) ヘッダーとボディー部をJavaScriptで分離
	 * @og.rev 3.8.8.5 (2007/03/09) 表示の仕方を修正しました。
	 * @og.rev 4.0.0.0 (2005/01/31) 新規作成(getColumnClassName ⇒ getColumnDbType)
	 *
	 * @param  startNo    表示開始位置
	 * @param  pageSize   表示件数
	 *
	 * @return  DBTableModelから作成された HTML文字列
	 * @og.rtnNotNull
	 */
	@Override
	public String create( final int startNo, final int pageSize )  {
		if( getRowCount() == 0 ) { return ""; }	// 暫定処置

		final int lastNo = getLastNo( startNo, pageSize );

		final StringBuilder out = new StringBuilder( BUFFER_LARGE )
			.append( getCountForm( startNo,pageSize ) )
			.append( getHeader() )
			.append("<tbody>").append( CR );

		String clmCls = "<td>";
		int bgClrCnt = 0;
		final int clmCnt = getColumnCount();	// 3.5.5.7 (2004/05/10)
		for(int column = 0; column < clmCnt; column++) {
			if( !isColumnDisplay( column ) ) { continue; }
			final int dummyRow = bgClrCnt++ ;		// 3.8.8.5 (2007/03/09)

			out.append("<tr").append( getBgColorCycleClass( dummyRow ) ).append('>')		// 6.0.2.5 (2014/10/31) char を append する。
				.append( CR );
			// 3.8.8.5 (2007/03/09) numberType 属性を考慮
			if( isNumberDisplay() ) {
				out.append( "<td>" ).append( getNumberData( dummyRow ) ).append( "</td>" )
					.append( CR );
			}
			out.append( "<td>" )
				.append( getColumnLabel(column) ).append("</td>")
				.append( CR );

			if( ! noClass ) {
				clmCls = "<td class=\"" + getColumnDbType(column) + "\" >" ;	// 4.0.0 (2005/01/31)
			}

			for( int row=startNo; row<lastNo; row++ ) {
				out.append( clmCls )
					.append( getValueLabel(row,column) )
					.append("</td>").append( CR );
			}
			out.append("</tr>").append( CR );
		}
		out.append("</tbody>").append( CR )
			.append("</table>").append( CR )
			.append( getScrollBarEndDiv() );	// 3.8.0.3 (2005/07/15)

		return out.toString();
	}

	/**
	 * 内容をクリア(初期化)します。
	 *
	 */
	@Override
	public void clear() {
		super.clear();
		noClass 		= false;
	}

	/**
	 * テーブルのバックグラウンドカラーの値をセットします。
	 * 行番号は, ０から始まるので、偶数を HTML_BG_COLOR_ROW0 、
	 * 奇数行を HTML_BG_COLOR_ROW1 とします。
	 * setBgColorCycle で、設定値変換しています。
	 * なお、このクラスでは、最初の行に、row_h クラス属性を付加します。
	 *
	 * @og.rev 3.8.8.5 (2007/03/09) ヘッダー部の色付け
	 *
	 * @param	indx 先頭からの連番( ０から始める )
	 *
	 * @return	行の色を指定する class 属性( cssファイルで指定 )
	 */
	@Override
	protected String getBgColorCycleClass( final int indx ) {
		return ( indx == 0 )
					? " class=\"row_h\""
					: super.getBgColorCycleClass( indx );
	}

	/**
	 * DBTableModel から テーブルのタグ文字列を作成して返します。
	 *
	 * @return  テーブルのタグ文字列
	 * @og.rtnNotNull
	 */
	@Override
	protected String getTableHead() {

		final StringBuilder buf = new StringBuilder( BUFFER_MIDDLE );
		if( isNumberDisplay() ) {
			buf.append("<colgroup class=\"S9\">");
		}
		buf.append("<colgroup class=\"HEADER\" />")
			.append(CR);

		return buf.toString();
	}

	/**
	 * フォーマットメソッドを使用できるかどうかを問い合わせます。
	 *
	 * @return  使用可能(true)/ 使用不可能 (false)
	 */
	@Override
	public boolean canUseFormat() {
		return false;
	}

	/**
	 * カラムのクラス名(X,S9 など)のセットを行うかどうか指定します。
	 *
	 * "true" で、クラス属性を設定しません。これは、ＣＳＳファイルに書かれている属性を
	 * 使用しないことを意味します。
	 * 初期値は、"false" です。
	 *
	 * @param	flag クラス名使用の有無(true:使用しない/false:使用する。)
	 */
	public void setBodyNoClass( final String flag ) {
		noClass = StringUtil.nval( flag,noClass );
	}

	/**
	 * 表示項目の編集(並び替え)が可能かどうかを返します。
	 *
	 * @og.rev 5.1.6.0 (2010/05/01) 新規追加
	 *
	 * @return	表示項目の編集(並び替え)が可能かどうか(false:不可能)
	 */
	@Override
	public boolean isEditable() {
		return false;
	}
}
