/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.fukurou.util.StringUtil;
import org.opengion.fukurou.util.ToString;						// 6.1.1.0 (2015/01/17)
import org.opengion.fukurou.util.Shell;

import static org.opengion.fukurou.util.StringUtil.nval ;
import static org.opengion.fukurou.util.HybsConst.BR;		// 6.1.0.0 (2014/12/26) refactoring

import java.io.File;

/**
 * ネイティブなプロセスを作成し、バッチファイルやアプリケーションを起動するタグです。
 *
 * プロセスの状態に応じて、下記のリターンコードが得られます。
 *
 *     SHELL.RTN_CODE = 0  : 正常    (ページの残りを評価する。)
 *     SHELL.RTN_CODE = 1  : 処理中  (ページの残りを評価する。)
 *     SHELL.RTN_CODE = 9  : 取消    (ページの残りを評価する。)
 *     SHELL.RTN_CODE = -1 : 異常終了(ページの残りの処理を行わない。)
 *
 * 異常終了の場合は、このタグ以降の処理を中止します。

 * wait 属性を使用して、処理の終了を待たずに、復帰することも出来ます。この場合は、
 * SHELL.RTN_CODE = 1 となり、そのまま、処理は継続されます。このSHELL.RTN_CODE は、
 * リクエスト変数にセットされていますので、そのまま、value タグで、取得可能です。
 *
 * action 属性に指定できるアクション名は、以下のとおりです。
 *
 *         RUN      Shellオブジェクトを生成/実行します。(セッションに登録しません。)
 *         SET      Shellオブジェクトを生成/実行し、セッションに登録します。
 *                  すでにセッションに登録済みのShellオブジェクトが実行中の場合は、途中経過を報告します。
 *         GET      セッションに登録されている Shellオブジェクトを呼び出します。
 *                  セッションに登録されていない場合は、異常終了になります。
 *         REMOVE   セッションから、Shellオブジェクトを削除します。
 *                  Shellが実行中の場合は強制終了します。
 *
 * @og.formSample
 * ●形式：&lt;og:shell program="…" /&gt;
 * ●body：なし
 *
 * ●Tag定義：
 *   &lt;og:shell
 *       program          ○【TAG】シェルファイルを実行する時に引き渡すコマンドを指定します(必須)。
 *       useBatch           【TAG】BATCHプロセスを実行するのかどうか[true/false]を指定します(初期値:false)
 *       wait               【TAG】プロセスの終了を待つかどうか[true/false]を指定します(初期値:true[待つ])
 *       timeout            【TAG】プロセスの実行処理のタイムアウトを設定します (初期値:SHELL_TIMEOUT[=3600])
 *       stdout             【TAG】実行結果を標準出力に出力するかどうか[true/false]を指定します(初期値:false)
 *       stderr             【TAG】実行結果をエラー出力に出力するかどうか[true/false]を指定します(初期値:false)
 *       info               【TAG】インフォメーション情報を出力するかどうか[true/false]を指定します(初期値:false)
 *       action             【TAG】アクション(RUN,SET,GET,REMOVE)をセットします(初期値:RUN)
 *       workDir            【TAG】作業ディレクトリを指定します(初期値:Java仮想マシンの作業ディレクトリ)
 *       envKeys            【TAG】環境変数のキーをＣＳＶ形式で指定します(初期値:現状のJava仮想マシンの環境)
 *       envVals            【TAG】環境変数の値をＣＳＶ形式で指定します(初期値:現状のJava仮想マシンの環境)
 *       display            【TAG】処理結果を画面に表示するかどうか[true/false]指定します(初期値:true)
 *       caseKey            【TAG】このタグ自体を利用するかどうかの条件キーを指定します(初期値:null)
 *       caseVal            【TAG】このタグ自体を利用するかどうかの条件値を指定します(初期値:null)
 *       caseNN             【TAG】指定の値が、null/ゼロ文字列 でない場合(Not Null=NN)は、このタグは使用されます(初期値:判定しない)
 *       caseNull           【TAG】指定の値が、null/ゼロ文字列 の場合は、このタグは使用されます(初期値:判定しない)
 *       caseIf             【TAG】指定の値が、true/TRUE文字列の場合は、このタグは使用されます(初期値:判定しない)
 *       debug              【TAG】デバッグ情報を出力するかどうか[true/false]を指定します(初期値:false)
 *   /&gt;
 *
 * ●使用例
 *    &lt;og:shell
 *        program  = "プログラム"      dir /S や、abc.bat  など
 *        action   = "アクション"      [RUN/SET/GET/REMOVE]
 *        useBatch = "バッチ使用"      バッチ：true/false    (初期値:false)
 *        wait     = "終了待ち"        処理終了まで待つ:true/待たない:false    (初期値:true)
 *        stdout   = "標準出力使用"    標準出力使用:true/未使用:false     (初期値:false)
 *        stderr   = "エラー出力使用"  エラー出力使用:true/未使用:false   (初期値:false)
 *        info     = "状態表示"        シェルの状態表示:true/未使用:false (初期値:false)
 *    /&gt;
 *
 *    &lt;og:value command="GET" key="SHELL.RTN_CODE" /&gt;
 *
 * @og.group その他部品
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class ShellTag extends CommonTagSupport {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "6.3.4.0 (2015/08/01)" ;
	private static final long serialVersionUID = 634020150801L ;

	/** action 引数に渡す事の出来る コマンド  実行 {@value} */
	public static final String ACT_RUN  = "RUN" ;
	/** action 引数に渡す事の出来る コマンド  登録 {@value} */
	public static final String ACT_SET  = "SET" ;
	/** action 引数に渡す事の出来る コマンド  取得 {@value} */
	public static final String ACT_GET  = "GET" ;
	/** action 引数に渡す事の出来る コマンド  削除 {@value} */
	public static final String ACT_REMOVE  = "REMOVE" ;

	/** action 引数に渡す事の出来る コマンド リスト  */
	private static final String[] ACTION_LIST = { ACT_RUN , ACT_SET , ACT_GET , ACT_REMOVE };

	private static final String RTN_CODE = "SHELL.RTN_CODE" ;

	private String	program		;
	private boolean	useBatch	;			// BATCHプロセスを実行するのかどうか(初期値:false)
	private boolean	stdout		;			// 標準出を出力するかどうか(初期値:false)
	private boolean	stderr		;			// エラー出力を出力するかどうか(初期値:false)
	private boolean	wait		= true;		// プロセスの終了を待つ(true)/待たない(false) (初期値:true)
	private boolean	info		;			// インフォメーションを出力するかどうか(初期値:false)
	private boolean	display		= true;		// 処理結果を画面に表示するかどうか(初期値:true)
	private String	action   	= ACT_RUN;
	private File	workDir		;
	private String[]	envKeys	;
	private String[]	envVals	;

	// 3.6.1.0 (2005/01/05) タイムアウト時間を設定
	private int timeout	= HybsSystem.sysInt( "SHELL_TIMEOUT" );

	/**
	 * Taglibの終了タグが見つかったときに処理する doEndTag() を オーバーライドします。
	 *
	 * @og.rev 3.1.1.2 (2003/04/04) Tomcat4.1 対応。release2() を doEndTag()で呼ぶ。
	 * @og.rev 3.1.9.0 (2003/05/16) Shell への stdout と stderr の取得設定は廃止。ShellTag では、有効。
	 * @og.rev 3.3.3.0 (2003/07/09) デバッグ時の表示ロジックを追加。
	 * @og.rev 6.3.4.0 (2015/08/01) caseKey,caseVal,caseNN,caseNull,caseIf 属性対応
	 *
	 * @return	後続処理の指示
	 */
	@Override
	public int doEndTag() {
		debugPrint();		// 4.0.0 (2005/02/28)
		if( !useTag() ) { return EVAL_PAGE ; }	// 6.3.4.0 (2015/08/01)

		final int rtnCode = actionExec( action );

		if( isDebug() ) {
			jspPrint( "Return Code=[" + rtnCode + "]" );
		}

		setRequestAttribute( RTN_CODE, String.valueOf( rtnCode ) );

		// rtnCode = 0  : 正常    (ページの残りを評価する。)
		// rtnCode = 1  : 処理中  (ページの残りを評価する。)
		// rtnCode = 9  : 取消    (ページの残りを評価する。)
		// rtnCode = -1 : 異常終了(ページの残りの処理を行わない。)

		if( rtnCode >= 0 ) {
			return EVAL_PAGE ;		// ページの残りを評価する。
		}
		else {
			return SKIP_PAGE ;		// ページの残りの処理を行わない。
		}
	}

	/**
	 * タグリブオブジェクトをリリースします。
	 * キャッシュされて再利用されるので、フィールドの初期設定を行います。
	 *
	 * @og.rev 2.0.0.4 (2002/09/27) カスタムタグの release() メソッドを、追加
	 * @og.rev 3.1.1.2 (2003/04/04) Tomcat4.1 対応。release2() を doEndTag()で呼ぶ。
	 * @og.rev 3.1.9.0 (2003/05/16) action コマンド追加
	 * @og.rev 3.3.3.0 (2003/07/09) action コマンドの動きを修正。
	 * @og.rev 3.6.0.1 (2004/09/29) display 属性を追加
	 * @og.rev 3.6.1.0 (2005/01/05) timeout 属性を追加
	 *
	 */
	@Override
	protected void release2() {
		super.release2();
		program  = null;
		useBatch = false;	// BATCHプロセスを実行するのかどうか(初期値:false)
		stdout   = false;	// 標準出を出力するかどうか(初期値:false)
		stderr   = false;	// エラー出力を出力するかどうか(初期値:false)
		wait     = true;	// プロセスの終了を待つ(true)/待たない(false) (初期値:true)
		action   = ACT_RUN;
		info     = false;
		display  = true;	// 3.6.0.1 (2004/09/29)
		timeout	 = HybsSystem.sysInt( "SHELL_TIMEOUT" );
	}

	/**
	 * アクションを実行します。
	 *
	 * アクションは,指定のアクションコマンドに対応する処理を入力データに
	 * 対して行います。[RUN/SET/GET/REMOVE]
	 *
	 * @og.rev 3.1.9.0 (2003/05/16) 新規追加
	 *
	 * @param   action アクションコマンド (public static final 宣言されている文字列)
	 *
	 * @return  リターンコード    0:正常  1:実行中  9:取消  -1:異常終了
	 */
	private int actionExec( final String action ) {
		int rtnCode = 1;
		if(      ACT_RUN.equals(     action ) ) { rtnCode = shellRun(); }
		else if( ACT_SET.equals(     action ) ) { rtnCode = shellSet(); }
		else if( ACT_GET.equals(     action ) ) { rtnCode = shellGet(); }
		else if( ACT_REMOVE.equals(  action ) ) { rtnCode = shellRemove(); }

		return rtnCode ;
	}

	/**
	 * Shell オブジェクトを新規作成します。
	 *
	 * @og.rev 3.3.3.0 (2003/07/09) 新規追加
	 * @og.rev 3.6.1.0 (2005/01/05) タイムアウト時間を設定
	 *
	 * @return	Shellオブジェクト
	 */
	private Shell makeShell() {
		final Shell shell = new Shell();
		shell.setCommand( program,useBatch );
		shell.setWait( wait );
		shell.setTimeout( timeout );		// 3.6.1.0 (2005/01/05)
		shell.setWorkDir( workDir );
		shell.setEnvP( makeEnvpArray( envKeys,envVals ) );

		if( isDebug() ) {
			jspPrint( shell.getCommand() + BR );
		}

		return shell ;
	}

	/**
	 * 環境変数設定用の文字列配列を作成します。
	 * これは、キー配列と値配列を、key=value という文字列にして、配列を返します。
	 *
	 * @og.rev 3.3.3.0 (2003/07/09) 新規追加
	 * @og.rev 3.6.0.0 (2004/09/22) 引数のキー、値がnull 等の場合は、長さ０配列を返します。
	 *
	 * @param	keys	環境変数のキー配列
	 * @param	vals	環境変数の値配列
	 *
	 * @return  環境変数設定用の文字列配列(キー=値の文字列を配列にセット)
	 * @og.rtnNotNull
	 */
	private String[] makeEnvpArray( final String[] keys, final String[] vals ) {
		if( keys == null || keys.length == 0 || vals == null || vals.length == 0 ) {
			return new String[0];
		}

		if( keys.length != vals.length ) {
			final String errMsg = "キーとバリューの個数が異なります。"
						+ CR
						+ " keys.length=[" + keys.length + "]  vals.length=[" + vals.length + "]"
						+ CR
						+ " keys=" + StringUtil.array2csv( keys )
						+ CR
						+ " vals=" + StringUtil.array2csv( vals );		// 5.1.8.0 (2010/07/01) errMsg 修正
			throw new HybsSystemException( errMsg );
		}

		String[] envp = new String[keys.length];
		for( int i=0; i<envp.length; i++ ) {
			envp[i] = keys[i] + "=" + vals[i] ;
		}

		return envp ;
	}

	/**
	 * RUN アクションを実行します。
	 *
	 * Shellオブジェクトを生成/実行します。(セッションに登録しません。)
	 *
	 * @og.rev 3.1.9.0 (2003/05/16) 新規追加
	 * @og.rev 3.3.3.0 (2003/07/09) Shell オブジェクトを、makeShell() メソッドより作成します。
	 *
	 * @return  リターンコード    0:正常  1:実行中  9:取消  -1:異常終了
	 */
	private int shellRun() {
		final Shell shell = makeShell();
		final int rtnCode = shell.exec();		// 0 は正常終了を示す
		shellView( shell );

		return rtnCode ;
	}

	/**
	 * SET アクションを実行します。
	 *
	 * Shellオブジェクトを生成/実行し、セッションに登録します。
	 * wait 等でShellの終了を待たない方式では、有用です。
	 *
	 * @og.rev 3.1.9.0 (2003/05/16) 新規追加
	 *
	 * @return  リターンコード    0:正常  1:実行中  9:取消  -1:異常終了
	 */
	private int shellSet() {
		final int rtnCode;

		Shell shell = (Shell)getSessionAttribute( HybsSystem.SHELL_KEEP_KEY );
		if( shell == null || shell.isEnd() ) {
			shell = makeShell();
			rtnCode = shell.exec();		// 0 は正常終了を示す
			setSessionAttribute( HybsSystem.SHELL_KEEP_KEY,shell );
		}
		else {
			rtnCode = shell.exitValue();
		}
		shellView( shell );

		return rtnCode ;
	}

	/**
	 * GET アクションを実行します。
	 *
	 * セッションに登録されている Shellオブジェクトを呼び出します。
	 *
	 * @og.rev 3.1.9.0 (2003/05/16) 新規追加
	 * @og.rev 3.3.3.0 (2003/07/09) アクション時の内容を出力します。
	 *
	 * @return  リターンコード    0:正常  1:実行中  9:取消  -1:異常終了
	 */
	private int shellGet() {
		int rtnCode = 0;

		final Shell shell = (Shell)getSessionAttribute( HybsSystem.SHELL_KEEP_KEY );
		if( shell != null ) { rtnCode = shell.exitValue(); }
		shellView( shell );

		return rtnCode ;
	}

	/**
	 * REMOVE アクションを実行します。
	 *
	 * セッションから、Shellオブジェクトを削除します。
	 * Shell が実行中ならば、強制終了させます。
	 *
	 * @og.rev 3.1.9.0 (2003/05/16) 新規追加
	 * @og.rev 3.3.3.0 (2003/07/09) アクション時の内容を出力します。
	 *
	 * @return  リターンコード    0:正常  1:実行中  9:取消  -1:異常終了
	 */
	private int shellRemove() {
		int rtnCode = 0;

		final Shell shell = (Shell)getSessionAttribute( HybsSystem.SHELL_KEEP_KEY );
		if( shell != null ) {
			shell.destroy();
			rtnCode = shell.exitValue();
		}
		shellView( shell );

		removeSessionAttribute( HybsSystem.SHELL_KEEP_KEY );

		return rtnCode ;
	}

	/**
	 * シェルの実行内容を、画面に表示します。
	 *
	 * @og.rev 3.3.3.0 (2003/07/09) 新規追加
	 * @og.rev 3.6.0.1 (2004/09/29) display 属性を考慮
	 * @og.rev 4.0.0.0 (2007/10/18) メッセージリソース統合( getResource().getMessage ⇒ getResource().getLabel )
	 *
	 * @param shell   シェルオブジェクト
	 */
	private void shellView( final Shell shell ) {
		if( shell == null ) {
			// MSG0066=処理は存在していません。
			jspPrint( getResource().getLabel( "MSG0066" ) );
			return ;
		}

		final int rtnCode = shell.exitValue();

		final String msg ;
		switch( rtnCode ) {
			case Shell.OK      : msg = "MSG0063"; break;	// MSG0063=完了しました。
			case Shell.RUNNING : msg = "MSG0067"; break;	// MSG0067=処理は実行中です。
			case Shell.CANCEL  : msg = "MSG0069"; break;	// MSG0069=処理をキャンセルしました。
	//		case Shell.ERROR   : msg = "MSG0068"; break;	// MSG0068=処理の実行中にエラーが発生しました。
			default            : msg = "MSG0068"; break;	// MSG0068=処理の実行中にエラーが発生しました。
		}

		// 3.6.0.1 (2004/09/29) display 属性を考慮(display=true か、ERROR の場合は表示)
		if( display || rtnCode == Shell.ERROR ) {
			jspPrint( getResource().getLabel( msg ) + BR );
		}

		if( info ) {
			jspPrint( shell.toString() );
		}
		if( stdout ) {
			jspPrint( StringUtil.htmlFilter( shell.getStdoutData() ) );
		}
		if( stderr ) {
			jspPrint( "<div class=\"row_mark\">"  );
			jspPrint( StringUtil.htmlFilter( shell.getStderrData() ) );
			jspPrint( "</div>" );
		}
	}

	/**
	 * 【TAG】シェルファイルを実行する時に引き渡すコマンドを指定します。
	 *
	 * @og.tag シェルファイルを実行する時に引き渡すコマンド
	 *
	 * @param   val シェルコマンド
	 */
	public void setProgram( final String val ) {
		program = getRequestParameter( val );
	}

	/**
	 * 【TAG】BATCHプロセスを実行するのかどうか[true/false]を指定します(初期値:false)。
	 *
	 * @og.tag
	 * BATCHファイルの実行と、EXEファイルの実行では、cmd.exe 経由するか
	 * どうか、判断が必要です。true(BATCHプロセス)では、cmd.exe 経由で実行、
	 * false(EXEファイル)では、直接実行します。
	 * 初期値は、false(EXEファイル)です。
	 *
	 * @param   flag BATCHプロセスを実行するのかどうか [true:BATCHプロセス/false:EXEファイル]
	 */
	public void setUseBatch( final String flag ) {
		useBatch = nval( getRequestParameter( flag ),useBatch );
	}

	/**
	 * 【TAG】実行結果を標準出力に出力するかどうか[true/false]を指定します(初期値:false)。
	 *
	 * @og.tag 実行結果の標準出力を出力するかどうか(初期値:false)
	 *
	 * @param   flag 実行結果の標準出力 [true:出力する/false:しない]
	 */
	public void setStdout( final String flag ) {
		stdout = nval( getRequestParameter( flag ),stdout );
	}

	/**
	 * 【TAG】実行結果をエラー出力に出力するかどうか[true/false]を指定します(初期値:false)。
	 *
	 * @og.tag 実行結果のエラー出力を出力するかどうか(初期値:false)
	 *
	 * @param   flag 実行結果のエラー出力 [true:出力する/false:しない]
	 */
	public void setStderr( final String flag ) {
		stderr = nval( getRequestParameter( flag ),stderr );
	}

	/**
	 * 【TAG】プロセスの終了を待つかどうか[true:待つ/false:待たない]を指定します(初期値:true[待つ])。
	 *
	 * @og.tag プロセスの終了を待つ(true)/待たない(false) (初期値:true)
	 *
	 * @param   flag 終了待ち [true:待つ/false:待たない]
	 */
	public void setWait( final String flag ) {
		wait = nval( getRequestParameter( flag ),wait );
	}

	/**
	 * 【TAG】プロセスの実行処理のタイムアウトを設定します
	 *		(初期値:SHELL_TIMEOUT[={@og.value org.opengion.hayabusa.common.SystemData#SHELL_TIMEOUT}])。
	 *
	 * @og.tag
	 * ゼロ(0) の場合は、割り込みが入るまで待ちつづけます。
	 * その場合、確実に終了させる手段を用意しておかないと、いつまで待っても以下の処理が
	 * 実行されない(デッドロック)状態になってしまいます。
	 * (初期値:システム定数のSHELL_TIMEOUT[={@og.value org.opengion.hayabusa.common.SystemData#SHELL_TIMEOUT}])。
	 *
	 * @og.rev 3.6.1.0 (2005/01/05) 新規追加
	 *
	 * @param	tout	タイムアウト時間(秒) (ゼロは、無制限)
	 * @see		org.opengion.hayabusa.common.SystemData#SHELL_TIMEOUT
	 */
	public void setTimeout( final String tout ) {
		timeout = nval( getRequestParameter( tout ),timeout );
	}

	/**
	 * 【TAG】インフォメーション情報を出力するかどうか[true/false]を指定します(初期値:false)。
	 *
	 * @og.tag インフォメーション情報を出力するかどうか(初期値:false)
	 *
	 * @og.rev 3.1.9.0 (2003/05/16) 新規追加
	 *
	 * @param   flag インフォメーション情報 [true:出力する/false:しない]
	 */
	public void setInfo( final String flag ) {
		info = nval( getRequestParameter( flag ),info );
	}

	/**
	 * 【TAG】処理結果を画面に表示するかどうか[true/false]指定します(初期値:true)。
	 *
	 * @og.tag
	 * 処理結果が、異常でない場合のみ、この機能が有効になります。
	 * 各種処理を連続で行う場合、最後にエラーが発生したにもかかわらず、
	 * 先に正常終了している場合に、不要なメッセージを出すと、紛らわしい為です。
	 *
	 * @og.rev 3.6.0.1 (2004/09/29) 新規追加
	 *
	 * @param   flag 処理結果を画面に表示するかどうか [true:出力する/false:しない]
	 */
	public void setDisplay( final String flag ) {
		display = nval( getRequestParameter( flag ),display );
	}

	/**
	 * 【TAG】アクション[RUN/SET/GET/REMOVE]をセットします(初期値:RUN)。
	 *
	 * @og.tag
	 * アクションは,HTMLから(get/post)指定されますので,ACT_xxx で設定される
	 * フィールド定数値のいづれかを、指定できます。
	 * 初期値は、RUN です。
	 *
	 * RUN      Shellオブジェクトを生成/実行します。(セッションに登録しません。)
	 * SET      Shellオブジェクトを生成/実行し、セッションに登録します。
	 * GET      セッションに登録されている Shellオブジェクトを呼び出します。
	 * REMOVE   セッションから、Shellオブジェクトを削除します。
	 *          Shellが実行中の場合は強制終了します。
	 *
	 * @og.rev 3.1.9.0 (2003/05/16) 新規追加
	 * @og.rev 3.5.6.2 (2004/07/05) 文字列の連結にStringBuilderを使用します。
	 * @og.rev 6.3.4.0 (2015/08/01) Arrays.toString から String.join に置き換え。
	 *
	 * @param	act アクション [RUN/SET/GET/REMOVE]
	 * @see		<a href="../../../../constant-values.html#org.opengion.hayabusa.taglib.ShellTag.ACT_RUN">アクション定数</a>
	 */
	public void setAction( final String act ) {
		action = nval( getRequestParameter( act ),action );
		if( !check( action, ACTION_LIST ) ) {
			final String errMsg = "指定のアクションは実行できません。アクションエラー"	+ CR
							+ "action=[" + action + "] " 								+ CR
							+ "actionList=" + String.join( ", " , ACTION_LIST ) ;
			throw new HybsSystemException( errMsg );

//			final StringBuilder errMsg = new StringBuilder( BUFFER_MIDDLE )
//				.append( "指定のアクションは実行できません。アクションエラー" )
//				.append( CR )
//				.append( "action=[" ).append( action ).append( "] " )
//				.append( CR );
//			for( int i=0; i<ACTION_LIST.length; i++ ) {
//				errMsg.append( " | " ).append( ACTION_LIST[i] );
//			}
//			errMsg.append( " | " );
//			throw new HybsSystemException( errMsg.toString() );
		}
	}

	/**
	 * 【TAG】作業ディレクトリを指定します(初期値:Java仮想マシンの作業ディレクトリ)。
	 *
	 * @og.tag
	 * シェルを実行する、作業ディレクトリを指定します。
	 * 絶対パスで指定することも、相対パスで指定することも出来ます。
	 * 相対パスの場合は、ドキュメントベースの物理パスが基準になります。
	 * (そのJSPファイル自身のディレクトリが基準ではありません。)
	 * 指定しない場合は、このJava仮想マシンの作業ディレクトリで実行されます。
	 *
	 * @og.rev 3.3.3.0 (2003/07/09) 新規追加
	 *
	 * @param   dir 作業ディレクトリ
	 */
	public void setWorkDir( final String dir ) {
		final String work = HybsSystem.url2dir( getRequestParameter( dir ) );
		workDir = new File( work );
		if( ! workDir.isDirectory() ) {
			final String errMsg = "指定の作業ディレクトリは、使用できません。"
							+ CR
							+ "workDir=[" + workDir + "] "
							+ CR ;
			throw new HybsSystemException( errMsg );
		}
	}

	/**
	 * 【TAG】環境変数のキーをＣＳＶ形式で指定します(初期値:現状のJava仮想マシンの環境)。
	 *
	 * @og.tag
	 * シェル実行時の環境変数のキーを、ＣＳＶV形式で指定します。
	 * 指定がない場合は、現状のJava仮想マシンの環境が引き継がれます。
	 * envVals と対応していなければなりません。
	 * 分解方法は、CSV変数を先に分解してから、getRequestParameter で値を取得します。
	 * こうしないとデータ自身にカンマを持っている場合に分解をミスる為です。
	 *
	 * @og.rev 3.3.3.0 (2003/07/09) 新規追加
	 * @og.rev 3.5.6.2 (2004/07/05) CommonTagSupport#getCSVParameter を使用
	 *
	 * @param   keys 環境変数のキー
	 */
	public void setEnvKeys( final String keys ) {
		envKeys = getCSVParameter( keys );
	}

	/**
	 * 【TAG】環境変数の値をＣＳＶ形式で指定します(初期値:現状のJava仮想マシンの環境)。
	 *
	 * @og.tag
	 * シェル実行時の環境変数の値を、ＣＳＶV形式で指定します。
	 * 指定がない場合は、現状のJava仮想マシンの環境が引き継がれます。
	 * envKeys と対応していなければなりません。
	 * 分解方法は、CSV変数を先に分解してから、getRequestParameter で値を取得します。
	 * こうしないとデータ自身にカンマを持っている場合に分解をミスる為です。
	 *
	 * @og.rev 3.3.3.0 (2003/07/09) 新規追加
	 * @og.rev 3.5.6.2 (2004/07/05) CommonTagSupport#getCSVParameter を使用
	 *
	 * @param   vals 環境変数のキー
	 */
	public void setEnvVals( final String vals ) {
		envVals = getCSVParameter( vals );
	}

	/**
	 * このオブジェクトの文字列表現を返します。
	 * 基本的にデバッグ目的に使用します。
	 *
	 * @return このクラスの文字列表現
	 * @og.rtnNotNull
	 */
	@Override
	public String toString() {
		return ToString.title( this.getClass().getName() )
				.println( "VERSION"			,VERSION	)
				.println( "program"			,program	)
				.println( "useBatch"		,useBatch	)
				.println( "stdout"			,stdout		)
				.println( "stderr"			,stderr		)
				.println( "wait"			,wait		)
				.println( "info"			,info		)
				.println( "display"			,display	)
				.println( "action"			,action		)
//				.println( "ACTION_LIST"		,ACTION_LIST	)
				.println( "workDir"			,workDir	)
				.println( "envKeys"			,envKeys	)
				.println( "envVals"			,envVals	)
				.println( "timeout"			,timeout	)
				.println( "Other..."	,getAttributes().getAttribute() )
				.fixForm().toString() ;
	}
}
