/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import org.opengion.fukurou.util.Attributes;
import org.opengion.fukurou.util.ToString;						// 6.1.1.0 (2015/01/17)
import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.html.ViewMarker;
import org.opengion.hayabusa.html.ViewLink_LINK;

import java.util.Locale ;
// import java.io.ObjectOutputStream;
import java.io.ObjectInputStream;
import java.io.IOException;

/**
 * HTML のaタグと同等のリンクを作成するタグで、さらに国際化対応と引数の受け渡しが可能です。
 *
 * DBTableModelオブジェクトを表示する、共通オブジェクトです。
 * このオブジェクトに、 データ(DBTableModel)と、コントローラ(EntryTagForm)を与えて、
 * 外部からコントロールすることで、各種形式で データ(DBTableModel)を表示させることが
 * 可能です。
 *
 * @og.formSample
 * ●形式：&lt;og:viewLink command="…" &gt; &lt;og:link name="…" ... /&gt; &lt;/og:viewLink &gt;
 * ●body：あり(EVAL_BODY_BUFFERED:BODYを評価し、{&#064;XXXX} を解析します)
 *
 * ●Tag定義：
 *   &lt;og:viewLink
 *       command          ○【TAG】コマンド (NEW,RENEW,RESET,REVIEW)をセットします(必須)。
 *       viewLinkId         【TAG】(通常使いません)requestから取得する ViewLink に対応する Attributes オブジェクトの ID
 *       caseKey            【TAG】このタグ自体を利用するかどうかの条件キーを指定します(初期値:null)
 *       caseVal            【TAG】このタグ自体を利用するかどうかの条件値を指定します(初期値:null)
 *       caseNN             【TAG】指定の値が、null/ゼロ文字列 でない場合(Not Null=NN)は、このタグは使用されます(初期値:判定しない)
 *       caseNull           【TAG】指定の値が、null/ゼロ文字列 の場合は、このタグは使用されます(初期値:判定しない)
 *       caseIf             【TAG】指定の値が、true/TRUE文字列の場合は、このタグは使用されます(初期値:判定しない)
 *       debug              【TAG】デバッグ情報を出力するかどうか[true/false]を指定します(初期値:false)
 *   &gt;   ... Body ...
 *   &lt;/og:viewLink&gt;
 *
 * ●使用例
 *     QueryTagとViewTagの間にviewLinkを使います。
 *     &lt;og:viewLink command="{&#064;command}"&gt;
 *         &lt;og:link column="NOSYN" gamenId="01PATTERN1" href="query.jsp" target="QUERY"
 *             keys="NOSYN,NMSYN,CDBK" value="[NOSYN],[NMSYN],[CDBK]" /&gt;
 *         &lt;og:link column="NMSYN" gamenId="01PATTERN1" href="query.jsp" target="QUERY"
 *             keys="NMSYN" value="[NMSYN]" /&gt;
 *         &lt;og:link column="CDBK" gamenId="01PATTERN1" href="query.jsp" target="QUERY"
 *             keys="NOSYN,NMSYN" value="[NOSYN],[NMSYN]" /&gt;
 *     &lt;/og:viewLink&gt;
 *
 * @og.group 画面表示
 *
 * @version  4.0
 * @author	 Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class ViewLinkTag extends CommonTagSupport {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "6.3.4.0 (2015/08/01)" ;
	private static final long serialVersionUID = 634020150801L ;

	/** command 引数に渡す事の出来る コマンド  新規 {@value} */
	public static final String CMD_NEW		= "NEW" ;
	/** command 引数に渡す事の出来る コマンド  再検索 {@value} */
	public static final String CMD_RENEW	= "RENEW" ;
	/** command 引数に渡す事の出来る コマンド  リセット {@value}  */
	public static final String CMD_RESET	= "RESET" ;		// 3.5.4.0 (2003/11/25)
	/** command 引数に渡す事の出来る コマンド  再表示 {@value}  */
	public static final String CMD_REVIEW	= "REVIEW" ;		// 3.5.4.0 (2003/11/25)

	/** command 引数に渡す事の出来る コマンド リスト  */
	private static final String[] COMMAND_LIST = { CMD_NEW , CMD_RENEW ,CMD_RESET , CMD_REVIEW };	// 3.5.4.0 (2003/11/25)

	private	String		viewLinkId	= HybsSystem.VIEWLINK_KEY;
	private	String		command		;
	private	transient ViewMarker	viewMarker	= new ViewLink_LINK();

	/**
	 * Taglibの開始タグが見つかったときに処理する doStartTag() を オーバーライドします。
	 *
	 * @og.rev 5.1.9.0 (2010/08/01) 戻り値を、EVAL_BODY_INCLUDE → EVAL_BODY_BUFFERED に変更
	 * @og.rev 6.3.4.0 (2015/08/01) caseKey,caseVal,caseNN,caseNull,caseIf 属性対応
	 *
	 * @return	後続処理の指示
	 */
	@Override
	public int doStartTag() {
//		if( check( command, COMMAND_LIST ) ) { return EVAL_BODY_BUFFERED ; }				// 5.1.9.0 (2010/08/01) 変更
		if( useTag() && check( command, COMMAND_LIST ) ) { return EVAL_BODY_BUFFERED ; }	// 6.3.4.0 (2015/08/01)
		else { return SKIP_BODY ; }
	}

	/**
	 * Taglibの終了タグが見つかったときに処理する doEndTag() を オーバーライドします。
	 *
	 * @og.rev 3.1.1.2 (2003/04/04) Tomcat4.1 対応。release2() を doEndTag()で呼ぶ。
	 * @og.rev 6.3.4.0 (2015/08/01) caseKey,caseVal,caseNN,caseNull,caseIf 属性対応
	 *
	 * @return	後続処理の指示
	 */
	@Override
	public int doEndTag() {
		debugPrint();		// 4.0.0 (2005/02/28)
//		if( check( command, COMMAND_LIST ) ) {
		if( useTag() && check( command, COMMAND_LIST ) ) {
			setRequestAttribute( viewLinkId,viewMarker );
		}

		return EVAL_PAGE ;
	}

	/**
	 * タグリブオブジェクトをリリースします。
	 * キャッシュされて再利用されるので、フィールドの初期設定を行います。
	 *
	 * @og.rev 2.0.0.4 (2002/09/27) カスタムタグの release() メソッドを、追加
	 * @og.rev 3.1.1.2 (2003/04/04) Tomcat4.1 対応。release2() を doEndTag()で呼ぶ。
	 *
	 */
	@Override
	protected void release2() {
		super.release2();
		viewLinkId	= HybsSystem.VIEWLINK_KEY;
		command		= null;
		viewMarker	= new ViewLink_LINK();
	}

	/**
	 * 内部タグの LinkTag より、個々のカラムの値を書き換える 為の
	 * リンク文字列を受け取る。
	 *
	 * 複数の値を受け取って、後ほど、すべてのカラムに対して処理を行います。
	 *
	 * @og.rev 3.1.2.0 (2003/04/07) taglib パッケージ内部で使用している箇所を protected 化する。
	 * @og.rev 4.0.0.0 (2005/08/31) 同一カラムの複数登録を許可します。
	 *
	 * @param	attri	リンクアトリビュート
	 */
	protected void addAttribute( final Attributes attri ) {
		viewMarker.addAttribute( attri );
	}

	/**
	 * 【TAG】コマンド (NEW,RENEW,RESET,REVIEW)をセットします。
	 *
	 * @og.tag
	 * コマンドは,HTMLから(get/post)指定されますので,CMD_xxx で設定される
	 * フィールド定数値のいづれかを、指定できます。
	 *
	 * @param	cmd コマンド (public static final 宣言されている文字列)
	 * @see		<a href="../../../../constant-values.html#org.opengion.hayabusa.taglib.ViewLinkTag.CMD_NEW">コマンド定数</a>
	 */
	public void setCommand( final String cmd ) {
		final String cmd2 = getRequestParameter( cmd );
		if( cmd2 != null && cmd2.length() > 0 ) { command = cmd2.toUpperCase(Locale.JAPAN); }
	}

	/**
	 * 【TAG】(通常使いません)requestから取得する ViewLink に対応する Attributes オブジェクトの ID。
	 *
	 * @og.tag
	 * ViewLink オブジェクトをこのキーで、登録することにより、
	 * ViewForm オブジェクトで、リンク情報を付加して表示させる。
	 * 初期値は、HybsSystem.VIEWLINK_KEY です。
	 *
	 * @og.rev 3.1.4.0 (2003/04/18) 新規追加
	 * @og.rev 3.5.6.3 (2004/07/12) {&#064;XXXX} 変数を使用できるように変更。
	 * @og.rev 3.5.6.4 (2004/07/16) LINK_ID を付加して、他のid と混同しないようにします。
	 *
	 * @param	id	オブジェクトID
	 */
	public void setViewLinkId( final String id ) {
		final String temp = getRequestParameter( id ) ;
		if( temp != null && temp.length() > 0 ) {
			viewLinkId = temp + TaglibUtil.LINK_ID;
		}
	}

//	/**
//	 * シリアライズ用のカスタムシリアライズ書き込みメソッド。
//	 *
//	 * @og.rev 4.0.0.0 (2006/09/31) 新規追加
//	 * @serialData 一部のオブジェクトは、シリアライズされません。
//	 *
//	 * @param	strm	ObjectOutputStreamオブジェクト
//	 * @throws IOException	入出力エラーが発生した場合
//	 */
//	private void writeObject( final ObjectOutputStream strm ) throws IOException {
//		strm.defaultWriteObject();
//	}

	/**
	 * シリアライズ用のカスタムシリアライズ読み込みメソッド
	 *
	 * ここでは、transient 宣言された内部変数の内、初期化が必要なフィールドのみ設定します。
	 *
	 * @og.rev 4.0.0.0 (2006/09/31) 新規追加
	 * @serialData 一部のオブジェクトは、シリアライズされません。
	 *
	 * @param	strm	ObjectInputStreamオブジェクト
	 * @see #release2()
	 * @throws IOException	シリアライズに関する入出力エラーが発生した場合
	 * @throws ClassNotFoundException	クラスを見つけることができなかった場合
	 */
	private void readObject( final ObjectInputStream strm ) throws IOException , ClassNotFoundException {
		strm.defaultReadObject();
		viewMarker = new ViewLink_LINK();
	}

	/**
	 * このオブジェクトの文字列表現を返します。
	 * 基本的にデバッグ目的に使用します。
	 *
	 * @return このクラスの文字列表現
	 * @og.rtnNotNull
	 */
	@Override
	public String toString() {
		return ToString.title( this.getClass().getName() )
				.println( "VERSION"		,VERSION	)
				.println( "viewLinkId"	,viewLinkId	)
				.println( "command"		,command	)
				.println( "Other..."	,getAttributes().getAttribute() )
				.fixForm().toString() ;
	}
}
