/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.table;

import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.db.AbstractTableFilter;
import org.opengion.hayabusa.db.DBTableModel;
import org.opengion.hayabusa.report2.QueueManager_DB;

import java.util.Map;

/**
 * TableFilter_REPORTDATA は、TableFilter インターフェースを継承した、DBTableModel 処理用の
 * 実装クラスです。
 *
 * ここでは、指定された要求NOに対してGE51(帳票明細データ)をGE52(帳票レイアウトテーブル)の定義に従って、
 * 分割し、DBTableModelを生成します。
 *
 * パラメータは、tableFilterタグの keys, vals にそれぞれ記述するか、BODY 部にCSS形式で記述します。
 * 【パラメータ】
 *  {
 *       SYSTEM_ID  :  ;		検索対象となる、システムID(必須)
 *       LISTID     :  ;		検索対象となる、帳票ID(必須)
 *       YKNO       :  ;		検索対象となる、要求番号(必須)
 *       KBTEXT     :  ;		H(ヘッダー),F(フッター),B(ボディー)のいずれかを指定(必須)
 *  }
 *
 * @og.formSample
 * ●形式：
 *      ① &lt;og:tableFilter classId="REPORTDATA" keys="SYSTEM_ID,LISTID,YKNO,KBTEXT" vals="GF,GF0001,111100,B" /&gt;
 *
 *      ② &lt;og:tableFilter classId="REPORTDATA" &gt;
 *               {
 *                   SYSTEM_ID : GF ;
 *                   LISTID    : GF0001 ;
 *                   YKNO      : 111100 ;
 *                   KBTEXT    : B ;
 *               }
 *         &lt;/og:tableFilter&gt;
 *
 * @see org.opengion.hayabusa.report2.QueueManager_DB.DBTableModelCreator
 * @og.rev 5.1.2.0 (2010/01/01) 新規作成
 * @og.rev 5.6.6.0 (2013/07/05) keys の整合性チェックを追加
 *
 * @version  0.9.0  2000/10/17
 * @author   Hiroki Nakamura
 * @since    JDK1.1,
 */
public class TableFilter_REPORTDATA extends AbstractTableFilter {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "5.6.6.1 (2013/07/12)" ;

	/**
	 * keys の整合性チェックを行うための初期設定を行います。
	 *
	 * @og.rev 5.6.6.1 (2013/07/12) keys の整合性チェック対応
	 *
	 * @param	keysMap keys の整合性チェックを行うための Map
	 */
	@Override
	protected void init( final Map<String,String> keysMap ) {
		keysMap.put( "SYSTEM_ID"	, "検索対象となる、システムID(必須)"							);
		keysMap.put( "LISTID"		, "検索対象となる、帳票ID(必須)"								);
		keysMap.put( "YKNO"			, "検索対象となる、要求番号(必須)"								);
		keysMap.put( "KBTEXT"		, "H(ヘッダー),F(フッター),B(ボディー)のいずれかを指定(必須)"	);
	}

	/**
	 * DBTableModel処理を実行します。
	 *
	 * @og.rev 5.5.2.6 (2012/05/25) protected変数を、private化したため、getterメソッドで取得するように変更
	 *
	 * @return 処理結果のDBTableModel
	 * @og.rtnNotNull
	 */
	public DBTableModel execute() {
		final String systemId	= getValue( "SYSTEM_ID" );
		final String listId   = getValue( "LISTID" );
		final String ykno     = getValue( "YKNO" );
		final String kbtext   = getValue( "KBTEXT" );

		if( systemId == null || systemId.isEmpty()
			|| listId == null || listId.isEmpty()
			|| ykno   == null || ykno.isEmpty()
			|| kbtext == null || kbtext.isEmpty() ) {		// 6.1.0.0 (2014/12/26) refactoring
			final String errMsg = "SYSTEM_ID,LISTID,YKNO,KBTEXTを全て指定して下さい。";
			throw new HybsSystemException( errMsg );
		}

		if( kbtext.length() > 1 || "HFB".indexOf( kbtext ) < 0 ) {
			final String errMsg = "KBTEXTは、H(ヘッダー),F(フッター),B(ボディー)のいずれかを指定して下さい";
			throw new HybsSystemException( errMsg );
		}

		final QueueManager_DB.DBTableModelCreator creator
			= new QueueManager_DB.DBTableModelCreator( systemId, listId, ykno, kbtext, getResource() );		// 5.5.2.6 (2012/05/25)

		return creator.getTable();
	}
}
