/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import org.opengion.hayabusa.html.ViewStackTableParam;

import static org.opengion.fukurou.util.StringUtil.nval ;

// import java.io.ObjectOutputStream;
// import java.io.ObjectInputStream;
// import java.io.IOException;

/**
 * viewタグの viewFormType が HTMLGanttTable の場合にパラメータを設定します。
 *
 * ガントチャートを表示する、ViewForm_HTMLGanttTable クラスに対して、各種パラメータを
 * 設定します。
 * パラメータが設定されていない場合は、ViewStackTableParam の初期値が
 * 使用されます。
 * (パラメータを使用するには、viewタグのuseParam 属性をtrueに設定する必要があります。)
 *
 * @og.formSample
 * ●形式：&lt;og:stackParam stackColumns="PRJCODE,RESOURCE_ID" /&gt;
 * ●body：なし
 *
 * ●Tag定義：
 *   &lt;og:stackParam
 *       stackColumns     ○【TAG】積上単位とするカラム名(CSV形式)をセットします (必須)
 *       costColumn         【TAG】内部積上げする場合の工数カラムをセットします
 *       innerStack         【TAG】工数の内部積み上げをするかどうか[true/false]指定します(初期値:false)
 *       debug              【TAG】デバッグ情報を出力するかどうか[true/false]を指定します(初期値:false)
 *   /&gt;
 *
 * @og.rev 5.5.7.0 (2012/10/01) 新規作成
 * @og.group 画面表示
 *
 * @version  5.0
 * @author	 Takahashi Masakazu
 * @since    JDK5.0,
 */
// public class ViewStackParamTag extends ViewParamTag {
public class ViewStackParamTag extends ViewParamImpl {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "5.6.1.2 (2013/02/22)" ;

	private static final long serialVersionUID = 561220130222L ;

	/**
	 * 【TAG】積上単位とするカラム名(CSV形式)をセットします。
	 *
	 * @og.tag
	 * 工数積上げのキーとなるカラム名を指定します。
	 * 
	 * 複数ある場合は、CSV形式で指定できます。
	 * テーブルモデルは積上単位＋タスク単位でOrderByがかかっている必要があります。
	 *
	 * @param	stackColumns 積上カラム名 (CSV形式)
	 */
	public void setStackColumns( final String stackColumns ) {
		putParam( ViewStackTableParam.STACK_COLUMNS_KEY ,
				  nval( getRequestParameter( stackColumns ),null ) );
	}

	/**
	 * 【TAG】工数関係のカラム（工数,開始日,終了日）を指定します。
	 *
	 * @og.tag
	 * 工数関係のカラム名を指定します。
	 * 工数(x.y)、開始日(yyyyMMdd)、終了日(yyyyMMdd)
	 * の３つのカラムをCSV形式で設定して下さい。
	 * エンジン内部で工数積上げを行う場合は必須です。
	 * 
	 * テーブルモデルは積上単位＋タスク単位でOrderByがかかっている必要があります。
	 *
	 * @og.rev 5.5.8.3 (2012/11/17)
	 *
	 * @param	costColumns 工数のカラム名
	 */
	public void setCostColumns( final String costColumns ) {
		putParam( ViewStackTableParam.COST_COLUMNS_KEY ,
				  nval( getRequestParameter( costColumns ),null ) );
	}

	/**
	 * 【TAG】エンジン内部で積上げするかどうかを指定します
	 *		(初期値:INNER_STACK_VALUE[={@og.value ViewStackTableParam#INNER_STACK_VALUE}])。
	 *
	 * @og.tag
	 * エンジン内部で積上げを行うかどうかを指定します。
	 * 内部で積上げを行った場合は、HTMLStackGanttTableではガント行の出力を行いません。
	 * 初期値はfalse（積上げしない）です。
	 *
	 * @og.rev 5.5.8.3 (2012/11/17)
	 *
	 * @param	flag 内部積上げ可否 [true:する/false:しない]
	 */
	public void setInnerStack( final String flag ) {
		putParam( ViewStackTableParam.INNER_STACK_KEY ,
				  nval( getRequestParameter( flag ),ViewStackTableParam.INNER_STACK_VALUE ) );
	}

	/**
	 * 【TAG】エンジン内部で積上げする場合に休日も積上げるかどうかを指定します
	 *		(初期値:STACK_HOLIDAY_VALUE[={@og.value ViewStackTableParam#STACK_HOLIDAY_VALUE}])。
	 *
	 * @og.tag
	 * エンジン内部で積上げを行う場合に休日も積上げるかどうかを指定します。
	 * zoomがDAYの場合のみ有効です。
	 * 初期値はtrue（積上げする）です。
	 *
	 * @og.rev 5.5.8.3 (2012/11/17)
	 *
	 * @param	flag 休日積上げ可否 [true:する/false:しない]
	 */
	public void setStackHoliday( final String flag ) {
		putParam( ViewStackTableParam.STACK_HOLIDAY_KEY ,
				  nval( getRequestParameter( flag ),ViewStackTableParam.STACK_HOLIDAY_VALUE ) );
	}

	/**
	 * 【TAG】その行の能力値カラムを指定します
	 *		(初期値:CAP_COLUMN_VALUE[={@og.value ViewStackTableParam#CAP_COLUMN_VALUE}])。
	 *
	 * @og.tag
	 * 内部積上げをする際、各行の能力のカラム名を指定します。
	 * カラムを指定しない場合は1となります。
	 *
	 * @og.rev 5.6.1.2 (2013/02/22)
	 *
	 * @param	capColumn 能力値のカラム名
	 */
	public void setCapacityColumn( final String capColumn ) {
		putParam( ViewStackTableParam.CAP_COLUMN_KEY ,
				  nval( getRequestParameter( capColumn ),ViewStackTableParam.CAP_COLUMN_VALUE  ) );
	}

	/**
	 * タグの名称を、返します。
	 * 自分自身のクラス名より、自動的に取り出せないため、このメソッドをオーバーライドします。
	 *
	 * @return  タグの名称
	 * @og.rtnNotNull
	 */
	@Override
	protected String getTagName() {
		return "stackParam" ;
	}

//	/**
//	 * シリアライズ用のカスタムシリアライズ書き込みメソッド。
//	 *
//	 * @serialData 一部のオブジェクトは、シリアライズされません。
//	 *
//	 * @param	strm	ObjectOutputStreamオブジェクト
//	 * @throws IOException	入出力エラーが発生した場合
//	 */
//	private void writeObject( final ObjectOutputStream strm ) throws IOException {
//		strm.defaultWriteObject();
//	}

//	/**
//	 * シリアライズ用のカスタムシリアライズ読み込みメソッド
//	 *
//	 * ここでは、transient 宣言された内部変数の内、初期化が必要なフィールドのみ設定します。
//	 *
//	 * @serialData 一部のオブジェクトは、シリアライズされません。
//	 *
//	 * @param	strm	ObjectInputStreamオブジェクト
//	 * @see #release2()
//	 * @throws IOException	シリアライズに関する入出力エラーが発生した場合
//	 * @throws ClassNotFoundException	クラスを見つけることができなかった場合
//	 */
//	private void readObject( final ObjectInputStream strm ) throws IOException , ClassNotFoundException {
//		strm.defaultReadObject();
//	}
}
