/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.column;

import org.opengion.hayabusa.db.AbstractEditor;
import org.opengion.hayabusa.db.CellEditor;
import org.opengion.hayabusa.db.DBColumn;
import org.opengion.fukurou.util.XHTMLTag;
import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.fukurou.util.TagBuffer;

import static org.opengion.fukurou.util.StringUtil.isNull;				// 6.3.3.0 (2015/07/25)

/**
 * カラムのデータをチェックボックスで編集する場合に使用するエディタークラスです。
 *
 * 値としては、"0" と、"1" のみ 有効です。("0" がOFF状態で、"1"がON状態です。)
 * チェックボックスは、通常チェックが入らないと何も送信されません。"0"から"1"への
 * 状態変化は、チェックされるためリクエスト値が送信されますが、"1"から"0"の場合は、
 * 送信されない為、データのクリアを取得することが出来ません。
 * このクラスでは、状態にかかわらず、hidden で、"0" を送信しています。受信側では、
 * 常に"0"がくる為、何もチェックされない場合は、"0"が設定されたことになります。
 * "1"が設定されると、"0"と"1"が両方(同じキーで)取得されます。その場合は、"1"を
 * 見つけることで、"1"をセットされたことが判ります。
 *
 * 編集パラメータに"SEQ"と記述することで正方向にしか選べないシークチェックボックスを実現できます。
 * これにより、シーケンスにステータスを順に挙げていくような、チェックボックス
 * を作成することが出来ます。(逆に戻れないメニュー)
 *
 * このエディタはeventColumnに対応していません。
 *
 * カラムの表示に必要な属性は, DBColumn オブジェクト より取り出します。
 * このクラスは、DBColumn オブジェクト毎に１つ作成されます。
 *
 * @og.rev 3.5.3.1 (2003/10/31) 新規作成
 * @og.group データ編集
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class Editor_CHBOX extends AbstractEditor {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "6.3.3.0 (2015/07/25)" ;

	private final boolean seqFlag ;			// 3.6.0.6 (2004/10/22)
	private final String  HIDDEN0 ;			// 3.6.0.6 (2004/10/22) , 6.1.1.0 (2015/01/17) 大文字に変更
	private final String  HIDDEN1 ;			// 3.6.0.6 (2004/10/22) , 6.1.1.0 (2015/01/17) 大文字に変更

	/**
	 * デフォルトコンストラクター。
	 * このコンストラクターで、基本オブジェクトを作成します。
	 *
	 * @og.rev 3.6.0.6 (2004/10/22) シーケンスアクセス機能(seqFlag)を追加します
	 * @og.rev 3.6.0.6 (2004/10/22) 値が"1"の場合は、hiddenで "0" を出すことでクリアに対応
	 */
	public Editor_CHBOX() {
		// 4.3.4.4 (2009/01/01)
		seqFlag   = false;	// 3.6.0.6 (2004/10/22)
		HIDDEN0 = "";
		HIDDEN1 = "";
	}

	/**
	 * コンストラクター。
	 *
	 * @og.rev 3.5.6.0 (2004/06/18) XHTMLTag の 内部配列 INPUT_KEY を隠蔽します。
	 * @og.rev 3.6.0.6 (2004/10/22) シーケンスアクセス機能(seqFlag)を追加します
	 * @og.rev 3.6.0.6 (2004/10/22) 値が"1"の場合は、hiddenで "0" を出すことでクリアに対応
	 * @og.rev 6.3.3.0 (2015/07/25) maxlength は不要なので削除
	 *
	 * @param	clm	DBColumnオブジェクト
	 */
	private Editor_CHBOX( final DBColumn clm ) {
		super( clm );
		seqFlag    = "SEQ".equals( clm.getEditorParam() );	// 3.6.0.6 (2004/10/22)
		final String  readonly = clm.isWritable() ? null : "readonly" ;

		attributes.set( "type"		, "checkbox"	);
		attributes.set( "readonly"	,  readonly		);
		attributes.set( "NO_MAXLEN"	, "true"		);			// 6.3.3.0 (2015/07/25) maxlength は不要なので削除

		tagBuffer.add( XHTMLTag.inputAttri( attributes ) );

		HIDDEN0 = "<input type=\"hidden\" value=\"0\" name=\"" + name ;		// 3.6.0.6 (2004/10/22)
		HIDDEN1 = "<input type=\"hidden\" value=\"1\" name=\"" + name ;		// 3.6.0.6 (2004/10/22)
	}

	/**
	 * 各オブジェクトから自分のインスタンスを返します。
	 * 自分自身をキャッシュするのか、新たに作成するのかは、各サブクラスの実装に
	 * まかされます。
	 *
	 * @param	clm	DBColumnオブジェクト
	 *
	 * @return	CellEditorオブジェクト
	 * @og.rtnNotNull
	 */
	public CellEditor newInstance( final DBColumn clm ) {
		return new Editor_CHBOX( clm );
	}

	/**
	 * データの編集用文字列を返します。
	 *
	 * @og.rev 3.5.5.5 (2004/04/23) 新規に Attributes オブジェクトを作成する方式を止めます。
	 * @og.rev 3.6.0.6 (2004/10/22) シーケンスアクセス機能(seqFlag)を追加します
	 * @og.rev 3.6.0.6 (2004/10/22) 値が"1"の場合は、hiddenで "0" を出すことでクリアに対応
	 * @og.rev 3.8.0.1 (2005/06/17) 値にかかわらず、hiddenで "0" を出すことでクリアに対応
	 * @og.rev 6.3.3.0 (2015/07/25) CHBOX に、id属性を出力します。
	 *
	 * @param	value 入力値
	 *
	 * @return	データの編集用文字列
	 * @og.rtnNotNull
	 */
	@Override
	public String getValue( final String value ) {
		// 6.1.1.0 (2015/01/17) TagBufferの連結記述
		return new TagBuffer( "input" )
					.add( "name"		, name )
					.add( "id"			, name , isNull( attributes.get( "id" ) ) )		// 6.3.3.0 (2015/07/25)
					.add( "value"		, "1" )
					.add( tagBuffer.makeTag() )
					.add( "checked"		, "checked"		, "1".equals( value ) )
					.add( "disabled"	, "disabled"	, "1".equals( value ) && seqFlag )
					.makeTag()
				// 3.6.0.6 (2004/10/22) seqFlag=trueで、値が"1"の場合、
				// disabled では値が飛ばないので、hidden で "1" を出力する。
				+ ( "1".equals( value ) && seqFlag ? HIDDEN1 : HIDDEN0 )
				+ "\" />" ;
	}

	/**
	 * name属性を変えた、データ表示/編集用のHTML文字列を作成します。
	 * テーブル上の name に 行番号を付加して、名前_行番号 で登録するキーを作成し,
	 * リクエスト情報を１つ毎のフィールドで処理できます。
	 *
	 * @og.rev 3.5.5.0 (2004/03/12) 名前と行番号の区切り記号("__")を、HybsSystem.JOINT_STRING  に変更。
	 * @og.rev 3.5.5.5 (2004/04/23) 新規に Attributes オブジェクトを作成する方式を止めます。
	 * @og.rev 3.6.0.6 (2004/10/22) シーケンスアクセス機能(seqFlag)を追加します
	 * @og.rev 3.6.0.6 (2004/10/22) 値が"1"の場合は、hiddenで "0" を出すことでクリアに対応
	 * @og.rev 3.8.0.1 (2005/06/17) 値にかかわらず、hiddenで "0" を出すことでクリアに対応
	 * @og.rev 6.3.3.0 (2015/07/25) CHBOX に、id属性を出力します。
	 *
	 * @param   row   行番号
	 * @param   value 値
	 *
	 * @return  データ表示/編集用の文字列
	 * @og.rtnNotNull
	 */
	@Override
	public String getValue( final int row,final String value ) {
		final String newName = name + HybsSystem.JOINT_STRING + row;						// 6.3.3.0 (2015/07/25)

		// 6.1.1.0 (2015/01/17) TagBufferの連結記述
		return new TagBuffer( "input" )
//					.add( "name"		, name + HybsSystem.JOINT_STRING + row )
					.add( "name"		, newName )
					.add( "id"			, newName , isNull( attributes.get( "id" ) ) )		// 6.3.3.0 (2015/07/25)
					.add( "value"		, "1" )
					.add( tagBuffer.makeTag() )
					.add( "checked"		, "checked"		, "1".equals( value ) )
					.add( "disabled"	, "disabled"	, "1".equals( value ) && seqFlag )
					.makeTag( row,value )
				// 3.6.0.6 (2004/10/22) seqFlag=trueで、値が"1"の場合、
				// disabled では値が飛ばないので、hidden で "1" を出力する。
				+ ( "1".equals( value ) && seqFlag ? HIDDEN1 : HIDDEN0 )
				+ HybsSystem.JOINT_STRING + row + "\" />" ;

	}
}
