/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.table;

import java.io.File;
import java.io.PrintWriter;
import java.util.Map;

import org.opengion.fukurou.util.ErrorMessage;
import org.opengion.fukurou.util.FileUtil;
import org.opengion.fukurou.util.FixLengthData;
import org.opengion.fukurou.util.StringUtil;
import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.db.AbstractTableFilter;
import org.opengion.hayabusa.db.DBTableModel;
import org.opengion.hayabusa.resource.ResourceFactory;
import org.opengion.hayabusa.resource.ResourceManager;

/**
 * TableFilter_DBARG_OUT は、TableFilter インターフェースを継承した、DBTableModel 処理用の
 * 実装クラスです。
 *
 * ここでは、テーブル一覧の検索結果より、GF81 のARG変数チェックテーブルから
 * 必要な情報を取得し、ARG変数作成スクリプトを作成します。
 * 出力ファイルは、オブジェクト名＋".sql" という命名規則で作成します。
 * 検索では、(SYSTEM_ID,TBLSYU,OBJ_NAME,SEQNO,CLM,CLM_NAME,CLS_NAME,USE_LENGTH,MAX_LENGTH,TABLE_NAME)
 * の項目を取得する必要があります。
 *
 * パラメータは、tableFilterタグの keys, vals にそれぞれ記述するか、BODY 部にCSS形式で記述します。
 * 【パラメータ】
 *  {
 *       DIR : {&#064;BASE_DIR}/sql/install/07_ARG ;   出力ファイルの基準フォルダ(必須)
 *       XML : false ;                                 XML出力を行うかどうか[true/false]を指定します(初期値:false)。
 *  }
 *
 * @og.formSample
 * ●形式：
 *      select SYSTEM_ID,TBLSYU,OBJ_NAME,SEQNO,CLM,CLM_NAME,CLS_NAME,USE_LENGTH,MAX_LENGTH,TABLE_NAME from GF81
 *      ① &lt;og:tableFilter classId="DBARG_OUT" keys="DIR" vals="{&#064;BASE_DIR}/sql/install/07_ARG" /&gt;
 *
 *      ② &lt;og:tableFilter classId="DBARG_OUT" &gt;
 *               {
 *                   DIR : {&#064;BASE_DIR}/sql/install/07_ARG ;
 *                   XML : false ;
 *               }
 *         &lt;/og:tableFilter&gt;
 *
 * @og.rev 5.6.6.0 (2013/07/05) keys の整合性チェックを追加
 *
 * @version  0.9.0  2000/10/17
 * @author   Kazuhiko Hasegawa
 * @since    JDK1.1,
 */
public class TableFilter_DBARG_OUT extends AbstractTableFilter {
	/** このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "6.3.7.0 (2015/09/04)" ;

	// 5.6.6.0 (2013/07/05) 検索項目を増やしておきます。
	private static final String[] KEYS = { "SYSTEM_ID","TBLSYU","OBJ_NAME","SEQNO","CLM","CLM_NAME","CLS_NAME","USE_LENGTH","MAX_LENGTH","TABLE_NAME" };

	private static final int SYSTEM_ID		= 0;
 //	private static final int TBLSYU			= 1;
	private static final int OBJ_NAME		= 2;
	private static final int SEQNO			= 3;
	private static final int CLM			= 4;
	private static final int CLM_NAME		= 5;
	private static final int CLS_NAME		= 6;
	private static final int USE_LENGTH		= 7;
 //	private static final int MAX_LENGTH		= 8;
 //	private static final int TABLE_NAME		= 9;

 //	private static final String ENCODE = "Windows-31J" ;
	private static final String ENCODE = "UTF-8" ;			// 5.6.7.0 (2013/07/27) sqlも UTF-8 で出力

	// 5.6.6.0 (2013/07/05) ヘッダー部作成用
	private static final String CMNT  = "************************************************************************" ;

	private static final int X = FixLengthData.X ;		// type 定数
	private static final int S = FixLengthData.S ;		// type 定数
	private static final int K = FixLengthData.K ;		// type 定数
	private static final int T = FixLengthData.T ;		// addLen 定数
	private static final int T3= FixLengthData.T3 ;		// addLen 定数

	/** 5.6.7.0 (2013/07/27) 各種定数  */
	// 6.0.2.3 (2014/10/10) AbstractTableFilter へ移動

	private ResourceManager resource	;			// 5.6.6.0 (2013/07/05) ヘッダー用のリソース

	/**
	 * keys の整合性チェックを行うための初期設定を行います。
	 *
	 * @og.rev 5.6.6.1 (2013/07/12) keys の整合性チェック対応
	 *
	 * @param	keysMap keys の整合性チェックを行うための Map
	 */
	@Override
	protected void init( final Map<String,String> keysMap ) {
		keysMap.put( "DIR"	, "出力ファイルの基準フォルダ(必須)"						);
		keysMap.put( "XML"	, "XML出力を行うかどうか[true/false]を指定(初期値:false)"	);		// 5.6.7.0 (2013/07/27) xml対応
	}

	/**
	 * DBTableModel処理を実行します。
	 *
	 * @og.rev 4.0.0.0 (2007/11/28) メソッドの戻り値をチェックします。
	 * @og.rev 5.5.2.6 (2012/05/25) protected変数を、private化したため、getterメソッドで取得するように変更
	 * @og.rev 5.6.7.0 (2013/07/27) xml 出力対応
	 * @og.rev 6.0.2.3 (2014/10/10) EXEC_END_TAG 自体にCRを追加。キャッシュします。
	 * @og.rev 6.3.7.0 (2015/09/04) AutoCloseableを使用したtry-with-resources構築に対応。
	 *
	 * @return 処理結果のDBTableModel
	 */
	public DBTableModel execute() {
		isXml = StringUtil.nval( getValue( "XML" ), false );	// 5.6.7.0 (2013/07/27) xml 出力対応
		// 6.0.2.3 (2014/10/10) EXEC_END_TAG 自体にCRを追加。キャッシュします。
		execEndTag = isXml ? CR + EXEC_END_TAG : ";" ;

		final File dir = new File( getValue( "DIR" ) );
		if( ! dir.exists() && !dir.mkdirs() ) {
			final String errMsg = "所定のフォルダが作成できませんでした。[" + dir + "]" ;
			// 4.3.4.4 (2009/01/01)
			throw new HybsSystemException( errMsg );
		}

		// カンマ,カラム,クラス,(,桁数,),記号(--),表示順,名称
		final int[] addLen = new int[] { 0,T,0,0,0,T3,0,1,0 };	// 各データ間のスペース
		final int[] type   = new int[] { X,X,X,X,S,X, X,S,K };	// 各データの種別 X:半角 S:空白前埋め K:全角混在
		final FixLengthData fixData = new FixLengthData( addLen,type );

		String[] data  = null;
		String oldObjName  = null;

		final DBTableModel table = getDBTableModel();		// 5.5.2.6 (2012/05/25) インターフェースにgetterメソッド追加
		final int[]  clmNo = getTableColumnNo( KEYS );
		final int rowCnt = table.getRowCount();
		for( int row=0; row<rowCnt; row++ ) {
			String objName = null;
			try {
				data = table.getValues( row );
				objName = data[clmNo[OBJ_NAME]];

				// 5.6.6.0 (2013/07/05) 初めに一度だけ作成しておきます。
				if( resource == null ) {
					final String systemId = data[clmNo[SYSTEM_ID]];
					resource = ResourceFactory.newInstance( systemId,"ja",false );
				}

				final boolean blk = ! objName.equals( oldObjName ) ;
				if( row > 0 && blk ) {
					// 5.6.7.0 (2013/07/27) xml 出力対応
					// 6.3.7.0 (2015/09/04) AutoCloseableを使用したtry-with-resources構築に対応。
					try( final PrintWriter writer = FileUtil.getPrintWriter( new File( dir,oldObjName + ( isXml ? ".xml" : ".sql" ) ),ENCODE ) ) {
						if( isXml ) { writer.println( XML_START_TAG ); }	// 5.6.7.0 (2013/07/27) xml 出力対応
						writer.print( makeHeadLine( oldObjName ) );
						writer.print( fixData.getAllFixData() );
						writer.println( makeEndLine( oldObjName ) );
						if( isXml ) { writer.println( XML_END_TAG ); }		// 5.6.7.0 (2013/07/27) xml 出力対応
//						writer.close();
						fixData.clear();
					}
				}

				final String[] outData = makeLineList( clmNo,data,blk );
				fixData.addListData( outData );

				oldObjName = objName ;
			}
			catch( RuntimeException ex ) {
				final ErrorMessage errMessage = makeErrorMessage( "TableFilter_DBARG_OUT Error",ErrorMessage.NG );
				errMessage.addMessage( row+1,ErrorMessage.NG,"ARG",ex.getMessage() );
				errMessage.addMessage( row+1,ErrorMessage.NG,"ARG",StringUtil.array2csv( data ) );
				errMessage.addMessage( row+1,ErrorMessage.NG,"ARG","OBJ_NAME=[" + objName + "]" );
			}
		}

		// 常に、一回り遅れてデータ出力している為、最後のデータを出力しておく必要がある。
		// 5.6.7.0 (2013/07/27) xml 出力対応
		// 6.3.7.0 (2015/09/04) AutoCloseableを使用したtry-with-resources構築に対応。
		try( final PrintWriter writer = FileUtil.getPrintWriter( new File( dir,oldObjName + ( isXml ? ".xml" : ".sql" ) ),ENCODE ) ) {
			if( isXml ) { writer.println( XML_START_TAG ); }	// 5.6.7.0 (2013/07/27) xml 出力対応
			writer.print( makeHeadLine( oldObjName ) );
			writer.print( fixData.getAllFixData() );
			writer.println( makeEndLine( oldObjName ) );
			if( isXml ) { writer.println( XML_END_TAG ); }		// 5.6.7.0 (2013/07/27) xml 出力対応
//			writer.close();
			fixData.clear();
		}

		return table;
	}

	/**
	 * ヘッダーとして使用する文字列を作成します。
	 *
	 * @og.rev 5.6.6.0 (2013/07/05) ヘッダーとして各種情報をセットします。
	 * @og.rev 5.6.7.0 (2013/07/27) xml 出力対応
	 * @og.rev 6.0.2.3 (2014/10/10) EXEC_END_TAG 自体にCRを追加
	 *
	 * @param	objName	オブジェクト名
	 *
	 * @return	ヘッダーとして使用する文字列
	 * @og.rtnNotNull
	 */
	private String makeHeadLine( final String objName ) {
		// objName から、ARG を取り除いた文字列が、画面IDになるはず。リソースから検索
		final String gamenId = objName.substring( 0,objName.length()-3 );		// 3文字は、"ARG" 分
		final String gamenNm = resource.getLabel( gamenId ) ;

		final String LINE1 = objName ;
		final String LINE2 = gamenId + " ( " + gamenNm + " )" ;
		final String LINE3 = "Created : " + HybsSystem.getDate() ;

		final int[] addLen = new int[] { 0,0,0 };	// 各データ間のスペース
		final int[] type   = new int[] { X,K,X };	// 各データの種別 X:半角 S:空白前埋め K:全角混在
		final FixLengthData fixData = new FixLengthData( addLen,type );

		final String[][] outData = new String[][] {
			{ "/**",	CMNT ,	"**/" },
			{ "/* ",	LINE1,	" */" },
			{ "/* ",	LINE2,	" */" },
			{ "/* ",	LINE3,	" */" },
			{ "/**",	CMNT ,	"**/" },
		};

		// 5.6.6.0 (2013/07/05) 簡易メソッドを利用
		fixData.addAllListData( outData );

		final StringBuilder buf = new StringBuilder( BUFFER_MIDDLE );
		fixData.getAllFixData( buf );

		if( isXml ) { buf.append( EXEC_START_TAG ).append( CR ); }			// 5.6.7.0 (2013/07/27) xml 出力対応
		buf.append( "DROP TYPE " ).append( objName ).append( "_ARRAY" )
		// 6.0.2.3 (2014/10/10) EXEC_END_TAG 自体にCRを追加
			.append( execEndTag ).append( CR );								// 6.0.2.3 (2014/10/10)

		if( isXml ) { buf.append( EXEC_START_TAG ).append( CR ); }			// 5.6.7.0 (2013/07/27) xml 出力対応
		buf.append( "CREATE OR REPLACE TYPE " ).append( objName )
			.append( " AS OBJECT" ).append( CR )
			.append( " (" ).append( CR );

		return buf.toString() ;
	}

	/**
	 * 各行に相当する文字列の配列を作成します。
	 * カンマ,カラム,クラス,(,桁数,),記号(--),表示順,名称
	 *
	 * @og.rev 5.5.1.9 (2012/04/18) useLen.length=0対応
	 * @og.rev 5.6.6.0 (2013/07/05) カラムの表示順も出力します。
	 *
	 * @param	clmNo	カラム番号配列
	 * @param	data	１行分の入力データ配列
	 * @param	first	最初の行かどうか[true:最初/false:それ以降]
	 *
	 * @return	１行分に相当する結果配列(カンマ,カラム,クラス(桁数),コメント記号,表示順,名称)
	 * @og.rtnNotNull
	 */
	private String[] makeLineList( final int[] clmNo,final String[] data,final boolean first ) {
		// カンマ,カラム,クラス(桁数),記号(--),表示順,名称
		String[] outData = new String[9];					// 5.6.6.0 (2013/07/05) カラムの表示順追加

		outData[0] = first ? "   " : " , " ;			// 0:カンマ
		outData[1] = data[clmNo[CLM]] ;						// 1:カラム

		final String clsName = data[clmNo[CLS_NAME]];
		if( clsName.startsWith( "CLOB" ) || clsName.startsWith( "DATE" ) ) {
			data[clmNo[USE_LENGTH]] = null;
		}
		final String useLen = data[clmNo[USE_LENGTH]];
		if( useLen != null && ! useLen.equals( "0" ) && useLen.length() > 0 ) { // 5.5.1.9 (2012/04/19)
			outData[2] = clsName ;							// 2:クラス
			outData[3] = " ( " ;							// 3:(
			outData[4] = useLen ;							// 4:桁数
			outData[5] = " )" ;								// 5:)
		}
		else {
			outData[2] = clsName ;							// NUMBER型桁数オープン対応。ARGで使えるか不明。
		}

		final String seqno  = data[clmNo[SEQNO]] ;				// 表示順
		final String nameJA = data[clmNo[CLM_NAME]] ;				// 名称
		if( nameJA != null ) {
			outData[6] = "-- " ;							// 3:コメント記号
			outData[7] = seqno ;							// 4:表示順
			outData[8] = nameJA ;							// 5:名称
		}

		return outData ;
	}

	/**
	 * 最後の行に相当する文字列を作成します。
	 *
	 * @og.rev 5.6.7.0 (2013/07/27) xml 出力対応
	 * @og.rev 6.0.2.3 (2014/10/10) EXEC_END_TAG の置き換え＋"/" の対応
	 *
	 * @param	objName	オブジェクト名
	 *
	 * @return	最後の行に相当する文字列
	 * @og.rtnNotNull
	 */
	private String makeEndLine( final String objName ) {
		// 6.0.2.3 (2014/10/10) EXEC_END_TAG の置き換え＋"/" の判定が必要。
		final String e_e_tag = execEndTag + (isXml ? CR : CR + "/" + CR) ;

		final StringBuilder buf = new StringBuilder( BUFFER_MIDDLE )
			.append( "  )"  )

		// 6.0.2.3 (2014/10/10) EXEC_END_TAG 自体にCRを追加
			.append( e_e_tag )													// 6.0.2.3 (2014/10/10)
			.append( CR );

		if( isXml ) { buf.append( EXEC_START_TAG ).append( CR ); }				// 5.6.7.0 (2013/07/27) xml 出力対応
		buf.append( "CREATE OR REPLACE TYPE " )
			.append( objName ).append( "_ARRAY AS VARRAY(10000) OF " ).append( objName )	// 6.0.2.3 (2014/10/10) 1000 → 10000
			// 6.0.2.3 (2014/10/10) EXEC_END_TAG 自体にCRを追加
			.append( e_e_tag );												// 6.0.2.3 (2014/10/10)

		return buf.toString() ;
	}
}
