/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.report;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.OutputStreamWriter;
import java.io.UnsupportedEncodingException;

import org.opengion.fukurou.util.StringUtil;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.report.AbstractCSVPrintPointService;

/**
 * 標準的なCSV形式でデータを作成します。
 * CSVの出力先はGE50系テーブルで指定した場所です。
 *
 * @og.group 帳票システム
 *
 * @version  5.9.0.0
 * @author	 Masakazu Takahashi
 * @since    JDK6.0,
 */
public class CSVPrintPointService_DEFAULT extends AbstractCSVPrintPointService {

	private static final String CR		= System.getProperty("line.separator");
	private final StringBuilder strCSV	= new StringBuilder();	// CSVはこれに吐く

	private final String	csvEncode 	= HybsSystem.sys("REPORT_CSV_TEXT_ENCODE");

	/**
	 * 発行処理
	 * ファイル出力
	 *
	 * @return 結果 [true:正常/false:異常]
	 */
	@Override
	public boolean execute(){
		System.out.print( "CSV create ... " );
		BufferedWriter bw = null;
		String filename= outdir;
		boolean flg = false;

		try {
			makeheader();
			makebody();

			bw = getWriter(filename,false,csvEncode);
			bw.write( strCSV.toString() );
			bw.flush();
			bw.close();

			flg = true;
			
//			if( prgfile != null && prgfile.length() > 0){
//				makeShellCommand();
//				flg = programRun();
//			}

		}
		catch ( Throwable ex ) {
			errMsg.append( "CSV Print Request Execution Error. " ).append( CR );
			errMsg.append( "==============================" ).append( CR );
			errMsg.append( "SYSTEM_ID=[" ).append( systemId ).append( "] , " );
			errMsg.append( "YKNO=["    ).append( ykno    ).append( "] , " );
			errMsg.append( ex.toString() );
			errMsg.append( CR );
//			throw new RuntimeException( errMsg.toString() );
			throw new RuntimeException( errMsg.toString(), ex );
		}
		return flg;
	}

	/**
	 * ヘッダの出力
	 *
	 */
	private void makeheader(){
		//ヘッダデータを出力する場合はここで指定する。
		//strCSV.append( listid ).append( CR );
		
		//１行目にカラム名を出力します。
		// メインテーブルはNULLではない
		for( int clmNo=0; clmNo<table.getColumnCount(); clmNo++ ) {
			// 先頭以外はカンマを付ける
			if( clmNo > 0 ){ strCSV.append( "," ); } 
			strCSV.append("\"").append( table.getColumnName( clmNo )).append( "\"" );
		}
		if( tableH != null){
			for( int clmNo=0; clmNo<tableH.getColumnCount(); clmNo++ ) {
				strCSV.append( "," ); 
				strCSV.append("\"H_").append( tableH.getColumnName( clmNo )).append("\"");
			}
		}
		if( tableF != null){
			for( int clmNo=0; clmNo<tableF.getColumnCount(); clmNo++ ) {
				strCSV.append( "," ); 
				strCSV.append("\"F_").append( tableF.getColumnName( clmNo )).append("\"");
			}
		}
		strCSV.append( CR );
	}



	/**
	 * 本体の出力を行います
	 */
	private void makebody(){

		for( int rowNo=0; rowNo<table.getRowCount(); rowNo++ ) {
			// カラム単位の処理
			for( int clmNo=0; clmNo<table.getColumnCount(); clmNo++ ) {
				// 先頭以外はカンマを付ける
				if( clmNo > 0 ){ strCSV.append( "," ); } 
				// 全てダブルクウォートで囲う
				strCSV.append("\"").append( StringUtil.replace(table.getValue( rowNo, clmNo ),"\"","\"\"" ) ).append("\"");
			}
			
			//ヘッダ、フッタは毎行に必ず付加します。
			//例え複数行あったとしても先頭行のみ有効です
			//ヘッダ
			if( tableH != null){
				int rowNoH=0;
				for( int clmNo=0; clmNo<tableH.getColumnCount(); clmNo++ ) {
					// 必ずカンマを付ける
					strCSV.append( "," ); 
					// 全てダブルクウォートで囲う
					strCSV.append("\"").append( StringUtil.replace(tableH.getValue( rowNoH, clmNo ),"\"","\"\"" ) ).append("\"");
				}
			}
			
			//フッタ
			if( tableF != null ){
				int rowNoF=0;
				for( int clmNo=0; clmNo<tableF.getColumnCount(); clmNo++ ) {
					// 必ずカンマを付ける
					strCSV.append( "," ); 
					// 全てダブルクウォートで囲う
					strCSV.append("\"").append( StringUtil.replace(table.getValue( rowNoF, clmNo ),"\"","\"\"" ) ).append("\"");
				}
			}

			strCSV.append( CR );
		}
	}


	/**
	 * ファイル書き込み用のライターを返します。
	 *
	 * @param fileName ファイル名
	 * @param append アベンドするか
	 * @param encode エンコード
	 *
	 * @return ライター
	 */
	private BufferedWriter getWriter( final String fileName, final boolean append, final String encode) {
		File file = new File ( fileName );
		BufferedWriter bw;

		try {
			bw = new BufferedWriter( new OutputStreamWriter( new FileOutputStream( file, append ), encode ) );
		}
		catch ( UnsupportedEncodingException ex ) {
			errMsg.append( "[ERROR] Input File is written by Unsupported Encoding" );
			throw new HybsSystemException( ex );
		}
		catch ( FileNotFoundException ex ) {
			errMsg.append( "[ERROR] File not Found" );
			throw new HybsSystemException( ex );
		}
		return bw;
	}

	/**
	 * シェルコマンドの文字列を作成します。
	 *
	 * @og.rev 5.4.3.9 引数変更
	 *
	 * @return 結果 [true:正常/false:異常]
	 */
	private boolean makeShellCommand() {
		StringBuilder buf = new StringBuilder( HybsSystem.BUFFER_SMALL );

		buf.append( prgdir + HybsSystem.FS + prgfile ).append( " " );		// 実行するコマンド
		buf.append( "\"" ).append( ykno ).append( "\" " );			// 引数１：要求NO
		buf.append( "\"" ).append( prtid ).append( "\" " );			// 引数２：プリンタID
		buf.append( "\"" ).append( prtName ).append( "\" " );		// 引数３：プリンタID
		buf.append( "\"" ).append( hostName ).append( "\" " );		// 引数４：ホスト名
		buf.append( "\"" ).append( portnm ).append( "\" " );			// 引数５：プリンタポート 5.4.3.1(2011/12/27)

		shellCmd = buf.toString();
		System.out.println( CR + shellCmd + CR );

		return true;
	}
}
