/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.resource;

import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.fukurou.util.ApplicationInfo;
import org.opengion.fukurou.db.DBUtil;

import java.util.HashMap;
import java.util.Map;
import java.util.WeakHashMap;
import java.util.Collections ;

/**
 * systemId と lang に対応したラベルデータを作成するデータロードクラスです。
 *
 * ラベルデータは、項目(CLM)に対して、各種ラベル情報を持っています。
 * ラベルデータは、名前(ORG)と名前(短)と名前(長)を持っています。従来のラベルは、表示名称と
 * して、一種類しか持っていませんでした。
 * 名前(ORG)は、従来の表示名称にあたります。これは、一般的なラベルとして
 * 使用されます。名前(短)は、テーブル一覧のヘッダーの様に、特殊なケースで、
 * 簡略化された名称を使用するときに利用されます。この切り替えは、自動で判断されます。
 * 名前(短)に、なにも設定されていない場合は、名前(長)が自動的に使用されますので
 * 初期データ移行時には、そのまま、通常時もテーブルヘッダー時も同じ文字列が
 * 使用されます。
 * 名前(短)と名前(長)は、コメント情報が存在する場合は、Tips表示を行います。
 *
 * ラベルデータを作成する場合は、同一ラベルで、作成区分(KBSAKU)違いの場合は、
 * 最も大きな作成区分を持つコードを使用します。
 * 作成区分(KBSAKU)='0' のデータは、マスタリソースとして、エンジンとともに
 * 配布されるリソースになります。
 *
 * 読込フラグ(FGLOAD)='1'のラベルリソースは、このLabelDataLoaderオブジェクトが
 * 構築された時に、すべてキャッシュとして内部メモリに読み取ります。
 * 読込フラグが、'1' 以外のデータは、初期起動時には、メモリにキャッシュされず
 * 実際に使用されるまで、オブジェクトが作成されません。
 * これは、使用されるかどうか判らないラベルデータを、予め作成しないことで、メモリの
 * 節約を図っています。
 * ただし、リソースのキャッシュに、WeakHashMap クラスを使用しているため、
 * メモリオーバー時には、クリアされるため、単独での読み取りも行います。
 *
 * SYSTEM_ID='**' は、共通リソースです。
 * これは、システム間で共通に使用されるリソース情報を登録しておきます。
 *
 * @og.rev 4.0.0.0 (2004/12/31) 新規作成
 * @og.group リソース管理
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
final class LabelDataLoader {
	// リソースの接続先を、取得します。
	private final String DBID = HybsSystem.sys( "RESOURCE_DBID" );

	// ＤＢリソースの初期一括読み込みのクエリー
	// 4.3.5.7 (2009/03/22) FGLOADの影響で個別システムのリソースが読まれない問題の対応
	private static final String QUERY = "select CLM,SNAME,LNAME,DESCRIPTION"
									+ " ,FGLOAD"
									+ " from GEA08 where SYSTEM_ID in ( ?,'**')"
									+ " and LANG=? and FGJ='1'"
									+ " order by SYSTEM_ID,CLM,KBSAKU" ;

	// ＤＢリソースの個別読み込み時のクエリー
	// 6.3.1.1 (2015/07/10) FGLOAD,UNIQ 追加
	private static final String QUERY2 = "select CLM,SNAME,LNAME,DESCRIPTION"
									+ " ,FGLOAD,UNIQ,SYSTEM_ID"				// 6.3.1.1 (2015/07/10)
									+ " from GEA08 where SYSTEM_ID in ( ?,'**')"
									+ " and LANG=? and CLM=? and FGJ='1'"
									+ " order by SYSTEM_ID,KBSAKU" ;

	// 6.3.1.1 (2015/07/10) 読込フラグ(FGLOAD) のマーカー設定追加。
	private static final boolean useFgloadAutoset = HybsSystem.sysBool( "USE_FGLOAD_AUTOSET" );

	// 6.3.1.1 (2015/07/10) FGLOAD更新(UNIQ だけで指定可能だが、万一を想定して、SYSTEM_IDとCLMを条件に追記)
	private static final String UPDATE2 = "update GEA08 set FGLOAD='2' where UNIQ=? and SYSTEM_ID=? and CLM=?";

	private final Map<String,LabelData> pool = Collections.synchronizedMap( new WeakHashMap<>() );	// キャッシュ用プール
	private final String  SYSTEM_ID ;		// システムID
	private final String  LANG ;			// 言語

	/** コネクションにアプリケーション情報を追記するかどうか指定 */
	public static final boolean USE_DB_APPLICATION_INFO  = HybsSystem.sysBool( "USE_DB_APPLICATION_INFO" ) ;

	// 3.8.7.0 (2006/12/15) アクセスログ取得の為,ApplicationInfoオブジェクトを設定
	private final ApplicationInfo appInfo;

	/**
	 *  SystemId と lang 毎に ファクトリオブジェクトを作成します。
	 *
	 * @og.rev 3.8.7.0 (2006/12/15) アクセスログ取得の為,ApplicationInfoオブジェクトを設定
	 *
	 * @param systemId システムID
	 * @param lang	言語
	 * @param initLoad リソースデータの先読み可否(true:先読みする)
	 */
	LabelDataLoader( final String systemId,final String lang,final boolean initLoad ) {
		SYSTEM_ID = systemId;
		LANG	  = lang;

		// 3.8.7.0 (2006/12/15) アクセスログ取得の為,ApplicationInfoオブジェクトを設定
		if( USE_DB_APPLICATION_INFO ) {
			appInfo = new ApplicationInfo();
			// ユーザーID,IPアドレス,ホスト名
			appInfo.setClientInfo( SYSTEM_ID,HybsSystem.HOST_ADRS,HybsSystem.HOST_NAME );
			// 画面ID,操作,プログラムID
			appInfo.setModuleInfo( "LabelDataLoader",null,null );
		}
		else {
			appInfo = null;
		}

		// ApplicationInfo の設定が終わってから実行します。
		if( initLoad ) { loadDBResource(); }
	}

	/**
	 * ＤＢリソースより ラベルデータを取得、設定します。
	 * 取得データは、CLM,SNAME,LNAME,DESCRIPTION です。
	 *
	 * @og.rev 3.8.7.0 (2006/12/15) アクセスログ取得の為,ApplicationInfoオブジェクトを設定
	 * @og.rev 4.3.5.7 (2009/03/22)  FGLOADの影響でシステム個別リソースが読まれない問題対応
	 */
	private void loadDBResource() {
		final String[] args = new String[] { SYSTEM_ID,LANG };

		final String[][] vals = DBUtil.dbExecute( QUERY,args,appInfo,DBID );

		final int len = vals.length;
		for( int i=0; i<len; i++ ) {
			if( "1".equals( vals[i][LabelData.FG_LOAD] ) ){ // 4.3.5.7 (2009/03/22)
				pool.put( vals[i][0],new LabelData( vals[i] ) );
			}
			// より上の作成区分で、FGLOAD='0'(個別読込)が来た場合は、下位のFGLOAD='1'(一括読込)を破棄
			else if( pool.get( vals[i][0]) != null ){
				pool.remove( vals[i][0] );
			}
		}

		System.out.println( "  LabelDataLoader [" + len + "] loaded" );
	}

	/**
	 * LabelData オブジェクトを取得します。
	 * 作成したLabelDataオブジェクトは，内部にプールしておき，同じリソース要求が
	 * あったときは，プールの LabelDataを返します。
	 * リソースDBに存在しない場合は、NULLラベルを作成します。このオブジェクトも
	 * キャッシュします。
	 * 読込フラグ(FGLOAD)が '1' のデータは、起動時に先読みします。
	 * それ以外のデータは、ここでキー要求が発生した時点で読み込みます。
	 * 読込フラグ(FGLOAD) のマーカー設定モード(USE_FGLOAD_AUTOSET)を使用する(true)場合は、
	 * 追加読み込み(先読みされていないカラム)に対して、読込フラグ(FGLOAD)を 2:使用実績 に
	 * 設定します。(次回起動時の、初期読み込みは行いません。)
	 *
	 * @og.rev 3.8.7.0 (2006/12/15) アクセスログ取得の為,ApplicationInfoオブジェクトを設定
	 * @og.rev 6.3.1.1 (2015/07/10) 読込フラグ(FGLOAD) のマーカー設定追加。
	 *
	 * @param   key        ラベルのキー
	 *
	 * @return  LabelDataオブジェクト
	 * @og.rtnNotNull
	 */
	public LabelData getLabelData( final String key ) {
		LabelData label = pool.get( key ) ;

		if( label == null ) {
			final String[] args = new String[] { SYSTEM_ID,LANG,key };
			final String[][] vals = DBUtil.dbExecute( QUERY2,args,appInfo,DBID );	// SYSTEM_ID='**' も含む

			if( vals.length > 0 ) {
				final int row=vals.length-1;						// 最後の検索結果
//				label = new LabelData( vals[vals.length-1] );		// 最後のデータ
				label = new LabelData( vals[row] );					// 最後のデータ

				// 6.3.1.1 (2015/07/10) 読込フラグ(FGLOAD) のマーカー設定追加。
				if( useFgloadAutoset ) {
					// 1:一括読込 と、2:使用実績 以外のﾘｿｰｽは、2:使用実績 をセットする。(SYSTEM_ID='**'は含まない)
					final String fgld  = vals[row][LabelData.FG_LOAD];
					final String sysld = vals[row][LabelData.SYSTEM_ID];
					if( !"1".equals( fgld ) && !"2".equals( fgld ) && !"**".equals( sysld ) ) {
						final String[] args2 = new String[] { vals[row][LabelData.UNIQ],SYSTEM_ID,key };
						DBUtil.dbExecute( UPDATE2,args2,appInfo,DBID );		// FGLOAD を、2:使用実績 にセット
					}
				}
			}
			else {
				label = new LabelData( key );		// null ラベル
			}
			pool.put( key,label );
		}

		return label ;
	}

	/**
	 * 指定されたクエリを発行し、ラベルマップを作成します。
	 *
	 * ここで作成されたラベル情報は、内部的にキャッシュされません。
	 * 各画面で一時的にラベル情報を追加したい場合に使用します。
	 *
	 * 発行するクエリでは、第1カラムにラベルキーを、第2カラムにラベル名称を設定します。
	 * 第3カラムが存在する場合は、名称(短)として使用されます。(必須ではありません)
	 * クエリが指定されていない又は、検索したカラム数が1以下の場合はエラーを返します。
	 *
	 * @og.rev 4.3.4.0 (2008/12/01) 新規作成
	 *
	 * @param	query		ラベルマップを作成するクエリ
	 *
	 * @return  ラベルマップ
	 */
	public Map<String, LabelData> getLabelMap( final String query ) {
		if( query == null || query.isEmpty() ) {
			final String errMsg = "ラベルを取得するクエリが指定されていません。";
			throw new HybsSystemException( errMsg );
		}

		final String[][] rtn = DBUtil.dbExecute( query, new String[0], appInfo );
		if( rtn == null || rtn.length == 0 ) { // データが存在しない場合はそのまま終了します。
			return null;
		}

		final int confSize = rtn[0].length;
		if( confSize < 2 ) {
			final String errMsg = "ラベルキー、ラベル名称の指定は必須です。"
						+ " SQL=" + query ;					// 5.1.8.0 (2010/07/01) errMsg 修正
			throw new HybsSystemException( errMsg );
		}

		final Map<String, LabelData> labelMap = new HashMap<>();
		for( int i=0; i<rtn.length; i++ ) {
			String[] ldconf = new String[5];
			ldconf[0] = rtn[i][0];
			ldconf[1] = ( confSize == 2 ? rtn[i][1] : rtn[i][2] ); // SNAME
			ldconf[2] = rtn[i][1]; // LNAME
			ldconf[3] = "";
			ldconf[4] = "";

			labelMap.put( rtn[i][0], new LabelData( ldconf ) );
		}
		return labelMap;
	}

	/**
	 * LabelData オブジェクトのキャッシュを個別にクリアします。
	 * リソースデータの更新など、一部分の更新時に、すべてのキャッシュを
	 * 破棄するのではなく、指定の分のみ破棄できる機能です。
	 *
	 * @param   key        ラベルのキー
	 */
	public void clear( final String key ) {
		pool.remove( key );
	}

	/**
	 * LabelData オブジェクトのキャッシュをクリアします。
	 *
	 */
	public void clear() {
		pool.clear();
	}
}
