/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

// import java.io.IOException;
// import java.io.ObjectInputStream;
// import java.io.ObjectOutputStream;
import java.util.ArrayList;
import java.util.List;
import java.util.Locale;
import java.util.Map;

import org.opengion.fukurou.util.StringUtil;
import org.opengion.fukurou.util.XHTMLTag;
import org.opengion.fukurou.util.ToString;						// 6.1.1.0 (2015/01/17)
import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.db.DBColumn;
import org.opengion.hayabusa.db.DBColumnConfig;
import org.opengion.hayabusa.db.DBEditConfig;
import org.opengion.hayabusa.db.DBLastSql;
import org.opengion.hayabusa.db.DBTableModel;
import org.opengion.hayabusa.db.DBTableModelSorter;
import org.opengion.hayabusa.html.FormatterType;
import org.opengion.hayabusa.html.TableFormatter;
import org.opengion.hayabusa.html.ViewCrossTableParam;
import org.opengion.hayabusa.html.ViewForm;
import org.opengion.hayabusa.html.ViewFormFactory;
import org.opengion.hayabusa.html.ViewMarker;
import org.opengion.hayabusa.resource.RoleMode;
import org.opengion.hayabusa.resource.UserInfo;
import org.opengion.hayabusa.resource.GUIInfo;

import static org.opengion.fukurou.util.StringUtil.nval;

/**
 * データベースの検索結果を表示するタグです。
 *
 * 検索結果は、DBTableModel にセットされます。
 * このタグに、データ(DBTableModel)と、
 * コントローラ(ViewForm)を与えて、 外部からコントロールすることで、テキストフィールドやテーブルの形で表示したり、
 * 入力可/不可、表示可/不可の設定を行うことができます。
 *
 * @og.formSample
 * ●形式：&lt;og:view /&gt;
 *     &lt;og:view
 *         viewFormType = &quot;HTMLTable&quot;
 *         command      = &quot;{&#064;command}&quot;
 *         startNo      = &quot;0&quot;
 *         pageSize     = &quot;20&quot;
 *         language     = &quot;ja&quot; /&gt;
 *
 *     viewFormType : ViewFormオブジェクトを作成する時のキー(オプション)     ViewFormFactory参照
 *     command      : 処理コマンド ( PREV,NEXT,FIRST,LAST,VIEW )(オプション)
 *     startNo      : データの書き始めの初期値を指定(オプション)
 *     pageSize     : データのページ数(画面表示件数)を指定(オプション)
 *     language     : ロケール(オプション)
 *
 * ●body：あり(EVAL_BODY_BUFFERED:BODYを評価し、{&#064;XXXX} を解析します)
 *
 * ●Tag定義：
 *   &lt;og:view
 *       viewFormType        【TAG】ViewForm オブジェクトを作成するクラス({@og.doc03Link viewFormType ViewForm_**** クラス})を指定します
 *       command             【TAG】コマンド (PREV,NEXT,FIRST,LAST,NEW,RENEW,VIEW,REVIEW,RESET)をセットします
 *       scope               【TAG】キャッシュする場合のスコープ[request/page/session/applicaton]を指定します(初期値:session)
 *       startNo             【TAG】表示データを作成する場合の表示の開始行番号をセットします
 *       pageSize            【TAG】表示データを作成する場合の１ページの行数をセットします
 *       pagePlus            【TAG】１ページの行数の増加分をセットします(初期値:0)
 *       writable            【TAG】行が書き込み可能かどうか[true/false]を設定します (初期値:DBTableModel#DEFAULT_WRITABLE[=true])
 *       checked             【TAG】行のチェックボックスの初期値を設定するかどうか[true/false]を指定します(初期値:null)
 *       rowspan             【TAG】表示データを作成する場合のフォーマットの行数をセットします(初期値:2)
 *       skip                【TAG】NEXT等で、選択行以外をスキップするのかどうか[true/false]を指定します (初期値:ViewForm#DEFAULT_SKIP[=false])
 *       columnWritable      【TAG】書き込み可能カラム名を、CSV形式で与えます
 *       noWritable          【TAG】書き込み不可カラム名を、CSV形式で与えます
 *       columnDisplay       【TAG】表示可能カラム名を、CSV形式で与えます
 *       clmSeqDisplay       【TAG】表示可能カラム名を、カラム順に、CSV形式で与えます
 *       noDisplay           【TAG】表示不可カラム名を、CSV形式で与えます
 *       columnGroup         【TAG】同一表示データをまとめる(表示しない)カラム名を、CSV形式で与えます
 *       noGroup             【TAG】同一表示データをまとめないカラム名を、CSV形式で与えます
 *       groupClass          【TAG】同一表示データをまとめる場合の、表示方法を指定します
 *       tableSorterKeys     【TAG】カラム名リンクソートを表示するカラム名を、CSV形式で与えます
 *       tableId             【TAG】(通常使いません)sessionから所得する DBTableModelオブジェクトの ID
 *       viewFormId          【TAG】(通常使いません)sessionから取得する ViewForm オブジェクトの ID
 *       viewLinkId          【TAG】(通常つかいません)requestから取得する ViewLink に対応する Attributes オブジェクトの ID
 *       viewMarkerId        【TAG】(通常使いません)requestから取得する ViewMarker に対応する Attributes オブジェクトの ID
 *       editMarkerId        【TAG】(通常使いません)requestから取得する EditMarker に対応する Attributes オブジェクトの ID
 *       selectedType        【TAG】表示時の選択用オブジェクトのタイプ[checkbox/radio/hidden]を指定します(初期値:checkbox)
 *       numberType          【TAG】表示時の行番号の表示方法[sequence/none/delete/skip=カラム名/view=カラム名]を指定します(初期値:sequence)
 *       optionTypeAttributes【TAG】テーブル等のチェックボックスに属性を付加します
 *       noMessage           【TAG】検索結果メッセージを表示しないかどうか[true/false]を設定します(初期値:false[表示する])
 *       backLinkCount       【TAG】ページの先頭へのリンクの間隔をセットします (初期値:VIEW_BACK_LINK_COUNT[=0])
 *       headerSkipCount     【TAG】ヘッダーを出力する間隔をセットします(初期値:システムパラメータ の VIEW_HEADER_SKIP_COUNT)
 *       useCheckControl     【TAG】チェックボックスの全チェックを選択するJavaScript の機能を使用するかどうかを指定します (初期値:VIEW_USE_CHECK_CONTROL[=true])
 *       useTableSorter      【TAG】ヘッダーにソート用リンクを作成するかどうか[true/false]を指定します (初期値:VIEW_USE_TABLE_SORTER[=true])
 *       useParam            【TAG】パラメーターファイルの使用をするかどうか[true/false]を指定します(初期値:false)
 *       useConsistency      【TAG】Consistency キー による整合性チェックを行うかどうか[true/false]を指定します(初期値:true)
 *       bgColorCycle        【TAG】テーブルのバックグラウンドカラーの入れ替えのサイクルをセットします(初期値は、1:ゼブラ)
 *       bgColorClsClm       【TAG】テーブルのバックグラウンドカラーのクラスに指定するカラム名を指定します
 *       useScrollBar        【TAG】スクロールバー用のDIV要素を出力するかどうか[true/false]を指定します
										(初期値:VIEW_USE_SCROLLBAR[={@og.value SystemData#VIEW_USE_SCROLLBAR}])
 *       maxScrollBarSize    【TAG】ヘッダー固定のスクロールバー使用時の最大行数を指定します
										(初期値:VIEW_SCROLLBAR_MAX_ROW_SIZE[={@og.value SystemData#VIEW_SCROLLBAR_MAX_ROW_SIZE}])
 *       loadMarker          【TAG】viewLink や viewMarker の情報取り込みを制御する(初期値は、NEW,RENEW,REVIEW,RESET)
 *       useSelectedRow      【TAG】SEL_ROW機能[true:有効/false:無効]を指定します (初期値:VIEW_USE_SELECTED_ROW[=true])
 *       useSelRowColor      【TAG】SEL_ROW使用時の選択行色変更[true:有効/false:無効]を指定します (初期値:VIEW_USE_SELROW_COLOR[=true])
 *       useHilightRow       【TAG】HILIGHT_ROW機能[true:有効/false:無効]を指定します (初期値:VIEW_USE_HILIGHT_ROW[=true])
 *       useAfterHtmlTag     【TAG】処理時間やページサイズなどの情報出力[true:有効/false:無効]を指定します(初期値:true)
 *       writableControl     【TAG】設定値に "_" が含まれている場合にレンデラーを使用するカラムをCSV形式で指定します
 *       popupReturnKeys     【TAG】ogPopup で検索結果の値を返すキーを、CSV形式で指定します
 *       clazz               【TAG】table要素に対して class 属性を設定します(初期値:viewTable)
 *       language            【TAG】タグ内部で使用する言語コード[ja/en/zh/…]を指定します
 *       height              【TAG】ビューの高さを指定します
 *       width               【TAG】ビューの幅を指定します
 *       fileDwnClms         【TAG】ファイルダウンロード時の抜出カラム名(CSV形式)を指定します
 *       skipNoEdit          【TAG】編集行のみを表示するかを指定します(初期値:false)
 *       noTransition        【TAG】(通常は使いません)画面遷移を行わない形式の登録方法を使用するかを指定します
 *       useHilightCheckOnly 【TAG】選択行ハイライトの制御をチェックボックスのみで有効にするかを指定します (初期値:VIEW_USE_HILIGHT_CHECK_ONLY[=false])
 *       useHilightRadioClick【TAG】選択行ハイライトの制御でラジオのクリックイベントを発行するかどうかを指定します (初期値:VIEW_USE_HILIGHT_RADIO_CLICK[=true])
 *       columnBulkSet       【TAG】一括入力カラムのアシスト機能を利用するカラム名を、CSV形式で与えます
 *       useTimeView         【TAG】処理時間を表示する TimeView を表示するかどうかを指定します
 *										(初期値:VIEW_USE_TIMEBAR[={@og.value SystemData#VIEW_USE_TIMEBAR}])。
 *       caseKey             【TAG】このタグ自体を利用するかどうかの条件キーを指定します(初期値:null)
 *       caseVal             【TAG】このタグ自体を利用するかどうかの条件値を指定します(初期値:null)
 *       caseNN              【TAG】指定の値が、null/ゼロ文字列 でない場合(Not Null=NN)は、このタグは使用されます(初期値:判定しない)
 *       caseNull            【TAG】指定の値が、null/ゼロ文字列 の場合は、このタグは使用されます(初期値:判定しない)
 *       caseIf              【TAG】指定の値が、true/TRUE文字列の場合は、このタグは使用されます(初期値:判定しない)
 *       mainView            【TAG】(通常は使いません)セッションに情報を登録するかどうか[true/false]指定します。(初期値:true)
 *       debug               【TAG】デバッグ情報を出力するかどうか[true/false]を指定します(初期値:false)
 *   &gt;   ... Body ...
 *   &lt;/og:view&gt;
 *
 * ●使用例
 *       &lt;og:view viewFormId="HTMLTable" command="NEW" startNo="0" pageSize="5" / &gt;
 *
 * @og.group 画面表示
 *
 * @version  4.0
 * @author	 Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class ViewFormTag extends CommonTagSupport {
	/** このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "6.3.6.0 (2015/08/16)" ;
	private static final long serialVersionUID = 636020150816L ;

	/** command 引数に渡す事の出来る コマンド  プレビュー {@value}  */
	public static final String CMD_PREV  = "PREV" ;
	/** command 引数に渡す事の出来る コマンド  ネクスト {@value}  */
	public static final String CMD_NEXT  = "NEXT" ;
	/** command 引数に渡す事の出来る コマンド  ファースト {@value}  */
	public static final String CMD_FIRST = "FIRST";
	/** command 引数に渡す事の出来る コマンド  ラスト {@value}  */
	public static final String CMD_LAST  = "LAST" ;
	/** command 引数に渡す事の出来る コマンド  新規 {@value}  */
	public static final String CMD_NEW	 = "NEW" ;
	/** command 引数に渡す事の出来る コマンド  再検索 {@value}  */
	public static final String CMD_RENEW = "RENEW" ;
	/** command 引数に渡す事の出来る コマンド  表示 {@value}  */
	public static final String CMD_VIEW  = "VIEW" ;
	/** command 引数に渡す事の出来る コマンド  再表示 {@value}  */
	public static final String CMD_REVIEW  = "REVIEW" ;
	/** command 引数に渡す事の出来る コマンド  リセット {@value}  */
	public static final String CMD_RESET  = "RESET" ;

	/** command 引数に渡す事の出来る コマンド リスト  */
	private static final String[] COMMAND_LIST = { CMD_PREV , CMD_NEXT , CMD_FIRST , CMD_LAST , CMD_NEW , CMD_RENEW , CMD_VIEW , CMD_REVIEW , CMD_RESET };

	// 3.8.0.8 (2005/10/03)
	/** コマンドリストに存在しない場合の値 従来の CMD_VIEW と区別させるため。 */
	private static final String CMD_XXX  = "XXX" ;

	private static final String START_NO_KEY	= "startNo";
	private static final String PAGE_SIZE_KEY 	= "pageSize";
	private static final String PAGE_PLUS_KEY 	= "pagePlus";

	private transient DBTableModel		table ;
	private transient ViewForm			form  ;
//	private List<TableFormatter>		values;
	private transient List<TableFormatter>		values;		// 6.3.9.0 (2015/11/06) transient 追加

	private String		tableId			= HybsSystem.TBL_MDL_KEY;
	private String		viewFormId		= HybsSystem.VIEWFORM_KEY;
	private String		viewFormType	;
	private String		viewLinkId		= HybsSystem.VIEWLINK_KEY;
	private String		viewMarkerId	= HybsSystem.VIEW_MARK_KEY;
	private String		editMarkerId	= HybsSystem.EDIT_MARK_KEY;		// 3.8.6.1 (2006/10/20)
	private String		command			;
	private int			startNo			= -1;		// HTML画面上の表示開始位置(初期値)
	private int			tempStartNo		= -1;		// 再検索時の表示開始位置
	private int			pageSize 		= -1;		// HTML画面上に表示する件数
	private int			pagePlus 		= -1;		// 3.5.6.4 (2004/07/16)
	private String		columnWritable	;
	private String		noWritable		;
	private String		columnDisplay	;
	private String		noDisplay		;
	private String		clmSeqDisplay	;		// 5.1.6.0 (2010/05/01) ViewForm_HTMLSeqClmTable 対応
	private String		columnGroup		;		// 3.8.5.0 (2006/03/20)
	private String		noGroup			;		// 5.1.8.0 (2010/07/01)
	private String		groupClass		;		// 5.1.8.0 (2010/07/01)
	private boolean		noMessage		;
	private String		writable 		;
	private String		checked			;
	private boolean		skip 			= ViewForm.DEFAULT_SKIP;
	private String		rowspan			= "2";
	private String		selectedType	= "checkbox";
	private String		numberType		= "sequence";		// 3.5.1.0 (2003/10/03) 追加
	private String		optTypeAttri	;
	private int			backLinkCount	= HybsSystem.sysInt( "VIEW_BACK_LINK_COUNT" );
	private int			headerSkipCount	= HybsSystem.sysInt( "VIEW_HEADER_SKIP_COUNT" );	// 0:通常ヘッダ、n:n回ごとに現れる
	private String		bodyString		;
	private int			useCheckControl = -1 ;
	private String		useTableSorter	;		// 3.5.4.7 (2004/02/06)

	private long dyStart				;		// 3.5.4.7 (2004/02/06) 実行時間測定用のDIV要素を出力します。
	private boolean	useTimeView			= HybsSystem.sysBool( "VIEW_USE_TIMEBAR" );		// 6.3.6.0 (2015/08/16)
	private boolean	useParam			;		// 3.5.4.8 (2004/02/23) パラメータTagの使用を宣言します。
	private Map<String,String> viewParam;		// 3.5.4.8 (2004/02/23) ViewParamTag のパラメータを受け取ります。
	private List<String[]> viewArrayList;		// 5.5.9.0 (2012/12/03) viewObjectから変更

	private transient DBLastSql	   lastSql	;	// 6.0.2.4 (2014/10/17) Edit機能で、JSPソース変更時の対応
	private transient DBEditConfig editConf	;	// 6.0.2.4 (2014/10/17) Edit機能で、JSPソース変更時の対応

	// 3.5.5.8 (2004/05/20) 整合性チェックを行う為の、Consistency キー を出力するかどうかを指定します。
	private boolean	useConsistency	= HybsSystem.sysBool( "USE_CONSISTENCY" ) ;

	// 0(ゼブラなし)、-1(ワーニング)、-2以下(エラー)、1(ゼブラ)、2以上(行数まとめ)
	private int bgColorCycle		= 1 ;		// 3.5.6.2 (2004/07/05) ゼブラ模様の指定を、ViewForm としてサポート
	private String	bgColorClsClm	;			// 5.1.8.0 (2010/07/01) 行に対して、動的にクラス属性を付与するカラム名を指定します。
	private String	useScrollBar	;
	private int		maxScrollBarSize = HybsSystem.sysInt( "VIEW_SCROLLBAR_MAX_ROW_SIZE" ) ;		// 6.2.4.0 (2015/05/15)

	private String	sortKeys		;			// 3.6.0.0 (2004/09/17) カラムソートリンクを出すカラム名を指定。

	// 3.6.0.4 (2004/10/14) viewLink や viewMarker の情報取り込みを制御
	// 前後の , が重要です！
	private String		loadMarker	= ",NEW,RENEW,REVIEW,RESET,";

	// 3.7.1.1 (2005/05/23) SEL_ROW機能の有効/無効を指定します。
	private String	useSelectedRow	;

	// 3.7.1.1 (2005/05/31) SEL_ROW機能使用時に、BG_COLOR_ROWSEL の使用 有/無を指定します。
	private String	useSelRowColor	;

	// 3.7.1.1 (2005/05/23) 処理時間やページサイズなどの情報出力の有効/無効を指定します。
	private boolean	useAfterHtmlTag	= true ;

	// 3.8.0.9 (2005/10/17) 設定値に "_" が含まれている場合に、レンデラーを使用します。
	private String	writableControl	;

	// 3.8.6.0 (2006/09/29) HILIGHT_ROW機能の有効/無効を指定します。
	private String	useHilightRow	;

	// 3.8.6.1 (2006/10/20) ogPopup で検索結果の値を返すキーを、CSV形式で指定します。
	private String	popupReturnKeys	;

	// 4.0.0 (2007/04/16) tableタグを出力するときに付与します。
	private String	clazz			= "viewTable";

	// 4.2.0.0 (2008/03/18) ビューの幅と高さを指定できるようにします。
	private String	height			;
	private String	width			;

	// 4.3.0.0 (2008/07/04) fileUD ファイルダウンロード時の抜出カラム名(CSV形式)
	private String	fileDwnClms		;

	private boolean	skipNoEdit		;	// 4.3.1.0 (2008/09/08)
	private boolean	noTransition	;	// 4.3.3.0 (2008/10/01)

	// 4.3.7.0 (2009/06/01) HILIGHT_ROW機能有効時に制御を行チェックボックスのみにします。
	private String	isHiRowCheckOnly;			// 6.0.2.5 (2014/10/31) refactoring
	// 6.2.2.0 (2015/03/27) hilightRow で、行選択時にイベント発行可否を指定できるようにする。
	private String	isHiRadioClick;

	// 5.2.1.0 (2010/10/01) 一括入力カラムのアシスト機能を利用するカラムを指定します。
	private String	columnBulkSet	;
	private boolean	useDefBulkset	= HybsSystem.sysBool( "VIEW_USE_DEFAULT_BULKSET" ) ;

	// 5.7.6.1 (2014/05/09) startNoのセッション保存を指定します。
	private boolean	isMainView		= true;	

	// 6.0.3.0 (2014/11/13) Edit機能の超特殊処理
	// 6.2.5.0 (2015/06/05) Edit機能でも、条件によっては、書き込み許可するので、特殊処理廃止

	/**
	 * Taglibの開始タグが見つかったときに処理する doStartTag() を オーバーライドします。
	 *
	 * @og.rev 3.5.4.7 (2004/02/06) DBTableModelのソート機能 useTableSorter を追加
	 * @og.rev 3.5.4.7 (2004/02/06) 実行時間測定用に、開始時刻を取得します。
	 * @og.rev 3.5.4.8 (2004/02/23) useParam を、BODY部の処理判定に加えます。
	 * @og.rev 3.5.6.0 (2004/06/18) DBRowHeader のパッケージプライベート化に伴なう変更
	 * @og.rev 3.5.6.1 (2004/06/25) viewFormType変更時のviewForm 再セット
	 * @og.rev 3.5.6.2 (2004/07/05) command="VIEW" などで、ありえないとしていた処理を追加
	 * @og.rev 3.5.6.3 (2004/07/12) markerSet で table の整合性もチェックする。
	 * @og.rev 3.5.6.4 (2004/07/16) DBTableModelのソート時には、viewを初期設定する。
	 * @og.rev 3.6.0.0 (2004/09/17) scope が request の場合(検索)ヘッダーソートリンクは使えない。
	 * @og.rev 3.6.0.0 (2004/09/22) pageSize の form への設定見直し。
	 * @og.rev 3.6.0.0 (2004/09/25) viewLink , viewMarker のリクエストメモリ削除の中止
	 * @og.rev 3.6.0.4 (2004/10/14) viewLink や viewMarker の情報取り込みを制御する loadMarker を追加
	 * @og.rev 3.6.0.5 (2004/10/18) viewLink , viewMarker のリクエストメモリ削除を復活します。
	 * @og.rev 3.8.0.8 (2005/10/03) ヘッダーソート時に、startNo を 0 に戻します。
	 * @og.rev 3.8.8.5 (2007/03/09) markerSet する条件を強化する。
	 * @og.rev 3.8.8.9 (2007/05/29) ソート時のviewLink,viewMarker の取り込み
	 * @og.rev 3.8.9.3 (2007/08/28) soryKeysのみを指定した場合に、ソート用のテーブルモデルが作成されないバグを修正。
	 * @og.rev 4.1.0.0(2008/01/11)システムパラメータ(useCheckControl)の読取は doEndTag 関数から doStartTag 関数に移動します。
	 * @og.rev 5.1.6.0 (2010/05/01) ViewForm_HTMLSeqClmTable クラス によるカラム表示順処理を行います。
	 * @og.rev 5.1.6.0 (2010/05/01) 左右分割スクロール画面でヘッダースクロールが2回処理されるバグを修正
	 * @og.rev 5.1.6.0 (2010/05/01) ユーザー毎の画面項目並び替え対応
	 * @og.rev 5.1.7.0 (2010/06/01) ViewFormに画面IDをセットし、セットされた画面IDと同じ場合のみViewFormを復元する。
	 * @og.rev 5.1.7.0 (2010/06/01) DBColumnの{&#064;XXXX}変数対応
	 * @og.rev 5.1.7.0 (2010/06/01) ヘッダーソート後に画面遷移し、戻ってきた際にstartNoがクリアされるバグを修正
	 * @og.rev 5.1.8.0 (2010/07/01) 全チェックを押した状態で画面遷移した場合にチェック状態を保持する。
	 * @og.rev 5.1.8.0 (2010/07/01) 同時設定不可(columnWritable,noWritable,columnDisplay,noDisplay,columnGroup,noGroup) チェック
	 * @og.rev 5.2.2.0 (2010/11/01) caseKey 、caseVal 属性対応
	 * @og.rev 5.3.0.0 (2010/12/01) 左右分割スクロール(SplitViewTag)対応
	 * @og.rev 5.3.6.0 (2011/06/01) Edit機能見直し
	 * @og.rev 5.4.1.0 (2011/11/01) 5.1.8.0 (2010/07/01)の全チェック保持機能を廃止
	 * @og.rev 6.0.2.4 (2014/10/17) Edit機能で、JSPソース変更時の対応
	 * @og.rev 6.0.3.0 (2014/11/13) Edit機能の超特殊処理
	 * @og.rev 6.2.0.0 (2015/02/27) PopupのBulkSet Off化にも使うので、HTMLSeqClmTable の条件は外す。
	 * @og.rev 6.2.5.0 (2015/06/05) Edit機能改善。viewLink,viewMarker の取り込みは、ソート時と、DBEditConfig 使用時とする。
	 *
	 * @return	後続処理の指示
	 */
	@Override
	public int doStartTag() {
		// 5.2.2.0 (2010/11/01) caseKey 、caseVal 属性対応
		if( !useTag() ) { return SKIP_BODY ; }	// Body を評価しない
		dyStart = System.currentTimeMillis();

		// 5.3.0.0 (2010/12/01) 左右分割スクロール(SplitViewTag)対応
		splitViewParamSet();

		// 5.1.8.0 (2010/07/01) 同時設定不可(columnWritable,noWritable,columnDisplay,noDisplay,columnGroup,noGroup) チェック
		if( columnWritable != null && noWritable != null ) {
			final String errMsg = "columnWritable と noWritable は同時に指定できません。"
							+ "columnWritable = [" + columnWritable
							+ "] , noWritable = [" + noWritable
							+ "]";
			throw new HybsSystemException( errMsg );
		}

		if( columnDisplay != null && noDisplay != null ) {
			final String errMsg = "columnDisplay と noDisplay は同時に指定できません。"
							+ "columnDisplay = [" + columnDisplay
							+ "] , noDisplay = [" + noDisplay
							+ "]";
			throw new HybsSystemException( errMsg );
		}

		if( columnGroup != null && noGroup != null ) {
			final String errMsg = "columnGroup と noGroup は同時に指定できません。"
							+ "columnGroup = [" + columnGroup
							+ "] , noGroup = [" + noGroup
							+ "]";
			throw new HybsSystemException( errMsg );
		}

		table = (DBTableModel)getObject( tableId );
		if( table == null ) { return SKIP_BODY ; }

		// 5.1.6.0 (2010/05/01)ViewForm_HTMLSeqClmTable クラス によるカラム表示順処理を行います。
		// clmSeqDisplay 属性が設定されていると、クラス指定を変更し、columnDisplay に、clmSeqDisplay を
		// 設定し、noDisplay を、削除(null をセット)します。
		if( clmSeqDisplay != null ) {
			viewFormType	= "HTMLSeqClmTable" ;
		}

		// 6.0.2.4 (2014/10/17) Edit機能で、JSPソース変更時の対応
		final String guikey = getGUIInfoAttri( "KEY" );
		lastSql = (DBLastSql)getSessionAttribute( HybsSystem.DB_LAST_SQL_KEY );
		// ここではtableId,scope,項目名などで厳密にチェックを行います。
		if( lastSql != null && lastSql.isGuiMatch( guikey ) && lastSql.isTableMatch( tableId, getScope() ) ) {
			final String editName = getRequestValue( "editName" );
			if( editName != null ) {
				editConf = getUser().getEditConfig( guikey, editName );
				if( editConf != null ) {
					viewFormType = "HTMLSeqClmTable";
				}
			}
		}

		// 5.1.7.0 (2010/06/01) DBColumnの{@XXXX}変数対応
		final int clmSize = table.getColumnCount();
		for( int clmNo=0; clmNo<clmSize; clmNo++ ) {
			final DBColumn dbColumn = table.getDBColumn( clmNo );
			if( dbColumn.isNeedsParamParse() ){
				final DBColumnConfig config = dbColumn.getConfig();
				config.setRendererParam( getRequestParameter( config.getRendererParam() ) );
				config.setEditorParam( getRequestParameter( config.getEditorParam() ) );
				final DBColumn newColumn = new DBColumn( config );
				table.setDBColumn( clmNo, newColumn );
			}
		}

		// 3.8.8.9 (2007/05/29) ソート時の viewLink,viewMarker の取り込み
		// 実際は、Formオブジェクトが変わった時も取り込む必要があるが、過去のソースとの関係で、今回は見送る。
		boolean useSorter = false;

		// 3.5.4.7 (2004/02/06) DBTableModelのソート機能 追加
		// 登録後に同じソート順で再表示させるために、リクエストキャッシュに登録します。
		// 3.8.9.3 (2007/08/28) soryKeysのみを指定した場合に、ソート用のテーブルモデルが作成されないバグを修正。
		// 5.1.6.0 (2010/05/01) 左右分割スクロール画面でヘッダースクロールが2回処理されるバグを修正
		if( "session".equals( getScope() )
			&& ( nval( useTableSorter,sysBool( "VIEW_USE_TABLE_SORTER" ) ) || sortKeys != null )
			&& !( table.getConsistencyKey().equals( getRequestAttribute( HybsSystem.VIEW_SORT_KEY ) ) ) ) {

			final String clm = getRequestValue( HybsSystem.SORT_COLUMNS );
			// 若干トリッキー。初めてクリックされるまで、DBTableModelSorter は作成しない。
			if( clm != null && clm.length() > 0 ) {
				useSorter = true;		// 3.8.8.9 (2007/05/29)
				startQueryTransaction( tableId );		// 3.6.0.8 (2004/11/19)
				final int clmNo = table.getColumnNo( clm );
				boolean sortAscending = nval( getRequestValue( HybsSystem.SORT_ASCENDING ),true );
				if( table instanceof DBTableModelSorter ) {
					if( !CMD_RESET.equals( command ) ) {
						((DBTableModelSorter)table).sortByColumn( clmNo );
					}
					sortAscending = ((DBTableModelSorter)table).isAscending() ;	// ソート方向
				}
				else {
					final DBTableModelSorter temp = new DBTableModelSorter();
					temp.setModel( table );
					temp.sortByColumn( clmNo,sortAscending );
					table = temp;
				}
				// 3.6.0.8 (2004/11/19) トランザクションチェックを行います。
				if( ! commitTableObject( tableId, table ) ) {
					jspPrint( "ViewFormTag " + getResource().getLabel( "ERR0041" ) ); // 5.8.8.0 (2015/06/05)
					return SKIP_PAGE;
				}
				if( CMD_VIEW.equals( command ) ) {
					setRequestCacheData( HybsSystem.SORT_COLUMNS,clm );
					setRequestCacheData( HybsSystem.SORT_ASCENDING,String.valueOf( sortAscending ) );
				}

				// 5.1.6.0 (2010/05/01)
				setRequestAttribute( HybsSystem.VIEW_SORT_KEY, table.getConsistencyKey() );

	 			// 3.8.0.8 (2005/10/03) ヘッダーソート時に、startNo を 0 に戻します。
				// 5.1.7.0 (2010/06/01) ヘッダーソート後に画面遷移し、戻ってきた際にstartNoがクリアされるバグを修正
				if( !CMD_RENEW.equals( command ) ) {
					startNo = 0;
					tempStartNo = startNo;
				}
			}
		}

		if( CMD_NEW.equals( command ) || CMD_RENEW.equals( command ) || CMD_REVIEW.equals( command ) || CMD_RESET.equals( command ) ) {
			form = ViewFormFactory.newInstance( viewFormType );		// 3.5.6.1 (2004/06/25)
			form.init( table );
			table.setDefaultRowWritable( DBTableModel.DEFAULT_WRITABLE );
			table.setDefaultRowChecked(  DBTableModel.DEFAULT_CHECKED );
		}
		else {
			// 3.5.6.1 (2004/06/25) viewFormType変更時のviewForm 再セット
			final ViewForm oldForm = (ViewForm)getObject( viewFormId );
			// 5.1.7.0 (2010/06/01) ViewFormに画面IDをセットし、セットされた画面IDと同じ場合のみViewFormを復元する。
			if( oldForm != null && guikey.equals( oldForm.getGamenId() ) ) {	// 6.0.2.4 (2014/10/17) 
				// 6.2.5.0 (2015/06/05) USE_EDIT_WITH_VIEW_CHANGE で、切り替え処理を行う。
				final String oldFormType = oldForm.getViewFormType();
				if( "HTMLSeqClmTable".equals( oldFormType ) && HybsSystem.sysBool( "USE_EDIT_WITH_VIEW_CHANGE" ) ) {
					viewFormType = "HTMLSeqClmTable" ;
				}
				if( oldFormType != null && oldFormType.equals( viewFormType ) ) {
					form = oldForm ;
					form.init( table );		// 6.2.5.0 (2015/06/05) 
				}
				else {
					form = ViewFormFactory.newInstance( viewFormType );
					// 6.2.5.0 (2015/06/05) form が切り替わった時のみ、markerSet する。
					form.init( table );
					// 3.5.6.3 (2004/07/12) markerSet で table の整合性もチェックする。
					final DBTableModel oldTable = oldForm.getDBTableModel();
					if( table.getConsistencyKey().equals( oldTable.getConsistencyKey() ) ) {
						// 3.8.8.5 (2007/03/09) markerSet する条件を強化する。
						// 3.8.8.9 (2007/05/29) ソート時の viewLink,viewMarker の取り込み
						// 6.2.5.0 (2015/06/05) viewLink,viewMarker の取り込みは、ソート時と、DBEditConfig 使用時とする。
						if( "true".equalsIgnoreCase( loadMarker ) ||
							loadMarker.indexOf( "," + command + "," ) >= 0 ||
							"HTMLSeqClmTable".equals( viewFormType ) ||			// 6.2.5.0 (2015/06/05)
							"HTMLSeqClmTable".equals( oldFormType )  ||			// 6.2.5.0 (2015/06/05)
							useSorter ) {
								form.markerSet( oldForm );
						}
					}
				}
			}
			else {
				// 3.5.6.2 (2004/07/05) command="VIEW" などで、ありえないとしていた処理を追加
				form = ViewFormFactory.newInstance( viewFormType );
				form.init( table );
			}
		}

		// 5.1.7.0 (2010/06/01) ViewFormに画面IDをセットし、セットされた画面IDと同じ場合のみViewFormを復元する。
		form.setGamenId( guikey );						// 6.0.2.4 (2014/10/17) 

		// 3.6.0.4 (2004/10/14) ViewMarker 等のセットは、loadMarker で制御します。
		if( "true".equalsIgnoreCase( loadMarker ) || loadMarker.indexOf( "," + command + "," ) >= 0 ) {
			final ViewMarker viewLink = (ViewMarker)getRequestAttribute( viewLinkId );
			if( viewLink != null ) {
				form.setViewLink( viewLink );
				removeRequestAttribute( viewLinkId );	// 3.6.0.5 (2004/10/18) 復活
			}

			final ViewMarker viewMarker = (ViewMarker)getRequestAttribute( viewMarkerId );
			if( viewMarker != null ) {
				form.setViewMarker( viewMarker );
				removeRequestAttribute( viewMarkerId );	// 3.6.0.5 (2004/10/18) 復活
			}

			// 3.8.6.1 (2006/10/24) editMarker を追加する。
			// 6.2.0.0 (2015/02/27) Popup(isRenderer=false のviewMarker)のBulkSet Off化対応
			final ViewMarker editMarker = (ViewMarker)getRequestAttribute( editMarkerId );
			if( editMarker != null ) {
				form.setEditMarker( editMarker );
				removeRequestAttribute( editMarkerId );
				// 6.0.3.0 (2014/11/13) Edit機能の超特殊処理
				// editMarkerのカラムは非表示設定する必要がある。
				// 6.2.0.0 (2015/02/27) PopupのBulkSet Off化にも使うので、HTMLSeqClmTable の条件は外す。
				// 6.2.5.0 (2015/06/05) Edit機能でも、条件によっては、書き込み許可するので、特殊処理廃止
			}
		}
		// 3.8.0.5 (2005/08/26) false の場合は、無条件に、form のviewLinkとViewMarkerをクリアします。
		if( "false".equalsIgnoreCase( loadMarker ) ) {
			form.setViewLink( null );
			form.setViewMarker( null );
			form.setEditMarker( null );		// 3.8.6.1 (2006/10/24)
		}
		// 4.1.0.0(2008/01/11)システムパラメータ(useCheckControl)の読取は doEndTag 関数から doStartTag 関数に移動します。
		if( useCheckControl < 0 ) {
			useCheckControl = intCase( sys( "VIEW_USE_CHECK_CONTROL" ) ) ;
		}
		setTableRowWritable( table );

		setDefaultParameter();

	//	if( pagePlus != 0 ) {	※ なぜ != だったのか謎。pagePlusがマイナス時は？
		if( pagePlus > 0 ) {
			form.setPageSize( pageSize + pagePlus );
		}
		else {
			form.setPageSize( pageSize );				// 3.6.0.0 (2004/09/22) 追加
		}

		// 3.5.4.8 (2004/02/23)
		if( form.canUseFormat() || useParam ) { return EVAL_BODY_BUFFERED ; }	// JSP API 1.2

		return SKIP_BODY ;
	}

	/**
	 * Taglibのタグ本体を処理する doAfterBody() を オーバーライドします。
	 *
	 * @og.rev 3.1.1.0 (2003/03/28) ボディの内容を取得する処理を、CommonTagSupport で行ｖう。
	 * @og.rev 3.1.1.2 (2003/04/04) Tomcat4.1 対応。ボディが存在する場合のみボディが呼ばれる対応。
	 * @og.rev 3.5.4.0 (2003/11/25) 外部からフォーマットが指定されない場合は、内部のフォーマットを登録する。
	 * @og.rev 5.1.7.0 (2010/06/01) フォーマットの{&#064;XXXX}の値に[が含まれる場合は、サイニタイズ("\\]\\"に変換)する。
	 *
	 * @return	後続処理の指示(SKIP_BODY)
	 */
	@Override
	public int doAfterBody() {
		bodyString = getSanitizedBodyString();
		return SKIP_BODY ;
	}

	/**
	 * Taglibの終了タグが見つかったときに処理する doEndTag() を オーバーライドします。
	 *
	 * @og.rev 3.0.1.3 (2003/03/11) startNo を隠しフィールドに持つ処理を、追加
	 * @og.rev 3.1.1.2 (2003/04/04) Tomcat4.1 対応。release2() を doEndTag()で呼ぶ。
	 * @og.rev 3.5.0.0 (2003/03/28) noClass 属性の追加。および、ロジック整理。
	 * @og.rev 3.5.1.0 (2003/10/03) numberType 属性の追加
	 * @og.rev 3.5.2.0 (2003/10/20) headerSkipCount属性を追加
	 * @og.rev 3.5.2.0 (2003/10/20) form が null の場合の条件分岐を追加
	 * @og.rev 3.5.3.1 (2003/10/31) Table系のViewで、skip属性を使用する。
	 * @og.rev 3.5.4.0 (2003/11/25) command="RESET"時に、戻った画面のpageSize がクリアされてしまう対応。
	 * @og.rev 3.5.4.0 (2003/11/25) フォーマット文字列を、TableFormatter オブジェクトを使用するように変更。
	 * @og.rev 3.5.4.1 (2003/12/01) startNo を隠しフィールドに持つ処理を廃止。(キャッシュを使用)
	 * @og.rev 3.5.4.3 (2004/01/05) useCheckControl属性を追加
	 * @og.rev 3.5.4.7 (2004/02/06) DBTableModelのソート機能 useTableSorter を追加
	 * @og.rev 3.5.4.7 (2004/02/06) makeStartNoKeyメソッドの廃止(setRequestCacheDataを利用)
	 * @og.rev 3.5.4.7 (2004/02/06) 実行時間測定用のDIV要素を出力しておきます。
	 * @og.rev 3.5.4.8 (2004/02/23) ViewParamTag のパラメータを追加します。
	 * @og.rev 3.5.5.5 (2004/04/23) DBTableModel の整合性パラメータを hidden で出力します。
	 * @og.rev 3.5.5.8 (2004/05/20) makeConsistencyKey メソッドに処理ロジックを移します。
	 * @og.rev 3.5.6.2 (2004/07/05) bgColorCycle 変数の追加。
	 * @og.rev 3.5.6.4 (2004/07/16) スクロールバー用のDIV要素の出力(開始)
	 * @og.rev 3.6.0.0 (2004/09/17) カラムソートリンクを出すカラム名を指定。
	 * @og.rev 3.6.0.0 (2004/09/22) pageSize の hidden 出力は、初期設定値とします。
	 * @og.rev 3.7.0.3 (2005/03/01) 指定の行番号まで画面をスクロールさせる機能を追加。
	 * @og.rev 3.7.1.1 (2005/05/23) 処理時間やページサイズなどの情報出力の有効/無効を指定します。
	 * @og.rev 3.8.0.3 (2005/07/15) ヘッダー固定のスクロールタイプをセットします。
	 * @og.rev 3.8.0.9 (2005/10/17) writableControl 属性を追加します。
	 * @og.rev 3.8.6.0 (2006/09/29) useHilightRow 属性を追加します。
	 * @og.rev 3.8.6.1 (2006/10/20) ポップアップで値を返すカラム文字列(CSV形式)
	 * @og.rev 4.0.0.0 (2007/11/27) ヘッダーカラムのエディター、レンデラー適用対応
	 * @og.rev 4.1.2.0 (2008/03/12) columnWritable,noWritable に、"null" を指定可能とする。
	 * @og.rev 4.2.0.0 (2008/03/18) ビューの高さ、幅の指定を可能にします。
	 * @og.rev 4.2.1.0 (2008/04/11) null チェック方法変更
	 * @og.rev 4.3.0.0 (2008/07/04) ロールモードによる設定
	 * @og.rev 4.3.0.0 (2008/07/04) fileUD 対応。抜き出すカラムを指定します。
	 * @og.rev 4.3.1.0 (2008/09/08) canUseFormatのエラーハンドリングは各サブクラスで行う。editOnly属性対応。
	 * @og.rev 4.3.3.0 (2008/10/01) noTransition 属性を追加します。
	 * @og.rev 4.3.6.0 (2009/04/01) eventColumn対応
	 * @og.rev 4.3.7.0 (2009/06/01) 行ハイライト機能でチェックボックスでのみON/OFFができるオプションを追加
	 * @og.rev 4.3.8.0 (2009/08/01) noTransition値取得メソッド名変更
	 * @og.rev 5.1.3.0 (2010/02/01) noTransitionのコントロールは、requestで行う。
	 * @og.rev 5.1.6.0 (2010/05/01) fileDwnClmsが指定されない場合は、Viewで表示されたカラムをセット
	 * @og.rev 5.1.7.0 (2010/06/01) 動的プルダウン実装見直し
	 * @og.rev 5.1.7.0 (2010/06/01) HilightRow用のDivは必ず出す(画面遷移なし登録(JavaScript)の正規表現でマッチさせるため))
	 * @og.rev 5.1.8.0 (2010/07/01) form.setNoGroup( noGroup ) , form.setGroupClass( groupClass ) 追加
	 * @og.rev 5.1.8.0 (2010/07/01) form.setBgColorClsClm( bgColorClsClm ) 追加
	 * @og.rev 5.1.8.0 (2010/07/01) Divを必ず出すのは、画面遷移なし登録の場合のみとする(CustomData対応)
	 * @og.rev 5.2.1.0 (2010/10/01) 一括入力カラムのアシスト機能を利用するカラムを指定します。(columnBulkSet)
	 * @og.rev 5.2.2.0 (2010/11/01) useHeaderColumn 属性を、ViewCrossTableParam.USE_HEADER_COLUMN をキーに、取得する
	 * @og.rev 5.2.2.0 (2010/11/01) caseKey 、caseVal 属性対応
	 * @og.rev 5.5.8.3 (2012/11/17) viewObject追加
	 * @og.rev 5.5.9.0 (2012/12/03) viewObjectをviewArrayListに変更
	 * @og.rev 5.7.6.1 (2014/05/09) mainView対応
	 * @og.rev 6.0.2.4 (2014/10/17) Edit機能で、JSPソース変更時の対応
	 * @og.rev 6.0.3.0 (2014/11/13) Edit機能の超特殊処理
	 * @og.rev 6.2.2.0 (2015/03/27) hilightRow で、行選択時にイベント発行可否を指定できるようにする。
	 * @og.rev 6.2.5.0 (2015/06/05) Edit機能改善。条件によっては、書き込み許可するので、特殊処理廃止
	 *
	 * @return	後続処理の指示
	 */
	@Override
	public int doEndTag() {
		debugPrint();		// 4.0.0 (2005/02/28)
		// 5.2.2.0 (2010/11/01) caseKey 、caseVal 属性対応
		if( !useTag() ) { return EVAL_PAGE ; }

		// noTransition = isNoTransitionRequest() || noTransition; // 4.3.3.0 (2008/10/01) 追加
		noTransition = isNoTransitionRequest(); // 5.1.3.0 (2010/02/01)
		if( form == null ) { return EVAL_PAGE ; }	// 3.5.2.0 (2003/10/20)
		if( form.canUseFormat() ) {
			if( values == null && bodyString != null && !useParam ) {
				final TableFormatter format = new TableFormatter();
				format.setFormatType( FormatterType.TYPE_HEAD );
				format.setFormat( bodyString );
				format.setRowspan( rowspan );
				values = new ArrayList<>();
				values.add( format );
				form.setFormatterList( values );
			}
			else if( values != null ) {
				form.setFormatterList( values );
			}
		}

		if( table != null ) {
			// 4.2.0.0 (2008/03/18) ビューの高さ、幅を指定する。null回避はAbstractViewFormで行う
			form.setHeight( height );
			form.setWidth( width );

			form.setColumnWritable( columnWritable );
			form.setNoWritable( noWritable );
			form.setColumnDisplay( columnDisplay );
			form.setNoDisplay( noDisplay );
			form.setColumnGroup( columnGroup );				// 3.8.5.0 (2006/03/20)
			form.setNoGroup( noGroup );						// 5.1.8.0 (2010/07/01) 追加
			form.setGroupClass( groupClass );				// 5.1.8.0 (2010/07/01) 追加
			form.setTableSorterKeys( sortKeys );			// 3.6.0.0 (2004/09/17) 追加
			form.setNoMessage( noMessage );
			form.setSelectedType( selectedType );
			form.setUseCheckControl( useCheckControl );		// 3.5.4.3 (2004/01/05) 追加
			form.setNumberType( numberType );				// 3.5.1.0 (2003/10/03) 追加
			form.setOptionTypeAttributes( optTypeAttri );
			form.setBackLinkCount( backLinkCount );
			form.setHeaderSkipCount( headerSkipCount );		// 3.5.2.0 (2003/10/20) 追加
			if( useParam ) { 
				form.setParam( viewParam ); 	// 3.5.4.8 (2004/02/23)
				form.setViewArrayList( viewArrayList ); // 5.5.9.0 (2012/12/03)
			}
			form.setBgColorCycle( bgColorCycle );			// 3.5.6.2 (2004/07/05) 追加
			form.setBgColorClsClm( bgColorClsClm );			// 5.1.8.0 (2010/07/01) 追加
			form.setWritableControl( writableControl );		// 3.8.0.9 (2005/10/17) 追加

			// 5.2.1.0 (2010/10/01) 初期バルクセットが true で、かつ、変数が未設定の場合は、全カラム(*)選択とする。
			if( useDefBulkset && columnBulkSet == null ) { columnBulkSet = "*"; }
			// selectedType が radio の場合は、カラムの設定を行わない。
			if( "radio".equalsIgnoreCase( selectedType ) ) { columnBulkSet = null; }
			form.setColumnBulkSet( columnBulkSet );			// 5.2.1.0 (2010/10/01) 追加

			// 6.0.2.4 (2014/10/17) Edit機能で、JSPソース変更時の対応
			// editConf が null でないという事は、lastSql の条件がクリアされている。
			if( editConf != null ) {
				// 6.0.3.0 (2014/11/13) Edit機能の超特殊処理
				// setColumnDisplay(String) の後でないと、リセットされてしまう。
				// 6.2.5.0 (2015/06/05) Edit機能でも、条件によっては、書き込み許可するので、特殊処理廃止
	//			if( editMarkerClmNos != null ) {
	//				for( int i=0; i<editMarkerClmNos.length; i++ ) {
	//					form.setColumnDisplay( editMarkerClmNos[i] , false );	// 表示しない
	//				}
	//			}

				final String orgClms = form.getViewClms();		// form.setColumnDisplay(String) の再セット前に取り出す。
				lastSql.setOrgClmNames( orgClms );	
				final String userViewClms = editConf.getViewClms( orgClms );
				makeEditTable( userViewClms );					// ここで、clmSeqDisplay を再セットしている。
			}

			// 6.0.2.4 (2014/10/17) makeEditTable メソッドで、clmSeqDisplay を再セットしている。
			if( clmSeqDisplay != null ) {
				form.setColumnDisplay( clmSeqDisplay );		// noDisplay は無視されます。
			}

			// 3.6.0.0 (2004/09/22) pageSize の hidden 出力は、初期設定値とします。
			final int oldPageSize = pageSize;
			pageSize = form.getPageSize();
			commandExec( command );

			form.setStartNo( startNo );
			form.setSkip( skip );							// 3.5.3.1 (2003/10/31)
			form.setSkipNoEdit( skipNoEdit );				// 4.3.1.0 (2008/09/08)
			final boolean useTblSorter = "session".equals( getScope() ) &&
						nval( useTableSorter,sysBool( "VIEW_USE_TABLE_SORTER" ) );
			form.setUseTableSorter( useTblSorter );		// 3.5.4.7 (2004/02/06)

			// 3.6.0.0 (2004/09/17)
			// tableSorter 時のリンクに追加するリクエスト変数のキャッシュMapを追加する。
			if( useTblSorter || sortKeys != null ) {
				// 4.0.0 (2005/01/31)
				final String memKey = HybsSystem.REQ_CACHE_KEY + getGUIInfoAttri( "KEY" );
				final Map<?,?> mem = (Map<?,?>)getSessionAttribute( memKey );		// 4.3.3.6 (2008/11/15) Generics警告対応
				form.makeTableSorterQuery( mem );
			}

			// 3.5.6.4 (2004/07/16) スクロールバー用のDIV要素の出力(開始)
			// 3.8.0.3 (2005/07/15) ヘッダー固定のスクロールタイプをセットします。
			// 4.0.0.0 (2007/04/10) ヘッダー固定のスクロールタイプは、一つにする。

			// 6.2.4.0 (2015/05/15) maxScrollBarSize 追加
			final boolean useScBar =nval( useScrollBar,sysBool( "VIEW_USE_SCROLLBAR" ) )
								& Math.min( pageSize,table.getRowCount() ) < maxScrollBarSize ;

			form.setUseScrollBar( useScBar );

			// 3.8.6.1 (2006/10/20) ポップアップで値を返すカラム文字列(CSV形式)
			form.setPopupReturnKeys( popupReturnKeys );
			form.setTableClass( clazz );		// 4.0.0 (2007/04/16)

			// 3.7.0.3 (2005/03/01) 指定の行番号まで画面をスクロールさせる機能を追加。{I} は無視する。
			final String rowval = nval( getRequestValue( "SEL_ROW" ),null ) ;

			final boolean useSelRow = nval( useSelectedRow,sysBool( "VIEW_USE_SELECTED_ROW" ) );
			if( useSelRow && rowval != null && !rowval.equalsIgnoreCase( "{I}" ) ) {
				final boolean useSelRowClr = nval( useSelRowColor,sysBool( "VIEW_USE_SELROW_COLOR" ) ) ;
				form.setScrollRowNo( Integer.parseInt( rowval ),useSelRowClr );
			}

			// 4.0.0 (2005/11/30) カラムロールを判定して、アクセス不可の場合は表示させないようにする。
			final int clmSize = table.getColumnCount();
			final UserInfo userInfo = getUser();
			for( int clmNo=0; clmNo<clmSize; clmNo++ ) {
				final DBColumn dbColumn = table.getDBColumn( clmNo );

				// 4.3.6.0 (2009/04/01) セッションに生SQLを登録
				// 5.1.7.0 (2010/06/01) 動的プルダウン実装見直し
				final String evCol = dbColumn.getEventColumn();
				if( evCol != null && evCol.length() > 0 ){
					addEventColumn( dbColumn );
				}

				// 4.3.0.0 (2008/07/04) ロールモードによる設定
				final byte bitMode = userInfo.getAccessBitMode( dbColumn.getRoleMode() );
				// アクセス不許可の場合のみ、セットする。
				// それ以外は、columnDisplay ,noDisplay 属性が適用されている。
				if( !RoleMode.isRead( bitMode ) ) {
					form.setColumnDisplay( clmNo,false );
				}
				// 書込み不許可の場合のみ、セットする。
				// それ以外は、columnWritable ,noWritable 属性が適用されている。
				if( !RoleMode.isWrite( bitMode ) ) {
					form.setColumnWritable( clmNo,false );
				}
			}

			// 3.8.6.0 (2006/09/29) useHilightRow 属性追加
			final boolean useHiRow = nval( useHilightRow,sysBool( "VIEW_USE_HILIGHT_ROW" ) );
			// 5.1.7.0 (2010/06/01) HilightRow用のDivは必ず出す(画面遷移なし登録(JavaScript)の正規表現でマッチさせるため))
			// 5.1.8.0 (2010/07/01) Divを必ず出すのは、画面遷移なし登録の場合のみとする(CustomData対応)
			if( useHiRow || noTransition ) {
				jspPrint( "<div id=\"divHlr\"" );
				if( useHiRow ) {
					// 4.3.7.0 (2009/06/01) 6.0.2.5 (2014/10/31) refactoring
					final boolean useHiCkOnly = nval( isHiRowCheckOnly ,sysBool( "VIEW_USE_HILIGHT_CHECK_ONLY"  ) );
					// 6.2.2.0 (2015/03/27) hilightRow で、行選択時にイベント発行可否を指定できるようにする。
					final boolean useRdoClick = nval( isHiRadioClick   ,sysBool( "VIEW_USE_HILIGHT_RADIO_CLICK" ) );
					jspPrint( " onclick=\"hilightRow( event, " + useHiCkOnly + ","  + useRdoClick + ");\"" );
				}
				jspPrint( ">" );
			}

			// 5.2.2.0 (2010/11/01) useHeaderColumn 属性を、ViewCrossTableParam.USE_HEADER_COLUMN をキーに、取得する
			if( viewParam != null &&
					"true".equalsIgnoreCase( viewParam.get( ViewCrossTableParam.USE_HEADER_COLUMN ) ) ) {
				form.setResourceManager( getResource() );
			}

			form.setNoTransition( noTransition ); // 4.3.3.0 (2008/10/01) 追加

			jspPrint( form.create( startNo,pageSize ) );

			// 5.1.8.0 (2010/07/01) Divを必ず出すのは、画面遷移なし登録の場合のみとする(CustomData対応)
			if( useHiRow || noTransition ) {
				jspPrint( "</div>" );
			}

			// 5.7.6.1 (2014/05/09) mainView時のみstartNoを登録する
			if( isMainView ){
				setRequestCacheData( START_NO_KEY,String.valueOf( tempStartNo ) );	// 3.5.4.7 (2004/02/06)
			}

			// 3.6.0.0 (2004/09/22) pageSize の hidden 出力は、初期設定値とします。
			if( useAfterHtmlTag ) {	// 3.7.1.1 (2005/05/23)
				jspPrint( makePageSize( oldPageSize ) );		// 3.6.0.0 (2004/09/22)
			}

			makeConsistencyKey();		// 3.5.5.8 (2004/05/20)

			setObject( viewFormId, form );

			// 3.5.4.8 (2004/02/23) 超特殊処理 DBTableModel を View で置き換えます。
			// 5.2.2.0 (2010/11/01) saveTableId 属性を、ViewCrossTableParam.SAVE_SCOPE_KEY をキーに、取得する
			if( viewParam != null ) {
				final String saveTableId = viewParam.get( ViewCrossTableParam.SAVE_TABLEID_KEY );	// 5.2.2.0 (2010/11/01)
				if( saveTableId != null ) {
					final String saveScope = viewParam.get( ViewCrossTableParam.SAVE_SCOPE_KEY );
					setObject( saveTableId, form.getDBTableModel() , saveScope );
				}
			}

			// 6.0.2.4 (2014/10/17) Edit機能で、JSPソース変更時の対応
			// 5.1.6.0 (2010/05/01) Viewで表示した項目をDBLastSqlにセットします。
			if( table.getRowCount() > 0 && lastSql != null ) {
				setViewClms( lastSql );
			}
		}

//		if( useAfterHtmlTag ) {	// 3.7.1.1 (2005/05/23)
		if( useTimeView && useAfterHtmlTag ) {		// 6.3.6.0 (2015/08/16)
			// 3.5.4.7 (2004/02/06)
			final long dyTime = System.currentTimeMillis()-dyStart;
			jspPrint( "<div id=\"viewTime\" value=\"" + (dyTime) + "\"></div>" );	// 3.5.6.3 (2004/07/12)
		}

		return EVAL_PAGE ;
	}

	/**
	 * タグリブオブジェクトをリリースします。
	 * キャッシュされて再利用されるので、フィールドの初期設定を行います。
	 *
	 * @og.rev 2.0.0.4 (2002/09/27) カスタムタグの release() メソッドを、追加
	 * @og.rev 2.0.0.4 (2002/09/27) カスタムタグの release() メソッドを、追加
	 * @og.rev 3.0.1.3 (2003/03/11) startNo を隠しフィールドに持つ処理を、追加
	 * @og.rev 3.1.1.2 (2003/04/04) Tomcat4.1 対応。release2() を doEndTag()で呼ぶ。
	 * @og.rev 3.5.2.0 (2003/10/20) headerSkipCount属性を追加
	 * @og.rev 3.5.3.0 (2003/10/27) backLinkCount,headerSkipCount属性の初期値をPropertyファイルから設定
	 * @og.rev 3.5.4.0 (2003/11/25) format 関係の文字列を廃止。TableFormatter クラスを使用する。
	 * @og.rev 3.5.4.3 (2004/01/05) useCheckControl属性を追加
	 * @og.rev 3.5.4.7 (2004/02/06) DBTableModelのソート機能 useTableSorter を追加
	 * @og.rev 3.5.4.7 (2004/02/06) 実行時間測定用に、dyStart を追加します。
	 * @og.rev 3.5.4.8 (2004/02/23) ViewParamTag のパラメータを追加します。
	 * @og.rev 3.5.5.8 (2004/05/20) useConsistency 属性を追加します。
	 * @og.rev 3.5.6.2 (2004/07/05) bgColorCycle 変数の追加。
	 * @og.rev 3.5.6.3 (2004/07/12) useScrollBar 変数の追加。
	 * @og.rev 3.6.0.0 (2004/09/17) sortKeys 変数の追加。カラムソートリンクを出すカラム名を指定。
	 * @og.rev 3.6.0.4 (2004/10/14) loadMarker 変数の追加。
	 * @og.rev 3.7.0.1 (2005/01/31) 全件チェックコントロール変更(useCheckControl)
	 * @og.rev 3.7.1.1 (2005/05/23) SEL_ROW機能の有効/無効を指定します。
	 * @og.rev 3.7.1.1 (2005/05/23) 処理時間やページサイズなどの情報出力の有効/無効を指定します。
	 * @og.rev 3.8.0.9 (2005/10/17) writableControl 属性を追加します。
	 * @og.rev 3.8.6.0 (2006/09/29) useHilightRow 属性を追加します。
	 * @og.rev 3.8.6.1 (2006/10/20) editMarkerId , popupReturnKeys属性を追加します。
	 * @og.rev 4.3.0.0 (2008/07/04) fileUD ファイルダウンロード時の抜出カラム名(CSV形式)
	 * @og.rev 4.3.1.0 (2008/09/08) isSkipNoEdit属性対応
	 * @og.rev 4.3.3.0 (2008/10/01) noTransition 属性を追加します。
	 * @og.rev 4.3.7.0 (2009/06/01) useHilightCheckOnly 属性を追加します。
	 * @og.rev 5.1.6.0 (2010/05/01) clmSeqDisplay 属性を追加します。
	 * @og.rev 5.1.8.0 (2010/07/01) noGroup , noGroup 属性を追加します。
	 * @og.rev 5.1.8.0 (2010/07/01) bgColorClsClm 属性を追加します。
	 * @og.rev 5.2.1.0 (2010/10/01) 一括入力カラムのアシスト機能を利用する、columnBulkSet 属性を追加します。
	 * @og.rev 5.5.9.0 (2012/12/03) viewArrayList追加
	 * @og.rev 5.7.6.1 (2014/05/09) mainView追加
	 * @og.rev 6.0.2.4 (2014/10/17) Edit機能で、JSPソース変更時の対応
	 * @og.rev 6.0.3.0 (2014/11/13) Edit機能の超特殊処理。editMarkerClmNos 追加
	 * @og.rev 6.2.2.0 (2015/03/27) hilightRow で、行選択時にイベント発行可否を指定できるようにする。isHiRadioClick 追加
	 * @og.rev 6.2.4.0 (2015/05/15) maxScrollBarSize 追加
	 * @og.rev 6.2.5.0 (2015/06/05) Edit機能改善。条件によっては、書き込み許可するので、特殊処理廃止
	 * @og.rev 6.3.6.0 (2015/08/16) useTimeView の初期値を、VIEW_USE_TIMEBAR にする。
	 */
	@Override
	protected void release2() {
		super.release2();
		tableId				= HybsSystem.TBL_MDL_KEY;
		viewFormId			= HybsSystem.VIEWFORM_KEY;
		viewFormType		= null;
		viewLinkId			= HybsSystem.VIEWLINK_KEY;
		viewMarkerId		= HybsSystem.VIEW_MARK_KEY;
		editMarkerId		= HybsSystem.EDIT_MARK_KEY;		// 3.8.6.1 (2006/10/20)
		command				= null;
		startNo				= -1;			// HTML画面上の表示開始位置(初期値)
		tempStartNo			= -1;			// 再検索時の表示開始位置
		pageSize 			= -1;			// HTML画面上に表示する件数
		pagePlus 			= -1;			// 3.5.6.4 (2004/07/16)
		table				= null;
		form 				= null;
		columnWritable		= null;
		noWritable			= null;
		columnDisplay		= null;
		noDisplay			= null;
		clmSeqDisplay		= null;			// 5.1.6.0 (2010/05/01) ViewForm_HTMLSeqClmTable 対応
		columnGroup			= null;			// 3.8.5.0 (2006/03/20)
		noGroup				= null;			// 5.1.8.0 (2010/07/01)
		groupClass			= null;			// 5.1.8.0 (2010/07/01)
		noMessage			= false;
		writable 			= null;
		checked				= null;
		skip 				= ViewForm.DEFAULT_SKIP;
		rowspan				= "2";
		selectedType		= "checkbox";
		numberType			= "sequence";
		optTypeAttri		= null;
		values				= null;
		backLinkCount		= HybsSystem.sysInt( "VIEW_BACK_LINK_COUNT" );
		headerSkipCount		= HybsSystem.sysInt( "VIEW_HEADER_SKIP_COUNT" );	// 0:通常ヘッダ、n:n回ごとに現れる
		bodyString			= null;
		useCheckControl		= -1 ;
		useTableSorter		= null ;		// 3.5.4.7 (2004/02/06)
//		dyStart				= 0;			// 3.5.4.7 (2004/02/06)
		useTimeView			= HybsSystem.sysBool( "VIEW_USE_TIMEBAR" );	// 6.3.6.0 (2015/08/16)
		viewParam			= null;			// 3.5.4.8 (2004/02/23)
		useParam			= false;		// 3.5.4.8 (2004/02/23)
		useConsistency		= HybsSystem.sysBool( "USE_CONSISTENCY" );		// 3.5.5.8 (2004/05/20)
		bgColorCycle		= 1 ;			// 3.5.6.2 (2004/07/05)
		bgColorClsClm		= null;			// 5.1.8.0 (2010/07/01)
		useScrollBar		= null;
		maxScrollBarSize	= HybsSystem.sysInt( "VIEW_SCROLLBAR_MAX_ROW_SIZE" ) ;		// 6.2.4.0 (2015/05/15)
		sortKeys			= null;			// 3.6.0.0 (2004/09/17)
		loadMarker			= ",NEW,RENEW,REVIEW,RESET,";					// 4.0.0.0 (2007/11/26)
		useSelectedRow		= null;
		useSelRowColor		= null;
		useAfterHtmlTag		= true;			// 3.7.1.1 (2005/05/23)
		writableControl		= null;			// 3.8.0.9 (2005/10/17)
		useHilightRow		= null;			// 3.8.6.0 (2006/09/29)
		popupReturnKeys		= null;			// 3.8.6.1 (2006/10/20)
		clazz				= "viewTable";	// 4.0.0 (2007/04/16)
		height				= null;			// 4.2.0.0 (2008/03/18)
		width				= null;			// 4.2.0.0 (2008/03/18)
		fileDwnClms			= null;			// 4.3.0.0 (2008/07/04)
		skipNoEdit			= false;		// 4.3.1.0 (2008/09/08)
		noTransition		= false;		// 4.3.3.0 (2008/10/01)
		isHiRowCheckOnly	= null;			// 4.3.7.0 (2009/06/01) 		// 6.0.2.5 (2014/10/31) refactoring
		isHiRadioClick		= null;			// 6.2.2.0 (2015/03/27)
		columnBulkSet		= null;			// 5.2.1.0 (2010/10/01)
		useDefBulkset		= HybsSystem.sysBool( "VIEW_USE_DEFAULT_BULKSET" ) ;		// 5.2.1.0 (2010/10/01)
		viewArrayList		= null;			// 5.5.9.0 (2012/12/03)
		isMainView			= true;			// 5.7.6.1 (2014/05/09)
		lastSql				= null;			// 6.0.2.4 (2014/10/17) Edit機能で、JSPソース変更時の対応
		editConf			= null;			// 6.0.2.4 (2014/10/17) Edit機能で、JSPソース変更時の対応
	}

	/**
	 * 左右分割スクロール(SplitViewTag)が存在すれば、パラメータの初期値設定などを行います。
	 *
	 * @og.rev 5.3.0.0 (2010/12/01) 左右分割スクロール(SplitViewTag)対応
	 * @og.rev 5.5.4.0 (2012/07/02) 2回目のnoDisplayを追記型に変更
	 */
	private void splitViewParamSet() {
		final SplitViewTag splitView = (SplitViewTag)findAncestorWithClass( this,SplitViewTag.class );
		if( splitView != null ) {
			final String  fixDisplay  = splitView.getFixDisplay();
			final boolean isFirstStep = splitView.isFirstStep();

			if( isFirstStep ) {	// １回目(左側:固定部)
				// Viewタグの左側には、fixDisplay のカラムを columnDisplay に設定する。
				columnDisplay = fixDisplay;
				noDisplay     = null;
			}
			else {				// ２回目(右側:可変部)
				// Viewタグの右側には、command="VIEW" , noMessage="true" , useSelectedRow="false"
				// numberType="delete" をセットするとともに、fixDisplay のカラムを noDisplay の
				columnDisplay = null;
				if( noDisplay != null && noDisplay.length() > 0 ){
					noDisplay     += ","+fixDisplay; // 5.5.4.0 (2012/07/02) CSV形式で追記する
				}
				else{
					noDisplay     = fixDisplay;
				}

				command			= "VIEW";
				numberType		= "delete" ;
				noMessage		= true;
				useSelectedRow	= "false";
			}
		}
	}

	/**
	 * 【TAG】(通常は使いません)結果のDBTableModelを、sessionに登録するときのキーを指定します
	 *		(初期値:HybsSystem#TBL_MDL_KEY[={@og.value HybsSystem#TBL_MDL_KEY}])。
	 *
	 * @og.tag
	 * 検索結果より、DBTableModelオブジェクトを作成します。これを、下流のviewタグ等に
	 * 渡す場合に、通常は、session を利用します。その場合の登録キーです。
	 * query タグを同時に実行して、結果を求める場合、同一メモリに配置される為、
	 * この tableId 属性を利用して、メモリ空間を分けます。
	 *		(初期値:HybsSystem#TBL_MDL_KEY[={@og.value HybsSystem#TBL_MDL_KEY}])。
	 *
	 * @param	id テーブルID (sessionに登録する時のID)
	 */
	public void setTableId( final String id ) {
		tableId   = nval( getRequestParameter( id ),tableId );	// 3.8.0.9 (2005/10/17)
	}

	/**
	 * 【TAG】(通常使いません)sessionから取得する ViewForm オブジェクトの ID。
	 *
	 * @og.tag
	 * 表示処理後に，(内部ポインタを書き換えた)ViewForm オブジェクトを
	 * 同じキーで、sessionに登録します。
	 * 初期値は、HybsSystem.VIEWFORM_KEY です。
	 *
	 * @og.rev 3.3.1.1 (2003/07/03) {&#064;XXXX} 変数が使用できるように修正。
	 * @og.rev 3.5.6.4 (2004/07/16) VIEW_ID を付加して、他のid と混同しないようにします。
	 *
	 * @param	id ViewFormオブジェクトID
	 */
	public void setViewFormId( final String id ) {
		final String temp = getRequestParameter( id ) ;
		if( temp != null && temp.length() > 0 ) {
			viewFormId = temp + TaglibUtil.VIEW_ID;
		}
	}

	/**
	 * 【TAG】(通常使いません)requestから取得する ViewLink に対応する Attributes オブジェクトの ID。
	 *
	 * @og.tag
	 * ViewForm オブジェクトに、ViewLink オブジェクトをこのキーより取得することにより、
	 * カラムに対して、リンク情報を付加して表示させる。
	 * 初期値は、HybsSystem.VIEWLINK_KEY です。
	 *
	 * @og.rev 3.1.4.0 (2003/04/18) 初期値を考慮して、属性登録を行うように変更。
	 * @og.rev 3.3.1.1 (2003/07/03) {&#064;XXXX} 変数が使用できるように修正。
	 * @og.rev 3.5.6.4 (2004/07/16) LINK_ID を付加して、他のid と混同しないようにします。
	 *
	 * @param	id ViewLinkオブジェクトID
	 */
	public void setViewLinkId( final String id ) {
		final String temp = getRequestParameter( id ) ;
		if( temp != null && temp.length() > 0 ) {
			viewLinkId = temp + TaglibUtil.LINK_ID;
		}
	}

	/**
	 * 【TAG】(通常使いません)requestから取得する ViewMarker に対応する Attributes オブジェクトの ID。
	 *
	 * @og.tag
	 * ViewForm オブジェクトに、ViewMarker オブジェクトをこのキーより取得することにより、
	 * カラムに対して、マーカー情報を付加して表示させる。
	 * 初期値は、HybsSystem.VIEW_MARK_KEY です。
	 *
	 * @og.rev 3.1.4.0 (2003/04/18) 新規追加
	 * @og.rev 3.3.1.1 (2003/07/03) {&#064;XXXX} 変数が使用できるように修正。
	 * @og.rev 3.5.6.4 (2004/07/16) MARK_ID を付加して、他のid と混同しないようにします。
	 *
	 * @param	id ViewMarkerオブジェクトID
	 */
	public void setViewMarkerId( final String id ) {
		final String temp = getRequestParameter( id ) ;
		if( temp != null && temp.length() > 0 ) {
			viewMarkerId = temp + TaglibUtil.MARK_ID;
		}
	}

	/**
	 * 【TAG】(通常使いません)requestから取得する EditMarker に対応する Attributes オブジェクトの ID。
	 *
	 * @og.tag
	 * ViewForm オブジェクトに、EditMarker オブジェクト(実際は、ViewMarker オブジェクト)を
	 * このキーより取得することにより、カラムに対して、マーカー情報を付加して表示させる。
	 * 初期値は、HybsSystem.EDIT_MARK_KEY です。
	 *
	 * @og.rev 3.8.6.1 (2006/10/20) 新規追加
	 *
	 * @param	id EditMarkerオブジェクトID (実際は、ViewMarker オブジェクト)
	 */
	public void setEditMarkerId( final String id ) {
		final String temp = getRequestParameter( id ) ;
		if( temp != null && temp.length() > 0 ) {
			editMarkerId = temp + TaglibUtil.MARK_ID;
		}
	}

	/**
	 * 【TAG】ViewForm オブジェクトを作成するクラス({@og.doc03Link viewFormType ViewForm_**** クラスの ****})を指定します。
	 *
	 * @og.tag
	 * これは、org.opengion.hayabusa.html 以下の ViewForm_**** クラスの **** を
	 * 与えます。これらは、ViewForm インターフェースを継承したサブクラスです。
	 * 属性クラス定義の {@link org.opengion.hayabusa.html.ViewForm  ViewForm} を参照願います。
	 * {@og.doc03Link viewFormType ViewForm_**** クラス}
	 *
	 * @param	id ViewFormクラス
	 * @see		org.opengion.hayabusa.html.ViewForm  ViewFormのサブクラス
	 */
	public void setViewFormType( final String id ) {
		viewFormType = getRequestParameter( id );

		if( viewFormType == null || viewFormType.isEmpty() ) {
			final String errMsg = "viewForm の id が設定されていません。";
			throw new HybsSystemException( errMsg );
		}
	}

	/**
	 * 【TAG】コマンド (PREV,NEXT,FIRST,LAST,NEW,RENEW,VIEW,REVIEW,RESET)をセットします。
	 *
	 * @og.tag
	 * コマンドは,HTMLから(get/post)指定されますので,CMD_xxx で設定される
	 * フィールド定数値のいづれかを、指定できます。
	 *
	 * @og.rev 3.8.0.8 (2005/10/03) その他のコマンドを CMD_XXX にします。
	 * @og.rev 6.3.4.0 (2015/08/01) checkメソッド内で、nullチェックも行っている。
	 *
	 * @param	cmd コマンド (public static final 宣言されている文字列)
	 * @see		<a href="../../../../constant-values.html#org.opengion.hayabusa.taglib.ViewFormTag.CMD_PREV">コマンド定数</a>
	 */
	public void setCommand( final String cmd ) {
		final String cmd2 = getRequestParameter( cmd );
		if( cmd2 != null && cmd2.length() > 0 ) { command = cmd2.toUpperCase(Locale.JAPAN); }

//		if( command == null ||
//			 (command.trim()).isEmpty() ||
//			! check( command, COMMAND_LIST ) ) {
		if( ! check( command, COMMAND_LIST ) ) {
				// 3.8.0.8 (2005/10/03) その他のコマンドを CMD_XXX にします。
				command = CMD_XXX ;
		}
	}

	/**
	 * 【TAG】表示データを作成する場合の表示の開始行番号(先頭は0)をセットします。
	 *
	 * @og.tag
	 * セットされていない場合は，変更しません。
	 *
	 * @og.rev 3.0.1.3 (2003/03/11) startNo を隠しフィールドに持つ処理を、追加
	 *
	 * @param	no 表示開始
	 */
	public void setStartNo( final String no ) {
		startNo = nval( getRequestParameter( no ),startNo );
		tempStartNo = startNo;
	}

	/**
	 * 【TAG】表示データを作成する場合の１ページの行数をセットします。
	 *
	 * @og.tag
	 * セットされていない場合は，変更しません。
	 *
	 * @param	sz 表示件数
	 */
	public void   setPageSize( final String sz ) {
		pageSize = nval( getRequestParameter( sz ),pageSize );
	}

	/**
	 * 【TAG】１ページの行数の増加分をセットします(初期値:0)。
	 *
	 * @og.tag
	 * セットされていない場合は，０(増加なし)です。
	 *
	 * @param	pp １ページの行数
	 */
	public void   setPagePlus( final String pp ) {
		pagePlus = nval( getRequestParameter( pp ),pagePlus );
	}

	/**
	 * 【TAG】ページの先頭へのリンクの間隔をセットします
	 *		(初期値:VIEW_BACK_LINK_COUNT[={@og.value SystemData#VIEW_BACK_LINK_COUNT}])。
	 *
	 * @og.tag
	 * 0 にセットすると、リンクは作成されません。
	 * テーブルのヘッダー固定ができないケースでは、データ件数が多いと、登録ボタンが見えなくなります。
	 * その場合、スクロールして上に上がるより、行番号に作成されリンクで、一気に上に戻るほうが早く処理できます。
	 * その場合、段組み等していると、間隔を少なくしておく必要が出てきます。
	 * ヘッダー固定を利用する場合は、不要です。
	 * (初期値:システム定数のVIEW_BACK_LINK_COUNT[={@og.value SystemData#VIEW_BACK_LINK_COUNT}])。
	 *
	 * @param	blc リンクの間隔
	 * @see		org.opengion.hayabusa.common.SystemData#VIEW_BACK_LINK_COUNT
	 */
	public void   setBackLinkCount( final String blc ) {
		backLinkCount = nval( getRequestParameter( blc ),backLinkCount );
	}

	/**
	 * 【TAG】ヘッダーを出力する間隔をセットします
	 *		(初期値:VIEW_HEADER_SKIP_COUNT[={@og.value SystemData#VIEW_HEADER_SKIP_COUNT}])。
	 *
	 * @og.tag
	 * 0を指定すると、繰り返しません(つまり、ヘッダーを最初に表示するだけです。)
	 * 数字を指定すると、その回数毎に、ヘッダーをデータの行に挿入します。
	 * (初期値:システム定数のVIEW_HEADER_SKIP_COUNT[={@og.value SystemData#VIEW_HEADER_SKIP_COUNT}])。
	 *
	 * @og.rev 3.5.2.0 (2003/10/20) 新規作成
	 *
	 * @param	hsc ヘッダーの表示間隔 (0:通常ヘッダ,n:n回ごとに現れる)
	 * @see		org.opengion.hayabusa.common.SystemData#VIEW_HEADER_SKIP_COUNT
	 */
	public void   setHeaderSkipCount( final String hsc ) {
		headerSkipCount = nval( getRequestParameter( hsc ),headerSkipCount );
	}

	/**
	 * 【TAG】書き込み可能カラム名を、CSV形式で与えます。
	 *
	 * @og.tag
	 * これは、書き込み不可カラム名の指定(noWritable)と同時にセットする
	 * ことは出来ません。
	 * "*" を指定すると、すべてのカラムを(columnWritable)指定したことになります。
	 *
	 * @og.rev 3.3.1.1 (2003/07/03) {&#064;XXXX} 変数が使用できるように修正。
	 * @og.rev 4.1.2.0 (2008/03/12) 引数に、"null" を指定可能とする。
	 * @og.rev 4.2.1.0 (2008/04/11) null チェック方法変更
	 * @og.rev 4.2.1.1 (2008/04/30) 引数にA,B,{&#064;XXXX}等が指定された場合に{&#064;XXXX}の変換が行われないバグを修正
	 * @og.rev 5.1.8.0 (2010/07/01) noWritable 同時セットチェックを、メソッドではなく、本体で行う。
	 *
	 * @param	columnName	書込可能カラム名 (CSV形式) (例："OYA,KO,HJO,SU,DYSET,DYUPD")
	 * @see		#setNoWritable( String )
	 */
	public void setColumnWritable( final String columnName ) {
		columnWritable = nval( getRequestParameter(columnName),null );
		if( "null".equalsIgnoreCase( columnWritable ) ) { columnWritable = ""; }
	}

	/**
	 * 【TAG】書き込み不可カラム名を、CSV形式で与えます。
	 *
	 * @og.tag
	 * これは、書き込み可能カラム名の指定(columnWritable)と同時にセットする
	 * ことは出来ません。
	 * "*" を指定すると、すべてのカラムを(noWritable)指定したことになります。
	 *
	 * @og.rev 3.3.1.1 (2003/07/03) {&#064;XXXX} 変数が使用できるように修正。
	 * @og.rev 4.1.2.0 (2008/03/12) 引数に、"null" を指定可能とする。
	 * @og.rev 4.2.1.0 (2008/04/11) null チェック方法変更
	 * @og.rev 4.2.1.1 (2008/04/30) 引数にA,B,{&#064;XXXX}等が指定された場合に{&#064;XXXX}の変換が行われないバグを修正
	 * @og.rev 5.1.8.0 (2010/07/01) noWritable 同時セットチェックを、メソッドではなく、本体で行う。
	 *
	 * @param	columnName	書込不可カラム名 (CSV形式) (例："OYA,KO,HJO,SU,DYSET,DYUPD")
	 * @see		#setColumnWritable( String )
	 */
	public void setNoWritable( final String columnName ) {
		noWritable = nval( getRequestParameter(columnName),null );
		if( "null".equalsIgnoreCase( noWritable ) ) { noWritable = ""; }
	}

	/**
	 * 【TAG】表示可能カラム名を、CSV形式で与えます。
	 *
	 * @og.tag
	 * これは、表示不可カラム名の指定(noDisplay)と同時にセットする
	 * ことは出来ません。
	 * "*" を指定すると、すべてのカラムを(columnDisplay)指定したことになります。
	 *
	 * @og.rev 3.3.1.1 (2003/07/03) {&#064;XXXX} 変数が使用できるように修正。
	 * @og.rev 5.1.8.0 (2010/07/01) noWritable 同時セットチェックを、メソッドではなく、本体で行う。
	 *
	 * @param	columnName	表示可能カラム名 (CSV形式) (例："OYA,KO,HJO,SU,DYSET,DYUPD")
	 * @see		#setNoDisplay( String )
	 */
	public void setColumnDisplay( final String columnName ) {
		columnDisplay = nval( getRequestParameter(columnName),null );
		if( "null".equalsIgnoreCase( columnDisplay ) ) { columnDisplay = ""; }
	}

	/**
	 * 【TAG】表示不可カラム名を、CSV形式で与えます。
	 *
	 * @og.tag
	 * これは、表示可能カラム名の指定(columnDisplay)と同時にセットする
	 * ことは出来ません。
	 * "*" を指定すると、すべてのカラムを(noDisplay)指定したことになります。
	 *
	 * @og.rev 3.3.1.1 (2003/07/03) {&#064;XXXX} 変数が使用できるように修正。
	 * @og.rev 5.1.8.0 (2010/07/01) noWritable 同時セットチェックを、メソッドではなく、本体で行う。
	 *
	 * @param	columnName	表示不可カラム (CSV形式) (例："OYA,KO,HJO,SU,DYSET,DYUPD")
	 * @see		#setColumnDisplay( String )
	 */
	public void setNoDisplay( final String columnName ) {
		noDisplay = nval( getRequestParameter(columnName),null );
		if( "null".equalsIgnoreCase( noDisplay ) ) { noDisplay = ""; }
	}

	/**
	 * 【TAG】表示可能カラム名を、カラム順に、CSV形式で与えます。
	 *
	 * @og.tag
	 * これは、ViewForm_HTMLSeqClmTable 用に特別に用意された属性です。
	 * 機能的には、columnDisplay 属性で、カラムの表示順まで指定できる機能を
	 * 用意すべきですが、過去のタグとの互換性の関係で、別に用意しています。
	 * (この属性を使用すると、通常の表示より、処理速度が低下します。)
	 *  例："OYA,KO,HJO,SU,DYSET,DYUPD" この順番で表示され、指定以外のカラムは表示されません。
	 *
	 * @og.rev 5.1.6.0 (2010/05/01) clmSeqDisplay 属性を追加します。
	 *
	 * @param	columnName	カラム名
	 */
	public void setClmSeqDisplay( final String columnName ) {
		clmSeqDisplay = nval( getRequestParameter(columnName),null );
	}

	/**
	 * 【TAG】同一表示データをまとめる(表示しない)カラム名を、CSV形式で与えます。
	 *
	 * @og.tag
	 * これは、カラムのグループ化指定を行います。
	 * 同じカラム番号で一つ上の行番号と同じ内容の場合に、特殊な表示方法で表示します。
	 * 対応は、表示(Renderer)時のみとします。
	 * 特殊な表示方法の指定は、groupClass 属性で指定します。
	 * (例：groupClass にアルファベット以外の文字・・そのまま表示、アルファベットはクラス属性を付与します。)
	 * カラム単位なので、新しいゼブラによる色分けは行いません。(任意のカラムに適用できる為)
	 * また、ファイル出力などのデータ自身は変更されていませんので、そのままデータが落ちます。
	 *
	 * これは、同一表示データをまとめないカラム名の指定(noGroup)と同時にセットする
	 * ことは出来ません。
	 * "*" を指定すると、すべてのカラムを(columnGroup)指定したことになります。
	 *
	 * @og.rev 3.8.5.0 (2006/03/20) 新規追加
	 * @og.rev 5.1.8.0 (2010/07/01) noWritable 同時セットチェックを、メソッドではなく、本体で行う。
	 *
	 * @param	group	グループカラム (CSV形式) (例："OYA,KO,HJO,SU,DYSET,DYUPD")
	 * @see		#setNoGroup( String )
	 * @see		#setGroupClass( String )
	 */
	public void setColumnGroup( final String group ) {
		columnGroup = nval( getRequestParameter( group ),null );
		if( "null".equalsIgnoreCase( columnGroup ) ) { columnGroup = ""; }
	}

	/**
	 * 【TAG】同一表示データをまとめないカラム名を、CSV形式で与えます。
	 *
	 * @og.tag
	 * これは、カラムのグループ化指定で、同一でもまとめないカラムを指定するほうが簡単な場合に
	 * 使用します。例えば、キー情報以外は、グループ化指定したい場合などに便利です。
	 * 動作は、columnGroup の動きと同じです。(まとめないカラムを指定するだけです。)

	 * これは、同一表示データをまとめるカラム名の指定(columnGroup)と同時にセットする
	 * ことは出来ません。
	 * "*" を指定すると、すべてのカラムを(noGroup)指定したことになります。
	 *
	 * @og.rev 5.1.8.0 (2010/07/01) 新規追加
	 *
	 * @param	group	グループ除外カラム (CSV形式) (例："OYA,KO,HJO,SU,DYSET,DYUPD")
	 * @see		#setColumnGroup( String )
	 * @see		#setGroupClass( String )
	 */
	public void setNoGroup( final String group ) {
		noGroup = nval( getRequestParameter( group ),null );
		if( "null".equalsIgnoreCase( noGroup ) ) { noGroup = ""; }
	}

	/**
	 * 【TAG】同一表示データをまとめる場合の、表示方法を指定します。
	 *
	 * @og.tag
	 * これは、カラムのグループ化指定(columnGroup)を行う場合の、まとめ表示の方法を指定します。
	 * なにも指定しない場合は、まとめカラムは、表示しない(ゼロ文字列)になります。
	 * その場合、先の行と同じ場合も、変更されたカラムが、NULL(ゼロ文字列)であった場合も、
	 * 同じ用に表示されるため、区別が付きません。
	 * そこで、前の行と同じデータの場合に、特殊な処理を行うことで、区別できるようにします。
	 * 指定方法が特殊なので、注意が必要です。
	 *   記号の場合：
	 *       記号(-- , → , ↓ , * など)が指定された場合は、そのままの文字に置き換えられます。
	 *       初期値は、空文字列(ゼロ文字列)です。
	 *   アルファベットの場合：
	 *       アルファベット(a-z,A-Z)の場合は、&lt;span class="アルファベット"&gt;元の文字&lt;/span&gt;
	 * 例えば、バックカラーに色を付ける、文字を薄い灰色にする、などできます。
	 * ただし、データ量が圧倒的に増えるため、大量のデータ等で使用するのは避けたほうが良いと思います。
	 *
	 * @og.rev 3.8.5.0 (2006/03/20) 新規追加
	 *
	 * @param	grpCls まとめ表示の方法
	 */
	public void setGroupClass( final String grpCls ) {
		groupClass = nval( getRequestParameter(grpCls),null );
	}

	/**
	 * 【TAG】カラム名リンクソートを表示するカラム名を、CSV形式で与えます。
	 *
	 * @og.tag
	 * ヘッダーにソート用リンクを作成する useTableSorter 属性 に対して、
	 * カラム個別に作成する場合のカラム名をCSV形式で指定します。
	 * この sortKeys 属性は、useTableSorter 属性 と無関係に、指定した
	 * カラムのみ、リンクを表示します。
	 * "*" を指定すると、すべてのカラムを(tableSorterKeys)指定したことになります。
	 * 例："OYA,KO,HJO,SU,DYSET,DYUPD"
	 *
	 * @og.rev 3.6.0.0 (2004/09/17) 新規作成
	 *
	 * @param	columnName	カラム名
	 */
	public void setTableSorterKeys( final String columnName ) {
		sortKeys = nval( getRequestParameter(columnName),null );
	}

	/**
	 * 【TAG】検索結果メッセージを表示しないかどうか[true/false]を設定します(初期値:false[表示する])。
	 *
	 * @og.tag
	 * 初期値は、表示する(true以外)です。
	 *
	 * @param	flag 検索結果非表示 [true:表示しない/それ以外:表示する]
	 */
	public void setNoMessage( final String flag ) {
		noMessage = nval( getRequestParameter( flag ),noMessage );
	}

	/**
	 * 【TAG】行が書き込み可能かどうか[true/false]を設定します
	 *		(初期値:DBTableModel#DEFAULT_WRITABLE[={@og.value DBTableModel#DEFAULT_WRITABLE}])。
	 *
	 * @og.tag
	 * 書込み許可する場合は, true、許可しない場合は, true 以外を設定します。
	 * ここで、false を設定すると全行書込み禁止になります。true を設定した
	 * 場合、デフォルトで書込み許可が与えられ、先頭カラムが writable であれば、
	 * その値を取り込んで、行ごとの書込み許可/不許可が決定されます。
	 * writable,checked,useCheckControl については、writable が最優先で
	 * 次に、useCheckControl="2"(または、checked)、次に、checked を判定します。
	 * checked="true"の場合は、useCheckControl が、"1"(true)以上の場合、
	 * "2"(checked)に設定されます。
	 * (初期値:DBTableModel#DEFAULT_WRITABLE[={@og.value DBTableModel#DEFAULT_WRITABLE}])。
	 *
	 * @param	flag 書き込み許可 [true:許可/false:不可]
	 * @see		org.opengion.hayabusa.db.DBTableModel#DEFAULT_WRITABLE
	 */
	public void setWritable( final String flag ) {
		writable = getRequestParameter( flag );
	}

	/**
	 * テーブルデータに書込み許可／チェック状態を設定します。
	 *
	 * @og.rev 3.6.0.0 (2004/09/17) writable が false の場合全件チェック用チェックボックスを出さない。
	 * @og.rev 3.7.0.1 (2005/01/31) 全件チェックコントロールのロジック変更(useCheckControl)
	 * @og.rev 3.8.8.5 (2007/03/09) 全件チェック(useCheckControl)と、個別チェック(checked)の整合性
	 * @og.rev 5.5.5.6 (2012/08/31) ユーザーと画面の関係が、リードオンリーの場合は、writable=falseを内部設定する。
	 * @og.rev 5.5.6.1 (2012/09/03) writable自動セットは指定コマンドのみ
	 *
	 * @param	table	DBTableModelオブジェクト
	 */
	private void setTableRowWritable( final DBTableModel table ) {

		// 5.5.5.6 (2012/08/31) リードオンリーの場合は、writable=falseを内部設定する。
		// このタグで、writable が指定された場合は、そちらを優先する。(つまり、null の場合は、内部設定する。)
		// 5.5.6.1 (2012/09/03) この処理を行うのは指定コマンドのみ
		if( ( writable == null || writable.isEmpty() ) && check( command, COMMAND_LIST ) ) {
			final GUIInfo guiInfo = (GUIInfo)getSessionAttribute( HybsSystem.GUIINFO_KEY );
			if( ! guiInfo.isWrite() ){
				writable = "FALSE" ;
			}
		}

		if( writable != null && writable.length() > 0 ) {
			final boolean flag = "TRUE".equalsIgnoreCase( writable );
			table.setDefaultRowWritable( flag );
			if( ! flag ) { useCheckControl = 0; }			// 3.7.0.1 (2005/01/31)
		}

		// 全チェックONの場合は、個別チェックもONにする。
		if( useCheckControl == 2 ) { checked = "true"; }	// 3.8.8.5 (2007/03/09)

		if( checked != null && checked.length() > 0 ) {
			final boolean flag = "TRUE".equalsIgnoreCase( checked );
			table.setDefaultRowChecked( flag );
			// 個別チェックONの場合は、全チェックもONにする。
			if( flag && useCheckControl >= 1 ) { useCheckControl = 2; }			// 3.8.8.5 (2007/03/09)
		}
	}

	/**
	 * 【TAG】行のチェックボックスの初期値を設定するかどうか[true/false]を指定します(初期値:null)。
	 *
	 * @og.tag
	 * 書き込み可能な行(rowWritable == true)のチェックボックスに対して
	 * 初期値を 選択済みにするか、非選択済みにするかを指定します。
	 * writable,checked,useCheckControl については、writable が最優先で
	 * 次に、useCheckControl="2"(または、checked)、次に、checked を判定します。
	 * checked="true"の場合は、useCheckControl が、"1"(true)以上の場合、
	 * "2"(checked)に設定されます。
	 *
	 * @param	flag 選択済設定 [true:選択済み/それ以外:非選択状態]
	 */
	public void setChecked( final String flag ) {
		checked = getRequestParameter( flag );
	}

	/**
	 * 【TAG】NEXT等で、選択行以外をスキップするのかどうか[true/false]を指定します
	 *		(初期値:ViewForm#DEFAULT_SKIP[={@og.value ViewForm#DEFAULT_SKIP}])。
	 *
	 * @og.tag
	 * 明細情報の登録/編集時に、必要な個所のみを順番に処理する場合に使用します。
	 * (初期値:ViewForm#DEFAULT_SKIP[={@og.value ViewForm#DEFAULT_SKIP}])。
	 *
	 * @param	flag スキップ  [true:する/それ以外:しない]
	 * @see		org.opengion.hayabusa.html.ViewForm#DEFAULT_SKIP
	 */
	public void setSkip( final String flag ) {
		skip = nval( getRequestParameter( flag ),skip );
	}

	/**
	 * 【TAG】表示データを作成する場合のフォーマットの行数(rowspan)をセットします(初期値:2)。
	 *
	 * @og.tag 表示データを作成する場合のフォーマットの行数をセットします。
	 *
	 * @og.rev 3.3.1.1 (2003/07/03) {&#064;XXXX} 変数が使用できるように修正。
	 *
	 * @param	rowspan フォーマットの行数
	 */
	public void setRowspan( final String rowspan ) {
		this.rowspan = getRequestParameter(rowspan);
	}

	/**
	 * 【TAG】表示時の選択用オブジェクトのタイプ[checkbox/radio/hidden]を指定します(初期値:checkbox)。
	 *
	 * @og.tag
	 * 複数選択可能時は "checkbox" を指定します。
	 * 一つだけ選ばせる場合は, "radio" を指定します。
	 * 隠しフィールドで全件を選ぶ場合は、"hidden" を指定します。
	 * 初期値は、"checkbox" です。
	 *
	 * @og.rev 3.3.1.1 (2003/07/03) {&#064;XXXX} 変数が使用できるように修正。
	 *
	 * @param	type 選択用タイプ [checkbox:複数選択/radio:一つだけ/hidden:全件選択]
	 */
	public void setSelectedType( final String type ) {
		selectedType = nval( getRequestParameter(type),selectedType );
	}

	/**
	 * 【TAG】表示時の行番号の表示方法[sequence/none/delete/skip=カラム名/view=カラム名]を指定します(初期値:sequence)。
	 *
	 * @og.tag
	 * ・sequenceは、１から始まる連番です。
	 * ・none を指定すると、番号は表示されません。
	 * ・delete を指定すると、行番号表示そのもののタグを取り除きます。(3.5.5.0 追加)
	 * ・skip=カラム名を指定すると、そのカラムの値が NULL の場合は、番号をスキップします。
	 * ・view=カラム名を指定すると、そのカラムの値が、使用されます。(ラベルは付きません)
	 * skip=XX と、view=XX は、=の前後にスペースを入れないで下さい。
	 * 初期値は、"sequence" です。
	 *
	 * @og.rev 3.5.1.0 (2003/10/03) 新規作成
	 *
	 * @param	type 行番号表示タイプ [sequence:連番/none:非表示/delete:タグ削除/skip=カラム名/view=カラム名]
	 */
	public void setNumberType( final String type ) {
		numberType = nval( getRequestParameter(type),numberType );
	}

	/**
	 * 【TAG】テーブル等のチェックボックスに属性を付加します。
	 *
	 * @og.tag
	 * JavaScript などの HTML基本タグ以外の属性を、そのまま
	 * チェックボックス/ラジオボタン等に使用します。
	 * そのときに、(')は、(")に置き換えます。
	 *
	 * @param	attributes チェックボックス追加属性
	 */
	public void setOptionTypeAttributes( final String attributes ) {
		final String optAttri = getRequestParameter( attributes );
		if( optAttri != null && optAttri.length() > 0 ) {
			optTypeAttri = optAttri.replace( '\'', '\"' );
		}
	}

	/**
	 * 【TAG】チェックボックスの全チェックを選択するJavaScript の機能を使用するかどうかを指定します
	 *		(初期値:VIEW_USE_CHECK_CONTROL[={@og.value SystemData#VIEW_USE_CHECK_CONTROL}])。
	 *
	 * @og.tag
	 * "true" で、コントロール用のチェックボックスが現れて、この機能を使用することができるようになります。
	 * false は、従来どおりです。
	 * 使用するにあたり、jsp/commpn/default.js にJavaScriptを設定しておきます。(設定済み)
	 * false は、0でも代用可能です。true は、1でも代用可能です。
	 * checked は、2でも代用可能です。checked は、初期値を、チェック済みにします。
	 * 注意：0,1,2,true,false,checked 以外の文字をセットすると、エラーが発生します。
	 * writable,checked,useCheckControl については、writable が最優先で
	 * 次に、useCheckControl="2"(または、checked)、次に、checked を判定します。
	 * checked="true"の場合は、useCheckControl が、"1"(true)以上の場合、
	 * "2"(checked)に設定されます。
	 * (初期値:ユーザー定数のVIEW_USE_CHECK_CONTROL[={@og.value SystemData#VIEW_USE_CHECK_CONTROL}])。
	 *
	 * @og.rev 3.5.4.3 (2004/01/05) 新規追加
	 * @og.rev 3.7.0.1 (2005/01/31) 全件チェックコントロールのロジック変更(useCheckControl)
	 *
	 * @param	flag 全チェック選択可否 [true,1:使用する/false,0:使用しない/checked,2:初期値チェック済み]
	 * @see		org.opengion.hayabusa.common.SystemData#VIEW_USE_CHECK_CONTROL
	 */
	public void setUseCheckControl( final String flag ) {
		useCheckControl = intCase( getRequestParameter(flag) );

		if( useCheckControl < 0 ) {
			final String errMsg = "useCheckControl は、false(0)/true(1)/2 のいずれかしか登録できません。["
						 + flag + "]";
			throw new HybsSystemException( errMsg );
		}
	}

	/**
	 * 【TAG】ヘッダーにソート用リンクを作成するかどうか[true/false]を指定します
	 *		(初期値:VIEW_USE_TABLE_SORTER[={@og.value SystemData#VIEW_USE_TABLE_SORTER}])。
	 *
	 * @og.tag
	 * "true" で、ヘッダーにソート用リンクを作成します。
	 * false は、従来どおりです。
	 * 検索済みのDBTableModelを、メモリ上でソートし、再表示させます。
	 *		(初期値:ユーザー定数のVIEW_USE_TABLE_SORTER[={@og.value SystemData#VIEW_USE_TABLE_SORTER}])。
	 *
	 * @og.rev 3.5.4.7 (2004/02/06) 新規追加
	 *
	 * @param	flag ヘッダーソートリンク [true:作成する/false:作成しない]
	 * @see		org.opengion.hayabusa.common.SystemData#VIEW_USE_TABLE_SORTER
	 */
	public void setUseTableSorter( final String flag ) {
		useTableSorter = nval( getRequestParameter(flag),useTableSorter );
	}

	/**
	 * 表示データをプレビュー可能かどうかを返します。
	 *
	 * 不可能な状態とは,すでに一番先頭から表示されている場合です。
	 *
	 * @return	プレビュー可能(true)／不可能(false)
	 */
	private boolean isPrev() {
		return startNo > 0;				// 6.2.6.0 (2015/06/19) refactoring
	}

	/**
	 * 表示データをネクスト可能かどうかを返します。
	 *
	 * 不可能な状態とは,すでに一番後ろが表示されている場合です。
	 *
	 * @return	ネクスト可能(true)／不可能(false)
	 */
	private boolean	 isNext() {
		return (startNo + pageSize) < table.getRowCount();				// 6.2.6.0 (2015/06/19) refactoring
	}

	/**
	 * コマンドを実行します。
	 *
	 * コマンドは,HTMLから(get/post)指定されますので,setCommand()メソッドで
	 * 登録します。
	 * コマンドを登録すると同時に,実行も行ないます。
	 *
	 * @og.rev 3.8.0.8 (2005/10/03) その他のコマンドを CMD_XXX にします。
	 *
	 * @param	command コマンド (public static final 宣言されている文字列)
	 * @see		<a href="../../../../constant-values.html#org.opengion.hayabusa.taglib.ViewFormTag.CMD_PREV">コマンド定数</a>
	 */
	private void commandExec( final String command ) {
		if( 	 CMD_PREV.equals(	command ) ) { prev() ;	}
		else if( CMD_NEXT.equals(	command ) ) { next() ;	}
		else if( CMD_FIRST.equals(	command ) ) { first() ; }
		else if( CMD_LAST.equals(	command ) ) { last() ;	}
	//	else if( CMD_NEW.equals(	command ) ) {} 		// 何もしない
	//	else if( CMD_REVIEW.equals( command ) ) {} 		// 何もしない
	//	else if( CMD_RESET.equals(	command ) ) {} 		// 何もしない
		else if( CMD_VIEW.equals(	command ) ) { view() ;	}
		// 3.8.0.8 (2005/10/03) その他のコマンドを CMD_XXX にします。
		else if( CMD_XXX.equals(	command ) ) { view() ;	}
		this.command = null;
	}

	/**
	 * HTMLデータの表示をプレビュー(戻し)します。
	 *
	 * @og.rev 3.0.1.3 (2003/03/11) startNo を隠しフィールドに持つ処理を、追加
	 *
	 */
	private void prev() {
		if( ! isPrev() ) { return ; }			// prev 出来ない。
		tempStartNo = tempStartNo - pageSize;
		if( tempStartNo < 0 ) { tempStartNo = 0; }
		if( skip ) {
			for( int i=startNo-1; i>=0; i-- ) {
				if( form.isMarked( i ) ) {
					startNo = i;
					break;
				}
			}
			return ;
		}

		startNo = startNo - pageSize;
		if( startNo < 0 ) {
			startNo = 0;
		}
	}

	/**
	 * HTMLデータの表示をネクスト(先送り)します。
	 *
	 * @og.rev 3.0.1.3 (2003/03/11) startNo を隠しフィールドに持つ処理を、追加
	 *
	 */
	private void next() {
		if( ! isNext() ) { return ; }			// next 出来ない。
		tempStartNo = tempStartNo + pageSize;
		if( skip ) {
			final int end  = table.getRowCount();
			for( int i=startNo+1; i<end; i++ ) {
				if( form.isMarked( i ) ) {
					startNo = i;
					break;
				}
			}
			return ;
		}

		startNo = startNo + pageSize;
	}

	/**
	 * HTMLデータの表示をファースト(最初)します。
	 *
	 * @og.rev 3.0.1.3 (2003/03/11) startNo を隠しフィールドに持つ処理を、追加
	 *
	 */
	private void first() {
		tempStartNo = 0;
		if( tempStartNo < 0 ) { tempStartNo = 0; }
		if( skip ) {
			final int end  = table.getRowCount();
			for( int i=startNo; i<end; i++ ) {
				if( form.isMarked( i ) ) {
					startNo = i;
					break;
				}
			}
			return ;
		}

		startNo = 0;
	}

	/**
	 * HTMLデータの表示をラスト(最後)します。
	 *
	 * @og.rev 3.0.1.3 (2003/03/11) startNo を隠しフィールドに持つ処理を、追加
	 *
	 */
	private void last() {
		tempStartNo = table.getRowCount() - pageSize;
		if( tempStartNo < 0 ) { tempStartNo = 0; }
		if( skip ) {
			for( int i=startNo; i>=0; i-- ) {
				if( form.isMarked( i ) ) {
					startNo = i;
					break;
				}
			}
			return ;
		}

		startNo = table.getRowCount() - pageSize;
		if( startNo < 0 ) { startNo = 0; }
	}

	/**
	 * HTMLデータの表示をします。
	 *
	 */
	private void view() {
		if( skip ) {
			final int end  = table.getRowCount();
			for( int i=startNo; i<end; i++ ) {
				if( form.isMarked( i ) ) {
					startNo = i;
					break;
				}
			}
			return ;
		}
	}

	/**
	 * startNo,pageSize,pagePlus の初期値を設定する。
	 *
	 * @og.rev 3.0.1.3 (2003/03/11) startNo を隠しフィールドに持つ処理を、追加
	 * @og.rev 3.5.4.1 (2003/12/01) startNo をキャッシュから取り出すように変更。
	 * @og.rev 3.5.4.7 (2004/02/06) getStartNoKeyメソッドの廃止(getRequestCacheDataを利用)
	 * @og.rev 3.7.0.2 (2005/02/14) NEXT 時に何も選択しないで追加した場合は、１件目から表示する。
	 * @og.rev 3.8.0.8 (2005/10/03) その他のコマンドを CMD_XXX にします。
	 *
	 */
	private void setDefaultParameter() {
		// 3.7.0.2 (2005/02/14)
		// 3.8.0.8 (2005/10/03) その他のコマンドを CMD_XXX にします。
		if( CMD_XXX.equals( command ) ) {
			final String[] selected = getRequestValues( HybsSystem.ROW_SEL_KEY ) ;
			if( selected == null || selected.length == 0 ) {
				startNo = 0;
				tempStartNo = startNo;
			}
		}

		if( startNo<0 ) {
			startNo = nval( getRequestCacheData( START_NO_KEY ),form.getStartNo() ) ;
			tempStartNo = startNo;
		}

		if( pageSize<0 ) {
			pageSize = nval( getRequestValue( PAGE_SIZE_KEY ),form.getPageSize() ) ;
		}

		if( pagePlus<0 ) {
			pagePlus = nval( getRequestValue( PAGE_PLUS_KEY ),0 ) ;
		}

	}

	/**
	 * 内部タグの TheadTag / TbodyTag / TfootTag より、個々のフォーマットを定義する
	 * 為のTableFormatter オブジェクトを受け取る。
	 *
	 * @og.rev 3.5.4.0 (2003/11/25) 新規登録(setAttributes メソッドより移行)
	 *
	 * @param	form TableFormatterオブジェクト
	 */
	protected void addFormatter( final TableFormatter form ) {
		if( values == null ) { values = new ArrayList<>(); }
		values.add( form );
	}

	/**
	 * pageSize をhidden属性で出力するタグを作成します。
	 *
	 * @og.rev 3.5.4.0 (2003/11/25) command="RESET"時に、戻った画面のpageSize がクリアされてしまう対応。
	 * @og.rev 3.5.5.5 (2004/04/23) hidden の出力に、XHTMLTag.hidden を使用します。
	 *
	 * @param	pageSize	ページサイズ
	 *
	 * @return	hidden属性で出力するタグ
	 * @og.rtnNotNull
	 */
	private String makePageSize( final int pageSize ) {
		final String name   = PAGE_SIZE_KEY;
		final String value  = String.valueOf( pageSize );

		return XHTMLTag.hidden( name,value );	// 3.5.5.5 (2004/04/23)
	}

	/**
	 * consistencyKey をhidden属性で出力するタグを作成します。
	 * 出力する条件は、useConsistency="true" (初期値)、で
	 * scope が、 request 以外の場合です。
	 * tableId が設定された場合は、consistencyKey に、tableId を加えた
	 * キーにします。
	 *
	 * @og.rev 3.5.5.5 (2004/04/23) DBTableModel の整合性パラメータを hidden で出力します。
	 * @og.rev 3.5.5.5 (2004/04/23) hidden の出力に、XHTMLTag.hidden を使用します。
	 * @og.rev 3.5.5.8 (2004/05/20) makeConsistencyKey メソッドに処理ロジックを移します。
	 *
	 */
	private void makeConsistencyKey() {
		if( useConsistency && ! "request".equals( getScope() ) ) {
			final String name   = HybsSystem.CONSISTENCY_KEY;
			final String value  = table.getConsistencyKey() ;

			jspPrint( XHTMLTag.hidden( name,value ) );
		}
	}

	/**
	 * "false","true","checked" の文字列を、それぞれ、0,1,2 に対応した数字に変換します。
	 * それ以外の文字列は、-1 にします。
	 *
	 * @og.rev 3.7.0.1 (2005/01/31) 新規作成
	 *
	 * @param    flag 変換対象 [true→1/false→0/checked→2]
	 *
	 * @return   設定値(0,1,2の数字、不正な場合は、-1 を返す。
	 */
	private static int intCase( final String flag ) {
		int rtn = -1;
		if( "FALSE".equalsIgnoreCase( flag ) || "0".equalsIgnoreCase( flag ) ) {
			rtn = 0;
		}
		else if( "TRUE".equalsIgnoreCase( flag ) || "1".equalsIgnoreCase( flag ) ) {
			rtn = 1;
		}
		else if( "CHECKED".equalsIgnoreCase( flag ) || "2".equalsIgnoreCase( flag ) ) {
			rtn = 2;
		}
		return rtn ;
	}

	/**
	 * ViewForm のサブクラスに渡すパラメータマップ。
	 *
 	 * @og.rev 3.5.4.8 (2004/02/23) ViewParamTag のパラメータを追加します。
	 *
	 * @param	map パラメータマップ
	 */
	protected void setParam( final Map<String,String> map ) {
		viewParam = map ;
	}

	/**
	 * ViewForm のサブクラスに渡すパラメータ(文字列配列のArrayList)を設定します。
	 *
 	 * @og.rev 5.5.8.3 (2012/11/17) 新規作成（スタックガント対応）
 	 * @og.rev 5.5.9.0 (2012/12/03) objectからArrayListに変更
	 *
	 * @param	list 文字列配列のArrayList
	 */
	protected void setViewArrayList( final List<String[]> list ) {
		viewArrayList = list;
	}

	/**
	 * 【TAG】パラメーターファイルの使用をするかどうか[true/false]を指定します(初期値:false)。
	 *
	 * @og.tag
	 * "true" で、パラメータファイルを読み込むため、BODY部を処理します。
	 * false は、パラメータファイうを使用しません。
	 * パラメータファイルは、各ViewFormのサブクラス用に用意します。
	 * 初期値は、false です。(false)
	 *
	 * @og.rev 3.5.4.8 (2004/02/23) 新規追加
	 *
	 * @param	flag パラメーター使用可否 [true:使用する/false:使用しない]
	 */
	public void setUseParam( final String flag ) {
		useParam = nval( getRequestParameter(flag),useParam );
	}

	/**
	 * 【TAG】Consistency キー による整合性チェックを行うかどうか[true/false]を指定します(初期値:true)。
	 *
	 * @og.tag
	 * 検索結果を DBTableModel にセットする時に、整合性キーの Consistency キーを
	 * 作成します。これを、Viewタグでhidden出力しておき、Entryタグでデータ書き換え時に
	 * 整合性チェックを行います。これは、IEの戻るボタンで戻った場合に、画面の
	 * キーと検索結果の DBTableModel の内容が一致しない場合のエラーチェックに
	 * なります。
	 * この属性は、何らかのケースで、このエラーチェックを行いたくない場合に、
	 * false に設定することで、整合性チェックを行いません。
	 * 初期値は、true(整合性チェックを行う)です。
	 *
	 * @og.rev 3.5.5.7 (2004/05/10) 新規登録
	 *
	 * @param	ck	整合性チェック [true:行う/false:行わない]
	 */
	public void setUseConsistency( final String ck ) {
		useConsistency = nval( getRequestParameter( ck ),useConsistency );
	}

	/**
	 * 【TAG】テーブルのバックグラウンドカラーの入れ替えのサイクルをセットします(初期値は、1:ゼブラ)。
	 *
	 * @og.tag
	 * 0(ゼブラなし)、-1(ワーニング)、-2以下(エラー)、1(ゼブラ)、2以上(行数まとめ)
	 * 初期値は、1(ゼブラ)です。
	 *
	 * @og.rev 3.5.6.2 (2004/07/05) 新規追加
	 *
	 * @param  sycle  ゼブラ指定 [0:ゼブラなし/-1:警告色/-2:エラー色/1:ゼブラ/2以上:行数まとめ]
	 */
	public void setBgColorCycle( final String sycle ) {
		bgColorCycle = nval( getRequestParameter( sycle ),bgColorCycle );
	}

	/**
	 * 【TAG】テーブルのバックグラウンドカラーのクラスに指定するカラム名を指定します。
	 *
	 * @og.tag
	 * 通常のゼブラ模様は、tr 属性のクラス指定として、row_0、row_1 が指定されます。
	 * その代わりに、ここで指定されたカラムの値がクラス属性として設定されます。
	 * (指定するのはカラム名です。行単位にクラス属性の値を変えることが可能です。)
	 * 選択行(row_sel)は、優先して使用されます。
	 * 出力されるクラス名は、"row_" + 属性値 になります。
	 *
	 * @og.rev 5.1.8.0 (2010/07/01) 新規追加
	 *
	 * @param  clsClm ゼブラ模様指定カラム
	 */
	public void setBgColorClsClm( final String clsClm ) {
		bgColorClsClm = nval( getRequestParameter( clsClm ),bgColorClsClm );
	}

	/**
	 * 【TAG】スクロールバー用のDIV要素を出力するかどうか[true/false]を指定します
	 *		(初期値:VIEW_USE_SCROLLBAR[={@og.value SystemData#VIEW_USE_SCROLLBAR}])。
	 *
	 * @og.tag
	 * スクロールバー対応する為、テーブルの先頭に、DIV要素を出力します。
	 * (初期値:ユーザー定数のVIEW_USE_SCROLLBAR[={@og.value SystemData#VIEW_USE_SCROLLBAR}])。
	 *
	 * @og.rev 3.5.6.3 (2004/07/12) 新規追加
	 *
	 * @param  useBar  スクロールバー用使用可否 [true:する/false:しない]
	 * @see		org.opengion.hayabusa.common.SystemData#VIEW_USE_SCROLLBAR
	 */
	public void setUseScrollBar( final String useBar ) {
		useScrollBar = nval( getRequestParameter( useBar ),useScrollBar );
	}

	/**
	 * 【TAG】ヘッダー固定のスクロールバー使用時の最大行数を指定します
	 *			(初期値:VIEW_SCROLLBAR_MAX_ROW_SIZE[={@og.value SystemData#VIEW_SCROLLBAR_MAX_ROW_SIZE}])。
	 *
	 * @og.tag
	 * ヘッダー固定のスクロールバーは、JavaScriptで処理しているため、
	 * 大量のデータを読み取ると、非常に時間がかかったり、場合によっては
	 * ブラウザがハングアップします。
	 * そこで、ある表示件数以上の場合は、強制的にスクロールバーを
	 * オフにします。
	 * 指定以上のデータ件数を、表示する場合に、オフにします。
	 * (初期値:VIEW_SCROLLBAR_MAX_ROW_SIZE[={@og.value SystemData#VIEW_SCROLLBAR_MAX_ROW_SIZE}])
	 *
	 * @og.rev 6.2.4.0 (2015/05/15) 新規作成
	 *
	 * @param  size  スクロールバー使用時の最大行数
	 * @see		org.opengion.hayabusa.common.SystemData#VIEW_SCROLLBAR_MAX_ROW_SIZE
	 */
	public void setMaxScrollBarSize( final String size ) {
		maxScrollBarSize = nval( getRequestParameter( size ),maxScrollBarSize );
	}

	/**
	 * 【TAG】viewLink や viewMarker の情報取り込みを制御する(初期値は、NEW,RENEW,REVIEW,RESET)。
	 *
	 * @og.tag
	 * viewLink や viewMarker の情報取り込みは、command が、NEW,RENEW,REVIEW,RESET
	 * の場合のみ行われています。
	 * この属性で、"true" を指定すると、コマンドに無関係に取り込みます。
	 * "false" の場合は、form にキャッシュしているリンク、マーカーを削除します。
	 * 取り込み時に、viewLink や viewMarker オブジェクトが存在しない場合は、
	 * 何もしません。これは、viewMarker オブジェクト自身がすでに設定済みの場合(VIEWなど)
	 * NEXTやPREVでは、viewMarkerオブジェクトは、リクエスト変数に設定されない為です。
	 * この場合は、登録しませんが、既存の viewMarkerオブジェクトを持っているはずなので、
	 * なにも設定しなくても viewMarker は適用されます。
	 * 初期値は、NEW,RENEW,REVIEW,RESET です。
	 *
	 * @og.rev 3.6.0.4 (2004/10/14) 新規追加
	 *
	 * @param  load  情報取り込みするキー
	 */
	public void setLoadMarker( final String load ) {
		loadMarker = nval( getRequestParameter( load ),loadMarker );
	}

	/**
	 * 【TAG】SEL_ROW機能[true:有効/false:無効]を指定します
	 *		(初期値:VIEW_USE_SELECTED_ROW[={@og.value SystemData#VIEW_USE_SELECTED_ROW}])。
	 *
	 * @og.tag
	 * SEL_ROW機能は、戻るリンクから戻ったときや、登録したときに、選択していた
	 * 行番号や、先にクリックされた行番号を返す機能で、SEL_ROW リクエストに設定された
	 * 行がマーカーされます。この機能の有効(true)/無効(false)を指定します。
	 * この属性で、"true" を指定すると、有効になります。
	 * "false" の場合は、無効です。
	 * (初期値:ユーザー定数のVIEW_USE_SELECTED_ROW[={@og.value SystemData#VIEW_USE_SELECTED_ROW}])。
	 *
	 * @og.rev 3.7.1.1 (2005/05/23) 新規追加
	 *
	 * @param  selRow  SEL_ROW機能有無 [true:有効/false:無効]
	 * @see		org.opengion.hayabusa.common.SystemData#VIEW_USE_SELECTED_ROW
	 */
	public void setUseSelectedRow( final String selRow ) {
		useSelectedRow = nval( getRequestParameter( selRow ),useSelectedRow );
	}

	/**
	 * 【TAG】SEL_ROW使用時の選択行色変更[true:有効/false:無効]を指定します
	 *		(初期値:VIEW_USE_SELROW_COLOR[={@og.value SystemData#VIEW_USE_SELROW_COLOR}])。
	 *
	 * @og.tag
	 * SEL_ROW機能は、現在選択されている行まで、スクロールする機能ですが、そのとき、
	 * 選択行にマーカーするかどうかを、指定します。
	 * この属性で、"true" を指定すると、有効になります。
	 * "false" の場合は、無効です。
	 * (初期値:ユーザー定数のVIEW_USE_SELROW_COLOR[={@og.value SystemData#VIEW_USE_SELROW_COLOR}])。
	 *
	 * @og.rev 3.7.1.1 (2005/05/31) 新規追加
	 *
	 * @param  selRowColor  SEL_ROW選択行色変更 [true:有効/false:無効]
	 * @see		org.opengion.hayabusa.common.SystemData#VIEW_USE_SELROW_COLOR
	 */
	public void setUseSelRowColor( final String selRowColor ) {
		useSelRowColor = nval( getRequestParameter( selRowColor ),useSelRowColor );
	}

	/**
	 * 【TAG】 処理時間やページサイズなどの情報出力[true:有効/false:無効]を指定します(初期値:true)。
	 *
	 * @og.tag
	 * ViewForm で、CustomData などの 非HTML表示ビューを使用する場合、
	 * 最後に、処理時間とページサイズ(NEXT時に使用)などの情報を、hidden で
	 * 出力していますが、これを制御するフラグをセットします。
	 * true(有効)にすると、これらのHTMLが出力されます。false にすると、出力されません。
	 * 初期値は、true(有効) です。
	 *
	 * @og.rev 3.7.1.1 (2005/05/23) 新規追加
	 *
	 * @param  useTag  情報出力の有効/無効 [true:有効/false:無効]
	 */
	public void setUseAfterHtmlTag( final String useTag ) {
		useAfterHtmlTag = nval( getRequestParameter( useTag ),useAfterHtmlTag );
	}

	/**
	 * 【TAG】設定値に "_" が含まれている場合にレンデラーを使用するカラムをCSV形式で指定します。
	 *
	 * @og.tag
	 * これは、従来の カラム定義の WRITABLE エディターと同等の働きを行うように
	 * カラム属性を指定します。
	 * WRITABLE エディターは、設定値にアンダーバー "_" が含まれている場合に、
	 * その値を書込み禁止にする機能です。これは、エディター自身が値を判断して
	 * 書き込み許可か禁止かを判断しています。
	 * この動きを汎用的にするため、指定のカラムをCSV形式(CSV)で指定
	 * することにより、レンデラーとエディターを設定値によって動的に切り替える
	 * 機能を実現します。
	 * その場合、表示/編集ともに、先頭のアンダーバーは削除されます。
	 * また、全カラムについて、有効にする場合は、writableControl="*" を設定します。
	 *
	 * @og.rev 3.8.0.9 (2005/10/17) 新規追加
	 *
	 * @param  wrtCtrl 書込制御カラム (CSV形式)
	 */
	public void setWritableControl( final String wrtCtrl ) {
		writableControl = nval( getRequestParameter( wrtCtrl ),writableControl );
	}

	/**
	 * 【TAG】HILIGHT_ROW機能[true:有効/false:無効]を指定します
	 *		(初期値:VIEW_USE_HILIGHT_ROW[={@og.value SystemData#VIEW_USE_HILIGHT_ROW}])。
	 *
	 * @og.tag
	 * HILIGHT_ROW機能は、チェックボックスやラジオボタンで行選択した際に、どの行が選択されたか
	 * 判るように行のハイライト表示を行う機能です。
	 * さらに、行の任意の箇所をクリックしても、チェックが入り、ハイライト表示されます。
	 * この属性で、"true" を指定すると、有効になります。
	 * "false" の場合は、無効です。
	 * (初期値:ユーザー定数のVIEW_USE_HILIGHT_ROW[={@og.value SystemData#VIEW_USE_HILIGHT_ROW}])。
	 *
	 * @og.rev 3.8.6.0 (2006/09/29) HILIGHT_ROW機能の有効/無効を指定します。
	 *
	 * @param  hilightRow  HILIGHT_ROW有無 [true:有効/false:無効]
	 * @see		org.opengion.hayabusa.common.SystemData#VIEW_USE_HILIGHT_ROW
	 */
	public void setUseHilightRow( final String hilightRow ) {
		useHilightRow = nval( getRequestParameter( hilightRow ),useHilightRow );
	}

	/**
	 * 【TAG】ogPopup で検索結果の値を返すキーを、CSV形式で指定します。
	 *
	 * @og.tag
	 * popup の検索結果を返す画面で、結果のラジオボタンにイベントセットします。
	 * この場合、オープンもとのwindow に値を返しますが、そのキーをCSV形式で
	 * 指定します。
	 * なお、このメソッドは、一覧表示(HTMLTable)関係のビューのみでサポートして
	 * いますが、チェックメソッドの関係で、それ以外のビューに適用しても素通り
	 * するようにします。(エラーにしません)
	 *
	 * @og.rev 3.8.6.1 (2006/10/20) 新規追加
	 *
	 * @param  rtnKeys ポップアップ戻りキー (CSV形式)
	 */
	public void setPopupReturnKeys( final String rtnKeys ) {
		popupReturnKeys = nval( getRequestParameter( rtnKeys ),popupReturnKeys );
	}

	/**
	 * 【TAG】table要素に対して class 属性を設定します(初期値:viewTable)。
	 *
	 * @og.tag
	 * Javaの言語使用上、class で作成できないため、代用として
	 * clazz を使用している。
	 * html で作成される属性は、 class で作成される。
	 * 初期値は、"viewTable" です。
	 *
	 * @og.rev 4.0.0.0 (2007/04/16) 新規追加
	 *
	 * @param	cls	class属性
	 */
	public void setClazz( final String cls ) {
		clazz = nval( getRequestParameter( cls ),clazz );
	}

	/**
	 * 【TAG】ビューの高さを指定します。
	 *
	 * @og.tag
	 * id=posDivのDIV要素に高さを指定します。
	 * javascriptからこの高さを見てビューの高さを決定します。
	 * 単位も指定して下さい。
	 *
	 * @og.rev 4.2.0.0 (2008/03/18) 新規追加
	 *
	 * @param	high	ビューの高さ
	 */
	public void setHeight( final String high ) {
		height = nval( getRequestParameter( high ), height );
	}

	/**
	 * 【TAG】ビューの幅を指定します。
	 *
	 * @og.tag
	 * id=posDivのDIV要素に幅を指定します。
	 * javascriptからこの値を見てビューの幅を決定します。
	 * 単位も指定して下さい。(%指定推奨)
	 *
	 * @og.rev 4.2.0.0 (2008/03/18) 新規追加
	 *
	 * @param	wide	指定するビューの幅
	 */
	public void setWidth( final String wide ) {
		width = nval( getRequestParameter( wide ), width );
	}

	/**
	 * 【TAG】ファイルダウンロード時の抜出カラム名(CSV形式)を指定します。
	 *
	 * @og.tag
	 * fileUDのファイル抜出時のカラムをカンマ形式で指定します。
	 *
	 * @og.rev 4.3.0.0 (2008/07/04) 新規追加
	 *
	 * @param	dwnClms	抜出カラム名 (CSV形式)
	 */
	public void setFileDwnClms( final String dwnClms ) {
		fileDwnClms = nval( getRequestParameter( dwnClms ), fileDwnClms );
	}

	/**
	 * 【TAG】編集行のみを表示するかを指定します(初期値:false)。
	 *
	 * @og.tag
	 * 編集行のみを表示するかを指定します。trueがセットされた場合、編集行(改廃Cがついている行)
	 * のみが出力されます。
	 *
	 * @og.rev 4.3.1.0 (2008/09/08)
	 *
	 * @param	sne	編集行のみ出力 [true:編集行のみ/false:全行]
	 */
	public void setSkipNoEdit( final String sne ) {
		skipNoEdit = nval( getRequestParameter( sne ), skipNoEdit );
	}

	/**
	 * 【TAG】(通常は使いません)画面遷移を行わない形式の登録方法を使用するかを指定します。
	 *
	 * @og.tag
	 * 画面遷移なしの登録を行うかどうかを指定します。
	 * trueが指定された場合、HTML出力時に、各行に対して、行番号と改廃Cが属性情報として付加されます。
	 * ここで出力された情報を元に、JavaScriptの入れ替え処理を行います。
	 *
	 * この値は、og:headタグで設定値、または前画面からの値を継承するため、通常、この属性ではセットしません。
	 *
	 * @og.rev 4.3.3.0 (2008/10/01) 新規追加
	 * @og.rev 5.1.3.0 (2010/02/01) noTransition、ajaxSubmitのコントロールは、requestで行う。
	 *
	 * @param   noTrnstn 画面遷移なし登録 [true:画面遷移なし登録/false:通常登録]
	 */
	public void setNoTransition( final String noTrnstn ) {
		setNoTransitionRequest( nval( getRequestParameter( noTrnstn ), isNoTransitionRequest() ) );
	}

	/**
	 * 【TAG】選択行ハイライトの制御をチェックボックスのみで有効にするかを指定します
	 *		(初期値:VIEW_USE_HILIGHT_CHECK_ONLY[={@og.value SystemData#VIEW_USE_HILIGHT_CHECK_ONLY}])。
	 *
	 * @og.tag
	 * 選択行ハイライトの制御をチェックボックスのみで有効にするかを指定します。
	 * trueの場合、チェックボックスのみで有効になります。
	 * falseの場合、行全体の各要素で有効になります。
	 * trueにした場合、チェックボックスが存在しない行に対しては、ハイライトすることはできません。
	 * (初期値:ユーザー定数のVIEW_USE_HILIGHT_CHECK_ONLY[={@og.value SystemData#VIEW_USE_HILIGHT_CHECK_ONLY}])。
	 *
	 * @og.rev 4.3.7.0 (2009/06/01) 新規追加
	 *
	 * @param   hiCkOnly 選択行ハイライト制御 [true:チェックボックスのみ/false:行全体]
	 * @see		org.opengion.hayabusa.common.SystemData#VIEW_USE_HILIGHT_CHECK_ONLY
	 */
	public void setUseHilightCheckOnly( final String hiCkOnly ) {
		isHiRowCheckOnly = nval( getRequestParameter( hiCkOnly ), isHiRowCheckOnly );		// 6.0.2.5 (2014/10/31) refactoring
	}

	/**
	 * 【TAG】選択行のハイライト表示を行った場合に、行選択でRadioボタンにclickイベントを発行させるかどうかを指定します
	 *		(初期値:VIEW_USE_HILIGHT_RADIO_CLICK[={@og.value SystemData#VIEW_USE_HILIGHT_RADIO_CLICK}])。
	 *
	 * @og.tag
	 * trueの場合、clickイベントを発行させます。
	 * falseの場合、従来通り、ハイライト表示でラジオのチェックは入りますが、イベントは発行されません。
	 * 互換性を考えれば、false ですが、POPUP系で、行選択してしまうと、選択できなくなるため、
	 * 行選択で、即POPUPが閉じる true にしておきます。
	 * (初期値:ユーザー定数のVIEW_USE_HILIGHT_RADIO_CLICK[={@og.value SystemData#VIEW_USE_HILIGHT_RADIO_CLICK}])。
	 *
	 * @og.rev 6.2.2.0 (2015/03/27) hilightRow で、行選択時にイベント発行可否を指定できるようにする。
	 *
	 * @param   useClick 行選択でRadioボタンのclickイベント [true:発行する/false:発行しない]
	 * @see		org.opengion.hayabusa.common.SystemData#VIEW_USE_HILIGHT_RADIO_CLICK
	 */
	public void setUseHilightRadioClick( final String useClick ) {
		isHiRadioClick = nval( getRequestParameter( useClick ), isHiRadioClick );		// 6.0.2.5 (2014/10/31) refactoring
	}

	/**
	 * 【TAG】一括入力カラムのアシスト機能を利用するカラム名を、CSV形式で与えます。
	 *
	 * @og.tag
	 * これは、カラムの書き込み時に、一括入力可能なカラムをヘッダーに表示し、
	 * 一括登録できる機能を提供します。
	 * この機能は、jsp/common/bulkSet.js JavaScript とともに提供されます。
	 * IE については、クリップボードも利用できます。Ctrl-C と Ctrl-V でそれぞれ
	 * 一連のカラムの値の取り出しと書き込みが可能になります。
	 * (プルダウンの場合はオプションに上記が追加されます)
	 * "*" を指定すると、すべてのカラムを(columnBulkSet)指定したことになります。
	 * "-" を指定すると、すべてのカラムを(columnBulkSet)除外したことになります。
	 *
	 * システムリソースで、VIEW_USE_DEFAULT_BULKSET を true に設定すると、初期値が "*"
	 * になりますので、個別の view に設定しなくてもこの機能が使用できるようになります。
	 *
	 * この機能は一括入力の手間削減が目的なので、selectedType が radio の場合は、
	 * この機能は無効になります。
	 *
	 * @og.rev 5.2.1.0 (2010/10/01) 新規作成
	 * @og.rev 5.4.4.0 (2012/02/01) コメント追加
	 *
	 * @param	columnName	一括入力カラム (CST形式) (例："OYA,KO,HJO,SU,DYSET,DYUPD")
	 */
	public void setColumnBulkSet( final String columnName ) {
		columnBulkSet = nval( getRequestParameter(columnName),columnBulkSet );
	}

	/**
	 * 【TAG】処理時間を表示する TimeView を表示するかどうか[true:する/false:しない]を指定します
	 *		(初期値:VIEW_USE_TIMEBAR[={@og.value SystemData#VIEW_USE_TIMEBAR}])。
	 *
	 * @og.tag
	 * true に設定すると、処理時間を表示するバーイメージが表示されます。
	 * これは、DB検索、APサーバー処理、画面表示の各処理時間をバーイメージで
	 * 表示させる機能です。処理時間の目安になります。
	 * (初期値:VIEW_USE_TIMEBAR[={@og.value SystemData#VIEW_USE_TIMEBAR}])。
	 *
	 * @og.rev 6.3.6.0 (2015/08/16) useTimeView の初期値を、VIEW_USE_TIMEBAR にする。
	 *
	 * @param	flag	処理時間を表示 [true:する/false:しない]
	 */
	public void setUseTimeView( final String flag ) {
		useTimeView = nval( getRequestParameter( flag ),useTimeView );
	}

	/**
	 * タグの名称を、返します。
	 * 自分自身のクラス名より、自動的に取り出せないため、このメソッドをオーバーライドします。
	 *
	 * @og.rev 4.0.0.0 (2005/01/31) 新規追加
	 *
	 * @return  タグの名称
	 * @og.rtnNotNull
	 */
	@Override
	protected String getTagName() {
		return "view" ;
	}

	/**
	 * seqClmTableを使用するかどうかをチェックします
	 *
	 * ここではtableId,scope,項目名などで厳密にチェックを行い、これらが元の情報と合致しない場合は、
	 * ViewFormTypeの置き換えを行わないようにします。
	 *
	 * @og.rev 5.1.6.0 (2010/05/01) 新規追加
	 * @og.rev 5.3.6.0 (2011/06/01) Edit機能見直しによりメソッド名変更
	 * @og.rev 5.7.7.3 (2014/06/27) カラムが存在しない場合の挙動変更
	 * @og.rev 6.0.2.4 (2014/10/17) Edit機能で、JSPソース変更時の対応
	 * @og.rev 6.2.4.1 (2015/05/22) カラムが DBTableModel に存在しない場合、return から continue に変更
	 *
	 * @param	userViewClms	表示カラム
	 */
	private void makeEditTable( final String userViewClms ) {

			// 1Queryに対して複数Viewがある場合(左右分割など)、Viewの各項目定義が'|'で分割されてセットされています。
			final int vwCnt = nval( (String)getRequestAttribute( "h_vwCnt" ), 0 );
			final String[] clmGroup = StringUtil.csv2Array( userViewClms, '|' );
			if( clmGroup.length > vwCnt ) {
				// 6.0.2.4 (2014/10/17) カラムの存在チェックと、取り除きを同時に行います。
				// AAA,!BBB,CCC,DDD,!EEE から!がついたカラムのみを取り除く(!がある項目は非表示)
				final String[] clms = StringUtil.csv2Array( clmGroup[vwCnt] );
				final StringBuilder buf = new StringBuilder( BUFFER_MIDDLE );
				for( int i=0; i<clms.length; i++ ) {
					final String clm = clms[i];
					// !がついたカラムは、スルーします。
					if( StringUtil.startsChar( clm , '!' ) ) { continue; }	// 6.2.0.0 (2015/02/27) １文字 String.startsWith
					if( table.getColumnNo( clm, false ) < 0 ) {
						continue;	// 6.2.4.1 (2015/05/22) カラムが DBTableModel に存在しない
					}
					else {
						if( buf.length() > 0 ) { buf.append( ',' ); }	// 最初以外は、カンマを追加する。	// 6.0.2.5 (2014/10/31) char を append する。
						buf.append( clm );
					}
				}
				// 行カウントカラムを表示する。
				if( table.getColumnNo( "rowCount", false ) >= 0 ) {
					buf.append( ",rowCount" );
				}

					clmSeqDisplay	= buf.toString();	// 6.0.2.4 (2014/10/17)
					setRequestAttribute( "h_vwCnt", String.valueOf( vwCnt+1 ) );
			}
	}

	/**
	 * Viewで表示した項目をDBLastSqlにセットします。
	 *
	 * ここではtableId,scopeなどで厳密にチェックを行い、これらが元の情報と合致しない場合は、
	 * DBLastSqlへのセットを行わないようにします。
	 *
	 * @og.rev 5.1.6.0 (2010/05/01) 新規追加
	 * @og.rev 5.3.9.0 (2011/09/01) ヘッダーソートを行った場合、編集対象カラムが正しく表示されないバグを修正
	 * @og.rev 6.0.2.4 (2014/10/17) Edit機能で、JSPソース変更時の対応
	 *
	 * @param	lastSql	DBLastSqlオブジェクト
	 */
	private void setViewClms( final DBLastSql lastSql ) {
		// 4.3.0.0 (2008/07/04) fileUD 対応。抜き出すカラムを指定します。
			final String guikey = getGUIInfoAttri( "KEY" );
			if( lastSql.isGuiMatch( guikey ) && lastSql.isTableMatch( tableId, getScope() ) && lastSql.isViewEditable() ) {
				// 5.1.6.0 (2010/05/01) fileDwnClmsが指定されない場合は、Viewで表示されたカラムをセット
				if( form.isEditable() ) {
					// 5.3.9.0 (2011/09/01) リクエストに対して初めて処理する場合に対象カラムを一旦クリアする。
					final boolean vwBln = nval( (String)getRequestAttribute( "h_vwBln" ), false );
					if( !vwBln ) {
						lastSql.setViewClmNames( null );
						lastSql.setClmNames( null );
						setRequestAttribute( "h_vwBln", "true" );
					}

					String clmNames = lastSql.getViewClmNames();
					if( clmNames == null || clmNames.isEmpty() ) {
						clmNames = nval( fileDwnClms, form.getViewClms() );
					}
					else {
						// 1Queryに対して複数Viewがある場合(左右分割など)、Viewの各項目定義が'|'で分割されてセットします。
						clmNames += "|" + nval( fileDwnClms, form.getViewClms() );
					}
					lastSql.setViewClmNames( clmNames );
					lastSql.setClmNames( clmNames.replace( "|", "," ) );
				}
				else {
					lastSql.setViewNotEditable();
				}
				setSessionAttribute( HybsSystem.DB_LAST_SQL_KEY,lastSql );
			}
	}

	/**
	 * 【TAG】(通常使いません)Viewに関する情報をセッションに情報を登録するかどうかを指定します(初期値:true)。
	 *
	 * @og.tag
	 * 同一画面IDで、複数のViewを表示する場合に利用します。
	 * View関連の情報をセッションに登録し利用していますが、複数のViewが存在する場合に前の情報が上書きされてしまいます。
	 * 複数のViewを利用する際に、メイン以外のViewの値をfalseに指定する事でセッションへの保存をしません。
	 * (QueryTagのmainTransも同時に設定するか検討してください）
	 *
	 * 現段階では、セッション保存抑制を行う値はstartNoのみです。
	 *
	 * @og.rev 5.7.6.1 (2010/05/09) 新規作成
	 *
	 * @param  flag メインView指定 [true:メイン/false:その他]
	 */
	public void setMainView( final String flag ) {
		isMainView = nval( getRequestParameter( flag ),isMainView );
	}

//	/**
//	 * シリアライズ用のカスタムシリアライズ書き込みメソッド。
//	 *
//	 * @og.rev 4.0.0.0 (2006/09/31) 新規追加
//	 * @serialData 一部のオブジェクトは、シリアライズされません。
//	 *
//	 * @param	strm	ObjectOutputStreamオブジェクト
//	 * @throws IOException	入出力エラーが発生した場合
//	 */
//	private void writeObject( final ObjectOutputStream strm ) throws IOException {
//		strm.defaultWriteObject();
//	}

//	/**
//	 * シリアライズ用のカスタムシリアライズ読み込みメソッド
//	 *
//	 * ここでは、transient 宣言された内部変数の内、初期化が必要なフィールドのみ設定します。
//	 *
//	 * @og.rev 4.0.0.0 (2006/09/31) 新規追加
//	 * @serialData 一部のオブジェクトは、シリアライズされません。
//	 *
//	 * @param	strm	ObjectInputStreamオブジェクト
//	 * @see #release2()
//	 * @throws IOException	シリアライズに関する入出力エラーが発生した場合
//	 * @throws ClassNotFoundException	クラスを見つけることができなかった場合
//	 */
//	private void readObject( final ObjectInputStream strm ) throws IOException , ClassNotFoundException {
//		strm.defaultReadObject();
//	}

	/**
	 * このオブジェクトの文字列表現を返します。
	 * 基本的にデバッグ目的に使用します。
	 *
	 * @return このクラスの文字列表現
	 * @og.rtnNotNull
	 */
	@Override
	public String toString() {
		return ToString.title( this.getClass().getName() )
				.println( "VERSION"			,VERSION			)
				.println( "tableId"			,tableId			)
				.println( "viewFormId"		,viewFormId			)
				.println( "viewFormType"	,viewFormType		)
				.println( "viewLinkId"		,viewLinkId			)
				.println( "viewMarkerId"	,viewMarkerId		)
				.println( "command"			,command			)
				.println( "startNo"			,startNo			)
				.println( "tempStartNo"		,tempStartNo		)
				.println( "pageSize"		,pageSize 			)
				.println( "pagePlus"		,pagePlus 			)
				.println( "columnWritable"	,columnWritable		)
				.println( "noWritable"		,noWritable			)
				.println( "columnDisplay"	,columnDisplay		)
				.println( "clmSeqDisplay"	,clmSeqDisplay		)
				.println( "noDisplay"		,noDisplay			)
				.println( "columnGroup"		,columnGroup		)
				.println( "noMessage"		,noMessage			)
				.println( "writable"		,writable 			)
				.println( "checked"			,checked			)
				.println( "skip"			,skip 				)
				.println( "rowspan"			,rowspan			)
				.println( "selectedType"	,selectedType		)
				.println( "numberType"		,numberType			)
				.println( "optTypeAttri"	,optTypeAttri		)
				.println( "backLinkCount"	,backLinkCount		)
				.println( "headerSkipCount"	,headerSkipCount	)
				.println( "bodyString"		,bodyString			)
				.println( "useCheckControl"	,useCheckControl 	)
				.println( "useTableSorter"	,useTableSorter		)
				.println( "useParam"		,useParam			)
				.println( "useConsistency"	,useConsistency		)
				.println( "bgColorCycle"	,bgColorCycle		)
				.println( "useScrollBar"	,useScrollBar		)
				.println( "sortKeys"		,sortKeys			)
				.println( "loadMarker"		,loadMarker			)
				.println( "useSelectedRow"	,useSelectedRow		)
				.println( "useSelRowColor"	,useSelRowColor		)
				.println( "useAfterHtmlTag"	,useAfterHtmlTag	)
				.println( "writableControl"	,writableControl	)
				.println( "hight"			,height				)
				.println( "width"			,width				)
				.println( "skipNoEdit"		,skipNoEdit			)
				.println( "noTransition"	,noTransition		)
				.println( "useHilightCheckOnly",isHiRowCheckOnly)		// 6.0.2.5 (2014/10/31) refactoring
//				.println( "COMMAND_LIST"	,COMMAND_LIST		)
				.println( "Other..."	,getAttributes().getAttribute() )
				.fixForm().toString() ;
	}
}
