/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.fukurou.model;

import java.util.ArrayList;
import java.util.List;

import static org.opengion.fukurou.util.HybsConst.BUFFER_MIDDLE;	// 6.1.0.0 (2014/12/26) refactoring

/**
 * [PN],[OYA] などの [] で指定されたカラムで表されたフォーマットデータに対して、
 * DataModel オブジェクトを適用して 各カラムに実データを割り当てるオブジェクトです。
 *
 * カラム名には、特殊カラム名が使用できます。これは、DataModel に存在しないカラム名
 * ですが、値を返すことが出来ます。
 * <pre>
 * [KEY.カラム名] : 行番号付きカラム名
 * [I]            : 行番号
 * [ROW.ID]       : 行毎のチェックボックスのID
 * [ROW.JSON]     : 行毎の全データのJavaScriptオブジェクト形式 { key:val,key:val,... }
 * カラムの前に修飾記号(#,$,!)を付けるとフォーマットを変更できます。
 * ただし、FormatTextField 系 と FormatTable 系で、出力される形式が異なります。
 *                  FormatTextField 系               FormatTable 系
 * [#カラム名]    : TDなしのラベルと入力フィールド   ラベルを出力
 * [$カラム名]    : TDなしの入力フィールドのみ       レンデラーを出力
 * [!カラム名]    : TDなしの値のみ                   値を出力
 *
 * </pre>
 * @og.group 画面表示
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class Formatter {
	/** カラムID(連結文字列)行番号の連結文字列を定義 {@value} */
	public static final String JOINT_STRING = "__" ;

	/** テーブル表示のチェックボックスを特定する id の 名称( id は、この名称＋行番号) {@value} */
	public static final String ROW_ID_KEY = "cb";	// 3.6.0.0 (2004/09/17)

	/** 特殊カラム名の定義: 行番号 [I]	*/
	public static final int SYS_ROWNUM	= -1;		// [KEY.カラム名],[I],[ROW.ID]
	/** 特殊カラム名の定義: [ROW.JSON]	*/
	public static final int SYS_JSON	= -2;		// [ROW.JSON]
	/** 特殊カラム名の定義: 非表示		*/
	public static final int NO_DISPLAY	= -9;		// 6.2.0.1 (2015/03/06) 非表示のマーカー

	private final DataModel<?>	model	;			// 4.3.3.6 (2008/11/15) Generics警告対応
	private int[]				clmNos	;			// フォーマットのカラム番号配列
	private String[]			format	;
	private String[]			clmKeys	;			// フォーマットのカラム名配列
	private char[]				type	;			// '#':ラベルのみ  '$':レンデラー '!':値のみ  その他:通常

	/**
	 * データモデルを指定してフォーマッターを構築します。
	 *
	 * @param  model データモデル
	 */
	public Formatter( final DataModel<?> model ) {		// 4.3.3.6 (2008/11/15) Generics警告対応
		this.model = model;
	}

	/**
	 * フォーマットをセットします。
	 *
	 * @param	fmt  [カラム名]形式のフォーマットデータ
	 */
	public void setFormat( final String fmt ) {
		makeFormatList( fmt );
		advanceFormat();
	}

	/**
	 * フォーマットをセットします。
	 *
	 * @param	fmt  [カラム名]形式のフォーマットデータ
	 */
	private void makeFormatList( final String fmt ) {
		int start = 0;
		int index = fmt.indexOf( '[' );
		final List<String> clmKeyList = new ArrayList<>();
		final List<String> formatList = new ArrayList<>();
		while( index >= 0 ) {
			final int end = fmt.indexOf( ']',index );
			if( end < 0 ) {
				final String errMsg = "[ と ] との対応関係がずれています。"
								+ "format=[" + fmt + "] : index=" + index ;
				throw new RuntimeException( errMsg );
			}

			// [ より前方の文字列は、formatList へ
			if( index > 0 ) { formatList.add( fmt.substring( start,index ) ); }
			else 			{ formatList.add( "" ); }	// ][ と連続しているケース

			// [XXXX] の XXXX部分を処理
			clmKeyList.add( fmt.substring( index+1,end ) );

			start = end+1 ;
			index = fmt.indexOf( '[',start );
		}
		// ] の後方部分は、formatList へ
		formatList.add( fmt.substring( start ) );

		format  = formatList.toArray( new String[formatList.size()] );
		clmKeys = clmKeyList.toArray( new String[clmKeyList.size()] );
	}

	/**
	 * 追加機能フォーマットを作成します。
	 *
	 */
	private void advanceFormat() {

		final int size = clmKeys.length ;
		clmNos = new int[size];
		type   = new char[size];

		// カラム番号の設定と、特殊カラム名処理
		String clm ;
		for( int i=0; i<size; i++ ) {
			clm = clmKeys[i];
			final char ch = clm.charAt(0);
			if( ch == '#' || ch == '$' || ch == '!' ) {
				type[i] = ch;
				clm = clm.substring(1);
				clmKeys[i]	= clm;
				clmNos[i] = model.getColumnNo( clm );
			}
			// [KEY.カラム名] 機能追加
			else if( clm.startsWith( "KEY." ) ) {
				clmNos[i]	= SYS_ROWNUM;
				format[i]	= format[i] + clm.substring(4) + JOINT_STRING ;
			}
			// [I] 機能追加
			else if( "I".equals( clm ) ) {
				clmNos[i]	= SYS_ROWNUM;
			}
			// [ROW.ID] 機能追加
			else if( "ROW.ID".equals( clm ) ) {
				clmNos[i] = SYS_ROWNUM;
				format[i] = format[i] + ROW_ID_KEY ;
			}
			// [ROW.JSON] 機能追加
			else if( "ROW.JSON".equals( clm ) ) {
				clmNos[i] = SYS_JSON;
			}
			else {
				clmNos[i] = model.getColumnNo( clm );
			}
		}
	}

	/**
	 * column にあるセルの属性値をStringに変換して返します。
	 *
	 * @param	row	処理中の行番号
	 * @param	clm	値が参照されるカラム番号
	 *
	 * @return	指定されたセルの値
	 *
	 */
	public String getValue( final int row,final int clm ) {
		final String rtn ;
		if( clm >= 0 ) {
			rtn = String.valueOf( model.getValue( row,clm ) );
		}
		else if( clm == SYS_ROWNUM ) {
			rtn = String.valueOf( row );
		}
		else if( clm == SYS_JSON ) {
			rtn = getJson( row );
		}
		else {
			final String errMsg = "指定のカラム番号に該当する処理が見つかりません。"
							+ "clm=[" + clm + "]" ;
			throw new RuntimeException( errMsg );
		}

		return rtn ;
	}

	/**
	 * 指定の 行番号に対する、DataModel を元に作成したフォーマット文字列を返します。
	 *
	 * @param	row 	行番号( [I]フォーマット処理用 )
	 *
	 * @return  指定のObject配列を元に作成したフォーマット文字列
	 */
	public String getFormatString( final int row ) {
		return getFormatString( row, null );
	}

	/**
	 * 指定の 行番号に対する、DataModel を元に作成したフォーマット文字列を返します。
	 * データはseparatorで指定された区切り文字で囲まれて返されます。
	 *
	 * @og.rev 4.3.1.1 (2008/08/23) switch に、default label が存在しないため、追加
	 *
	 * @param	row 	行番号( [I]フォーマット処理用 )
	 * @param	separator	セパレーター
	 *
	 * @return  内部のDataModelを元に作成したフォーマット文字列
	 * @og.rtnNotNull
	 */
	public String getFormatString( final int row, final String separator ) {
		final StringBuilder rtnStr = new StringBuilder( BUFFER_MIDDLE );

		final int count = clmNos.length;
		for( int i=0; i<count; i++ ) {
			rtnStr.append( format[i] );
			if( clmNos[i] == SYS_ROWNUM ) {
				rtnStr.append( row );
			}
			else if( clmNos[i] == SYS_JSON ) {
				rtnStr.append( getJson( row ) );
			}
			else {
				if( separator == null || separator.isEmpty() ) {
					rtnStr.append( model.getValue( row,clmNos[i] ) );
				}
				else {
					// 4.3.1.1 (2008/08/23) default label が存在しないため、追加
					switch( model.getNativeType( clmNos[i] ) ) {
						case INT:
						case LONG:
						case DOUBLE:
							rtnStr.append( model.getValue( row,clmNos[i] ) );
							break;
						case STRING:
						case CALENDAR:
							rtnStr.append( separator ).append( model.getValue( row,clmNos[i] ) ).append( separator );
							break;
						default:
							throw new AssertionError( "Unexpected enumrated value! " + model.getNativeType( clmNos[i] ) );
					}
				}
			}
		}
		rtnStr.append( format[count] );

		return rtnStr.toString();
	}

	/**
	 * 引数の DataModel を元に作成したフォーマット文字列を返します。
	 * これは、簡易処理で、DataModel オブジェクトは、実質的には、LineModel です。
	 * １行分の配列データを、毎回 clone() するのがちょっとなのですが、
	 * パッケージの関連から、引数が、DataModel オブジェクトになっています。
	 *
	 * @og.rev 6.3.2.0 (2015/07/10) LineModelで、Formatter処理できるように、対応します。
	 *
	 * @param	model 	DataModelオブジェクト(実質的には、LineModelオブジェクト)
	 *
	 * @return  引数のDataModelを元に作成したフォーマット文字列
	 * @og.rtnNotNull
	 */
	public String getLineFormatString( final DataModel<Object> model ) {
		final StringBuilder rtnStr = new StringBuilder( BUFFER_MIDDLE );

		final int count = clmNos.length;
		for( int i=0; i<count; i++ ) {
			rtnStr.append( format[i] )
				.append( model.getValue( 0,clmNos[i] ) );		// 行番号は、0 にしておきます。
		}
		rtnStr.append( format[count] );

		return rtnStr.toString();
	}

	/**
	 * 先のフォーマット情報の［カラム名］を、クエスチョンマークに置き換えたフォーマットを返します。
	 *
	 * これは、java.sql.PreparedStatement 形式のQuery文字列を合成する場合に使用します。
	 *
	 * @return  PreparedStatement形式のQuery文字列
	 * @og.rtnNotNull
	 */
	public String getQueryFormatString() {
		final StringBuilder rtnStr = new StringBuilder( BUFFER_MIDDLE );

		final int count = clmKeys.length;
		for( int i=0; i<count; i++ ) {
			rtnStr.append( format[i] ).append( '?' );
		}
		rtnStr.append( format[count] );

		return rtnStr.toString();
	}

	/**
	 * フォーマットのカラム名配列を返します。
	 *
	 * @return	フォーマットのカラム名配列
	 * @og.rtnNotNull
	 */
	public String[] getClmKeys() {
		return clmKeys.clone();
	}

	/**
	 * フォーマットのカラム番号配列を返します。
	 *
	 * @return	フォーマットのカラム番号配列
	 * @og.rtnNotNull
	 */
	public int[] getClmNos() {
		return clmNos.clone();
	}

	/**
	 * フォーマット配列を返します。
	 *
	 * @return	フォーマット配列
	 * @og.rtnNotNull
	 */
	public String[] getFormat() {
		return format.clone();
	}

	/**
	 * タイプ文字列配列を返します。
	 * タイプとは、[XXX] の記述で、[#XXX] は、XXXカラムのラベルを、[$XXX]は、XXXカラムの
	 * レンデラーを、[!XXX} は、値のみ取り出す指定を行います。
	 * 主に、TextField系のフォーマットとTable系では、意味合いが異なりますので、
	 * ご注意ください。
	 *
	 * @return タイプ文字列配列 '#':ラベルのみ  '$':レンデラー '!':値のみ  その他:通常
	 * @og.rtnNotNull
	 */
	public char[] getType() {
		return type.clone();
	}

	/**
	 * 行毎の全データのJavaScriptオブジェクト形式 を返します。
	 *
	 * JavaScriptオブジェクト形式とは、{ key:val,key:val,... }の形式で、
	 * ここでは、内部設定された DataModel のすべてのカラム名をキーに、
	 * 引数で渡された 配列を 値として使用します。
	 *
	 * @param	row	(DataModelの)行番号
	 *
	 * @return  指定の行番号に対応する全データのJSON形式データ
	 * @og.rtnNotNull
	 */
	public String getJson( final int row ) {
		final String[] names = model.getNames();
		final Object[] vals  = model.getValues( row );

		final StringBuilder rtnStr = new StringBuilder( BUFFER_MIDDLE );

		rtnStr.append( "{'I':'" ).append( row ).append( '\'' );	// 行番号

		for( int i=0; i<names.length; i++ ) {
			rtnStr.append( ",'" ).append( names[i] ).append( "':'" );
			rtnStr.append( vals[i] ).append( '\'' );
		}
		rtnStr.append( '}' );		// 6.0.2.5 (2014/10/31) char を append する。

		return rtnStr.toString();
	}
}
