/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.report;

import java.util.Map;

import org.opengion.fukurou.util.Shell;
import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.db.DBTableModel;
import static org.opengion.fukurou.system.HybsConst.CR;				// 6.1.0.0 (2014/12/26) refactoring
import static org.opengion.fukurou.system.HybsConst.BUFFER_MIDDLE;	// 6.1.0.0 (2014/12/26) refactoring

/**
 * RFIDPrintPoint インターフェース のデフォルト実装クラスです。
 * execute() をオーバーライドして、各種RFIDシステムに対応して下さい。
 *
 * @og.group 帳票システム
 *
 * @version  4.0
 * @author	 Masakazu Takahashi
 * @since    JDK6.0,
 */
public abstract class AbstractRFIDPrintPointService implements RFIDPrintPointService {

	protected final StringBuilder errMsg = new StringBuilder( BUFFER_MIDDLE );	// エラーメッセージ

	protected String		ykno		;
	protected String 		systemId	;
	protected String		fgrun		;
	protected String		hostName	;
	protected String		prtName		;
	protected DBTableModel	table		;
	protected String		prgdir		;
	protected String		prgfile		;
	protected String		outdir		;
	/** 6.4.3.1 (2016/02/12) 作成元のMapを、HashMap から ConcurrentHashMap に置き換え。  */
	protected Map<String, String[]>	rfidLayout;
	protected String		prtid		;			// 5.4.3.0 (2011/12/26)
	protected String		portnm		;			// 5.4.3.1 (2011/12/27)
	protected String		listid		;			// 5.4.3.4 (2012/01/12)
	protected String		layoutFile	;			// 5.4.3.9 (2012/01/25)

	protected String		fgkan		= GE50Access.FG_ERR2;	// 初期値はアプリエラー
	protected String		shellCmd	;

//	protected int TIMEOUT = HybsSystem.sysInt( "REPORT_DAEMON_TIMEOUT" ); //Shellタイムアウト
	private static final int TIMEOUT = HybsSystem.sysInt( "REPORT_DAEMON_TIMEOUT" ); //Shellタイムアウト 6.3.9.1 (2015/11/27) private static final化

	/**
	 * デフォルトコンストラクター
	 *
	 * @og.rev 6.4.2.0 (2016/01/29) PMD refactoring. Each class should declare at least one constructor.
	 */
	protected AbstractRFIDPrintPointService() {
		super();		// これも、自動的に呼ばれるが、空のメソッドを作成すると警告されるので、明示的にしておきます。
	}

	/**
	 * RFID発行処理
	 * RFIDのシステムに応じてこのメソッドをオーバーライドします
	 * 実行後はfgkanの値を正しい値でセットしなおして下さい。
	 *
	 * @return 結果 [true:正常/false:異常]
	 */
	public abstract boolean execute();

	/**
	 * 帳票起動された要求番号をセットします。
	 *
	 * @param   no 要求NO
	 */
	public void setYkno( final String no ) {
		ykno = no;
	}

	/**
	 * システムIDをセットします。
	 *
	 * @param   id システムID
	 */
	public void setSystemId( final String id ) {
		systemId = id;
	}

	/**
	 * 実行方法をセットします。
	 *
	 * @param   flag 実行方法
	 */
	public void setFgrun( final String flag ) {
		fgrun = flag;
	}

	/**
	 * 帳票デーモンが実行されているホスト名をセットします。
	 *
	 * @param   host ホスト名
	 */
	public void setHostName( final String host ) {
		hostName = host;
	}

	/**
	 * プリンター名をセットします。
	 *
	 * @param   printerName	プリンタ名
	 */
	public void setPrinterName( final String printerName ) {
		prtName = printerName;
	}

	/**
	 * DBTableModel をセットします。
	 *
	 * @param   tbl DBTableModelオブジェクト
	 */
	public void setTable( final DBTableModel tbl ) {
		table = tbl;
	}

	/**
	 * REPORT_RFID_SERVICE_CLASSから起動するバッチ等のプログラムをセットします。
	 *
	 * @param dir バッチプログラムディレクトリ
	 */
	public void setPrgDir( final String dir ){
		prgdir = dir;
	}

	/**
	 * REPORT_RFID_SERVICE_CLASSから起動するバッチ等のプログラムをセットします。
	 *
	 * @param file バッチプログラム名
	 */
	public void setPrgFile( final String file ){
		prgfile = file;
	}

	/**
	 * ファイル出力時のディレクトリを指定します。
	 *
	 * @og.rev 4.3.2.2 (2008/09/22) 板金 RFID対応
	 * @param dir ファイル出力ディレクトリ
	 */
	public void setOutDir( final String dir ){
		outdir = dir;
	}

	/**
	 * ファイル出力時のディレクトリを指定します。
	 *
	 * @og.rev 5.4.3.0 (2011/12/26) _DEFAULT対応
	 * @param id プリンタID
	 */
	public void setPrtId( final String id ){
		prtid = id;
	}

	/**
	 * プリンタのポート名。
	 *
	 * @og.rev 5.4.3.1 (2011/12/27)
	 * @param port ポート名
	 */
	public void setPortnm( final String port ){
		portnm = port;
	}

	/**
	 * RFIDのレイアウトデータをセットします。
	 *
	 * キーはカラム名です。
	 * 値はGE58で定義された出力区分及びパラメーターの配列です。
	 * 配列の各インデックス番号の意味は、RFIDPrintRequestを参照して下さい。
	 *
	 * @param   layout RFIDレイアウト
	 */
	public void setLayout( final Map<String, String[]> layout ) {
		rfidLayout = layout;
	}

	/**
	 * 帳票IDをセットします。
	 *
	 * @param   id 帳票ID
	 */
	public void setListId( final String id ) {
		listid = id;
	}

	/**
	 * 帳票IDをセットします。
	 *
	 * @og.rev 5.4.3.9 (2012/01/25)
	 * @param file 雛形ファイル名
	 */
	public void setLayoutFile( final String file ) {
		layoutFile = file;
	}

	/**
	 * 完成フラグを返します。
	 *
	 * @return 完成フラグ String
	 */
	public String getFgkan(){
		return fgkan;
	}

	/**
	 * エラーメッセージを返します。
	 *
	 * @return エラーメッセージ String
	 * @og.rtnNotNull
	 */
	public String getErrMsg(){
		return errMsg.toString();
	}

	/**
	 * シェルの実行を行います。
	 *
	 * @og.rev 5.4.3.0 (2011/12/26)
	 *
	 * @return 結果 [true:正常/false:異常]
	 */
	protected boolean programRun(){
		final Shell shell = new Shell();
		shell.setCommand( shellCmd,true );		// BATCHプロセスで実行する
		shell.setWait( true );					// プロセスの終了を待つ
		shell.setTimeout( TIMEOUT );

		final int rtnCode = shell.exec();				// 0 は正常終了を示す

		if( rtnCode != 0 ) {
			errMsg.append( "Shell Command exequte Error." ).append( CR )
				.append( "==============================" ).append( CR )
				.append( shellCmd ).append( CR )
				.append( shell.getStdoutData() ).append( CR )
				.append( shell.getStderrData() ).append( CR )
				.append( CR );
			return false;
		}

		return true;
	}

}
