/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import org.opengion.fukurou.util.XHTMLTag;

/**
 * 指定の位置に画像を配置するHTML拡張タグです。
 *
 * @og.formSample
 * ●形式：&lt;og:img src="･･･" alt="･･･"; /&gt;
 * ●body：なし
 *
 * ●Tag定義：
 *   &lt;og:img
 *       name               【HTML】名前を指定します
 *       src              ○【HTML】表示させる画像のURLを指定します(必須)。
 *       alt              ○【HTML】画像が表示されない場合、代わりに表示させるテキストを指定します(必須)。
 *       longdesc           【HTML】画像部分についての詳しい情報があるURL(longdesc)を指定します
 *       width              【HTML】横幅をピクセル数または％で指定します
 *       height             【HTML】縦幅をピクセル数または％で指定します
 *       usemap             【HTML】クライアントサイドクリッカブルを指定します
 *       ismap              【HTML】サーバーサイドクリッカブルを指定します
 *       clazz              【HTML】要素に対して class 属性を設定します
 *       id                 【HTML】要素に対して固有の名前(id)をつける場合に設定します
 *       title              【HTML】要素に対する補足的情報(title)を設定します
 *       style              【HTML】この要素に対して適用させるスタイルシート(style)を設定します
 *       lang               【HTML】要素の内容と他の属性値の言語(lang,xml:lang)を指定します
 *       dir                【HTML】文字表記の方向(dir)を指定します
 *       onClick            【HTML】JavaScriptのイベント onClick を設定します(例:onClick="renew('query.jsp','QUERY');")
 *       onBlur             【HTML】JavaScriptのイベント onBlur を設定します(例:onBlur="this.value=value.toUpperCase();")
 *       onFocus            【HTML】JavaScriptのイベント onFocus を設定します
 *       ondblClick         【HTML】JavaScriptのイベント ondblClick を設定します
 *       onMouseDown        【HTML】JavaScriptのイベント onMouseDown を設定します
 *       onMouseUp          【HTML】JavaScriptのイベント onMouseUp を設定します
 *       onMouseMove        【HTML】JavaScriptのイベント onMouseMove を設定します
 *       onMouseOut         【HTML】JavaScriptのイベント onMouseOut を設定します
 *       onMouseOver        【HTML】JavaScriptのイベント onMouseOver を設定します
 *       roles              【TAG】ロールをセットします
 *       caseKey            【TAG】このタグ自体を利用するかどうかの条件キーを指定します(初期値:null) 5.7.7.2 (2014/06/20)
 *       caseVal            【TAG】このタグ自体を利用するかどうかの条件値を指定します(初期値:null) 5.7.7.2 (2014/06/20)
 *       caseNN             【TAG】指定の値が、null/ゼロ文字列 でない場合(Not Null=NN)は、このタグは使用されます(初期値:判定しない) 5.7.7.2 (2014/06/20)
 *       caseNull           【TAG】指定の値が、null/ゼロ文字列 の場合は、このタグは使用されます(初期値:判定しない) 5.7.7.2 (2014/06/20)
 *       caseIf             【TAG】指定の値が、true/TRUE文字列の場合は、このタグは使用されます(初期値:判定しない)
 *       debug              【TAG】デバッグ情報を出力するかどうか[true/false]を指定します(初期値:false)
 *   /&gt;
 *
 * ●使用例
 *    &lt;og:img src="{&#064;URL}" alt="sample" /&gt;
 *
 *        src  : 表示させる画像のURL(必須)
 *        alt  : 画像が表示されない場合、代わりに表示させるテキスト(必須)
 *
 * @og.group 画面部品
 *
 * @version  4.0
 * @author   Kohei Naruse
 * @since    JDK5.0,
 */
public class ImageTag extends HTMLTagSupport {
	/** このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "6.4.2.0 (2016/01/29)" ;
	private static final long serialVersionUID = 642020160129L ;

	/**
	 * デフォルトコンストラクター
	 *
	 * @og.rev 6.4.2.0 (2016/01/29) PMD refactoring. Each class should declare at least one constructor.
	 */
	public ImageTag() {
		super();		// これも、自動的に呼ばれるが、空のメソッドを作成すると警告されるので、明示的にしておきます。
	}

	/**
	 * 指定位置に画像を配置するためのイメージタグを作成します。
	 *
	 * @return  イメージタグ
	 * @og.rtnNotNull
	 */
	@Override
	protected String makeTag() {
		// 6.1.1.0 (2015/01/17) refactoring. ロジックの見直し

		return XHTMLTag.img( getAttributes() ) ;
	}

	/**
	 * 【HTML】名前を指定します。
	 *
	 * @og.tag 名前を指定します。
	 *
	 * @param   name 名前
	 */
	public void setName( final String name ) {
		set( "name",getRequestParameter( name ) );
	}

	/**
	 * 【HTML】表示させる画像のURLを指定します。
	 *
	 * @og.tag 表示させる画像のURLを指定します。
	 *
	 * @param   src 画像のURL
	 */
	public void setSrc( final String src ) {
		set( "src",getRequestParameter( src ) );
	}

	/**
	 * 【HTML】画像が表示されない場合、代わりに表示させるテキストを指定します。
	 *
	 * @og.tag 画像が表示されない場合、代わりに表示させるテキストを指定します。
	 *
	 * @param   alt 代替テキスト
	 */
	public void setAlt( final String alt ) {
		set( "alt",getRequestParameter( alt ) );
	}

	/**
	 * 【HTML】画像部分についての詳しい情報があるURL(longdesc)を指定します。
	 *
	 * @og.tag longdescを指定します。
	 *
	 * @param   longdesc 詳しい情報があるURL
	 */
	public void setLongdesc( final String longdesc ) {
		set( "longdesc",getRequestParameter( longdesc ) );
	}

	/**
	 * 【HTML】横幅をピクセル数または％で指定します。
	 *
	 * @og.tag 横幅を指定します。
	 *
	 * @param   width 横幅(ピクセル数または％)
	 */
	public void setWidth( final String width ) {
		set( "width",getRequestParameter( width ) );
	}

	/**
	 * 【HTML】縦幅をピクセル数または％で指定します。
	 *
	 * @og.tag 縦幅を指定します。
	 *
	 * @param   height 縦幅(ピクセル数または％)
	 */
	public void setHeight( final String height ) {
		set( "height",getRequestParameter( height ) );
	}

	/**
	 * 【HTML】クライアントサイドクリッカブルを指定します。
	 *
	 * @og.tag クライアントサイドクリッカブルを指定します。
	 *
	 * @param   usemap クライアントサイドクリッカブル
	 */
	public void setUsemap( final String usemap ) {
		set( "usemap",getRequestParameter( usemap ) );
	}

	/**
	 * 【HTML】サーバーサイド・イメージマップを使用する場合に指定します。
	 *
	 * @og.tag サーバーサイドクリッカブルを指定します。
	 *
	 * @param   ismap クライアントサイドクリッカブル
	 */
	public void setIsmap( final String ismap ) {
		set( "ismap",getRequestParameter( ismap ) );
	}

	/**
	 * タグの名称を、返します。
	 * 自分自身のクラス名より、自動的に取り出せないため、このメソッドをオーバーライドします。
	 *
	 * @og.rev 4.0.0.0 (2005/01/31) 新規追加
	 *
	 * @return  タグの名称
	 * @og.rtnNotNull
	 */
	@Override
	protected String getTagName() {
		return "img" ;
	}
}
