/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.view;

/**
 * 検索結果を単純なリスト形式で表示するクラスです。
 *
 * このクラスでは、検索結果を単純なリストで表示します。
 * 表示のみでこの表示フォーマットを利用してデータ編集を行うことはできません。
 *
 * 各カラムのデータは、カンマによって連結され、またヘッダー部分も出力されません。
 * さらに各カラムの属性値に基づくclass属性等も一切出力されません。
 *
 * AbstractViewForm により、setter/getterメソッドのデフォルト実装を提供しています。
 * 各HTMLのタグに必要な setter/getterメソッドのみ，追加定義しています。
 *
 * AbstractViewForm を継承している為,ロケールに応じたラベルを出力させる事が出来ます。
 *
 * @og.group 画面表示
 *
 * @version  4.0
 * @author	 Hiroki Nakamura
 * @since    JDK5.0,
 */
public class ViewForm_HTMLSimpleList extends ViewForm_HTMLTable {
	/** このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "6.4.2.0 (2016/01/29)" ;

	// 警告時の行ごとに色を変更する時の、デフォルトクラス属性
	private static final String BG_WARNING_COLOR = " class=\"row_warning\"";

	// エラー時の行ごとに色を変更する時の、デフォルトクラス属性
	private static final String BG_ERROR_COLOR = " class=\"row_error\"";

	private String colorRow = "";

	/**
	 * デフォルトコンストラクター
	 *
	 * @og.rev 6.4.2.0 (2016/01/29) PMD refactoring. Each class should declare at least one constructor.
	 */
	public ViewForm_HTMLSimpleList() {
		super();		// これも、自動的に呼ばれるが、空のメソッドを作成すると警告されるので、明示的にしておきます。
	}

	/**
	 * DBTableModel から HTML文字列を作成して返します。
	 * startNo(表示開始位置)から、pageSize(表示件数)までのView文字列を作成します。
	 * 表示残りデータが pageSize 以下の場合は,残りのデータをすべて出力します。
	 *
	 * @og.rev 6.3.9.1 (2015/11/27) カラムが一つも表示されないケースは、考えないことにする。
	 *
	 * @param  startNo	  表示開始位置
	 * @param  pageSize   表示件数
	 *
	 * @return	DBTableModelから作成された HTML文字列
	 * @og.rtnNotNull
	 */
	@Override
	public String create( final int startNo, final int pageSize )  {
		if( getRowCount() == 0 ) { return ""; }

		final int lastNo = getLastNo( startNo, pageSize );

		final StringBuilder out = new StringBuilder( BUFFER_LARGE );

		final int clmCnt = getColumnCount();
		for( int row=startNo; row<lastNo; row++ ) {
			if( isSkip( row ) || isSkipNoEdit( row ) ) { continue; }

			if( row != startNo ) { out.append( "<br />" ); }

			out.append( "<span " ).append( getBgColorCycleClass(row) ).append( '>' );		// 6.0.2.5 (2014/10/31) char を append する。
			// 6.3.9.1 (2015/11/27) カラムが一つも表示されないケースは、考えないことにする。
//			boolean isFirstColumn = true;
//			for( int column=0; column<clmCnt; column++ ) {
//				if( isColumnDisplay( column ) ) {
//					// 6.0.2.5 (2014/10/31) char を append する。
//					if( isFirstColumn ) { out.append( row+1 ); }
//					out.append( ',' );
//					out.append( getValueLabel(row,column) );
//					isFirstColumn = false;
//				}
//			}
			out.append( row+1 );
			for( int column=0; column<clmCnt; column++ ) {
				if( isColumnDisplay( column ) ) {
					out.append( ',' ).append( getValueLabel(row,column) );			// row+1 を append しているので、これでよい。
				}
			}
			out.append( "</span>" );
		}

		return out.toString();
	}

	/**
	 * テーブルのバックグラウンドカラーの入れ替えのサイクルをセットします。
	 * 0以上(通常)、-1(ワーニング)、-2以下(エラー)
	 * 初期値は、0以上(通常)です。
	 *
	 * @param  sycle  0以上(通常)、-1(ワーニング)、-2以下(エラー)
	 */
	@Override
	public void setBgColorCycle( final int sycle ) {
		if( sycle == -1 ) {					// -1(ワーニング)
			colorRow		= BG_WARNING_COLOR ;
		}
		else if( sycle < -1 ) {				// -2以下(エラー)
			colorRow		= BG_ERROR_COLOR ;
		}
	}

	/**
	 * テーブルのバックグラウンドカラーの値をセットします。
	 *
	 * @param	row 行番号( ０から始める )
	 *
	 * @return	行の色を指定する class 属性( cssファイルで指定 )
	 */
	@Override
	protected String getBgColorCycleClass( final int row ) {
		return colorRow ;
	}

	/**
	 * 表示項目の編集(並び替え)が可能かどうかを返します。
	 *
	 * @og.rev 5.1.6.0 (2010/05/01) 新規追加
	 *
	 * @return	表示項目の編集(並び替え)が可能かどうか(false:不可能)
	 */
	@Override
	public boolean isEditable() {
		return false;
	}

}
