/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.io;

import java.io.File;										// 6.2.0.0 (2015/02/27)
import java.util.Arrays;									// 6.2.0.0 (2015/02/27)

import org.opengion.fukurou.model.TableModelHelper;			// 6.2.0.0 (2015/02/27) クラス名変更
import org.opengion.fukurou.model.EventReader_TEXT;			// 6.2.0.0 (2015/02/27) 

import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.io.AbstractTableReader;		// 6.2.0.0 (2015/02/27)

import static org.opengion.fukurou.system.HybsConst.CR ;		// 6.2.2.0 (2015/03/27)

/**
 * POI による、EXCELバイナリファイルを読み取る実装クラスです。
 *
 * ファイル名、シート名を指定して、データを読み取ることが可能です。
 * 第一カラムが # で始まる行は、コメント行なので、読み飛ばします。
 * カラム名の指定行で、カラム名が null の場合は、その列は読み飛ばします。
 *
 * 入力形式は、openXML形式にも対応しています。
 * ファイルの内容に応じて、.xlsと.xlsxのどちらで読み取るかは、内部的に
 * 自動判定されます。
 *
 * @og.rev 3.5.4.8 (2004/02/23) 新規作成
 * @og.rev 4.3.6.7 (2009/05/22) ooxml形式対応
 * @og.rev 6.2.0.0 (2015/02/27) TableReader クラスの呼び出し元メソッドの共通化(EXCEL,TEXT)。新規
 * @og.group ファイル入力
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class TableReader_Text extends AbstractTableReader {
	/** このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "6.4.2.0 (2016/01/29)" ;

	/**
	 * デフォルトコンストラクター
	 *
	 * @og.rev 6.4.2.0 (2016/01/29) PMD refactoring. Each class should declare at least one constructor.
	 */
	public TableReader_Text() {
		super();		// これも、自動的に呼ばれるが、空のメソッドを作成すると警告されるので、明示的にしておきます。
	}

	/**
	 * DBTableModel から 各形式のデータを作成して,BufferedReader より読み取ります。
	 * コメント/空行を除き、最初の行は、必ず項目名が必要です。
	 * それ以降は、コメント/空行を除き、データとして読み込んでいきます。
	 * このメソッドは、EXCEL 読み込み時に使用します。
	 *
	 * @og.rev 4.0.0.0 (2006/09/31) 新規追加
	 * @og.rev 5.1.6.0 (2010/05/01) columns 処理 追加
	 * @og.rev 5.1.6.0 (2010/05/01) skipRowCountの追加
	 * @og.rev 5.1.8.0 (2010/07/01) Exception をきちっと記述(InvalidFormatException)
	 * @og.rev 5.2.1.0 (2010/10/01) setTableColumnValues メソッドを経由して、テーブルにデータをセットする。
	 * @og.rev 5.5.1.2 (2012/04/06) HeaderData を try の上にだし、エラーメッセージを取得できるようにする。
	 * @og.rev 5.5.7.2 (2012/10/09) sheetNos 追加による複数シートのマージ読み取りサポート
	 * @og.rev 5.5.8.2 (2012/11/09) HeaderData に デバッグフラグを渡します。
	 * @og.rev 6.0.2.5 (2014/10/31) debug=true 時に、進捗が見えるようにします。
	 * @og.rev 6.1.0.0 (2014/12/26) Excel関係改善
	 * @og.rev 6.2.0.0 (2015/02/27) TableReader クラスの呼び出し元メソッドの共通化(EXCEL,TEXT)。新規
	 * @og.rev 6.2.1.0 (2015/03/13) TableReaderModel を外部からセットします。
	 * @og.rev 6.2.3.0 (2015/05/01) 行読み飛ばし nullSkipClm追加
	 *
	 * @param   file 読み取り元ファイル
	 * @param   enc ファイルのエンコード文字列(未使用)
	 */
	@Override
	public void readDBTable( final File file , final String enc ) {
		final boolean isDebug = isDebug();			// 5.5.7.2 (2012/10/09) デバッグ情報

		if( isDebug ) { System.out.println( " Filename=" + file ) ; }

		final TableModelHelper helper = new TableModelHelper() {

			/**
			 * カラム名配列がそろった段階で、イベントが発生します。
			 *
			 * openGion での標準的な処理は、colNo==0 の時に、val の先頭が、#NAME
			 * で始まるレコードを、名前配列として認識します。
			 * #value( String,int,int ) で、この #NAME だけは、継続処理されます。
			 * その上で、#NAME レコードが終了した時点で、カラム名配列が完成するので
			 * そこで初めて、このメソッドが呼ばれます。
			 *
			 * @og.rev 6.0.3.0 (2014/11/13) 新規作成
			 * @og.rev 6.1.0.0 (2014/12/26) omitNames 属性を追加
			 *
			 * @param   names  カラム名配列
			 * @see		#value( String,int,int )
			 */
			@Override
			public void columnNames( final String[] names ) {
				setTableDBColumn( names ) ;
			}

			/**
			 * row にあるセルのオブジェクト値がそろった段階で、イベントが発生します。
			 *
			 * @og.rev 6.0.3.0 (2014/11/13) 新規作成
			 * @og.rev 6.2.1.0 (2015/03/13) setTableColumnValuesに、行番号を引数に追加
			 *
			 * @param   vals    文字列値の１行分の配列
			 * @param   rowNo   行番号(0～)
			 */
			@Override
			public void values( final String[] vals,final int rowNo ) {
				if( isDebug && rowNo % 100 == 0 ) {
					System.out.println( "  No.[" + rowNo + "]=" + Arrays.toString( vals ) ) ;
				}
				setTableColumnValues( vals,rowNo );		// 6.2.1.0 (2015/03/13)
			}
		};

		helper.setDebug( isDebug );							// 6.2.0.0 (2015/02/27)
		helper.setConstData( constKeys , constAdrs );		// 6.1.0.0 (2014/12/26) 外部から固定値情報を指定。
		helper.setNames( columns , isUseNumber() );			// 6.1.0.0 (2014/12/26) 外部からカラム名配列を指定。
		helper.setSkipRowCount( getSkipRowCount() );		// 6.1.0.0 (2014/12/26) 外部からスキップ行数を指定。
		helper.setNullBreakClm( nullBreakClm );				// 6.2.0.0 (2015/02/27) 外部からnullBreakClmを指定。
		helper.setNullSkipClm( nullSkipClm );				// 6.2.3.0 (2015/05/01) 外部からnullSkipClmを指定。

		final EventReader_TEXT reader = new EventReader_TEXT();
		reader.setSeparator( separator );					// 6.2.0.0 (2015/02/27) 外部からエンコードを指定。
		reader.setEncode( enc );							// 6.2.0.0 (2015/02/27) 外部からエンコードを指定。

		reader.eventReader( file , helper );

		// 最後まで、#NAME が見つから無かった場合
		if( !helper.isNameSet() ) {
			final String errMsg = "最後まで、#NAME が見つかりませんでした。"
							+ CR
							+ "ファイル形式が異なるか、もしくは損傷している可能性があります。"
							+ "Class=[Text], File=[" + file + "], Encode=[" + enc + "], Separator=[" + separator + "]"
							+ CR ;
			throw new HybsSystemException( errMsg );
		}

		if( isDebug ) { System.out.println( "  TableReader End." ) ; }
	}
}
