/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.io;

import org.opengion.hayabusa.db.ColumnActionListener;		// 6.2.2.0 (2015/03/27)

import java.io.File;										// 6.2.0.0 (2015/02/27)

/**
 * DBTableModel インターフェース のオブジェクトをReader を用いて入力する為の，共通インターフェースです。
 *
 * @og.group ファイル入力
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public interface TableReader {

	/**
	 * ヘッダー情報の入力時の区切り文字
	 */
	String TAB_SEPARATOR = "\t";		// 項目区切り文字

	/**
	 * DBTableModel から 各形式のデータを作成して,Reader より読み取ります。
	 *
	 * @og.rev 3.5.4.3 (2004/01/05) 引数に、BufferedReader を受け取ル要に変更します。
	 * @og.rev 6.2.0.0 (2015/02/27) TableReader クラスの呼び出し元メソッドの共通化(EXCEL,TEXT)。新規
	 *
	 * @param   filename 読み取り元ファイル名
	 * @param   enc ファイルのエンコード文字列
	 */
	void readDBTable( final File filename , final String enc );

	/**
	 * データを読み込む場合の,区切り文字をセットします。
	 *
	 * なお，このメソッドは,サブクラスによっては,使用しない場合があります。
	 * もし，使用しないサブクラスを作成する場合は, UnsupportedOperationException
	 * を throw するように,サブクラスで実装して下さい。
	 *
	 * @param   separator 区切り文字
	 */
	void setSeparator( final String separator ) ;

	/**
	 * DBTableModelのデータとしてEXCELファイルを読み込むときのシート名を設定します。
	 * これにより、複数の形式の異なるデータを順次読み込むことや、シートを指定して
	 * 読み取ることが可能になります。
	 * sheetNos と sheetName が同時に指定された場合は、sheetNos が優先されます。エラーにはならないのでご注意ください。
	 * のでご注意ください。
	 * このメソッドは、isExcel() == true の場合のみ利用されます。
	 *
	 * @og.rev 3.5.4.2 (2003/12/15) 新規追加
	 *
	 * @param   sheetName シート名
	 * @see		#setSheetNos( String ) 
	 */
	void setSheetName( final String sheetName ) ;

	/**
	 * EXCELファイルを読み込むときのシート番号を指定します(初期値:0)。
	 *
	 * EXCEL読み込み時に複数シートをマージして取り込みます。
	 * シート番号は、0 から始まる数字で表します。
	 * ヘッダーは、最初のシートのカラム位置に合わせます。（ヘッダータイトルの自動認識はありません。）
	 * よって、指定するシートは、すべて同一レイアウトでないと取り込み時にカラムのずれが発生します。
	 * 
	 * シート番号の指定は、CSV形式で、複数指定できます。また、N-M の様にハイフンで繋げることで、
	 * N 番から、M 番のシート範囲を一括指定可能です。また、"*" による、全シート指定が可能です。
	 * これらの組み合わせも可能です。（ 0,1,3,5-8,10-* ）
	 * ただし、"*" に関しては例外的に、一文字だけで、すべてのシートを表すか、N-* を最後に指定するかの
	 * どちらかです。途中には、"*" は、現れません。
	 * シート番号は、重複(1,1,2,2)、逆転(3,2,1) での指定が可能です。これは、その指定順で、読み込まれます。
	 * sheetNos と sheetName が同時に指定された場合は、sheetNos が優先されます。エラーにはならないのでご注意ください。
	 * このメソッドは、isExcel() == true の場合のみ利用されます。
	 * 
	 * 初期値は、0（第一シート） です。
	 *
	 * @og.rev 5.5.7.2 (2012/10/09) 新規追加
	 *
	 * @param   sheetNos EXCELファイルのシート番号（0から始まる）
	 * @see		#setSheetName( String ) 
	 */
	void setSheetNos( final String sheetNos ) ;

	/**
	 * EXCELファイルを読み込むときのシート単位の固定値を設定するためのカラム名とアドレスを指定します。
	 * カラム名は、CSV形式で指定します。
	 * 対応するアドレスを、EXCEL上の行-列を０から始まる整数でCSV形式で指定します。
	 * これにより、シートの一か所に書かれている情報を、DBTableModel のカラムに固定値として
	 * 設定することができます。
	 * 例として、DB定義書で、テーブル名をシートの全レコードに設定したい場合などに使います。
	 * このメソッドは、isExcel() == true の場合のみ利用されます。
	 *
	 * @og.rev 5.5.8.2 (2012/11/09) 新規追加
	 *
	 * @param   constKeys 固定値となるカラム名(CSV形式)
	 * @param   constAdrs 固定値となるアドレス(行-列,行-列,・・・)
	 */
	void setSheetConstData( final String constKeys,final String constAdrs ) ;

	/**
	 * ここに指定されたカラム列に NULL が現れた時点で読み取りを中止します。
	 *
	 * これは、指定のカラムは必須という事を条件に、そのレコードだけを読み取る処理を行います。
	 * 複数Sheetの場合は、次のSheetを読みます。
	 * 現時点では、Excel の場合のみ有効です。
	 *
	 * @og.rev 5.5.8.2 (2012/11/09) 新規追加
	 *
	 * @param   clm カラム列
	 */
	void setNullBreakClm( final String clm ) ;

	/**
	 * ここに指定されたカラム列に NULL が現れたレコードは読み飛ばします。
	 *
	 * 例えば、更新対象カラムで、null の場合は、何もしない、などのケースで使用できます。
	 * 複数カラムの場合は、AND条件やOR条件などが、考えられるため、
	 * カラムを一つにまとめて、指定してください。
	 *
	 * @og.rev 6.2.3.0 (2015/05/01) 行読み飛ばし nullSkipClm追加
	 *
	 * @param   clm カラム列
	 */
	void setNullSkipClm( final String clm ) ;

	/**
	 * 読み取り元ファイルのカラム列を、外部(タグ)より指定します。
	 * ファイルに記述された #NAME より優先して使用されます。
	 *
	 * @og.rev 3.5.4.5 (2004/01/23) 新規作成
	 *
	 * @param   clms 読み取り元ファイルのカラム列(CSV形式)
	 */
	void setColumns( final String clms ) ;

	/**
	 * 行番号情報を、使用している(true)/していない(false)を指定します。
	 *
	 * 通常のフォーマットでは、各行の先頭に行番号が出力されています。
	 * 読み取り時に、#NAME 属性を使用する場合は、この行番号を無視しています。
	 * #NAME 属性を使用せず、columns 属性でカラム名を指定する場合(他システムの
	 * 出力ファイルを読み取るケース等)では、行番号も存在しないケースがあり、
	 * その様な場合に、useNumber="false" を指定すれば、データの最初から読み取り始めます。
	 * この場合、出力データのカラムの並び順が変更された場合、columns 属性も
	 * 指定しなおす必要がありますので、できるだけ、#NAME 属性を使用するように
	 * してください。
	 * なお、EXCEL 入力には、この設定は適用されません。(暫定対応)
	 * 初期値は、true(使用する) です。
	 *
	 * @og.rev 3.7.0.5 (2005/04/11) 新規追加
	 *
	 * @param	useNumber	行番号情報 [true:使用している/false:していない]
	 */
	void setUseNumber( final boolean useNumber ) ;

	/**
	 * データの読み飛ばし件数を設定します。
	 *
	 * TAB区切りテキストやEXCEL等のデータの読み始めの初期値を指定します。
	 * ファイルの先頭行が、０行としてカウントしますので、設定値は、読み飛ばす
	 * 件数になります。(１と指定すると、１件読み飛ばし、２行目から読み込みます。)
	 * 読み飛ばしは、コメント行などは、無視しますので、実際の行数分読み飛ばします。
	 * ＃NAME属性や、columns 属性は、有効です。
	 *
	 * @og.rev 5.1.6.0 (2010/05/01) 新規作成
	 *
	 * @param	count 読み始めの初期値
	 */
	void setSkipRowCount( final int count ) ;

	/**
	 * ColumnActionListenerオブジェクトを設定します。
	 *
	 * ColumnActionListenerオブジェクトは、カラム名配列設定時と、それに対応する値配列設定時に
	 * 呼ばれるイベントリスナーです。
	 * 具体的なテーブル処理は、このイベントを使用して書き込みを行います。
	 *
	 * @og.rev 6.2.2.0 (2015/03/27) 新規作成
	 *
	 * @param	listener	ColumnActionListenerオブジェクト
	 */
	void setColumnActionListener( final ColumnActionListener listener ) ;

	/**
	 * デバッグ情報を出力するかどうかを指定します。
	 *
	 * EXCELなどを読み取る場合、シートマージで読み取ると、エラー時の行番号が、連番になるため、
	 * どのシートなのか、判らなくなります。
	 * そこで、どうしてもわからなくなった場合に備えて、デバッグ情報を出力できるようにします。
	 * 通常は使用しませんので、設定を無視します。
	 * 初期値は、false:デバッグ情報を出力しない です。
	 *
	 * @og.rev 5.5.7.2 (2012/10/09) 新規作成
	 *
	 * @param	useDebug	デバッグ情報を出力するかどうかを指定
	 */
	void setDebug( final boolean useDebug ) ;
}
