/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.fukurou.system;

// import java.util.Set;
// import java.util.LinkedHashSet;

import java.io.PrintStream ;
import java.io.PrintWriter ;

/**
 * 共通的に使用されるRuntimeExceptionクラスです。
 *
 * RuntimeException を継承しているため、try{} catch() {} は不要です。
 * 本システムでは、すべてこのエクセプションクラスを継承させたクラスを作成し、用途によって、
 * 使い分けるようにします。つまり、他のどのような、Throwable が発生したとしても、一旦、
 * try{} catch() {} で受けて、このクラスのサブクラスを、再度 throw させます。
 * そして、必要であれば、try{} catch() {} を用いて捕まえて、それぞれの対応処理を行います。
 *
 * このクラスには、元々の発生したエクセプション( Throwable )を引数にとり、
 * その printStackTrace()情報を、自分自身のトレース情報に含めます。
 * また、引数にオブジェクトを渡すことができますので、object.toString() で、オブジェクトの
 * 状態を表示できるようにしておけば、手軽にデバッグに使うことが可能になります。
 *
 * このRuntimeExceptionを使うと、ThrowUtil#ogStackTrace(String,Throwable) に準拠した
 * StackTrace を作成します。これは、最初の３行＋org.opengionを含む行だけのエラーを出力
 * しますので、エラーログの容量を大幅に抑えることが可能です。
 *
 * @og.group エラー処理
 * @og.rev 6.4.2.0 (2016/01/29) 新規作成
 *
 * @version  6.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK8.0,
 */
public class OgRuntimeException extends RuntimeException {
	private static final long serialVersionUID = 642020160129L ;

//	private static final int MAX_NORMAL_STACK = 3;				// 先頭から、通常にスタックトレースする行数。
//
//	private static final StackTraceElement ST_TR_EL = new StackTraceElement( "...", "", "", 0 );

	private final Throwable orgTh ;
	private       String    addMsg ;

	/**
	 *  詳細メッセージを指定しないで OgRuntimeException を構築します。
	 *
	 * @og.rev 6.4.2.0 (2016/01/29) 新規作成。
	 *
	 * @see		java.lang.RuntimeException#RuntimeException()
	 */
	public OgRuntimeException() {
		super();
		orgTh = null;
	}

	/**
	 *  指定された詳細メッセージを持つ OgRuntimeException を構築します。
	 *
	 * @og.rev 6.4.2.0 (2016/01/29) 新規作成。
	 *
	 * @param	msg	詳細メッセージ
	 * @see		java.lang.RuntimeException#RuntimeException(String)
	 */
	public OgRuntimeException( final String msg ) {
		super( msg );
		orgTh = null;
	}

	/**
	 *  指定されたThrowableオブジェクトを持つ OgRuntimeException を構築します。
	 *
	 * @og.rev 6.4.2.0 (2016/01/29) 新規作成。
	 *
	 * @param	th	例外Throwableオブジェクト
	 * @see		java.lang.RuntimeException#RuntimeException(Throwable)
	 */
	public OgRuntimeException( final Throwable th ) {
		super();
		orgTh = th ;
	}

	/**
	 *  指定された詳細メッセージと、Throwableオブジェクトを持つ OgRuntimeException を構築します。
	 *
	 * @og.rev 6.4.2.0 (2016/01/29) 新規作成。
	 *
	 * @param	msg	詳細メッセージ
	 * @param	th	例外Throwableオブジェクト
	 * @see		java.lang.RuntimeException#RuntimeException(String,Throwable)
	 */
	public OgRuntimeException( final String msg,final Throwable th ) {
//		super( msg,th );
		super( msg );
		orgTh = th ;
	}

	/**
	 *  指定された追加メッセージを、printStackTrace() 出力時に、合成します。
	 *
	 * 主に、openGionバージョンや、その他追記したいメッセージを登録することで、
	 * printStackTrace() 出力時に、合成できます。
	 * 複数のメッセージは登録できません。最後に登録したメッセージのみ使用されます。
	 * なお、null のメッセージは登録できません。
	 *
	 * @og.rev 6.4.2.0 (2016/01/29) 新規作成。
	 *
	 * @param	msg	追加メッセージ
	 */
	public void addMessage( final String msg ) {
		if( msg != null ) {
			addMsg = msg;
		}
	}

	/**
	 * このスロー可能オブジェクトおよびそのバックトレースを標準エラー・ストリームに出力します。
	 *
	 * ここのメソッドは、このThrowableオブジェクトのスタック・トレースを、System.errフィールドの
	 * 値であるエラー出力ストリームで出力します。出力の先頭行には、このオブジェクトに対する
	 * toString()メソッドの結果が含まれます。
	 * 残りの行は、以前にfillInStackTrace()メソッドによって記録されたデータを表します。
	 * この情報の書式は実装によって異なりますが、典型的な書式の例を次に示します。 
	 *
	 * @og.rev 6.4.2.0 (2016/01/29) 新規作成。
	 * @see		java.lang.RuntimeException#printStackTrace()
	 */
	@Override
	public void printStackTrace() {
		System.err.println( ThrowUtil.ogStackTrace( addMsg,orgTh ) );
	}

	/**
	 * このスロー可能オブジェクトとそのバックトレースを指定された印刷ストリームに出力します。
	 *
	 * @og.rev 6.4.2.0 (2016/01/29) 新規作成。
	 *
	 * @param	ps 出力する印刷ストリーム
	 * @see		java.lang.RuntimeException#printStackTrace(PrintStream)
	 */
	@Override
	public void printStackTrace( final PrintStream ps ) {
		ps.println( ThrowUtil.ogStackTrace( addMsg,orgTh ) );
	}

	/**
	 * このスロー可能オブジェクトとそのバックトレースを指定されたプリント・ライターに出力します。
	 *
	 * @og.rev 6.4.2.0 (2016/01/29) 新規作成。
	 *
	 * @param	pw 出力するプリント・ライター
	 * @see		java.lang.RuntimeException#printStackTrace(PrintWriter)
	 */
	@Override
	public void printStackTrace( final PrintWriter pw ) {
		pw.println( ThrowUtil.ogStackTrace( addMsg,orgTh ) );
	}

//	/**
//	 * printStackTrace()によって出力されるスタック・トレース情報にプログラムでアクセスできるようにします。
//	 *
//	 * この処理は、ThrowUtil#getStackTrace(Throwable) 経由で返される StackTraceElement 配列です。
//	 * 
//	 * 通常、スタック・トレース情報は、膨大なメッセージが含まれるため、その中の、"org.opengion" を
//	 * 含む箇所だけを、抜粋します。
//	 *
//	 * @og.rev 6.4.2.0 (2016/01/29) 新規作成。
//	 *
//	 * @return	このスロー可能オブジェクトに関するスタック・トレースを表す、スタック・トレース要素の配列。
//	 * @see		java.lang.RuntimeException#getStackTrace()
//	 * @see		org.opengion.fukurou.system.ThrowUtil#getStackTrace(Throwable)
//	 */
//	@Override
//	public StackTraceElement[] getStackTrace() {
//		return ThrowUtil.getStackTrace( orgTh );

//		final Set<StackTraceElement> steSet = new LinkedHashSet<>();		// 重複の取り除きと、スタック配列の順序の維持
//
//		final StackTraceElement[] stEles = orgTh == null || orgTh instanceof OgRuntimeException		// 自分自身を呼ぶと、ループしてしまうため。
//												? super.getStackTrace()
//												: orgTh.getStackTrace() ;
//
//		int idx = 0;
//		for( final StackTraceElement stEle : stEles ) {
//			if( idx++ < MAX_NORMAL_STACK ) {						// MAX_NORMAL_STACK 件までは無条件に出力する。
//				steSet.add( stEle );								// 同一エラー行は１度しか出さない。
//				continue;
//			}
//
//			final String cls = stEle.getClassName();
//			if( cls.contains( "org.opengion" ) ) {
//				steSet.add( stEle );								// 同一エラー行は１度しか出さない。
//			}
//			else {
//				steSet.add( ST_TR_EL );								// 重複が許されないので、最初に ... 出力されるだけ。
//			}
//		}
//
//		return steSet.toArray( new StackTraceElement[steSet.size()] );
//	}
}
