/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.db;

import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.resource.CodeData;
import static org.opengion.fukurou.system.HybsConst.CR ;				// 6.1.0.0 (2014/12/26)
import static org.opengion.fukurou.system.HybsConst.BUFFER_LARGE;		// 6.1.0.0 (2014/12/26) refactoring

/**
 * データのコード情報を取り扱うクラスです。
 *
 * コードのキーとラベルの情報から、HTMLのメニューやリストを作成するための オプション
 * タグを作成したり、与えられたキーをもとに、チェック済みのオプションタグを作成したり
 * します。
 *
 * @og.rev 6.2.2.4 (2015/04/24) 新規作成
 * @og.group 選択データ制御
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class Selection_BITBOX extends Selection_NULL {
	private final CodeData codeData ;

	/**
	 * コンストラクター
	 *
	 * @og.rev 6.2.2.4 (2015/04/24) 新規追加
	 *
	 * @param	cdData	コードデータオブジェクト
	 */
	public Selection_BITBOX( final CodeData cdData ) {
		super();		// 6.4.1.1 (2016/01/16) PMD refactoring. It is a good practice to call super() in a constructor
		if( cdData == null ) {
			final String errMsg = "コードリソースが定義されていません。" + CR ;
			throw new HybsSystemException( errMsg );
		}

		codeData = cdData ;
	}

	/**
	 * 初期値が選択済みの 選択肢(オプション)を返します。
	 * このオプションは、引数の値を初期値とするオプションタグを返します。
	 * ※ このクラスでは実装されていません。
	 *
	 * @og.rev 6.2.2.4 (2015/04/24) 新規追加
	 *
	 * @param	selectValue	選択されている値
	 * @param	seqFlag	シーケンスアクセス機能の指定
	 * @param	useShortLabel	短ラベルの指定
	 *
	 * @return  オプションタグ
	 */
	@Override
	public String getOption( final String selectValue,final boolean seqFlag, final boolean useShortLabel ) {
		final String errMsg = "このクラスでは実装されていません。";
		throw new UnsupportedOperationException( errMsg );
	}

	/**
	 * 初期値が選択済みの 選択肢(オプション)を返します。
	 * このオプションは、引数の値を初期値とするオプションタグを返します。
	 *
	 * @og.rev 6.2.2.4 (2015/04/24) 新規追加
	 * @og.rev 6.4.4.0 (2016/03/11) ラベルとラジオボタンの間に、全角スペースをひとつ入れます。
	 *
	 * @param   name         ラジオの name
	 * @param   selectValue  選択されている値
	 * @param   useLabel     ラベル表示の有無 [true:有/false:無]
	 *
	 * @return  オプションタグ
	 * @og.rtnNotNull
	 */
	@Override
	public String getOption( final String name,final String selectValue,final boolean useLabel ) {
		String value = null;
		try {
			final String inputTag = "<input type=\"checkbox\" name=\"" + name + "\" value=\"" ;
			final StringBuilder buf = new StringBuilder( BUFFER_LARGE );
			final int size = codeData.getSize();
			final int selVal = selectValue == null || selectValue.isEmpty() ? 0 : Integer.parseInt( selectValue );	// 数値化して、論理和の準備をする。
			for( int i=0; i<size; i++ ) {
				value = codeData.getCodeKey(i);
				final int val = value == null || value.isEmpty() ? 0 : Integer.parseInt( value );	// 数値化して、論理和の準備をする。
				if( useLabel ) { buf.append( "<label>" ); }
				buf.append( inputTag ).append( value ).append( '"' );	// 6.0.2.5 (2014/10/31) char を append する。
				if( ( val & selVal ) != 0 ) {							// 論理積
					buf.append( " checked=\"checked\"" );
				}
				buf.append( "/>" );
//				if( useLabel ) { buf.append( codeData.getShortLabel(i) ).append( "</label>" ); }
				if( useLabel ) { buf.append( codeData.getShortLabel(i) ).append( '　' ).append( "</label>" ); }
			}
			return buf.toString();
		}
		catch( final NumberFormatException ex ) {		// 文字列が解析可能な数値を含まない場合
			final String errMsg = "BITBOXで、数値変換できませんでした。 " + CR
								+ " name=[" + name + ", selectValue=[" + selectValue + "] , value=[" + value + "]" + CR
								+ ex.getMessage() ;
			return errMsg;
		}
	}

	/**
	 * 選択肢(value)に対するラベルを返します。
	 * 選択肢(value)が、存在しなかった場合は、選択肢そのものを返します。
	 * このメソッドでは、短縮ラベルを返すかどうかを指定するフラグを指定します。
	 * getValueLabel( XX,false ) は、getValueLabel( XX ) と同じです。
	 *
	 * @og.rev 6.2.2.4 (2015/04/24) 新規追加
	 *
	 * @param	selectValue	選択肢の値
	 * @param	isSLbl	短縮ラベルを [true:使用する/false:しない] (未使用)
	 *
	 * @return  選択肢のラベル
	 * @see     #getValueLabel( String )
	 */
	@Override
	public String getValueLabel( final String selectValue,final boolean isSLbl ) {
		String value = null;
		try {
			final StringBuilder buf = new StringBuilder( BUFFER_LARGE );
			final int size = codeData.getSize();
			// 6.4.2.1 (2016/02/05) PMD refactoring. Useless parentheses.
//			final int selVal = (selectValue == null || selectValue.isEmpty()) ? 0 : Integer.parseInt( selectValue );	// 数値化して、論理積の準備をする。
			final int selVal = selectValue == null || selectValue.isEmpty() ? 0 : Integer.parseInt( selectValue );		// 数値化して、論理積の準備をする。
			for( int i=0; i<size; i++ ) {
				value = codeData.getCodeKey(i);
				// 6.4.2.1 (2016/02/05) PMD refactoring. Useless parentheses.
//				final int val = (value == null || value.isEmpty()) ? 0 : Integer.parseInt( value );	// 数値化して、論理積の準備をする。
				final int val = value == null || value.isEmpty() ? 0 : Integer.parseInt( value );	// 数値化して、論理積の準備をする。
				if( ( val & selVal ) == 0 ) {							// 論理積
					buf.append( '□' );
				}
				else {
					buf.append( '■' );
				}

				if( isSLbl ) { buf.append( codeData.getShortLabel(i) ).append( '　' ); }
			}
			return buf.toString();
		}
		catch( final NumberFormatException ex ) {		// 文字列が解析可能な数値を含まない場合
			final String errMsg = "BITBOXで、数値変換できませんでした。 " + CR
								+ " selectValue=[" + selectValue + "] , value=[" + value + "]" + CR
								+ ex.getMessage() ;
			return errMsg;
		}
	}

}
