/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.db;

import org.opengion.hayabusa.common.HybsSystem;

/**
 * Queryオブジェクトを取得する為に使用する，ファクトリクラスです。
 *
 *  Queryオブジェクト の識別ID を元に、QueryFactory.newInstance( String id )
 * メソッドで，Queryオブジェクトを取得します。
 *
 * 実装とマッピングの関係から,識別ID は、システムパラメータ で 定義します
 *
 * @og.rev 3.6.0.8 (2004/11/19) キャッシュ()ではなく、オブジェクトを直接生成します。
 * @og.group データ表示
 * @og.group データ編集
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public final class QueryFactory {
	// 3.1.0.0 (2003/03/20) Hashtable を使用している箇所で、非同期でも構わない箇所を、HashMap に置換え。
	/** newInstance() 時のデフォルトクラス {@value} */
	public static final String DEFAULT  = "JDBC" ;

	/**
	 *  デフォルトコンストラクターをprivateにして、
	 *  オブジェクトの生成をさせないようにする。
	 *
	 */
	private QueryFactory() {
	}

	/**
	 * 標準的な Queryオブジェクト(JDBCQuery)を取得します。
	 * 過去に使用された Queryオブジェクト はプールから取得されます。
	 * ただし、内部変数はすべてクリアされますので、 一旦取り出した
	 * オブジェクトを保持したい場合は，各アプリケーション側で保持して下さい。
	 *
	 * @return  Queryオブジェクト
	 * @og.rtnNotNull
	 */
	public static Query newInstance() {
		return newInstance( DEFAULT );
	}

	/**
	 * 識別id に応じた Queryオブジェクトを取得します。
	 * ただし、内部変数はすべてクリアされますので、 一旦取り出した
	 * オブジェクトを保持したい場合は，各アプリケーション側で保持して下さい。
	 *
	 * @og.rev 3.6.0.8 (2004/11/19) キャッシュ廃止。直接生成します。
	 * @og.rev 4.0.0.0 (2005/01/31) キーの指定を、Query. から、Query_ に変更します。
	 * @og.rev 5.3.7.0 (2011/07/01) ゼロ文字列もDefaultを適用
	 * @og.rev 6.0.4.0 (2014/11/28) NullPointerException が発生するので、事前にチェックします。
	 * @og.rev 6.4.3.3 (2016/03/04) HybsSystem.newInstance(String,String) への置き換え。
	 *
	 * @param   id Queryインターフェースを実装したサブクラスの識別id
	 *
	 * @return  Queryオブジェクト
	 * @og.rtnNotNull
	 */
	public static Query newInstance( final String id ) {
		final String type = id == null || id.isEmpty() ? DEFAULT : id ;		// 6.4.2.1 (2016/02/05) PMD refactoring. Useless parentheses.

		return HybsSystem.newInstance( "Query_" , type );
	}
}
