/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */

package org.opengion.fukurou.queue;

import org.opengion.fukurou.util.StringUtil;

import com.amazonaws.auth.AWSCredentials;
import com.amazonaws.auth.AWSStaticCredentialsProvider;
import com.amazonaws.auth.BasicAWSCredentials;
import com.amazonaws.auth.InstanceProfileCredentialsProvider;
import com.amazonaws.regions.Regions;
import com.amazonaws.services.sqs.AmazonSQS;
import com.amazonaws.services.sqs.AmazonSQSClientBuilder;
import com.amazonaws.services.sqs.model.SendMessageRequest;

/**
 * SQSサーバへのメッセージキュー送信クラス
 * 
 *  SQSサーバへのメッセージキュー送信用のクラスです。
 * AmazonSQSへの送信が可能です。
 * 
 *  @og.group メッセージ連携
 *  
 *  @og.rev 5.10.14.0 (2019/08/01) 新規作成
 *  
 * @version 5
 * @author oota
 * @since JDK7
 */
public class QueueSend_SQS implements QueueSend {
	String sqsUrl = "";
	Boolean batch = false;
	AmazonSQS client;

	/**
	 * 接続処理
	 * SQSサーバに接続を行います。
	 * 
	 * @og.rev 5.10.15.0 (2019/08/30) 引数追加対応
	 *
	 * @param jmsServer 接続先url
	 * @param sqsAccessKey sqsアクセスキー
	 * @param sqsSecretKey sqsシークレットキー
	 */
	@Override
	public void connect(final String jmsServer, final String sqsAccessKey, final String sqsSecretKey) {
		sqsUrl = jmsServer;

		try {
				if (StringUtil.isNull(sqsAccessKey)) {
					// IAMロールによる認証
					client = AmazonSQSClientBuilder.standard()
							.withCredentials(new InstanceProfileCredentialsProvider(false)).build();
				} else {
					AWSCredentials credentials = new BasicAWSCredentials(sqsAccessKey, sqsSecretKey);
					
// proxy環境でのテスト用。proxyホストの情報を入力して、実行します。
//					ClientConfiguration conf = new ClientConfiguration();
//					conf.setProtocol(Protocol.HTTPS);
//					conf.setProxyHost("mtc-px14");
//					conf.setProxyPort(8081);
					
					client = AmazonSQSClientBuilder.standard()
							.withCredentials(new AWSStaticCredentialsProvider(credentials))
//							.withClientConfiguration(conf)
							.withRegion(Regions.AP_NORTHEAST_1.getName()).build();
				}
		} catch (Exception e) {
			throwErrMsg("SQSサーバの接続に失敗しました。" + e.getMessage());
		}
	}

	/**
	 * エラーメッセージ送信
	 * 
	 * @og.rev 5.10.15.0 (2019/08/30) hybs除外
	 * 
	 * @param errMsg エラーメッセージ
	 */
	public void throwErrMsg(final String errMsg) {
			throw new RuntimeException( errMsg );
	}

	/**
	 * メッセージ送信
	 * MQサーバにメッセージキューを送信します。
	 * 
	 * @param info 送信メッセージキュー情報 
	 */
	@Override
	public void sendMessage(QueueInfo info) {
		if(client == null) {
			throwErrMsg("SQSサーバに接続されていません。");
		}
		
		SendMessageRequest request = new SendMessageRequest(sqsUrl, info.getMessage());

		/** 情報設定 */
		// FIFOタイプのみ設定します。
		// グループID
		request.setMessageGroupId(info.getSqsFifoGroupId());
		// 重複禁止ID
		request.setMessageDeduplicationId(info.getSqsFifoDedupliId());

		/** 送信処理 */
		try {
			client.sendMessage(request);
		} catch (Exception e) {
			throwErrMsg("キューの送信処理に失敗しました。" + e.getMessage());
		}
	}

	/**
	 * クローズ処理
	 * SQSサーバとの接続をクローズします。
	 */
	@Override
	public void close() {
		if(client != null) {
			client.shutdown();
		}
	}

	@Override
	public void setBatchFlg(final Boolean batchFlg) {
		// SQSの場合は、バッチとWeb上で共通処理のため、フラグ設定の影響はありません
		batch = batchFlg;
	}

	/**
	 * テスト用メソッド
	 * proxy環境下で実行する場合は、
	 * connectメソッドのproxy設定のコメントを外して、
	 * 実行する必要があります。
	 * 
	 * @param args 引数
	 */
	public static void main(String[] args) {
		// 接続先
		String url = System.getProperty("URL");
		// グループID
		String groupId = System.getProperty("GROUPID");
		// アクセスキー
		String accessKey = System.getProperty("CLOUD_SQS_ACCESS_KEY");
		// シークレットキー
		String secretKey = System.getProperty("CLOUD_SQS_SECRET_KEY");
		// 送信メッセージ(乱数は重複排除IDのテスト用)
		String message = "サンプル送信メッセージ：" + Math.random();
		
		// SQSにメッセージ送信
		QueueSend queueSend = new QueueSend_SQS();
		
		// キュー情報の設定
		QueueInfo queueInfo = new QueueInfo();
		
		queueInfo.setSqsFifoGroupId(groupId);

		// メッセージ
		queueInfo.setMessage(message);
		
		// 接続
		queueSend.connect(url, accessKey, secretKey);
		// 送信
		queueSend.sendMessage(queueInfo);
		// クローズ
		queueSend.close();
	}
}
