/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.table;

import java.util.Calendar;									// 7.0.1.3 (2018/11/12)

import org.opengion.hayabusa.db.AbstractTableFilter;
import org.opengion.hayabusa.db.DBTableModel;

import org.opengion.fukurou.util.ErrorMessage;
import org.opengion.fukurou.util.StringUtil;
import org.opengion.fukurou.util.HybsDateUtil;				// 7.0.1.3 (2018/11/12)
import org.opengion.fukurou.system.DateSet;					// 7.0.1.3 (2018/11/12)

/**
 * TableFilter_KEY_BREAK は、TableFilter インターフェースを継承した、DBTableModel 処理用の
 * 実装クラスです。
 *
 * ここでは、指定のカラムに対して、キーブレイクが発生したときのデータ処理方法を指定できます。
 * 主として、グルーピング処理を行うのですが、ソートされデータの並び順で、キーブレイクするため、
 * 同一キーが存在していても、並び順が離れている場合、別のキーとしてブレイクします。
 *
 * GROUP_KEY  : キーブレイクの判定を行うカラムを、CSV形式で設定します。
 * OUT_TYPE   : 出力するデータのタイプを指定します。
 *                first : 最初のデータ(ブレイク直後のデータ)を出力します。(初期値)
 *                last  : 最後のデータ(ブレイク直前のデータ)を出力します。
 *                range : 最初のデータと最後のデータを出力します。
 * 
 * firstは、キーブレイク時のデータを残します。つまり、キーの最初に現れたデータです。
 * lastは、キーブレイクの直前のデータを残します。これは、同一キーの最後のデータということになります。
 * rangeは、firstと、last つまり、同値キーの最初と最後のデータを残します。
 * 
 * もし、キーが、1行だけの場合、firstも、lastも、同じ行を指すことになります。
 * その場合、rangeは、その1行だけになります(2行出力されません)。
 * 
 * 例：機種と日付と、状況Fがあったとして、日付、機種、状況F でソートし、機種をグループキー、
 * 状況Fをブレイクキーとすれば、日付の順に、機種の中で、状況Fがブレークしたときのみ、
 * データを残す、ということが可能になります。7.0.0.1 (2018/10/09) Delete
 *
 * OUT_TYPE に、lastか、range を指定した場合のみ、最大、最小、平均、中間、個数の集計処理が行えます。
 * これらの設定は、指定のカラムのデータに反映されます。
 * MIN_CLM  : キーブレイク時に、指定のカラムの最小値をデータに書き込みます。
 * MAX_CLM  : キーブレイク時に、指定のカラムの最大値をデータに書き込みます。
 * AVG_CLM  : キーブレイク時に、指定のカラムの平均値をデータに書き込みます。
 * MID_CLM  : キーブレイク時に、指定のカラムの最小値と最大値の中間の値をデータに書き込みます。
 * CNT_CLM  : キーブレイク時に、指定のカラムのデータ件数をデータに書き込みます。
 *
 * これらのカラムの値は、数値で表現できるもので無ければなりません。
 * 例えば、20180101000000 のような、日付でも数字のみなら、OKです。
 *
 * パラメータは、tableFilterタグの keys, vals にそれぞれ記述するか、BODY 部にCSS形式で記述します。
 *
 * @og.formSample
 * ●形式：
 *		① &lt;og:tableFilter classId="KEY_BREAK"
 *				keys="GROUP_KEY,OUT_TYPE"
 * 				vals='"CLM5,CLM6....",first' /&gt;
 *
 *		② &lt;og:tableFilter classId="KEY_BREAK" &gt;
 *				 {
 *					 GROUP_KEY  : CLM5,CLM6....	;
 *					 OUT_TYPE : first			;
 *				 }
 *		   &lt;/og:tableFilter&gt;
 *
 * @og.rev 6.7.9.1 (2017/05/19) 新規追加
 * @og.rev 7.0.0.1 (2018/10/09) ｸﾞﾙｰﾌﾟで、まとめる処理を止めます。
 * @og.rev 7.0.1.1 (2018/10/22) ﾛｼﾞｯｸ見直し
 *
 * @version  6.7  2017/05/19
 * @author   Kazuhiko Hasegawa
 * @since    JDK1.8,
 */
public class TableFilter_KEY_BREAK extends AbstractTableFilter {
	/** このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "7.0.1.3 (2018/11/12)" ;

	/**
	 * デフォルトコンストラクター
	 *
	 * @og.rev 6.7.9.1 (2017/05/19) 新規追加
	 * @og.rev 7.0.0.1 (2018/10/09) 最小,最大,平均,件数 を集計するためのｷｰﾜｰﾄﾞ追加
	 * @og.rev 7.0.1.3 (2018/11/12) MID_CLM(最小値と最大値の中間の値)のｷｰﾜｰﾄﾞ追加
	 */
	public TableFilter_KEY_BREAK() {
		super();
		initSet( "GROUP_KEY"	, "キーブレイクの判定を行うカラムを、CSV形式で設定します。" );
		initSet( "OUT_TYPE"	, "出力するデータのタイプを指定[first/last/range]を指定します。(初期値:first 最初のデータ)"	);
		initSet( "MIN_CLM"	, "キーブレイク時に、指定のカラムの最小値をデータに書き込みます。"	);
		initSet( "MAX_CLM"	, "キーブレイク時に、指定のカラムの最大値をデータに書き込みます。"	);
		initSet( "AVG_CLM"	, "キーブレイク時に、指定のカラムの平均値をデータに書き込みます。"	);
		initSet( "MID_CLM"	, "キーブレイク時に、指定のカラムの最小値と最大値の中間の値をデータに書き込みます。"	);
		initSet( "CNT_CLM"	, "キーブレイク時に、指定のカラムのデータ件数をデータに書き込みます。"	);
	}

	/**
	 * DBTableModel処理を実行します。
	 *
	 * @og.rev 6.7.9.1 (2017/05/19) 新規追加
	 * @og.rev 7.0.0.1 (2018/10/09) 最小,最大,平均,件数 を集計するためのｷｰﾜｰﾄﾞ追加
	 * @og.rev 7.0.1.1 (2018/10/22) ﾛｼﾞｯｸ見直し
	 * @og.rev 7.0.1.3 (2018/11/12) MID_CLM(最小値と最大値の中間の値)のｷｰﾜｰﾄﾞ追加
	 *
	 * @return 処理結果のDBTableModel
	 */
	public DBTableModel execute() {
		final DBTableModel table  = getDBTableModel();
		final DBTableModel rtnTbl = table.newModel();				// 削除ではなく、追加していきます。
		final int rowCnt = table.getRowCount();
		if( rowCnt == 0 ) { return rtnTbl; }						// 7.0.1.3 (2018/11/12) row<=rowCnt を追加したので、0件なら即終了

		final String[]	brkClms	= StringUtil.csv2Array( getValue( "GROUP_KEY" ) );

		// 7.0.0.1 (2018/10/09) 最小,最大,平均,件数 を集計するためのｷｰﾜｰﾄﾞ追加
		final String	outType = StringUtil.nval( getValue( "OUT_TYPE"  ), "first" ) ;

		final boolean	useFirst = "first".equalsIgnoreCase( outType ) || "range".equalsIgnoreCase( outType );	// firstかrange時に使用
		final boolean	useLast  = "last".equalsIgnoreCase( outType )  || "range".equalsIgnoreCase( outType ) ;	// lastかrange 時に使用

		// 7.0.0.1 (2018/10/09) 最小,最大,平均,件数 を集計するためのｷｰﾜｰﾄﾞ追加(useLast=true のときのみ使用)
		final int	minClmNo = table.getColumnNo( getValue( "MIN_CLM"  ), false ) ;	// カラムが存在しなければ、-1
		final int	maxClmNo = table.getColumnNo( getValue( "MAX_CLM"  ), false ) ;	// カラムが存在しなければ、-1
		final int	avgClmNo = table.getColumnNo( getValue( "AVG_CLM"  ), false ) ;	// カラムが存在しなければ、-1
		final int	midClmNo = table.getColumnNo( getValue( "MID_CLM"  ), false ) ;	// 7.0.1.3 (2018/11/12) カラムが存在しなければ、-1
		final int	cntClmNo = table.getColumnNo( getValue( "CNT_CLM"  ), false ) ;	// カラムが存在しなければ、-1

		final int[]    brkClmNo = new int[brkClms.length];			// ブレイクキーカラムの番号

		for( int i=0; i<brkClms.length; i++ ) {
			brkClmNo[i] = table.getColumnNo( brkClms[i],false );	// カラムが存在しなければ、-1
		}

		// 7.0.0.1 (2018/10/09) 最小,最大,平均,件数 を集計するためのｷｰﾜｰﾄﾞ追加(useLast=true のときのみ使用)
		double	minData  = Double.POSITIVE_INFINITY ;		// 仮数部の桁数の限界は15桁なので、日付型(14桁)は、処理できる。
		double	maxData  = Double.NEGATIVE_INFINITY ;
		double	total    = 0.0 ;
		int		cntData  = 0 ;		// 
		boolean	isLong	 = true;	// データに、少数点以下をつけるかどうかの判定です。
		double	midMin	 = Double.POSITIVE_INFINITY ;
		double	midMax	 = Double.NEGATIVE_INFINITY ;

		String oldBlkKeys = null;	// 前回ブレイクキーの値

//		final int rowCnt = table.getRowCount();

		String[] oldData = null;
		// 7.0.1.3 (2018/11/12) 最後のデータの処理を行うために、row<=rowCnt と1回余計に回します。
		for( int row=0; row<=rowCnt; row++ ) {
			final String[] data = row == rowCnt ? null : table.getValues( row );		// row<=rowCnt の影響
			try {
				final String brkKeys = getKeys( brkClmNo , data );		// ブレークキー(data==nullの場合、ゼロ文字列)
				if( !brkKeys.equalsIgnoreCase( oldBlkKeys ) ) {			// キーブレイク
					if( row>0 ) {
						if( minClmNo >= 0 ) { oldData[minClmNo] = isLong ? String.valueOf( Math.round( minData ) ) : String.valueOf( minData ) ; }
						if( maxClmNo >= 0 ) { oldData[maxClmNo] = isLong ? String.valueOf( Math.round( maxData ) ) : String.valueOf( maxData ) ; }
						if( avgClmNo >= 0 ) { oldData[avgClmNo] = String.format( "%.3f", total/cntData ); }
						if( midClmNo >= 0 ) { oldData[midClmNo] = getMiddle( midMin,midMax ); }
						if( cntClmNo >= 0 ) { oldData[cntClmNo] = String.valueOf( cntData ); }

						minData = Double.POSITIVE_INFINITY  ;
						maxData = Double.NEGATIVE_INFINITY  ;
						total   = 0.0 ;
						midMin  = Double.POSITIVE_INFINITY  ;
						midMax  = Double.NEGATIVE_INFINITY  ;

						if( useLast ) {
							// useFirst=true で、cntData == 1 の場合は、First行は削除します(1件を2件に増やさない)。
							if( useFirst ) {
								final int rCnt = rtnTbl.getRowCount();
								if( cntData == 1 ) {						// 1行しかない場合は、First行は削除します(1件を2件に増やさない)
									rtnTbl.removeValue( rCnt-1 );
								}
								else {
									final String[] fstData = rtnTbl.getValues( rCnt-1 );	// 前のデータ=First行に、最大、最小等のデータを反映させます。
									if( minClmNo >= 0 ) { fstData[minClmNo] = oldData[minClmNo]; }
									if( maxClmNo >= 0 ) { fstData[maxClmNo] = oldData[maxClmNo]; }
									if( avgClmNo >= 0 ) { fstData[avgClmNo] = oldData[avgClmNo]; }
									if( midClmNo >= 0 ) { fstData[midClmNo] = oldData[midClmNo]; }
									if( cntClmNo >= 0 ) { fstData[cntClmNo] = oldData[cntClmNo]; }
								}
							}

							rtnTbl.addColumnValues( oldData );			// ブレイクした一つ前＝最後のデータ
						}
						if( row == rowCnt ) { break; }					// 最後のデータの処理を行うために、row<=rowCnt と1回余計に回します。
					}

					if( useFirst ) {									// useLast=true で、cntData == 1 の場合は、登録しません
						rtnTbl.addColumnValues( data );					// ブレイク時のデータを登録します。
					}

					oldBlkKeys = brkKeys;
					cntData    = 0 ;
				}
				oldData = data;											// 一つ前のデータ
				cntData++;				// 毎回、カラムのある無しを判定するより、早そうなので常にカウントしておきます。

				// ブレイク時も集計処理は行います。
//				if( useLast ) {
					if( minClmNo >= 0 && !StringUtil.isNull( data[minClmNo] ) ) {
						if( isLong && data[minClmNo].indexOf( '.' ) >= 0 ) { isLong = false; }			// 一度、false になると、戻らない。
						minData = Math.min( minData, Double.parseDouble( data[minClmNo] ) );
					}
					if( maxClmNo >= 0 && !StringUtil.isNull( data[maxClmNo] ) ) {
						if( isLong && data[maxClmNo].indexOf( '.' ) >= 0 ) { isLong = false; }			// 一度、false になると、戻らない。
						maxData = Math.max( maxData, Double.parseDouble( data[maxClmNo] ) );
					}
					if( avgClmNo >= 0 && !StringUtil.isNull( data[avgClmNo] ) ) {
						total += Double.parseDouble( data[avgClmNo] );
					}
					if( midClmNo >= 0 && !StringUtil.isNull( data[midClmNo] ) ) {
						final double mid = Double.parseDouble( data[midClmNo] );
						midMin = Math.min( midMin, mid );
						midMax = Math.max( midMax, mid );
					}
//				}
			}
			catch( final RuntimeException ex ) {	// そのまま、継続して処理を行う。
				// 6.5.0.1 (2016/10/21) ErrorMessage をまとめるのと、直接 Throwable を渡します。
				makeErrorMessage( "TableFilter_KEY_BREAK Error",ErrorMessage.NG )
					.addMessage( row+1,ErrorMessage.NG,"KEY_BREAK" , StringUtil.array2csv( data ) )
					.addMessage( ex );
			}
		}
//		// 一番最後に書込みが行われないので。
//		if( useLast ) {
//			if( minClmNo >= 0 ) { oldData[minClmNo] = isLong ? String.valueOf( Math.round( minData ) ) : String.valueOf( minData ) ; }
//			if( maxClmNo >= 0 ) { oldData[maxClmNo] = isLong ? String.valueOf( Math.round( maxData ) ) : String.valueOf( maxData ) ; }
//			if( avgClmNo >= 0 ) { oldData[avgClmNo] = String.format( "%.3f", total/cntData ); }
//			if( midClmNo >= 0 ) { oldData[midClmNo] = getMiddle( midMin,midMax ); }
//			if( cntClmNo >= 0 ) { oldData[cntClmNo] = String.valueOf( cntData ); }
//
//			rtnTbl.addColumnValues( oldData );
//		}

		return rtnTbl;
	}

	/**
	 * 最小値と最大値の中間の値の文字列を作成します。
	 *
	 * 特殊系で、8桁か、14桁の場合、日付文字として中間の日付を求めます。
	 *
	 * @og.rev 7.0.1.3 (2018/11/12) MID_CLM(最小値と最大値の中間の値)のｷｰﾜｰﾄﾞ追加
	 *
	 * @param min 最小値
	 * @param max 最大値
	 * @return 中間の値の文字列
	 */
	private String getMiddle( final double min , final double max ) {
		final String minStr = String.valueOf( Math.round( min ) );		// 14桁の場合、2.0181103000000E13 見たいな表記になるため。
		final String maxStr = String.valueOf( Math.round( max ) );
		final int minLen = minStr.length();

		final String midStr ;
		if( minLen == maxStr.length() && ( minLen == 8 || minLen == 14 ) ) {
			final Calendar minCal = HybsDateUtil.getCalendar( minStr );
			final Calendar maxCal = HybsDateUtil.getCalendar( maxStr );
			final long     midTim = ( maxCal.getTimeInMillis() + minCal.getTimeInMillis() ) / 2 ;

			if( minLen == 8 ) {
				midStr = DateSet.getDate( midTim , "yyyyMMdd" );
			}
			else {	// 14桁しかありえない
				midStr = DateSet.getDate( midTim , "yyyyMMddHHmmss" );
			}
		}
		else {
			midStr = String.format( "%.3f", ( max + min ) / 2.0 );		// 日付型でなければ、minStr,maxStr は使わないので。
		}

		return midStr;
	}

	/**
	 * キーの配列ｱﾄﾞﾚｽと、１行分のデータ配列から、キーとなる文字列を作成します。
	 *
	 * @og.rev 6.7.9.1 (2017/05/19) 新規追加
	 * @og.rev 7.0.1.3 (2018/11/12) 最後のデータの処理を行うために、row<=rowCnt と1回余計に回す対応
	 *
	 * @param clms キーの配列ｱﾄﾞﾚｽ
	 * @param rowData １行分のデータ配列
	 * @return キーとなる文字列
	 */
	private String getKeys( final int[] clms , final String[] rowData ) {
		if( rowData == null ) { return ""; }					// rowData がnull の場合は、キーブレイクとなる

		final StringBuilder buf = new StringBuilder();
		for( int i=0; i<clms.length; i++ ) {
			if( clms[i] >= 0 ) {
				buf.append( rowData[clms[i]] ).append( ':' );
			}
		}
		return buf.toString();
	}
}
