/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.io;

import java.io.PrintWriter;

import org.opengion.fukurou.util.StringUtil;
import org.opengion.hayabusa.db.DBTableModel;

/**
 * 加工なしダブルクォート区切り文字指定データの書き出しクラスです。
 *
 * DefaultTableWriter を継承して，データの出力部のみオーバーライドして，
 * データそのものを加工なしで、ダブルコーテーションで処理して出力します。
 * 本来は、DefaultTableWriter の出力形態のはずですが、過去の互換性との関係で、
 * なまデータを出力するクラスを、追加作成しました。
 * 従来の CSV2 は、このクラスを使用してください。
 *
 * @og.rev 3.7.0.3 (2005/03/01) 新規作成
 * @og.group ファイル出力
 *
 * @version  4.0
 * @author	 Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class TableWriter_Data2 extends TableWriter_Default {
	/** このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "6.4.2.0 (2016/01/29)" ;

	/**
	 * デフォルトコンストラクター
	 *
	 * @og.rev 6.4.2.0 (2016/01/29) PMD refactoring. Each class should declare at least one constructor.
	 */
	public TableWriter_Data2() { super(); }		// これも、自動的に呼ばれるが、空のメソッドを作成すると警告されるので、明示的にしておきます。

	/**
	 * PrintWriter に DBTableModelのテーブル情報を書き込みます。
	 *
	 * @og.rev 3.7.0.3 (2005/03/01) 新規作成
	 * @og.rev 3.8.0.1 (2005/06/17) DBTypeが NVAR の場合は、元のUnicodeに戻します。
	 * @og.rev 5.1.6.0 (2010/05/01) DbType の初期値(dbType)を利用する。
	 * @og.rev 5.2.1.0 (2010/10/01) useRenderer 対応
	 * @og.rev 6.0.1.2 (2014/08/08) カラム飛ばしできる機能を追加
	 * @og.rev 6.0.4.0 (2014/11/28) データ出力用のレンデラー
	 *
	 * @param	table  DBTableModelオブジェクト
	 * @param	writer PrintWriterオブジェクト
	 */
	@Override
	protected void writeData( final DBTableModel table,final PrintWriter writer ) {
		final int numberOfRows = table.getRowCount();
		final String separator = getSeparator();
		final boolean useNumber = isUseNumber();
		final boolean useRenderer = isUseRenderer();	// 5.2.1.0 (2010/10/01)

		for( int row=0; row<numberOfRows; row++ ) {
			if( useNumber ) {
				writer.print( quotation( String.valueOf( row+1 ) ) );
				writer.print( separator );
			}

			for( int i=0; i<numberOfColumns; i++ ) {
				if( i != 0 ) { writer.print( separator ); }
				final int clm = clmNo[i];
				if( clm < 0 ) { continue; }			// 6.0.1.2 (2014/08/08) カラム飛ばし

				String val = table.getValue(row,clm);
				if( dbType[i] == NVAR ) {
					val = StringUtil.getReplaceEscape( val );
				}
				// 5.2.1.0 (2010/10/01) useRenderer 対応
				else if( useRenderer ) {
					// 6.0.4.0 (2014/11/28) データ出力用のレンデラー
					val = dbColumn[clm].getWriteValue( val );
				}

				writer.print( quotation( val ) );
			}
			writer.println();
		}
	}
}
