/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.db;

import java.io.File;
import java.io.FileFilter;
import java.util.Arrays;
import java.nio.file.FileSystems;
import java.nio.file.PathMatcher;

import org.opengion.fukurou.system.LogWriter;
// import org.opengion.fukurou.util.StringUtil;
// import static org.opengion.fukurou.system.HybsConst.CR ;
import static org.opengion.fukurou.system.HybsConst.BUFFER_MIDDLE;

/**
 * ファイルのプルダウンリストの作成するクラスです。
 *
 * ファイルの一覧リストからプルダウンリストを作成します。
 * オプションタグを作成したり、与えられたキーをもとに、チェック済みのオプションタグを
 * 作成したりします。
 *
 * @og.rev 7.2.4.0 (2020/05/11) 新規追加
 * @og.group 選択データ制御
 *
 * @version  7.1
 * @author   Kazuhiko Hasegawa
 * @since    JDK11.0,
 */
public class Selection_FILES extends Selection_NULL {
	private final String	CACHE ;

	/**
	 * コンストラクター
	 *
	 * @og.rev 7.2.4.0 (2020/05/11) 新規追加
	 *
	 * @param	param	パラメータ文字列(;で条件を区切ります)
	 */
	public Selection_FILES( final String param ) {
		super();

		final String[] prms = param.split(";");
		final String   from = prms[0];
		final boolean  nameOnly = prms.length > 1 && "nameOnly".equalsIgnoreCase( prms[1] ) ;

		final PathMatcher match = prms.length > 2 ? FileSystems.getDefault().getPathMatcher(prms[2]) : null;
		final FileFilter PATH_FLTR = file -> {
	 		return match == null || match.matches( file.toPath().getFileName() ) ;
		};

		final File[] list = new File( from ).listFiles( PATH_FLTR );
		Arrays.sort( list );

		final StringBuilder buf = new StringBuilder( BUFFER_MIDDLE );
		for( int i=0; i<list.length; i++ ) {
			String value = list[i].getName();
			if( nameOnly ) {
				final int ad = value.lastIndexOf( '.' );
				if( ad >= 0 ) { value = value.substring( 0,ad ); }
			}

			buf.append( "<option value=\"" ).append( value )
				.append( "\">" ).append( value ).append( "</option>" );
		}

		CACHE = buf.toString();
	}

	/**
	 * 初期値が選択済みの 選択肢(オプション)を返します。
	 * このオプションは、引数の値を初期値とするオプションタグを返します。
	 * このクラスでは、useShortLabel は、無視されます。(常に、false です)
	 *
	 * @og.rev 7.2.4.0 (2020/05/11) 新規追加
	 *
	 * @param   selectValue  選択されている値
	 * @param   seqFlag  シーケンスアクセス機能 [true:ON/false:OFF]
	 * @param   useShortLabel ラベル(短)をベースとしたオプション表示を行うかどうか(常にfalse)。
	 *
	 * @return  オプションタグ
	 * @og.rtnNotNull
	 */
	@Override
	public String getOption( final String selectValue,final boolean seqFlag, final boolean useShortLabel ) {
		// マッチするアドレスを探す。キーの前後のダブルクオートを加味して検索
		final String selVal = "\"" + selectValue + "\"" ;

		final int indx = CACHE.indexOf( selVal );

		if( indx < 0 ) {
			if( selectValue != null && selectValue.length() > 0 ) {
				final String errMsg = "コードに存在しない値が指定されました。"
							+ " value=[" + selectValue + "]" ;
				LogWriter.log( errMsg );
			}
			return CACHE;
		}
		else {
			final int addIndx = indx + selVal.length() ;	// selected の挿入位置

			final StringBuilder buf = new StringBuilder( BUFFER_MIDDLE );

			if( seqFlag ) {
				buf.append( "<option value=\"" ).append( selectValue ).append( '"' );		// 6.0.2.5 (2014/10/31) char を append する。
			}
			else {
				buf.append( CACHE.substring( 0,addIndx ) );
			}
			buf.append( " selected=\"selected\"" )
				.append( CACHE.substring( addIndx ) );
			return buf.toString() ;
		}
	}

	/**
	 * 選択肢(value)に対するラベルを返します。
	 * 選択肢(value)が、存在しなかった場合は、選択肢そのものを返します。
	 * getValueLabel( XX,false ) は、getValueLabel( XX ) と同じです。
	 *
	 * @og.rev 7.2.4.0 (2020/05/11) 新規追加
	 *
	 * @param	selectValue	選択肢の値
	 * @param	isSLbl	短縮ラベルを [true:使用する/false:しない](常に false)
	 *
	 * @return  選択肢のラベル
	 * @see     #getValueLabel( String )
	 */
	@Override
	public String getValueLabel( final String selectValue,final boolean isSLbl ) {
		// マッチするアドレスを探す。キーの前後のダブルクオートを加味して検索
		final String selVal = "\"" + selectValue + "\"" ;

		final int indx = CACHE.indexOf( selVal );

		if( indx < 0 ) {
			// マッチしなければ、選択肢そのものを返す。
			return selectValue;
		}
		else {
			// マッチすれば、キー以下のBODY部の文字列を切り出して返す。
			final int stIdx = indx + selVal.length() + 1 ;			// ＋１ は、">" の位置
			final int edIdx = CACHE.indexOf( '<',stIdx );				// 終了アドレス

			return CACHE.substring( stIdx,edIdx );
		}
	}
}
