/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.fukurou.process;

import org.opengion.fukurou.system.OgRuntimeException ;		// 6.4.2.0 (2016/01/29)
import org.opengion.fukurou.util.Argument;
import org.opengion.fukurou.util.HybsFileFilter;
import org.opengion.fukurou.system.LogWriter;

import java.io.File;
import java.util.Map ;
import java.util.LinkedHashMap ;
import java.util.Stack;

/**
 * Process_FileSearch は、指定のフォルダ以下のファイルを一覧する、FirstProcess
 * インターフェースと、ChainProcess インターフェースの実装クラスです。
 *
 * Fileﾘｽﾄは、FileLineModel オブジェクトを使用します。
 * ファイル属性(Level,File,Length,Modify,LineCnt,Biko,MD5) と、
 * オプションで、FILEPATH,ADDRESS,FILENAME 属性を文字列で準備できます。
 *
 * 指定の条件に合致するファイルを検索し、LineModel のサブクラスである、
 * FileLineModel オブジェクトを作成して、下流に渡します。
 * FileLineModel オブジェクトには、ファイル属性(Level,File,Length,Modify)
 * が設定されます。
 *
 * 引数文字列中にスペースを含む場合は、ダブルコーテーション("") で括って下さい。
 * 引数文字列の 『=』の前後には、スペースは挟めません。必ず、-key=value の様に
 * 繋げてください。
 *
 * ※ 6.3.1.1 (2015/07/10)
 *    ignoreCase の導入と同時に、初期値を、Grep , GrepChange , GrepChangeExcel と『異なる』
 *    true:区別しない にしました。
 *    これは、混乱する可能性が高いのですが、ファイル関連の検索において、通常は、大文字小文字の
 *    区別なしで検索するのが多く、Grep の区別するケースとは異なると判断しました。
 *    実際は、ignoreCase 属性を毎回記述したくないというのが、本当の所です。
 *
 * ※ 6.4.0.2 (2015/12/11)
 *    prefix,suffix,instr,equals に、("|"で複数指定可) の説明をJavaDocに追加。
 *    useDIR 属性に、onlyパラメータを追加[false:File/true:File+Dir/only:Dir]。
 * ※ 7.2.6.0 (2020/06/30)
 *    useDIR 属性は廃止。
 *
 * @og.formSample
 *  Process_FileSearch -start=d:/ -suffix=jsp
 *
 *     -start=開始フォルダ         ：検索を開始するフォルダ
 *   [ -ignoreCase=[true/false]  ] ：検索時に大文字小文字を区別しないかどうか(初期値:区別しない[true])
 *   [ -prefix=接頭辞            ] ：File････,View････,など、指定の接頭辞で始まるファイルを検索("|"で複数指定可)
 *   [ -unprefix=不接頭辞        ] ：File････,View････,など、指定の接頭辞で始まらないファイルを検索("|"で複数指定可)
 *   [ -preDir=接頭辞ﾌｫﾙﾀﾞ       ] ：(追加：7.2.6.0 (2020/06/30))File････,View････,など、指定の接頭辞で始まるフォルダを検索("|"で複数指定可)			// 6.4.3.2 (2016/02/19)
 *   [ -unpreDir=不接頭辞ﾌｫﾙﾀﾞ   ] ：(追加：7.2.6.0 (2020/06/30))File････,View････,など、指定の接頭辞で始まらないフォルダを検索("|"で複数指定可)		// 7.2.6.0 (2020/06/30)
 *   [ -suffix=接尾辞            ] ：.txt|.java|.jsp.... など、指定の接尾辞で終わるファイルを検索("|"で複数指定可)
 *   [ -unsuffix=不接尾辞        ] ：.txt|.java|.jsp.... など、指定の接尾辞で終わらないファイルを検索("|"で複数指定可)
 *   [ -instr=部分文字列         ] ：ファイル名と一致する部分文字列を指定("|"で複数指定可)
 *   [ -uninstr=不部分文字列     ] ：ファイル名と一致しな部分文字列を指定("|"で複数指定可)
 *   [ -instrDir=部分文字列      ] ：(追加：7.2.6.0 (2020/06/30))フォルダと一致する部分文字列を指定("|"で複数指定可)									// 7.2.6.0 (2020/06/30)
 *   [ -uninstrDir=不部分文字列  ] ：(追加：7.2.6.0 (2020/06/30))フォルダと一致しな部分文字列を指定("|"で複数指定可)									// 7.2.6.0 (2020/06/30)
 *   [ -equals=一致         廃止 ] ：ファイル名と一致する文字列を指定("|"で複数指定可)    7.2.7.0 (2020/08/07) 廃止
 *   [ -notequals=不一致    廃止 ] ：ファイル名と一致しない文字列を指定("|"で複数指定可)  7.2.7.0 (2020/08/07) 廃止
 *   [ -fileEequals=一致         ] ：ファイル名と一致する文字列を指定("|"で複数指定可)    7.2.7.0 (2020/08/07) 追加
 *   [ -unFileEquals=不一致      ] ：ファイル名と一致しない文字列を指定("|"で複数指定可)  7.2.7.0 (2020/08/07) 追加
 *   [ -match=正規表現           ] ：ファイル名と一致する正規表現を指定
 *   [ -unmatch=正規表現         ] ：ファイル名と一致しない正規表現を指定
 *   [ -matchDir=正規表現        ] ：(追加：7.2.6.0 (2020/06/30))フォルダと一致する正規表現を指定														// 7.2.6.0 (2020/06/30)
 *   [ -unmatchDir=不一致ﾌｫﾙﾀﾞ   ] ：(追加：7.2.6.0 (2020/06/30))フォルダと一致しない正規表現を指定														// 7.2.6.0 (2020/06/30)
 *   [ -modify=YYYYMMDD          ] ：指定日付け以降に変更されたファイルを検索
 *             YYYYMMDD   : YYYYMMDD 形式での指定日の 00:00:00 を基準時刻
 *             TODAY      : 実行日の 00:00:00 を基準時刻
 *             YESTERDAY  : 実行日前日の 00:00:00 を基準時刻
 *             LAST_WEEK  : 実行日の先週(7日前) 00:00:00 を基準時刻
 *             MONTH      : 実行月の 1日 00:00:00 を基準時刻
 *             LAST_MONTH : 実行前月の 同日 00:00:00 を基準時刻
 *             LAST_YEAR  : 実行前年の 同月同日 00:00:00 を基準時刻
 *   [ -unmodify=YYYYMMDD        ] ：(追加：7.2.6.0 (2020/06/30))指定日付け以前に変更されたファイルを検索(形式はmodifyと同一)
 *   [ -useDIR=[false/true/only] ] ：(廃止：7.2.6.0 (2020/06/30))判定をファイルだけでなく、ディレクトリでも行うかどうかを指定[false:File/true:File+Dir/only:Dir](初期値:false)
 *   [ -larger=サイズ(Byte)      ] ：ファイルの大きさが指定のバイト数と同じか大きいファイルを検索
 *   [ -smaller=サイズ(Byte)     ] ：ファイルの大きさが指定のバイト数より小さいファイルを検索
 *   [ -isHidden=[true/false]    ] ：true:HIDDENファイルのみ検索/false:NORMALファイルのみ検索(初期値:null)
 *   [ -maxLevel=最大階層数      ] ：ディレクトリの階層を下がる最大数(初期値:256)
 *   [ -useLineCnt=行数計算      ] ：ファイルの行数をカウントするかどうかを指定(初期値:false)
 *   [ -useMD5=MD5計算値         ] ：ファイルのMD5計算を行うかどうかを指定(初期値:false)
 *   [ -useOmitCmnt=[false/true] ] ：コメント部分を削除した行数と文字数計算を行うかどうかを指定(初期値:false)
 *   [ -useFilePath=[false/true] ] ：FILEPATH,ADDRESS,FILENAME 属性を文字列で準備します(初期値:false)
 *   [ -modifyForm=日付ﾌｫｰﾏｯﾄ    ] ：MODIFY 属性のDate型から文字列に変換するフォーマットを指定します(初期値:null)
 *   [ -encode=エンコード名      ] ：コメント削除時の文字数計算で利用するファイルのエンコード(初期値:JISAutoDetect)
 *   [ -inPath=入力共通パス      ] ：BIKO作成用のファイルパスから削除する部分(文字数のみ)
 *   [ -outPath=出力追加パス     ] ：BIKO作成用のファイルパスに追加する部分
 *   [ -errAbend=[true/false]    ] ：異常発生時に、処理を中断(true)するか、継続(false)するかを指定する(初期値:true[中断する])
 *   [ -display=[false/true]     ] ：trueは、検索状況を表示します(初期値:false)
 *   [ -debug=[false/true]       ] ：デバッグ情報を標準出力に表示する(true)かしない(false)か(初期値:false[表示しない])
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class Process_FileSearch extends AbstractProcess implements FirstProcess , ChainProcess {

	private Stack<FileListStack> dirs	;
	private File			file		;
	private HybsFileFilter	filter		;
	private FileLineModel	newData		;
	private int				level		= 1;

	private String			startDir	;
	private int				maxLevel	= 256;
	private int				inCount		;
	private int				outCount	;
	private int				inPathLen	;		// 4.2.3.0 (2008/05/26) BIKO欄用
	private String			outPath		;		// 4.3.1.1 (2008/08/23) BIKO欄用
	private boolean			errAbend	= true;	// 6.3.1.0 (2015/06/28) 中断する
	private boolean			display		;		// false:表示しない
	private boolean			debug		;		// 5.7.3.0 (2014/02/07) デバッグ情報

	/** staticイニシャライザ後、読み取り専用にするので、ConcurrentHashMap を使用しません。 */
	private static final Map<String,String> MUST_PROPARTY   ;		// ［プロパティ］必須チェック用 Map
	/** staticイニシャライザ後、読み取り専用にするので、ConcurrentHashMap を使用しません。 */
	private static final Map<String,String> USABLE_PROPARTY ;		// ［プロパティ］整合性チェック Map

	static {
		MUST_PROPARTY = new LinkedHashMap<>();
		MUST_PROPARTY.put( "start",	"検索を開始するフォルダ(必須)" );

		USABLE_PROPARTY = new LinkedHashMap<>();
		USABLE_PROPARTY.put( "ignoreCase"	, "大文字小文字を区別しないかどうか(初期値:区別しない[true])" );		// 6.3.1.1 (2015/07/10)
		USABLE_PROPARTY.put( "preDir"		, "File････,View････,など、指定の接頭辞で始まるフォルダを検索" );		// 6.4.3.2 (2016/02/19)
		USABLE_PROPARTY.put( "unpreDir"		, "File････,View････,など、指定の接頭辞で始まらないフォルダを検索" );	// 7.2.6.0 (2020/06/30)
		USABLE_PROPARTY.put( "prefix"		, "File････,View････,など、指定の接頭辞で始まるファイルを検索" );
		USABLE_PROPARTY.put( "unprefix"		, "File････,View････,など、指定の接頭辞で始まらないファイルを検索" );
		USABLE_PROPARTY.put( "suffix"		, ".txt|.java|.jsp.... など、指定の接尾辞で終わるファイルを検索" );
		USABLE_PROPARTY.put( "unsuffix"		, ".txt|.java|.jsp.... など、指定の接尾辞で終わらないファイルを検索" );
		USABLE_PROPARTY.put( "instr"		, "ファイル名と一致する部分文字列を指定" );
		USABLE_PROPARTY.put( "uninstr"		, "ファイル名と一致しない部分文字列を指定" );
		USABLE_PROPARTY.put( "instrDir"		, "フォルダと一致する部分文字列を指定" );								// 7.2.6.0 (2020/06/30)
		USABLE_PROPARTY.put( "uninstrDir"	, "フォルダと一致しない部分文字列を指定" );								// 7.2.6.0 (2020/06/30)
//		USABLE_PROPARTY.put( "equals"		, "ファイル名と一致する文字列を指定" );									// 7.2.7.0 (2020/08/07) 廃止
//		USABLE_PROPARTY.put( "notequals"	, "ファイル名と一致しない文字列を指定" );								// 7.2.7.0 (2020/08/07) 廃止
		USABLE_PROPARTY.put( "fileEquals"	, "ファイル名と一致する文字列を指定" ); 								// 7.2.7.0 (2020/08/07) 追加
		USABLE_PROPARTY.put( "unFileEquals"	, "ファイル名と一致しない文字列を指定" );								// 7.2.7.0 (2020/08/07) 追加
		USABLE_PROPARTY.put( "match"		, "ファイル名と一致する正規表現を指定" );
		USABLE_PROPARTY.put( "unmatch"		, "ファイル名と一致しない正規表現を指定" );
		USABLE_PROPARTY.put( "matchDir"		, "フォルダと一致する正規表現を指定" );									// 7.2.6.0 (2020/06/30)
		USABLE_PROPARTY.put( "unmatchDir"	, "フォルダと一致しない正規表現を指定" );								// 7.2.6.0 (2020/06/30)
		USABLE_PROPARTY.put( "modify"		, "指定日付け以降に変更されたファイルを検索" +
										CR + "YYYYMMDD   : YYYYMMDD 形式での指定日の 00:00:00 を基準時刻" +
										CR + "TODAY      : 実行日の 00:00:00 を基準時刻" +
										CR + "YESTERDAY  : 実行日前日の 00:00:00 を基準時刻" +
										CR + "LAST_WEEK  : 実行日の先週(7日前) 00:00:00 を基準時刻" +
										CR + "MONTH      : 実行月の 1日 00:00:00 を基準時刻" +
										CR + "LAST_MONTH : 実行前月の 同日 00:00:00 を基準時刻" +
										CR + "LAST_YEAR  : 実行前年の 同月同日 00:00:00 を基準時刻"
		);
		USABLE_PROPARTY.put( "unmodify"		, "指定日付け以前に変更されたファイルを検索" );							// 7.2.6.0 (2020/06/30)
//		USABLE_PROPARTY.put( "useDIR"		, "判定をファイルだけでなく、ディレクトリでも行うかどうかを指定[false:File/true:File+Dir/only:Dir](初期値:false)" );	// 7.2.6.0 (2020/06/30) 廃止
		USABLE_PROPARTY.put( "larger"		, "ファイルの大きさが指定のバイト数と同じか大きいファイルを検索" );
		USABLE_PROPARTY.put( "smaller"		, "ファイルの大きさが指定のバイト数より小さいファイルを検索" );
		USABLE_PROPARTY.put( "isHidden"		, "true:HIDDENファイルのみ検索/false:NORMALファイルのみ検索(初期値:null)" );
		USABLE_PROPARTY.put( "maxLevel"		, "ディレクトリの階層を下がる最大数(初期値:256)" );
		USABLE_PROPARTY.put( "useLineCnt"	, "ファイルの行数をカウントするかどうかを指定(初期値:false)" );
		USABLE_PROPARTY.put( "useMD5"		, "ファイルのMD5計算を行うかどうかを指定(初期値:false)" );		// 5.7.2.1 (2014/01/17)
		USABLE_PROPARTY.put( "useOmitCmnt"	, "コメント部分を削除した行数と文字数計算を行うかどうかを指定(初期値:false)" );		// 5.7.4.0 (2014/03/07)
		USABLE_PROPARTY.put( "useFilePath"	, "FILEPATH,ADDRESS,FILENAME 属性を文字列で準備します(初期値:false)" );				// 6.3.1.0 (2015/06/28)
		USABLE_PROPARTY.put( "modifyForm"	, "MODIFY 属性のDate型から文字列に変換するフォーマットを指定します(初期値:null)" );	// 6.3.1.1 (2015/07/10)
		USABLE_PROPARTY.put( "encode"		, "コメント削除時の文字数計算で利用するファイルのエンコード(初期値:JISAutoDetect)" );	// 5.7.4.0 (2014/03/07)
		USABLE_PROPARTY.put( "inPath"		, "BIKO作成用のファイルパスから削除する部分(文字数のみ)" );
		USABLE_PROPARTY.put( "outPath"		, "BIKO作成用のファイルパスに追加する部分" );
		USABLE_PROPARTY.put( "errAbend"		, "異常発生時に、処理を中断(true)するか、継続(false)するか" +
										CR + "(初期値:true:中断する)" );			// 6.3.1.0 (2015/06/28)
		USABLE_PROPARTY.put( "display"		, "trueは、検索状況を表示します(初期値:false)" );
		USABLE_PROPARTY.put( "debug"		, "デバッグ情報を標準出力に表示する(true)かしない(false)か" +
										CR + "(初期値:false:表示しない)" );			// 5.7.3.0 (2014/02/07) デバッグ情報
	}

	/**
	 * デフォルトコンストラクター。
	 * このクラスは、動的作成されます。デフォルトコンストラクターで、
	 * super クラスに対して、必要な初期化を行っておきます。
	 *
	 */
	public Process_FileSearch() {
		super( "org.opengion.fukurou.process.Process_FileSearch",MUST_PROPARTY,USABLE_PROPARTY );
	}

	/**
	 * プロセスの初期化を行います。初めに一度だけ、呼び出されます。
	 * 初期処理(ファイルオープン、ＤＢオープン等)に使用します。
	 *
	 * @og.rev 4.2.2.0 (2008/05/10) 行数カウントの使用有無
	 * @og.rev 4.3.1.1 (2008/08/23) BIKO 欄にoutPath 属性を追加します。
	 * @og.rev 5.1.2.0 (2010/01/01) useDIR 属性を追加します。
	 * @og.rev 5.7.2.1 (2014/01/17) useMD5 属性を追加します。
	 * @og.rev 5.7.4.0 (2014/03/07) useOmitCmnt,encode 属性を追加します。
	 * @og.rev 5.7.4.3 (2014/03/28) larger,smaller属性を文字列に変更
	 * @og.rev 5.7.5.0 (2014/04/04) isHidden属性を追加します。
	 * @og.rev 6.3.1.0 (2015/06/28) errAbend属性追加。
	 * @og.rev 6.3.1.0 (2015/06/28) FILEPATH,ADDRESS,FILENAME 属性追加
	 * @og.rev 6.3.1.1 (2015/07/10) メソッドチェーン化と、大文字/小文字の区別なし(ignoreCase=true)対応
	 * @og.rev 6.3.1.1 (2015/07/10) Modify のフォーマットを、指定可能にします。
	 * @og.rev 6.4.0.2 (2015/12/11) useDIR 属性に、onlyパラメータを追加します[false:File/true:File+Dir/only:Dir]。
	 * @og.rev 7.2.6.0 (2020/06/30) ディレクトリ処理を統一します。
	 * @og.rev 7.2.7.0 (2020/08/07) equals,notequals 廃止、fileEquals,unFileEquals 追加
	 *
	 * @param   paramProcess データベースの接続先情報などを持っているオブジェクト
	 */
	public void init( final ParamProcess paramProcess ) {
		final Argument arg = getArgument();

		startDir = arg.getProparty( "start" );

		final String inPath	= arg.getProparty( "inPath" );
		if( inPath != null ) { inPathLen = inPath.length(); }

		final boolean ignoreCase= arg.getProparty( "ignoreCase"	,true	);	// 6.3.1.1 (2015/07/10)
		final String preDir		= arg.getProparty( "preDir"		);		// 6.4.3.2 (2016/02/19)
		final String unpreDir	= arg.getProparty( "unpreDir"	);		// 7.2.6.0 (2020/06/30)
		final String prefix		= arg.getProparty( "prefix"		);
		final String unprefix	= arg.getProparty( "unprefix"	);		// 5.1.2.0 (2010/01/01) 追加
		final String suffix		= arg.getProparty( "suffix"		);
		final String unsuffix	= arg.getProparty( "unsuffix"	);		// 5.1.2.0 (2010/01/01) 追加
		final String instr		= arg.getProparty( "instr"		);
		final String uninstr	= arg.getProparty( "uninstr"	);		// 5.1.2.0 (2010/01/01) 追加
		final String instrDir	= arg.getProparty( "instrDir"	);		// 7.2.6.0 (2020/06/30)
		final String uninstrDir	= arg.getProparty( "uninstrDir"	);		// 7.2.6.0 (2020/06/30)
//		final String equals		= arg.getProparty( "equals"		);											// 7.2.7.0 (2020/08/07) 廃止
//		final String notequals	= arg.getProparty( "notequals"	);			// 5.1.2.0 (2010/01/01) 追加    // 7.2.7.0 (2020/08/07) 廃止
		final String fileEquals		= arg.getProparty( "fileEquals"		);									// 7.2.7.0 (2020/08/07) 追加
		final String unFileEquals	= arg.getProparty( "unFileEquals"	);	// 5.1.2.0 (2010/01/01) 追加    // 7.2.7.0 (2020/08/07) 追加
		final String match		= arg.getProparty( "match"		);
		final String unmatch	= arg.getProparty( "unmatch"	);
		final String matchDir	= arg.getProparty( "matchDir"	);		// 7.2.6.0 (2020/06/30)
		final String unmatchDir	= arg.getProparty( "unmatchDir"	);		// 7.2.6.0 (2020/06/30)
		final String modify		= arg.getProparty( "modify"		);
		final String unmodify	= arg.getProparty( "unmodify"	);		// 7.2.6.0 (2020/06/30)
		final String larger		= arg.getProparty( "larger"		);
		final String smaller	= arg.getProparty( "smaller"	);
		final String isHidden	= arg.getProparty( "isHidden"	);		// 5.7.5.0 (2014/04/04) 追加
		maxLevel				= arg.getProparty( "maxLevel",maxLevel);
		outPath					= arg.getProparty( "outPath"	);
		// 4.2.2.0 (2008/05/10) 行数カウントの使用有無
		final boolean useLineCnt	= arg.getProparty( "useLineCnt",false );
		// 5.7.2.1 (2014/01/17) ファイルのMD5の計算有無
		final boolean useMD5		= arg.getProparty( "useMD5",false );
		// 5.7.4.0 (2014/03/07) コメント部分を削除した行数と文字数計算を行うかどうか
		final boolean useOmitCmnt	= arg.getProparty( "useOmitCmnt",false );
		// 6.3.1.0 (2015/06/28) FILEPATH,ADDRESS,FILENAME 属性追加
		final boolean useFilePath	= arg.getProparty( "useFilePath",false );
		// 6.3.1.1 (2015/07/10) MODIFY 属性のDate型から文字列に変換するフォーマットを指定します
		final String modifyForm	= arg.getProparty( "modifyForm",null );
		// 5.7.4.0 (2014/03/07) コメント削除時の文字数計算で利用するファイルのエンコード(初期値:JISAutoDetect)
		final String encode			= arg.getProparty( "encode" , "JISAutoDetect" );

//		// 5.1.2.0 (2010/01/01) 判定をディレクトリでも行うかどうか
//		// 6.4.0.2 (2015/12/11) useDIR 属性に、onlyパラメータを追加します[false:File/true:File+Dir/only:Dir]。
//		// 7.2.6.0 (2020/06/30) useDIR属性 廃止
//		final String useDIR = arg.getProparty( "useDIR","false" );

		errAbend= arg.getProparty( "errAbend"	, errAbend );			// 6.3.1.0 (2015/06/28) errAbend属性追加
		display	= arg.getProparty( "display"	, display );
		debug	= arg.getProparty( "debug"		, debug );				// 5.7.3.0 (2014/02/07) デバッグ情報

		// 6.3.1.1 (2015/07/10) メソッドチェーン化と、大文字/小文字の区別なし(ignoreCase=true)対応
		// 7.2.6.0 (2020/06/30)
//		filter = new HybsFileFilter( useDIR,ignoreCase )
		filter = new HybsFileFilter( ignoreCase )	// 7.2.6.0 (2020/06/30) useDIR属性 廃止
			.startsDir(		preDir		)			// 6.4.3.2 (2016/02/19)
			.startsDir(		unpreDir	, true	)	// 7.2.6.0 (2020/06/30)
			.startsWith(	prefix		)
			.startsWith(	unprefix	, true  )
			.endsWith(		suffix		)
			.endsWith(		unsuffix	, true  )
			.instr(			instr		)
			.instr(			uninstr		, true  )
			.instrDir(		instrDir	)			// 7.2.6.0 (2020/06/30)
			.instrDir(		uninstrDir	, true  )	// 7.2.6.0 (2020/06/30)
//			.fileEquals(	equals		)			// 7.2.7.0 (2020/08/07) 廃止
//			.fileEquals(	notequals	, true  )	// 7.2.7.0 (2020/08/07) 廃止
			.fileEquals(	fileEquals	)			// 7.2.7.0 (2020/08/07) 追加
			.fileEquals(	unFileEquals, true  )	// 7.2.7.0 (2020/08/07) 追加
			.matches(		match		)
			.matches(		unmatch		, true  )
			.matchDir(		matchDir	)			// 7.2.6.0 (2020/06/30)
			.matchDir(		unmatchDir	, true  )	// 7.2.6.0 (2020/06/30)
			.lastModified(	modify		)
			.lastModified(	unmodify	, true	)	// 7.2.6.0 (2020/06/30)
			.isLarger(		larger		)			// 5.7.4.3 (2014/03/28) larger,smaller属性を文字列に変更
			.isSmaller(		smaller		)
			.isHidden(		isHidden	);			// 5.7.5.0 (2014/04/04) 追加

		final File tempFile = new File( startDir );
		if( display ) { println( "start=[" + tempFile + "]" ); }		// 5.7.3.0 (2014/02/07)
		if( tempFile.isDirectory() ) {
			dirs = new Stack<>();
			final File[] fileList = tempFile.listFiles( filter );
			dirs.push( new FileListStack( fileList, level ) );
		}
		else {
			dirs = new Stack<>();
			final File[] fileList = new File[] { tempFile };
			dirs.push( new FileListStack( fileList, level ) );
		}

		newData = new FileLineModel( useLineCnt,useMD5,useOmitCmnt,useFilePath );	// 6.3.1.0 (2015/06/28)
		newData.setEncode( encode );
		newData.setModifyForm( modifyForm );										// 6.3.1.1 (2015/07/10)
	}

	/**
	 * このデータの処理において、次の処理が出来るかどうかを問い合わせます。
	 * この呼び出し１回毎に、次のデータを取得する準備を行います。
	 *
	 * @og.rev 5.3.8.0 (2011/08/01) 処理中の状態を表示するための println を追加
	 *
	 * @return	処理できる:true / 処理できない:false
	 */
	public boolean next() {
		while( !dirs.empty() ) {
			final FileListStack fStack = dirs.pop();

			level = fStack.getLevel();
			if( level > maxLevel ) { continue; }

			final File[] fileList = fStack.getFileList();
			if( fileList == null ) { continue; }

			int address = fStack.getAddress();
			for( ; address<fileList.length; address++ ) {
				inCount++ ;
				if( fileList[address].isDirectory() ) {
					if( debug ) { println( "file Add=" + fileList[address].getAbsolutePath() ); }			// 5.7.3.0 (2014/02/07) デバッグ情報
					final File[] newList = fileList[address].listFiles( filter );
					dirs.push( new FileListStack( newList,level+1) );
				}
				else {
					file = fileList[address];
					if( debug ) { println( "file=" + file ); }			// 5.7.3.0 (2014/02/07) デバッグ情報
					fStack.setAddress( address+1 );
					dirs.push( fStack );
					return true;
				}
			}
		}
		return false;
	}

	/**
	 * 最初に、 行データである LineModel を作成します
	 * FirstProcess は、次々と処理をチェインしていく最初の行データを
	 * 作成して、後続の ChainProcess クラスに処理データを渡します。
	 *
	 * @og.rev 4.2.3.0 (2008/05/26) BIKO 欄に展開ファイル名を記述します。
	 * @og.rev 4.3.1.1 (2008/08/23) BIKO 欄にoutPath 属性を追加します。
	 * @og.rev 6.3.1.0 (2015/06/28) errAbend属性追加。
	 * @og.rev 6.3.9.0 (2015/11/06) コンストラクタで初期化されていないフィールドを null チェックなしで利用している(findbugs)
	 *
	 * @param	rowNo	処理中の行番号
	 *
	 * @return	処理変換後のLineModel
	 */
	public LineModel makeLineModel( final int rowNo ) {
		outCount++ ;
		// 6.3.1.0 (2015/06/28) 取りあえず、エラーの発生しそうな箇所で対応しておく。
		try {
			newData.setFileVals( level,file );
		}
		catch( final RuntimeException ex ) {
			// 6.3.1.1 (2015/07/10) throwExceptionメソッドを使用
			final String errMsg = "rowNo=[" + rowNo + "] , file =[" + file + "]" ;
			throwException( errMsg,ex,errAbend );
		}

		// 6.3.9.0 (2015/11/06) コンストラクタで初期化されていないフィールドを null チェックなしで利用している(findbugs)
		if( file == null ) {
			final String errMsg = "#next() 実行しておかないと、file が初期化されません。" ;
			throw new OgRuntimeException( errMsg );
		}

		// 4.3.1.1 (2008/08/23)
		String biko = null;
		// 4.2.3.0 (2008/05/26) BIKO 欄追加
		if( inPathLen > 0 ) {
			biko = file.getAbsolutePath().substring( inPathLen );
		}

		if( outPath != null ) {
			if( biko == null ) {
				biko = outPath + file.getName() ;
			}
			else {
				biko = outPath + biko ;
			}
		}
		if( biko != null ) {
			newData.setBiko( biko );
		}

		newData.setRowNo( rowNo );

		return newData;
	}

	/**
	 * 引数の LineModel を処理するメソッドです。
	 * 変換処理後の LineModel を返します。
	 * 後続処理を行わない場合(データのフィルタリングを行う場合)は、
	 * null データを返します。つまり、null データは、後続処理を行わない
	 * フラグの代わりにも使用しています。
	 * なお、変換処理後の LineModel と、オリジナルの LineModel が、
	 * 同一か、コピー(クローン)かは、各処理メソッド内で決めています。
	 * ドキュメントに明記されていない場合は、副作用が問題になる場合は、
	 * 各処理ごとに自分でコピー(クローン)して下さい。
	 *
	 * @param	data	オリジナルのLineModel
	 *
	 * @return	処理変換後のLineModel
	 */
	public LineModel action( final LineModel data ) {

		final FileLineModel fileData ;
		if( data instanceof FileLineModel ) {
			fileData = (FileLineModel)data ;
		}
		else {
			// これは、プログラマーの問題なので、errAbend 対象外
			final String errMsg = "データが FileLineModel オブジェクトではありません。" + CR ;
			throw new OgRuntimeException( errMsg );
		}

		if( debug ) { println( "Before:" + data.dataLine() ); }		// 5.1.2.0 (2010/01/01) display の条件変更

		final File inFile = fileData.getFile() ;
		final File[] fileList = inFile.listFiles( filter );

		LineModel rtn = null;
		if( fileList != null && fileList.length > 0 ) {
			rtn = data;
		}

		if( display && rtn != null ) { println( rtn.dataLine() ); }		// 5.1.2.0 (2010/01/01) display の条件変更
		return rtn ;
	}

	/**
	 * プロセスの終了を行います。最後に一度だけ、呼び出されます。
	 * 終了処理(ファイルクローズ、ＤＢクローズ等)に使用します。
	 *
	 * @param   isOK トータルで、OKだったかどうか[true:成功/false:失敗]
	 */
	public void end( final boolean isOK ) {
		dirs		= null;
		file		= null;
		filter		= null;
		newData		= null;
	}

	/**
	 * プロセスの処理結果のレポート表現を返します。
	 * 処理プログラム名、入力件数、出力件数などの情報です。
	 * この文字列をそのまま、標準出力に出すことで、結果レポートと出来るような
	 * 形式で出してください。
	 *
	 * @return   処理結果のレポート
	 */
	public String report() {
		final String report = "[" + getClass().getName() + "]" + CR
				+ TAB + "Start Folder : " + startDir  + CR
				+ TAB + "Search Count : " + inCount   + CR
				+ TAB + "Output Count : " + outCount ;

		return report ;
	}

	/**
	 * このクラスの使用方法を返します。
	 *
	 * @return	このクラスの使用方法
	 * @og.rtnNotNull
	 */
	public String usage() {
		final StringBuilder buf = new StringBuilder( 3000 )
			.append( "Process_FileSearch は、指定のフォルダ以下のファイルを一覧する、FirstProcess"	).append( CR )
			.append( "インターフェースと、ChainProcess インターフェースの実装クラスです。"			).append( CR )
			.append( CR )
			.append( "指定の条件に合致するファイルを検索し、ファイル属性(Level,File,Length,Modify)"	).append( CR )
			.append( "を元に、LineModelを作成し、下流に渡します。"									).append( CR )
			.append( CR )
			.append( "引数文字列中に空白を含む場合は、ダブルコーテーション(\"\") で括って下さい。"	).append( CR )
			.append( "引数文字列の 『=』の前後には、空白は挟めません。必ず、-key=value の様に"		).append( CR )
			.append( "繋げてください。"																).append( CR )
			.append( CR )
			.append( "  -start=開始フォルダ    ：検索を開始するフォルダ"								).append( CR )
			.append( "[ -ignoreCase=true/false]：大文字小文字を区別しないかどうか(初期値:しない[true])" ).append( CR )
			.append( "[ -preDir=接頭辞ﾌｫﾙﾀﾞ   ]：File････,View････,などの接頭辞で始まるフォルダ"		).append( CR )			// 6.4.3.2 (2016/02/19)
			.append( "[ -prefix=接頭辞        ]：File････,View････,などの接頭辞で始まるファイル"		).append( CR )
			.append( "[ -unprefix=不接頭辞    ]：File････,View････,などの接頭辞で始まらないファイル"	).append( CR )
			.append( "[ -suffix=接尾辞        ]：.txt|.java|.jsp.... などの接尾辞で終わるファイル"		).append( CR )
			.append( "[ -unsuffix=不接尾辞    ]：.txt|.java|.jsp.... などの接尾辞で終わらないファイル"	).append( CR )
			.append( "[ -instr=部分文字列     ]：ファイル名と一致する部分文字列"						).append( CR )
			.append( "[ -uninstr=不部分文字列 ]：ファイル名と一致しな部分文字列"						).append( CR )
			.append( "[ -instrDir=部分文字列  ]：フォルダ名と一致する部分文字列"						).append( CR )
			.append( "[ -uninstrDir=不部分文字列 ]：フォルダ名と一致しな部分文字列"						).append( CR )
//			.append( "[ -equals=一致          ]：ファイル名と一致する文字列"							).append( CR )			// 7.2.7.0 (2020/08/07) 廃止
//			.append( "[ -notequals=不一致     ]：ファイル名と一致しない文字列"							).append( CR )			// 7.2.7.0 (2020/08/07) 廃止
			.append( "[ -fileEquals=一致      ]：ファイル名と一致する文字列"							).append( CR )			// 7.2.7.0 (2020/08/07) 追加
			.append( "[ -unFileEquals=不一致  ]：ファイル名と一致しない文字列"							).append( CR )			// 7.2.7.0 (2020/08/07) 追加
			.append( "[ -match=正規表現       ]：ファイル名と一致する正規表現"							).append( CR )
			.append( "[ -unmatch=正規表現     ]：ファイル名と一致しない正規表現"						).append( CR )
			.append( "[ -matchDir=正規表現    ]：フォルダ名と一致する正規表現"							).append( CR )
			.append( "[ -unmatchDir=正規表現  ]：フォルダ名と一致しない正規表現"						).append( CR )
			.append( "[ -modify=YYYYMMDD      ]：指定日付け以降に変更されたファイル"					).append( CR )
			.append( "          YYYYMMDD   : YYYYMMDD 形式での指定日の 00:00:00 を基準時刻"				).append( CR )
			.append( "          TODAY      : 実行日の 00:00:00 を基準時刻"								).append( CR )
			.append( "          YESTERDAY  : 実行日前日の 00:00:00 を基準時刻"							).append( CR )
			.append( "          LAST_WEEK  : 実行日の先週(7日前) 00:00:00 を基準時刻"					).append( CR )
			.append( "          MONTH      : 実行月の 1日 00:00:00 を基準時刻"							).append( CR )
			.append( "          LAST_MONTH : 実行前月の 同日 00:00:00 を基準時刻"						).append( CR )
			.append( "          LAST_YEAR  : 実行前年の 同月同日 00:00:00 を基準時刻"					).append( CR )
			.append( "[ -unmodify=YYYYMMDD    ]：指定日付け以前に変更されたファイル"					).append( CR )
//			.append( "[ -useDIR=[false/true/only]]：判定をディレクトリ名も含めて行うかどうか[false:File/true:File+Dir/only:Dir](初期値:false)").append( CR )
			.append( "[ -larger=サイズ(Byte)  ]：大きさが指定のバイト数と同じか大きいファイル"			).append( CR )
			.append( "[ -smaller=サイズ(Byte) ]：大きさが指定のバイト数より小さいファイル"				).append( CR )
			.append( "[ -isHidden=[false/true]]：true:HIDDENのみ検索/false:NORMALのみ検索(初期値:null)"	).append( CR )
			.append( "[ -maxLevel=最大階層数  ]：ディレクトリの階層を下がる最大数(初期値:256)"			).append( CR )
			.append( "[ -useLineCnt=行数計算  ]：ファイルの行数をカウントするかどうか(初期値:false)"	).append( CR )
			.append( "[ -useMD5=MD5計算値     ]：ファイルのMD5計算を行うかどうかを指定(初期値:false)" 	).append( CR )
			.append( "[ -useOmitCmnt=[false/true] ]：コメント部分を削除した行数と文字数計算を行うかどうか(初期値:false)" 	).append( CR )
			.append( "[ -useFilePath=[false/true] ]：FILEPATH,ADDRESS,FILENAME 属性を準備(初期値:false)" ).append( CR )
			.append( "[ -modifyForm=日付ﾌｫｰﾏｯﾄ]：MODIFY 属性のDate型から文字列に変換するフォーマットを指定します(初期値:null)" ).append( CR )
			.append( "[ -inPath=入力共通パス  ]：BIKO作成用のファイルパスから削除する文字列"			).append( CR )
			.append( "[ -outPath=出力追加パス ]：BIKO作成用のファイルパスに追加する文字列"				).append( CR )
			.append( "[ -display=[false/true] ]：trueは、検索状況を表示(初期値:false)"					).append( CR )
			.append( "[ -debug=[false/true]   ]：trueは、デバッグ状況を表示(初期値:false)"				).append( CR )
			.append( CR ).append( CR )
			.append( getArgument().usage() ).append( CR );

		return buf.toString();
	}

	/**
	 * このクラスは、main メソッドから実行できません。
	 *
	 * @param	args	コマンド引数配列
	 */
	public static void main( final String[] args ) {
		LogWriter.log( new Process_FileSearch().usage() );
	}

	/**
	 * このクラスはファイルをスタックを使用して展開する場合の
	 * 個々の状態を保持する為のクラスです。
	 *
	 * @version  4.0
	 * @author   Kazuhiko Hasegawa
	 * @since    JDK5.0,
	 */
	private static final class FileListStack {
		private int address ;
		private final File[] files;
		private final int level;

		/**
		 * コンストラクター
		 * 初期値を設定します。
		 * ファイルの配列については、コピーせずそのまま内部配列にセットしています。
		 *
		 * @param files ファイルの配列(ファイルリスト)
		 * @param	level	レベル(指定のstartフォルダからの階層数)
		 */
		FileListStack( final File[] files,final int level ) {
			this.files   = files;
			this.address = 0;
			this.level   = level;
		}

		/**
		 * ファイルリストのアドレスを設定します。
		 * スタックから取り出した後、配列を前回の続きからサーチする場合に使用します。
		 *
		 * @param	address	ファイルリストのアドレス
		 */
		/* default */ void setAddress( final int address ) {
			this.address = address;
		}

		/**
		 * ファイルリストのアドレスを取り出します。
		 *
		 * @return	ファイルリストのアドレス
		 */
		/* default */ int getAddress() {
			return address;
		}

		/**
		 * ファイルリストを取り出します。
		 * ファイルの配列については、コピーせずそのまま内部配列を返しています。
		 *
		 * @return ファイルリスト配列
		 */
		/* default */ File[] getFileList() {
			return files;
		}

		/**
		 * 階層レベルを取り出します。
		 *
		 * @return	レベル
		 */
		/* default */ int getLevel() {
			return level;
		}
	}
}
