/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.query;

import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.db.AbstractQuery;
import org.opengion.fukurou.util.ErrorMessage;
import org.opengion.fukurou.util.StringUtil;

import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;

/**
 * 引数引き当て(PreparedStatement) を利用した登録系Queryです。
 *
 * java.sql.PreparedStatement を用いて、データベース検索処理を行います。
 * 引数に、指定した値を配列で渡します。
 * 内部変数の受け渡しのデフォルト実装は、AbstractQuery クラスを継承している
 * ため,ここでは、execute() メソッドを実装しています。
 * このクラスでは、ステートメント文を execute() する事により,データベースを
 * 検索した結果を DBTableModel に割り当てます。
 *
 * このクラスは、Query で使用されるより、内部の DBTableModelUtilから、
 * 利用されるケースが主です。
 * Query で使用する場合は、JDBCPrepared ではなく、JDBCTableUpdate を
 * 使用することを、ご検討ください。
 *
 * ※postgres8.3以降では、数値型の列に対して、実行した場合は、型相違のエラーが発生します。
 * postgres8.3以降で利用する場合は、postgres側に暗黙の型変換(CAST)の実装を検討して下さい。
 *
 * @og.formSample
 * 例：
 *     可変引数付きのＳＱＬ文を実行します。
 *     これは、INSERT,UPDATE,DELETE など、どのようなSQL文でも実行できます。
 *     names 属性で指定するのは、DBTableModelのカラム名で、その値が順番に、
 *     引数(?記号)の個所に設定されます。
 *     選択されたデータ(行)の数だけ、繰り返し実行されます。
 *
 * jsp/TYPE1A/copy.jsp
 * &lt;og:value scope=&quot;session&quot;
 *            key=&quot;names&quot;
 *            value=&quot;CLM,NAME_JA,LABEL_NAME,KBSAKU,SYSTEM_ID,LANG&quot; /&gt;
 * &lt;og:value scope=&quot;session&quot; key=&quot;SQL&quot; &gt;
 *     INSERT INTO GEA08
 *         (CLM,NAME_JA,LABEL_NAME,KBSAKU,SYSTEM_ID,LANG,
 *          FGJ,DYSET,DYUPD,USRSET,USRUPD,PGUPD)
 *     VALUES
 *         (?,?,?,?,?,?,
 *          &#39;1&#39;,&#39;{&#064;USER.YMDH}&#39;,&#39;{&#064;USER.YMDH}&#39;,&#39;{&#064;USER.ID}&#39;,&#39;{&#064;USER.ID}&#39;,&#39;{&#064;GUI.KEY}&#39;)
 * &lt;/og:value&gt;
 *
 * jsp/TYPE1A/entry.jsp
 * &lt;h:tableUpdate
 *     command   = &quot;{&#064;command}&quot;
 *     queryType = &quot;JDBCPrepared&quot;
 *     names     = &quot;{&#064;names}&quot;  &gt;
 * {&#064;SQL}
 * &lt;/og:tableUpdate&gt;
 *
 * &lt;!-- 前画面で指定のSQL文を削除します。(scope=&quot;session&quot;なので削除が必要。) --&gt;
 * &lt;og:value scope=&quot;session&quot; key=&quot;names&quot; command=&quot;REMOVE&quot; /&gt;
 * &lt;og:value scope=&quot;session&quot; key=&quot;SQL&quot;   command=&quot;REMOVE&quot; /&gt;
 *
 * 以下はSELECTで使用する場合の例
 * ※値はnamesに指定した値が、?に順番に設定されます。
 * (andタグのplaceHolder属性は、valueに指定した式を、実行する or 実行しないの判定に利用されます。)
 * &lt;og:query command="NEW" queryType="JDBCPrepared" names="NO,NAME,KBN"&gt;
 *      SELECT
 *          NO,NAME,KBN
 *      FROM
 *          T01
 *      &lt;og:where&gt;
 *          &lt;og:and value="NO = ?" placeHolder="{&#064;NO}" /&gt;
 *          &lt;og:and value="NAME LIKE ? || '%'" placeHolder="{&#064;NAME}" /&gt;
 *          &lt;og:and value="KBN IN (?)" multi="true" placeHolder="{&#064;KBN}"/&gt;
 *      &lt;/og:where&gt;
 * &lt;/og:query&gt;
 *
 * @og.group データ表示
 * @og.group データ編集
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class Query_JDBCPrepared extends AbstractQuery {
	/** このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "6.9.3.0 (2018/03/26)" ;

	/**
	 * デフォルトコンストラクター
	 *
	 * @og.rev 6.4.2.0 (2016/01/29) PMD refactoring. Each class should declare at least one constructor.
	 */
	public Query_JDBCPrepared() { super(); }		// これも、自動的に呼ばれるが、空のメソッドを作成すると警告されるので、明示的にしておきます。

	/**
	 * 引数配列付のクエリーを実行します。
	 * 処理自体は, #execute() と同様に、各サブクラスの実装に依存します。
	 * これは、PreparedQuery で使用する引数を配列でセットするものです。
	 * select * from emp where deptno = ? and job = ? などの PreparedQuery の
	 * ? 部分の引数を
	 * 順番にセットしていきます。
	 *
	 * @og.rev 2.1.2.3 (2002/12/02) データベース更新時に、更新フラグをセットするように変更
	 * @og.rev 2.3.1.3 (2003/01/28) Open Cursor が、大量に残る件の対応。ResultSet を close()
	 * @og.rev 3.1.1.0 (2003/03/28) 同期メソッド(synchronized付き)を非同期に変更する。
	 * @og.rev 3.3.3.1 (2003/07/18) ＤＢ登録時の後ろスペースを削除する。
	 * @og.rev 3.5.6.0 (2004/06/18) PreparedStatement をexecute 間で使いまわします。
	 * @og.rev 3.8.0.8 (2005/10/03) エラーメッセージの出力順をメッセージ＋Queryに変更します。
	 * @og.rev 5.3.8.0 (2011/08/01) pstmt.setObject で、useParamMetaData の判定を避けるため、pstmt.setString で代用(PostgreSQL対応)
	 * @og.rev 6.3.6.1 (2015/08/28) close(),realClose() 廃止。Queryはキャッシュしません。
	 * @og.rev 6.4.2.1 (2016/02/05) try-with-resources 文で記述。
	 * @og.rev 6.9.3.0 (2018/03/26) DB_FETCH_SIZE追加。
	 *
	 * @param   args オブジェクトの引数配列(可変長引数)
	 */
	@Override
	public void execute( final String... args ) {			// 6.1.1.0 (2015/01/17) refactoring

		// 6.4.2.1 (2016/02/05) try-with-resources 文
		try( final PreparedStatement pstmt = getConnection().prepareStatement( getStatement() ) ) {
			pstmt.setQueryTimeout( DB_MAX_QUERY_TIMEOUT );

			// 6.1.1.0 (2015/01/17) refactoring. 可変引数にしたため、null は来ない。
			for( int i=0; i<args.length; i++ ) {
				pstmt.setString( i+1,StringUtil.rTrim( args[i] ) );	// 5.3.8.0 (2011/08/01) 処理の簡素化
			}

			if( pstmt.execute() ) {
				try( final ResultSet resultSet = pstmt.getResultSet() ) {
					resultSet.setFetchSize( DB_FETCH_SIZE );		// 6.9.3.0 (2018/03/26)
					createTableModel( resultSet );
					setUpdateFlag( false );
				}
			}
			else {
				setExecuteCount( pstmt.getUpdateCount() );
			}

			setErrorCode( ErrorMessage.OK );
		}
		catch( final SQLException ex ) {		// catch は、close() されてから呼ばれます。
			setErrorCode( ErrorMessage.EXCEPTION );

			final String errMsg = ex.getMessage() + ":" + ex.getSQLState() + CR
						+ getStatement() + CR;
			throw new HybsSystemException( errMsg,ex );		// 3.5.5.4 (2004/04/15) 引数の並び順変更
		}
	}

}
