/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.fukurou.util;

import java.util.Arrays ;

import static org.opengion.fukurou.system.HybsConst.CR;				// 6.1.0.0 (2014/12/26) refactoring
import static org.opengion.fukurou.system.HybsConst.BUFFER_MIDDLE;	// 6.1.0.0 (2014/12/26) refactoring

/**
 * ToString.java は、共通的に使用される toString() を簡素化するクラスです。
 * デバッグ情報を出力する最も一般的な方法は、内部文字列の出力です。
 * 通常、キーと値をペアで記述するため、StringBuilder で append しながら作成するにしても
 * コーディング的にあまり見栄えの良い形にはなりません。
 * ここでは、それらを簡易的に整形して出力できるように、メソッドを用意しました。
 *
 * @og.group ユーティリティ
 *
 * @version  4.0
 * @author	 Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public final class ToString {

	/** ４つ分のスペースです。	*/
	private static final String SPACE = "    " ;
	/** ３つ分のカンマです。	*/
	private static final String COMMA = " , " ;

	private final StringBuilder buf ;

	/** 最後に改行記号を入れたかどうか。	*/
	private boolean lastCR = true ;

	/** キー文字列の最大長(すべてASCII換算)	*/
	private int maxSize	;

	/**
	 * タイトルを指定するコンストラクター
	 * タイトルは、先頭にスペースなしで入れます。最後は改行ありです。
	 *
	 * @param title タイトル文字列
	 */
	public ToString( final String title ) {
		buf = new StringBuilder( BUFFER_MIDDLE );
		buf.append( "TITLE = [" ).append( title ).append( ']' ).append( CR );		// 6.0.2.5 (2014/10/31) char を append する。
		lastCR = true ;
	}

	/**
	 * 簡易的にオブジェクトを構築する static メソッド
	 * タイトルは、先頭にスペースなしで入れます。最後は改行ありです。
	 *
	 * @param	title タイトル文字列
	 *
	 * @return	ToStringオブジェクト
	 * @og.rtnNotNull
	 */
	public static ToString title( final String title ) {
		return new ToString( title );
	}

	/**
	 * 改行なしのキーと値のペアを設定します。
	 * 前回、改行で終わっている場合は、キーから始めます。
	 * そうでない場合、" , " を出力してから、書き始めます。
	 *
	 * @param	key	キー文字列
	 * @param	val	値文字列
	 *
	 * @return	自分自身
	 * @og.rtnNotNull
	 */
	public ToString print( final String key,final Object val ) {
		if( lastCR ) {	buf.append( SPACE ); }
		else {			buf.append( COMMA ); }
		lastCR = false;

		// 6.0.2.5 (2014/10/31) char を append する。
		// 6.4.1.1 (2016/01/16) PMD refactoring. Avoid if (x != y) ..; else ..;
		if( key == null ) {
			buf.append( "   [" ).append( String.valueOf( val ) ).append( ']' );
		}
		else {
			final int len = key.length();
			if( len > maxSize ) { maxSize = len; }
			buf.append( key ).append( " = [" ).append( String.valueOf( val ) ).append( ']' );
		}

		return this ;
	}

	/**
	 * 改行ありのキーと値のペアを設定します。
	 * 前回、改行で終わっている場合は、キーから始めます。
	 * そうでない場合、" , " を出力してから、書き始めます。
	 *
	 * @param	key	キー文字列
	 * @param	val	値文字列
	 *
	 * @return	自分自身
	 * @og.rtnNotNull
	 */
	public ToString println( final String key,final Object val ) {
		print( key,val );

		buf.append( CR );
		lastCR = true;

		return this ;
	}

	/**
	 * 改行なしのキーと値配列のペアを設定します。
	 * 前回、改行で終わっている場合は、キーから始めます。
	 * そうでない場合、" , " を出力してから、書き始めます。
	 *
	 * @param	key	キー文字列
	 * @param	val	値配列
	 *
	 * @return	自分自身
	 * @og.rtnNotNull
	 */
	public ToString print( final String key,final Object[] val ) {
		final String str = Arrays.toString( val );

		return print( key,str.substring( 1,str.length()-1 ) );
	}

	/**
	 * 改行ありのキーと値配列のペアを設定します。
	 * 前回、改行で終わっている場合は、キーから始めます。
	 * そうでない場合、" , " を出力してから、書き始めます。
	 *
	 * @param	key	キー文字列
	 * @param	val	値配列
	 *
	 * @return	自分自身
	 * @og.rtnNotNull
	 */
	public ToString println( final String key,final Object[] val ) {
		final String str = Arrays.toString( val );

		return println( key,str.substring( 1,str.length()-1 ) );
	}

	/**
	 * 改行のみ設定します。
	 *
	 * @return	自分自身
	 * @og.rtnNotNull
	 */
	public ToString println() {
		buf.append( CR );
		lastCR = true;

		return this;
	}

	/**
	 * 改行付きの値のみ設定します。
	 *
	 * @param	val	値
	 *
	 * @return	自分自身
	 * @og.rtnNotNull
	 */
	public ToString println( final Object val ) {
		return println( null,val );
	}

	/**
	 * 改行付きの値のみ設定します。
	 *
	 * @param	val	値配列
	 *
	 * @return	自分自身
	 * @og.rtnNotNull
	 */
	public ToString println( final Object[] val ) {
		final String str = Arrays.toString( val );

		return println( null,str.substring( 1,str.length()-1 ) );
	}

	/**
	 * 先頭のキーの位置を最大値に合わせて、整形します。
	 *
	 * @return	自分自身
	 * @og.rtnNotNull
	 */
	public ToString fixForm() {
		final String searchKey = CR + SPACE ;
		final int    skLen = searchKey.length() ;

		final char[] ch = new char[maxSize];
		Arrays.fill( ch,' ' );
		final String MAX_SPACE = new String( ch );

		int adrs = buf.indexOf( searchKey );
		while( adrs >= 0 ) {
			final int eq = buf.indexOf( "=",adrs );
			if( eq >= 0 && buf.charAt(adrs+skLen) != ' ' ) {
				final int nextAdrs  = buf.indexOf( searchKey,adrs+skLen );
				if( nextAdrs < 0 || eq < nextAdrs ) {
					buf.insert( eq, MAX_SPACE.substring( eq-(adrs+skLen)-1 ) );
				}
			}
			adrs = buf.indexOf( searchKey,adrs+skLen );
		}

		return this;
	}

	/**
	 * 内部バッファを文字列にして返します。
	 *
	 * @return	内部バッファを文字列にして返します。
	 * @og.rtnNotNull
	 */
	@Override
	public String toString() {
		return buf.toString() ;
	}
}
