/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.query;

import org.opengion.hayabusa.db.AbstractQuery;
import org.opengion.hayabusa.db.DBTableModel;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.fukurou.util.ErrorMessage;
import org.opengion.fukurou.util.StringUtil;
// import org.opengion.fukurou.util.HybsDateUtil;			// 5.5.8.5 (2012/11/27)
import org.opengion.fukurou.model.Formatter;
import org.opengion.fukurou.db.DBUpdater;				// 6.9.3.0 (2018/03/26)

import java.sql.Connection;
import java.sql.PreparedStatement;
// import java.sql.ParameterMetaData;
import java.sql.SQLException;

/**
 * 引数引き当て(PreparedStatement) を利用した登録系Queryです。
 *
 * java.sql.PreparedStatement を用いて、データベース登録処理を行います。
 * 引数の指定方法は、DBTableModele のカラム名に対応する名称を、SQL文の[カラム名]形式で
 * 記述します。これを解析して、実際に実行する PreparedStatement に対応する文字列を
 * 作成します。
 * たとえば、INSERT INTO GEXX (CLM,NAME_JA,LABEL_NAME) VALUES ([CLM],[NAME_JA],[LABEL_NAME] )
 * と記述すれば、内部で、DBTableModele のカラム名に対応する値を取り出し、SQL文として、
 * INSERT INTO GEXX (CLM,NAME_JA,LABEL_NAME) VALUES (?,?,? ) を実行します。
 *
 * @og.formSample
 * ●使用例
 *
 *    ・QUERYを直接書く場合
 *    【entry.jsp】
 *        &lt;og:tableUpdate
 *            command   = &quot;{&#064;command}&quot;
 *            queryType = &quot;JDBCTableUpdate&quot;
 *        &gt;
 *            INSERT INTO GE41
 *                (CLM,NAME_JA,LABEL_NAME,KBSAKU,SYSTEM_ID,LANG,
 *                 FGJ,DYSET,DYUPD,USRSET,USRUPD,PGUPD)
 *            VALUES
 *                ([CLM],[NAME_JA],[LABEL_NAME],[KBSAKU],[SYSTEM_ID],[LANG],
 *                 &#39;1&#39;,&#39;{&#064;USER.YMDH}&#39;,&#39;{&#064;USER.YMDH}&#39;,&#39;{&#064;USER.ID}&#39;,&#39;{&#064;USER.ID}&#39;,&#39;{&#064;GUI.KEY}&#39;)
 *        &lt;/og:tableUpdate&gt;
 *
 * @og.rev 4.0.0.0 (2005/01/31) 新規作成
 * @og.group データ編集
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class Query_JDBCTableUpdate extends AbstractQuery {
	/** このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "7.2.9.1 (2020/10/23)" ;

	/**
	 * デフォルトコンストラクター
	 *
	 * @og.rev 6.4.2.0 (2016/01/29) PMD refactoring. Each class should declare at least one constructor.
	 */
	public Query_JDBCTableUpdate() { super(); }		// これも、自動的に呼ばれるが、空のメソッドを作成すると警告されるので、明示的にしておきます。

	/**
	 * 引数配列付のクエリーを実行します。
	 * 処理自体は, #execute() と同様に、各サブクラスの実装に依存します。
	 * これは、PreparedQuery で使用する引数を配列でセットするものです。
	 * select * from emp where deptno = ? and job = ? などの PreparedQuery の
	 * [カラム名] 部分の引数を、DBTableModelから順番にセットしていきます。
	 *
	 * @og.rev 3.8.0.8 (2005/10/03) エラーメッセージの出力順をメッセージ＋Queryに変更します。
	 * @og.rev 4.0.0.0 (2007/05/09) ParameterMetaData を使用したパラメータ設定追加。
	 * @og.rev 4.0.0.0 (2007/09/25) isOracle から useParamMetaData に変更
	 * @og.rev 5.3.8.0 (2011/08/01) useParamMetaData を ConnectionFactory経由で取得。(PostgreSQL対応)、setNull 対応
	 * @og.rev 5.5.5.4 (2012/08/18) useParamMetaData 処理を、ループの外に出す。(PostgreSQL対応)
	 * @og.rev 5.5.5.4 (2012/08/18) DATE オブジェクトを登録できるようにする。
	 * @og.rev 5.5.8.5 (2012/11/27) TIMESTAMP型でも処理できるようにします。
	 * @og.rev 5.6.9.4 (2013/10/31) エラーメッセージに１行前の情報も出力します。
	 * @og.rev 6.3.6.1 (2015/08/28) close(),realClose() 廃止。Queryはキャッシュしません。
	 * @og.rev 6.4.2.1 (2016/02/05) try-with-resources 文で記述。
	 * @og.rev 6.4.3.4 (2016/03/11) Formatterに新しいコンストラクターを追加する。
	 * @og.rev 6.4.4.2 (2016/04/01) contains 判定を行う新しいメソッドを使用します。
	 * @og.rev 6.4.6.0 (2016/05/27) isNumber , isDate 追加。
	 * @og.rev 6.9.3.0 (2018/03/26) DBUpdaterクラスを利用したバッチ更新を追加します。
	 * @og.rev 6.9.8.0 (2018/05/28) エラー時に、わかりやすいように、引数を、パラメータ順にします。
	 * @og.rev 7.2.9.1 (2020/10/23) isTimeのメソッドを統合します。
	 *
	 * @param   rowNo 選択された行番号配列(登録する対象行)
	 * @param   table DBTableModelオブジェクト(登録する元データ)
	 */
	@Override
	public void execute( final int[] rowNo, final DBTableModel table ) {

		int row = 0;			// エラー時に表示するエラー行番号
		int executeCount = 0;	// 処理件数
		final Formatter form = new Formatter( table,getStatement() );	// 6.4.3.4 (2016/03/11)
		final int[] clmNos = form.getClmNos();		// 引数の個数分の配列。カラム番号を保存
		final String query = form.getQueryFormatString();
		final int    cnt   = clmNos.length;			// 引数の個数(カラムの個数ではありません。)

		// 5.5.5.4 (2012/08/18) Timestamp オブジェクトを登録できるようにする。
		boolean useTimeStamp = false;
		// 6.3.9.1 (2015/11/27) Found 'DD'-anomaly for variable(PMD)
		final boolean[] isTime = new boolean[cnt];
		for( int j=0; j<cnt; j++ ) {
			// 5.5.8.5 (2012/11/27) TIMESTAMP型でも処理できるようにします。
			// 6.4.4.2 (2016/04/01) contains 判定を行う新しいメソッドを使用します。
			isTime[j] = table.getDBColumn( clmNos[j] ).isDateType();	// 6.4.6.0 (2016/05/27)
			if( !useTimeStamp && isTime[j] ) { useTimeStamp = true; }	// isTime[j] == true 時に、一度だけ実行される。
		}

		final Connection conn = getConnection();
		// 6.9.8.0 (2018/05/28) エラー時に、わかりやすいように、引数を、パラメータ順にします。
		final String[] vals = new String[cnt];

		// 6.4.2.1 (2016/02/05) try-with-resources 文
		try( PreparedStatement pstmt = conn.prepareStatement( query ) ) {		// 更新系なので、setFetchSize は不要。
			pstmt.setQueryTimeout( DB_MAX_QUERY_TIMEOUT );
		//	((oracle.jdbc.OraclePreparedStatement)pstmt).setExecuteBatch(50);
			// 4.0.0.0 (2007/09/25) isOracle から usePMeta に変更
			final boolean usePMeta = useParameterMetaData();					// 5.3.8.0 (2011/08/01)

			final DBUpdater dbUpper = new DBUpdater( cnt,pstmt,usePMeta,useTimeStamp ? isTime : null );		// 7.2.9.1 (2020/10/23)

			for( int i=0; i<rowNo.length; i++ ) {
				row = rowNo[i];
				final String[] data = table.getValues( row );
				for( int j=0; j<cnt; j++ ) {
					vals[j] = StringUtil.rTrim( data[ clmNos[j] ] );
				}

				dbUpper.execute( vals );
			}

			executeCount = dbUpper.execEnd();

			setExecuteCount( executeCount );
			setErrorCode( ErrorMessage.OK );
		}
		catch( final SQLException ex) {		// catch は、close() されてから呼ばれます。
			setErrorCode( ErrorMessage.EXCEPTION );
			final StringBuilder errMsg = new StringBuilder( BUFFER_MIDDLE )
				.append( ex.getMessage() ).append( ':' ).append( ex.getSQLState() ).append( CR )
				.append( "  QUERY=" ).append( getStatement() ).append( CR )
				.append( "  ROW =[" ).append( (row+1) ).append( ']' ).append( CR )
				// 6.9.8.0 (2018/05/28) エラー時に、わかりやすいように、引数を、パラメータ順にします。
				.append( "  VALS=[" ).append( StringUtil.array2csv( vals )).append( ']' )	// 6.9.8.0 (2018/05/28)
				.append( CR ) ;
			// 5.6.9.4 (2013/10/31)
			if( row > 0 ) {
				// 6.9.8.0 (2018/05/28) エラー時に、わかりやすいように、引数を、パラメータ順にします。
				for( int j=0; j<cnt; j++ ) {
					vals[j] = StringUtil.rTrim( table.getValue( row-1,clmNos[j] ) );
				}

				errMsg.append("  ROW(-1) =[" ).append( row ).append( ']' ).append( CR )		// 6.3.9.0 (2015/11/06) This statement may have some unnecessary parentheses(PMD)
					.append(  "  VALS(-1)=[" ).append( StringUtil.array2csv( vals )).append( ']' )	// 6.9.8.0 (2018/05/28)
					.append( CR ) ;
			}
			throw new HybsSystemException( errMsg.toString(),ex );		// 3.5.5.4 (2004/04/15) 引数の並び順変更
		}
	}
}
