/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.fukurou.process;

import java.util.HashMap;
import java.util.Map;

import org.opengion.fukurou.queue.QueueInfo;
import org.opengion.fukurou.queue.QueueSend;
import org.opengion.fukurou.queue.QueueSendFactory;
import org.opengion.fukurou.util.Argument;
import org.opengion.fukurou.util.LogWriter;

/**
 *Process_QueueSendは、MQ or SQSにメッセージキューを送信する、
 *FirstProcessインタフェースの実装クラスです。
 *
 *@og.formSample
 * 1)MQにメッセージを送信する場合
 *   java -DmqUserId=[mqﾕｰｻﾞid] -DmqPassword=[mqﾊﾟｽﾜｰﾄﾞ] -cp [ｸﾗｽﾊﾟｽ] org.opengion.fukurou.process.MainProcess org.opengion.fukurou.process.Process_Logger -logFile=System.out org.opengion.fukurou.process.Process_QueueSend -queueType=MQ -jmsServer=[mqｻｰﾊﾞｰ] -groupId=[ｸﾞﾙｰﾌﾟID] -message=[ﾒｯｾｰｼﾞ]
 * 2)SQSにメッセージを送信する場合
 *   java -cp [ｸﾗｽﾊﾟｽ] org.opengion.fukurou.process.MainProcess org.opengion.fukurou.process.Process_Logger -logFile=System.out org.opengion.fukurou.process.Process_QueueSend -accessKey=[awsｱｸｾｽｷｰ] -secretKey=[awsｼｰｸﾚｯﾄｷｰ] -queueType=SQS -jmsServer=[sqsｻｰﾊﾞｰ] -groupId=[ｸﾞﾙｰﾌﾟID] -message=[ﾒｯｾｰｼﾞ]
 *
 *※proxy環境から、外部のMQやSQSサーバにはﾌﾟﾛｷｼ情報を渡して、実行する必要があります。
 *-Dhttp.proxyHost=[proxyﾎｽﾄ] -Dhttp.proxyPort=[proxyﾎﾟｰﾄ] -Dhttps.proxyHost=[proxyﾎｽﾄ] -Dhttps.proxyPort=[proxyﾎﾟｰﾄ]
 *
 * -queueType=キュータイプ       :MQ or SQS
 * -jmsServer=キューサーバー     :キューサーバーのURLを指定
 * -groupId=グループID           :キュー格納先のグループID
 * -message=送信メッセージ       :キューに格納するメッセージ
 * [-sccessKey=アクセスキー]     :SQSに接続用のアクセスキーです(aws上で取得)
 * [-secretKey=シークレットキー] :SQSに接続用のシークレットキーです(aws上で取得)
 * 
 * コマンド例
 * java -Dhttp.proxyHost=proxyhost -Dhttp.proxyPort=8080 -Dhttps.proxyHost=proxyhost -Dhttps.proxyPort=8080 -cp H:\sample\* ^
 * org.opengion.fukurou.process.MainProcess ^
 * org.opengion.fukurou.process.Process_Logger -logFile=System.out ^
 * org.opengion.fukurou.process.Process_QueueSend -accessKey=[ｱｸｾｽｷｰ] -secretKey=[ｼｰｸﾚｯﾄｷｰ] -queueType=SQS ^
 * -jmsServer=https://sqs.ap-northeast-1.amazonaws.com/000000000000/otfifo01.fifo -groupId=sample -message=sendMsg
 *
 * @og.rev 5.10.17.1 (2019/11/15) 新規追加
 *
 * @verion 5
 * @since JDK7
 */
public class Process_QueueSend extends AbstractProcess implements FirstProcess{
	private static final String name = "";
	private static final Map<String, String> mustProperty;
	private static final Map<String, String> usableProperty;

	QueueSend queueSend;

	private String queueType;
	private String jmsServer;
	private String groupId;
	private String message;

	static {
		mustProperty = new HashMap<String,String>();
		mustProperty.put("queueType", "キュータイプ");
		mustProperty.put("jmsServer", "jms接続先");
		mustProperty.put("groupId", "グループID");
		mustProperty.put("message", "メッセージ");

		usableProperty = new HashMap<String,String>();
		// SQS用
		usableProperty.put("accessKey", "アクセスキ");
		usableProperty.put("secretKey",  "シークレットキー");
	}

	/**
	 * コンストラクター
	 */
	public Process_QueueSend() {
		super(name, mustProperty, usableProperty);
	}

	/**
	 * プロセスの初期化を行います。初めに一度だけ、呼び出されます。
	 * 初期処理(ファイルオープン、ＤＢオープン等)に使用します。
	 *
	 * @param   paramProcess データベースの接続先情報などを持っているオブジェクト
	 */
	@Override
	public void init(ParamProcess paramProcess) {
		Argument arg = getArgument();

		queueType = arg.getProparty("queueType");
		jmsServer = arg.getProparty("jmsServer");
		groupId = arg.getProparty("groupId");
		message = arg.getProparty("message");
		final String accessKey = arg.getProparty("accessKey");
		final String secretKey = arg.getProparty("secretKey");

		queueSend = QueueSendFactory.newQueueSend(queueType);

		// バッチ実行
		queueSend.setBatchFlg(true);

		// 接続処理
		queueSend.connect(jmsServer, accessKey, secretKey);
	}

	/**
	 * プロセスの終了を行います。最後に一度だけ、呼び出されます。
	 * 終了処理(ファイルクローズ、ＤＢクローズ等)に使用します。
	 *
	 * @param   isOK トータルで、OKだったかどうか[true:成功/false:失敗]
	 */
	@Override
	public void end(boolean isOK) {
		queueType = "";
		jmsServer = "";
		groupId = "";
		message = "";

		if(queueSend != null) {
			queueSend.close();
		}
		queueSend = null;
	}

	/**
	 * このデータの処理において、次の処理が出来るかどうかを問い合わせます。
	 * この呼び出し１回毎に、次のデータを取得する準備を行います。
	 *
	 * @return 処理できる:true / 処理できない:false
	 **/
	@Override
	public boolean next() {
		QueueInfo queueInfo = new QueueInfo();
		queueInfo.setJmsServer(jmsServer);
		queueInfo.setMqQueueName(groupId);
		queueInfo.setSqsFifoGroupId(groupId);
		queueInfo.setMessage(message);

		queueSend.sendMessage(queueInfo);

		return false;
	}

	/**
	 * 最初に、 行データである LineModel を作成します
	 * FirstProcess は、次々と処理をチェインしていく最初の行データを
	 * 作成して、後続の ChainProcess クラスに処理データを渡します。
	 *
	 * @param rowNo 処理中の行番号
	 *
	 * @return 処理変換後のLineModel
	 * */
	@Override
	public LineModel makeLineModel(int rowNo) {
		// 後続のChainProcessは実行しません。
		return null;
	}

	/**
	 * プロセスの処理結果のレポート表現を返します。
	 * 処理プログラム名、入力件数、出力件数などの情報です。
	 * この文字列をそのまま、標準出力に出すことで、結果レポートと出来るような
	 * 形式で出してください。
	 *
	 * @return   処理結果のレポート
	 */
	@Override
	public String report() {
		final String report = "[" + getClass().getName() + "]" + CR
				+ TAB + "queueType:" + queueType + CR
				+ TAB + "jmsServer:" + jmsServer + CR
				+ TAB + "gropuId:" + groupId + CR
				+ TAB + "message:" + message;
		return report;
	}

	/**
	 * このクラスの使用方法を返します。
	 *
	 * @return	このクラスの使用方法
	 */
	@Override
	public String usage() {
		StringBuilder buf = new StringBuilder();

		buf.append("Process_QueueSendは、MQ or SQSにメッセージキューを送信する、").append( CR );
		buf.append("FirstProcessインタフェースの実装クラスです。").append(CR);
		buf.append(CR);
		buf.append("-queueType=キュータイプ   :MQ or SQS").append(CR);
		buf.append("-jmsServer=キューサーバー :キューサーバーのURLを指定").append(CR);
		buf.append("-groupId=グループID       :キュー格納先のグループID").append(CR);
		buf.append("-message=送信メッセージ   :キューに格納するメッセージ").append(CR);
		buf.append("[-sccessKey=アクセスキー]").append(CR);
		buf.append("[-secretKey=シークレットキー]").append(CR);
		buf.append( CR ).append( CR );
		buf.append( getArgument().usage() ).append( CR );

		return buf.toString();
	}

	/**
	 * このクラスは、main メソッドから実行できません。
	 *
	 * @param	args	コマンド引数配列
	 */
	public static void main(final String[] args) {
		LogWriter.log(new Process_QueueSend().usage());
	}

}
